/*
C
C  _______________________________________________________________
C
C*   Licence
C    =======
C
C    You may use or modify this code for your own non commercial
C    purposes for an unlimited time. 
C    In any case you should not deliver this code without a special 
C    permission of ZIB.
C    In case you intend to use the code commercially, we oblige you
C    to sign an according licence agreement with ZIB.
C
C
C  _______________________________________________________________
C
*/

#include "stdio.h"
#include "math.h"
 
#include "kask.h"
#include "kasksol.h"
#include "kaskass.h"

/*
   Direct solution of a positive definit matrix by a rational
   cholesky decomposition (copied from Wilkinson/ Reinsch;
   Handbok for Automatic Computation, Vol. II, Linear Algebra)
   and a forward/ backward substitution.
*/
 
/*
    CholDec(a,d,l,n) factorizes a=ldl**T
         a: symmetric positive definit Matrix
         d: vector, containing diagonal components
         l: triangular matrix, l[i][i]=1.0
         n: dimension
    error: a indefinit
 
Ref.: Wilkinson/Reinsch, p.21, routine symdet
*/
 
int CholDec(a, d, l ,n)
  REAL **a, *d, **l;
  int n;
  {
    int i, j, k;
    REAL x, y, z;
 
    Set0MatFull(l, n, true);
    Set0VecFull(d, n);
    for (i = 0; i<n; i++)
      for (j = 0; j<i+1; j++)
        {
          x = a[i][j];
          if (i==j)
            {
              for (k = j-1; k>=0; k--)
                {
                  y = l[i][k];
                  z = (l[i][k] = y*d[k]);
                  x -= y*z;
                }
              if (x==ZERO) return false;
              d[i] = ONE/x;
            }
          else
            {
              for (k = j-1; k>=0; k--) x -= l[i][k]*l[j][k];
              l[i][j]  = x;
            }
        }
    for (k = 0; k<n; k++) l[k][k] = ONE;
    return true;
  }
 
/*
   CholSol(l,p,b,x,n) forward/backward substitution
     l: triangular matrix
     p: diagonal elements (d from chol_dec)
     b: rightside of ax=b
     x: solution
     n: dimension
 
Ref.: Wilkinson/Reinsch, p.22, routine symsol
*/
 
int CholSol(l, p, b, x, n)
  REAL **l, *p, *b, *x;
  int n;
  {
    int i, k;
    REAL y;
 
    Set0VecFull(x, n);
    for (i = 0; i<n; i++)      /* Forward substitution */
      {
        y = b[i];
        for (k = i-1; k>=0; k--) y -= l[i][k]*x[k];
        x[i] = y;
      }
    for (i = n-1; i>=0; i--)   /* Backward substitution */
      {
        y = x[i]*p[i];
        for (k = i+1; k<n; k++) y -= l[k][i]*x[k];
        x[i] = y;
      }
    return true;
  }
 
int ChkSol(a, b, x, d, n, verboseP)
  REAL **a, *b, *x, *d;
  int n, verboseP;
  {
    int i, k;
    REAL s, amax;
    REAL zmin = RMAX, zmax = RMIN;

    amax = ZERO;
    for (i = 0; i<n; i++)
      {
        s = ZERO;
        for (k = 0; k<n; k++) s += ((i<k)?a[k][i]:a[i][k])*x[k];
        if (fabs(s-b[i])>amax) amax = fabs(s-b[i]);
      }
/*
    sprintf(globBuf, "Solve: maximal deviation %e (ChkSol)\n",
			amax);
	ZIBStdOut(globBuf);
*/
    for (k = 0; k<n; k++)
      {
        if (d[k]<zmin) zmin = d[k];
        if (d[k]>zmax) zmax = d[k];
      }
    if (verboseP)
	  {
		sprintf(globBuf,"Solve: min/max(diag) %9.3e/%9.3e (ChkSol)\n",
				zmin, zmax);
		ZIBStdOut(globBuf);
	  }
    return zmin>=ZERO;
  }
