      SUBROUTINE NLEQ1S(N,NFMAX,FCN,JAC,X,XSCAL,RTOL,IOPT,IERR,
     $LIWK,IWK,LI2WK,I2WK,LRWK,RWK)
C*    Begin Prologue NLEQ1S
      INTEGER N,NFMAX
      EXTERNAL FCN,JAC
      DOUBLE PRECISION X(N),XSCAL(N)
      DOUBLE PRECISION RTOL
      INTEGER IOPT(50)
      INTEGER IERR
      INTEGER LIWK
      INTEGER IWK(LIWK)
      INTEGER LI2WK
      INTEGER I2WK(LI2WK)
      INTEGER LRWK
      DOUBLE PRECISION RWK(LRWK)
C     ------------------------------------------------------------
C
C*  Title
C
C     Numerical solution of systems of nonlinear (NL) equations (EQ)
C     with a sparse (S) Jacobian -
C     especially designed for numerically sensitive problems.
C
C*  Written by        U. Nowak, L. Weimann 
C*  Purpose           Solution of systems of highly nonlinear equations
C                     with a sparse Jacobian matrix
C*  Method            Damped affine invariant Newton method
C                     (see references below)
C*  Category          F2a. - Systems of nonlinear equations
C*  Keywords          Nonlinear equations, Newton methods,
C                     sparse Jacobian
C*  Version           2.3
C*  Revision          September 1991
C*  Latest Change     June 1992
C*  Library           CodeLib
C*  Code              Fortran 77, Double Precision
C*  Environment       Standard Fortran 77 environment on PC's,
C                     workstations and hosts.
C*  Copyright     (c) Konrad Zuse Zentrum fuer
C                     Informationstechnik Berlin
C                     Heilbronner Str. 10, D-1000 Berlin 31
C                     phone 0049+30+89604-0, 
C                     telefax 0049+30+89604-125
C*  Contact           Lutz Weimann 
C                     ZIB, Numerical Software Development 
C                     phone: 0049+30+89604-185 ;
C                     e-mail:
C                     RFC822 notation: weimann@sc.zib-berlin.de
C                     X.400: C=de;A=dbp;P=zib-berlin;OU=sc;S=Weimann
C
C*    References:
C
C     /1/ P. Deuflhard:
C         Newton Techniques for Highly Nonlinear Problems -
C         Theory and Algorithms.
C         Academic press Inc. (To be published)
C
C     /2/ U. Nowak, L. Weimann:
C         A Family of Newton Codes for Systems of Highly Nonlinear
C         Equations - Algorithm, Implementation, Application.
C         ZIB, Technical Report TR 90-10 (December 1990)
C
C  ---------------------------------------------------------------
C
C* Licence
C    You may use or modify this code for your own non commercial
C    purposes for an unlimited time. 
C    In any case you should not deliver this code without a special 
C    permission of ZIB.
C    In case you intend to use the code commercially, we oblige you
C    to sign an according licence agreement with ZIB.
C
C* Warranty 
C    This code has been tested up to a certain level. Defects and
C    weaknesses, which may be included in the code, do not establish
C    any warranties by ZIB. ZIB does not take over any liabilities
C    which may follow from acquisition or application of this code.
C
C* Software status 
C    This code is under care of ZIB and belongs to ZIB software class 1.
C
C     ------------------------------------------------------------
C
C*    Summary:
C     ========
C     Damped Newton-algorithm for systems of highly nonlinear
C     equations - damping strategy due to Ref. (1).
C
C     (The iteration is done by subroutine NIINT currently. NLEQ1S
C      itself does some house keeping and builds up workspace.)
C
C     Jacobian computation by user supplied subroutine JAC.
C
C     The numerical solution of the arising linear equations is
C     done by means of the MA28 Harwell subroutines package for
C     the solution of sparse linear systems.
C     For special purposes these routines may be substituted.
C
C     This is a driver routine for the core solver NIINT.
C
C     ------------------------------------------------------------
C
C*    Parameters list description (* marks inout parameters)
C     ======================================================
C
C*    External subroutines (to be supplied by the user)
C     =================================================
C 
C     (Caution: Arguments declared as (input) must not
C               be altered by the user subroutines ! )
C
C     FCN(N,X,F,IFAIL) Ext    Function subroutine
C       N              Int    Number of vector components (input)
C       X(N)           Dble   Vector of unknowns (input)
C       F(N)           Dble   Vector of function values (output)
C       IFAIL          Int    FCN evaluation-failure indicator. (output)
C                             On input:  Has always value 0 (zero).
C                             On output: Indicates failure of FCN eval-
C                                uation, if having a value <= 2.
C                             If <0: NLEQ1S will be terminated with 
C                                    error code = 82, and IFAIL stored
C                                    to IWK(23).
C                             If =1: A new trial Newton iterate will
C                                    computed, with the damping factor
C                                    reduced to it's half.
C                             If =2: A new trial Newton iterate will
C                                    computed, with the damping factor
C                                    reduced by a reduct. factor, which
C                                    must be output through F(1) by FCN,
C                                    and it's value must be >0 and < 1.
C                             Note, that if IFAIL = 1 or 2, additional
C                             conditions concerning the damping factor,
C                             e.g. the minimum damping factor or the
C                             bounded damping strategy may also influ-
C                             ence the value of the reduced damping 
C                             factor.
C      
C     
C     JAC(N,X,DFDX,IROW,ICOL,NFILL,IFAIL)
C                    Ext    Jacobian matrix subroutine (input)
C       N            Int    Number of vector components (input)
C       X(N)         Dble   Vector of unknowns (input)
C       DFDX(NFILL)  Dble   DFDX(k) must get the partial derivative
C                           of IROW(k)-th component of FCN with 
C                           respect to X(ICOL(k)) (output)
C                           (see also parameters IROW and ICOL)
C       IROW(NFILL)  Int    IROW(k) must get the row index of
C                           the Jacobian matrix element DFDX(k) 
C                           (output)
C       ICOL(NFILL)  Int    ICOL(k) must get the column index of
C                           the Jacobian matrix element DFDX(k)
C                           (output)
C     * NFILL        Int    On input, it holds the length NFMAX
C                           of the arrays DFDX,IROW,ICOL .
C                           On output, it must return the number
C                           of currently used or needed elements
C                           of DFDX to hold the Jacobian matrix.
C                           If the output value becomes greater
C                           than the input value, NLEQ1S
C                           terminates with error code 11 .
C       IFAIL        Int    JAC evaluation-failure indicator. (output)
C                           Has always value 0 (zero) on input.
C                           Indicates failure of JAC evaluation
C                           and causes termination of NLEQ1S,
C                           if set to a negative value on output
C
C*    Input parameters of NLEQ1S
C     ==========================
C
C     N              Int    Number of unknowns
C     NFMAX          Int    Maximum number of nonzero elements
C                           of the Jacobian matrix
C   * X(N)           Dble   Initial estimate of the solution
C   * XSCAL(N)       Dble   User scaling (lower threshold) of the 
C                           iteration vector X(N)
C   * RTOL           Dble   Required relative precision of
C                           solution components -
C                           RTOL.GE.EPMACH*TEN*N
C   * IOPT(50)       Int    Array of run-time options. Set to zero
C                           to get default values (details see below)
C
C*    Output parameters of NLEQ1S
C     ===========================
C
C   * X(N)           Dble   Solution values ( or final values,
C                           respectively )
C   * XSCAL(N)       Dble   After return with IERR.GE.0, it contains
C                           the latest internal scaling vector used
C                           After return with IERR.EQ.-1 in onestep-
C                           mode it contains a possibly adapted 
C                           (as described below) user scaling vector:
C                           If (XSCAL(I).LT. SMALL) XSCAL(I) = SMALL ,
C                           If (XSCAL(I).GT. GREAT) XSCAL(I) = GREAT .
C                           For SMALL and GREAT, see section machine
C                           constants below  and regard note 1.
C   * RTOL           Dble   Finally achieved (relative) accuracy
C                           The estimated absolute error of component i
C                           of x_out is approximately given by
C                             abs_err(i) = RTOL * XSCAL_out(i) ,
C                           where (approximately)
C                             XSCAL_out(i) = 
C                                max(abs(X_out(i)),XSCAL_in(i)).
C     IERR           Int    Return value parameter
C                           =-1 sucessfull completion of one iteration
C                               step, subsequent iterations are needed 
C                               to get a solution. (stepwise mode only) 
C                           = 0 successfull completion of iteration
C                           > 0 see list of error messages below
C
C     Note 1.
C        The machine dependent values SMALL, GREAT and EPMACH are
C        gained from calls of the machine constants function D1MACH.
C        As delivered, this function is adapted to use constants 
C        suitable for all machines with IEEE arithmetic. If you use
C        another type of machine, you have to change the DATA state-
C        ments for IEEE arithmetic in D1MACH into comments and to 
C        uncomment the set of DATA statements suitable for your machine.
C
C*    Workspace parameters of NLEQ1S
C     ==============================
C
C     LIWK           Int    Declared dimension of integer
C                           workspace.
C                           Required minimum (for standard linear system
C                           solver) : 8*N+57
C   * IWK(LIWK)      Int    Integer Workspace
C     LI2WK          Int    Declared dimension of Short Integer 
C                           workspace.
C                           Required minimum: Int(7.5*NFMAX)+5*N
C   * I2WK(LI2WK)    ShInt  Short Integer Workspace
C                           In the current implementation the same 
C                           integer type as this of IWK (INTEGER*4).
C     LRWK           Int    Declared dimension of real workspace.
C                           Required minimum (for standard linear system
C                           solver): 3*NFMAX+(11+NBROY)*N+62 
C                           NBROY = Maximum number of Broyden steps
C                           (Default: if Broyden steps are enabled, e.g.
C                                                IOPT(32)=1            -
C                                       NBROY = MAX( INT(NFMAX/N), 10 ),
C                                     else (if IOPT(32)=0) - 
C                                       NBROY=0 ;
C                            see equally named IWK-field below)
C   * RWK(LRWK)      Dble   Real Workspace
C
C     Note 2a.  A test on sufficient workspace is made. If this
C               test fails, IERR is set to 10 and an error-message
C               is issued from which the minimum of required
C               workspace size can be obtained.
C               The actually required workspace strongly depends on the
C               number of fill-in elements of the LU-decomposition.
C               As long as the first decomposition fails due to lack
C               of workspace, the output values IWK(15), IWK(18),
C               IWK(19) (minimum required workspace) are just estimates.
C               Note that the option QFIXPT = IOPT(37) affects the
C               minimum required workspace.
C
C     Note 2b.  The first 50 elements of IWK and RWK are partially 
C               used as input for internal algorithm parameters (for
C               details, see below). In order to set the default values
C               of these parameters, the fields must be set to zero.
C               Therefore, it's recommanded always to initialize the
C               first 50 elements of both workspaces to zero.
C
C*   Options IOPT:
C    =============
C
C     Pos. Name   Default  Meaning
C
C       1  QSUCC  0        =0 (.FALSE.) initial call:
C                             NLEQ1S is not yet initialized, i.e. this is
C                             the first call for this nonlinear system.
C                             At successfull return with MODE=1,
C                             QSUCC is set to 1.
C                          =1 (.TRUE.) successive call:
C                             NLEQ1S is initialized already and is now
C                             called to perform one or more following
C                             Newton-iteration steps.
C                             ATTENTION:
C                                Don't destroy the contents of
C                                IOPT(i) for 1 <= i <= 50 ,
C                                IWK(j)  for 1 <= j < NIWKFR,
C                                I2WK(k) for 1 <= k < NI2WKF and
C                                RWK(l)  for 1 <= l < NRWKFR.
C                                (Nevertheless, some of the options, e.g.
C                                 FCMIN, SIGMA, MPR..., can be modified
C                                 before successive calls.)
C       2  MODE   0        =0 Standard mode initial call:
C                             Return when the required accuracy for the
C                             iteration vector is reached. User defined
C                             parameters are evaluated and checked.
C                             Standard mode successive call:
C                             If NLEQ1S was called previously with MODE=1,
C                             it performs all remaining iteration steps.
C                          =1 Stepwise mode:
C                             Return after one Newton iteration step.
C       3..8               Reserved
C       9  ISCAL  0        Determines how to scale the iterate-vector:
C                          =0 The user supplied scaling vector XSCAL is
C                             used as a (componentwise) lower threshold
C                             of the current scaling vector
C                          =1 The vector XSCAL is always used as the
C                             current scaling vector
C      10                  Reserved
C      11  MPRERR 0        Print error messages
C                          =0 No output
C                          =1 Error messages
C                          =2 Warnings additionally
C                          =3 Informal messages additionally
C      12  LUERR  6        Logical unit number for error messages
C      13  MPRMON 0        Print iteration Monitor
C                          =0 No output
C                          =1 Standard output
C                          =2 Summary iteration monitor additionally
C                          =3 Detailed iteration monitor additionally
C                          =4,5,6 Outputs with increasing level addi-
C                             tional increasing information for code
C                             testing purposes. Level 6 produces
C                             in general extremely large output!
C      14  LUMON  6        Logical unit number for iteration monitor
C      15  MPRSOL 0        Print solutions
C                          =0 No output
C                          =1 Initial values and solution values
C                          =2 Intermediate iterates additionally
C      16  LUSOL  6        Logical unit number for solutions
C      17  MPRLIN 0        Print linear solvers monitor
C                          = 0 or 1 : No output
C                          = 2 or 3: Linear solvers statistics after
C                                    each Jacobians LU decomposition
C                          = 4 Additional output for test purposes
C      18  LULIN  6        Logical unit number for sparse linear solvers
C                          monitor       
C      19  MPRTIM 0        Output level for the time monitor
C                          = 0 : no time measurement and no output
C                          = 1 : time measurement will be done and
C                                summary output will be written -
C                                regard note 4a.
C      20  LUTIM  6        Logical output unit for time monitor
C      21..30              Reserved
C      31  NONLIN 3        Problem type specification
C                          =1 Linear problem
C                             Warning: If specified, no check will be
C                             done, if the problem is really linear, and
C                             NLEQ1S terminates unconditionally after
C                             one Newton-iteration step.
C                          =2 Mildly nonlinear problem
C                          =3 Highly nonlinear problem
C                          =4 Extremely nonlinear problem
C      32  QRANK1 0        =0 (.FALSE.) Rank-1 updates by Broyden-
C                             approximation are inhibited.
C                          =1 (.TRUE.) Rank-1 updates by Broyden-
C                             approximation are allowed.
C      33..34              Reserved
C      35  QNSCAL 0        Inhibit automatic row scaling: 
C                          =0 (.FALSE.) Automatic row scaling of
C                             the linear system is activ: 
C                             Rows i=1,...,N will be divided by
C                             max j=1,...,N (abs(a(i,j))) 
C                          =1 (.TRUE.) No row scaling of the linear
C                             system. Recommended only for well row-
C                             scaled nonlinear systems.
C      36                  Reserved
C      37  QFIXPT 0        Fixed sparse pattern option:
C                          =0 (.FALSE.) The sparse pattern of the Jacobian
C                             may vary for different iterates
C                          =1 (.TRUE.) The sparse pattern of the Jacobian
C                             is fixed for all calls.
C                             Using this option saves 2 NFMAX units of
C                             the integer workspace I2WK and one compa-
C                             rison of the old against the new sparse 
C                             pattern for each iteration step.
C                             The comparison is done in order to have
C                             a chance of using the fast-factor routine
C                             MA28B instead of the analyze-factor
C                             routine MA28A.
C      38  IBDAMP          Bounded damping strategy switch:
C                          =0 The default switch takes place, dependent
C                             on the setting of NONLIN (=IOPT(31)):
C                             NONLIN = 0,1,2,3 -> IBDAMP = off ,
C                             NONLIN = 4 -> IBDAMP = on
C                          =1 means always IBDAMP = on 
C                          =2 means always IBDAMP = off 
C      39  IORMON          Convergence order monitor 
C                          =0 Standard option is IORMON=2 
C                          =1 Convergence order is not checked,
C                             the iteration will be always proceeded
C                             until the solution has the required 
C                             precision RTOL (or some error condition
C                             occured)
C                          =2 Use additional 'weak stop' criterion:
C                             Convergence order is monitored
C                             and termination due to slowdown of the
C                             convergence may occur.
C                          =3 Use additional 'hard stop' criterion:
C                             Convergence order is monitored
C                             and termination due to superlinear 
C                             convergence slowdown may occur. 
C                          In case of termination due to convergence
C                          slowdown, the warning code IERR=4 will be
C                          set.
C                          In cases, where the Newton iteration con-
C                          verges but superlinear convergence order has
C                          never been detected, the warning code IERR=5 
C                          is returned.
C      40..45              Reserved
C      46..50              User options (see note 4b)
C
C     Note 3:
C         If NLEQ1S terminates with IERR=2 (maximum iterations)
C         or  IERR=3 (small damping factor), you may try to continue
C         the iteration by increasing NITMAX or decreasing FCMIN
C         (see RWK) and setting QSUCC to 1.
C
C     Note 4a:
C        The integrated time monitor calls the machine dependent
C        subroutine SECOND to get the current time stamp in form
C        of a real number (Single precision). As delivered, this
C        subroutine always return 0.0 as time stamp value. Refer
C        to the compiler- or library manual of the FORTRAN compiler
C        which you currently use to find out how to get the current
C        time stamp on your machine.
C
C     Note 4b:
C         The user options may be interpreted by the user replacable
C         routines NISOUT, NIFACT, NISOLV - the distributed version
C         of NISOUT currently uses IOPT(46) as follows:
C         0 = standard plotdata output (may be postprocessed by a user-
C             written graphical program)
C         1 = plotdata output is suitable as input to the graphical
C             package GRAZIL (based on GKS), which has been developed
C             at ZIB. 
C
C
C*   Optional INTEGER input/output in IWK:
C    =======================================
C
C     Pos. Name          Meaning
C
C      1   NITER  IN/OUT Number of Newton-iterations
C      2                 reserved
C      3   NCORR  IN/OUT Number of corrector steps
C      4   NFCN   IN/OUT Number of FCN-evaluations
C      5   NJAC   IN/OUT Number of Jacobian generations or
C                        JAC-calls
C      6..8              Reserved
C      9   NREJR1 IN/OUT Number of rejected Newton iteration steps
C                        done with a rank-1 approximated Jacobian
C     10..11             Reserved
C     12   IDCODE IN/OUT Output: The 8 decimal digits program identi-
C                        fication number ppppvvvv, consisting of the
C                        program code pppp and the version code vvvv.
C                        Input: If containing a negative number,
C                        it will only be overwritten by the identi-
C                        fication number, immediately followed by
C                        a return to the calling program.      
C     13                 Reserved
C     14   NI2WKF OUT    First element of I2WK which is free to be used
C                        as workspace between Newton iteration steps
C                        for MA28 linear solver: 4*NFMAX+5*N
C     15   LI2WKA OUT    Length of I2WK currently required
C     16   NIWKFR OUT    First element of IWK which is free to be used
C                        as workspace between Newton iteration steps
C                        for MA28 linear solver: 56
C     17   NRWKFR OUT    First element of RWK which is free to be used
C                        as workspace between Newton iteration steps
C                        for MA28 linear solver: 2*NFMAX+(6+NBROY)*N+61
C     18   LIWKA  OUT    Length of IWK currently required
C     19   LRWKA  OUT    Length of RWK currently required
C     20..22             Reserved
C     23   IFAIL  OUT    Set in case of failure of NIFACT (IERR=80),
C                        N2SOLV (IERR=81), FCN (IERR=82) or JAC(IERR=83)
C                        to the nonzero IFAIL value returned by the 
C                        routine indicating the failure .
C     24..30             Reserved
C     31   NITMAX IN     Maximum number of permitted iteration
C                        steps (default: 50)
C     32                 Reserved
C     33   NEW    IN/OUT Count of consecutive rank-1 updates
C     34                 Reserved
C     35   NFILLA INTERN Number of nonzero elements of previous (in)
C                        or current (out) Jacobian
C     36   NBROY  IN     Maximum number of possible consecutive 
C                        iterative Broyden steps. The total real 
C                        workspace needed (RWK) depends on this value
C                        (see LRWK above).
C                        Default is max( NFMAX/N , 10 )
C                        (see parameters N and NFMAX) -
C                        provided that Broyden is allowed. 
C                        If Broyden is inhibited, NBROY is always set to
C                        zero.
C     37..45             Reserved
C     46   MINIRN  OUT   MA28: Minimum length of array IRN, for success
C                              on future runs
C     47   MINICN  OUT   MA28: Minimum length of array ICN, for success
C                              on future runs
C     48   IRNCP   OUT   MA28: Set to number of compresses on array IRN
C     49   ICNCP   OUT   MA28: Set to number of compresses on array ICN
C     50   IRANK   OUT   MA28: Estimated rank of matrix
C
C*   Optional REAL input/output in RWK:
C    ====================================
C
C     Pos. Name          Meaning
C
C      1..16             Reserved
C     17   CONV   OUT    The achieved relative accuracy after the  
C                        current step
C     18   SUMX   OUT    Natural level (not Normx of printouts)
C                        of the current iterate, i.e. Sum(DX(i)**2),
C                        where DX = scaled Newton correction.
C     19   DLEVF  OUT    Standard level (not Normf of printouts)
C                        of the current iterate, i.e. Norm2(F(X)),
C                        where F =  nonlinear problem function.
C     20   FCBND  IN     Bounded damping strategy restriction factor
C                        (Default is 10)
C     21   FCSTRT IN     Damping factor for first Newton iteration -
C                        overrides option NONLIN, if set (see note 5)
C     22   FCMIN  IN     Minimal allowed damping factor (see note 5)
C     23   SIGMA  IN     Broyden-approximation decision parameter
C                        Required choice: SIGMA.GE.1. Increasing this
C                        parameter make it less probable that the algo-
C                        rith performs  rank-1 updates.
C                        Rank1 updates are inhibited, if 
C                        SIGMA.GT.1/FCMIN is set. (see note 5)
C     24   SIGMA2 IN     Decision parameter about increasing damping
C                        factor to corrector if predictor is small.
C                        Required choice: SIGMA2.GE.1. Increasing this
C                        parameter make it less probable that the algo-
C                        rith performs rank-1 updates.
C     25..50             Reserved
C
C     Note 5:
C       The default values of the internal parameters may be obtained
C       from the monitor output with at least IOPT field MPRMON set to 2
C       and by initializing the corresponding RWK-fields to zero. 
C
C*   Error and warning messages:
C    ===========================
C
C      1    Termination, since jacobian matrix became singular
C      2    Termination after NITMAX iterations ( as indicated by
C           input parameter NITMAX=IWK(31) )
C      3    Termination, since damping factor became to small
C      4    Warning: Superlinear or quadratic convergence slowed down
C           near the solution.
C           Iteration has been stopped therefore with an approximation
C           of the solution not such accurate as requested by RTOL,
C           because possibly the RTOL requirement may be too stringent
C           (i.e. the nonlinear problem is ill-conditioned)
C      5    Warning: Iteration stopped with termination criterion 
C           (using RTOL as requested precision) satisfied, but no 
C           superlinear or quadratic convergence has been indicated yet.
C           Therefore, possibly the error estimate for the solution may
C           not match good enough the really achieved accuracy.
C     10    Integer or real workspace too small
C     11    Insufficient storage for nonzero Jacobian matrix
C           elements - possibly NFMAX too small.
C           The number of needed nonzero elements NFILL as returned
C           by user routine JAC is stored to IWK(35) (NFILLA-field).
C     12    Integer or real workspace are far too small, minimum
C           estimates given do not regard additional amounts of
C           workspace needed by sparse linear solver. 
C     20    Bad input to one or both of the  dimensional parameters
C           N and NFMAX
C     21    Nonpositive value for RTOL supplied
C     22    Negative scaling value via vector XSCAL supplied
C     30    One or more fields specified in IOPT are invalid
C           (for more information, see error-printout)
C     80    Error signalled by linear solver routine NIFACT,
C           for more detailed information see IFAIL-value
C           stored to IWK(23)
C     81    Error signalled by linear solver routine NISOLV,
C           for more detailed information see IFAIL-value
C           stored to IWK(23)
C           (not used by standard routine NISOLV)
C     82    Error signalled by user routine FCN (Nonzero value
C           returned via IFAIL-flag; stored to IWK(23) )
C     83    Error signalled by user routine JAC (Nonzero value
C           returned via IFAIL-flag; stored to IWK(23) )
C
C     Note 6 : in case of failure:
C        -    use non-standard options
C        -    use another initial guess
C        -    or reformulate model
C        -    or apply continuation techniques (soon available
C             for sparse Jacobian problems)
C
C*    Machine dependent constants used:
C     =================================
C
C     DOUBLE PRECISION EPMACH  in  NIPCHK, NIINT
C     DOUBLE PRECISION GREAT   in  NIPCHK
C     DOUBLE PRECISION SMALL   in  NIPCHK, NIINT, NISCAL
C
C*    Subroutines called: NIPCHK, NIINT, NISLVI
C
C     ------------------------------------------------------------
C*    End Prologue
C
C*    Summary of changes:
C     ===================
C      
C     2.2.1  91, June  3    Time monitor included
C     2.2.2  91, June  3    Bounded damping strategy implemented
C     2.2.3  91, July 26    FCN-count changed for anal. Jacobian,
C                           RWK structured compatible to other 2.2.3
C                           version codes,
C            91, July 31    output MINIRN, MINICN, IRNCP, ICNCP,
C                           IRANK (Common MA28FD) to IWK(46..50)
C                           New options: MPRLIN, LULIN (IOPT(17..18))
C     2.2.4  91, August 14  Convergence order monitor included
C     2.2.5  91, August 15  Iterative Broyden included
C     2.2.6  91, Sept.  17  Damping factor reduction by FCN-fail imple-
C                           mented
C     2.3    91, Dec.   20  New Release for CodeLib
C   
C     ------------------------------------------------------------
C
C     PARAMETER (IRWKI=xx, LRWKI=yy)  
C     IRWKI: Start position of internally used RWK part
C     LRWKI: Length of internally used RWK part
C     (current values see parameter statement below)
C
C     INTEGER L4,L5,L51,L6,L61,L62,L63,L7,L71,L8,L9,L10,L11,L12,L121,
C             L13,L14,L20
C     Starting positions in RWK of formal array parameters of internal
C     routine N1INT (dynamically determined in driver routine NLEQ1,
C     dependent on N and options setting)
C
C     Further RWK positions (only internally used)
C
C     Position  Name     Meaning
C
C     IRWKI     FCKEEP   Damping factor of previous successfull iter.
C     IRWKI+1   FCA      Previous damping factor
C     IRWKI+2   FCPRI    A priori estimate of damping factor
C     IRWKI+3   DMYCOR   Number My of latest corrector damping factor
C                        (kept for use in rank-1 decision criterium)
C     IRWKI+(4..LRWKI-1) Free
C
C     Internal arrays stored in RWK (see routine NIINT for descriptions)
C
C     Position  Array         Type   Remarks
C
C     L4        A(NFMAX)      Perm
C     L41       DXSAVE(N,NBROY)
C                             Perm   NBROY=IWK(36) (Default: N or 0)
C     L5        DX(N)         Perm  
C     L51       DXQ(N)        Perm 
C     L6        XA(N)         Perm
C     L61       F(N)          Perm
C     L62       FW(N)         Perm
C     L63       XWA(N)        Perm
C     L7        FA(N)         Perm
C     L8                      low: Perm   Start position of array work-
C                             high: Temp  space needed for linear solver
C                               (See NISLVI for details on "low" and 
C                                "high") 
C     L10       XW(N)         Temp
C     L12       DXQA(N)       Temp
C     L121      T1(N)         Temp
C
C     Internal arrays stored in I2WK
C
C     L30       IROWA(NFMAX)  Perm   only used if QFIXPT=IOPT(37)=0
C     L31       ICOLA(NFMAX)  Perm   only used if QFIXPT=IOPT(37)=0
C     L33       IROW(NFMAX)   Perm
C     L34       ICOL(NFMAX)   Perm
C     L35,L36                 low: Perm, high: Temp   
C                                    Start position of integer worksp. 
C                                    needed for linear solver
C                                    (L35 for QFIXPT=1,L36 for QFIXPT=0)
C
      EXTERNAL NIINT, NISLVI
      INTRINSIC DBLE
      INTEGER IRWKI, LRWKI
      PARAMETER (IRWKI=51, LRWKI=10)  
      DOUBLE PRECISION ONE
      PARAMETER (ONE=1.0D0)
      DOUBLE PRECISION TEN
      PARAMETER (TEN=1.0D1)
      DOUBLE PRECISION ZERO
      PARAMETER (ZERO=0.0D0)
      INTEGER NITMAX,LUERR,LUMON,LUSOL,MPRERR,MPRMON,MPRSOL,
     $NRWKFR,NRW,NIWKFR,NIW,NI2WKF,NI2W,NONLIN
      INTEGER L4,L41,L5,L51,L6,L61,L62,L63,L7,L8,L9,L10,L12,L121,
     $L20,L30,L31,L33,L34,L35,L36
      DOUBLE PRECISION FC,FCMIN,PERCI,PERCR
      LOGICAL QINIMO,QRANK1,QFCSTR,QSUCC,QFIXPT,QBDAMP
      CHARACTER CHGDAT*20, PRODCT*8
C     Which version ?
      LOGICAL QVCHK
      INTEGER IVER
      PARAMETER( IVER=21512300 )
C
C     Version: 2.3               Latest change:
C     -----------------------------------------
C
      DATA      CHGDAT      /'December 20, 1991   '/
      DATA      PRODCT      /'NLEQ1S'/
C*    Begin
      IERR = 0
      QVCHK = IWK(12).LT.0
      IWK(12) = IVER
      IF (QVCHK) RETURN
C        Print error messages?
      MPRERR = IOPT(11)
      LUERR = IOPT(12)
      IF (LUERR .EQ. 0) THEN
        LUERR = 6
        IOPT(12)=LUERR
      ENDIF
C        Print iteration monitor?
      MPRMON = IOPT(13)
      LUMON = IOPT(14)
      IF (LUMON .LE. 0 .OR. LUMON .GT. 99) THEN
        LUMON = 6
        IOPT(14)=LUMON
      ENDIF
C        Print intermediate solutions?
      MPRSOL = IOPT(15)
      LUSOL = IOPT(16)
      IF (LUSOL .EQ. 0) THEN
        LUSOL = 6
        IOPT(16)=LUSOL
      ENDIF
C        Print linear solvers monitor?
      MPRLIN = IOPT(17)
      LULIN = IOPT(18)
      IF (LULIN .EQ. 0) THEN
        LULIN = 6
        IOPT(18)=LULIN
      ENDIF
C        Print time summary statistics?
      MPRTIM = IOPT(19)
      LUTIM = IOPT(20)
      IF (LUTIM .EQ. 0) THEN
        LUTIM = 6
        IOPT(20)=LUTIM
      ENDIF
      QSUCC = IOPT(1).EQ.1
      QINIMO = MPRMON.GE.1.AND..NOT.QSUCC
C     Print NLEQ1S heading lines
      IF(QINIMO)THEN
10000   FORMAT('    N L E Q 1 S *****  V e r s i o n  ',
     $         '2 . 3 ***',//,1X,'Newton-Method ',
     $         'for the solution of nonlinear systems',//)
        WRITE(LUMON,10000)
      ENDIF
C     Check input parameters and options
      CALL NIPCHK(N,NFMAX,X,XSCAL,RTOL,IOPT,IERR,LIWK,IWK,LRWK,RWK)
C     Exit, if any parameter error was detected till here
      IF (IERR.NE.0) RETURN 
      QFIXPT = IOPT(37).EQ.1
      QRANK1=IOPT(32).EQ.1
      IF (QRANK1) THEN
        NBROY=IWK(36)
        IF (NBROY.EQ.0) NBROY=MAX(INT(NFMAX/N),10)
        IWK(36)=NBROY
      ELSE
        NBROY=0
      ENDIF
C     WorkSpace: RWK
      L4=IRWKI+LRWKI
      L41=L4+NFMAX
      L5=L41+NBROY*N
      L51=L5+N
      L6=L51+N
      L61=L6+N
      L62=L61+N
      L63=L62+N
      L7=L63+N
      L8=L7+N
      NRWKFR = L8
      L9=LRWK+1
      L10=L9-N
      L12=L10-N
      L121=L12-N
      LRWKT=LRWK+1-L121
C     End WorkSpace at NRW
C     WorkSpace: IWK
      L20=51
      NIWKFR = L20
      LIWKT = 0
C     End WorkSpace at NIW
C     WorkSpace: I2WK
      IF (QFIXPT) THEN
        L33=1
        L34=L33+NFMAX
        L35=L34+NFMAX
        NI2WKF=L35
        LI2WKT=0
      ELSE
        L30=1
        L31=L30+NFMAX
        L33=L31+NFMAX
        L34=L33+NFMAX
        L36=L34+NFMAX
        NI2WKF=L36
        LI2WKT=0
      ENDIF
C     End WorkSpace at NI2W
      NIWLA = NIWKFR
      NI2WLA = NI2WKF
      NRWLA = NRWKFR
      IFAIL=0
      IF(NRWLA.GT.LRWK.OR.NIWLA.GT.LIWK.OR.NI2WLA.GT.LI2WK) THEN
        NI2W = NI2WKF+LI2WKT-1
        NIW = NIWKFR+LIWKT-1
        NRW = NRWKFR+LRWKT-1
        IFAIL=12
      ENDIF
      LIWL = LIWK-LIWKT-NIWLA+1
      LI2WL = LI2WK-LI2WKT-NI2WLA+1
      LRWL = LRWK-LRWKT-NRWLA+1
      IF (IFAIL.EQ.0) THEN
        CALL NISLVI(N,NFMAX,IOPT,IFAIL,LIWL,IWK(NIWLA),NIWKFR,NILUSE,
     $              LI2WL,I2WK(NI2WLA),NI2WKF,NI2LUS,LRWL,RWK(NRWLA),
     $              NRWKFR,NRLUSE)
        NIWKFR = NIWLA+NIWKFR
        LIWKT = LIWKT + NILUSE
        NI2WKF = NI2WLA+NI2WKF
        LI2WKT = LI2WKT + NI2LUS
        NRWKFR = NRWLA+NRWKFR
        LRWKT = LRWKT + NRLUSE
C
C       Store lengths of currently required workspaces
        NI2W = NI2WKF+LI2WKT-1
        NIW = NIWKFR+LIWKT-1
        NRW = NRWKFR+LRWKT-1
        IWK(15) = NI2W
      ENDIF
      IWK(18) = NIWKFR-1
      IWK(19) = NRWKFR-1
C     Free workspaces, not used between steps
      IWK(14) = NI2WKF
      IWK(16) = NIWKFR
      IWK(17) = NRWKFR
C
      IF(NRW.GT.LRWK.OR.NIW.GT.LIWK.OR.NI2W.GT.LI2WK.OR.IFAIL.NE.0)THEN
        IERR=10
        IF (IFAIL.NE.0) IERR=IFAIL
      ELSE
        IF(QINIMO)THEN
          PERCR  = DBLE(NRW) / DBLE(LRWK) * 100.0D0
          PERCI  = DBLE(NIW) / DBLE(LIWK) * 100.0D0
          PERCI2 = DBLE(NI2W)/ DBLE(LI2WK)* 100.0D0
C         Print statistics concerning workspace usage
10050     FORMAT(' Real    Workspace declared as ',I9,
     $    ' is used up to ',I9,' (',F5.1,' percent)',//,
     $    ' Integer Workspace declared as ',I9,
     $    ' is used up to ',I9,' (',F5.1,' percent)',//,
     $    ' Short integer Workspace declared as ',I9,
     $    ' is used up to ',I9,' (',F5.1,' percent)',//)
          WRITE(LUMON,10050)LRWK,NRW,PERCR,LIWK,NIW,PERCI,
     $                      LI2WK,NI2W,PERCI2
        ENDIF
        IF(QINIMO)THEN
10051     FORMAT(/,' N =',I4,/,
     $    ' Maximum number of nonzero Jacobian elements : ',I5,//, 
     $    ' Prescribed relative precision',D10.2,/)
          WRITE(LUMON,10051)N,NFMAX,RTOL
10057     FORMAT(' Automatic row scaling of the Jacobian is ',A,/)
          IF (IOPT(35).EQ.1) THEN
            WRITE(LUMON,10057) 'inhibited'
          ELSE
            WRITE(LUMON,10057) 'allowed'
          ENDIF
        ENDIF
        NONLIN=IOPT(31)
        IF (IOPT(38).EQ.0) QBDAMP = NONLIN.EQ.4
        IF (IOPT(38).EQ.1) QBDAMP = .TRUE.
        IF (IOPT(38).EQ.2) QBDAMP = .FALSE.
        IF (QBDAMP) THEN
          IF (RWK(20).LT.ONE) RWK(20) = TEN
        ENDIF
        IF (QINIMO) THEN
10064     FORMAT(' Rank-1 updates are ',A)
          IF (QRANK1) THEN
            WRITE(LUMON,10064) 'allowed'
          ELSE
            WRITE(LUMON,10064) 'inhibited'
          ENDIF
10065     FORMAT(' Problem is specified as being ',A)
          IF (NONLIN.EQ.1) THEN
            WRITE(LUMON,10065) 'linear'
          ELSE IF (NONLIN.EQ.2) THEN
            WRITE(LUMON,10065) 'mildly nonlinear'
          ELSE IF (NONLIN.EQ.3) THEN
            WRITE(LUMON,10065) 'highly nonlinear'
          ELSE IF (NONLIN.EQ.4) THEN
            WRITE(LUMON,10065) 'extremely nonlinear'
          ENDIF
10066     FORMAT(' Bounded damping strategy is ',A,:,/, 
     $           ' Bounding factor is ',D10.3)
          IF (QBDAMP) THEN
            WRITE(LUMON,10066) 'active', RWK(20)
          ELSE
            WRITE(LUMON,10066) 'off'
          ENDIF
        ENDIF
C       Maximum permitted number of iteration steps
        NITMAX=IWK(31)
        IF (NITMAX.LE.0) NITMAX=50
        IWK(31)=NITMAX
10068   FORMAT(' Maximum permitted number of iteration steps : ',
     $         I6)
        IF (QINIMO) WRITE(LUMON,10068) NITMAX
C       Initial damping factor for highly nonlinear problems
        QFCSTR=RWK(21).GT.ZERO
        IF (.NOT.QFCSTR) THEN
          RWK(21)=1.0D-2
          IF (NONLIN.EQ.4) RWK(21)=1.0D-4
        ENDIF
C       Minimal permitted damping factor
        IF (RWK(22).LE.ZERO) THEN
          RWK(22)=1.0D-4
          IF (NONLIN.EQ.4) RWK(22)=1.0D-8
        ENDIF
        FCMIN=RWK(22)
C       Rank1 decision parameter SIGMA
        IF (RWK(23).LT.ONE) RWK(23)=3.0D0
        IF (.NOT.QRANK1) RWK(23)=10.0D0/FCMIN
C       Decision parameter about increasing too small predictor
C       to greater corrector value
        IF (RWK(24).LT.ONE) RWK(24)=10.0D0/FCMIN       
C       Starting value of damping factor (FCMIN.LE.FC.LE.1.0)
        IF(NONLIN.LE.2.AND..NOT.QFCSTR)THEN
C         for linear or mildly nonlinear problems
          FC = ONE
        ELSE
C         for highly or extremely nonlinear problems
          FC = RWK(21)
        ENDIF
        RWK(21)=FC
        IF (MPRMON.GE.2.AND..NOT.QSUCC) THEN
10069     FORMAT(//,' Internal parameters:',//,
     $      ' Starting value for damping factor FCSTART = ',D9.2,/,
     $      ' Minimum allowed damping factor FCMIN = ',D9.2,/,
     $      ' Rank-1 updates decision parameter SIGMA = ',D9.2)
          WRITE(LUMON,10069) RWK(21),FCMIN,RWK(23)
        ENDIF
C
C       Initialize and start time measurements monitor
C
        IF ( IOPT(1).EQ.0 .AND. MPRTIM.NE.0 ) THEN
          CALL MONINI (' NLEQ1S',LUTIM)
          CALL MONDEF (0,'NLEQ1S')
          CALL MONDEF (1,'FCN')
          CALL MONDEF (2,'Jacobi')
          CALL MONDEF (3,'MA28AD')
          CALL MONDEF (4,'MA28BD')
          CALL MONDEF (5,'MA28CD')
          CALL MONDEF (6,'Output')
          CALL MONSRT ()
        ENDIF
C
C
        IERR=-1
C       If IERR is unmodified on exit, successive steps are required
C       to complete the Newton iteration
        IF (NBROY.EQ.0) NBROY=1
        CALL NIINT(N,NFMAX,FCN,JAC,X,XSCAL,RTOL,NITMAX,NONLIN,IOPT,IERR,
     $  LRWK,RWK,NRWLA,LRWL,LIWK,IWK,NIWLA,LIWL,LI2WK,I2WK,NI2WLA,
     $  LI2WL,NBROY,
     $  RWK(L4),RWK(L41),RWK(L5),RWK(L51),RWK(L6),RWK(L63),RWK(L61),
     $  RWK(L7),RWK(L10),RWK(L62),RWK(L12),RWK(L121),
     $  I2WK(L30),I2WK(L31),I2WK(L33),I2WK(L34),RWK(21),RWK(22),RWK(23),
     $  RWK(24),RWK(IRWKI+1),RWK(IRWKI),RWK(IRWKI+2),RWK(IRWKI+3),
     $  RWK(17),RWK(18),RWK(19),
     $  MPRERR,MPRMON,MPRSOL,LUERR,LUMON,LUSOL,
     $  IWK(1),IWK(3),IWK(4),IWK(5),IWK(9),IWK(33),IWK(35),QBDAMP)
C
        IF (MPRTIM.NE.0.AND.IERR.NE.-1.AND.IERR.NE.10) CALL MONEND
C
        IWK(18) = NIW
        IWK(19) = NRW
C       Sparse linear solvers output information:
        DO 20 I=0,4
          IWK(46+I) = IWK(NIWLA+I)
20      CONTINUE
C
      ENDIF
C     Print statistics
      IF (MPRMON.GE.1.AND.IERR.NE.-1.AND.IERR.NE.10.AND.IERR.NE.12) THEN
10080   FORMAT(/, '   ******  Statistics * ', A8, ' *******', /,
     $            '   ***  Newton iterations : ', I7,'  ***', /,
     $            '   ***  Corrector steps   : ', I7,'  ***', /,
     $            '   ***  Rejected rk-1 st. : ', I7,'  ***', /,
     $            '   ***  Jacobian eval.    : ', I7,'  ***', /,
     $            '   ***  Function eval.    : ', I7,'  ***', /,
     $            '   *************************************', /)
        WRITE (LUMON,10080) PRODCT,IWK(1),IWK(3),IWK(9),IWK(5),
     $  IWK(4)
      ENDIF
C     Print workspace requirements, if insufficient
      IF (IERR.EQ.10 .OR. IERR.EQ.12) THEN
10090   FORMAT(///,20('*'),'Workspace Error',20('*'))
        IF (MPRERR.GE.1) WRITE(LUERR,10090)
        IF(NRW.GT.LRWK)THEN
10091     FORMAT(/,' Real Workspace dimensioned as',1X,I9,
     $    1X,'must be enlarged at least up to ',
     $    I9,/)
          IF (MPRERR.GE.1) WRITE(LUERR,10091)LRWK,NRW
        ENDIF
        IF(NIW.GT.LIWK)THEN
10092     FORMAT(/,' Integer Workspace dimensioned as ',
     $    I9,' must be enlarged at least up ',
     $    'to ',I9,/)
          IF (MPRERR.GE.1) WRITE(LUERR,10092)LIWK,NIW
        ENDIF
        IF(NI2W.GT.LI2WK)THEN
10093     FORMAT(/,' Short integer Workspace dimensioned as ',
     $    I9,' must be enlarged at least up ',
     $    'to ',I9,/)
          IF (MPRERR.GE.1) WRITE(LUERR,10093)LI2WK,NI2W
        ENDIF
        IF (IERR.EQ.12 .AND. MPRERR.GE.1) WRITE(LUERR,10094)
10094   FORMAT(' The above given amount does not include additional',/,
     $         ' workspace needed for the sparse linear solver !',/)
      ENDIF
C     End of subroutine NLEQ1S
      RETURN
      END
C
C
      SUBROUTINE NIPCHK(N,NFMAX,X,XSCAL,RTOL,IOPT,IERR,LIWK,IWK,LRWK,
     $                  RWK)
C*    Begin Prologue NIPCHK
      INTEGER N,NFMAX
      DOUBLE PRECISION X(N),XSCAL(N)
      DOUBLE PRECISION RTOL
      INTEGER IOPT(50)
      INTEGER IERR
      INTEGER LIWK
      INTEGER IWK(LIWK)
      INTEGER LRWK
      DOUBLE PRECISION RWK(LRWK)
C     ------------------------------------------------------------
C
C*    Summary :
C
C     N I P C H K : Checking of input parameters and options
C                   for NLEQ1S.
C
C*    Parameters:
C     ===========
C
C     See parameter description in driver routine.
C
C*    Subroutines called: D1MACH
C
C*    Machine dependent constants used:
C     =================================
C
C     EPMACH = relative machine precision
C     GREAT = squareroot of maxreal divided by 10
C     SMALL = squareroot of "smallest positive machine number
C             divided by relative machine precision"
      DOUBLE PRECISION EPMACH,GREAT,SMALL
C
C     ------------------------------------------------------------
C*    End Prologue
C
      EXTERNAL D1MACH
      INTRINSIC DBLE
      DOUBLE PRECISION ONE
      PARAMETER (ONE=1.0D0)
      DOUBLE PRECISION TEN
      PARAMETER (TEN=1.0D1)
      DOUBLE PRECISION ZERO
      PARAMETER (ZERO=0.0D0)
C
      PARAMETER (NUMOPT=50)
      INTEGER IOPTL(NUMOPT),IOPTU(NUMOPT)
      DOUBLE PRECISION D1MACH,TOLMIN,TOLMAX,DEFSCL
C
      DATA IOPTL /0,0,0,0,0,0,0,0,0,0,0,1,0,1,0,1,0,1,0,1,
     $            0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
     $            0,0,0,0,0,0,0,0,0,0,
     $            -9999999,-9999999,-9999999,-9999999,-9999999/
      DATA IOPTU /1,1,0,0,0,0,0,0,1,0,3,99,6,99,3,99,4,99,1,99,
     $            0,0,0,0,0,0,0,0,0,0,4,1,0,0,1,
     $            0,1,2,3,0,0,0,0,0,0,
     $            9999999,9999999,9999999,9999999,9999999/
C
      EPMACH = D1MACH(3)
      GREAT  = DSQRT(D1MACH(2)/TEN)
      SMALL  = D1MACH(6)
      IERR = 0
C        Print error messages?
      MPRERR = IOPT(11)
      LUERR = IOPT(12)
      IF (LUERR .LE. 0 .OR. LUERR .GT. 99) THEN
        LUERR = 6
        IOPT(12)=LUERR
      ENDIF
C
C     Checking dimensional parameters N and NFMAX
      IF ( N.LE.0 .OR. NFMAX.LE.0 ) THEN
        IF (MPRERR.GE.1)  WRITE(LUERR,10011) N,NFMAX
10011   FORMAT(/,' Error: Bad input to dimensional parameters supplied',
     $         /,8X,'choose N and NFMAX positive',/,
     $         8X,'your input is: N = ',I5,' NFMAX = ',I5)
        IERR = 20
      ENDIF
C
C     Problem type specification by user
      NONLIN=IOPT(31)
      IF (NONLIN.LE.0) NONLIN=3
      IOPT(31)=NONLIN
C
C     Checking and conditional adaption of the user-prescribed RTOL
      IF (RTOL.LE.ZERO) THEN
        IF (MPRERR.GE.1) 
     $      WRITE(LUERR,'(A)') '0Error: Nonpositive RTOL supplied'
        IERR = 21
      ELSE
        TOLMIN = EPMACH*TEN*DBLE(N)
        IF(RTOL.LT.TOLMIN) THEN
          RTOL = TOLMIN
          IF (MPRERR.GE.2) 
     $      WRITE(LUERR,10012) 'increased ','smallest',RTOL
        ENDIF
        TOLMAX = 1.0D-1
        IF(RTOL.GT.TOLMAX) THEN
          RTOL = TOLMAX
          IF (MPRERR.GE.2) 
     $      WRITE(LUERR,10012) 'decreased ','largest',RTOL
        ENDIF
10012   FORMAT(/,' Warning: User prescribed RTOL ',A,'to ',
     $         'reasonable ',A,' value RTOL = ',D11.2)
      ENDIF
C     
C     Test user prescribed accuracy and scaling on proper values
      IF (N.LE.0) RETURN 
      IF (NONLIN.GE.3) THEN
        DEFSCL = RTOL
      ELSE
        DEFSCL = ONE
      ENDIF
      DO 10 I=1,N
        IF (XSCAL(I).LT.ZERO) THEN
          IF (MPRERR.GE.1) THEN 
            WRITE(LUERR,10013) I
10013       FORMAT(/,' Error: Negative value in XSCAL(',I5,') supplied')
          ENDIF
          IERR = 22
        ENDIF
        IF (XSCAL(I).EQ.ZERO) XSCAL(I) = DEFSCL
        IF ( XSCAL(I).GT.ZERO .AND. XSCAL(I).LT.SMALL ) THEN
          IF (MPRERR.GE.2) THEN
            WRITE(LUERR,10014) I,XSCAL(I),SMALL
10014       FORMAT(/,' Warning: XSCAL(',I5,') = ',D9.2,' too small, ',
     $             'increased to',D9.2)
          ENDIF
          XSCAL(I) = SMALL
        ENDIF
        IF (XSCAL(I).GT.GREAT) THEN
          IF (MPRERR.GE.2) THEN
            WRITE(LUERR,10015) I,XSCAL(I),GREAT
10015       FORMAT(/,' Warning: XSCAL(',I5,') = ',D9.2,' too big, ',
     $             'decreased to',D9.2)
          ENDIF
          XSCAL(I) = GREAT
        ENDIF
10    CONTINUE
C     Checks options
      DO 20 I=1,30
        IF (IOPT(I).LT.IOPTL(I) .OR. IOPT(I).GT.IOPTU(I)) THEN
          IERR=30
          IF (MPRERR.GE.1) THEN
            WRITE(LUERR,20001) I,IOPT(I),IOPTL(I),IOPTU(I)
20001       FORMAT(' Invalid option specified: IOPT(',I2,')=',I12,';',
     $             /,3X,'range of permitted values is ',I8,' to ',I8)
          ENDIF
        ENDIF
20    CONTINUE
C     End of subroutine NIPCHK
      RETURN
      END
C
      SUBROUTINE NIINT(N,NFMAX,FCN,JAC,X,XSCAL,RTOL,NITMAX,NONLIN,IOPT,
     $IERR,LRWK,RWK,NRWLA,LNRWLA,LIWK,IWK,NIWLA,LNIWLA,LI2WK,I2WK,
     $NI2WLA,LNI2WL,NBROY,
     $A,DXSAVE,DX,DXQ,XA,XWA,F,FA,XW,FW,DXQA,T1,IROWA,ICOLA,IROW,
     $ICOL,FC,FCMIN,SIGMA,SIGMA2,
     $FCA,FCKEEP,FCPRI,DMYCOR,CONV,SUMX,DLEVF,MPRERR,MPRMON,MPRSOL,
     $LUERR,LUMON,LUSOL,NITER,NCORR,NFCN,NJAC,NREJR1,NEW,NFILLA,QBDAMP)
C*    Begin Prologue NIINT
      INTEGER N,NFMAX
      EXTERNAL FCN,JAC
      DOUBLE PRECISION X(N),XSCAL(N)
      DOUBLE PRECISION RTOL
      INTEGER NITMAX,NONLIN
      INTEGER IOPT(50)
      INTEGER IERR
      INTEGER LRWK
      DOUBLE PRECISION RWK(LRWK)
      INTEGER NRWLA,LNRWLA,LIWK
      INTEGER IWK(LIWK)
      INTEGER NIWLA,LNIWLA,LI2WK
      INTEGER I2WK(LI2WK)
      INTEGER NI2WLA,LNI2WL,NBROY
      DOUBLE PRECISION A(NFMAX),DXSAVE(N,NBROY)
      DOUBLE PRECISION DX(N),DXQ(N),XA(N),XWA(N),F(N),FA(N)
      DOUBLE PRECISION XW(N),FW(N),DXQA(N),T1(N)
      INTEGER IROWA(NFMAX),ICOLA(NFMAX),IROW(NFMAX),ICOL(NFMAX)
      DOUBLE PRECISION FC,FCMIN,SIGMA,SIGMA2,FCA,FCKEEP,CONV,SUMX,DLEVF,
     $                 FCPRI,DMYCOR
      INTEGER MPRERR,MPRMON,MPRSOL,LUERR,LUMON,LUSOL,NITER,
     $NCORR,NFCN,NJAC,NREJR1,NEW,NFILLA
      LOGICAL QBDAMP
C     ------------------------------------------------------------
C
C*    Summary :
C
C     N I I N T : Core routine for NLEQ1S .
C     Damped Newton-algorithm for systems of highly nonlinear
C     equations especially designed for numerically sensitive
C     problems.
C
C*    Parameters:
C     ===========
C
C       N,NFMAX,FCN,JAC,X,XSCAL,RTOL   
C                         See parameter description in driver routine
C
C       NITMAX      Int    Maximum number of allowed iterations
C       NONLIN      Int    Problem type specification
C                          (see IOPT-field NONLIN)
C       IOPT        Int    See parameter description in driver routine
C       IERR        Int    See parameter description in driver routine
C       LRWK        Int    Length of real workspace
C       RWK(LRWK)   Dble   Real workspace array
C       NRWLA       Int    Index of first element of RWK-partition
C                          for linear solver
C       LNRWLA      Int    Length of RWK-partition for linear solver
C       LIWK        Int    Length of integer workspace
C       IWK(LIWK)   Int    Integer workspace array
C       NIWLA       Int    Index of first element of IWK-partition
C                          for linear solver
C       LNIWLA      Int    Length of IWK-partition for linear solver
C       LI2WK       Int    Length of short integer workspace
C       I2WK(LI2WK) ShInt  Short integer workspace array
C       NI2WLA      Int    Index of first element of I2WK-partition
C                          for linear solver
C       LNI2WL      Int    Length of I2WK-partition for linear solver
C       NBROY       Int    Maximum number of possible consecutive
C                          iterative Broyden steps. (See IWK(36))
C       A(NFMAX)    Dble   Holds the Jacobian matrix (decomposed form
C                          after call of linear decomposition
C                          routine)
C       DXSAVE(X,NBROY)
C                   Dble   Used to save the quasi Newton corrections of
C                          all previously done consecutive Broyden
C                          steps.
C       DX(N)       Dble   Current Newton correction
C       DXQ(N)      Dble   Simplified Newton correction J(k-1)*X(k)
C       XA(N)       Dble   Previous Newton iterate
C       XWA(N)      Dble   Scaling factors used for latest decomposed
C                          Jacobian for column scaling - may differ
C                          from XW, if Broyden updates are performed
C       F(N)        Dble   Function (FCN) value of current iterate
C       FA(N)       Dble   Function (FCN) value of previous iterate
C       XW(N)       Dble   Scaling factors for iteration vector
C       FW(N)       Dble   Scaling factors for rows of the system
C       DXQA(N)     Dble   Previous Newton correction
C       T1(N)       Dble   Workspace for linear solvers and internal
C                          subroutines
C       IROWA(NFMAX) ShInt Holds the values of IROW(NFMAX) from the 
C                          previous call of routine JAC. Used to 
C                          check for changes of the Jacobian's sparse
C                          structure.
C       ICOLA(NFMAX) ShInt Holds the values of ICOL(NFMAX) from the 
C                          previous call of routine JAC.
C                          See also IROWA(NFMAX) .
C       IROW(NFMAX)  ShInt Holds row indices of Jacobian matrix as
C                          supplied by the routine JAC
C       ICOL(NFMAX)  ShInt Holds column indices of Jacobian matrix as
C                          supplied by the routine JAC.
C       FC          Dble   Current Newton iteration damping factor.
C       FCMIN       Dble   Minimum permitted damping factor. If
C                          FC becomes smaller than this value, one
C                          of the following may occur:
C                          a.    Recomputation of the Jacobian
C                                matrix by means of difference
C                                approximation (instead of Rank1
C                                update), if Rank1 - update
C                                previously was used
C                          b.    Fail exit otherwise
C       SIGMA2      Dble   Decision parameter for damping factor
C                          increasing to corrector value
C       FCA         Dble   Previous Newton iteration damping factor.
C       FCKEEP      Dble   Keeps the damping factor as it is at start
C                          of iteration step.
C       CONV        Dble   Scaled maximum norm of the Newton-
C                          correction. Passed to RWK-field on output.
C       SUMX        Dble   Square of the natural level (see equal-
C                          named IOPT-output field)
C       DLEVF       Dble   Square of the standard level (see equal-
C                          named IOPT-output field)
C       MPRERR,MPRMON,MPRSOL,LUERR,LUMON,LUSOL,
C       NITER,NCORR,NFCN,NJAC,NREJR1,NEW,NFILLA :
C                          See description of equal named IWK-fields
C                          in the driver subroutine
C       QBDAMP      Logic  Flag, that indicates, whether bounded damping
C                          strategy is active:
C                          .true.  = bounded damping strategy is active
C                          .false. = normal damping strategy is active
C
C*    Internal double variables
C     =========================
C
C       CONVA    Holds the previous value of CONV .
C       DMUE     Temporary value used during computation of damping 
C                factors predictor.
C       FCDNM    Used to compute the denominator of the damping 
C                factor FC during computation of it's predictor,
C                corrector and aposteriori estimate (in the case of
C                performing a Rank1 update) .
C       FCK2     Aposteriori estimate of FC.
C       FCMIN2   FCMIN**2 . Used for FC-predictor computation.
C       FCNUMP   Gets the numerator of the predictor formula for FC.
C       FCNMP2   Temporary used for predictor numerator computation.
C       FCNUMK   Gets the numerator of the corrector computation 
C                of FC .
C       SUMXA    Natural level of the previous iterate.
C       TH       Temporary variable used during corrector- and 
C                aposteriori computations of FC.
C
C*    Internal integer variables
C     ==========================
C
C     IFAIL      Gets the return value from subroutines called from
C                NIINT (NIFACT, NISOLV, FCN, JAC)
C     ISCAL      Holds the scaling option from the IOPT-field ISCAL      
C     MODE       Matrix storage mode (see IOPT-field MODE) 
C     NRED       Count of successive corrector steps
C
C
C*    Internal logical variables
C     ==========================
C
C     QGENJ      Jacobian updating technique flag:
C                =.TRUE.  : Call of analytical subroutine JAC or
C                           numerical differentiation
C                =.FALSE. : rank1- (Broyden-) update
C     QINISC     Iterate initial-scaling flag:
C                =.TRUE.  : at first call of NISCAL
C                =.FALSE. : at successive calls of NISCAL
C     QSUCC      See description of IOPT-field QSUCC.
C     QJCRFR     Jacobian refresh flag:
C                set to .TRUE. if damping factor gets too small
C                and Jacobian was computed by rank1-update. 
C                Indicates, that the Jacobian needs to be recomputed
C                by subroutine JAC.
C     QSCALE     Holds the value of .NOT.QNSCAL. See description
C                of IOPT-field QNSCAL.
C     QSTRUC     Sparse structure changes of Jacobian (determined
C                in each iteration step):
C                =.TRUE.  : Sparse structure has been changed
C                =.FALSE. : New Jacobians sparse structure is the
C                           same as the previous
C
C*    Subroutines called:
C     ===================
C
C       NIFACT, NISOLV, NILVLS, NISCRF, NISOUT, NIPRV1, NIPRV2, NISCAL,
C       MONON,  MONOFF
C
C*    Functions called:
C     =================
C
C       D1MACH, WNORM
C
C
C*    Machine constants used
C     ======================
C
      DOUBLE PRECISION EPMACH,SMALL
C 
C     ------------------------------------------------------------
C*    End Prologue
      EXTERNAL NIFACT, NISOLV, NILVLS, NISCRF, NISOUT, NIPRV1,
     $         NIPRV2, NISCAL,
     $         MONON,  MONOFF, D1MACH, WNORM
      INTRINSIC DSQRT,DMIN1,MAX0,MIN0
      DOUBLE PRECISION ONE
      PARAMETER (ONE=1.0D0)
      DOUBLE PRECISION ZERO
      PARAMETER (ZERO=0.0D0)
      DOUBLE PRECISION HALF
      PARAMETER (HALF=0.5D0)
      DOUBLE PRECISION TEN
      PARAMETER (TEN=10.0D0)
      INTEGER IFAIL,ILOOP,ISCAL,K,MODE,NRED,L1
      DOUBLE PRECISION ALFA1,ALFA2,ALFA,BETA,
     $CONVA,DLEVXA,DMYPRI,D1MACH,DXANRM,DXNRM,WNORM,FCDNM,FCMIN2,
     $FCBND,FCBH,FCK2,FCH,FCNUMP,FCCOR,FCNMP2,FCNUMK,FCREDU,DLEVFN,
     $SUMXA,SUM1,SUM2,TH,RSMALL
      LOGICAL QGENJ,QINISC,QSUCC,QJCRFR,QSCALE,QSTRUC,QNEXT,QREP,QRANK1,
     $        QMIXIO,QFIXPT
CWEI
      INTRINSIC DLOG
      DOUBLE PRECISION CLIN0,CLIN1,CALPHA,CALPHK,ALPHAE,ALPHAK,ALPHAA,
     $                 SUMXA0,SUMXA1,SUMXA2,SUMXTE,FCMON,DLOG
      INTEGER ICONV, IORMON
      LOGICAL QMSTOP
      SAVE CLIN0,CLIN1,CALPHA,ALPHAE,ALPHAK,ALPHAA,SUMXA0,SUMXA1,SUMXA2,
     $     ICONV,QMSTOP
C
      EPMACH = D1MACH(3)
      SMALL  = D1MACH(6)
C*    Begin
C       ----------------------------------------------------------
C       1 Initialization
C       ----------------------------------------------------------
C       1.1 Control-flags and -integers
        QSUCC = IOPT(1).EQ.1
        QSCALE = .NOT. IOPT(35).EQ.1
        QRANK1 = IOPT(32).EQ.1
        IORMON = IOPT(39)
        IF (IORMON.EQ.0) IORMON=2
        QMIXIO = LUMON.EQ.LUSOL .AND. MPRMON.NE.0 .AND. MPRSOL.NE.0
        QFIXPT =IOPT(37).EQ.1
        ISCAL = IOPT(9)
        MODE = IOPT(2)
        MPRTIM = IOPT(19)
C       ----------------------------------------------------------
C       1.2 Derivated dimensional parameters
C       Yet none
C       ----------------------------------------------------------
C       1.3 Derivated internal parameters
        FCMIN2 = FCMIN*FCMIN
        TOLMIN = DSQRT(TEN*EPMACH)
        RSMALL = DSQRT(TEN*RTOL)
C       ----------------------------------------------------------
C       1.4 Adaption of input parameters, if necessary
        IF(FC.LT.FCMIN) FC = FCMIN
        IF(FC.GT.ONE) FC = ONE
C       ----------------------------------------------------------
C       1.5 Initial preparations
        QJCRFR = .FALSE.
        IFAIL = 0
        FCBND = ZERO
        IF (QBDAMP) FCBND = RWK(20)
C       ----------------------------------------------------------
C       1.5.1 Miscellaneous preparations of first iteration step
        IF (.NOT.QSUCC) THEN
          NITER = 0
          NCORR = 0
          NREJR1 = 0
          NFCN = 0
          NJAC = 0
          QINISC = .TRUE.
          QGENJ = .TRUE.
          FCKEEP = FC
          FCA = FC
          FCPRI = FC
          FCK2 = FC
          CONV = ZERO
          DO 1521 L1=1,N
            XA(L1)=X(L1)
1521      CONTINUE
CWEI      
          ICONV = 0
          ALPHAE = ZERO
          SUMXA1 = ZERO
          SUMXA0 = ZERO
          CLIN0  = ZERO
          QMSTOP = .FALSE.
C         ------------------------------------------------------
C         1.6 Print monitor header
          IF(MPRMON.GE.2 .AND. .NOT.QMIXIO)THEN
16003       FORMAT(///,2X,66('*'))
            WRITE(LUMON,16003)
16004       FORMAT(/,8X,'It',7X,'Normf ',10X,'Normx ',8X,
     $             'Damp.Fct.',3X,'New')
            WRITE(LUMON,16004)
          ENDIF
C         --------------------------------------------------------
C         1.7 Startup step
C         --------------------------------------------------------
C         1.7.1 Computation of the residual vector
          IF (MPRTIM.NE.0) CALL MONON(1)
          CALL FCN(N,X,F,IFAIL)
          IF (MPRTIM.NE.0) CALL MONOFF(1)
          NFCN = NFCN+1
C     Exit, if ...
          IF (IFAIL.NE.0) THEN
            IERR = 82
            GOTO 4299
          ENDIF
        ELSE
          QINISC = .FALSE.
        ENDIF
C
C       Main iteration loop
C       ===================
C
C       Repeat
2       CONTINUE
C         --------------------------------------------------------
C         2 Startup of iteration step
          IF (.NOT.QJCRFR) THEN
C           ------------------------------------------------------
C           2.1 Scaling of variables X(N)
            CALL NISCAL(N,X,XA,XSCAL,XW,ISCAL,QINISC,IOPT,LRWK,RWK)
            QINISC = .FALSE.
            IF(NITER.NE.0)THEN
              DO 2200 L1=1,N
                DXQA(L1)=DXQ(L1)
2200          CONTINUE
C             ----------------------------------------------------
C             2.2.1 Aposteriori estimate of damping factor
              FCNUMP = ZERO
              DO 2201 L1=1,N
                FCNUMP=FCNUMP+(DX(L1)/XW(L1))**2
2201          CONTINUE
              TH = FC-ONE
              FCDNM = ZERO
              DO 2202 L1=1,N
                FCDNM=FCDNM+((DXQA(L1)+TH*DX(L1))/XW(L1))**2
2202          CONTINUE
C             --------------------------------------------------
C             2.2.2 Decision criterion for Jacobian updating
C                   technique:
C                   QGENJ.EQ..TRUE. numerical differentation,
C                   QGENJ.EQ..FALSE. rank1 updating
              QGENJ = .TRUE.
              IF (FC.EQ.FCPRI) THEN
                QGENJ = FC.LT.ONE.OR.FCA.LT.ONE.OR.DMYCOR.LE.FC*SIGMA
     $                  .OR. .NOT.QRANK1 .OR. NEW+2.GT.NBROY 
                FCA = FC
              ELSE
                DMYCOR = FCA*FCA*HALF*DSQRT(FCNUMP/FCDNM)
                IF (NONLIN.LE.3) THEN
                  FCCOR = DMIN1(ONE,DMYCOR)
                ELSE
                  FCCOR = DMIN1(ONE,HALF*DMYCOR)
                ENDIF
                FCA = DMAX1(DMIN1(FC,FCCOR),FCMIN)
C$Test-begin
                IF (MPRMON.GE.5) THEN
                  WRITE(LUMON,22201) FCCOR, FC, DMYCOR, FCNUMP,
     $                               FCDNM
22201             FORMAT (/, ' +++ aposteriori estimate +++', /,
     $                    ' FCCOR  = ', D18.10,'  FC     = ', D18.10, /,
     $                    ' DMYCOR = ', D18.10,'  FCNUMP = ', D18.10, /,
     $                    ' FCDNM  = ', D18.10,/,
     $                    ' ++++++++++++++++++++++++++++',/)
                ENDIF
C$Test-end 
              ENDIF
              FCK2 = FCA
C             ------------------------------------------------------
C             2.2.2 Computation of the numerator of damping
C                   factor predictor
              FCNMP2 = ZERO
              DO 221 L1=1,N
                FCNMP2=FCNMP2+(DXQA(L1)/XW(L1))**2
221           CONTINUE
              FCNUMP = FCNUMP*FCNMP2
            ENDIF
          ENDIF
          QJCRFR =.FALSE.
C         --------------------------------------------------------
C         2.3 Jacobian matrix (stored to array A(NFMAX))
C         --------------------------------------------------------
C         2.3.1 Jacobian generation by routine JAC or
C               difference approximation (If QGENJ.EQ..TRUE.)
C               - or -
C               Rank-1 update of Jacobian (If QGENJ.EQ..FALSE.)
          IF (QGENJ) THEN
            NEW = 0
            NFILL  = NFMAX
            IFAIL = NJAC + 1
            IF (MPRTIM.NE.0) CALL MONON(2)
            CALL JAC(N,X,A,IROW,ICOL,NFILL,IFAIL)
            IF (MPRTIM.NE.0) CALL MONOFF(2)
            NJAC = NJAC + 1
C     Exit, If ...
            IF (IFAIL.LT.0) THEN
              IERR = 83
              GOTO 4299
            ENDIF
            IF(NFILL.GT.NFMAX)THEN
              NFILLA = NFILL
              IERR = 11
              GOTO 4299
            ENDIF
          ELSE
            NEW = NEW+1
          ENDIF
          IF ( NEW.EQ.0 ) THEN
C           ------------------------------------------------------
C           2.3.2 Save scaling values
            DO 232 L1=1,N
              XWA(L1) = XW(L1)
232         CONTINUE
C           --------------------------------------------------------
C           2.3.3 Check, if re-determination of sparse
C                 structure of Jacobian is needed.
C                 If so, QSTRUC  = .TRUE. is set,
C                 otherwise QSTRUC  = .FALSE. is set.
            IF (QFIXPT) THEN
              QSTRUC = NJAC.EQ.1
            ELSE
              IF(NJAC.GT.1)THEN
                IF(NFILL.NE.NFILLA)THEN
                  QSTRUC  = .TRUE.
                ELSE
                  QSTRUC  = .FALSE.
                  DO 2331 I=1,NFILL
                    IF(IROW(I).NE.IROWA(I).OR.ICOL(I).NE.ICOLA(I))THEN
                      QSTRUC  = .TRUE.
                      GOTO 2339
                    ENDIF
2331              CONTINUE
2339              CONTINUE
                ENDIF
              ELSE
                QSTRUC  = .TRUE.
              ENDIF
C             --------------------------------------------------------
C             2.3.4 Save new Jacobian structure characteristics
              IF(QSTRUC)THEN
                NFILLA  = NFILL
                DO 2341 L1=1,NFILL
                  IROWA(L1)=IROW(L1)
2341            CONTINUE
                DO 2342 L1=1,NFILL
                  ICOLA(L1)=ICOL(L1)
2342            CONTINUE
              ENDIF
            ENDIF
C           --------------------------------------------------------
C           2.4 Prepare solution of the linear system
C           --------------------------------------------------------
C           2.4.1 internal column scaling of matrix A
            DO 2412 L1=1,NFILL
              A(L1)=-A(L1)*XW(ICOL(L1))
2412        CONTINUE
C           ------------------------------------------------------
C           2.4.2 Row scaling of matrix A
            IF (QSCALE) THEN
              CALL NISCRF(N,NFILL,A,IROW,ICOL,FW)
            ELSE
              DO 242 K=1,N
                FW(K)=ONE
242           CONTINUE
            ENDIF
          ENDIF
C         --------------------------------------------------------
C         2.4.3 Save and scale values of F(N)
          DO 243 L1=1,N
            FA(L1)=F(L1)
            T1(L1)=F(L1)*FW(L1)
243       CONTINUE
C         --------------------------------------------------------
C         3 Central part of iteration step
C         --------------------------------------------------------
C         3.1 Solution of the linear system
C         --------------------------------------------------------
C         3.1.1 Decomposition of sparse (N,N)-matrix A
          IF (NEW.EQ.0) THEN
            CALL NIFACT(N,NFILL,A,IROW,ICOL,IOPT,QSTRUC,IFAIL,LNIWLA,
     $                 IWK(NIWLA),LNI2WL,I2WK(NI2WLA),LNRWLA,RWK(NRWLA))
C     Exit Repeat If ...
            IF(IFAIL.NE.0) THEN
              IF (IFAIL.EQ.1) THEN
                IERR = 1
              ELSE
                IERR = 80
              ENDIF
              GOTO 4299
            ENDIF
          ENDIF
C         --------------------------------------------------------
C         3.1.2 Solution of linear (N,N)-system
          IF(NEW.EQ.0) THEN 
            IF (MPRTIM.NE.0) CALL MONON(5)
            CALL NISOLV(N,NFILL,A,IROW,ICOL,T1,IOPT,IFAIL,LNIWLA,
     $                 IWK(NIWLA),LNI2WL,I2WK(NI2WLA),LNRWLA,RWK(NRWLA))
            IF (MPRTIM.NE.0) CALL MONOFF(5)
C     Exit Repeat If ...
            IF(IFAIL.NE.0)  THEN
              IERR = 81
              GOTO 4299
            ENDIF
          ELSE  
            ALFA1=ZERO
            ALFA2=ZERO
            DO 3121 I=1,N
              ALFA1=ALFA1+DX(I)*DXQ(I)/XW(I)**2
              ALFA2=ALFA2+DX(I)**2/XW(I)**2
3121        CONTINUE
            ALFA=ALFA1/ALFA2
            BETA=ONE-ALFA
            DO 3122 I=1,N
              T1(I)=(DXQ(I)+(FCA-ONE)*ALFA*DX(I))/BETA
3122        CONTINUE
            IF(NEW.EQ.1) THEN
              DO 3123 I=1,N
                DXSAVE(I,1)=DX(I)
3123          CONTINUE
            ENDIF
            DO 3124 I=1,N
              DXSAVE(I,NEW+1)=T1(I)
              DX(I)=T1(I)
              T1(I)=T1(I)/XW(I)
3124        CONTINUE
          ENDIF
C         --------------------------------------------------------
C         3.2 Evaluation of scaled natural level function SUMX
C             scaled maximum error norm CONV
C             evaluation of (scaled) standard level function
C             DLEVF ( DLEVF only, if MPRMON.GE.2 )
C             and computation of ordinary Newton corrections 
C             DX(N)
          CALL NILVLS(N,T1,XW,F,DX,CONV,SUMX,DLEVF,MPRMON,NEW.EQ.0)
          DO 32 L1=1,N
            XA(L1)=X(L1)
32        CONTINUE
          SUMXA = SUMX
          DLEVXA = DSQRT(SUMXA/DBLE(FLOAT(N)))
          CONVA = CONV
          DXANRM = WNORM(N,DX,XW)
C         --------------------------------------------------------
C         3.3 A - priori estimate of damping factor FC
          IF(NITER.NE.0.AND.NONLIN.NE.1.AND.NEW.EQ.0)THEN
C           ------------------------------------------------------
C           3.3.1 Computation of the denominator of a-priori
C                 estimate
            FCDNM = ZERO
            DO 331 L1=1,N
              FCDNM=FCDNM+((DX(L1)-DXQA(L1))/XW(L1))**2
331         CONTINUE
            FCDNM = FCDNM*SUMX
C           ------------------------------------------------------
C           3.3.2 New damping factor
            IF(FCDNM.GT.FCNUMP*FCMIN2 .OR.
     $        (NONLIN.EQ.4 .AND. FCA**2*FCNUMP .LT. 4.0D0*FCDNM)) THEN
              DMYPRI = FCA*DSQRT(FCNUMP/FCDNM)
              FCPRI = DMIN1(DMYPRI,ONE)
              IF (NONLIN.EQ.4) FCPRI = DMIN1(HALF*DMYPRI,ONE)
            ELSE
              FCPRI = ONE
C$Test-begin
              DMYPRI = -1.0D0
C$Test-end
            ENDIF
C$Test-begin
            IF (MPRMON.GE.5) THEN
              WRITE(LUMON,33201) FCPRI, FC, FCA, DMYPRI, FCNUMP,
     $                           FCDNM
33201         FORMAT (/, ' +++ apriori estimate +++', /,
     $                ' FCPRI  = ', D18.10, '  FC     = ', D18.10, /,
     $                ' FCA    = ', D18.10, '  DMYPRI = ', D18.10, /,
     $                ' FCNUMP = ', D18.10, '  FCDNM  = ', D18.10, /,
     $                   ' ++++++++++++++++++++++++', /)
            ENDIF
C$Test-end 
            FC = DMAX1(FCPRI,FCMIN)
            IF (QBDAMP) THEN
              FCBH = FCA*FCBND
              IF (FC.GT.FCBH) THEN
                FC = FCBH
                IF (MPRMON.GE.4)
     $            WRITE(LUMON,*) ' *** incr. rest. act. (a prio) ***'
              ENDIF
              FCBH = FCA/FCBND
              IF (FC.LT.FCBH) THEN
                FC = FCBH
                IF (MPRMON.GE.4)
     $            WRITE(LUMON,*) ' *** decr. rest. act. (a prio) ***'
              ENDIF
            ENDIF
          ENDIF
CWEI
          IF (IORMON.GE.2) THEN
            SUMXA2=SUMXA1
            SUMXA1=SUMXA0
            SUMXA0=DLEVXA
            IF (SUMXA0.EQ.ZERO) SUMXA0=SMALL
C           Check convergence rates (linear and superlinear)
C           ICONV : Convergence indicator
C                   =0: No convergence indicated yet
C                   =1: Damping factor is 1.0d0
C                   =2: Superlinear convergence detected (alpha >=1.2)
C                   =3: Quadratic convergence detected (alpha > 1.8)
            FCMON = DMIN1(FC,FCMON)
            IF (FCMON.LT.ONE) THEN
              ICONV = 0
              ALPHAE = ZERO
            ENDIF
            IF (FCMON.EQ.ONE .AND. ICONV.EQ.0) ICONV=1
            IF (NITER.GE.1) THEN
              CLIN1 = CLIN0
              CLIN0 = SUMXA0/SUMXA1
            ENDIF
            IF (ICONV.GE.1.AND.NITER.GE.2) THEN
              ALPHAK = ALPHAE
              ALPHAE = ZERO
              IF (CLIN1.LE.0.95D0) ALPHAE = DLOG(CLIN0)/DLOG(CLIN1)
              IF (ALPHAK.NE.ZERO) ALPHAK =0.5D0*(ALPHAE+ALPHAK)
              ALPHAA = DMIN1(ALPHAK,ALPHAE)
              CALPHK = CALPHA
              CALPHA = ZERO
              IF (ALPHAE.NE.ZERO) CALPHA = SUMXA1/SUMXA2**ALPHAE
              SUMXTE = DSQRT(CALPHA*CALPHK)*SUMXA1**ALPHAK-SUMXA0
              IF (ALPHAA.GE.1.2D0 .AND. ICONV.EQ.1) ICONV = 2
              IF (ALPHAA.GT.1.8D0) ICONV = 3
              IF (MPRMON.GE.4)  WRITE(LUMON,32001) ICONV, ALPHAE, 
     $                            CALPHA, CLIN0, ALPHAK, SUMXTE
32001         FORMAT(' ** ICONV: ',I1,'  ALPHA: ',D9.2,
     $               '  CONST-ALPHA: ',D9.2,'  CONST-LIN: ',D9.2,' **',
     $               /,' **',11X,'ALPHA-POST: ',D9.2,' CHECK: ',D9.2,
     $               25X,'**')
              IF ( ICONV.GE.2 .AND. ALPHAA.LT.0.9D0 ) THEN
                 IF (IORMON.EQ.3) THEN
                   IERR = 4
                   GOTO 4299
                 ELSE
                   QMSTOP = .TRUE.
                 ENDIF 
              ENDIF
            ENDIF
          ENDIF
          FCMON = FC
C
C         --------------------------------------------------------
C         3.4 Save natural level for later computations of
C             corrector and print iterate
          FCNUMK = SUMX
          IF (MPRMON.GE.2) THEN
            IF (MPRTIM.NE.0) CALL MONON(6)
            CALL NIPRV1(DLEVF,DLEVXA,FCKEEP,NITER,NEW,MPRMON,LUMON,
     $                  QMIXIO)
            IF (MPRTIM.NE.0) CALL MONOFF(6)
          ENDIF
          NRED = 0
          QNEXT = .FALSE.
          QREP  = .FALSE.   
C         QREP = ITER .GT. ITMAX   or  QREP = ITER .GT. 0
C
C         Damping-factor reduction loop
C         ================================
C         DO (Until)
34        CONTINUE
C           ------------------------------------------------------
C           3.5 Preliminary new iterate
            DO 35 L1=1,N
              X(L1)=XA(L1)+DX(L1)*FC
35          CONTINUE
C           -----------------------------------------------------
C           3.5.2 Exit, if problem is specified as being linear
C     Exit Repeat If ...
            IF( NONLIN.EQ.1 )THEN
              IERR = 0
              GOTO 4299
            ENDIF
C           ------------------------------------------------------
C           3.6.1 Computation of the residual vector
            IF (MPRTIM.NE.0) CALL MONON(1)
            CALL FCN(N,X,F,IFAIL)
            IF (MPRTIM.NE.0) CALL MONOFF(1)
            NFCN = NFCN+1
C     Exit, If ...
            IF(IFAIL.LT.0)THEN
              IERR = 82
              GOTO 4299
            ENDIF
            IF(IFAIL.EQ.1 .OR. IFAIL.EQ.2) THEN
              IF (IFAIL.EQ.1) THEN
                FCREDU = HALF
              ELSE
                FCREDU = F(1)
C     Exit, If ...
                IF (FCREDU.LE.0 .OR. FCREDU.GE.1) THEN
                  IERR = 83
                  GOTO 4299
                ENDIF
              ENDIF
              IF (MPRMON.GE.2) THEN
36101           FORMAT(8X,I2,' FCN could not be evaluated  ',
     $                 8X,F7.5,4X,I2)
                WRITE(LUMON,36101)NITER,FC,NEW
              ENDIF
              FCH = FC
              FC = FCREDU*FC
              IF (FCH.GT.FCMIN) FC = DMAX1(FC,FCMIN)
              IF (QBDAMP) THEN
                FCBH = FCH/FCBND
                IF (FC.LT.FCBH) THEN
                  FC = FCBH
                  IF (MPRMON.GE.4) WRITE(LUMON,*)
     $               ' *** decr. rest. act. (FCN redu.) ***'
                ENDIF
              ENDIF
              IF (FC.LT.FCMIN) THEN
                IERR = 3
                GOTO 4299
              ENDIF  
C     Break DO (Until) ...
              GOTO 3109
            ENDIF
            DO 361 L1=1,N
              T1(L1)=F(L1)*FW(L1)
361         CONTINUE
C           ------------------------------------------------------
C           3.6.2 Solution of linear (N,N)-system
            IF (MPRTIM.NE.0) CALL MONON(5)
            CALL NISOLV(N,NFILL,A,IROW,ICOL,T1,IOPT,IFAIL,LNIWLA,
     $             IWK(NIWLA),LNI2WL,I2WK(NI2WLA),LNRWLA,RWK(NRWLA))
            IF (MPRTIM.NE.0) CALL MONOFF(5)
C     Exit Repeat If ...
            IF(IFAIL.NE.0)  THEN
              IERR = 81
              GOTO 4299
            ENDIF
            IF(NEW.GT.0) THEN 
              DO 3630 I=1,N
                DXQ(I) = T1(I)*XWA(I)
3630          CONTINUE                   
              DO 363 ILOOP=1,NEW 
                SUM1=ZERO
                SUM2=ZERO
                DO 3631 I=1,N
                  SUM1=SUM1+(DXQ(I)*DXSAVE(I,ILOOP))/ XW(I)**2
                  SUM2=SUM2+(DXSAVE(I,ILOOP)/XW(I))**2
3631            CONTINUE
                BETA=SUM1/SUM2
                DO 3632 I=1,N
                  DXQ(I)=DXQ(I)+BETA*DXSAVE(I,ILOOP+1)
                  T1(I) = DXQ(I)/XW(I)
3632            CONTINUE
363           CONTINUE
            ENDIF
C           ------------------------------------------------------
C           3.6.3 Evaluation of scaled natural level function
C                 SUMX
C                 scaled maximum error norm CONV and evaluation
C                 of (scaled) standard level function DLEVFN
            CALL NILVLS(N,T1,XW,F,DXQ,CONV,SUMX,DLEVFN,MPRMON,NEW.EQ.0)
            DXNRM = WNORM(N,DXQ,XW)
C           ------------------------------------------------------
C           3.6.4 Convergence test
C     Exit Repeat If ...
            IF ( DXNRM.LE.RTOL .AND. DXANRM.LE.RSMALL .AND. 
     $           FC.EQ.ONE ) THEN
              IERR = 0
              GOTO 4299
            ENDIF
C           
            FCA = FC
C           ----------------------------------------------------
C           3.6.5 Evaluation of reduced damping factor
            TH = FCA-ONE
            FCDNM = ZERO
            DO 39 L1=1,N
              FCDNM=FCDNM+((DXQ(L1)+TH*DX(L1))/XW(L1))**2
39          CONTINUE
            IF (FCDNM.NE.ZERO) THEN
              DMYCOR = FCA*FCA*HALF*DSQRT(FCNUMK/FCDNM)
            ELSE
              DMYCOR = 1.0D+35
            ENDIF
            IF (NONLIN.LE.3) THEN
              FCCOR = DMIN1(ONE,DMYCOR)
            ELSE
              FCCOR = DMIN1(ONE,HALF*DMYCOR)
            ENDIF
C$Test-begin
            IF (MPRMON.GE.5) THEN
              WRITE(LUMON,39001) FCCOR, FC, DMYCOR, FCNUMK,
     $                           FCDNM, FCA
39001         FORMAT (/, ' +++ corrector computation +++', /,
     $                ' FCCOR  = ', D18.10, '  FC     = ', D18.10, /,
     $                ' DMYCOR = ', D18.10, '  FCNUMK = ', D18.10, /,
     $                ' FCDNM  = ', D18.10, '  FCA    = ', D18.10, /,
     $                   ' +++++++++++++++++++++++++++++', /)
                ENDIF
C$Test-end 
C           ------------------------------------------------------
C           3.7 Natural monotonicity test
            IF(SUMX.GT.SUMXA)THEN
C             ----------------------------------------------------
C             3.8 Output of iterate
              IF (MPRMON.GE.3) THEN
                IF (MPRTIM.NE.0) CALL MONON(6)
                CALL NIPRV2(DLEVFN,DSQRT(SUMX/DBLE(FLOAT(N))),FC,
     $                      NITER,MPRMON,LUMON,QMIXIO,'*')
                IF (MPRTIM.NE.0) CALL MONOFF(6)
              ENDIF
              IF (QMSTOP) THEN
                IERR = 4
                GOTO 4299
              ENDIF
              FCH = DMIN1(FCCOR,HALF*FC)
              IF (FC.GT.FCMIN) THEN
                FC=DMAX1(FCH,FCMIN)
              ELSE
                FC=FCH
              ENDIF
              IF (QBDAMP) THEN
                FCBH = FCA/FCBND
                IF (FC.LT.FCBH) THEN
                  FC = FCBH
                  IF (MPRMON.GE.4)
     $              WRITE(LUMON,*) ' *** decr. rest. act. (a post) ***'
                ENDIF
              ENDIF
CWEI
              FCMON = FC
C
C$Test-begin
              IF (MPRMON.GE.5) THEN
                WRITE(LUMON,39002) FC
39002           FORMAT (/, ' +++ corrector setting 1 +++', /,
     $                  ' FC     = ', D18.10, /,
     $                     ' +++++++++++++++++++++++++++', /)
              ENDIF
C$Test-end 
              QREP = .TRUE.
              NCORR = NCORR+1
              NRED = NRED+1
C             ----------------------------------------------------
C             3.10 If damping factor is too small:
C                  Refresh Jacobian,if current Jacobian was computed
C                  by a Rank1-update, else fail exit
              QJCRFR  = FC.LT.FCMIN.OR.NEW.GT.0.AND.NRED.GT.1
C     Exit Repeat If ...
              IF(QJCRFR.AND.NEW.EQ.0)THEN
                IERR = 3
                GOTO 4299
              ENDIF
            ELSE
              IF (.NOT.QREP .AND. FCCOR.GT.SIGMA2*FC) THEN
                IF (MPRMON.GE.3) THEN
                  IF (MPRTIM.NE.0) CALL MONON(6)
                  CALL NIPRV2(DLEVFN,DSQRT(SUMX/DBLE(FLOAT(N))),FC,
     $                        NITER,MPRMON,LUMON,QMIXIO,'+')
                  IF (MPRTIM.NE.0) CALL MONOFF(6)
                ENDIF
                FC = FCCOR
C$Test-begin
                IF (MPRMON.GE.5) THEN
                  WRITE(LUMON,39003) FC
39003             FORMAT (/, ' +++ corrector setting 2 +++', /,
     $                    ' FC     = ', D18.10, /,
     $                       ' +++++++++++++++++++++++++++', /)
                ENDIF
C$Test-end 
                QREP = .TRUE.
              ELSE
                QNEXT = .TRUE.
              ENDIF
            ENDIF
3109      CONTINUE
          IF(.NOT.(QNEXT.OR.QJCRFR)) GOTO  34
C         UNTIL ( expression - negated above)
C         End of damping-factor reduction loop
C         =======================================
          IF(QJCRFR)THEN
C           ------------------------------------------------------
C           3.11 Restore former values for repeting iteration
C                step
            NREJR1 = NREJR1+1
            DO 3111 L1=1,N
              X(L1)=XA(L1)
3111        CONTINUE
            DO 3112 L1=1,N
              F(L1)=FA(L1)
3112        CONTINUE
            IF(MPRMON.GE.2)THEN
31130           FORMAT(8X,I2,' Not accepted damping factor ',
     $                 8X,F7.5,4X,I2)
                WRITE(LUMON,31130)NITER,FC,NEW
            ENDIF
            FC = FCKEEP
            FCA = FCK2
            IF(NITER.EQ.0)THEN
              FC = FCMIN
            ENDIF
            QGENJ = .TRUE.
          ELSE
C           ------------------------------------------------------
C           4 Preparations to start the following iteration step
C           ------------------------------------------------------
C           4.1 Print values
            IF (MPRMON.GE.3) THEN
              IF (MPRTIM.NE.0) CALL MONON(6)
              CALL NIPRV2(DLEVFN,DSQRT(SUMX/DBLE(FLOAT(N))),FC,NITER+1,
     $                    MPRMON,LUMON,QMIXIO,'*')
              IF (MPRTIM.NE.0) CALL MONOFF(6)
            ENDIF
C           Print the natural level of the current iterate and return
C           it in one-step mode
            SUMX = SUMXA
            IF(MPRSOL.GE.2.AND.NITER.NE.0) THEN
              IF (MPRTIM.NE.0) CALL MONON(6)
              CALL NISOUT(N,XA,2,IOPT,RWK,LRWK,IWK,LIWK,MPRSOL,LUSOL)
              IF (MPRTIM.NE.0) CALL MONOFF(6)
            ELSE IF(MPRSOL.GE.1.AND.NITER.EQ.0)THEN
              IF (MPRTIM.NE.0) CALL MONON(6)
              CALL NISOUT(N,XA,1,IOPT,RWK,LRWK,IWK,LIWK,MPRSOL,LUSOL)
              IF (MPRTIM.NE.0) CALL MONOFF(6)
            ENDIF
            NITER = NITER+1
            DLEVF = DLEVFN
C     Exit Repeat If ...
            IF(NITER.GE.NITMAX)THEN
              IERR = 2
              GOTO 4299
            ENDIF
            FCKEEP = FC
C           ------------------------------------------------------
C           4.2 Return, if in one-step mode
C Exit Subroutine If ...
            IF (MODE.EQ.1) THEN
              IOPT(1)=1
              RETURN
            ENDIF
          ENDIF
        GOTO 2
C       End Repeat
4299    CONTINUE
C       End of main iteration loop
C       ==========================
C       ----------------------------------------------------------
C       9 Exits
C       ----------------------------------------------------------
C       9.1 Solution exit
        IF(IERR.EQ.0 .OR. IERR.EQ.4)THEN
          IF (NONLIN.NE.1) THEN
            IF ( IERR.EQ.0 ) THEN
              DO 91 L1=1,N
                X(L1)=X(L1)+DXQ(L1)
91            CONTINUE
            ELSE IF (ALPHAA.GT.ZERO .AND. IORMON.EQ.3) THEN
              DO 92 L1=1,N
                X(L1)=X(L1)+DX(L1)
92            CONTINUE
            ENDIF
C           Print final monitor output
            IF(MPRMON.GE.2) THEN
              IF (IERR.EQ.0) THEN
                IF (MPRTIM.NE.0) CALL MONON(6)
                CALL NIPRV2(DLEVFN,DSQRT(SUMX/DBLE(FLOAT(N))),FC,
     $                      NITER+1,MPRMON,LUMON,QMIXIO,'*')
                IF (MPRTIM.NE.0) CALL MONOFF(6)
              ELSE IF (IORMON.EQ.3) THEN
                IF (MPRTIM.NE.0) CALL MONON(6)
                CALL NIPRV1(DLEVFN,DSQRT(SUMXA/DBLE(FLOAT(N))),FC,
     $                      NITER,NEW,MPRMON,LUMON,QMIXIO)
                IF (MPRTIM.NE.0) CALL MONOFF(6)
              ENDIF
            ENDIF
            IF (  IORMON.GE.2 ) THEN
              IF ( ICONV.LE.1 .AND. ALPHAE .NE. ZERO 
     $                        .AND. ALPHAK .NE. ZERO ) IERR = 5
            ENDIF
C
            IF(MPRMON.GE.1) THEN
91001         FORMAT(///' Solution of nonlinear system ',
     $        'of equations obtained within ',I3,
     $        ' iteration steps',//,' Achieved relative accuracy',D10.3)
              IF (IERR.EQ.4) THEN
                WRITE(LUMON,91001) NITER,DSQRT(SUMXA/DBLE(FLOAT(N)))
              ELSE
                WRITE(LUMON,91001) NITER+1,DSQRT(SUMX/DBLE(FLOAT(N)))
              ENDIF 
            ENDIF
          ELSE
            IF(MPRMON.GE.1) THEN
91002         FORMAT(///' Solution of linear system ',
     $        'of equations obtained by NLEQ1S',//,' No estimate ',
     $        'available for the achieved relative accuracy')
                WRITE(LUMON,91002)
            ENDIF
          ENDIF
        ENDIF
C       ----------------------------------------------------------
C       9.2 Fail exit messages
C       ----------------------------------------------------------
C       9.2.1 Termination, since jacobian matrix became singular
        IF(IERR.EQ.1.AND.MPRERR.GE.1)THEN
92101     FORMAT(/,' Iteration terminates due to ',
     $    'singular jacobian matrix',/)
          WRITE(LUERR,92101)
        ENDIF
C       ----------------------------------------------------------
C       9.2.2 Termination after more than NITMAX iterations
        IF(IERR.EQ.2.AND.MPRERR.GE.1)THEN
92201     FORMAT(/,' Iteration terminates after NITMAX ',
     $    '=',I3,'  Iteration steps')
          WRITE(LUERR,92201)NITMAX
        ENDIF
C       ----------------------------------------------------------
C       9.2.3 Damping factor FC became too small
        IF(IERR.EQ.3.AND.MPRERR.GE.1)THEN
92301     FORMAT(/,' Damping factor has become too ',
     $    'small: lambda =',D10.3,2X,/)
          WRITE(LUERR,92301)FC
        ENDIF
CWEI
C       ----------------------------------------------------------
C       9.2.4.1 Superlinear convergence slowed down
        IF(IERR.EQ.4.AND.MPRERR.GE.1)THEN
92401     FORMAT(/,' Warning: Monotonicity test failed after ',A,
     $           ' convergence was already checked;',/,
     $    ' RTOL requirement may be too stringent',/)
92402     FORMAT(/,' Warning: ',A,' convergence slowed down;',/,
     $    ' RTOL requirement may be too stringent',/)
          IF (QMSTOP) THEN
            IF (ICONV.EQ.2) WRITE(LUERR,92401) 'superlinear'
            IF (ICONV.EQ.3) WRITE(LUERR,92401) 'quadratic'
          ELSE
            IF (ICONV.EQ.2) WRITE(LUERR,92402) 'superlinear'
            IF (ICONV.EQ.3) WRITE(LUERR,92402) 'quadratic'
          ENDIF
        ENDIF
C       ----------------------------------------------------------
C       9.2.4.2 Convergence criterion satisfied before superlinear
C               convergence has been established
        IF(IERR.EQ.5.AND.MPRERR.GE.1)THEN
92410     FORMAT(/,' Warning: No quadratic or superlinear convergence ',
     $           'established yet',/,
     $           10X,'your solution may perhaps may be less accurate ',
     $           /,10X,'as indicated by the standard error estimate')
          WRITE(LUERR,92410)
        ENDIF
C       ----------------------------------------------------------
C       9.2.5 Error exit due to linear solver routine NIFACT
        IF(IERR.EQ.80.AND.MPRERR.GE.1)THEN
92501     FORMAT(/,' Error ',I5,' signalled by linear solver NIFACT')
          WRITE(LUERR,92501) IFAIL
        ENDIF
C       ----------------------------------------------------------
C       9.2.6 Error exit due to linear solver routine NISOLV
        IF(IERR.EQ.81.AND.MPRERR.GE.1)THEN
92601     FORMAT(/,' Error ',I5,' signalled by linear solver NISOLV')
          WRITE(LUERR,92601) IFAIL
        ENDIF
C       ----------------------------------------------------------
C       9.2.7 Error exit due to fail of user function FCN
        IF(IERR.EQ.82.AND.MPRERR.GE.1)THEN
92701     FORMAT(/,' Error ',I5,' signalled by user function FCN')
          WRITE(LUERR,92701) IFAIL
        ENDIF
C       ----------------------------------------------------------
C       9.2.8 Error exit due to fail of user function JAC
        IF(IERR.EQ.83.AND.MPRERR.GE.1)THEN
92801     FORMAT(/,' Error ',I5,' signalled by user function JAC')
          WRITE(LUERR,92801) IFAIL
        ENDIF
        IF(IERR.GE.80.AND.IERR.LE.83) IWK(23) = IFAIL
        IF ((IERR.EQ.82.OR.IERR.EQ.83).AND.NITER.LE.1.AND.MPRERR.GE.1)
     $  THEN
          WRITE (LUERR,92810)
92810     FORMAT(' Try to find a better initial guess for the solution')
        ENDIF
C       ----------------------------------------------------------
C       9.3 Common exit
        IF (MPRERR.GE.3.AND.IERR.NE.0.AND.IERR.NE.4.AND.NONLIN.NE.1)
     $    THEN
93100     FORMAT(/,'    Achieved relative accuracy',D10.3,2X)
          WRITE(LUERR,93100)CONVA
        ENDIF
        SUMX = SUMXA
        IF(MPRSOL.GE.2.AND.NITER.NE.0) THEN
          IF (MPRTIM.NE.0) CALL MONON(6)
          CALL NISOUT(N,XA,2,IOPT,RWK,LRWK,IWK,LIWK,MPRSOL,LUSOL)
          IF (MPRTIM.NE.0) CALL MONOFF(6)
        ELSE IF(MPRSOL.GE.1.AND.NITER.EQ.0)THEN
          IF (MPRTIM.NE.0) CALL MONON(6)
          CALL NISOUT(N,XA,1,IOPT,RWK,LRWK,IWK,LIWK,MPRSOL,LUSOL)
          IF (MPRTIM.NE.0) CALL MONOFF(6)
        ENDIF
        IF (IERR.NE.4) NITER = NITER+1
        DLEVF = DLEVFN
        IF(MPRSOL.GE.1)THEN
C         Print Solution or final iteration vector
          IF(IERR.EQ.0)THEN
             MODEFI = 3
          ELSE
             MODEFI = 4
          ENDIF
          IF (MPRTIM.NE.0) CALL MONON(6)
          CALL NISOUT(N,X,MODEFI,IOPT,RWK,LRWK,IWK,LIWK,MPRSOL,LUSOL)
          IF (MPRTIM.NE.0) CALL MONOFF(6)
        ENDIF
C       Return the latest internal scaling to XSCAL
        DO 93 I=1,N
          XSCAL(I)=XW(I)
93      CONTINUE
C       End of exits
C       End of subroutine NIINT
      RETURN
      END
C
      SUBROUTINE NISCAL(N,X,XA,XSCAL,XW,ISCAL,QINISC,IOPT,LRWK,RWK)
C*    Begin Prologue SCALE
      INTEGER N
      DOUBLE PRECISION X(N),XSCAL(N),XA(N),XW(N)
      INTEGER ISCAL
      LOGICAL QINISC
      INTEGER IOPT(50),LRWK
      DOUBLE PRECISION RWK(LRWK)
C     ------------------------------------------------------------
C
C*    Summary :
C    
C     S C A L E : To be used in connection with NLEQ1S .
C       Computation of the internal scaling vector XW used for the
C       Jacobian matrix, the iterate vector and it's related
C       vectors - especially for the solution of the linear system
C       and the computations of norms to avoid numerical overflow.
C
C*    Input parameters
C     ================
C
C     N         Int     Number of unknowns
C     X(N)      Dble    Current iterate
C     XA(N)     Dble    Previous iterate
C     XSCAL(N)  Dble    User scaling passed from parameter XSCAL
C                       of interface routine NLEQ1S
C     ISCAL     Int     Option ISCAL passed from IOPT-field
C                       (for details see description of IOPT-fields)
C     QINISC    Logical = .TRUE.  : Initial scaling
C                       = .FALSE. : Subsequent scaling
C     IOPT(50)  Int     Options array passed from NLEQ1S parameter list
C     LRWK      Int     Length of real workspace
C     RWK(LRWK) Dble    Real workspace (see description above)
C
C*    Output parameters
C     =================
C
C     XW(N)     Dble   Scaling vector computed by this routine
C                      All components must be positive. The follow-
C                      ing relationship between the original vector
C                      X and the scaled vector XSCAL holds:
C                      XSCAL(I) = X(I)/XW(I) for I=1,...N
C
C*    Subroutines called: D1MACH
C
C*    Machine constants used
C     ======================
C
      DOUBLE PRECISION SMALL
C
C     ------------------------------------------------------------
C*    End Prologue
      EXTERNAL D1MACH
      INTRINSIC DABS,DMAX1
      DOUBLE PRECISION D1MACH,HALF
      PARAMETER (HALF=0.5D0)
      INTEGER MPRMON,LUMON
      SMALL  = D1MACH(6)
C*    Begin
      DO 1 L1=1,N
        IF (ISCAL.EQ.1) THEN
          XW(L1) = XSCAL(L1)
        ELSE
          XW(L1)=DMAX1(XSCAL(L1),(DABS(X(L1))+DABS(XA(L1)))*HALF,SMALL)
        ENDIF
1     CONTINUE
C$Test-Begin
      MPRMON = IOPT(13)
      IF (MPRMON.GE.6) THEN
        LUMON = IOPT(14)
        WRITE(LUMON,*) ' '
        WRITE(LUMON,*) ' ++++++++++++++++++++++++++++++++++++++++++'
        WRITE(LUMON,*) '      X-components   Scaling-components    '
        WRITE(LUMON,10) (X(L1), XW(L1), L1=1,N)
10      FORMAT('  ',D18.10,'  ',D18.10)
        WRITE(LUMON,*) ' ++++++++++++++++++++++++++++++++++++++++++'
        WRITE(LUMON,*) ' '
      ENDIF
C$Test-End
C     End of subroutine NISCAL
      RETURN
      END
C
      SUBROUTINE NISCRF(N,NFILL,A,IROW,ICOL,FW)
C*    Begin Prologue SCROWF
      INTEGER N,NFILL
      DOUBLE PRECISION A(NFILL)
      INTEGER IROW(NFILL),ICOL(NFILL)
      DOUBLE PRECISION FW(N)
C     ------------------------------------------------------------
C
C*    Summary :
C
C     S C R O W F : Row Scaling of a (M,N)-matrix in full storage
C                   mode
C
C*    Input parameters (* marks inout parameters)
C     ===========================================
C
C       N           Int    Number of rows and columns of the matrix
C     * A(NFILL)    Dble   Array holding real values of matrix to 
C                          be scaled
C       IROW(NFILL) ShInt  Array of row indices of matrix (corre-
C                          sponding to elements in A with equal
C                          array indices)
C       ICOL(NFILL) ShInt  Array of column indices of matrix (corre-
C                          sponding to elements in A with equal
C                          array indices)
C
C*    Output parameters
C     =================
C
C       FW(N)       Dble   Row scaling factors - FW(i) contains
C                          the factor by which the i-th row of A
C                          has been multiplied
C
C     ------------------------------------------------------------
C*    End Prologue
      INTRINSIC DABS
      DOUBLE PRECISION ONE
      PARAMETER (ONE=1.0D0)
      DOUBLE PRECISION ZERO
      PARAMETER (ZERO=0.0D0)
      INTEGER IROWL1,L1
      DOUBLE PRECISION S1,S2
C*    Begin
      DO 1 L1=1,N
        FW(L1) = ZERO
1     CONTINUE
      DO 2 L1=1,NFILL
        IROWL1 = IROW(L1)
        S1 = FW(IROWL1)
        S2 = DABS(A(L1))
        IF (S2.GT.S1) FW(IROWL1)=S2
2     CONTINUE
      DO 3 L1=1,N
        IF (FW(L1).GT.ZERO) THEN
          FW(L1)=ONE/FW(L1)
        ELSE
          FW(L1)=ONE
        ENDIF
3     CONTINUE
      DO 4 L1=1,NFILL
        A(L1)=A(L1)*FW(IROW(L1))
4     CONTINUE
C     End of subroutine NISCRF
      RETURN
      END
C
      SUBROUTINE NISLVI(N,NFMAX,IOPT,IFAIL,LIWK,IWK,LPIWK,LTIWK,
     $                  LI2WK,I2WK,LPI2WK,LTI2WK,
     $                  LRWK,RWK,LPRWK,LTRWK)
C*    Begin Prologue SOLINI
      INTEGER N,NFMAX
      INTEGER IOPT(50)
      INTEGER IFAIL
      INTEGER LIWK
      INTEGER IWK(LIWK)
      INTEGER LPIWK,LTIWK,LI2WK
      INTEGER I2WK(LI2WK)
      INTEGER LPI2WK,LTI2WK,LRWK
      DOUBLE PRECISION RWK(LRWK)
      INTEGER LPRWK,LTRWK
C     ------------------------------------------------------------
C
C*    Summary :
C
C     S O L I N I : Initialize linear algebra subprograms for 
C                   factorization of a sparse (N,N)-matrix
C
C*    Input parameters (* marks inout parameters)
C     ===========================================
C
C     N             Int    Order of the linear system
C     NFMAX         Int    Dimension of the matrix array A -
C                          the maximum number of nonzero elements
C                          of the sparse matrix, NFMAX <= 32767
C     IOPT(50)      Int    Option vector passed from NLEQ1S
C
C*    Output parameters
C     =================
C
C     IFAIL         Int    Error indicator returned by this routine:
C                          = 0 matrix decomposition successfull
C                          =10 supplied workspace too small
C
C*    Workspace parameters
C     ====================
C
C     LIWK          Int    Length of integer workspace (In)
C     IWK(LIWK)     Int    Integer Workspace supplied for linear solver
C     LPIWK         Int    Length of integer Workspace used by 
C                          linear solver across successive calls - i.e.
C                          'permanent' (out)       
C     LTIWK         Int    Length of integer Workspace used by 
C                          linear solver - but not across successive
C                          calls - i.e. 'temporary' (out)       
C     LI2WK         Int    Length of short integer workspace (in) 
C     I2WK(LI2WK)   ShInt  Short integer Workspace supplied for 
C                          linear solver
C     LPI2WK        Int    Length of short integer Workspace used by 
C                          linear solver across successive calls - i.e.
C                          'permanent' (out)       
C     LTI2WK        Int    Length of short integer Workspace used by 
C                          linear solver - but not across successive
C                          calls - i.e. 'temporary' (out)       
C     LRWK          Int    Length of real workspace (in)      
C     RWK(LRWK)     Dble   Real Workspace supplied for linear solver
C     LPRWK         Int    Length of real Workspace used by 
C                          linear solver across successive calls - i.e.
C                          'permanent' (out)
C     LTRWK         Int    Length of real Workspace used by 
C                          linear solver - but not across successive
C                          calls - i.e. 'temporary' (out)
C
C*    Subroutines called:  none
C
C     ------------------------------------------------------------
C*    End Prologue
       INTRINSIC MIN0,IDINT
      INTEGER IRM1,IRM2
      LOGICAL QSUCC
      INTEGER ICNCP,IRNCP
      DOUBLE PRECISION THRSH1,THRSH2
      PARAMETER ( THRSH1=1.0D-2, THRSH2=1.0D-6 )
      DOUBLE PRECISION EPSQ,RESID,RMIN,UQ
      LOGICAL ABORT1,ABORT2,GROW,IBLOCK
      COMMON /MA28ED/ LP,MP,IBLOCK,GROW
      COMMON /MA28FD/ EPSQ,RMIN,RESID,IRNCP,ICNCP,MINIRN,MINICN,
     $IRANK,ABORT1,ABORT2
C*    Begin
      QSUCC = IOPT(1).EQ.1
      IFAIL = 0
      LTIWK = 8*N + 2
      LPIWK = 5
      LPI2WK=2*NFMAX+5*N
      LTI2WK=IDINT(1.5D0*NFMAX)
      LPRWK = 2*NFMAX
      LTRWK = N + 2
      IRM2  = LI2WK-(LPI2WK+LTI2WK)
      IRM1  = MIN0(IRM2, LRWK-(LPRWK+LTRWK))
      IF (IRM1.LT.0) THEN
        IFAIL=10
        IWK(1)=-IRM1
      ELSE
        LICN  = 2*NFMAX
        LIRN  = IDINT(1.5D0*NFMAX)
        LICADD  = MIN0(2*NFMAX,IRM1)
        LICN  = LICN+LICADD
        LIRADD  = MIN0(IDINT(0.5D0*NFMAX),IRM2-LICADD)
        LIRN  = LIRN+LIRADD
        IWK(LIWK) = LICN
        IWK(LIWK-1) = LIRN
        LPI2WK=LICN+5*N
        LTI2WK=LIRN
        LPRWK = LICN
        LTRWK = N + 2
      ENDIF
C     ( Real workspace starts at RWK(NRWKFR) )
      IF (LIWK.GE.LPIWK+LTIWK.AND.LRWK.GE.LPRWK+LTRWK
     $   .AND.LI2WK.GE.LPI2WK+LTI2WK.AND.IFAIL.EQ.0) THEN
        IF (.NOT.QSUCC) THEN
          IBLOCK  = .FALSE.
          EPSQ  = 10.D0*THRSH2
        ENDIF
        IF (IOPT(11).GT.0) THEN
          LP  = IOPT(12)
          MP  = IOPT(12)
        ELSE
          LP = 0
          MP = 0
        ENDIF
        UQ  = THRSH1
        RWK(LRWK)=UQ
        RWK(LRWK-1) = THRSH2
        DO 10 J=1,5
          IWK(J)=0
10      CONTINUE
      ELSE
        IFAIL=10
      ENDIF
      RETURN
      END
C
      SUBROUTINE NIFACT(N,NFILL,A,IROW,ICOL,IOPT,QSTRUC,IFAIL,LIWK,IWK,
     $LI2WK,I2WK,LRWK,RWK)
C*    Begin Prologue FACT
      INTEGER N,NFILL
      DOUBLE PRECISION A(NFILL)
      INTEGER IROW(NFILL),ICOL(NFILL)
      INTEGER IOPT(50)
      LOGICAL QSTRUC
      INTEGER IFAIL
      INTEGER LIWK
      INTEGER IWK(LIWK)
      INTEGER LI2WK
      INTEGER I2WK(LI2WK)
      INTEGER LRWK
      DOUBLE PRECISION RWK(LRWK)
C     ------------------------------------------------------------
C
C*    Summary :
C
C     F A C T : Call linear algebra subprogram for factorization of
C               a (N,N)-matrix
C
C*    Input parameters
C     ================
C
C     N             Int    Order of the linear system
C     NFILL         Int    Number of nonzero elements in the sparse
C                          matrix, NFILL <= NFMAX
C     A(NFMAX)      Dble   Matrix real evalues storage. 
C                          See routine NIINT.
C     IROW(NFMAX)   ShInt  Matrix row indices storage.
C     ICOL(NFMAX)   ShInt  Matrix column indices storage.
C     IOPT(50)      Int    Option vector passed from NLEQ1S
C     QSTRUC        Logic  (Input) Tells, if the sparse structure
C                          of the current matrix is different from 
C                          that of the matrix supplied in the previous
C                          call:
C                          = .TRUE.  : Structure is different
C                          = .FALSE. : Structure is the same as before
C
C*    Output parameters
C     =================
C
C     IFAIL         Int    Error indicator returned by this routine:
C                          = 0 matrix decomposition successfull
C                          = 1 singular matrix
C                          other : See IFLAG values of MA28 routines.
C
C*    Workspace parameters
C     ====================
C
C     LIWK          Int    Length of integer workspace (In)
C     IWK(LIWK)     Int    Integer Workspace supplied for linear solver
C     LI2WK         Int    Length of short integer workspace (in) 
C     I2WK(LI2WK)   ShInt  Short integer Workspace supplied for 
C                          linear solver
C     LRWK          Int    Length of real workspace (in)      
C     RWK(LRWK)     Dble   Real Workspace supplied for linear solver
C
C*    Subroutines called:  MA28AD, MA28BD
C
C     ------------------------------------------------------------
C*    End Prologue
      EXTERNAL MA28AD, MA28BD
      INTEGER NFILLK
      INTEGER LICN,LIRN,LIRBAS,MPRERR,LUERR
      INTEGER ICNCP,IRNCP
      INTEGER NDROP,MAXIT,NOITER,NSRCH,ISTART
      DOUBLE PRECISION EPSQ,RESID,RMIN
      DOUBLE PRECISION TOLQ,THEMAX,BIG,DXMAX,ERRMAX,DRES,CGCE
      LOGICAL ABORT1,ABORT2,GROW,IBLOCK,LBIG
      COMMON /MA28ED/ LP,MP,IBLOCK,GROW
      COMMON /MA28FD/ EPSQ,RMIN,RESID,IRNCP,ICNCP,MINIRN,MINICN,
     $                IRANK,ABORT1,ABORT2
      COMMON /MA28HD/ TOLQ,THEMAX,BIG,DXMAX,ERRMAX,DRES,CGCE,NDROP,
     $                MAXIT,NOITER,NSRCH,ISTART,LBIG
      CHARACTER*6 CRLIN
      DOUBLE PRECISION THRSH2
C*    Begin
      MPRERR = IOPT(11)
      LUERR = IOPT(12)
      MPRLIN = IOPT(17)
      LULIN = IOPT(18)
      MPRTIM = IOPT(19)
      IFAIL = 0
      LICN=IWK(LIWK)
      LIRN=IWK(LIWK-1)
      L1=LI2WK+1-LIRN
      THRSH2 = RWK(LRWK-1)
C     DO (Until)
1     CONTINUE
        NFILLK = NFILL
        DO 2 I=1,NFILL
          RWK(I)=A(I)
2       CONTINUE
        IF(QSTRUC)THEN
          LIRBAS=L1-1
          DO 3 I=1,NFILL
            I2WK(I+LIRBAS)=IROW(I)
            I2WK(I)=ICOL(I)
3         CONTINUE
          CRLIN = 'MA28AD'
          IF (MPRTIM.NE.0) CALL MONON(3)
          CALL MA28AD(N,NFILL,RWK(1),LICN,I2WK(L1),LIRN,I2WK(1),
     $                RWK(LRWK),I2WK(1+LICN),IWK(LIWK-8*N-1),
     $                RWK(LRWK-N-1),IFAIL)
          IF (MPRTIM.NE.0) CALL MONOFF(3)
          QSTRUC = .FALSE.
        ELSE
          CRLIN = 'MA28BD'
          IF (MPRTIM.NE.0) CALL MONON(4)
          CALL MA28BD(N,NFILL,RWK(1),LICN,IROW,ICOL,I2WK(1),
     $                I2WK(1+LICN),IWK(LIWK-8*N-1),RWK(LRWK-N-1),IFAIL)
          IF (MPRTIM.NE.0) CALL MONOFF(4)
          QSTRUC = RMIN.LT.THRSH2.OR.IFAIL.GT.0
        ENDIF
        IWK(1) = MAX(MINIRN,IWK(1))
        IWK(2) = MAX(MINICN,IWK(2))
        IWK(3) = MAX(IRNCP, IWK(3))
        IWK(4) = MAX(ICNCP, IWK(4))
        IWK(5) = IRANK
        IF (MPRLIN.GE.2) 
     $    WRITE(LULIN,23) CRLIN,NFILLK,FLOAT(NFILLK)/FLOAT(N**2)*100.0,
     $                    MINIRN,MINICN,IRNCP,ICNCP,IRANK
23      FORMAT(' ++ ',A6,' - Nonzeros:',I6,' Nonzeros percentage: ',
     $                F5.1,'++',/,
     $         ' ++ MINIRN: ',I6,' MINICN: ',I6,21X,' ++',/,
     $         ' ++ IRNCP:  ',I6,' ICNCP:  ',I6,' IRANK:  ',I6,6X,' ++')
        IF (MPRLIN.GE.4) THEN
          WRITE(LULIN,24) RWK(LRWK), EPSQ, TOLQ, NSRCH, 
     $                    RMIN, RESID, THEMAX, NDROP, IFAIL
24        FORMAT(' ++ Input  values - U: ',D10.3,' EPS: ',D10.3,
     $                ' TOL: ',D10.3,8X,' ++',/,
     $           ' ++',16X,' NSRCH: ',I6,40X,' ++',/,
     $           ' ++ Output values - RMIN: ',D10.3,' RESID: ',D10.3,
     $                ' THEMAX: ',D10.3,' ++',/,
     $           ' ++',16X,' NDROP: ',I6,' IFAIL: ',I6,26X,' ++')
        ENDIF
      IF(.NOT.(.NOT.QSTRUC)) GOTO 1
C     UNTIL ( expression - negated above)
      IF (IFAIL.EQ.-1) IFAIL=1
      IF(IFAIL.LE.-7.AND.MPRERR.GT.0)THEN
C       Improper Jacobian matrix
21      FORMAT(/,' Formal bad Jacobian matrix detected by sparse',
     $          ' solver.',/,'See the ',
     $         'above printed message for more details.',/)
        WRITE(LUERR,21)
      ENDIF
      IF(IFAIL.LE.-3.AND.MPRERR.GT.0)THEN
C       Workspace for sparse solver too small
22      FORMAT(/,' Short integer and real workspace for ',
     $  'sparse linear solver is too small',/,
     $  ' more requested for each: ICN - ',I6,',  IRN - ',I6)
        IC = MAX0(0,MINICN-LICN)
        IR = MAX0(0,MINIRN-LIRN)
        WRITE(LUERR,22) IC, IR
      ENDIF
      RETURN
      END
C
      SUBROUTINE NISOLV(N,NFILL,A,IROW,ICOL,B,IOPT,IFAIL,LIWK,IWK,
     $LI2WK,I2WK,LRWK,RWK)
C*    Begin Prologue SOLVE
      INTEGER N,NFILL
      DOUBLE PRECISION A(NFILL)
      INTEGER IROW(NFILL),ICOL(NFILL)
      DOUBLE PRECISION B(N)
      INTEGER IOPT(50)
      INTEGER IFAIL
      INTEGER LIWK
      INTEGER IWK(LIWK)
      INTEGER LI2WK
      INTEGER I2WK(LI2WK)
      INTEGER LRWK
      DOUBLE PRECISION RWK(LRWK)
C     ------------------------------------------------------------
C
C*    Summary :
C
C     S O L V E : Call linear algebra subprogram for solution of
C                 the linear system A*Z = B
C
C*    Parameters
C     ==========
C
C     N,NFILL,A,IROW,ICOL,IOPT,IFAIL,LIWK,IWK,LI2WK,I2WK,LRWK,RWK :
C                        See description for subroutine NIFACT.          
C     B(N)       Dble    In:  Right hand side of the linear system
C                        Out: Solution of the linear system
C
C     Subroutines called: MA28CD
C
C     ------------------------------------------------------------
C*    End Prologue
      EXTERNAL MA28CD
      INTEGER ICNCP,IRNCP
      DOUBLE PRECISION EPSQ,RESID,RMIN
      LOGICAL ABORT1,ABORT2,GROW,IBLOCK
      COMMON /MA28ED/ LP,MP,IBLOCK,GROW
      COMMON /MA28FD/ EPSQ,RMIN,RESID,IRNCP,ICNCP,MINIRN,MINICN,
     $IRANK,ABORT1,ABORT2
C*    Begin
      LICN=IWK(LIWK)
      MTYPE = 1
      CALL MA28CD(N,RWK(1),LICN,I2WK(1),I2WK(1+LICN),B,RWK(LRWK-N-1),
     $            MTYPE)
      IFAIL = 0
      RETURN
      END
C
      SUBROUTINE NILVLS(N,DX1,XW,F,DXQ,CONV,SUMX,DLEVF,MPRMON,QDSCAL)
C*    Begin Prologue LEVELS
      INTEGER N,MPRMON
      DOUBLE PRECISION DX1(N),XW(N),F(N),DXQ(N)
      DOUBLE PRECISION CONV,SUMX,DLEVF
      LOGICAL QDSCAL
C
C     ------------------------------------------------------------
C
C*    Summary :
C
C     L E V E L S : To be used in connection with NLEQ1S .
C     provides descaled solution, error norm and level functions
C
C*    Input parameters (* marks inout parameters)
C     ===========================================
C
C       N              Int    Number of parameters to be estimated
C       DX1(N)         Dble   array containing the scaled Newton
C                             correction
C       XW(N)          Dble   Array containing the scaling values
C       F(N)           Dble   Array containing the residuum
C
C*    Output parameters
C     =================
C
C       DXQ(N)         Dble   Array containing the descaled Newton
C                             correction
C       CONV           Dble   Scaled maximum norm of the Newton
C                             correction
C       SUMX           Dble   Scaled natural level function value
C       DLEVF          Dble   Standard level function value (only
C                             if needed for print)
C       MPRMON         Int    Print information parameter (see
C                             driver routine NLEQ1S )
C       QDSCAL         Logic  .TRUE., if descaling of DX1 required,
C                             else .FALSE.
C
C     ------------------------------------------------------------
C*    End Prologue
      INTRINSIC DABS
      DOUBLE PRECISION ZERO
      PARAMETER (ZERO=0.0D0)
      INTEGER L1
      DOUBLE PRECISION S1
C*    Begin
      IF (QDSCAL) THEN
C       ------------------------------------------------------------
C       1.2 Descaling of solution DX1 ( stored to DXQ )
        DO 12 L1=1,N
          DXQ(L1)=DX1(L1)*XW(L1)
12      CONTINUE
      ENDIF
C     ------------------------------------------------------------
C     2 Evaluation of scaled natural level function SUMX and
C       scaled maximum error norm CONV
      CONV = ZERO
      DO 20 L1=1,N
        S1 = DABS(DX1(L1))
        IF(S1.GT.CONV) CONV=S1
20    CONTINUE
      SUMX = ZERO
      DO 21 L1=1,N
        SUMX = SUMX+DX1(L1)**2
21    CONTINUE
C     ------------------------------------------------------------
C     3 Evaluation of (scaled) standard level function DLEVF (only
C       if needed for print)
      IF(MPRMON.GE.2)THEN
        DLEVF = ZERO
        DO 3 L1=1,N
          DLEVF = DLEVF+F(L1)**2
3       CONTINUE
        DLEVF = DSQRT(DLEVF/DBLE(FLOAT(N)))
      ENDIF
C     End of subroutine NILVLS
      RETURN
      END
C
      SUBROUTINE NIPRV1(DLEVF,DLEVX,FC,NITER,NEW,MPRMON,LUMON,QMIXIO)
C*    Begin Prologue NIPRV1
      DOUBLE PRECISION DLEVF,DLEVX,FC
      INTEGER NITER,MPRMON,LUMON
      LOGICAL QMIXIO
C     ------------------------------------------------------------
C
C*    Summary :
C
C     N I P R V 1 : Printing of intermediate values (Type 1 routine)
C
C*    Parameters
C     ==========
C
C     DLEVF, DLEVX   See descr. of internal double variables of NIINT
C     FC,NITER,NEW,MPRMON,LUMON
C                  See parameter descr. of subroutine NIINT
C     QMIXIO Logical  = .TRUE.  , if LUMON.EQ.LUSOL
C                     = .FALSE. , if LUMON.NE.LUSOL
C
C     ------------------------------------------------------------
C*    End Prologue
C     Print Standard - and natural level
      IF(QMIXIO)THEN
1       FORMAT(2X,66('*'))
        WRITE(LUMON,1)
2       FORMAT(8X,'It',7X,'Normf ',10X,'Normx ',20X,'New')
        IF (MPRMON.GE.3) WRITE(LUMON,2)
3       FORMAT(8X,'It',7X,'Normf ',10X,'Normx ',8X,'Damp.Fct.',3X,'New')
        IF (MPRMON.EQ.2) WRITE(LUMON,3)
      ENDIF
4     FORMAT(6X,I4,5X,D10.3,2X,4X,D10.3,17X,I2)
      IF (MPRMON.GE.3.OR.NITER.EQ.0) 
     $  WRITE(LUMON,4) NITER,DLEVF,DLEVX,NEW
5     FORMAT(6X,I4,5X,D10.3,6X,D10.3,6X,F7.5,4X,I2)
      IF (MPRMON.EQ.2.AND.NITER.NE.0) 
     $  WRITE(LUMON,5) NITER,DLEVF,DLEVX,FC,NEW
      IF(QMIXIO)THEN
6       FORMAT(2X,66('*'))
        WRITE(LUMON,6)
      ENDIF
C     End of subroutine NIPRV1
      RETURN
      END
C
      SUBROUTINE NIPRV2(DLEVF,DLEVX,FC,NITER,MPRMON,LUMON,QMIXIO,
     $                  CMARK)
C*    Begin Prologue NIPRV2
      DOUBLE PRECISION DLEVF,DLEVX,FC
      INTEGER NITER,MPRMON,LUMON
      LOGICAL QMIXIO
      CHARACTER*1 CMARK
C     ------------------------------------------------------------
C
C*    Summary :
C
C     N I P R V 2 : Printing of intermediate values (Type 2 routine)
C
C*    Parameters
C     ==========
C
C     DLEVF, DLEVX   See descr. of internal double variables of N2INT
C     FC,NITER,MPRMON,LUMON
C                  See parameter descr. of subroutine N2INT
C     QMIXIO Logical  = .TRUE.  , if LUMON.EQ.LUSOL
C                     = .FALSE. , if LUMON.NE.LUSOL
C     CMARK Char*1    Marker character to be printed before DLEVX
C
C     ------------------------------------------------------------
C*    End Prologue
C     Print Standard - and natural level, and damping
C     factor
      IF(QMIXIO)THEN
1       FORMAT(2X,66('*'))
        WRITE(LUMON,1)
2       FORMAT(8X,'It',7X,'Normf ',10X,'Normx ',8X,'Damp.Fct.')
        WRITE(LUMON,2)
      ENDIF
3     FORMAT(6X,I4,5X,D10.3,4X,A1,1X,D10.3,2X,4X,F7.5)
      WRITE(LUMON,3)NITER,DLEVF,CMARK,DLEVX,FC
      IF(QMIXIO)THEN
4       FORMAT(2X,66('*'))
        WRITE(LUMON,4)
      ENDIF
C     End of subroutine NIPRV2
      RETURN
      END
C
      SUBROUTINE NISOUT(N,X,MODE,IOPT,RWK,NRW,IWK,NIW,MPRINT,LUOUT)
C*    Begin Prologue SOLOUT
      INTEGER N
      DOUBLE PRECISION X(N)
      INTEGER NRW
      INTEGER MODE
      INTEGER IOPT(50)
      DOUBLE PRECISION RWK(NRW)
      INTEGER NIW
      INTEGER IWK(NIW)
      INTEGER MPRINT,LUOUT
C     ------------------------------------------------------------
C
C*    Summary :
C
C     S O L O U T : Printing of iterate (user customizable routine)
C
C*    Input parameters
C     ================
C
C     N         Int Number of equations/unknowns
C     X(N)   Dble   iterate vector
C     MODE          =1 This routine is called before the first
C                      Newton iteration step
C                   =2 This routine is called with an intermedi-
C                      ate iterate X(N)
C                   =3 This is the last call with the solution
C                      vector X(N)
C                   =4 This is the last call with the final, but
C                      not solution vector X(N)
C     IOPT(50)  Int The option array as passed to the driver
C                   routine(elements 46 to 50 may be used
C                   for user options)
C     MPRINT    Int Solution print level 
C                   (see description of IOPT-field MPRINT)
C     LUOUT     Int the solution print unit 
C                   (see description of see IOPT-field LUSOL)
C
C
C*    Workspace parameters
C     ====================
C
C     NRW, RWK, NIW, IWK    see description in driver routine
C
C*    Use of IOPT by this routine
C     ===========================
C
C     Field 46:       =0 Standard output
C                     =1 GRAZIL suitable output
C
C     ------------------------------------------------------------
C*    End Prologue
      LOGICAL QGRAZ,QNORM
C*    Begin
      QNORM = IOPT(46).EQ.0
      QGRAZ = IOPT(46).EQ.1
      IF(QNORM) THEN
1        FORMAT('  ',A,' data:',/)
         IF (MODE.EQ.1) THEN
101        FORMAT('  Start data:',/,'  N =',I5,//,
     $            '  Format: iteration-number, (x(i),i=1,...N), ',
     $            'Normf , Normx ',/)
           WRITE(LUOUT,101) N
           WRITE(LUOUT,1) 'Initial'
         ELSE IF (MODE.EQ.3) THEN
           WRITE(LUOUT,1) 'Solution'
         ELSE IF (MODE.EQ.4) THEN
           WRITE(LUOUT,1) 'Final'
         ENDIF
2        FORMAT(' ',I5)
C        WRITE          NITER
         WRITE(LUOUT,2) IWK(1)
3        FORMAT((12X,3(D18.10,1X)))
         WRITE(LUOUT,3)(X(L1),L1=1,N)
C        WRITE          DLEVF,  DLEVX
         WRITE(LUOUT,3) RWK(19),DSQRT(RWK(18)/DBLE(FLOAT(N)))
         IF(MODE.EQ.1.AND.MPRINT.GE.3) THEN
           WRITE(LUOUT,1) 'Intermediate'
         ELSE IF(MODE.GE.3) THEN
           WRITE(LUOUT,1) 'End'
         ENDIF
      ENDIF
      IF(QGRAZ) THEN
        IF(MODE.EQ.1) THEN
10        FORMAT('&name com',I3.3,:,255(7(', com',I3.3,:),/))
          WRITE(LUOUT,10)(I,I=1,N+2)
15        FORMAT('&def  com',I3.3,:,255(7(', com',I3.3,:),/))
          WRITE(LUOUT,15)(I,I=1,N+2)
16        FORMAT(6X,': X=1, Y=',I3)
          WRITE(LUOUT,16) N+2
        ENDIF
20      FORMAT('&data ',I5)
C        WRITE          NITER
        WRITE(LUOUT,20) IWK(1) 
21      FORMAT((6X,4(D18.10)))
        WRITE(LUOUT,21)(X(L1),L1=1,N)
C        WRITE          DLEVF,  DLEVX
        WRITE(LUOUT,21) RWK(19),DSQRT(RWK(18)/DBLE(FLOAT(N)))
        IF(MODE.GE.3) THEN
30        FORMAT('&wktype 3111',/,'&atext x ''iter''')
          WRITE(LUOUT,30)
35        FORMAT('&vars = com',I3.3,/,'&atext y ''x',I3,'''',
     $           /,'&run')
          WRITE(LUOUT,35) (I,I,I=1,N)
36        FORMAT('&vars = com',I3.3,/,'&atext y ''',A,'''',
     $           /,'&run')
          WRITE(LUOUT,36) N+1,'Normf ',N+2,'Normx '
C39       FORMAT('&stop')
C         WRITE(LUOUT,39)
        ENDIF
      ENDIF
C     End of subroutine NISOUT
      RETURN
      END
      DOUBLE PRECISION FUNCTION WNORM(N,Z,XW)
      INTEGER N
      DOUBLE PRECISION Z(N), XW(N)
C     ------------------------------------------------------------
C
C*    Summary :
C
C     E N O R M : Return the norm to be used in exit (termination)
C                 criteria
C
C*    Input parameters
C     ================
C
C     N         Int Number of equations/unknowns
C     Z(N)     Dble  The vector, of which the norm is to be computed
C     XW(N)    Dble  The scaling values of Z(N)
C
C*    Output
C     ======
C
C     WNORM(N,Z,XW)  Dble  The mean square root norm of Z(N) subject
C                          to the scaling values in XW(N):
C                          = Sqrt( Sum(1,...N)((Z(I)/XW(I))**2) / N )
C
C     ------------------------------------------------------------
C*    End Prologue
      INTEGER I
      DOUBLE PRECISION S
C*    Begin
      S = 0.0D0
      DO 10 I=1,N
        S = S + ( Z(I)/XW(I) ) ** 2
10    CONTINUE
      WNORM = DSQRT( S / DBLE(FLOAT(N)) )
C     End of function WNORM
      RETURN
      END
C*    End package
C
C
C*    Group  Sparse Linear System Solver MA28 from HARWELL Library
C
C I AND J ARE IBM FORTRAN DOUBLE AND SINGLE LENGTH VERSIONS  ISDJ/
C D AND S ARE STANDARD FORTRAN DOUBLE AND SINGLE LENGTH VERSIONS
C     SUBROUTINE MC20A (NC,MAXA,A,INUM,JPTR,JNUM,JDISP)              JS/
      SUBROUTINE MC20AD(NC,MAXA,A,INUM,JPTR,JNUM,JDISP)
C
      INTEGER   INUM(MAXA),JNUM(MAXA)                                
C     INTEGER*2 INUM(MAXA),JNUM(MAXA)                                DI/
C     REAL A(MAXA)                                                   JS/
      DOUBLE PRECISION A(MAXA),ACE,ACEP
      DIMENSION JPTR(NC)
C
C     ******************************************************************
C
      NULL=-JDISP
C**      CLEAR JPTR
      DO 60 J=1,NC
   60 JPTR(J)=0
C**      COUNT THE NUMBER OF ELEMENTS IN EACH COLUMN.
      DO 120 K=1,MAXA
      J=JNUM(K)+JDISP
      JPTR(J)=JPTR(J)+1
  120 CONTINUE
C**      SET THE JPTR ARRAY
      K=1
      DO 150 J=1,NC
      KR=K+JPTR(J)
      JPTR(J)=K
  150 K=KR
C
C**      REORDER THE ELEMENTS INTO COLUMN ORDER.  THE ALGORITHM IS AN
C        IN-PLACE SORT AND IS OF ORDER MAXA.
      DO 230 I=1,MAXA
C        ESTABLISH THE CURRENT ENTRY.
      JCE=JNUM(I)+JDISP
      IF(JCE.EQ.0) GO TO 230
      ACE=A(I)
      ICE=INUM(I)
C        CLEAR THE LOCATION VACATED.
      JNUM(I)=NULL
C        CHAIN FROM CURRENT ENTRY TO STORE ITEMS.
      DO 200 J=1,MAXA
C        CURRENT ENTRY NOT IN CORRECT POSITION.  DETERMINE CORRECT
C        POSITION TO STORE ENTRY.
      LOC=JPTR(JCE)
      JPTR(JCE)=JPTR(JCE)+1
C        SAVE CONTENTS OF THAT LOCATION.
      ACEP=A(LOC)
      ICEP=INUM(LOC)
      JCEP=JNUM(LOC)
C        STORE CURRENT ENTRY.
      A(LOC)=ACE
      INUM(LOC)=ICE
      JNUM(LOC)=NULL
C        CHECK IF NEXT CURRENT ENTRY NEEDS TO BE PROCESSED.
      IF(JCEP.EQ.NULL) GO TO 230
C        IT DOES.  COPY INTO CURRENT ENTRY.
      ACE=ACEP
      ICE=ICEP
  200 JCE=JCEP+JDISP
C
  230 CONTINUE
C
C**      RESET JPTR VECTOR.
      JA=1
      DO 250 J=1,NC
      JB=JPTR(J)
      JPTR(J)=JA
  250 JA=JB
      RETURN
      END
C     SUBROUTINE MC20B(NC,MAXA,A,INUM,JPTR)                          JS/
      SUBROUTINE MC20BD(NC,MAXA,A,INUM,JPTR)
C     REAL A(MAXA)                                                   JS/
      DOUBLE PRECISION A(MAXA),ACE
      INTEGER   INUM(MAXA)
C     INTEGER*2 INUM(MAXA)                                           DS/
      DIMENSION JPTR(NC)
C
C     ******************************************************************
C
      KMAX=MAXA
      DO 30 JJ=1,NC
      J=NC+1-JJ
      KLO=JPTR(J)+1
      IF(KLO.GT.KMAX)GO TO 30
      KOR=KMAX
      DO 25 KDUMMY=KLO,KMAX
C ITEMS KOR, KOR+1, .... ,KMAX ARE IN ORDER
      ACE=A(KOR-1)
      ICE=INUM(KOR-1)
      DO 10 K=KOR,KMAX
      IK=INUM(K)
      IF(IABS(ICE).LE.IABS(IK))GO TO 20
      INUM(K-1)=IK
10    A(K-1)=A(K)
      K=KMAX+1
20    INUM(K-1)=ICE
      A(K-1)=ACE
25    KOR=KOR-1
C        NEXT COLUMN
30    KMAX=KLO-2
      RETURN
      END
C I IS THE IBM FORTRAN VERSION                                       IS/
C S IS THE STANDARD FORTRAN VERSION
      SUBROUTINE MC13D(N,ICN,LICN,IP,LENR,IOR,IB,NUM,IW)
      INTEGER IP(N)
      INTEGER ICN(LICN),LENR(N),IOR(N),IB(N),IW(N,3)
C     INTEGER*2 ICN(LICN),LENR(N),IOR(N),IB(N),IW(N,3)                I/
      CALL MC13E(N,ICN,LICN,IP,LENR,IOR,IB,NUM,IW(1,1),IW(1,2),IW(1,3))
      RETURN
      END
      SUBROUTINE MC13E(N,ICN,LICN,IP,LENR,ARP,IB,NUM,LOWL,NUMB,PREV)
      INTEGER STP,DUMMY
      INTEGER IP(N)
C
C ARP(I) IS ONE LESS THAN THE NUMBER OF UNSEARCHED EDGES LEAVING
C     NODE I.  AT THE END OF THE ALGORITHM IT IS SET TO A
C     PERMUTATION WHICH PUTS THE MATRIX IN BLOCK LOWER
C     TRIANGULAR FORM.
C IB(I) IS THE POSITION IN THE ORDERING OF THE START OF THE ITH
C     BLOCK.  IB(N+1-I) HOLDS THE NODE NUMBER OF THE ITH NODE
C     ON THE STACK.
C LOWL(I) IS THE SMALLEST STACK POSITION OF ANY NODE TO WHICH A PATH
C     FROM NODE I HAS BEEN FOUND.  IT IS SET TO N+1 WHEN NODE I
C     IS REMOVED FROM THE STACK.
C NUMB(I) IS THE POSITION OF NODE I IN THE STACK IF IT IS ON
C     IT, IS THE PERMUTED ORDER OF NODE I FOR THOSE NODES
C     WHOSE FINAL POSITION HAS BEEN FOUND AND IS OTHERWISE ZERO.
C PREV(I) IS THE NODE AT THE END OF THE PATH WHEN NODE I WAS
C     PLACED ON THE STACK.
      INTEGER ICN(LICN),LENR(N),ARP(N),IB(N),LOWL(N),NUMB(N),          
     1PREV(N)                                                          
C     INTEGER *2 ICN(LICN),LENR(N),ARP(N),IB(N),LOWL(N),NUMB(N),     I/
C    1PREV(N)                                                        I/
C
C
C   ICNT IS THE NUMBER OF NODES WHOSE POSITIONS IN FINAL ORDERING HAVE
C     BEEN FOUND.
      ICNT=0
C NUM IS THE NUMBER OF BLOCKS THAT HAVE BEEN FOUND.
      NUM=0
      NNM1=N+N-1
C
C INITIALIZATION OF ARRAYS.
      DO 20 J=1,N
      NUMB(J)=0
      ARP(J)=LENR(J)-1
   20 CONTINUE
C
C
      DO 120 ISN=1,N
C LOOK FOR A STARTING NODE
      IF (NUMB(ISN).NE.0) GO TO 120
      IV=ISN
C IST IS THE NUMBER OF NODES ON THE STACK ... IT IS THE STACK POINTER.
      IST=1
C PUT NODE IV AT BEGINNING OF STACK.
      LOWL(IV)=1
      NUMB(IV)=1
      IB(N)=IV
C
C THE BODY OF THIS LOOP PUTS A NEW NODE ON THE STACK OR BACKTRACKS.
      DO 110 DUMMY=1,NNM1
      I1=ARP(IV)
C HAVE ALL EDGES LEAVING NODE IV BEEN SEARCHED.
      IF (I1.LT.0) GO TO 60
      I2=IP(IV)+LENR(IV)-1
      I1=I2-I1
C
C LOOK AT EDGES LEAVING NODE IV UNTIL ONE ENTERS A NEW NODE OR
C     ALL EDGES ARE EXHAUSTED.
      DO 50 II=I1,I2
      IW=ICN(II)
C HAS NODE IW BEEN ON STACK ALREADY.
      IF (NUMB(IW).EQ.0) GO TO 100
C UPDATE VALUE OF LOWL(IV) IF NECESSARY.
C 50  LOWL(IV)=MIN0(LOWL(IV),LOWL(IW))                      WEI;      S/
   50 IF (LOWL(IW).LT.LOWL(IV)) LOWL(IV)=LOWL(IW)
C
C THERE ARE NO MORE EDGES LEAVING NODE IV.
      ARP(IV)=-1
C IS NODE IV THE ROOT OF A BLOCK.
   60 IF (LOWL(IV).LT.NUMB(IV)) GO TO 90
C
C ORDER NODES IN A BLOCK.
      NUM=NUM+1
      IST1=N+1-IST
      LCNT=ICNT+1
C PEEL BLOCK OFF THE TOP OF THE STACK STARTING AT THE TOP AND
C     WORKING DOWN TO THE ROOT OF THE BLOCK.
      DO 70 STP=IST1,N
      IW=IB(STP)
      LOWL(IW)=N+1
      ICNT=ICNT+1
      NUMB(IW)=ICNT
      IF (IW.EQ.IV) GO TO 80
   70 CONTINUE
   80 IST=N-STP
      IB(NUM)=LCNT
C ARE THERE ANY NODES LEFT ON THE STACK.
      IF (IST.NE.0) GO TO 90
C HAVE ALL THE NODES BEEN ORDERED.
      IF (ICNT.LT.N) GO TO 120
      GO TO 130
C
C BACKTRACK TO PREVIOUS NODE ON PATH.
   90 IW=IV
      IV=PREV(IV)
C UPDATE VALUE OF LOWL(IV) IF NECESSARY.
C     LOWL(IV)=MIN0(LOWL(IV),LOWL(IW))                  WEI;          S/
      IF (LOWL(IW).LT.LOWL(IV)) LOWL(IV)=LOWL(IW)
      GO TO 110
C
C PUT NEW NODE ON THE STACK.
 100  ARP(IV)=I2-II-1
      PREV(IW)=IV
      IV=IW
      IST=IST+1
      LOWL(IV)=IST
      NUMB(IV)=IST
      K=N+1-IST
      IB(K)=IV
  110 CONTINUE
C
  120 CONTINUE
C
C
C PUT PERMUTATION IN THE REQUIRED FORM.
  130 DO 140 I=1,N
      II=NUMB(I)
 140  ARP(II)=I
      RETURN
      END
C I AND J ARE IBM FORTRAN SINGLE AND DOUBLE PRECISION CODES RESP.  JISD/
C S AND D ARE STANDARD FORTRAN SINGLE AND DOUBLE PRECISION CODES RESP.
C     SUBROUTINE MC22A(N,ICN,A,NZ,LENROW,IP,IQ,IW,IW1)               IS/
      SUBROUTINE MC22AD(N,ICN,A,NZ,LENROW,IP,IQ,IW,IW1)
C     REAL A(NZ)                                                     IS/
      DOUBLE PRECISION A(NZ),AVAL
      INTEGER IW(N,2)
      INTEGER   ICN(NZ),LENROW(N),IP(N),IQ(N),IW1(NZ)
C     INTEGER*2 ICN(NZ),LENROW(N),IP(N),IQ(N),IW1(NZ)                ID/
      IF (NZ.LE.0) GO TO 1000
      IF (N.LE.0) GO TO 1000
C SET START OF ROW I IN IW(I,1) AND LENROW(I) IN IW(I,2)
      IW(1,1)=1
      IW(1,2)=LENROW(1)
      DO 10 I=2,N
      IW(I,1)=IW(I-1,1)+LENROW(I-1)
 10   IW(I,2)=LENROW(I)
C PERMUTE LENROW ACCORDING TO IP.  SET OFF-SETS FOR NEW POSITION
C     OF ROW IOLD IN IW(IOLD,1) AND PUT OLD ROW INDICES IN IW1 IN
C     POSITIONS CORRESPONDING TO THE NEW POSITION OF THIS ROW IN A/ICN.
      JJ=1
      DO 20 I=1,N
      IOLD=IP(I)
      IOLD=IABS(IOLD)
      LENGTH=IW(IOLD,2)
      LENROW(I)=LENGTH
      IF (LENGTH.EQ.0) GO TO 20
      IW(IOLD,1)=IW(IOLD,1)-JJ
      J2=JJ+LENGTH-1
      DO 15 J=JJ,J2
 15   IW1(J)=IOLD
      JJ=J2+1
 20   CONTINUE
C SET INVERSE PERMUTATION TO IQ IN IW(.,2).
      DO 30 I=1,N
      IOLD=IQ(I)
      IOLD=IABS(IOLD)
 30   IW(IOLD,2)=I
C PERMUTE A AND ICN IN PLACE, CHANGING TO NEW COLUMN NUMBERS.
C
C ***   MAIN LOOP   ***
C EACH PASS THROUGH THIS LOOP PLACES A CLOSED CHAIN OF COLUMN INDICES
C     IN THEIR NEW (AND FINAL) POSITIONS ... THIS IS RECORDED BY
C     SETTING THE IW1 ENTRY TO ZERO SO THAT ANY WHICH ARE SUBSEQUENTLY
C     ENCOUNTERED DURING THIS MAJOR SCAN CAN BE BYPASSED.
      DO 200 I=1,NZ
      IOLD=IW1(I)
      IF (IOLD.EQ.0) GO TO 200
      IPOS=I
      JVAL=ICN(I)
C IF ROW IOLD IS IN SAME POSITIONS AFTER PERMUTATION GO TO 150.
      IF (IW(IOLD,1).EQ.0) GO TO 150
      AVAL=A(I)
C **  CHAIN LOOP  **
C EACH PASS THROUGH THIS LOOP PLACES ONE (PERMUTED) COLUMN INDEX
C     IN ITS FINAL POSITION  .. VIZ. IPOS.
      DO 100 ICHAIN=1,NZ
C NEWPOS IS THE ORIGINAL POSITION IN A/ICN OF THE ELEMENT TO BE PLACED
C IN POSITION IPOS.  IT IS ALSO THE POSITION OF THE NEXT ELEMENT IN
C     THE CHAIN.
      NEWPOS=IPOS+IW(IOLD,1)
C IS CHAIN COMPLETE ?
      IF (NEWPOS.EQ.I) GO TO 130
      A(IPOS)=A(NEWPOS)
      JNUM=ICN(NEWPOS)
      ICN(IPOS)=IW(JNUM,2)
      IPOS=NEWPOS
      IOLD=IW1(IPOS)
      IW1(IPOS)=0
C **  END OF CHAIN LOOP  **
 100  CONTINUE
 130  A(IPOS)=AVAL
 150  ICN(IPOS)=IW(JVAL,2)
C ***   END OF MAIN LOOP   ***
 200  CONTINUE
C
 1000 RETURN
      END
C J AND I ARE IBM DOUBLE AND SINGLE VERSIONS.    JISD/
C D AND S ARE STANDARD FORTRAN DOUBLE AND SINGLE VERSIONS.
C     SUBROUTINE MC23A(N,ICN,A,LICN,LENR,IDISP,IP,IQ,LENOFF,IW,IW1)  IS/
      SUBROUTINE MC23AD(N,ICN,A,LICN,LENR,IDISP,IP,IQ,LENOFF,IW,IW1)
C     REAL A(LICN)                                                   IS/
      DOUBLE PRECISION A(LICN)
      INTEGER IDISP(2),IW1(N,2)
      LOGICAL ABORT
      INTEGER   ICN(LICN),LENR(N),IP(N),IQ(N),LENOFF(N),IW(N,5)
C     INTEGER*2 ICN(LICN),LENR(N),IP(N),IQ(N),LENOFF(N),IW(N,5)      ID/
C INPUT ... N,ICN .. A,ICN,LENR ....
C
C SET UP POINTERS IW(.,1) TO THE BEGINNING OF THE ROWS AND SET LENOFF
C     EQUAL TO LENR.
C     COMMON /MC23B/ LP,NUMNZ,NUM,LARGE,ABORT                        IS/
      COMMON /MC23BD/ LP,NUMNZ,NUM,LARGE,ABORT
      SAVE /MC23BD/
      IW1(1,1)=1
      LENOFF(1)=LENR(1)
      IF (N.EQ.1) GO TO 20
      DO 10 I=2,N
      LENOFF(I)=LENR(I)
   10 IW1(I,1)=IW1(I-1,1)+LENR(I-1)
C IDISP(1) POINTS TO THE FIRST POSITION IN A/ICN AFTER THE
C     OFF-DIAGONAL BLOCKS AND UNTREATED ROWS.
   20 IDISP(1)=IW1(N,1)+LENR(N)
C
C FIND ROW PERMUTATION IP TO MAKE DIAGONAL ZERO-FREE.
      CALL MC21A(N,ICN,LICN,IW1,LENR,IP,NUMNZ,IW)
C
C POSSIBLE ERROR RETURN FOR STRUCTURALLY SINGULAR MATRICES.
      IF (NUMNZ.NE.N.AND.ABORT) GO TO 170
C
C IW1(.,2) AND LENR ARE PERMUTATIONS OF IW1(.,1) AND LENR/LENOFF
C     SUITABLE FOR ENTRY
C     TO MC13D SINCE MATRIX WITH THESE ROW POINTER AND LENGTH ARRAYS
C     HAS MAXIMUM NUMBER OF NON-ZEROS ON THE DIAGONAL.
      DO 30 II=1,N
      I=IP(II)
      IW1(II,2)=IW1(I,1)
   30 LENR(II)=LENOFF(I)
C
C FIND SYMMETRIC PERMUTATION IQ TO BLOCK LOWER TRIANGULAR FORM.
      CALL MC13D(N,ICN,LICN,IW1(1,2),LENR,IQ,IW(1,4),NUM,IW)
C
      IF (NUM.NE.1) GO TO 60
C
C ACTION TAKEN IF MATRIX IS IRREDUCIBLE.
C WHOLE MATRIX IS JUST MOVED TO THE END OF THE STORAGE.
      DO 40 I=1,N
      LENR(I)=LENOFF(I)
      IP(I)=I
   40 IQ(I)=I
      LENOFF(1)=-1
C IDISP(1) IS THE FIRST POSITION AFTER THE LAST ELEMENT IN THE
C     OFF-DIAGONAL BLOCKS AND UNTREATED ROWS.
      NZ=IDISP(1)-1
      IDISP(1)=1
C IDISP(2) IS THE POSITION IN A/ICN OF THE FIRST ELEMENT IN THE
C     DIAGONAL BLOCKS.
      IDISP(2)=LICN-NZ+1
      LARGE=N
      IF (NZ.EQ.LICN) GO TO 230
      DO 50 K=1,NZ
      J=NZ-K+1
      JJ=LICN-K+1
      A(JJ)=A(J)
   50 ICN(JJ)=ICN(J)
C 230 = RETURN
      GO TO 230
C
C DATA STRUCTURE REORDERED.
C
C FORM COMPOSITE ROW PERMUTATION ... IP(I) = IP(IQ(I)).
   60 DO 70 II=1,N
      I=IQ(II)
   70 IW(II,1)=IP(I)
      DO 80 I=1,N
   80 IP(I)=IW(I,1)
C
C RUN THROUGH BLOCKS IN REVERSE ORDER SEPARATING DIAGONAL BLOCKS
C     WHICH ARE MOVED TO THE END OF THE STORAGE.  ELEMENTS IN
C     OFF-DIAGONAL BLOCKS ARE LEFT IN PLACE UNLESS A COMPRESS IS
C     NECESSARY.
C
C IBEG INDICATES THE LOWEST VALUE OF J FOR WHICH ICN(J) HAS BEEN
C     SET TO ZERO WHEN ELEMENT IN POSITION J WAS MOVED TO THE
C     DIAGONAL BLOCK PART OF STORAGE.
      IBEG=LICN+1
C IEND IS THE POSITION OF THE FIRST ELEMENT OF THOSE TREATED ROWS
C     WHICH ARE IN DIAGONAL BLOCKS.
      IEND=LICN+1
C LARGE IS THE DIMENSION OF THE LARGEST BLOCK ENCOUNTERED SO FAR.
      LARGE=0
C
C NUM IS THE NUMBER OF DIAGONAL BLOCKS.
      DO 150 K=1,NUM
      IBLOCK=NUM-K+1
C I1 IS FIRST ROW (IN PERMUTED FORM) OF BLOCK IBLOCK.
C I2 IS LAST ROW (IN PERMUTED FORM) OF BLOCK IBLOCK.
      I1=IW(IBLOCK,4)
      I2=N
      IF (K.NE.1) I2=IW(IBLOCK+1,4)-1
      LARGE=MAX0(LARGE,I2-I1+1)
C GO THROUGH THE ROWS OF BLOCK IBLOCK IN THE REVERSE ORDER.
      DO 140 II=I1,I2
      INEW=I2-II+I1
C WE NOW DEAL WITH ROW INEW IN PERMUTED FORM (ROW IOLD IN ORIGINAL
C     MATRIX).
      IOLD=IP(INEW)
C IF THERE IS SPACE TO MOVE UP DIAGONAL BLOCK PORTION OF ROW GO TO 110
      IF (IEND-IDISP(1).GE.LENOFF(IOLD)) GO TO 110
C
C IN-LINE COMPRESS.
C MOVES SEPARATED OFF-DIAGONAL ELEMENTS AND UNTREATED ROWS TO
C     FRONT OF STORAGE.
      JNPOS=IBEG
      ILEND=IDISP(1)-1
      IF (ILEND.LT.IBEG) GO TO 190
      DO 90 J=IBEG,ILEND
      IF (ICN(J).EQ.0) GO TO 90
      ICN(JNPOS)=ICN(J)
      A(JNPOS)=A(J)
      JNPOS=JNPOS+1
   90 CONTINUE
      IDISP(1)=JNPOS
      IF (IEND-JNPOS.LT.LENOFF(IOLD)) GO TO 190
      IBEG=LICN+1
C RESET POINTERS TO THE BEGINNING OF THE ROWS.
      DO 100 I=2,N
  100 IW1(I,1)=IW1(I-1,1)+LENOFF(I-1)
C
C ROW IOLD IS NOW SPLIT INTO DIAG. AND OFF-DIAG. PARTS.
  110 IROWB=IW1(IOLD,1)
      LENI=0
      IROWE=IROWB+LENOFF(IOLD)-1
C BACKWARD SCAN OF WHOLE OF ROW IOLD (IN ORIGINAL MATRIX).
      IF (IROWE.LT.IROWB) GO TO 130
      DO 120 JJ=IROWB,IROWE
      J=IROWE-JJ+IROWB
      JOLD=ICN(J)
C IW(.,2) HOLDS THE INVERSE PERMUTATION TO IQ.
C     ..... IT WAS SET TO THIS IN MC13D.
      JNEW=IW(JOLD,2)
C IF (JNEW.LT.I1) THEN ....
C ELEMENT IS IN OFF-DIAGONAL BLOCK AND SO IS LEFT IN SITU.
      IF (JNEW.LT.I1) GO TO 120
C ELEMENT IS IN DIAGONAL BLOCK AND IS MOVED TO THE END OF THE STORAGE.
      IEND=IEND-1
      A(IEND)=A(J)
      ICN(IEND)=JNEW
      IBEG=MIN0(IBEG,J)
      ICN(J)=0
      LENI=LENI+1
  120 CONTINUE
C
      LENOFF(IOLD)=LENOFF(IOLD)-LENI
  130 LENR(INEW)=LENI
  140 CONTINUE
C
      IP(I2)=-IP(I2)
  150 CONTINUE
C RESETS IP(N) TO POSITIVE VALUE.
      IP(N)=-IP(N)
C IDISP(2) IS POSITION OF FIRST ELEMENT IN DIAGONAL BLOCKS.
      IDISP(2)=IEND
C
C THIS COMPRESS IS USED TO MOVE ALL OFF-DIAGONAL ELEMENTS TO THE
C     FRONT OF THE STORAGE.
      IF (IBEG.GT.LICN) GO TO 230
      JNPOS=IBEG
      ILEND=IDISP(1)-1
      DO 160 J=IBEG,ILEND
      IF (ICN(J).EQ.0) GO TO 160
      ICN(JNPOS)=ICN(J)
      A(JNPOS)=A(J)
      JNPOS=JNPOS+1
  160 CONTINUE
C IDISP(1) IS FIRST POSITION AFTER LAST ELEMENT OF OFF-DIAGONAL BLOCKS.
      IDISP(1)=JNPOS
      GO TO 230
C
C
C ERROR RETURN
  170 IF (LP.NE.0) WRITE(LP,180) NUMNZ
  180 FORMAT(33X,41H MATRIX IS STRUCTURALLY SINGULAR, RANK = ,I6)
      IDISP(1)=-1
      GO TO 210
  190 IF (LP.NE.0) WRITE(LP,200) N
  200 FORMAT(33X,33H LICN NOT BIG ENOUGH INCREASE BY ,I6)
      IDISP(1)=-2
  210 IF (LP.NE.0) WRITE(LP,220)
C
C 220 FORMAT(33H+ERROR RETURN FROM MC23A  BECAUSE)                   IS/
  220 FORMAT(33H+ERROR RETURN FROM MC23AD BECAUSE)
  230 RETURN
      END
C J AND I ARE IBM DOUBLE AND SINGLE VERSIONS      JISD/
C D AND S ARE STANDARD FORTRAN DOUBLE AND SINGLE VERSIONS.
C     SUBROUTINE MC24A(N,ICN,A,LICN,LENR,LENRL,W)  IS/
      SUBROUTINE MC24AD(N,ICN,A,LICN,LENR,LENRL,W)
C     REAL A(LICN),W(N)  IS/
      DOUBLE PRECISION A(LICN),W(N),AMAXL,WROWL,AMAXU,ZERO
      INTEGER   ICN(LICN),LENR(N),LENRL(N)
C     INTEGER*2 ICN(LICN),LENR(N),LENRL(N)                           ID/
C     DATA ZERO/0.0E0/  IS/
      DATA ZERO/0.0D0/
      AMAXL=ZERO
      DO 10 I=1,N
 10   W(I)=ZERO
      J0=1
      DO 100 I=1,N
      IF (LENR(I).EQ.0) GO TO 100
      J2=J0+LENR(I)-1
      IF (LENRL(I).EQ.0) GO TO 50
C CALCULATION OF 1-NORM OF L.
      J1=J0+LENRL(I)-1
      WROWL=ZERO
      DO 30 JJ=J0,J1
C30   WROWL=WROWL+ABS(A(JJ))  IS/
 30   WROWL=WROWL+DABS(A(JJ))
C AMAXL IS THE MAXIMUM NORM OF COLUMNS OF L SO FAR FOUND.
C     AMAXL=AMAX1(AMAXL,WROWL)  IS/
      AMAXL=DMAX1(AMAXL,WROWL)
      J0=J1+1
C CALCULATION OF NORMS OF COLUMNS OF U (MAX-NORMS).
 50   J0=J0+1
      IF (J0.GT.J2) GO TO 90
      DO 80 JJ=J0,J2
      J=ICN(JJ)
C80   W(J)=AMAX1(ABS(A(JJ)),W(J))  IS/
 80   W(J)=DMAX1(DABS(A(JJ)),W(J))
 90   J0=J2+1
 100  CONTINUE
C AMAXU IS SET TO MAXIMUM MAX-NORM OF COLUMNS OF U.
      AMAXU=ZERO
      DO 200 I=1,N
C200  AMAXU=AMAX1(AMAXU,W(I))  IS/
 200  AMAXU=DMAX1(AMAXU,W(I))
C GROFAC IS MAX U MAX-NORM TIMES MAX L 1-NORM.
      W(1)=AMAXL*AMAXU
      RETURN
      END
C I AND J ARE IBM FORTRAN SINGLE AND DOUBLE PRECISION CODES JISD/
C S AND D ARE STANDARD FORTRAN SINGLE AND DOUBLE PRECISION CODES RESP.
C     SUBROUTINE MA30A(NN,ICN,A,LICN,LENR,LENRL,IDISP,IP,IQ,IRN,  IS/
C    1LIRN,LENC,IFIRST,LASTR,NEXTR,LASTC,NEXTC,IPTR,IPC,U,IFLAG) IS/
      SUBROUTINE MA30AD(NN,ICN,A,LICN,LENR,LENRL,IDISP,IP,IQ,IRN,
     1LIRN,LENC,IFIRST,LASTR,NEXTR,LASTC,NEXTC,IPTR,IPC,U,IFLAG)
C     REAL A(LICN)                                                   IS/
      DOUBLE PRECISION A(LICN),U,AU,UMAX,AMAX,ZERO
      INTEGER IPTR(NN),PIVOT,PIVEND,DISPC,OLDPIV,OLDEND,PIVROW
      INTEGER ROWI
      INTEGER IPC(NN),IDISP(2)
      LOGICAL ABORT1,ABORT2,ABORT3
      INTEGER   ICN(LICN),LENR(NN),LENRL(NN),IP(NN),IQ(NN),LENC(NN),
     1IRN(LIRN)
C     INTEGER*2 ICN(LICN),LENR(NN),LENRL(NN),IP(NN),IQ(NN),LENC(NN), ID/
C    1IRN(LIRN)                                                      ID/
      INTEGER   IFIRST(NN),LASTR(NN),NEXTR(NN),LASTC(NN),NEXTC(NN)
C     INTEGER*2 IFIRST(NN),LASTR(NN),NEXTR(NN),LASTC(NN),NEXTC(NN)   ID/
C     COMMON /MA30E/ LP,ABORT1,ABORT2,ABORT3  IS/
      COMMON /MA30ED/ LP,ABORT1,ABORT2,ABORT3
C     COMMON /MA30F/ IRNCP,ICNCP,IRANK,MINIRN,MINICN  IS/
      COMMON /MA30FD/ IRNCP,ICNCP,IRANK,MINIRN,MINICN
      SAVE /MA30ED/,/MA30FD/
C     DATA UMAX/.9999/                                               IS/
      DATA UMAX/.999999999D0/
C     DATA ZERO/0.0/                                                 IS/
      DATA ZERO/0.0D0/
C
      MINIRN=0
      MINICN=IDISP(1)-1
      MOREI=0
      IRANK=NN
      IRNCP=0
      ICNCP=0
      IFLAG=0
C RESET U IF NECESSARY.
C     U=AMIN1(U,UMAX)                                                IS/
      U=DMIN1(U,UMAX)
C     U=AMAX1(U,ZERO)                                                IS/
      U=DMAX1(U,ZERO)
C IBEG IS THE POSITION OF THE NEXT PIVOT ROW AFTER ELIMINATION STEP
C     USING IT.
      IBEG=IDISP(1)
C IACTIV IS THE POSITION OF THE FIRST ENTRY IN THE ACTIVE PART OF A/ICN.
      IACTIV=IDISP(2)
C NZROW IS CURRENT NUMBER OF NON-ZEROS IN ACTIVE AND UNPROCESSED PART
C     OF ROW FILE ICN.
      NZROW=LICN-IACTIV+1
      MINICN=NZROW+MINICN
C
C COUNT THE NUMBER OF DIAGONAL BLOCKS AND SET UP POINTERS TO THE
C     BEGINNINGS OF THE ROWS.
C NUM IS THE NUMBER OF DIAGONAL BLOCKS.
      NUM=1
      IPTR(1)=IACTIV
      IF (NN.EQ.1) GO TO 20
      NNM1=NN-1
      DO 10 I=1,NNM1
      IF (IP(I).LT.0) NUM=NUM+1
   10 IPTR(I+1)=IPTR(I)+LENR(I)
C ILAST IS THE LAST ROW IN THE PREVIOUS BLOCK.
   20 ILAST=0
C
C ***********************************************
C ****    LU DECOMPOSITION OF BLOCK NBLOCK   ****
C ***********************************************
C
C EACH PASS THROUGH THIS LOOP PERFORMS LU DECOMPOSITION ON ONE
C     OF THE DIAGONAL BLOCKS.
      DO 950 NBLOCK=1,NUM
      ISTART=ILAST+1
      DO 30 IROWS=ISTART,NN
      IF (IP(IROWS).LT.0) GO TO 40
   30 CONTINUE
      IROWS=NN
   40 ILAST=IROWS
C N IS THE NUMBER OF ROWS IN THE CURRENT BLOCK.
C ISTART IS THE INDEX OF THE FIRST ROW IN THE CURRENT BLOCK.
C ILAST IS THE INDEX OF THE LAST ROW IN THE CURRENT BLOCK.
C IACTIV IS THE POSITION OF THE FIRST ELEMENT IN THE BLOCK.
C ITOP IS THE POSITION OF THE LAST ELEMENT IN THE BLOCK.
      N=ILAST-ISTART+1
      IF (N.NE.1) GO TO 100
C
C CODE FOR DEALING WITH 1X1 BLOCK.
      LENRL(ILAST)=0
      ISING=ISTART
      IF (LENR(ILAST).NE.0) GO TO 60
C BLOCK IS STRUCTURALLY SINGULAR.
      IRANK=IRANK-1
      ISING=-ISING
      IF (IFLAG.NE.2.AND.IFLAG.NE.-5) IFLAG=1
      IF (.NOT.ABORT1) GO TO 90
      IDISP(2)=IACTIV
      IFLAG=-1
      IF (LP.NE.0) WRITE(LP,50)
   50 FORMAT(67H ERROR RETURN FROM MA30A/AD BECAUSE MATRIX IS STRUCTURAL
     1LY SINGULAR)
C     RETURN
      GO TO 1110
   60 IF (A(IACTIV).NE.ZERO) GO TO 80
      ISING=-ISING
      IRANK=IRANK-1
      IPTR(ILAST)=0
      IF (IFLAG.NE.-5) IFLAG=2
      IF (.NOT.ABORT2) GO TO 80
      IDISP(2)=IACTIV
      IFLAG=-2
      IF (LP.NE.0) WRITE(LP,70)
   70 FORMAT(66H ERROR RETURN FROM MA30A/AD BECAUSE MATRIX IS NUMERICALL
     1Y SINGULAR)
      GO TO 1110
   80 A(IBEG)=A(IACTIV)
      ICN(IBEG)=ICN(IACTIV)
      IACTIV=IACTIV+1
      IPTR(ISTART)=0
      IBEG=IBEG+1
      NZROW=NZROW-1
   90 LASTR(ISTART)=ISTART
      LASTC(ISTART)=ISING
      GO TO 950
C
C NON-TRIVIAL BLOCK.
  100 ITOP=LICN
      IF (ILAST.NE.NN) ITOP=IPTR(ILAST+1)-1
C
C SET UP COLUMN ORIENTED STORAGE.
      DO 110 I=ISTART,ILAST
      LENRL(I)=0
  110 LENC(I)=0
      IF (ITOP-IACTIV.LT.LIRN) GO TO 120
      MINIRN=ITOP-IACTIV+1
      PIVOT=ISTART-1
      GO TO 1050
C
C CALCULATE COLUMN COUNTS.
  120 DO 130 II=IACTIV,ITOP
      I=ICN(II)
  130 LENC(I)=LENC(I)+1
C SET UP COLUMN POINTERS SO THAT IPC(J) POINTS TO POSITION AFTER END
C     OF COLUMN J IN COLUMN FILE.
      IPC(ILAST)=LIRN+1
      J1=ISTART+1
      DO 140 JJ=J1,ILAST
      J=ILAST-JJ+J1-1
  140 IPC(J)=IPC(J+1)-LENC(J+1)
      DO 160 INDROW=ISTART,ILAST
      J1=IPTR(INDROW)
      J2=J1+LENR(INDROW)-1
      IF (J1.GT.J2) GO TO 160
      DO 150 JJ=J1,J2
      J=ICN(JJ)
      IPOS=IPC(J)-1
      IRN(IPOS)=INDROW
      IPC(J)=IPOS
  150 CONTINUE
  160 CONTINUE
C DISPC IS THE LOWEST INDEXED ACTIVE LOCATION IN THE COLUMN FILE.
      DISPC=IPC(ISTART)
      NZCOL=LIRN-DISPC+1
      MINIRN=MAX0(NZCOL,MINIRN)
      NZMIN=1
C
C INITIALIZE ARRAY IFIRST.  IFIRST(I) = +/- K INDICATES THAT ROW/COL
C     K HAS I NON-ZEROS.  IF IFIRST(I) = 0, THERE IS NO ROW OR COLUMN
C     WITH I NON ZEROS.
      DO 170 I=1,N
  170 IFIRST(I)=0
C
C COMPUTE ORDERING OF ROW AND COLUMN COUNTS.
C FIRST RUN THROUGH COLUMNS (FROM COLUMN N TO COLUMN 1).
      DO 190 JJ=ISTART,ILAST
      J=ILAST-JJ+ISTART
      NZ=LENC(J)
      IF (NZ.NE.0) GO TO 180
      IPC(J)=0
      LASTC(J)=0
      GO TO 190
  180 ISW=IFIRST(NZ)
      IFIRST(NZ)=-J
      LASTC(J)=0
      NEXTC(J)=-ISW
      ISW1=IABS(ISW)
      IF (ISW.NE.0) LASTC(ISW1)=J
  190 CONTINUE
C NOW RUN THROUGH ROWS (AGAIN FROM N TO 1).
      DO 210 II=ISTART,ILAST
      I=ILAST-II+ISTART
      NZ=LENR(I)
      IF (NZ.NE.0) GO TO 200
      IPTR(I)=0
      LASTR(I)=0
      GO TO 210
  200 ISW=IFIRST(NZ)
      IFIRST(NZ)=I
      IF (ISW.GT.0) GO TO 205
      NEXTR(I)=0
      LASTR(I)=ISW
      GO TO 210
 205  NEXTR(I)=ISW
      LASTR(I)=LASTR(ISW)
      LASTR(ISW)=I
  210 CONTINUE
C
C
C **********************************************
C ****    START OF MAIN ELIMINATION LOOP    ****
C **********************************************
      DO 930 PIVOT=ISTART,ILAST
C
C FIRST FIND THE PIVOT USING MARKOWITZ CRITERION WITH STABILITY
C     CONTROL.
C JCOST IS THE MARKOWITZ COST OF THE BEST PIVOT SO FAR,.. THIS
C     PIVOT IS IN ROW IPIV AND COLUMN JPIV.
      NZ2=NZMIN
      JCOST=N*N
C
C EXAMINE ROWS/COLUMNS IN ORDER OF ASCENDING COUNT.
      DO 290 L=1,2
      LL=L
C A PASS WITH L EQUAL TO 2 IS ONLY PERFORMED IN THE CASE OF SINGULARITY.
      DO 280 NZ=NZ2,N
      IF (JCOST.LE.(NZ-1)**2) GO TO 380
      IJFIR=IFIRST(NZ)
      IF (IJFIR) 212,211,215
 211  IF (LL.EQ.1) NZMIN=NZ+1
      GO TO 280
 212  LL=2
      IJFIR=-IJFIR
      GO TO 245
 215  LL=2
C SCAN ROWS WITH NZ NON-ZEROS.
      DO 235 IDUMMY=1,N
      IF (IJFIR.EQ.0) GO TO 240
C ROW IJFIR IS NOW EXAMINED.
      I=IJFIR
      IJFIR=NEXTR(I)
C FIRST CALCULATE MULTIPLIER THRESHOLD LEVEL.
      AMAX=ZERO
      J1=IPTR(I)+LENRL(I)
      J2=IPTR(I)+LENR(I)-1
      DO 220 JJ=J1,J2
C220  AMAX=AMAX1(AMAX,ABS(A(JJ)))                                    IS/
  220 AMAX=DMAX1(AMAX,DABS(A(JJ)))
      AU=AMAX*U
C SCAN ROW FOR POSSIBLE PIVOTS
      DO 230 JJ=J1,J2
C     IF (ABS(A(JJ)).LE.AU.AND.L.EQ.1) GO TO 230                     IS/
      IF (DABS(A(JJ)).LE.AU.AND.L.EQ.1) GO TO 230
      J=ICN(JJ)
      KCOST=(NZ-1)*(LENC(J)-1)
      IF (KCOST.GE.JCOST) GO TO 230
C BEST PIVOT SO FAR IS FOUND.
      JCOST=KCOST
      IJPOS=JJ
      IPIV=I
      JPIV=J
      IF (JCOST.LE.(NZ-1)**2) GO TO 380
  230 CONTINUE
 235  CONTINUE
C
C COLUMNS WITH NZ NON-ZEROS NOW EXAMINED.
 240  IJFIR=IFIRST(NZ)
      IJFIR=-LASTR(IJFIR)
 245  IF (JCOST.LE.NZ*(NZ-1)) GO TO 380
      DO 270 IDUMMY=1,N
      IF (IJFIR.EQ.0) GO TO 280
      J=IJFIR
      IJFIR=NEXTC(IJFIR)
      I1=IPC(J)
      I2=I1+NZ-1
C SCAN COLUMN J.
      DO 260 II=I1,I2
      I=IRN(II)
      KCOST=(NZ-1)*(LENR(I)-LENRL(I)-1)
      IF (KCOST.GE.JCOST) GO TO 260
C PIVOT HAS BEST MARKOWITZ COUNT SO FAR ... NOW CHECK ITS
C     SUITABILITY ON NUMERIC GROUNDS BY EXAMINING THE OTHER NON-ZEROS
C     IN ITS ROW.
      J1=IPTR(I)+LENRL(I)
      J2=IPTR(I)+LENR(I)-1
C WE NEED A STABILITY CHECK ON SINGLETON COLUMNS BECAUSE OF POSSIBLE
C     PROBLEMS WITH UNDERDETERMINED SYSTEMS.
      AMAX=ZERO
      DO 250 JJ=J1,J2
C     AMAX=AMAX1(AMAX,ABS(A(JJ)))                                    IS/
      AMAX=DMAX1(AMAX,DABS(A(JJ)))
  250 IF (ICN(JJ).EQ.J) JPOS=JJ
C     IF (ABS(A(JPOS)).LE.AMAX*U.AND.L.EQ.1) GO TO 260               IS/
      IF (DABS(A(JPOS)).LE.AMAX*U.AND.L.EQ.1) GO TO 260
      JCOST=KCOST
      IPIV=I
      JPIV=J
      IJPOS=JPOS
      IF (JCOST.LE.NZ*(NZ-1)) GO TO 380
  260 CONTINUE
C
  270 CONTINUE
C
  280 CONTINUE
C
C MATRIX IS NUMERICALLY OR STRUCTURALLY SINGULAR  ... WHICH IT IS WILL
C     BE DIAGNOSED LATER.
      IRANK=IRANK-1
  290 CONTINUE
C ASSIGN REST OF ROWS AND COLUMNS TO ORDERING ARRAY.
C MATRIX IS STRUCTURALLY SINGULAR.
      IF (IFLAG.NE.2.AND.IFLAG.NE.-5) IFLAG=1
      IRANK=IRANK-ILAST+PIVOT+1
      IF (.NOT.ABORT1) GO TO 300
      IDISP(2)=IACTIV
      IFLAG=-1
      IF (LP.NE.0) WRITE(LP,50)
      GO TO 1110
  300 K=PIVOT-1
      DO 350 I=ISTART,ILAST
      IF (LASTR(I).NE.0) GO TO 350
      K=K+1
      LASTR(I)=K
      IF (LENRL(I).EQ.0) GO TO 340
      MINICN=MAX0(MINICN,NZROW+IBEG-1+MOREI+LENRL(I))
      IF (IACTIV-IBEG.GE.LENRL(I)) GO TO 320
C     CALL MA30D(A,ICN,IPTR(ISTART),N,IACTIV,ITOP,.TRUE.)  IS/
      CALL MA30DD(A,ICN,IPTR(ISTART),N,IACTIV,ITOP,.TRUE.)
C CHECK NOW TO SEE IF MA30D/DD HAS CREATED ENOUGH AVAILABLE SPACE.
      IF (IACTIV-IBEG.GE.LENRL(I)) GO TO 320
C CREATE MORE SPACE BY DESTROYING PREVIOUSLY CREATED LU FACTORS.
      MOREI=MOREI+IBEG-IDISP(1)
      IBEG=IDISP(1)
      IF (LP.NE.0) WRITE(LP,310)
      IFLAG=-5
  310 FORMAT(48H LU DECOMPOSITION DESTROYED TO CREATE MORE SPACE)
      IF (ABORT3) GO TO 1030
  320 J1=IPTR(I)
      J2=J1+LENRL(I)-1
      IPTR(I)=0
      DO 330 JJ=J1,J2
      A(IBEG)=A(JJ)
      ICN(IBEG)=ICN(JJ)
      ICN(JJ)=0
  330 IBEG=IBEG+1
      NZROW=NZROW-LENRL(I)
  340 IF (K.EQ.ILAST) GO TO 360
  350 CONTINUE
  360 K=PIVOT-1
      DO 370 I=ISTART,ILAST
      IF (LASTC(I).NE.0) GO TO 370
      K=K+1
      LASTC(I)=-K
      IF (K.EQ.ILAST) GO TO 940
  370 CONTINUE
C
C THE PIVOT HAS NOW BEEN FOUND IN POSITION (IPIV,JPIV) IN LOCATION
C     IJPOS IN ROW FILE.
C UPDATE COLUMN AND ROW ORDERING ARRAYS TO CORRESPOND WITH REMOVAL
C     OF THE ACTIVE PART OF THE MATRIX.
  380 ISING=PIVOT
      IF (A(IJPOS).NE.ZERO) GO TO 390
C NUMERICAL SINGULARITY IS RECORDED HERE.
      ISING=-ISING
      IF (IFLAG.NE.-5) IFLAG=2
      IF (.NOT.ABORT2) GO TO 390
      IDISP(2)=IACTIV
      IFLAG=-2
      IF (LP.NE.0) WRITE(LP,70)
      GO TO 1110
  390 OLDPIV=IPTR(IPIV)+LENRL(IPIV)
      OLDEND=IPTR(IPIV)+LENR(IPIV)-1
C CHANGES TO COLUMN ORDERING.
      DO 460 JJ=OLDPIV,OLDEND
      J=ICN(JJ)
      LC=LASTC(J)
      NC=NEXTC(J)
      IF (NC.NE.0) LASTC(NC)=LC
      IF (LC.EQ.0) GO TO 440
      NEXTC(LC)=NC
      GO TO 460
 440  NZ=LENC(J)
      ISW=IFIRST(NZ)
      IF (ISW.GT.0) LASTR(ISW)=-NC
      IF (ISW.LT.0) IFIRST(NZ)=-NC
  460 CONTINUE
C CHANGES TO ROW ORDERING.
      I1=IPC(JPIV)
      I2=I1+LENC(JPIV)-1
      DO 530 II=I1,I2
      I=IRN(II)
      LR=LASTR(I)
      NR=NEXTR(I)
      IF (NR.NE.0) LASTR(NR)=LR
      IF (LR.LE.0) GO TO 500
      NEXTR(LR)=NR
      GO TO 530
 500  NZ=LENR(I)-LENRL(I)
      IF (NR.NE.0) IFIRST(NZ)=NR
      IF (NR.EQ.0) IFIRST(NZ)=LR
  530 CONTINUE
C     RECORD THE COLUMN PERMUTATION IN LASTC(JPIV) AND THE ROW
C     PERMUTATION IN LASTR(IPIV).
      LASTC(JPIV)=ISING
      LASTR(IPIV)=PIVOT
C
C MOVE PIVOT TO POSITION LENRL+1 IN PIVOT ROW AND MOVE PIVOT ROW
C     TO THE BEGINNING OF THE AVAILABLE STORAGE.
C THE L PART AND THE PIVOT IN THE OLD COPY OF THE PIVOT ROW IS
C     NULLIFIED WHILE, IN THE STRICTLY UPPER TRIANGULAR PART, THE
C     COLUMN INDICES, J SAY, ARE OVERWRITTEN BY THE CORRESPONDING
C     ELEMENT OF IQ (IQ(J)) AND IQ(J) IS SET TO THE NEGATIVE OF THE
C     DISPLACEMENT OF THE COLUMN INDEX FROM THE PIVOT ELEMENT.
      IF (OLDPIV.EQ.IJPOS) GO TO 540
      AU=A(OLDPIV)
      A(OLDPIV)=A(IJPOS)
      A(IJPOS)=AU
      ICN(IJPOS)=ICN(OLDPIV)
      ICN(OLDPIV)=JPIV
C CHECK TO SEE IF THERE IS SPACE IMMEDIATELY AVAILABLE IN A/ICN TO
C     HOLD NEW COPY OF PIVOT ROW.
  540 MINICN=MAX0(MINICN,NZROW+IBEG-1+MOREI+LENR(IPIV))
      IF (IACTIV-IBEG.GE.LENR(IPIV)) GO TO 550
C     CALL MA30D(A,ICN,IPTR(ISTART),N,IACTIV,ITOP,.TRUE.)  IS/
      CALL MA30DD(A,ICN,IPTR(ISTART),N,IACTIV,ITOP,.TRUE.)
      OLDPIV=IPTR(IPIV)+LENRL(IPIV)
      OLDEND=IPTR(IPIV)+LENR(IPIV)-1
C CHECK NOW TO SEE IF MA30D/DD HAS CREATED ENOUGH AVAILABLE SPACE.
      IF (IACTIV-IBEG.GE.LENR(IPIV)) GO TO 550
C CREATE MORE SPACE BY DESTROYING PREVIOUSLY CREATED LU FACTORS.
      MOREI=MOREI+IBEG-IDISP(1)
      IBEG=IDISP(1)
      IF (LP.NE.0) WRITE(LP,310)
      IFLAG=-5
      IF (ABORT3) GO TO 1030
      IF (IACTIV-IBEG.GE.LENR(IPIV)) GO TO 550
C THERE IS STILL NOT ENOUGH ROOM IN A/ICN.
      IFLAG=-4
      GO TO 1030
C COPY PIVOT ROW AND SET UP IQ ARRAY.
  550 IJPOS=0
      J1=IPTR(IPIV)
C
      DO 570 JJ=J1,OLDEND
      A(IBEG)=A(JJ)
      ICN(IBEG)=ICN(JJ)
      IF (IJPOS.NE.0) GO TO 560
      IF (ICN(JJ).EQ.JPIV) IJPOS=IBEG
      ICN(JJ)=0
      GO TO 570
  560 K=IBEG-IJPOS
      J=ICN(JJ)
      ICN(JJ)=IQ(J)
      IQ(J)=-K
  570 IBEG=IBEG+1
C
      IJP1=IJPOS+1
      PIVEND=IBEG-1
      LENPIV=PIVEND-IJPOS
      NZROW=NZROW-LENRL(IPIV)-1
      IPTR(IPIV)=OLDPIV+1
      IF (LENPIV.EQ.0) IPTR(IPIV)=0
C
C REMOVE PIVOT ROW (INCLUDING PIVOT) FROM COLUMN ORIENTED FILE.
      DO 600 JJ=IJPOS,PIVEND
      J=ICN(JJ)
      I1=IPC(J)
      LENC(J)=LENC(J)-1
C I2 IS LAST POSITION IN NEW COLUMN.
      I2=IPC(J)+LENC(J)-1
      IF (I2.LT.I1) GO TO 590
      DO 580 II=I1,I2
      IF (IRN(II).NE.IPIV) GO TO 580
      IRN(II)=IRN(I2+1)
      GO TO 590
  580 CONTINUE
  590 IRN(I2+1)=0
  600 CONTINUE
      NZCOL=NZCOL-LENPIV-1
C
C GO DOWN THE PIVOT COLUMN AND FOR EACH ROW WITH A NON-ZERO ADD
C     THE APPROPRIATE MULTIPLE OF THE PIVOT ROW TO IT.
C WE LOOP ON THE NUMBER OF NON-ZEROS IN THE PIVOT COLUMN SINCE
C     MA30D/DD MAY CHANGE ITS ACTUAL POSITION.
C
      NZPC=LENC(JPIV)
      IF (NZPC.EQ.0) GO TO 870
      DO 820 III=1,NZPC
      II=IPC(JPIV)+III-1
      I=IRN(II)
C SEARCH ROW I FOR NON-ZERO TO BE ELIMINATED, CALCULATE MULTIPLIER,
C     AND PLACE IT IN POSITION LENRL+1 IN ITS ROW.
      J1=IPTR(I)+LENRL(I)
      IEND=IPTR(I)+LENR(I)-1
      DO 610 JJ=J1,IEND
      IF (ICN(JJ).NE.JPIV) GO TO 610
C IF PIVOT IS ZERO, REST OF COLUMN IS AND SO MULTIPLIER IS ZERO.
      AU=ZERO
      IF (A(IJPOS).NE.ZERO) AU=-A(JJ)/A(IJPOS)
      A(JJ)=A(J1)
      A(J1)=AU
      ICN(JJ)=ICN(J1)
      ICN(J1)=JPIV
      LENRL(I)=LENRL(I)+1
      GO TO 620
  610 CONTINUE
C GO TO 870 IF PIVOT ROW IS A SINGLETON.
  620 IF (LENPIV.EQ.0) GO TO 820
C NOW PERFORM NECESSARY OPERATIONS ON REST OF NON-PIVOT ROW I.
      ROWI=J1+1
      IOP=0
C IF ALL THE PIVOT ROW CAUSES FILL-IN GO TO 640
      IF (ROWI.GT.IEND) GO TO 640
C PERFORM OPERATIONS ON CURRENT NON-ZEROS IN ROW I.
C INNERMOST LOOP.
      DO 630 JJ=ROWI,IEND
      J=ICN(JJ)
      IF (IQ(J).GT.0) GO TO 630
      IOP=IOP+1
      PIVROW=IJPOS-IQ(J)
      A(JJ)=A(JJ)+AU*A(PIVROW)
      ICN(PIVROW)=-ICN(PIVROW)
  630 CONTINUE
  640 IFILL=LENPIV-IOP
C IF THERE IS NO FILL-IN GO TO 740.
      IF (IFILL.EQ.0) GO TO 740
C NOW FOR THE FILL-IN.
      MINICN=MAX0(MINICN,MOREI+IBEG-1+NZROW+IFILL+LENR(I))
C SEE IF THERE IS ROOM FOR FILL-IN.
C GET MAXIMUM SPACE FOR ROW I IN SITU.
      DO 650 JDIFF=1,IFILL
      JNPOS=IEND+JDIFF
      IF (JNPOS.GT.LICN) GO TO 660
      IF (ICN(JNPOS).NE.0) GO TO 660
  650 CONTINUE
C THERE IS ROOM FOR ALL THE FILL-IN AFTER THE END OF THE ROW SO IT
C     CAN BE LEFT IN SITU.
C NEXT AVAILABLE SPACE FOR FILL-IN.
      IEND=IEND+1
      GO TO 740
C JMORE SPACES FOR FILL-IN ARE REQUIRED IN FRONT OF ROW.
  660 JMORE=IFILL-JDIFF+1
      I1=IPTR(I)
C WE NOW LOOK IN FRONT OF THE ROW TO SEE IF THERE IS SPACE FOR
C     THE REST OF THE FILL-IN.
      DO 670 JDIFF=1,JMORE
      JNPOS=I1-JDIFF
      IF (JNPOS.LT.IACTIV) GO TO 680
      IF (ICN(JNPOS).NE.0) GO TO 690
  670 CONTINUE
  680 JNPOS=I1-JMORE
      GO TO 700
C WHOLE ROW MUST BE MOVED TO THE BEGINNING OF AVAILABLE STORAGE.
  690 JNPOS=IACTIV-LENR(I)-IFILL
C IF THERE IS SPACE IMMEDIATELY AVAILABLE FOR THE SHIFTED ROW GO TO 720.
  700 IF (JNPOS.GE.IBEG) GO TO 720
C     CALL MA30D(A,ICN,IPTR(ISTART),N,IACTIV,ITOP,.TRUE.)  IS/
      CALL MA30DD(A,ICN,IPTR(ISTART),N,IACTIV,ITOP,.TRUE.)
      I1=IPTR(I)
      IEND=I1+LENR(I)-1
      JNPOS=IACTIV-LENR(I)-IFILL
      IF (JNPOS.GE.IBEG) GO TO 720
C NO SPACE AVAILABLE SO TRY TO CREATE SOME BY THROWING AWAY PREVIOUS
C     LU DECOMPOSITION.
      MOREI=MOREI+IBEG-IDISP(1)-LENPIV-1
      IF (LP.NE.0) WRITE(LP,310)
      IFLAG=-5
      IF (ABORT3) GO TO 1030
C KEEP RECORD OF CURRENT PIVOT ROW.
      IBEG=IDISP(1)
      ICN(IBEG)=JPIV
      A(IBEG)=A(IJPOS)
      IJPOS=IBEG
      DO 710 JJ=IJP1,PIVEND
      IBEG=IBEG+1
      A(IBEG)=A(JJ)
  710 ICN(IBEG)=ICN(JJ)
      IJP1=IJPOS+1
      PIVEND=IBEG
      IBEG=IBEG+1
      IF (JNPOS.GE.IBEG) GO TO 720
C THIS STILL DOES NOT GIVE ENOUGH ROOM.
      IFLAG=-4
      GO TO 1030
  720 IACTIV=MIN0(IACTIV,JNPOS)
C MOVE NON-PIVOT ROW I.
      IPTR(I)=JNPOS
      DO 730 JJ=I1,IEND
      A(JNPOS)=A(JJ)
      ICN(JNPOS)=ICN(JJ)
      JNPOS=JNPOS+1
  730 ICN(JJ)=0
C FIRST NEW AVAILABLE SPACE.
      IEND=JNPOS
  740 NZROW=NZROW+IFILL
C INNERMOST FILL-IN LOOP WHICH ALSO RESETS ICN.
      DO 810 JJ=IJP1,PIVEND
      J=ICN(JJ)
      IF (J.LT.0) GO TO 800
      A(IEND)=AU*A(JJ)
      ICN(IEND)=J
      IEND=IEND+1
C
C PUT NEW ENTRY IN COLUMN FILE.
      MINIRN=MAX0(MINIRN,NZCOL+LENC(J)+1)
      JEND=IPC(J)+LENC(J)
      JROOM=NZPC-III+1+LENC(J)
      IF (JEND.GT.LIRN) GO TO 750
      IF (IRN(JEND).EQ.0) GO TO 790
  750 IF (JROOM.LT.DISPC) GO TO 760
C COMPRESS COLUMN FILE TO OBTAIN SPACE FOR NEW COPY OF COLUMN.
C     CALL MA30D(A,IRN,IPC(ISTART),N,DISPC,LIRN,.FALSE.)  IS/
      CALL MA30DD(A,IRN,IPC(ISTART),N,DISPC,LIRN,.FALSE.)
      IF (JROOM.LT.DISPC) GO TO 760
      JROOM=DISPC-1
      IF (JROOM.GE.LENC(J)+1) GO TO 760
C COLUMN FILE IS NOT LARGE ENOUGH.
      GO TO 1050
C COPY COLUMN TO BEGINNING OF FILE.
  760 JBEG=IPC(J)
      JEND=IPC(J)+LENC(J)-1
      JZERO=DISPC-1
      DISPC=DISPC-JROOM
      IDISPC=DISPC
      DO 770 II=JBEG,JEND
      IRN(IDISPC)=IRN(II)
      IRN(II)=0
  770 IDISPC=IDISPC+1
      IPC(J)=DISPC
      JEND=IDISPC
      DO 780 II=JEND,JZERO
  780 IRN(II)=0
  790 IRN(JEND)=I
      NZCOL=NZCOL+1
      LENC(J)=LENC(J)+1
C END OF ADJUSTMENT TO COLUMN FILE.
      GO TO 810
C
  800 ICN(JJ)=-J
  810 CONTINUE
      LENR(I)=LENR(I)+IFILL
C END OF SCAN OF PIVOT COLUMN.
  820 CONTINUE
C
C
C REMOVE PIVOT COLUMN FROM COLUMN ORIENTED STORAGE AND UPDATE ROW
C     ORDERING ARRAYS.
      I1=IPC(JPIV)
      I2=IPC(JPIV)+LENC(JPIV)-1
      NZCOL=NZCOL-LENC(JPIV)
      DO 860 II=I1,I2
      I=IRN(II)
      IRN(II)=0
      NZ=LENR(I)-LENRL(I)
      IF (NZ.NE.0) GO TO 830
      LASTR(I)=0
      GO TO 860
  830 IFIR=IFIRST(NZ)
      IFIRST(NZ)=I
      IF (IFIR) 840,855,850
 840  LASTR(I)=IFIR
      NEXTR(I)=0
      GO TO 860
 850  LASTR(I)=LASTR(IFIR)
      NEXTR(I)=IFIR
      LASTR(IFIR)=I
      GO TO 860
 855  LASTR(I)=0
      NEXTR(I)=0
      NZMIN=MIN0(NZMIN,NZ)
 860  CONTINUE
C RESTORE IQ AND NULLIFY U PART OF OLD PIVOT ROW.
  870 IPC(JPIV)=0
      IF (LENPIV.EQ.0) GO TO 930
      NZROW=NZROW-LENPIV
      JVAL=IJP1
      JZER=IPTR(IPIV)
      IPTR(IPIV)=0
      DO 880 JCOUNT=1,LENPIV
      J=ICN(JVAL)
      IQ(J)=ICN(JZER)
      ICN(JZER)=0
      JVAL=JVAL+1
  880 JZER=JZER+1
C ADJUST COLUMN ORDERING ARRAYS.
      DO 920 JJ=IJP1,PIVEND
      J=ICN(JJ)
      NZ=LENC(J)
      IF (NZ.NE.0) GO TO 890
      LASTC(J)=0
      GO TO 920
  890 IFIR=IFIRST(NZ)
      LASTC(J)=0
      IF (IFIR) 900,910,915
 900  IFIRST(NZ)=-J
      IFIR=-IFIR
      LASTC(IFIR)=J
      NEXTC(J)=IFIR
      GO TO 920
 910  IFIRST(NZ)=-J
      NEXTC(J)=0
      NZMIN=MIN0(NZMIN,NZ)
      GO TO 920
 915  LC=-LASTR(IFIR)
      LASTR(IFIR)=-J
      NEXTC(J)=LC
      IF (LC.NE.0) LASTC(LC)=J
  920 CONTINUE
  930 CONTINUE
C ********************************************
C ****    END OF MAIN ELIMINATION LOOP    ****
C ********************************************
C
C RESET IACTIV TO POINT TO THE BEGINNING OF THE NEXT BLOCK.
  940 IF (ILAST.NE.NN) IACTIV=IPTR(ILAST+1)
  950 CONTINUE
C
C ********************************************
C ****    END OF DEOMPOSITION OF BLOCK    ****
C ********************************************
C
C RECORD SINGULARITY (IF ANY) IN IQ ARRAY.
      IF (IRANK.EQ.NN) GO TO 970
      DO 960 I=1,NN
      IF (LASTC(I).GT.0) GO TO 960
      ISING=-LASTC(I)
      IQ(ISING)=-IQ(ISING)
      LASTC(I)=ISING
  960 CONTINUE
C
C RUN THROUGH LU DECOMPOSITION CHANGING COLUMN INDICES TO THAT OF NEW
C     ORDER AND PERMUTING LENR AND LENRL ARRAYS ACCORDING TO PIVOT
C     PERMUTATIONS.
  970 ISTART=IDISP(1)
      IEND=IBEG-1
      DO 980 JJ=ISTART,IEND
      JOLD=ICN(JJ)
  980 ICN(JJ)=LASTC(JOLD)
      DO 990 II=1,NN
      I=LASTR(II)
      NEXTR(I)=LENR(II)
  990 NEXTC(I)=LENRL(II)
      DO 1000 I=1,NN
      LENRL(I)=NEXTC(I)
 1000 LENR(I)=NEXTR(I)
C
C UPDATE PERMUTATION ARRAYS IP AND IQ.
      DO 1010 II=1,NN
      I=LASTR(II)
      J=LASTC(II)
      NEXTR(I)=IABS(IP(II)+0)
 1010 NEXTC(J)=IABS(IQ(II)+0)
      DO 1020 I=1,NN
      IF (IP(I).LT.0) NEXTR(I)=-NEXTR(I)
      IP(I)=NEXTR(I)
      IF (IQ(I).LT.0) NEXTC(I)=-NEXTC(I)
 1020 IQ(I)=NEXTC(I)
      IP(NN)=IABS(IP(NN)+0)
      IDISP(2)=IEND
      GO TO 1110
C
C   ***    ERROR RETURNS    ***
 1030 IDISP(2)=IACTIV
      IF (LP.EQ.0) GO TO 1110
      WRITE(LP,1040)
 1040 FORMAT(55H ERROR RETURN FROM MA30A/AD BECAUSE LICN NOT BIG ENOUGH)
      GO TO 1080
 1050 IF (IFLAG.EQ.-5) IFLAG=-6
      IF (IFLAG.NE.-6) IFLAG=-3
      IDISP(2)=IACTIV
      IF (LP.EQ.0) GO TO 1110
      IF (IFLAG.EQ.-3) WRITE(LP,1060)
      IF (IFLAG.EQ.-6) WRITE(LP,1070)
 1060 FORMAT(55H ERROR RETURN FROM MA30A/AD BECAUSE LIRN NOT BIG ENOUGH)
 1070 FORMAT(51H ERROR RETURN FROM MA30A/AD LIRN AND LICN TOO SMALL)
 1080 PIVOT=PIVOT-ISTART+1
      WRITE(LP,1090) PIVOT,NBLOCK,ISTART,ILAST
 1090 FORMAT(10H AT STAGE ,I5,10H IN BLOCK ,I5,
     116H WITH FIRST ROW ,I5,14H AND LAST ROW ,I5)
      IF (PIVOT.EQ.0) WRITE(LP,1100) MINIRN
 1100 FORMAT(34H TO CONTINUE SET LIRN TO AT LEAST ,I8)
C
C
 1110 RETURN
      END
C     SUBROUTINE MA30D(A,ICN,IPTR,N,IACTIV,ITOP,REALS)  IS/
      SUBROUTINE MA30DD(A,ICN,IPTR,N,IACTIV,ITOP,REALS)
C     REAL A(ITOP)                                                   IS/
      DOUBLE PRECISION A(ITOP)
      INTEGER IPTR(N)
      LOGICAL REALS
      INTEGER   ICN(ITOP)
C     INTEGER*2 ICN(ITOP)                                            ID/
C     COMMON /MA30F/ IRNCP,ICNCP,IRANK,MINIRN,MINICN  IS/
      COMMON /MA30FD/ IRNCP,ICNCP,IRANK,MINIRN,MINICN
      SAVE /MA30FD/
C IACTIV IS THE FIRST POSITION IN ARRAYS A/ICN FROM WHICH THE
C     COMPRESS STARTS.
C ON EXIT IACTIV EQUALS THE POSITION OF THE FIRST ELEMENT IN THE
C     COMPRESSED PART OF A/ICN
      IF (REALS) ICNCP=ICNCP+1
      IF (.NOT.REALS) IRNCP=IRNCP+1
C SET THE FIRST NON-ZERO ELEMENT IN EACH ROW TO THE NEGATIVE OF THE
C     ROW/COL NUMBER AND HOLD THIS ROW/COL INDEX IN THE ROW/COL
C     POINTER.  THIS IS SO THAT THE BEGINNING OF EACH ROW/COL CAN
C     BE RECOGNIZED IN THE SUBSEQUENT SCAN.
      DO 10 J=1,N
      K=IPTR(J)
      IF (K.LT.IACTIV) GO TO 10
      IPTR(J)=ICN(K)
      ICN(K)=-J
   10 CONTINUE
      KN=ITOP+1
      KL=ITOP-IACTIV+1
C GO THROUGH ARRAYS IN REVERSE ORDER COMPRESSING TO THE BACK SO
C     THAT THERE ARE NO ZEROS HELD IN POSITIONS IACTIV TO ITOP IN ICN.
C     RESET FIRST ELEMENT OF EACH ROW/COL AND POINTER ARRAY IPTR.
      DO 30 K=1,KL
      JPOS=ITOP-K+1
      IF (ICN(JPOS).EQ.0) GO TO 30
      KN=KN-1
      IF (REALS) A(KN)=A(JPOS)
      IF (ICN(JPOS).GE.0) GO TO 20
C FIRST NON-ZERO OF ROW/COL HAS BEEN LOCATED
      J=-ICN(JPOS)
      ICN(JPOS)=IPTR(J)
      IPTR(J)=KN
   20 ICN(KN)=ICN(JPOS)
   30 CONTINUE
      IACTIV=KN
      RETURN
      END
C     SUBROUTINE MA30B(N,ICN,A,LICN,LENR,LENRL,IDISP,IP,IQ,W,IW,  IS/
C    1IFLAG)  IS/
      SUBROUTINE MA30BD(N,ICN,A,LICN,LENR,LENRL,IDISP,IP,IQ,W,IW,
     1IFLAG)
C     REAL A(LICN),W(N)                                              IS/
      DOUBLE PRECISION A(LICN),W(N),AU,EPS,ROWMAX,ZERO,ONE,RMIN
      INTEGER IW(N),IDISP(2),PIVPOS
      LOGICAL ABORT1,ABORT2,ABORT3,STAB
      INTEGER   ICN(LICN),LENR(N),LENRL(N),IP(N),IQ(N)
C     INTEGER*2 ICN(LICN),LENR(N),LENRL(N),IP(N),IQ(N)               ID/
C     COMMON /MA30E/ LP,ABORT1,ABORT2,ABORT3  IS/
      COMMON /MA30ED/ LP,ABORT1,ABORT2,ABORT3
C     COMMON /MA30G/ EPS,RMIN  IS/
      COMMON /MA30GD/ EPS,RMIN
      SAVE /MA30ED/,/MA30GD/
C     DATA ZERO/0.0/,ONE/1.0/  IS/
      DATA ZERO/0.0D0/,ONE/1.0D0/
      STAB=EPS.LE.ONE
      RMIN=EPS
      ISING=0
      IFLAG=0
C 170 = RETURN...
      IF (N.EQ.1) GO TO 170
      DO 10 I=1,N
   10 W(I)=ZERO
C SET UP POINTERS TO THE BEGINNING OF THE ROWS.
      IW(1)=IDISP(1)
      DO 20 I=2,N
   20 IW(I)=IW(I-1)+LENR(I-1)
C
C   ****   START  OF MAIN LOOP    ****
C AT STEP I, ROW I OF A IS TRANSFORMED TO ROW I OF L/U BY ADDING
C     APPROPRIATE MULTIPLES OF ROWS 1 TO I-1.
C     .... USING ROW-GAUSS ELIMINATION.
      DO 140 I=1,N
C ISTART IS BEGINNING OF ROW I OF A AND ROW I OF L.
      ISTART=IW(I)
C IFIN IS END OF ROW I OF A AND ROW I OF U.
      IFIN=ISTART+LENR(I)-1
C ILEND IS END OF ROW I OF L.
      ILEND=ISTART+LENRL(I)-1
      IF (ISTART.GT.ILEND) GO TO 70
C LOAD ROW I OF A INTO VECTOR W.
      DO 30 JJ=ISTART,IFIN
      J=ICN(JJ)
   30 W(J)=A(JJ)
C
C ADD MULTIPLES OF APPROPRIATE ROWS OF  I TO I-1  TO ROW I.
      DO 50 JJ=ISTART,ILEND
      J=ICN(JJ)
C IPIVJ IS POSITION OF PIVOT IN ROW J.
      IPIVJ=IW(J)+LENRL(J)
C FORM MULTIPLIER AU.
      AU=-W(J)/A(IPIVJ)
      W(J)=AU
C AU * ROW J (U PART) IS ADDED TO ROW I.
      IPIVJ=IPIVJ+1
      JFIN=IW(J)+LENR(J)-1
      IF (IPIVJ.GT.JFIN) GO TO 50
C INNERMOST LOOP.
      DO 40 JAYJAY=IPIVJ,JFIN
      JAY=ICN(JAYJAY)
   40 W(JAY)=W(JAY)+AU*A(JAYJAY)
C
   50 CONTINUE
C RELOAD W BACK INTO A (NOW L/U)
      DO 60 JJ=ISTART,IFIN
      J=ICN(JJ)
      A(JJ)=W(J)
   60 W(J)=ZERO
C WE NOW PERFORM THE STABILITY CHECKS.
   70 PIVPOS=ILEND+1
      IF (IQ(I).GT.0) GO TO 120
C MATRIX HAD SINGULARITY AT THIS POINT IN MA30A/AD.
C IS IT THE FIRST SUCH PIVOT IN CURRENT BLOCK ?
      IF (ISING.EQ.0) ISING=I
C DOES CURRENT MATRIX HAVE A SINGULARITY IN THE SAME PLACE ?
      IF (PIVPOS.GT.IFIN) GO TO 80
      IF (A(PIVPOS).NE.ZERO) GO TO 150
C IT DOES .. SO SET ISING IF IT IS NOT THE END OF THE CURRENT BLOCK
C CHECK TO SEE THAT APPROPRIATE PART OF L/U IS ZERO OR NULL.
   80 IF (ISTART.GT.IFIN) GO TO 100
      DO 90 JJ=ISTART,IFIN
      IF (ICN(JJ).LT.ISING) GO TO 90
      IF (A(JJ).NE.ZERO) GO TO 150
   90 CONTINUE
  100 IF (PIVPOS.LE.IFIN) A(PIVPOS)=ONE
      IF (IP(I).GT.0.AND.I.NE.N) GO TO 140
C END OF CURRENT BLOCK ... RESET ZERO PIVOTS AND ISING.
      DO 110 J=ISING,I
      IF ((LENR(J)-LENRL(J)).EQ.0) GO TO 110
      JJ=IW(J)+LENRL(J)
      A(JJ)=ZERO
  110 CONTINUE
      ISING=0
      GO TO 140
C MATRIX HAD NON-ZERO PIVOT IN MA30A/AD AT THIS STAGE.
  120 IF (PIVPOS.GT.IFIN) GO TO 150
      IF (A(PIVPOS).EQ.ZERO) GO TO 150
      IF (.NOT.STAB) GO TO 140
      ROWMAX=ZERO
      DO 130 JJ=PIVPOS,IFIN
C 130 ROWMAX=AMAX1(ROWMAX,ABS(A(JJ)))                                IS/
  130 ROWMAX=DMAX1(ROWMAX,DABS(A(JJ)))
C     IF (ABS(A(PIVPOS))/ROWMAX.GE.RMIN) GO TO 140 IS/
      IF (DABS(A(PIVPOS))/ROWMAX.GE.RMIN) GO TO 140
      IFLAG=I
C     RMIN=ABS(A(PIVPOS))/ROWMAX  IS/
      RMIN=DABS(A(PIVPOS))/ROWMAX
C   ****    END OF MAIN LOOP    ****
  140 CONTINUE
C
      GO TO 170
C   ***   ERROR RETURN   ***
  150 IF (LP.NE.0) WRITE(LP,160) I
  160 FORMAT(55H ERROR RETURN FROM MA30B/BD SINGULARITY DETECTED IN ROW,
     1I8)
      IFLAG=-I
C
  170 RETURN
      END
C     SUBROUTINE MA30C(N,ICN,A,LICN,LENR,LENRL,LENOFF,IDISP,IP,IQ,  IS/
C    1X,W,MTYPE) IS/
      SUBROUTINE MA30CD(N,ICN,A,LICN,LENR,LENRL,LENOFF,IDISP,IP,IQ,
     1X,W,MTYPE)
C     REAL A(LICN),X(N),W(N)                                         IS/
      DOUBLE PRECISION A(LICN),X(N),W(N),WII,WI,RESID,ZERO
      INTEGER IDISP(2)
      LOGICAL NEG,NOBLOC
      INTEGER   ICN(LICN),LENR(N),LENRL(N),LENOFF(N),IP(N),IQ(N)
C     INTEGER*2 ICN(LICN),LENR(N),LENRL(N),LENOFF(N),IP(N),IQ(N)     ID/
C     COMMON /MA30H/ RESID  IS/
      COMMON /MA30HD/ RESID
      SAVE /MA30HD/
C     DATA ZERO/0.0/   IS/
      DATA ZERO/0.0D0/
C THE FINAL VALUE OF RESID IS THE MAXIMUM RESIDUAL FOR AN INCONSISTENT
C     SET OF EQUATIONS.
      RESID=ZERO
C NOBLOC IS .TRUE. IF SUBROUTINE BLOCK HAS BEEN USED PREVIOUSLY AND
C     IS .FALSE. OTHERWISE.  THE VALUE .FALSE. MEANS THAT LENOFF
C     WILL NOT BE SUBSEQUENTLY ACCESSED.
      NOBLOC=LENOFF(1).LT.0
      IF (MTYPE.EQ.2) GO TO 140
C
C WE NOW SOLVE   A * X = B.
C NEG IS USED TO INDICATE WHEN THE LAST ROW IN A BLOCK HAS BEEN
C     REACHED.  IT IS THEN SET TO TRUE WHEREAFTER BACKSUBSTITUTION IS
C     PERFORMED ON THE BLOCK.
      NEG=.FALSE.
C IP(N) IS NEGATED SO THAT THE LAST ROW OF THE LAST BLOCK CAN BE
C     RECOGNISED.  IT IS RESET TO ITS POSITIVE VALUE ON EXIT.
      IP(N)=-IP(N)
C PREORDER VECTOR ... W(I) = X(IP(I))
      DO 10 II=1,N
      I=IP(II)
      I=IABS(I)
   10 W(II)=X(I)
C LT HOLDS THE POSITION OF THE FIRST NON-ZERO IN THE CURRENT ROW OF THE
C     OFF-DIAGONAL BLOCKS.
      LT=1
C IFIRST HOLDS THE INDEX OF THE FIRST ROW IN THE CURRENT BLOCK.
      IFIRST=1
C IBLOCK HOLDS THE POSITION OF THE FIRST NON-ZERO IN THE CURRENT ROW
C     OF THE LU DECOMPOSITION OF THE DIAGONAL BLOCKS.
      IBLOCK=IDISP(1)
C IF I IS NOT THE LAST ROW OF A BLOCK, THEN A PASS THROUGH THIS LOOP
C     ADDS THE INNER PRODUCT OF ROW I OF THE OFF-DIAGONAL BLOCKS AND W
C     TO W AND PERFORMS FORWARD ELIMINATION USING ROW I OF THE LU
C     DECOMPOSITION.   IF I IS THE LAST ROW OF A BLOCK THEN, AFTER
C     PERFORMING THESE AFOREMENTIONED OPERATIONS, BACKSUBSTITUTION IS
C     PERFORMED USING THE ROWS OF THE BLOCK.
      DO 120 I=1,N
      WI=W(I)
      IF(NOBLOC)GO TO 30
      IF (LENOFF(I).EQ.0) GO TO 30
C OPERATIONS USING LOWER TRIANGULAR BLOCKS.
C LTEND IS THE END OF ROW I IN THE OFF-DIAGONAL BLOCKS.
      LTEND=LT+LENOFF(I)-1
      DO 20 JJ=LT,LTEND
      J=ICN(JJ)
   20 WI=WI-A(JJ)*W(J)
C LT IS SET THE BEGINNING OF THE NEXT OFF-DIAGONAL ROW.
      LT=LTEND+1
C SET NEG TO .TRUE. IF WE ARE ON THE LAST ROW OF THE BLOCK.
   30 IF (IP(I).LT.0) NEG=.TRUE.
      IF (LENRL(I).EQ.0) GO TO 50
C FORWARD ELIMINATION PHASE.
C IEND IS THE END OF THE L PART OF ROW I IN THE LU DECOMPOSITION.
      IEND=IBLOCK+LENRL(I)-1
      DO 40 JJ=IBLOCK,IEND
      J=ICN(JJ)
   40 WI=WI+A(JJ)*W(J)
C IBLOCK IS ADJUSTED TO POINT TO THE START OF THE NEXT ROW.
   50 IBLOCK=IBLOCK+LENR(I)
      W(I)=WI
      IF (.NOT.NEG) GO TO 120
C BACK SUBSTITUTION PHASE.
C J1 IS POSITION IN A/ICN AFTER END OF BLOCK BEGINNING IN ROW IFIRST
C     AND ENDING IN ROW I.
      J1=IBLOCK
C ARE THERE ANY SINGULARITIES IN THIS BLOCK?  IF NOT, CONTINUE WITH
C     THE BACKSUBSTITUTION.
C IF MTYPE=3 FIRST SINGULARITY IS SUPRESSED BY SETTING PIVOT TO ONE
      IB=I
      IF (IQ(I).GT.0 .OR. MTYPE.EQ.3) GO TO 70
      DO 60 III=IFIRST,I
      IB=I-III+IFIRST
      IF (IQ(IB).GT.0) GO TO 70
      J1=J1-LENR(IB)
C     RESID=AMAX1(RESID,ABS(W(IB)))                                  IS/
      RESID=DMAX1(RESID,DABS(W(IB)))
      W(IB)=ZERO
   60 CONTINUE
C ENTIRE BLOCK IS SINGULAR.
      GO TO 110
C EACH PASS THROUGH THIS LOOP PERFORMS THE BACK-SUBSTITUTION
C     OPERATIONS FOR A SINGLE ROW, STARTING AT THE END OF THE BLOCK AND
C     WORKING THROUGH IT IN REVERSE ORDER.
   70 DO 100 III=IFIRST,IB
      II=IB-III+IFIRST
C J2 IS END OF ROW II.
      J2=J1-1
C J1 IS BEGINNING OF ROW II.
      J1=J1-LENR(II)
C JPIV IS THE POSITION OF THE PIVOT IN ROW II.
      JPIV=J1+LENRL(II)
      JPIVP1=JPIV+1
C IF ROW  II OF U HAS NO NON-ZEROS GO TO 90.
      IF (J2.LT.JPIVP1) GO TO 90
      WII=W(II)
      DO 80 JJ=JPIVP1,J2
      J=ICN(JJ)
   80 WII=WII-A(JJ)*W(J)
      W(II)=WII
90    CONTINUE
      IF (MTYPE.NE.3) W(II)=W(II)/A(JPIV)
      MTYPE=1
  100 CONTINUE
  110 IFIRST=I+1
      NEG=.FALSE.
  120 CONTINUE
C
C REORDER SOLUTION VECTOR ... X(I) = W(IQINVERSE(I))
      DO 130 II=1,N
      I=IQ(II)
      I=IABS(I)
  130 X(I)=W(II)
      IP(N)=-IP(N)
      GO TO 310
C
C
C WE NOW SOLVE   ATRANSPOSE * X = B.
C PREORDER VECTOR ... W(I)=X(IQ(I))
  140 DO 150 II=1,N
      I=IQ(II)
      I=IABS(I)
  150 W(II)=X(I)
C LJ1 POINTS TO THE BEGINNING THE CURRENT ROW IN THE OFF-DIAGONAL
C     BLOCKS.
      LJ1=IDISP(1)
C IBLOCK IS INITIALIZED TO POINT TO THE BEGINNING OF THE BLOCK AFTER
C     THE LAST ONE 
      IBLOCK=IDISP(2)+1
C ILAST IS THE LAST ROW IN THE CURRENT BLOCK.
      ILAST=N
C IBLEND POINTS TO THE POSITION AFTER THE LAST NON-ZERO IN THE
C     CURRENT BLOCK.
      IBLEND=IBLOCK
C EACH PASS THROUGH THIS LOOP OPERATES WITH ONE DIAGONAL BLOCK AND
C     THE OFF-DIAGONAL PART OF THE MATRIX CORRESPONDING TO THE ROWS
C     OF THIS BLOCK.  THE BLOCKS ARE TAKEN IN REVERSE ORDER AND THE
C     NUMBER OF TIMES THE LOOP IS ENTERED IS MIN(N,NO. BLOCKS+1).
      DO 280 NUMBLK=1,N
      IF (ILAST.EQ.0) GO TO 290
      IBLOCK=IBLOCK-LENR(ILAST)
C THIS LOOP FINDS THE INDEX OF THE FIRST ROW IN THE CURRENT BLOCK..
C     IT IS FIRST AND IBLOCK IS SET TO THE POSITION OF THE BEGINNING
C     OF THIS FIRST ROW.
      DO 160 K=1,N
      II=ILAST-K
      IF (II.EQ.0) GO TO 170
      IF (IP(II).LT.0) GO TO 170
      IBLOCK=IBLOCK-LENR(II)
  160 CONTINUE
  170 IFIRST=II+1
C J1 POINTS TO THE POSITION OF THE BEGINNING OF ROW I (LT PART) OR PIVOT
      J1=IBLOCK
C FORWARD ELIMINATION.
C EACH PASS THROUGH THIS LOOP PERFORMS THE OPERATIONS FOR ONE ROW OF THE
C     BLOCK.  IF THE CORRESPONDING ELEMENT OF W IS ZERO THEN THE
C     OPERATIONS CAN BE AVOIDED.
      DO 200 I=IFIRST,ILAST
      IF (W(I).EQ.ZERO) GO TO 195
C IS ROW I SINGULAR?  IF SO, GO TO 210
      IF (IQ(I).LT.0) GO TO 210
C J2 FIRST POINTS TO THE PIVOT IN ROW I AND THEN IS MADE TO POINT TO THE
C     FIRST NON-ZERO IN THE U TRANSPOSE PART OF THE ROW.
      J2=J1+LENRL(I)
      WI=W(I)/A(J2)
      IF (LENR(I)-LENRL(I).EQ.1) GO TO 190
      J2=J2+1
C J3 POINTS TO THE END OF ROW I.
      J3=J1+LENR(I)-1
      DO 180 JJ=J2,J3
      J=ICN(JJ)
  180 W(J)=W(J)-A(JJ)*WI
  190 W(I)=WI
  195 J1=J1+LENR(I)
  200 CONTINUE
      GO TO 230
C DEALS WITH REST OF BLOCK WHICH IS SINGULAR.
  210 DO 220 II=I,ILAST
C     RESID=AMAX1(RESID,ABS(W(II)))                                  IS/
      RESID=DMAX1(RESID,DABS(W(II)))
      W(II)=ZERO
  220 CONTINUE
C BACK SUBSTITUTION.
C THIS LOOP DOES THE BACK SUBSTITUTION ON THE ROWS OF THE BLOCK IN
C     THE REVERSE ORDER DOING IT SIMULTANEOUSLY ON THE L TRANSPOSE PART
C     OF THE DIAGONAL BLOCKS AND THE OFF-DIAGONAL BLOCKS.
  230 J1=IBLEND
      DO 270 IBACK=IFIRST,ILAST
      I=ILAST-IBACK+IFIRST
C J1 POINTS TO THE BEGINNING OF ROW I.
      J1=J1-LENR(I)
      IF (LENRL(I).EQ.0) GO TO 250
C J2 POINTS TO THE END OF THE L TRANSPOSE PART OF ROW I.
      J2=J1+LENRL(I)-1
      DO 240 JJ=J1,J2
      J=ICN(JJ)
  240 W(J)=W(J)+A(JJ)*W(I)
  250 IF(NOBLOC)GO TO 270
C OPERATIONS USING LOWER TRIANGULAR BLOCKS.
      IF(LENOFF(I).EQ.0)GO TO 270
C LJ2 POINTS TO THE END OF ROW I OF THE OFF-DIAGONAL BLOCKS.
      LJ2=LJ1-1
C LJ1 POINTS TO THE BEGINNING OF ROW I OF THE OFF-DIAGONAL BLOCKS.
      LJ1=LJ1-LENOFF(I)
      DO 260 JJ=LJ1,LJ2
      J=ICN(JJ)
  260 W(J)=W(J)-A(JJ)*W(I)
  270 CONTINUE
      IBLEND=J1
      ILAST=IFIRST-1
  280 CONTINUE
C REORDER SOLUTION VECTOR ... X(I)=W(IPINVERSE(I))
  290 DO 300 II=1,N
      I=IP(II)
      I=IABS(I)
  300 X(I)=W(II)
C
  310 RETURN
      END
C I IS IBM SP AND S IS STANDARD SP    IS/
      SUBROUTINE MC21A(N,ICN,LICN,IP,LENR,IPERM,NUMNZ,IW)
      INTEGER IP(N)
      INTEGER ICN(LICN),LENR(N),IPERM(N),IW(N,4)
C     INTEGER*2 ICN(LICN),LENR(N),IPERM(N),IW(N,4)                    I/
      CALL MC21B(N,ICN,LICN,IP,LENR,IPERM,NUMNZ,IW(1,1),IW(1,2),IW(1,3),
     1IW(1,4))
      RETURN
      END
      SUBROUTINE MC21B(N,ICN,LICN,IP,LENR,IPERM,NUMNZ,PR,ARP,CV,OUT)
      INTEGER IP(N)
C   PR(I) IS THE PREVIOUS ROW TO I IN THE DEPTH FIRST SEARCH.
C IT IS USED AS A WORK ARRAY IN THE SORTING ALGORITHM.
C   ELEMENTS (IPERM(I),I) I=1, ... N  ARE NON-ZERO AT THE END OF THE
C ALGORITHM UNLESS N ASSIGNMENTS HAVE NOT BEEN MADE.  IN WHICH CASE
C (IPERM(I),I) WILL BE ZERO FOR N-NUMNZ ENTRIES.
C   CV(I) IS THE MOST RECENT ROW EXTENSION AT WHICH COLUMN I
C WAS VISITED.
C   ARP(I) IS ONE LESS THAN THE NUMBER OF NON-ZEROS IN ROW I
C WHICH HAVE NOT BEEN SCANNED WHEN LOOKING FOR A CHEAP ASSIGNMENT.
C   OUT(I) IS ONE LESS THAN THE NUMBER OF NON-ZEROS IN ROW I
C WHICH HAVE NOT BEEN SCANNED DURING ONE PASS THROUGH THE MAIN LOOP.
      INTEGER ICN(LICN),LENR(N),IPERM(N),PR(N),CV(N),
     1ARP(N),OUT(N)
C     INTEGER*2 ICN(LICN),LENR(N),IPERM(N),PR(N),CV(N),               I/
C    1ARP(N),OUT(N)                                                   I/
C
C   INITIALIZATION OF ARRAYS.
      DO 10 I=1,N
      ARP(I)=LENR(I)-1
      CV(I)=0
   10 IPERM(I)=0
      NUMNZ=0
C
C
C   MAIN LOOP.
C   EACH PASS ROUND THIS LOOP EITHER RESULTS IN A NEW ASSIGNMENT
C OR GIVES A ROW WITH NO ASSIGNMENT.
      DO 130 JORD=1,N
      J=JORD
      PR(J)=-1
      DO 100 K=1,JORD
C LOOK FOR A CHEAP ASSIGNMENT
      IN1=ARP(J)
      IF (IN1.LT.0) GO TO 60
      IN2=IP(J)+LENR(J)-1
      IN1=IN2-IN1
      DO 50 II=IN1,IN2
      I=ICN(II)
      IF (IPERM(I).EQ.0) GO TO 110
   50 CONTINUE
C   NO CHEAP ASSIGNMENT IN ROW.
      ARP(J)=-1
C   BEGIN LOOKING FOR ASSIGNMENT CHAIN STARTING WITH ROW J.
   60 OUT(J)=LENR(J)-1
C INNER LOOP.  EXTENDS CHAIN BY ONE OR BACKTRACKS.
      DO 90 KK=1,JORD
      IN1=OUT(J)
      IF (IN1.LT.0) GO TO 80
      IN2=IP(J)+LENR(J)-1
      IN1=IN2-IN1
C FORWARD SCAN.
      DO 70 II=IN1,IN2
      I=ICN(II)
      IF (CV(I).EQ.JORD) GO TO 70
C   COLUMN I HAS NOT YET BEEN ACCESSED DURING THIS PASS.
      J1=J
      J=IPERM(I)
      CV(I)=JORD
      PR(J)=J1
      OUT(J1)=IN2-II-1
      GO TO 100
   70 CONTINUE
C
C   BACKTRACKING STEP.
   80 J=PR(J)
      IF (J.EQ.-1) GO TO 130
   90 CONTINUE
C
  100 CONTINUE
C
C   NEW ASSIGNMENT IS MADE.
  110 IPERM(I)=J
      ARP(J)=IN2-II-1
      NUMNZ=NUMNZ+1
      DO 120 K=1,JORD
      J=PR(J)
      IF (J.EQ.-1) GO TO 130
      II=IP(J)+LENR(J)-OUT(J)-2
      I=ICN(II)
      IPERM(I)=J
  120 CONTINUE
C
  130 CONTINUE
C
C   IF MATRIX IS STRUCTURALLY SINGULAR, WE NOW COMPLETE THE
C PERMUTATION IPERM.
      IF (NUMNZ.EQ.N) RETURN
      DO 140 I=1,N
  140 ARP(I)=0
      K=0
      DO 160 I=1,N
      IF (IPERM(I).NE.0) GO TO 150
      K=K+1
      OUT(K)=I
      GO TO 160
  150 J=IPERM(I)
      ARP(J)=I
  160 CONTINUE
      K=0
      DO 170 I=1,N
      IF (ARP(I).NE.0) GO TO 170
      K=K+1
      IOUTK=OUT(K)
      IPERM(IOUTK)=I
  170 CONTINUE
      RETURN
      END
C I AND J ARE IBM SINGLE AND DOUBLE PRECISION CODES RESP.  JISD/
C S AND D ARE STANDARD SINGLE AND DOUBLE PRECISION CODES RESP.
C     SUBROUTINE MA28A(N,NZ,A,LICN,IRN,LIRN,ICN,U,IKEEP,  IS/
C    1IW,W,IFLAG)  IS/
      SUBROUTINE MA28AD(N,NZ,A,LICN,IRN,LIRN,ICN,U,IKEEP,
     1IW,W,IFLAG)
C THE PARAMETERS ARE AS FOLLOWS.....
C N     INTEGER  ORDER OF MATRIX  NOT ALTERED BY SUBROUTINE.
C NZ    INTEGER  NUMBER OF NON-ZEROS IN INPUT MATRIX  NOT ALTERED
C     BY SUBROUTINE.
C A     REAL/DOUBLE PRECISION ARRAY  LENGTH LICN.  HOLDS NON-ZEROS OF
C     MATRIX ON ENTRY AND NON-ZEROS OF FACTORS ON EXIT.  REORDERED BY
C     MC20A/AD AND MC23A/AD AND ALTERED BY MA30A/AD.
C LICN  INTEGER  LENGTH OF ARRAYS A AND ICN.  NOT ALTERED BY
C     SUBROUTINE.
C IRN   INTEGER*2 ARRAY  LENGTH LIRN.  HOLDS ROW INDICES ON INPUT ...
C     USED AS WORKSPACE BY MA30A/AD TO HOLD COLUMN ORIENTATION OF
C     MATRIX.
C LIRN  INTEGER  LENGTH OF ARRAY IRN.
C ICN   INTEGER*2 ARRAY  LENGTH LICN.  HOLDS COLUMN INDICES ON ENTRY
C     AND COLUMN INDICES OF DECOMPOSED MATRIX ON EXIT. REORDERED BY
C     MC20A/AD AND MC23A/AD AND ALTERED BY MA30A/AD.
C U     REAL/DOUBLE PRECISION VARIABLE  SET BY USER TO CONTROL
C     BIAS TOWARDS NUMERIC OR SPARSITY PIVOTING.  U=1.0 GIVES PARTIAL
C     PIVOTING WHILE U=0. DOES NOT CHECK MULTIPLIERS AT ALL.
C     VALUES OF U GREATER THAN ONE ARE TREATED AS ONE WHILE NEGATIVE
C     VALUES ARE TREATED AS ZERO.  NOT ALTERED BY SUBROUTINE.
C IKEEP  INTEGER*2 ARRAY  LENGTH 5*N  USED AS WORKSPACE BY MA28A/AD
C     (SEE LATER COMMENTS).  IT IS NOT REQUIRED TO BE SET ON ENTRY
C     AND, ON EXIT, IT CONTAINS INFORMATION ABOUT THE DECOMPOSITION.
C     IT SHOULD BE PRESERVED BETWEEN THIS CALL AND SUBSEQUENT CALLS
C     TO MA28B/BD OR MA28C/CD.
C     IKEEP(I,1),I=1,N  HOLDS THE TOTAL LENGTH OF THE PART OF ROW I
C     IN THE DIAGONAL BLOCK.
C     ROW IKEEP(I,2),I=1,N  OF THE INPUT MATRIX IS THE ITH ROW IN
C     PIVOT ORDER.
C     COLUMN IKEEP(I,3),I=1,N  OF THE INPUT MATRIX IS THE ITH COLUMN
C     IN PIVOT ORDER.
C     IKEEP(I,4),I=1,N  HOLDS THE LENGTH OF THE PART OF ROW I IN
C     THE L PART OF THE L/U DECOMPOSITION.
C     IKEEP(I,5),I=1,N  HOLDS THE LENGTH OF THE PART OF ROW I IN THE
C     OFF-DIAGONAL BLOCKS.  IF THERE IS ONLY ONE DIAGONAL BLOCK,
C     IKEEP(1,5) WILL BE SET TO -1.
C IW    INTEGER*2 ARRAY  LENGTH 10*N.  TO OBTAIN CORRECT ALIGNMENT
C     FOR THIS ARRAY, BECAUSE PARTS OF IT ARE USED AS INTEGER*4
C     WORKSPACE, THE USER SHOULD HAVE DECLARED IT TO BE AN
C     INTEGER ARRAY OF LENGTH 5*N.  IN THE STANDARD VERSION THIS
C     DISTINCTION DISAPPEARS AND THE LENGTH OF THIS WORK-ARRAY (IN
C     THE USER'S PROGRAM) SHOULD CONSEQUENTLY BE CHANGED TO 8*N.
C W     REAL/DOUBLE PRECISION ARRAY  LENGTH N.  USED BY MC24A/AD BOTH
C     AS WORKSPACE AND TO RETURN GROWTH ESTIMATE IN W(1).  THE USE OF
C     THIS ARRAY BY MA28A/AD IS THUS OPTIONAL DEPENDING ON COMMON
C     BLOCK LOGICAL VARIABLE GROW.
C IFLAG  INTEGER VARIABLE  USED AS ERROR FLAG BY ROUTINE.  A POSITIVE
C     OR ZERO VALUE ON EXIT INDICATES SUCCESS.  POSSIBLE NEGATIVE
C     VALUES ARE -1 THROUGH -14.
C     REAL A(LICN),W(N)  IS/
      DOUBLE PRECISION A(LICN),U,W(N),UPRIV,RMIN,EPS,RESID,ZERO,
     1THEMAX
      INTEGER IDISP(2),IPRIV4
      INTEGER   ICN(LICN),IRN(LIRN),IKEEP(N,5),IW(N,8),IPRIV2(2) 
C     INTEGER*2 ICN(LICN),IRN(LIRN),IKEEP(N,5),IW(N,10),IPRIV2(2)    ID/
      LOGICAL GROW,LBLOCK,ABORT,ABORT1,ABORT2,ABORT3,ABORTA,ABORTB
C COMMON BLOCKS ... COMMON BLOCK MA28F/FD IS USED MERELY
C     TO COMMUNICATE WITH COMMON BLOCK MA30F/FD  SO THAT THE USER
C     NEED NOT DECLARE THIS COMMON BLOCK IN HIS MAIN PROGRAM.
C THE COMMON BLOCK VARIABLES ARE AS FOLLOWS ...
C LP,MP  INTEGER  DEFAULT VALUE 6 (LINE PRINTER).  UNIT NUMBER
C     FOR ERROR MESSAGES AND DUPLICATE ELEMENT WARNING RESP.
C NLP,MLP  INTEGER  UNIT NUMBER FOR MESSAGES FROM MA30A/AD AND
C     MC23A/AD RESP.  SET BY MA28A/AD TO VALUE OF LP.
C LBLOCK  LOGICAL  DEFAULT VALUE TRUE.  IF TRUE MC23A/AD IS USED
C     TO FIRST PERMUTE THE MATRIX TO BLOCK LOWER TRIANGULAR FORM.
C GROW    LOGICAL  DEFAULT VALUE TRUE.  IF TRUE THEN AN ESTIMATE
C     OF THE INCREASE IN SIZE OF MATRIX ELEMENTS DURING L/U
C     DECOMPOSITION IS GIVEN BY MC24A/AD.
C EPS,RMIN,RESID  REAL/DOUBLE PRECISION VARIABLES NOT REFERENCED
C     BY MA28A/AD.
C IRNCP,ICNCP  INTEGER  SET TO NUMBER OF COMPRESSES ON ARRAYS IRN AND
C     ICN/A RESPECTIVELY.
C MINIRN,MINICN  INTEGER  MINIMUM LENGTH OF ARRAYS IRN AND ICN/A
C     RESPECTIVELY, FOR SUCCESS ON FUTURE RUNS.
C IRANK  INTEGER   ESTIMATED RANK OF MATRIX.
C MIRNCP,MICNCP,MIRANK,MIRN,MICN INTEGER VARIABLES.  USED TO
C     COMMUNICATE BETWEEN MA30F/FD AND MA28F/FD VALUES OF ABOVENAMED
C     VARIABLES WITH SOMEWHAT SIMILAR NAMES.
C ABORT1,ABORT2  LOGICAL VARIABLES WITH DEFAULT VALUE TRUE.  IF FALSE
C     THEN DECOMPOSITION WILL BE PERFORMED EVEN IF THE MATRIX IS
C     STRUCTURALLY OR NUMERICALLY SINGULAR RESPECTIVELY.
C ABORTA,ABORTB  LOGICAL VARIABLES USED TO COMMUNICATE VALUES OF
C     ABORT1 AND ABORT2 TO MA30A/AD.
C ABORT  LOGICAL  USED TO COMMUNICATE VALUE OF ABORT1 TO MC23A/AD.
C ABORT3  LOGICAL VARIABLE NOT REFERENCED BY MA28A/AD.
C IDISP   INTEGER ARRAY  LENGTH 2.  USED TO COMMUNICATE INFORMATION
C     ON DECOMPOSITION BETWEEN THIS CALL TO MA28A/AD AND SUBSEQUENT
C     CALLS TO MA28B/BD AND MA28C/CD.  ON EXIT, IDISP(1) AND
C     IDISP(2) INDICATE POSITION IN ARRAYS A AND ICN OF THE
C     FIRST AND LAST ELEMENTS IN THE L/U DECOMPOSITION OF THE
C     DIAGONAL BLOCKS, RESPECTIVELY.
C NUMNZ  INTEGER  STRUCTURAL RANK OF MATRIX.
C NUM    INTEGER  NUMBER OF DIAGONAL BLOCKS.
C LARGE  INTEGER  SIZE OF LARGEST DIAGONAL BLOCK.
C     INTERNAL VARIABLES AND WORKSPACE USED IN  MA28A/AD ARE DEFINED
C     WITHIN THE SUBROUTINE IMMEDIATELY PRIOR TO THEIR FIRST USE.
C     COMMON /MA28E/ LP,MP,LBLOCK,GROW  IS/
      COMMON /MA28ED/ LP,MP,LBLOCK,GROW
C     COMMON /MA28F/ EPS,RMIN,RESID,IRNCP,ICNCP,MINIRN,MINICN,  IS/
C    1IRANK,ABORT1,ABORT2  IS/
      COMMON /MA28FD/ EPS,RMIN,RESID,IRNCP,ICNCP,MINIRN,MINICN,
     1IRANK,ABORT1,ABORT2
C     COMMON /MA28G/ IDISP  IS/
      COMMON /MA28GD/ IDISP
C     COMMON /MA30E/ NLP,ABORTA,ABORTB,ABORT3  IS/
      COMMON /MA30ED/ NLP,ABORTA,ABORTB,ABORT3
C     COMMON /MA30F/ MIRNCP,MICNCP,MIRANK,MIRN,MICN  IS/
      COMMON /MA30FD/ MIRNCP,MICNCP,MIRANK,MIRN,MICN
C     COMMON /MC23B/ MLP,NUMNZ,NUM,LARGE,ABORT  IS/
      COMMON /MC23BD/ MLP,NUMNZ,NUM,LARGE,ABORT
      SAVE /MA28ED/,/MA28FD/,/MA28GD/,/MA30ED/,/MA30FD/,/MC23BD/
C EQUIVALENCE IS USED TO OBTAIN INTEGER*4 INFORMATION FROM INTEGER*2
C     ARRAYS  ...  SEE LATER COMMENTS ON THE SETTING OF IPRIV4.
      EQUIVALENCE(IPRIV4,IPRIV2(1))
C     DATA ZERO /0.0E0/  IS/
      DATA ZERO /0.0D0/
C SOME  INITIALIZATION AND TRANSFER OF INFORMATION BETWEEN
C     COMMON BLOCKS (SEE EARLIER COMMENTS).
      IFLAG=0
      ABORTA=ABORT1
      ABORTB=ABORT2
      ABORT=ABORT1
      MLP=LP
      NLP=LP
C UPRIV PRIVATE COPY OF U IS USED IN CASE IT IS OUTSIDE
C     RANGE  ZERO TO ONE  AND  IS THUS ALTERED BY MA30A/AD.
      UPRIV=U
C SIMPLE DATA CHECK ON INPUT VARIABLES AND ARRAY DIMENSIONS.
      IF (N.GT.0) GO TO 2
C     IF (N.GT.0.AND.N.LE.32767) GO TO 2                             ID/
      IFLAG=-8
      IF (LP.NE.0) WRITE(LP,1) N
 1    FORMAT(36X,17HN OUT OF RANGE = ,I10)
      GO TO 999
 2    IF (NZ.GT.0) GO TO 4
      IFLAG=-9
      IF (LP.NE.0) WRITE(LP,3) NZ
 3    FORMAT(36X,18HNZ NON POSITIVE = ,I10)
      GO TO 999
 4    IF (LICN.GE.NZ) GO TO 6
      IFLAG=-10
      IF (LP.NE.0) WRITE(LP,5) LICN
 5    FORMAT(36X,17HLICN TOO SMALL = ,I10)
      GO TO 999
 6    IF (LIRN.GE.NZ) GO TO 8
      IFLAG=-11
      IF (LP.NE.0) WRITE(LP,7) LIRN
 7    FORMAT(36X,17HLIRN TOO SMALL = ,I10)
      GO TO 999
C
C DATA CHECK TO SEE IF ALL INDICES LIE BETWEEN 1 AND N.
 8    DO 30 I=1,NZ
      IF (IRN(I).GT.0.AND.IRN(I).LE.N.AND.ICN(I).GT.0.AND.ICN(I).LE.N)
     1GO TO 30
      IF (IFLAG.EQ.0.AND.LP.NE.0) WRITE(LP,10)
 10   FORMAT(62H ERROR RETURN FROM MA28A/AD BECAUSE INDICES FOUND OUT OF
     1 RANGE)
      IFLAG=-12
      IF (LP.NE.0) WRITE(LP,20) I,A(I),IRN(I),ICN(I)
 20   FORMAT(1X,I6,22HTH ELEMENT WITH VALUE ,1PD22.14,
     130H IS OUT OF RANGE WITH INDICES ,I8,2H ,,I8)
 30   CONTINUE
      IF (IFLAG.LT.0) GO TO 1000
C
C SORT ELEMENTS INTO ROW ORDER.
C     CALL MC20A(N,NZ,A,ICN,IW,IRN,0)  IS/
      CALL MC20AD(N,NZ,A,ICN,IW,IRN,0)
C
C THESE TWO STATEMENTS (TOGETHER WITH THE EARLIER EQUIVALENCE STATEMENT)
C SET IPRIV4 EQUAL TO THE INTEGER*4 VALUE CONSISTING OF THE
C CONCATENATION OF THE TWO INTEGER*2 WORDS IW(1,1) AND IW(2,1).
C THIS IS NECESSARY IN THE IBM VERSION BECAUSE MC20A/AD EXPECTS AN
C INTEGER*4 ARGUMENT IN IW.  THE STATEMENTS CAN BE LEFT IN
C WITHOUT AFFECTING A STANDARD FORTRAN VERSION (ASSUMING THE
C EQUIVALENCE STATEMENT IS KEPT).
      IPRIV2(1)=IW(1,1)
      IPRIV2(2)=IW(1,2)
      IF (N.GT.1) IPRIV2(2)=IW(2,1)
C PART OF IKEEP IS USED HERE AS A WORK-ARRAY.  IKEEP(I,2) IS
C     THE LAST ROW TO HAVE A NON-ZERO IN COLUMN I.  IKEEP(I,3)
C     IS THE OFF-SET OF COLUMN I FROM THE START OF THE ROW.
      DO 40 I=1,N
      IKEEP(I,2)=0
 40   IKEEP(I,1)=0
C
C CHECK FOR DUPLICATE ELEMENTS .. SUMMING ANY SUCH ENTRIES AND
C     PRINTING A WARNING MESSAGE ON UNIT MP.
C MOVE IS EQUAL TO THE NUMBER OF DUPLICATE ELEMENTS FOUND.
      MOVE=0
C THE LOOP ALSO CALCULATES THE LARGEST ELEMENT IN THE MATRIX, THEMAX.
      THEMAX=ZERO
C J1 IS POSITION IN ARRAYS OF FIRST NON-ZERO IN ROW.
      J1=IPRIV4
      DO 80 I=1,N
      IF (I.NE.N) GO TO 45
      IPRIV4=NZ+1
      GO TO 49
C THESE STATEMENTS ARE USED AS ABOVE TO SET IPRIV4.
C THIS TIME THE CHANGE INDICATED BY THE SPECIAL COMMENT CARD IS
C REQUIRED FOR SUCCESSFUL OPERATION OF THE STANDARD VERSION.
 45   DO 47 L=1,2
      K=1
      J=I+1
C     J=2*I+L                                                        ID/
      IF (J.LE.N) GO TO 46
      J=J-N
      K=2
 46   IPRIV2(L)=IW(J,K)
 47   CONTINUE
 49   LENGTH=IPRIV4-J1
      IF (LENGTH.EQ.0) GO TO 80
      J2=IPRIV4-1
      NEWJ1=J1-MOVE
      DO 70 JJ=J1,J2
      J=ICN(JJ)
C     THEMAX=AMAX1(THEMAX,ABS(A(JJ))) IS/
      THEMAX=DMAX1(THEMAX,DABS(A(JJ)))
      IF (IKEEP(J,2).EQ.I) GO TO 50
C FIRST TIME COLUMN HAS OCURRED IN CURRENT ROW.
      IKEEP(J,2)=I
      IKEEP(J,3)=JJ-MOVE-NEWJ1
      IF (MOVE.EQ.0) GO TO 70
C SHIFT NECESSARY BECAUSE OF  PREVIOUS DUPLICATE ELEMENT.
      NEWPOS=JJ-MOVE
      A(NEWPOS)=A(JJ)
      ICN(NEWPOS)=ICN(JJ)
      GO TO 70
C DUPLICATE ELEMENT.
 50   MOVE=MOVE+1
      LENGTH=LENGTH-1
      JAY=IKEEP(J,3)+NEWJ1
      IF (MP.NE.0) WRITE(MP,60) I,J,A(JJ)
 60   FORMAT(31H DUPLICATE ELEMENT IN POSITION ,I8,2H ,,I8,
     112H WITH VALUE ,1PD22.14)
      A(JAY)=A(JAY)+A(JJ)
C     THEMAX=AMAX1(THEMAX,ABS(A(JAY))) IS/
      THEMAX=DMAX1(THEMAX,DABS(A(JAY)))
 70   CONTINUE
      IKEEP(I,1)=LENGTH
      J1=IPRIV4
 80   CONTINUE
C
C KNUM IS ACTUAL NUMBER OF NON-ZEROS IN MATRIX WITH ANY MULTIPLE
C     ENTRIES COUNTED ONLY ONCE.
      KNUM=NZ-MOVE
      IF (.NOT.LBLOCK) GO TO 100
C
C PERFORM BLOCK TRIANGULARISATION.
C     CALL MC23A(N,ICN,A,LICN,IKEEP,IDISP,IKEEP(1,2),IKEEP(1,3), I/
C    1IKEEP(1,5),IW(1,5),IW)  I/
C     CALL MC23A(N,ICN,A,LICN,IKEEP,IDISP,IKEEP(1,2),IKEEP(1,3), S/
C    1IKEEP(1,5),IW(1,3),IW)  S/
      CALL MC23AD(N,ICN,A,LICN,IKEEP,IDISP,IKEEP(1,2),IKEEP(1,3),
     1IKEEP(1,5),IW(1,3),IW)
C     CALL MC23AD(N,ICN,A,LICN,IKEEP,IDISP,IKEEP(1,2),IKEEP(1,3), J/
C    1IKEEP(1,5),IW(1,5),IW)  J/
      IF (IDISP(1).GT.0) GO TO 130
      IFLAG=-7
      IF(IDISP(1).EQ.-1) IFLAG=-1
      IF (LP.NE.0) WRITE(LP,90)
 90   FORMAT(36X,26HERROR RETURN FROM MC23A/AD)
      GO TO 999
C
C BLOCK TRIANGULARIZATION NOT REQUESTED.
C MOVE STRUCTURE TO END OF DATA ARRAYS IN PREPARATION FOR
C     MA30A/AD.
C ALSO SET LENOFF(1) TO -1 AND SET PERMUTATION ARRAYS.
 100  DO 110 I=1,KNUM
      II=KNUM-I+1
      NEWPOS=LICN-I+1
      ICN(NEWPOS)=ICN(II)
 110  A(NEWPOS)=A(II)
      IDISP(1)=1
      IDISP(2)=LICN-KNUM+1
      DO 120 I=1,N
      IKEEP(I,2)=I
 120  IKEEP(I,3)=I
      IKEEP(1,5)=-1
C
C PERFORM L/U DECOMOSITION ON DIAGONAL BLOCKS.
C130  CALL MA30A(N,ICN,A,LICN,IKEEP,IKEEP(1,4),IDISP,IKEEP(1,2),  S/
C    1IKEEP(1,3),IRN,LIRN,IW(1,2),IW(1,3),IW(1,4),IW(1,5),IW(1,6), S/
C    2IW(1,7),IW(1,8),IW,UPRIV,IFLAG)  S/
 130  CALL MA30AD(N,ICN,A,LICN,IKEEP,IKEEP(1,4),IDISP,IKEEP(1,2),
     1IKEEP(1,3),IRN,LIRN,IW(1,2),IW(1,3),IW(1,4),IW(1,5),IW(1,6),
     2IW(1,7),IW(1,8),IW,UPRIV,IFLAG)
C130  CALL MA30A(N,ICN,A,LICN,IKEEP,IKEEP(1,4),IDISP,IKEEP(1,2),  I/
C    1IKEEP(1,3),IRN,LIRN,IW(1,3),IW(1,4),IW(1,5),IW(1,6),IW(1,7), I/
C    2IW(1,8),IW(1,9),IW,UPRIV,IFLAG)  I/
C130  CALL MA30AD(N,ICN,A,LICN,IKEEP,IKEEP(1,4),IDISP,IKEEP(1,2),  J/
C    1IKEEP(1,3),IRN,LIRN,IW(1,3),IW(1,4),IW(1,5),IW(1,6),IW(1,7), J/
C    2IW(1,8),IW(1,9),IW,UPRIV,IFLAG)  J/
C
C TRANSFER COMMON BLOCK INFORMATION.
      MINIRN=MAX0(MIRN,NZ)
      MINICN=MAX0(MICN,NZ)
      IRNCP=MIRNCP
      ICNCP=MICNCP
      IRANK=MIRANK
      IF (IFLAG.GE.0) GO TO 140
      IF (LP.NE.0) WRITE(LP,135)
 135  FORMAT(36X,26HERROR RETURN FROM MA30A/AD)
      GO TO 999
C
C REORDER OFF-DIAGONAL BLOCKS ACCORDING TO PIVOT PERMUTATION.
 140  I1=IDISP(1)-1
C     IF (I1.NE.0) CALL MC22A(N,ICN,A,I1,IKEEP(1,5),IKEEP(1,2), IS/
C    1IKEEP(1,3),IW,IRN) IS/
      IF (I1.NE.0) CALL MC22AD(N,ICN,A,I1,IKEEP(1,5),IKEEP(1,2),
     1IKEEP(1,3),IW,IRN)
C
C OPTIONALLY CALCULATE ELEMENT GROWTH ESTIMATE.
      I1=IDISP(1)
      IEND=LICN-I1+1
C     IF (GROW) CALL MC24A(N,ICN,A(I1),IEND,IKEEP,IKEEP(1,4),W) IS/
      IF (GROW) CALL MC24AD(N,ICN,A(I1),IEND,IKEEP,IKEEP(1,4),W)
C INCREMENT GROWTH ESTIMATE BY ORIGINAL MAXIMUM ELEMENT.
      IF (GROW) W(1)=W(1)+THEMAX
      IF (GROW.AND.N.GT.1) W(2)=THEMAX
C SET FLAG IF THE ONLY ERROR IS DUE TO DUPLICATE ELEMENTS.
      IF (IFLAG.GE.0.AND.MOVE.NE.0) IFLAG=-14
      GO TO 1000
 999  IF (LP.NE.0) WRITE(LP,998)
 998  FORMAT(36H+ERROR RETURN FROM MA28A/AD BECAUSE )
 1000 RETURN
      END
      BLOCK DATA
C     REAL EPS,RMIN,RESID        IS/
      DOUBLE PRECISION EPS,RMIN,RESID
      LOGICAL LBLOCK,GROW,ABORT1,ABORT2
C     COMMON /MA28E/ LP,MP,LBLOCK,GROW   IS/
      COMMON /MA28ED/ LP,MP,LBLOCK,GROW
C     COMMON /MA28F/ EPS,RMIN,RESID,IRNCP,ICNCP,MINIRN,MINICN,   IS/
C    1IRANK,ABORT1,ABORT2    IS/
      COMMON /MA28FD/ EPS,RMIN,RESID,IRNCP,ICNCP,MINIRN,MINICN,
     1IRANK,ABORT1,ABORT2
      SAVE /MA28ED/,/MA28FD/
C     DATA EPS/1.0E-4/         IS/
      DATA EPS/1.0D-4/
      DATA LP/6/,MP/6/
      DATA LBLOCK/.TRUE./,GROW/.TRUE./
      DATA ABORT1/.TRUE./,ABORT2/.TRUE./
      END
C     SUBROUTINE MA28B(N,NZ,A,LICN,IVECT,JVECT,ICN,IKEEP, IS/
C    1IW,W,IFLAG) IS/
      SUBROUTINE MA28BD(N,NZ,A,LICN,IVECT,JVECT,ICN,IKEEP,
     1IW,W,IFLAG)
C THE PARAMETERS ARE AS FOLLOWS ...
C N      INTEGER  ORDER OF MATRIX  NOT ALTERED BY SUBROUTINE.
C NZ     INTEGER  NUMBER OF NON-ZEROS IN INPUT MATRIX  NOT ALTERED
C     BY SUBROUTINE.
C A      REAL/DOUBLE PRECISION ARRAY  LENGTH LICN.  HOLDS NON-ZEROS OF
C     MATRIX ON ENTRY AND NON-ZEROS OF FACTORS ON EXIT.  REORDERED BY
C     MA28D/DD AND ALTERED BY SUBROUTINE MA30B/BD.
C LICN   INTEGER  LENGTH OF ARRAYS A AND ICN.  NOT ALTERED BY
C     SUBROUTINE.
C IVECT,JVECT  INTEGER*2 ARRAYS  LENGTH NZ.  HOLD ROW AND COLUMN
C     INDICES OF NON-ZEROS RESPECTIVELY.  NOT ALTERED BY SUBROUTINE.
C ICN    INTEGER*2 ARRAY  LENGTH LICN.  SAME ARRAY AS OUTPUT FROM
C     MA28A/AD.  UNCHANGED BY MA28B/BD.
C IKEEP  INTEGER*2 ARRAY  LENGTH 5*N.  SAME ARRAY AS OUTPUT FROM
C     MA28A/AD.  UNCHANGED BY MA28B/BD.
C IW     INTEGER ARRAY  LENGTH 4*N (5*N IN STANDARD VERSION).
C     USED AS WORKSPACE BY MA28D/DD AND MA30B/BD.
C W      REAL/DOUBLE PRECISION ARRAY  LENGTH N.  USED AS WORKSPACE
C     BY MA28D/DD,MA30B/BD AND (OPTIONALLY) MC24A/AD.
C IFLAG  INTEGER  USED AS ERROR FLAG WITH POSITIVE OR ZERO VALUE
C     INDICATING SUCCESS.
C     REAL A(LICN),W(N),MEPS,MRMIN    IS/
      DOUBLE PRECISION A(LICN),W(N),EPS,MEPS,RMIN,MRMIN
     1                ,RESID,WMAX
      INTEGER IDISP(2),IW(N,5)     
C     INTEGER IDISP(2),IW(N,4)    ID/
      INTEGER   IKEEP(N,5),IVECT(NZ),JVECT(NZ),ICN(LICN)
C     INTEGER*2  IKEEP(N,5),IVECT(NZ),JVECT(NZ),ICN(LICN) ID/
      LOGICAL GROW,LBLOCK,ABORTA,ABORTB,ABORT1,ABORT2,ABORT3
C UNLESS OTHERWISE STATED COMMON BLOCK VARIABLES ARE AS IN MA28A/AD.
C     THOSE VARIABLES REFERENCED BY MA28B/BD ARE MENTIONED BELOW.
C LP,MP  INTEGERS  USED AS IN MA28A/AD AS UNIT NUMBER FOR ERROR AND
C     WARNING MESSAGES, RESPECTIVELY.
C NLP    INTEGER VARIABLE USED TO GIVE VALUE OF LP TO MA30E/ED.
C EPS    REAL/DOUBLE PRECISION  MA30B/BD WILL OUTPUT A POSITIVE VALUE
C     FOR IFLAG IF ANY MODULUS OF THE RATIO OF PIVOT ELEMENT TO THE
C     LARGEST ELEMENT IN ITS ROW (U PART ONLY) IS LESS THAN EPS (UNLESS
C     EPS IS GREATER THAN 1.0 WHEN NO ACTION TAKES PLACE).
C RMIN   REAL/DOUBLE PRECISION  VARIABLE EQUAL TO THE VALUE OF THIS MINI
C     RATIO IN CASES WHERE EPS IS LESS THAN OR EQUAL TO 1.0
C MEPS,MRMIN  REAL/DOUBLE PRECISION VARIABLES USED BY THE SUBROUTINE
C     TO COMMUNICATE BETWEEN COMMON BLOCKS MA28F/FD AND MA30G/GD.
C IDISP  INTEGER ARRAY  LENGTH 2  THE SAME AS THAT USED BY MA28A/AD.
C     IT IS UNCHANGED BY MA28B/BD.
C     COMMON /MA28E/ MP,LP,LBLOCK,GROW  IS/
      COMMON /MA28ED/ MP,LP,LBLOCK,GROW
C     COMMON /MA28F/ EPS,RMIN,RESID,IRNCP,ICNCP,MINIRN,MINICN,  IS/
C    1IRANK,ABORT1,ABORT2  IS/
      COMMON /MA28FD/ EPS,RMIN,RESID,IRNCP,ICNCP,MINIRN,MINICN,
     1IRANK,ABORT1,ABORT2
C     COMMON /MA28G/ IDISP  IS/
      COMMON /MA28GD/ IDISP
C     COMMON /MA30E/ NLP,ABORTA,ABORTB,ABORT3  IS/
      COMMON /MA30ED/ NLP,ABORTA,ABORTB,ABORT3
C     COMMON /MA30G/ MEPS,MRMIN  IS/
      COMMON /MA30GD/ MEPS,MRMIN
      SAVE /MA28ED/,/MA28FD/,/MA28GD/,/MA30ED/,/MA30GD/
      IFLAG=0
      MEPS=EPS
      NLP=LP
C SIMPLE DATA CHECK ON VARIABLES.
      IF (N.GT.0) GO TO 2
C     IF (N.GT.0.AND.N.LE.32767) GO TO 2  ID/
      IFLAG=-11
      IF (LP.NE.0) WRITE(LP,1) N
 1    FORMAT(36X,17HN OUT OF RANGE = ,I10)
      GO TO 999
 2    IF (NZ.GT.0) GO TO 4
      IFLAG=-10
      IF (LP.NE.0) WRITE(LP,3) NZ
 3    FORMAT(36X,18HNZ NON POSITIVE = ,I10)
      GO TO 999
 4    IF (LICN.GE.NZ) GO TO 6
      IFLAG=-9
      IF (LP.NE.0) WRITE(LP,5) LICN
 5    FORMAT(36X,17HLICN TOO SMALL = ,I10)
      GO TO 999
C
C6    CALL MA28D(N,A,LICN,IVECT,JVECT,NZ,ICN,IKEEP,IKEEP(1,4),  IS/
C    1IKEEP(1,5),IKEEP(1,2),IKEEP(1,3),IW(1,3),IW,W(1),IFLAG)  IS/
 6    CALL MA28DD(N,A,LICN,IVECT,JVECT,NZ,ICN,IKEEP,IKEEP(1,4),
     1IKEEP(1,5),IKEEP(1,2),IKEEP(1,3),IW(1,3),IW,W(1),IFLAG)
C WMAX IS LARGEST ELEMENT IN MATRIX.
      WMAX=W(1)
C IDUP EQUALS ONE IF THERE WERE DUPLICATE ELEMENTS, ZERO OTHERWISE.
      IDUP=0
      IF (IFLAG.EQ.(N+1)) IDUP=1
      IF (IFLAG.LT.0) GO TO 999
C
C PERFORM ROW-GAUSS ELIMINATION ON THE STRUCTURE RECEIVED FROM MA28D/DD
C     CALL MA30B(N,ICN,A,LICN,IKEEP,IKEEP(1,4),IDISP,IKEEP(1,2), IS/
C    1IKEEP(1,3),W,IW,IFLAG) IS/
      CALL MA30BD(N,ICN,A,LICN,IKEEP,IKEEP(1,4),IDISP,IKEEP(1,2),
     1IKEEP(1,3),W,IW,IFLAG)
C
C TRANSFER COMMON BLOCK INFORMATION.
      RMIN=MRMIN
      IF (IFLAG.GE.0) GO TO 200
      IFLAG=-2
      IF (LP.NE.0) WRITE(LP,100)
 100  FORMAT(36X,26HERROR RETURN FROM MA30B/BD)
      GO TO 999
C
C OPTIONALLY CALCULATE THE GROWTH PARAMETER.
 200  I1=IDISP(1)
      IEND=LICN-I1+1
C     IF (GROW) CALL MC24A(N,ICN,A(I1),IEND,IKEEP,IKEEP(1,4),W) IS/
      IF (GROW) CALL MC24AD(N,ICN,A(I1),IEND,IKEEP,IKEEP(1,4),W)
C INCREMENT ESTIMATE BY LARGEST ELEMENT IN INPUT MATRIX.
      IF (GROW) W(1)=W(1)+WMAX
C SET FLAG IF THE ONLY ERROR IS DUE TO DUPLICATE ELEMENTS.
      IF (IDUP.EQ.1.AND.IFLAG.GE.0) IFLAG=-14
      GO TO 1000
 999  IF (LP.NE.0) WRITE(LP,998)
 998  FORMAT(36H+ERROR RETURN FROM MA28B/BD BECAUSE )
 1000 RETURN
      END
C THIS SUBROUTINE NEED NEVER BE CALLED BY THE USER DIRECTLY.
C     IT SORTS THE USER'S MATRIX INTO THE STRUCTURE OF THE DECOMPOSED
C     FORM AND CHECKS FOR THE PRESENCE OF DUPLICATE ENTRIES OR
C     NON-ZEROS LYING OUTSIDE THE SPARSITY PATTERN OF THE DECOMPOSITION
C     IT ALSO CALCULATES THE LARGEST ELEMENT IN THE INPUT MATRIX.
C     SUBROUTINE MA28D(N,A,LICN,IVECT,JVECT,NZ,ICN,LENR,LENRL,  IS/
C    1LENOFF,IP,IQ,IW1,IW,W1,IFLAG)  IS/
      SUBROUTINE MA28DD(N,A,LICN,IVECT,JVECT,NZ,ICN,LENR,LENRL,
     1LENOFF,IP,IQ,IW1,IW,W1,IFLAG)
C     REAL A(LICN)    IS/
      DOUBLE PRECISION A(LICN),ZERO,W1,AA
      INTEGER IW(N,2),IDISP(2)
      INTEGER   ICN(LICN),IVECT(NZ),JVECT(NZ),IP(N),IQ(N),
     1LENR(N),IW1(N,3),LENRL(N),LENOFF(N)
C     INTEGER*2 ICN(LICN),IVECT(NZ),JVECT(NZ),IP(N),IQ(N),     ID/
C    1LENR(N),IW1(N,3),LENRL(N),LENOFF(N)  ID/
      LOGICAL LBLOCK,GROW,BLOCKL
C     COMMON /MA28E/ LP,MP,LBLOCK,GROW  IS/
      COMMON /MA28ED/ LP,MP,LBLOCK,GROW
C     COMMON /MA28G/ IDISP  IS/
      COMMON /MA28GD/ IDISP
      SAVE /MA28ED/,/MA28GD/
C     DATA ZERO/0.0E0/  IS/
      DATA ZERO/0.0D0/
      BLOCKL=LENOFF(1).GE.0
C IW1(I,3)  IS SET TO THE BLOCK IN WHICH ROW I LIES AND THE
C     INVERSE PERMUTATIONS TO IP AND IQ ARE SET IN IW1(.,1) AND
C     IW1(.,2) RESP.
C POINTERS TO BEGINNING OF THE PART OF ROW I IN DIAGONAL AND
C   OFF-DIAGONAL BLOCKS ARE SET IN IW(I,2) AND IW(I,1) RESP.
      IBLOCK=1
      IW(1,1)=1
      IW(1,2)=IDISP(1)
      DO 10 I=1,N
      IW1(I,3)=IBLOCK
      IF (IP(I).LT.0) IBLOCK=IBLOCK+1
      II=IABS(IP(I)+0)
      IW1(II,1)=I
      JJ=IQ(I)
      JJ=IABS(JJ)
      IW1(JJ,2)=I
      IF (I.EQ.1) GO TO 10
      IF(BLOCKL) IW(I,1)=IW(I-1,1)+LENOFF(I-1)
      IW(I,2)=IW(I-1,2)+LENR(I-1)
 10   CONTINUE
C PLACE EACH NON-ZERO IN TURN INTO ITS CORRECT LOCATION
C    IN THE A/ICN ARRAY.
      IDISP2=IDISP(2)
      DO 300 I=1,NZ
C NECESSARY TO AVOID REFERENCE TO UNASSIGNED ELEMENT OF ICN.
      IF(I.GT.IDISP2) GO TO 30
      IF (ICN(I).LT.0) GO TO 300
   30 IOLD=IVECT(I)
      JOLD=JVECT(I)
      AA=A(I)
C THIS IS A DUMMY LOOP FOR FOLLOWING A CHAIN OF INTERCHANGES.
C   IT WILL BE EXECUTED NZ TIMES IN TOTAL.
      DO 200 IDUMMY=1,NZ
C PERFORM SOME VALIDITY CHECKS ON IOLD AND JOLD.
      IF (IOLD.LE.N .AND. IOLD.GT.0
     1      .AND. JOLD.LE.N .AND. JOLD.GT.0) GO TO 60
      IF (LP.NE.0) WRITE(LP,40) I,A(I),IOLD,JOLD
 40   FORMAT(9H ELEMENT ,I6,12H WITH VALUE ,1PD22.14,
     1 13H HAS INDICES ,I8,2H ,,I8
     2 /36X,20HINDICES OUT OF RANGE)
      IFLAG=-12
      GO TO 340
 60   INEW=IW1(IOLD,1)
      JNEW=IW1(JOLD,2)
C ARE WE IN A VALID BLOCK AND IS IT DIAGONAL OR OFF-DIAGONAL?
      IF (IW1(INEW,3)-IW1(JNEW,3)) 70,100,90
 70   IFLAG=-13
      IF (LP.NE.0) WRITE(LP,80) IOLD,JOLD
 80   FORMAT(36X,8HNON-ZERO,I7,2H ,,I6,27H IN ZERO OFF-DIAGONAL BLOCK)
      GO TO 340
 90   J1=IW(INEW,1)
      J2=J1+LENOFF(INEW)-1
      GO TO 160
C ELEMENT IS IN DIAGONAL BLOCK.
 100  J1=IW(INEW,2)
      IF (INEW.GT.JNEW) GO TO 110
      J2=J1+LENR(INEW)-1
      J1=J1+LENRL(INEW)
      GO TO 160
 110  J2=J1+LENRL(INEW)
C BINARY SEARCH OF ORDERED LIST  .. ELEMENT IN L PART OF ROW.
      DO 140 JDUMMY=1,N
      MIDPT=(J1+J2)/2
      JCOMP=IABS(ICN(MIDPT)+0)
      IF (JNEW-JCOMP) 120,180,130
 120  J2=MIDPT
      GO TO 140
 130  J1=MIDPT
 140  CONTINUE
      IFLAG=-13
      IF (LP.NE.0) WRITE(LP,150) IOLD,JOLD
 150  FORMAT(36X,8H ELEMENT ,I6,2H ,,I6,23H WAS NOT IN L/U PATTERN)
      GO TO 340
C LINEAR SEARCH ... ELEMENT IN L PART OF ROW OR OFF-DIAGONAL BLOCKS.
 160  DO 170 MIDPT=J1,J2
      IF (IABS(ICN(MIDPT)+0).EQ.JNEW) GO TO 180
 170  CONTINUE
      IFLAG=-13
      IF (LP.NE.0) WRITE(LP,150) IOLD,JOLD
      GO TO 340
C EQUIVALENT ELEMENT OF ICN IS IN POSITION MIDPT.
 180  IF (ICN(MIDPT).LT.0) GO TO 250
      IF (MIDPT.GT.NZ.OR.MIDPT.LE.I) GO TO 220
      W1=A(MIDPT)
      A(MIDPT)=AA
      AA=W1
      IOLD=IVECT(MIDPT)
      JOLD=JVECT(MIDPT)
      ICN(MIDPT)=-ICN(MIDPT)
 200  CONTINUE
 220  A(MIDPT)=AA
      ICN(MIDPT)=-ICN(MIDPT)
      GO TO 300
 250  A(MIDPT)=A(MIDPT)+AA
C SET FLAG FOR DUPLICATE ELEMENTS.
      IFLAG=N+1
 300  CONTINUE
C RESET ICN ARRAY  AND ZERO ELEMENTS IN L/U BUT NOT IN A.
C ALSO CALCULATE MAXIMUM ELEMENT OF A.
  340 W1=ZERO
      DO 400 I=1,IDISP2
      IF (ICN(I).LT.0) GO TO 350
      A(I)=ZERO
      GO TO 400
 350  ICN(I)=-ICN(I)
C     W1=AMAX1(W1,ABS(A(I)))                IS/
      W1=DMAX1(W1,DABS(A(I)))
 400  CONTINUE
      RETURN
      END
C     SUBROUTINE MA28C(N,A,LICN,ICN,IKEEP,RHS,W,MTYPE)  IS/
      SUBROUTINE MA28CD(N,A,LICN,ICN,IKEEP,RHS,W,MTYPE)
C THE PARAMETERS ARE AS FOLLOWS ....
C N     INTEGER  ORDER OF MATRIX  NOT ALTERED BY SUBROUTINE.
C A      REAL/DOUBLE PRECISION ARRAY  LENGTH LICN.  THE SAME ARRAY AS
C     WAS USED IN THE MOST RECENT CALL TO MA28A/AD OR MA28B/BD.
C LICN  INTEGER  LENGTH OF ARRAYS A AND ICN.  NOT ALTERED BY
C     SUBROUTINE.
C ICN    INTEGER*2 ARRAY  LENGTH LICN.  SAME ARRAY AS OUTPUT FROM
C     MA28A/AD.  UNCHANGED BY MA28C/CD.
C IKEEP  INTEGER*2 ARRAY  LENGTH 5*N.  SAME ARRAY AS OUTPUT FROM
C     MA28A/AD.  UNCHANGED BY MA28C/CD.
C RHS    REAL/DOUBLE PRECISION ARRAY  LENGTH N.  ON ENTRY, IT HOLDS THE
C     RIGHT HAND SIDE.  ON EXIT, THE SOLUTION VECTOR.
C W      REAL/DOUBLE PRECISION ARRAY  LENGTH N. USED AS WORKSPACE BY
C     MA30C/CD.
C MTYPE  INTEGER  USED TO TELL MA30C/CD TO SOLVE THE DIRECT EQUATION
C     (MTYPE.NE.2) OR ITS TRANSPOSE (MTYPE.EQ.2).
C     IF MTYPE=3 FIRST SINGULARITY IS SUPRESSED BY SETTING PIVOT TO ONE
C     IN THIS CASE MTYPE REMAINS NOT UNCHANGED
C     REAL A(LICN),RHS(N),W(N),MRESID  IS/
      DOUBLE PRECISION A(LICN),RHS(N),W(N),MRESID,EPS,RMIN,RESID
      INTEGER IDISP(2)
      INTEGER   ICN(LICN),IKEEP(N,5)
C     INTEGER*2 ICN(LICN),IKEEP(N,5)  ID/
      LOGICAL ABORT1,ABORT2
C UNLESS OTHERWISE STATED COMMON BLOCK VARIABLES ARE AS IN MA28A/AD.
C     THOSE VARIABLES REFERENCED BY MA28C/CD ARE MENTIONED BELOW.
C RESID  REAL/DOUBLE PRECISION  VARIABLE RETURNS MAXIMUM RESIDUAL OF
C     EQUATIONS WHERE PIVOT WAS ZERO.
C MRESID  REAL/DOUBLE PRECISION VARIABLE USED BY MA28C/CD TO
C     COMMUNICATE BETWEEN MA28F/FD AND MA30H/HD.
C IDISP  INTEGER ARRAY  LENGTH 2  THE SAME AS THAT USED BY MA28A/AD.
C     IT IS UNCHANGED BY MA28B/BD.
C     COMMON /MA28F/ EPS,RMIN,RESID,IRNCP,ICNCP,MINIRN,MINICN,  IS/
C    1IRANK,ABORT1,ABORT2  IS/
      COMMON /MA28FD/ EPS,RMIN,RESID,IRNCP,ICNCP,MINIRN,MINICN,
     1IRANK,ABORT1,ABORT2
C     COMMON /MA28G/ IDISP  IS/
      COMMON /MA28GD/ IDISP
C     COMMON /MA30H/ MRESID  IS/
      COMMON /MA30HD/ MRESID
      SAVE /MA28FD/,/MA28GD/,/MA30HD/
C
C THIS SUBROUTINE PERFORMS THE SOLUTION OF THE SET OF EQUATIONS.
C     CALL MA30C(N,ICN,A,LICN,IKEEP,IKEEP(1,4),IKEEP(1,5),IDISP, IS/
C    1IKEEP(1,2),IKEEP(1,3),RHS,W,MTYPE)  IS/
      CALL MA30CD(N,ICN,A,LICN,IKEEP,IKEEP(1,4),IKEEP(1,5),IDISP,
     1IKEEP(1,2),IKEEP(1,3),RHS,W,MTYPE)
C
C TRANSFER COMMON BLOCK INFORMATION.
      RESID=MRESID
      RETURN
      END
C
C*    Group  Time monitor package
C
C*    Begin Prologue
C     ------------------------------------------------------------
C
C*  Title
C    
C     Monitor - A package for making multiple time measurements and
C               summary statistics
C
C*  Written by        U. Nowak, L. Weimann 
C*  Version           1.0
C*  Revision          January 1991
C*  Latest Change     January 1991
C*  Library           CodeLib
C*  Code              Fortran 77, Double Precision
C*  Environment       Standard Fortran 77 environment on PC's,
C*  Copyright     (c) Konrad Zuse Zentrum fuer
C                     Informationstechnik Berlin
C                     Heilbronner Str. 10, D-1000 Berlin 31
C                     phone 0049+30+89604-0, 
C                     telefax 0049+30+89604-125
C*  Contact           Lutz Weimann 
C                     ZIB, Numerical Software Development 
C                     phone: 0049+30+89604-185 ;
C                     e-mail: 
C                     RFC822 notation: weimann@sc.zib-berlin.de
C                     X.400: C=de;A=dbp;P=zib-berlin;OU=sc;S=Weimann
C
C  ---------------------------------------------------------------
C
C* Licence
C    You may use or modify this code for your own non commercial
C    purposes for an unlimited time. 
C    In any case you should not deliver this code without a special 
C    permission of ZIB.
C    In case you intend to use the code commercially, we oblige you
C    to sign an according licence agreement with ZIB.
C
C* Warranty 
C    This code has been tested up to a certain level. Defects and
C    weaknesses, which may be included in the code, do not establish
C    any warranties by ZIB. ZIB does not take over any liabilities
C    which may follow from aquisition or application of this code.
C
C* Software status 
C    This code is under care of ZIB and belongs to ZIB software class 1.
C
C  ---------------------------------------------------------------
C
C*    Summary:
C
C     Monitor is a package for generating time and summary statistics
C     about the execution of multiple program parts of any program.
C     Nested measurements of program parts are possible.
C     ------------------------------------------------------------
C
C*    Usage:
C
C     The usage of Monitor is naturally divided into three phases:
C     1. the initialization and setup phase before the start of
C        the program or subroutines package to be measured;
C     2. the run phase of the program to be measured;
C     3. the final evaluation call.
C
C     The phase 1 must start with exactly one call of the subroutine
C     MONINI, which passes a title string and a logical unit for
C     later statistics output and possible error messages to the
C     package. This call follows a number of calls of the subroutine
C     MONDEF, where each call associates an identification string
C     to a positive integer number, called the measurement index
C     - up to maxtab, where maxtab is a package constant. Multiple
C     measurement indices may be used for measurements of multiple
C     program parts. The index 0 must also be associated with some
C     identification string, and corresponds to all parts of the
C     measured program from the measurement start call till the final
C     evaluation call, which are not associated with specific positive
C     measurement indices. After all necessary MONDEF calls are done,
C     the measurements are started at begin of the program to be
C     measured by a parameterless call of MONSRT.
C     In phase 2, each program part to be measured must be immediately
C     preceeded by a call of the subroutine MONON with the associated 
C     measurement index, and must be immediately followed by a call of
C     the subroutine MONOFF with the same measurement index. Measure-
C     ments of nested program parts are possible, and nesting is allowed
C     up to the number mnest, where mnest is a package constant.
C     Calling MONOFF without a preceeding MONON call with the same 
C     measurement index, or calling one of these subroutines with a
C     measurement index not previously defined by a MONDEF call causes
C     an error stop of the program. 
C     Finally at the end of the program to be measured, the parameter-
C     less call of the subroutine MONEND closes all measurements and
C     prints the summary statistics.
C     As delivered, maxtab has a value 20 and mnest a value 10, but
C     both constants may be increased, if needed, to any possible
C     integer value, by simply changing it's values in the first 
C     parameter statement of the subroutine MONTOR below.
C
C*    Subroutines and their parameters:
C     =================================
C
C     MONINI(CIDENT,LUMON)  : Initialize Monitor
C       CIDENT  char*20  Identification string for the total measurement
C                        ( printed in summary )
C       LUMON   int      The logical unit for printing out the summary
C
C     MONDEF(MESIND,CIDMES) : Define one measurement index
C       MESIND  int      >=1 : measurement index for a specific part
C                        = 0 : measurement index for all remaining parts
C                              (i.e. not belonging to parts with 
C                               index >=1)
C       CIDMES  char*15  Identification string for the part associated
C                        with MESIND ( printed in summary )
C
C     MONSRT                : Start measurements
C       (no parameters)
C
C     MONON(MESIND)         : Start measurement of a specific part
C       MESIND  int      >=1 : measurement index for a specific part
C
C     MONOFF(MESIND)        : Stop measurement of a specific part
C       MESIND  int      >=1 : measurement index for a specific part
C
C     MONEND                : Finish measurements and print summary
C       (no parameters)
C
C
C*    Example:
C       Calling sequence:
C
C       CALL MONINI (' Example',6)
C       CALL MONDEF (0,'Solver')
C       CALL MONDEF (1,'User function')
C       CALL MONDEF (2,'User matrix')
C       CALL MONSRT ()
C       ...
C       program to be measured (part without specific measurement index)
C       ...
C 1     CONTINUE      
C       ...
C       CALL MONON (2)
C       ...  user matrix code ...
C       CALL MONOFF(2)
C       ...
C       program to be measured (part without specific measurement index)
C       ...
C       CALL MONON (1)
C       ...  user function code ...
C       CALL MONOFF(1)
C       ...
C       program to be measured (part without specific measurement index)
C       ...
C       IF (no termination) GOTO 1
C       ...
C       CALL MONEND ()
C     ------------------------------------------------------------
C 
      SUBROUTINE MONTOR
      PARAMETER(MAXTAB=20,MNEST=10)
      CHARACTER*15 NAME(MAXTAB),NAME0
      CHARACTER*20 TEXT 
      CHARACTER*(*) TEXTH 
      CHARACTER*(*) NAMEH   
      REAL SEC(MAXTAB),ASEC(MAXTAB),PC1(MAXTAB),PC2(MAXTAB)
      INTEGER COUNT(MAXTAB),INDACT(MNEST)
      LOGICAL QON(MAXTAB)
      INTEGER IOUNIT
C
      SAVE SEC,COUNT,ASEC,PC1,PC2,INDXO,TIME1,TIME0,MAXIND,NAME
      SAVE SEC0,NAME0,TEXT,MONI,QON,IONCNT,INDACT
C
C
      DATA MONI/6/ , INFO/1/ , IGRAPH/1/
C
      RETURN
C
C     initialize monitor
C
      ENTRY MONINI (TEXTH,IOUNIT)
C
      MONI=IOUNIT
      MAXIND=0
      TEXT=TEXTH
      DO 100 I=1,MAXTAB
        SEC(I)=0.
        ASEC(I)=0.
        COUNT(I)=0
        QON(I)=.FALSE.
100   CONTINUE
      DO 105 I=1,MNEST
        INDACT(I)=0
105   CONTINUE
C
      SEC0=0.
      IONCNT=0
      RETURN
C
C     define one monitor entry
C
      ENTRY MONDEF(INDX,NAMEH)
      IF(INDX.LT.0 .OR. INDX.GT.MAXTAB) GOTO 1190
      IF (INDX.GT.MAXIND) MAXIND=INDX
      IF (INDX.GT.0) THEN
        IF (COUNT(INDX).GT.0) GOTO 1290
      ENDIF
      IF (INDX.EQ.0) THEN
        NAME0 = NAMEH
      ELSE
        NAME(INDX) = NAMEH
      ENDIF
      RETURN
C
C     start monitor measurements
C 
      ENTRY MONSRT()
      CALL SECOND (TIME1)
C
C      if(igraph.gt.0) call gmini(maxind,name0,name)
C
      RETURN
C
C     start one measurement
C
      ENTRY MONON (INDX)
      IF(INDX.GT.MAXIND.OR.INDX.LE.0) GOTO 1010
      IF (QON(INDX)) GOTO 1030
      CALL SECOND(ASEC(INDX))
      QON(INDX)=.TRUE.
      IF (IONCNT.EQ.0) THEN
        SEC0=SEC0+ASEC(INDX)-TIME1
      ELSE
        INDXO=INDACT(IONCNT)
        SEC(INDXO)=SEC(INDXO)+ASEC(INDX)-ASEC(INDXO)
      ENDIF
      IONCNT=IONCNT+1
      INDACT(IONCNT)=INDX
      IF(INFO.GT.1) WRITE(MONI,*) ' enter',NAME(INDX),ASEC(INDX)
C
C      if(igraph.gt.0) call gmon(indx,sec0)
C
      RETURN
C
C     stop one measurement
C
      ENTRY MONOFF (INDX)
      IF(INDX.GT.MAXIND.OR.INDX.LE.0) GOTO 1010
      IF (.NOT. QON(INDX)) GOTO 1040
      CALL SECOND(TIME2)
      QON(INDX)=.FALSE.
      SEC(INDX)=SEC(INDX)+TIME2-ASEC(INDX)
      COUNT(INDX)=COUNT(INDX)+1
      IONCNT=IONCNT-1
      IF (IONCNT.EQ.0) THEN
        TIME1=TIME2
      ELSE
        ASEC(INDACT(IONCNT))=TIME2
      ENDIF
      IF(INFO.GT.1) WRITE(MONI,*) ' exit ',NAME(INDX),TIME2
C
C      if(igraph.gt.0) call gmoff(indx,sec(indx))
C
      RETURN
C
C     terminate monitor and print statistics
C
      ENTRY MONEND
      CALL SECOND (TIME0)
      SEC0=SEC0+TIME0-TIME1
C
      SUM=1.E-10
      DO 200 I=1,MAXIND
      SUM=SUM+SEC(I)
      IF(COUNT(I).LE.0) GOTO 200
      ASEC(I)=SEC(I)/FLOAT(COUNT(I))
200   CONTINUE
      SUM0=SUM+SEC0
C
      DO 250 I=1,MAXIND
      PC1(I)=100.*SEC(I)/SUM0
      PC2(I)=100.*SEC(I)/SUM
250   CONTINUE
      PC10=100.*SEC0/SUM0
      PC20=100.*SEC0/SUM
C
      WRITE(MONI,9500)
      WRITE(MONI,9510)
      WRITE(MONI,9505)
9500  FORMAT(///)
9510  FORMAT(1X,75('#'))
9505  FORMAT(' #',73X,'#')
      WRITE(MONI,9505)
      WRITE(MONI,9512) TEXT
9512  FORMAT(' #   Results from time monitor program for: ',A29,2X,'#')
      WRITE(MONI,9505)
      WRITE(MONI,9514) SUM0,SUM
9514  FORMAT(' #   Total time:',F11.3,5X,'Sum of parts:',F11.3,19X,'#')
      WRITE(MONI,9505)
      WRITE(MONI,9520)
9520  FORMAT(' #   ',2X,'name',12X,'calls',7X,'time',4X,'av-time',
     1       4X,'% total',6X,'% sum   #')
C
      I0=1
      WRITE(MONI,9550) NAME0,I0,SEC0,SEC0,PC10,PC20
9550  FORMAT(' #   ',A15,I8,F11.3,F11.4,F11.2,F11.2,'   #')
C
      DO 300 I=1,MAXIND
      WRITE(MONI,9550) NAME(I),COUNT(I),SEC(I),ASEC(I),PC1(I),PC2(I)
300   CONTINUE
C
C
      WRITE(MONI,9505)
      WRITE(MONI,9510)
      WRITE(MONI,9500)
C
C
C      IF(IGRAPH.GT.0) CALL GMEND
C
      RETURN
C
C  error exits
C
1010  CONTINUE
      WRITE(MONI,9010) INDX
9010  FORMAT(/,' error in subroutine monon or monoff',/,
     $         '   indx out of range    indx=',I4)
      GOTO 1111
C
1020  CONTINUE
      WRITE(MONI,9020) INDX
9020  FORMAT(/,' error in subroutine monoff',/,'   indx out of range',/,
     1         '   indx=',I4)
      GOTO 1111
C
1030  CONTINUE
      WRITE(MONI,9030) INDX
9030  FORMAT(/,' error in subroutine monon',/,
     $         '   measurement is already running for ',
     1         '   indx=',I4)
      GOTO 1111
C
1040  CONTINUE
      WRITE(MONI,9040) INDX
9040  FORMAT(/,' error in subroutine monoff',/,
     $         '   measurement has never been activated for ',
     1         '   indx=',I4)
      GOTO 1111
C
1190  CONTINUE
      WRITE(MONI,9190) MAXTAB,INDX
9190  FORMAT(/,' error in subroutine mondef',/,'   indx gt ',I4,/,
     1         '   indx=',I4)
      GOTO 1111
C
1290  CONTINUE
      WRITE(MONI,9290) INDX
9290  FORMAT(/,' error in subroutine mondef',/,'   indx = ',I4,
     1         '   already in use' )
      GOTO 1111
C
1111  STOP
C
C  end subroutine monitor
C
      END
C
C*    Group  Machine dependent subroutines and functions
C
      SUBROUTINE SECOND(TIME)
      REAL TIME
      TIME=0.0E0
      RETURN
      END
      DOUBLE PRECISION FUNCTION D1MACH(I)
C
C  DOUBLE-PRECISION MACHINE CONSTANTS
C
C  D1MACH( 1) = B**(EMIN-1), THE SMALLEST POSITIVE MAGNITUDE.
C
C  D1MACH( 2) = B**EMAX*(1 - B**(-T)), THE LARGEST MAGNITUDE.
C
C  D1MACH( 3) = B**(-T), THE SMALLEST RELATIVE SPACING.
C
C  D1MACH( 4) = B**(1-T), THE LARGEST RELATIVE SPACING.
C
C  D1MACH( 5) = LOG10(B)
C
C  TO ALTER THIS FUNCTION FOR A PARTICULAR ENVIRONMENT,
C  THE DESIRED SET OF DATA STATEMENTS SHOULD BE ACTIVATED BY
C  REMOVING THE C FROM COLUMN 1.
C  ON RARE MACHINES A STATIC STATEMENT MAY NEED TO BE ADDED.
C  (BUT PROBABLY MORE SYSTEMS PROHIBIT IT THAN REQUIRE IT.)
C
C  FOR IEEE-ARITHMETIC MACHINES (BINARY STANDARD), ONE OF THE FIRST
C  TWO SETS OF CONSTANTS BELOW SHOULD BE APPROPRIATE.
C
C  WHERE POSSIBLE, DECIMAL, OCTAL OR HEXADECIMAL CONSTANTS ARE USED
C  TO SPECIFY THE CONSTANTS EXACTLY.  SOMETIMES THIS REQUIRES USING
C  EQUIVALENT INTEGER ARRAYS.  IF YOUR COMPILER USES HALF-WORD
C  INTEGERS BY DEFAULT (SOMETIMES CALLED INTEGER*2), YOU MAY NEED TO
C  CHANGE INTEGER TO INTEGER*4 OR OTHERWISE INSTRUCT YOUR COMPILER
C  TO USE FULL-WORD INTEGERS IN THE NEXT 5 DECLARATIONS.
C
      INTEGER SMALL(4)
      INTEGER LARGE(4)
      INTEGER RIGHT(4)
      INTEGER DIVER(4)
      INTEGER LOG10(4)
C
      DOUBLE PRECISION DMACH(5)
C
      EQUIVALENCE (DMACH(1),SMALL(1))
      EQUIVALENCE (DMACH(2),LARGE(1))
      EQUIVALENCE (DMACH(3),RIGHT(1))
      EQUIVALENCE (DMACH(4),DIVER(1))
      EQUIVALENCE (DMACH(5),LOG10(1))
C
C     MACHINE CONSTANTS FOR IEEE ARITHMETIC MACHINES, SUCH AS THE AT&T
C     3B SERIES AND MOTOROLA 68000 BASED MACHINES (E.G. SUN 3 AND AT&T
C     PC 7300), IN WHICH THE MOST SIGNIFICANT BYTE IS STORED FIRST.
C
       DATA SMALL(1),SMALL(2) /    1048576,          0 /
       DATA LARGE(1),LARGE(2) / 2146435071,         -1 /
       DATA RIGHT(1),RIGHT(2) / 1017118720,          0 /
       DATA DIVER(1),DIVER(2) / 1018167296,          0 /
       DATA LOG10(1),LOG10(2) / 1070810131, 1352628735 /
C
C     MACHINE CONSTANTS FOR IEEE ARITHMETIC MACHINES AND 8087-BASED
C     MICROS, SUCH AS THE IBM PC AND AT&T 6300, IN WHICH THE LEAST
C     SIGNIFICANT BYTE IS STORED FIRST.
C
C      DATA SMALL(1),SMALL(2) /          0,    1048576 /
C      DATA LARGE(1),LARGE(2) /         -1, 2146435071 /
C      DATA RIGHT(1),RIGHT(2) /          0, 1017118720 /
C      DATA DIVER(1),DIVER(2) /          0, 1018167296 /
C      DATA LOG10(1),LOG10(2) / 1352628735, 1070810131 /
C
C     MACHINE CONSTANTS FOR AMDAHL MACHINES.
C
C      DATA SMALL(1),SMALL(2) /    1048576,          0 /
C      DATA LARGE(1),LARGE(2) / 2147483647,         -1 /
C      DATA RIGHT(1),RIGHT(2) /  856686592,          0 /
C      DATA DIVER(1),DIVER(2) /  873463808,          0 /
C      DATA LOG10(1),LOG10(2) / 1091781651, 1352628735 /
C
C     MACHINE CONSTANTS FOR THE BURROUGHS 1700 SYSTEM.
C
C      DATA SMALL(1) / ZC00800000 /
C      DATA SMALL(2) / Z000000000 /
C
C      DATA LARGE(1) / ZDFFFFFFFF /
C      DATA LARGE(2) / ZFFFFFFFFF /
C
C      DATA RIGHT(1) / ZCC5800000 /
C      DATA RIGHT(2) / Z000000000 /
C
C      DATA DIVER(1) / ZCC6800000 /
C      DATA DIVER(2) / Z000000000 /
C
C      DATA LOG10(1) / ZD00E730E7 /
C      DATA LOG10(2) / ZC77800DC0 /
C
C     MACHINE CONSTANTS FOR THE BURROUGHS 5700 SYSTEM.
C
C      DATA SMALL(1) / O1771000000000000 /
C      DATA SMALL(2) / O0000000000000000 /
C
C      DATA LARGE(1) / O0777777777777777 /
C      DATA LARGE(2) / O0007777777777777 /
C
C      DATA RIGHT(1) / O1461000000000000 /
C      DATA RIGHT(2) / O0000000000000000 /
C
C      DATA DIVER(1) / O1451000000000000 /
C      DATA DIVER(2) / O0000000000000000 /
C
C      DATA LOG10(1) / O1157163034761674 /
C      DATA LOG10(2) / O0006677466732724 /
C
C     MACHINE CONSTANTS FOR THE BURROUGHS 6700/7700 SYSTEMS.
C
C      DATA SMALL(1) / O1771000000000000 /
C      DATA SMALL(2) / O7770000000000000 /
C
C      DATA LARGE(1) / O0777777777777777 /
C      DATA LARGE(2) / O7777777777777777 /
C
C      DATA RIGHT(1) / O1461000000000000 /
C      DATA RIGHT(2) / O0000000000000000 /
C
C      DATA DIVER(1) / O1451000000000000 /
C      DATA DIVER(2) / O0000000000000000 /
C
C      DATA LOG10(1) / O1157163034761674 /
C      DATA LOG10(2) / O0006677466732724 /
C
C     MACHINE CONSTANTS FOR FTN4 ON THE CDC 6000/7000 SERIES.
C
C      DATA SMALL(1) / 00564000000000000000B /
C      DATA SMALL(2) / 00000000000000000000B /
C
C      DATA LARGE(1) / 37757777777777777777B /
C      DATA LARGE(2) / 37157777777777777774B /
C
C      DATA RIGHT(1) / 15624000000000000000B /
C      DATA RIGHT(2) / 00000000000000000000B /
C
C      DATA DIVER(1) / 15634000000000000000B /
C      DATA DIVER(2) / 00000000000000000000B /
C
C      DATA LOG10(1) / 17164642023241175717B /
C      DATA LOG10(2) / 16367571421742254654B /
C
C     MACHINE CONSTANTS FOR FTN5 ON THE CDC 6000/7000 SERIES.
C
C      DATA SMALL(1) / O"00564000000000000000" /
C      DATA SMALL(2) / O"00000000000000000000" /
C
C      DATA LARGE(1) / O"37757777777777777777" /
C      DATA LARGE(2) / O"37157777777777777774" /
C
C      DATA RIGHT(1) / O"15624000000000000000" /
C      DATA RIGHT(2) / O"00000000000000000000" /
C
C      DATA DIVER(1) / O"15634000000000000000" /
C      DATA DIVER(2) / O"00000000000000000000" /
C
C      DATA LOG10(1) / O"17164642023241175717" /
C      DATA LOG10(2) / O"16367571421742254654" /
C
C     MACHINE CONSTANTS FOR CONVEX C-1
C
C      DATA SMALL(1),SMALL(2) / '00100000'X, '00000000'X /
C      DATA LARGE(1),LARGE(2) / '7FFFFFFF'X, 'FFFFFFFF'X /
C      DATA RIGHT(1),RIGHT(2) / '3CC00000'X, '00000000'X /
C      DATA DIVER(1),DIVER(2) / '3CD00000'X, '00000000'X /
C      DATA LOG10(1),LOG10(2) / '3FF34413'X, '509F79FF'X /
C
C     MACHINE CONSTANTS FOR THE CRAY 1, XMP, 2, AND 3.
C
C      DATA SMALL(1) / 201354000000000000000B /
C      DATA SMALL(2) / 000000000000000000000B /
C
C      DATA LARGE(1) / 577767777777777777777B /
C      DATA LARGE(2) / 000007777777777777776B /
C
C      DATA RIGHT(1) / 376434000000000000000B /
C      DATA RIGHT(2) / 000000000000000000000B /
C
C      DATA DIVER(1) / 376444000000000000000B /
C      DATA DIVER(2) / 000000000000000000000B /
C
C      DATA LOG10(1) / 377774642023241175717B /
C      DATA LOG10(2) / 000007571421742254654B /
C
C     MACHINE CONSTANTS FOR THE DATA GENERAL ECLIPSE S/200
C
C     NOTE - IT MAY BE APPROPRIATE TO INCLUDE THE FOLLOWING LINE -
C     STATIC DMACH(5)
C
C      DATA SMALL/20K,3*0/,LARGE/77777K,3*177777K/
C      DATA RIGHT/31420K,3*0/,DIVER/32020K,3*0/
C      DATA LOG10/40423K,42023K,50237K,74776K/
C
C     MACHINE CONSTANTS FOR THE HARRIS SLASH 6 AND SLASH 7
C
C      DATA SMALL(1),SMALL(2) / '20000000, '00000201 /
C      DATA LARGE(1),LARGE(2) / '37777777, '37777577 /
C      DATA RIGHT(1),RIGHT(2) / '20000000, '00000333 /
C      DATA DIVER(1),DIVER(2) / '20000000, '00000334 /
C      DATA LOG10(1),LOG10(2) / '23210115, '10237777 /
C
C     MACHINE CONSTANTS FOR THE HONEYWELL DPS 8/70 SERIES.
C
C      DATA SMALL(1),SMALL(2) / O402400000000, O000000000000 /
C      DATA LARGE(1),LARGE(2) / O376777777777, O777777777777 /
C      DATA RIGHT(1),RIGHT(2) / O604400000000, O000000000000 /
C      DATA DIVER(1),DIVER(2) / O606400000000, O000000000000 /
C      DATA LOG10(1),LOG10(2) / O776464202324, O117571775714 /
C
C     MACHINE CONSTANTS FOR THE IBM 360/370 SERIES,
C     THE XEROX SIGMA 5/7/9 AND THE SEL SYSTEMS 85/86.
C
C      DATA SMALL(1),SMALL(2) / Z00100000, Z00000000 /
C      DATA LARGE(1),LARGE(2) / Z7FFFFFFF, ZFFFFFFFF /
C      DATA RIGHT(1),RIGHT(2) / Z33100000, Z00000000 /
C      DATA DIVER(1),DIVER(2) / Z34100000, Z00000000 /
C      DATA LOG10(1),LOG10(2) / Z41134413, Z509F79FF /
C
C     MACHINE CONSTANTS FOR THE INTERDATA 8/32
C     WITH THE UNIX SYSTEM FORTRAN 77 COMPILER.
C
C     FOR THE INTERDATA FORTRAN VII COMPILER REPLACE
C     THE Z'S SPECIFYING HEX CONSTANTS WITH Y'S.
C
C      DATA SMALL(1),SMALL(2) / Z'00100000', Z'00000000' /
C      DATA LARGE(1),LARGE(2) / Z'7EFFFFFF', Z'FFFFFFFF' /
C      DATA RIGHT(1),RIGHT(2) / Z'33100000', Z'00000000' /
C      DATA DIVER(1),DIVER(2) / Z'34100000', Z'00000000' /
C      DATA LOG10(1),LOG10(2) / Z'41134413', Z'509F79FF' /
C
C     MACHINE CONSTANTS FOR THE PDP-10 (KA PROCESSOR).
C
C      DATA SMALL(1),SMALL(2) / "033400000000, "000000000000 /
C      DATA LARGE(1),LARGE(2) / "377777777777, "344777777777 /
C      DATA RIGHT(1),RIGHT(2) / "113400000000, "000000000000 /
C      DATA DIVER(1),DIVER(2) / "114400000000, "000000000000 /
C      DATA LOG10(1),LOG10(2) / "177464202324, "144117571776 /
C
C     MACHINE CONSTANTS FOR THE PDP-10 (KI PROCESSOR).
C
C      DATA SMALL(1),SMALL(2) / "000400000000, "000000000000 /
C      DATA LARGE(1),LARGE(2) / "377777777777, "377777777777 /
C      DATA RIGHT(1),RIGHT(2) / "103400000000, "000000000000 /
C      DATA DIVER(1),DIVER(2) / "104400000000, "000000000000 /
C      DATA LOG10(1),LOG10(2) / "177464202324, "047674776746 /
C
C     MACHINE CONSTANTS FOR PDP-11 FORTRANS SUPPORTING
C     32-BIT INTEGERS (EXPRESSED IN INTEGER AND OCTAL).
C
C      DATA SMALL(1),SMALL(2) /    8388608,           0 /
C      DATA LARGE(1),LARGE(2) / 2147483647,          -1 /
C      DATA RIGHT(1),RIGHT(2) /  612368384,           0 /
C      DATA DIVER(1),DIVER(2) /  620756992,           0 /
C      DATA LOG10(1),LOG10(2) / 1067065498, -2063872008 /
C
C      DATA SMALL(1),SMALL(2) / O00040000000, O00000000000 /
C      DATA LARGE(1),LARGE(2) / O17777777777, O37777777777 /
C      DATA RIGHT(1),RIGHT(2) / O04440000000, O00000000000 /
C      DATA DIVER(1),DIVER(2) / O04500000000, O00000000000 /
C      DATA LOG10(1),LOG10(2) / O07746420232, O20476747770 /
C
C     MACHINE CONSTANTS FOR PDP-11 FORTRANS SUPPORTING
C     16-BIT INTEGERS (EXPRESSED IN INTEGER AND OCTAL).
C
C      DATA SMALL(1),SMALL(2) /    128,      0 /
C      DATA SMALL(3),SMALL(4) /      0,      0 /
C
C      DATA LARGE(1),LARGE(2) /  32767,     -1 /
C      DATA LARGE(3),LARGE(4) /     -1,     -1 /
C
C      DATA RIGHT(1),RIGHT(2) /   9344,      0 /
C      DATA RIGHT(3),RIGHT(4) /      0,      0 /
C
C      DATA DIVER(1),DIVER(2) /   9472,      0 /
C      DATA DIVER(3),DIVER(4) /      0,      0 /
C
C      DATA LOG10(1),LOG10(2) /  16282,   8346 /
C      DATA LOG10(3),LOG10(4) / -31493, -12296 /
C
C      DATA SMALL(1),SMALL(2) / O000200, O000000 /
C      DATA SMALL(3),SMALL(4) / O000000, O000000 /
C
C      DATA LARGE(1),LARGE(2) / O077777, O177777 /
C      DATA LARGE(3),LARGE(4) / O177777, O177777 /
C
C      DATA RIGHT(1),RIGHT(2) / O022200, O000000 /
C      DATA RIGHT(3),RIGHT(4) / O000000, O000000 /
C
C      DATA DIVER(1),DIVER(2) / O022400, O000000 /
C      DATA DIVER(3),DIVER(4) / O000000, O000000 /
C
C      DATA LOG10(1),LOG10(2) / O037632, O020232 /
C      DATA LOG10(3),LOG10(4) / O102373, O147770 /
C
C     MACHINE CONSTANTS FOR THE PRIME 50 SERIES SYSTEMS
C     WTIH 32-BIT INTEGERS AND 64V MODE INSTRUCTIONS,
C     SUPPLIED BY IGOR BRAY.
C
C      DATA SMALL(1),SMALL(2) / :10000000000, :00000100001 /
C      DATA LARGE(1),LARGE(2) / :17777777777, :37777677775 /
C      DATA RIGHT(1),RIGHT(2) / :10000000000, :00000000122 /
C      DATA DIVER(1),DIVER(2) / :10000000000, :00000000123 /
C      DATA LOG10(1),LOG10(2) / :11504046501, :07674600177 /
C
C     MACHINE CONSTANTS FOR THE SEQUENT BALANCE 8000
C
C      DATA SMALL(1),SMALL(2) / $00000000,  $00100000 /
C      DATA LARGE(1),LARGE(2) / $FFFFFFFF,  $7FEFFFFF /
C      DATA RIGHT(1),RIGHT(2) / $00000000,  $3CA00000 /
C      DATA DIVER(1),DIVER(2) / $00000000,  $3CB00000 /
C      DATA LOG10(1),LOG10(2) / $509F79FF,  $3FD34413 /
C
C     MACHINE CONSTANTS FOR THE UNIVAC 1100 SERIES.
C
C      DATA SMALL(1),SMALL(2) / O000040000000, O000000000000 /
C      DATA LARGE(1),LARGE(2) / O377777777777, O777777777777 /
C      DATA RIGHT(1),RIGHT(2) / O170540000000, O000000000000 /
C      DATA DIVER(1),DIVER(2) / O170640000000, O000000000000 /
C      DATA LOG10(1),LOG10(2) / O177746420232, O411757177572 /
C
C     MACHINE CONSTANTS FOR THE VAX UNIX F77 COMPILER
C
C      DATA SMALL(1),SMALL(2) /        128,           0 /
C      DATA LARGE(1),LARGE(2) /     -32769,          -1 /
C      DATA RIGHT(1),RIGHT(2) /       9344,           0 /
C      DATA DIVER(1),DIVER(2) /       9472,           0 /
C      DATA LOG10(1),LOG10(2) /  546979738,  -805796613 /
C
C     MACHINE CONSTANTS FOR THE VAX-11 WITH
C     FORTRAN IV-PLUS COMPILER
C
C      DATA SMALL(1),SMALL(2) / Z00000080, Z00000000 /
C      DATA LARGE(1),LARGE(2) / ZFFFF7FFF, ZFFFFFFFF /
C      DATA RIGHT(1),RIGHT(2) / Z00002480, Z00000000 /
C      DATA DIVER(1),DIVER(2) / Z00002500, Z00000000 /
C      DATA LOG10(1),LOG10(2) / Z209A3F9A, ZCFF884FB /
C
C     MACHINE CONSTANTS FOR VAX/VMS VERSION 2.2
C
C      DATA SMALL(1),SMALL(2) /       '80'X,        '0'X /
C      DATA LARGE(1),LARGE(2) / 'FFFF7FFF'X, 'FFFFFFFF'X /
C      DATA RIGHT(1),RIGHT(2) /     '2480'X,        '0'X /
C      DATA DIVER(1),DIVER(2) /     '2500'X,        '0'X /
C      DATA LOG10(1),LOG10(2) / '209A3F9A'X, 'CFF884FB'X /
C
C/6S
      IF (I .LT. 1  .OR.  I .GT. 6) GOTO 999
      IF (I .LE. 5 ) THEN
        D1MACH = DMACH(I)
      ELSE IF (I .EQ. 6) THEN
C       D1MACH = DSQRT(DMACH(1)/DMACH(3))
        D1MACH = 4.94D-32
      ENDIF
      RETURN
  999 WRITE(6,1999) I
 1999 FORMAT(' D1MACH - I OUT OF BOUNDS',I10)
      STOP
      END
