;% Copyright (c) 1990-1994 The MITRE Corporation
;% 
;% Authors: W. M. Farmer, J. D. Guttman, F. J. Thayer
;%   
;% The MITRE Corporation (MITRE) provides this software to you without
;% charge to use, copy, modify or enhance for any legitimate purpose
;% provided you reproduce MITRE's copyright notice in any copy or
;% derivative work of this software.
;% 
;% This software is the copyright work of MITRE.  No ownership or other
;% proprietary interest in this software is granted you other than what
;% is granted in this license.
;% 
;% Any modification or enhancement of this software must identify the
;% part of this software that was modified, by whom and when, and must
;% inherit this license including its warranty disclaimers.
;% 
;% MITRE IS PROVIDING THE PRODUCT "AS IS" AND MAKES NO WARRANTY, EXPRESS
;% OR IMPLIED, AS TO THE ACCURACY, CAPABILITY, EFFICIENCY OR FUNCTIONING
;% OF THIS SOFTWARE AND DOCUMENTATION.  IN NO EVENT WILL MITRE BE LIABLE
;% FOR ANY GENERAL, CONSEQUENTIAL, INDIRECT, INCIDENTAL, EXEMPLARY OR
;% SPECIAL DAMAGES, EVEN IF MITRE HAS BEEN ADVISED OF THE POSSIBILITY OF
;% SUCH DAMAGES.
;% 
;% You, at your expense, hereby indemnify and hold harmless MITRE, its
;% Board of Trustees, officers, agents and employees, from any and all
;% liability or damages to third parties, including attorneys' fees,
;% court costs, and other related costs and expenses, arising out of your
;% use of this software irrespective of the cause of said liability.
;% 
;% The export from the United States or the subsequent reexport of this
;% software is subject to compliance with United States export control
;% and munitions control restrictions.  You agree that in the event you
;% seek to export this software or any derivative work thereof, you
;% assume full responsibility for obtaining all necessary export licenses
;% and approvals and for assuring compliance with applicable reexport
;% restrictions.
;% 
;% 
;% COPYRIGHT NOTICE INSERTED: Mon Apr 11 11:42:27 EDT 1994


(herald READ-PRINT)


; Transforming sexps into expressions and vice versa.
; To ``print'' a formal symbol, we use its name, and to ``print'' a compound
; expression we find an sexp-builder and apply it to the expression.  The
; default is given below.  

(define (EXPRESSION->WITHLESS-SEXP expr)
  (if (formal-symbol? expr)
      (name expr)
      (let ((constructor (or (and (use-quasi-constructor-form?)
				  (quasi-constructor-if-enabled
				   (expression-quasi-constructor expr)))
			     (expression-constructor expr))))
	((or (sexp-builder constructor)
	     old-default-builder)
	 expr))))

(define (expression->sexp-new expr)
  (let ((with-vars
	 (set-difference
	  (set-difference
	   (free-variables expr)
	   (expression-var-name-conflict? expr))
	  (current-language-default-variables))))
    (if with-vars
	`(with ,(var-list->sexp with-vars)
	       ,(expression->withful-sexp expr))
	(expression->withful-sexp expr))))

(define (EXPRESSION->WITHFUL-SEXP expr)
  (if (formal-symbol? expr)
      (name expr)
      (let ((constructor (or (and (use-quasi-constructor-form?)
				  (quasi-constructor-if-enabled
				   (expression-quasi-constructor expr)))
			     (expression-constructor expr))))
	((or (sexp-builder constructor)
	     new-default-builder)
	 expr))))

; The default way to build an sexp is to use the name of the constructor and
; map through the components to get their sexp forms.  Withs must be inserted
; when the sorts of ambiguously named variables become unique.

(define (NEW-DEFAULT-BUILDER expr)
  (receive (constr comps)
    (if (and (use-quasi-constructor-form?)
	     (quasi-constructor-if-enabled
	      (expression-quasi-constructor expr)))
	(expression-quasi-constructor-&-components expr)
	(return (expression-constructor expr)
		(expression-components expr)))
    (let ((conflicts (expression-var-name-conflict? expr)))
      (if conflicts
	  (cons
	   (name constr)
	   (map 
	    (lambda (c)
	      (let ((with-vars
		     (set-intersection
		      (free-variables c)
		      (set-difference conflicts (expression-var-name-conflict? c)))))
		(if with-vars
		    `(with ,(var-list->sexp with-vars)
			   ,(expression->withful-sexp c))
		    (expression->withful-sexp c))))
	    comps))
	  (cons (name constr)
	      (map 
	       (lambda (c) (expression->withful-sexp c))
	       comps))))))

(define DEFAULT-BUILDER NEW-DEFAULT-BUILDER)

(define (OLD-DEFAULT-BUILDER expr)
  (receive (q-constr q-comps)
    (if (and (use-quasi-constructor-form?)
	     (quasi-constructor-if-enabled
	      (expression-quasi-constructor expr)))
	(expression-quasi-constructor-&-components expr)
	(return '#f '()))
    (receive (constr comps)
      (if q-constr
	  (return q-constr q-comps)
	  (return (expression-constructor expr)
		  (expression-components expr)))
      (cons (name constr)
	    (map 
	     (lambda (c) (expression->withless-sexp c))
	     comps)))))


; NAME-FORMAL-SYMBOL-ALIST is an association of names and formal symbols that
; have already been identified in the recursive analysis of the sexp.
; Particular decoders (EG the quantifiers) add to it.  Note that this procedure
; may return expressions with WITH as main constructor.  To discard the
; wrapping withs, call REMOVE-LEADING-WITHS, or else use QS->E or QR (which
; contain calls to it) to read the expression.  

(define (SEXP->EXPRESSION-1 sexp language name-formal-symbol-alist)
  (let ((error-test
	 (lambda (expr)
	   (if (contains-expression? language expr)
	       expr
	       (imps-error "SEXP->EXPRESSION-1: expression ~S fails to belong to language ~S"
		      expr language)))))
    (error-test
     (let ((constructor (find-sexp-constructor sexp)))
       (cond
	((transparent-constructor? constructor)
	 (let ((decoder (or (sexp-decoder constructor)	;use decoder 
			    rec-descent-sexp-decoder)))
	   (car						;but don't apply constructor 
	    (decoder sexp language name-formal-symbol-alist))))
	((or (quasi-constructor? constructor)
	     (constructor? constructor))		;for constructor,
	 (let ((decoder (or (sexp-decoder constructor)	;apply decoder 
			    rec-descent-sexp-decoder)))
	   (apply constructor
		  (decoder sexp language name-formal-symbol-alist))))
	((seek-symbol-form sexp)			;for formal symbol,
	 =>
	 (lambda (sexp)
	   (cond ((assq sexp name-formal-symbol-alist)	;look in alist
		  => cdr)				;and language
		 ((find-constant language sexp))
		 (else
		  (imps-error "SEXP->EXPRESSION-1:  Cannot locate ~A in ~A"
			 sexp language)))))
	(else						;otherwise, must be 
	 (apply						;application 
	  apply-operator
	  (map (lambda (c)
		 (sexp->expression-1 c language name-formal-symbol-alist))
	       sexp))))))))

(define (SEXP->EXPRESSION-2 sexp language name-formal-symbol-alist)
  (let ((constructor (find-sexp-constructor sexp)))
    (cond
     ((transparent-constructor? constructor)
      (let ((decoder (or (sexp-decoder constructor)	;use decoder 
			 rec-descent-sexp-decoder)))
	(car						;but don't apply constructor 
	 (decoder sexp language name-formal-symbol-alist))))
     ((constructor? constructor)			;for constructor,
      (let ((decoder (or (sexp-decoder constructor)	;apply decoder 
			 rec-descent-sexp-decoder)))
	(apply
	 constructor
	 (decoder sexp language name-formal-symbol-alist))))
     ((and (quasi-constructor? constructor)		;check if shadowed,
	   (let ((the-name
		  (seek-symbol-form (name constructor))))
	     (or (not the-name)
		 (and (not (assq the-name name-formal-symbol-alist))
		      (not (find-constant language the-name))))))
      (let ((decoder
	     (or (sexp-decoder constructor)		;otherwise treat as a
		 rec-descent-sexp-decoder)))		;constructor 

	(apply
	 constructor
	 (decoder sexp language name-formal-symbol-alist))))

     ((seek-symbol-form sexp)				;for formal symbol,
      =>
      (lambda (sexp)
	(cond ((assq sexp name-formal-symbol-alist)	;look in alist
	       => cdr)					;and language
	      ((find-constant language sexp))
	      (else
	       (imps-error "SEXP->EXPRESSION-2:  Cannot locate ~A in ~A"
			   sexp language)))))
     (else						;otherwise, must be 
      (apply						;application 
       apply-operator
       (map (lambda (c)
	      (sexp->expression-2 c language name-formal-symbol-alist))
	    sexp))))))


; Return two values.  The first is either NIL or the sole constructor or
; quasi-constructor whose name appears as a member of sexp.  The second is the
; sexp with the constructor name (if any) deleted.  Too many constructors cause
; an error.

(define (SEXP-CONSTRUCTOR+BODY sexp)
  (if (possible-symbol-form? sexp)
      (return nil sexp)
      (iterate iter ((remnant sexp)(constructors nil))
	(cond
	 ((null? remnant)
	  (if (null? (cdr constructors))
	      (return (car constructors)
		      (if constructors 
			  (delq (name (car constructors)) sexp)
			  sexp))
	      (imps-error "SEXP-CONSTRUCTOR:  Too many constructors ~S in sexp ~A"
			  constructors
			  (string-downcase! (format nil "~S" sexp)))))
	 ((and (possible-symbol-form? (car remnant))
	       (or (symbol->quasi-constructor (car remnant))
		   (symbol->constructor (car remnant))))
	  => (lambda (c) (iter (cdr remnant)(cons c constructors))))
	 (else (iter (cdr remnant) constructors))))))

; Return one value, either NIL or the sole constructor or quasi-constructor
; whose name appears as a member of sexp.  Too many constructors no longer
; cause an error.

(define (FIND-SEXP-CONSTRUCTOR sexp)
  (cond ((list? sexp)
	 (iterate iter ((remnant sexp)(constructors nil))
	   (cond
	    ((null? remnant)
	     ;; Used to be the following, but why screw around with this?
	     ;;
	     ;;(if (null? (cdr constructors))
	     ;; (car constructors)
	     ;; (imps-error "SEXP-CONSTRUCTOR:  Too many constructors ~S in sexp ~A"
	     ;; constructors
	     ;; (string-downcase! (format nil "~S" sexp))))
	     '#f)
	    ((and (possible-symbol-form? (car remnant))
		  (or (symbol->quasi-constructor (car remnant))
		      (symbol->constructor (car remnant))))
	     => (lambda (c) c))
	    (else (iter (cdr remnant) constructors)))))
	((eq? sexp 'truth)
	 the-true)
	((eq? sexp 'falsehood)
	 the-false)
	(else nil)))

; Procedure describing the default sexp-decoder 

(define (REC-DESCENT-SEXP-DECODER sexp language name-formal-symbol-alist)
  (map
   (lambda (c)
     (sexp->expression-1 c language name-formal-symbol-alist))
   (cdr sexp)))

(define (symbol-means-constructor-or-quasi-constructor? symbol)
  (or (symbol->constructor symbol)
      (symbol->quasi-constructor symbol)))

(define (symbol-means-binding-constructor? symbol)
  (cond ((symbol->constructor symbol)
	 => (lambda (c)
	      (or (binding-constructor? c)
		  (eq? c with)
		  (eq? c falselike))))
	(else '#f)))

; The main procedure for producing an sexp from an expression.  Wraps it in a
; WITH if appropriate. 

(define EXPRESSION->SEXP expression->sexp-new)

(define (EXPRESSION->SEXP-old expression)
  (expression->withless-sexp
   (if (or (closed? expression)
	   (subset? (free-variables expression)
		    (current-language-default-variables)) ;if necessary, apply 
	   (eq? with					;WITH to force variable 
		(expression-constructor expression)))	;sortings to be shown
       expression
       (with expression))))

; Construct an expression from a given sexp, suppressing WITHs. 

(define (SEXP->EXPRESSION language-or-theory sexp)
  (if (and (not (language? language-or-theory))
	   (not (theory? language-or-theory))
	   (or (language? sexp)
	       (theory? sexp)))
      (sexp->expression sexp language-or-theory)
      (remove-leading-withs 
       (let ((language (if (language? language-or-theory)
			   language-or-theory
			   (theory-language language-or-theory))))
	 (sexp->expression-1 sexp 
			     language
			     (default-sortings->name-var-alist language))))))

(define (default-sortings->name-var-alist language)
  (map
   (lambda (pair)
     (cons (car pair) (find-variable (car pair) (cdr pair))))
   (language-default-sortings language)))

(define (add-language-default-sorting language var-name sorting-list)
  (let ((sorting (list->sort language sorting-list)))
    (push (language-default-sortings language)
	  (cons var-name sorting))
    (set (current-language-default-variables)
	 (get-language-default-variables (current-language)))))

(define (get-language-default-variables language)
  (map
   (lambda (pair)
     (find-variable (car pair) (cdr pair)))
   (language-default-sortings language)))


(define current-language-default-variables
  (make-simple-switch 'current-language-default-variables list? '#f))

(define SEXP->WITHLESS-EXPRESSION sexp->expression) 


; Current theory and current language

(define current-theory
  (let ((the-theory '#f))
    (object
	(lambda () the-theory)
      ((setter self)
       (lambda (new)
	 (imps-enforce (object
			   (lambda (t)(or (not t) (theory? t)))
			 ((print self port)
			  (print '(lambda (t)(or (not t) (theory? t))) port)))
		       new)
	 (set (current-language-default-variables)
	      (get-language-default-variables (theory-language new)))
	 (set the-theory new))))))

;;; Formerly:  
;;;  (make-simple-switch 'current-theory (lambda (t)(or (not t) (theory? t))) '#f)

(let ((previous-theory-stack '()))

  (define (push-current-theory)
    (if (or (not (current-theory))
	    (theory? (current-theory)))
	(push previous-theory-stack (current-theory))
	(imps-error "push-current-theory: bad current-theory ~S" (current-theory))))

  (define (pop-current-theory)
    (if (or (not (car previous-theory-stack))
	    (theory? (car previous-theory-stack)))
	(block0
	 (current-theory)
	 (set (current-theory) (car previous-theory-stack))
	 (set previous-theory-stack (cdr previous-theory-stack)))
	(imps-error "pop-current-theory: bad top ~S" (car previous-theory-stack)))))

(define default-language 
  (make-simple-switch 'default-language (always '#t) '#f))

(define current-language
  (let ((use-theory? '#t))
    (object
	(lambda ()
	  (cond ((and use-theory? (current-theory)) => theory-language)
		(else (default-language))))
      ((setter self)
       (lambda (new)
	 (cond ((language? new)
		(imps-warning "set current-language: setting current language to ~S: why?~&"
			      new)
		(set use-theory? '#f)
		(set (current-language-default-variables)
		     (get-language-default-variables new))
		(set (default-language) new))
	       (else (set use-theory? '#t))))))))


(define (QR-SEXP sexp . language)
  (let ((language (if language
		      (car language)
		      (current-language))))
    (sexp->expression language sexp)))

(define QS->E qr-sexp)

;;;(define (imps-sexp-read-proc language input)
;;;  (if (string? input)
;;;      (with-input-from-string (port input)
;;;	(imps-sexp-read-proc language port))
;;;      (read input)))

(define (imps-sexp-read-proc language port)
  (ignore language)
  (read port))

;; (IMPS-READER) is a procedure which takes a language and a port and returns an 
;; s-expression. imps-string-read-proc is the more frequently used one.

(define IMPS-READER
  (make-simple-switch
   'imps-reader
   procedure?
   (lambda (language port)
     (ignore language)
     (read port))))

; IMPS-READ-PROCEDURE takes:
;  1.  A language
;  2.  an "input", namely either a string or a port.
; It returns an expression.  

(define (imps-read-procedure language input)
  (let ((port (if (port? input)
		  input
		  (string->input-port input))))
    (sexp->expression
     language
     ((imps-reader) language port))))


;;;(define imps-sexp-print-proc
;;;  (lambda (expr)
;;;    (string-downcase!					;downcase the string.  
;;;     (with-output-to-string
;;;      tmp
;;;      (set (line-length tmp) 60)			;force newlines 
;;;      (pretty-print
;;;       (expression->sexp expr)
;;;       tmp)))))

(define imps-sexp-print-proc
  (lambda (sexp port)
    (pretty-print sexp port)))

;;; I suppose there should be an IMPS-TEX-PRINTER switch also--JDR.

;; (IMPS-PRINTER) is a procedure which takes an s-expression and a port
;; and prints someting to this port. 

(define IMPS-PRINTER
  (make-simple-switch
   'imps-printer
   procedure?
   pretty-print))

; IMPS-EXPRESSION->STRING-PROCEDURE takes an expression and returns a string. 

(define (imps-expression->string-procedure expr)
  (with-output-to-string
    tmp
    ((imps-printer)
     (expression->sexp expr)
     tmp)))

(define (imps-print-expression expression port)
  (writes
   port
   (imps-expression->string-procedure expression)))


(let ((reader-stack '())
      (printer-stack '()))
  (define (push-current-syntax)
    (push printer-stack (imps-printer))
    (push reader-stack (imps-reader))
    (return))

  (define (pop-current-syntax)
    (set (imps-printer) (pop printer-stack))
    (set (imps-reader) (pop reader-stack))
    (return)))

(define (push-current-theory-and-syntax)
  (push-current-theory)
  (push-current-syntax))

(define (pop-current-theory-and-syntax)
  (pop-current-theory)
  (pop-current-syntax))

(define (imps-load-theory filespec)
  (push-current-theory-and-syntax)
  (unwind-protect
   ((*value t-implementation-env 'load-file)
    (filename->string
     ((*value t-implementation-env 'expand-filename)
      ((*value t-implementation-env 'get-default-filename)
       filespec)))
    imps-implementation-env
    '#t)
   (pop-current-theory-and-syntax)))

(define (imps-require-theory filespec)
  (push-current-theory-and-syntax)
  (unwind-protect
   (*require '() filespec imps-implementation-env)
   (pop-current-theory-and-syntax)))
     

(define (QR input . language)
  (let ((language (if language (car language) (current-language))))
    (imps-read-procedure language input)))

;;; (if (use-string-form?)
;;; 	(input-string->expression *parse* language str)
;;; 	(with-input-from-string (tmp str)
;;; 				(read-expression language tmp)))

(define (QP expr)
  (imps-expression->string-procedure expr))

;; Printing assumptions for sequents
;;
;; 
(define (ASSUMPTIONS->STRING assumptions)
  (format nil
	  (separated-string-append
	   "~%~%"
	   (map
	    imps-expression->string-procedure
	    assumptions))))
	
(define (ASSUMPTIONS->SEXP assumptions)
  (if
   assumptions
   (map
    (lambda (a)
      (expression->sexp a))
    assumptions)
   'Empty-context))

(define context-reader-read-table
  (let ((rt (make-read-table standard-read-table 'context-reader-read-table)))
    (set (read-table-entry rt #\[)
	 (read-table-entry rt #\SPACE))
    (set (read-table-entry rt #\])
	 (read-table-entry rt #\SPACE))
    (set (read-table-entry rt #\,)
	 (read-table-entry rt #\SPACE))
    rt))

(define imps-sexp-context-reader
  (lambda (language str)
    (let ((port (string->input-port str))
	  (reader (imps-reader)))
      ;;
      ;; This was to throw away [,] when we had 'em.  
      ;;
      ;;(set (port-read-table port) context-reader-read-table)
      (let ((things (iterate iter ((l '()))
		      (let ((new (reader language port)))
			(if (eof? new)
			    (reverse! l)
			    (iter (cons new l)))))))
	(if (eq? (car things) 'empty-context)
	    '()
	    (map
	     (lambda (sexp)
	       (sexp->expression language sexp))
	     things))))))

(define START-READING-EXPRESSION-LIST?
  (make-simple-switch 'reading-expression-list boolean? '#f))

(define imps-string-form-context-reader
  (lambda (language str)
    ;;The tokenizer may get confused with the first left bracket "[" when
    ;;reading a string of assumptions. To tell it to treat it specially, set a
    ;;flag.
    (if (or (string-empty? str)
	    (string-equal?
	     "empty-context"
	     (string-downcase!
	      (substring str 0 (string-length "empty-context")))))
	'()
	(bind (((start-reading-expression-list?) '#t))
	  (map
	   (lambda (sexp)
	     (sexp->expression language sexp))
	   (top-level-parse *parse* str))))))


; The IMPS-CONTEXT-READER has as its value a procedure which takes two
; arguments, namely a language and a string.   It returns a list of
; assumptions.  

(define IMPS-CONTEXT-READER
  (make-simple-switch 'imps-context-reader
		      procedure?
		      imps-sexp-context-reader))

(define (STRING->ASSUMPTIONS language string)
  ((imps-context-reader) language string))

(define (SEQUENT->SEXP sequent)
  (let ((assumptions (sequent-assumptions sequent)))
    (if (null? assumptions)
	(list 'empty-context '=> (expression->sexp (sequent-assertion sequent)))
	(append
	 (assumptions->sexp assumptions)
	 (list '=> (expression->sexp (sequent-assertion sequent)))))))

(define (SEXP->SEQUENT language-or-theory sexp)
  (receive (assumptions assertion)
    (iterate iter ((sexp sexp)
		   (assumptions-so-far nil))
      (cond ((null? sexp)
	     (imps-error "SEXP->SEQUENT: => not found"))
	    ((eq? (car sexp) 'empty-context)
	     (if (and (null? assumptions-so-far)
		      (eq? (cadr sexp) '=>))
		 (return
		  nil
		  (sexp->expression language-or-theory (caddr sexp)))
		 (imps-error "SEXP->SEQUENT misplaced \"Empty-Context\"")))
	    ((eq? (car sexp) '=>)
	     (return
	      assumptions-so-far
	      (sexp->expression language-or-theory (cadr sexp))))
	    (else
	     (iter (cdr sexp)
		   (cons (sexp->expression language-or-theory (car sexp))
			 assumptions-so-far)))))
    (build-sequent
     (if (theory? language-or-theory)
	 (build-context language-or-theory assumptions)
	 (build-context (current-theory) assumptions))
     assertion)))

(define (list-constants lang-or-theory)
  (let ((lang (if (theory? lang-or-theory)
		  (theory-language lang-or-theory)
		  lang-or-theory)))
    (walk
     (lambda (c)
       (format t "~A: ~A~%"
	       (let ((sym (name c)))
		 (if (symbol? sym)
		     (string-downcase (symbol->string sym))
		     sym))
	       (expression-sorting c)))
     (language-constants lang))))
    
(define (short-list-constants lang-or-theory)
  (let ((lang (if (theory? lang-or-theory)
		  (theory-language lang-or-theory)
		  lang-or-theory)))
    (walk
     (lambda (c)
       (format t "~A: ~A~%"
	       (let ((sym (name c)))
		 (if (symbol? sym)
		     (string-downcase (symbol->string sym))
		     sym))
	       (expression-sorting c)))
     (set-difference
      (language-constants lang)
      (collect-set
       language-constants
       (map theory-language (fixed-theories-set)))))))
