;% Copyright (c) 1990-1994 The MITRE Corporation
;% 
;% Authors: W. M. Farmer, J. D. Guttman, F. J. Thayer
;%   
;% The MITRE Corporation (MITRE) provides this software to you without
;% charge to use, copy, modify or enhance for any legitimate purpose
;% provided you reproduce MITRE's copyright notice in any copy or
;% derivative work of this software.
;% 
;% This software is the copyright work of MITRE.  No ownership or other
;% proprietary interest in this software is granted you other than what
;% is granted in this license.
;% 
;% Any modification or enhancement of this software must identify the
;% part of this software that was modified, by whom and when, and must
;% inherit this license including its warranty disclaimers.
;% 
;% MITRE IS PROVIDING THE PRODUCT "AS IS" AND MAKES NO WARRANTY, EXPRESS
;% OR IMPLIED, AS TO THE ACCURACY, CAPABILITY, EFFICIENCY OR FUNCTIONING
;% OF THIS SOFTWARE AND DOCUMENTATION.  IN NO EVENT WILL MITRE BE LIABLE
;% FOR ANY GENERAL, CONSEQUENTIAL, INDIRECT, INCIDENTAL, EXEMPLARY OR
;% SPECIAL DAMAGES, EVEN IF MITRE HAS BEEN ADVISED OF THE POSSIBILITY OF
;% SUCH DAMAGES.
;% 
;% You, at your expense, hereby indemnify and hold harmless MITRE, its
;% Board of Trustees, officers, agents and employees, from any and all
;% liability or damages to third parties, including attorneys' fees,
;% court costs, and other related costs and expenses, arising out of your
;% use of this software irrespective of the cause of said liability.
;% 
;% The export from the United States or the subsequent reexport of this
;% software is subject to compliance with United States export control
;% and munitions control restrictions.  You agree that in the event you
;% seek to export this software or any derivative work thereof, you
;% assume full responsibility for obtaining all necessary export licenses
;% and approvals and for assuring compliance with applicable reexport
;% restrictions.
;% 
;% 
;% COPYRIGHT NOTICE INSERTED: Mon Apr 11 11:42:27 EDT 1994


(herald SIMPLE-JAFFAR-UNIFICATION)


;;; CLASSES

(define-structure-type CLASS
  variables
  expressions
  exp-count)

(define (MAKE-AND-INITIALIZE-CLASS var)
  (let ((class (make-class)))
    (set (class-variables class) (list var))
    (set (class-expressions class) '())
    (set (class-exp-count class) 0)
    class))

(define (ADD-VARIABLES vars class)
  (or (null? vars)
      (set (class-variables class) 
	   (set-union vars (class-variables class)))))

(define (ADD-EXPRESSIONS exps class)
  (or (null? exps)
      (and (set (class-expressions class)
		(set-union exps (class-expressions class)))
	   (set (class-exp-count class)
		(length (class-expressions class))))))

(define (MERGE-TWO-CLASSES class1 class2)
  (set (class-variables class1)
       (set-union (class-variables class1)
		  (class-variables class2)))
  (set (class-expressions class1)
       (set-union (class-expressions class1)
		  (class-expressions class2)))
  (set (class-exp-count class1)
       (length (class-expressions class1))))
		  


;;; CLASS SYSTEMS

(define-structure-type CLASS-SYSTEM
  var-class-pairs)

(define (MAKE-AND-INITIALIZE-CLASS-SYSTEM)
  (let ((system (make-class-system)))
    (set (class-system-var-class-pairs system) '())
    system))

(define (GET-CLASS var system)
  (let ((entry (assq var (class-system-var-class-pairs system))))
    (if entry
	(cdr entry)
	(let* ((class (make-and-initialize-class var)))
	  (set (class-system-var-class-pairs system)
	       (cons (cons var class)
		     (class-system-var-class-pairs system)))
	  class))))
	  
(define (ADD-VARIABLES-TO-CLASS vars class system)
  (walk
   (lambda (var)
     (let* ((class-of-var (get-class var system))
	    (var-class-pair-of-var
	      (assq var (class-system-var-class-pairs system))))
       (merge-two-classes class class-of-var)
       (set (cdr var-class-pair-of-var) class)))
   vars))

(define (CYCLES? val)
  (eq? val 'cycles)) 


;;; JAFFAR'S UNIFICATION ALGORITHM 
;;;
;;; The following algorithm, based on Jaffar's unification algorithm 
;;; for rational terms, unifies a list of expressions.  If the list 
;;; of expressions is unifiable as standard expressions, a 
;;; MOST GENERAL UNIFIER is returned; if the list is only unifiable 
;;; as rational expressions, 'CYCLES is returned; otherwise 'FAIL 
;;; is returned.


(define (UNIFY-UNDER-EXOSCOPES exprs exoscopes)
  (let ((subst (unify exprs)))
    (if (and (succeed? subst)
	     (null-intersection? (subst-domain subst) exoscopes))
	subst
	(fail))))

(define (UNIFY exprs)
  (let ((subst (unify-expressions exprs)))
    (cond ((fail? subst) (fail))
	  ((cycles? subst) (fail))
	  (else subst))))				;succeed only if no cycles 


(define (UNIFY-EXPRESSIONS exps)
  ; exp is a list of expressions.
  (if (null? exps)
      the-empty-substitution
      (let ((c&f-pair (common-and-frontiers exps)))
	(cond ((fail? c&f-pair)
	       'fail)
	      ((null? (cdr c&f-pair))
	       the-empty-substitution)
	      (else
	       (let* ((class-system (make-and-initialize-class-system))
		      (result
		        (map-but-avoid
			  (lambda (x) (apply-frontier x class-system))
			  (cdr c&f-pair)
			  'fail)))
		 (if (and (not (fail? result))
			  (consistent-system? class-system))
		     (unfold-system class-system)
		     'fail)))))))
		     
(define	(COMMON-AND-FRONTIERS exps)
  ; exps is a non-null list of expressions.
  (receive (vars non-var-exps) (separate-out-variables exps)
    (cond ((and (= (length vars) 1)
		(= (length non-var-exps) 0))
	   (cons (car vars) '()))
	  ((and (= (length vars) 0)
		(= (length non-var-exps) 1))
	   (cons (car non-var-exps) '()))
	  ((not (null? vars))
	   (cons (car vars) (list (list vars non-var-exps))))
	  ((and (not (any? constant? non-var-exps))
		(every?
		 (lambda (x)
		   (and (eq? (expression-constructor (car non-var-exps))
			     (expression-constructor x))
			(= (length (expression-components (car non-var-exps)))
			   (length (expression-components x)))))
		 (cdr non-var-exps))
		(or (not (binding-expression? (car non-var-exps)))
		    (every?
		     (lambda (x)
		       (equal-sets? (newly-bound-variables (car non-var-exps))
				    (newly-bound-variables x)))
		     (cdr non-var-exps))))
	   (common-and-frontiers-of-compound-expressions non-var-exps))
	  (else
	   'fail))))

(define (COMMON-AND-FRONTIERS-OF-COMPOUND-EXPRESSIONS exps)
  ; exps is a non-null list of compound-expressions which have the same
  ; constructor, the same number of components, and the same newly bound
  ; variables if they are binding expressions.  
  (let* ((first-exp (car exps))
	 (components-list
	  (if (binding-expression? first-exp)
	      (map
	       (lambda (x) (list (car (expression-components x))))
	       exps)
	      (map expression-components exps)))
	 (c&f-pairs
	  (map-but-avoid
	   (lambda (x) (common-and-frontiers x))
	   (transpose-matrix components-list)
	   'fail)))
    (if (fail? c&f-pairs)
	'fail
	(let* ((new-components (map car c&f-pairs))
	       (common-components
		(if (binding-expression? first-exp)
		    (cons (car new-components)
			  (cdr (expression-components first-exp)))
		    new-components))
	       (common (compound-expression
			 (expression-constructor first-exp)
			 common-components))
	       (frontiers (apply append (map cdr c&f-pairs))))
	  (cons common frontiers)))))
		       
(define (SEPARATE-OUT-VARIABLES exps)
  (iterate sep ((vars '())
		(non-var-exps '()) 
		(rest (make-set exps)))
    (if (null? rest)
	(return vars non-var-exps)
	(let ((first (car rest)))
	  (if (variable? first)
	      (sep (cons first vars)
		   non-var-exps
		   (cdr rest))
	      (sep vars
		   (cons first non-var-exps)
		   (cdr rest)))))))

(define (APPLY-FRONTIER frontier system)
  (let* ((vars (car frontier))
	 (exps (cadr frontier))
	 (class (get-class (car vars) system)))
    (add-variables-to-class (cdr vars) class system)
    (add-expressions exps class)))
	
(define (CONSISTENT-SYSTEM? system)
  (let ((class (get-unreduced-class system)))
    (or (not class)
	(let ((c&f-pair (common-and-frontiers (class-expressions class))))
	  (and (not (eq? c&f-pair 'fail))
	       (let ((common (car c&f-pair))
		     (frontiers (cdr c&f-pair)))
		 (set (class-expressions class) (list common))
		 (set (class-exp-count class) 1)
		 (and (not (eq? (map-but-avoid
				 (lambda (x) (apply-frontier x system))
				 frontiers
				 'fail)
				'fail))
		      (consistent-system? system))))))))
	
(define (GET-UNREDUCED-CLASS system)
  (iterate search-for-class 
      ((classes (map cdr (class-system-var-class-pairs system))))
    (if (null? classes)
	'#F
	(if (> (class-exp-count (car classes)) 1)
	    (car classes)
	    (search-for-class (cdr classes))))))

(define (UNFOLD-SYSTEM system)
  (let ((vars (map car (class-system-var-class-pairs system))))
    (extend-substitution vars the-empty-substitution '() system)))

(define (EXTEND-SUBSTITUTION vars sub visited-vars system)
  (iterate ext-sub ((vars vars) (sub sub))
    (if (null? vars)
	sub
	(let ((new-sub (extend-subst-for-one-var
			 (car vars)
			 sub
			 visited-vars
			 system)))
	  (if (eq? new-sub 'cycles)
	      'cycles
	      (ext-sub (cdr vars) new-sub))))))

(define (EXTEND-SUBST-FOR-ONE-VAR var sub visited-vars system)
  (let ((entry (assq var (class-system-var-class-pairs system))))
    (if (not entry)
	sub
	(let ((class-vars (class-variables (cdr entry)))
	      (class-exp (if (null? (class-expressions (cdr entry)))
			     '#F
			     (car (class-expressions (cdr entry))))))
	  (cond ((not (null? (set-intersection visited-vars class-vars)))
		 'cycles)
		((not class-exp)
		 (add-subst-components-1
		   (car class-vars)
		   (cdr class-vars)
		   sub))
		((not (null? (set-intersection
			       (set-union visited-vars class-vars)
			       (free-variables class-exp))))
		 'cycles)
		(else
		 (let ((new-sub (extend-substitution
				  (free-variables class-exp)
				  sub
				  (set-union visited-vars class-vars)
				  system)))
		   (if (eq? new-sub 'cycles)
		       'cycles
		       (add-subst-components-2
			 class-vars
			 (apply-substitution new-sub class-exp)
			 new-sub)))))))))
			
	      	       
	  
(define (ADD-SUBST-COMPONENTS-1 target replacements sub)
  (iterate add-sub-comps-1
      ((replacements replacements)
       (sub sub))
    (if (null? replacements)
	sub
	(add-sub-comps-1
	  (cdr replacements)
	  (add-subst-component
	    (make-subst-component target (car replacements))
	    sub)))))

(define (ADD-SUBST-COMPONENTS-2 targets replacement sub)
  (iterate add-sub-comps-2
      ((targets targets)
       (sub sub))
    (if (null? targets)
	sub
	(add-sub-comps-2
	  (cdr targets)
	  (add-subst-component
	    (make-subst-component (car targets) replacement)
	    sub)))))

			       
