;% Copyright (c) 1990-1994 The MITRE Corporation
;% 
;% Authors: W. M. Farmer, J. D. Guttman, F. J. Thayer
;%   
;% The MITRE Corporation (MITRE) provides this software to you without
;% charge to use, copy, modify or enhance for any legitimate purpose
;% provided you reproduce MITRE's copyright notice in any copy or
;% derivative work of this software.
;% 
;% This software is the copyright work of MITRE.  No ownership or other
;% proprietary interest in this software is granted you other than what
;% is granted in this license.
;% 
;% Any modification or enhancement of this software must identify the
;% part of this software that was modified, by whom and when, and must
;% inherit this license including its warranty disclaimers.
;% 
;% MITRE IS PROVIDING THE PRODUCT "AS IS" AND MAKES NO WARRANTY, EXPRESS
;% OR IMPLIED, AS TO THE ACCURACY, CAPABILITY, EFFICIENCY OR FUNCTIONING
;% OF THIS SOFTWARE AND DOCUMENTATION.  IN NO EVENT WILL MITRE BE LIABLE
;% FOR ANY GENERAL, CONSEQUENTIAL, INDIRECT, INCIDENTAL, EXEMPLARY OR
;% SPECIAL DAMAGES, EVEN IF MITRE HAS BEEN ADVISED OF THE POSSIBILITY OF
;% SUCH DAMAGES.
;% 
;% You, at your expense, hereby indemnify and hold harmless MITRE, its
;% Board of Trustees, officers, agents and employees, from any and all
;% liability or damages to third parties, including attorneys' fees,
;% court costs, and other related costs and expenses, arising out of your
;% use of this software irrespective of the cause of said liability.
;% 
;% The export from the United States or the subsequent reexport of this
;% software is subject to compliance with United States export control
;% and munitions control restrictions.  You agree that in the event you
;% seek to export this software or any derivative work thereof, you
;% assume full responsibility for obtaining all necessary export licenses
;% and approvals and for assuring compliance with applicable reexport
;% restrictions.
;% 
;% 
;% COPYRIGHT NOTICE INSERTED: Mon Apr 11 11:42:27 EDT 1994


(herald DEFINITIONS)


;;; This file contains the apparatus for making definitions of the form
;;; c = e, where c is new constant and e is an expression.

(define-structure-type DEFINITION
  home-theory
  constant
  defining-expr
  axiom					; theorem (equation)
  theorems				; set of theorems
  usage-list				; a list of symbols

  implicit?				;t or f
  implicit-unfolding-macete		;elementary macete
  

  (((name self)
    (name (definition-constant self)))
   ((print self port)
    (format port "#{IMPS-definition ~A: ~S ~S}"
	    (object-hash self)
	    (name (definition-constant self))
	    (definition-defining-expr self)))))

(block (set (definition-implicit? (stype-master definition-stype)) '#f)
       (set (definition-implicit-unfolding-macete (stype-master definition-stype)) '#f))

;;; The switch below is used for making primitive symbols into defined symbols.

(define PRESERVE-THEORY-LANGUAGE?
  (make-simple-switch 'preserve-theory-language? boolean? '#f)) 

(define (DEFINEDNESS-FORMULA definition)
  (or (definition? definition)
      (imps-error "DEFINEDNESS-FORMULA: ~S ~A."
		  definition "is not a definition"))
  (is-defined (definition-defining-expr definition)))

(define (SORT-DEFINEDNESS-FORMULA definition)
  (or (definition? definition)
      (imps-error "SORT-DEFINEDNESS-FORMULA: ~S ~A."
		  definition "is not a definition"))
  (defined-in (definition-defining-expr definition)
              (expression-sorting (definition-constant definition))))

(define (ADD-SORT-DEFINEDNESS-THEOREM def)
  (let ((theory (definition-home-theory def)))
    (theory-add-theorem 
     theory
     (sort-definedness-formula def)
     (concatenate-symbol 'sort-definedness-formula-for- (name def) '_ (name theory)))))

(define (BUILD-DEFINITION theory the-name defining-expr defining-sort . usage-list)
  
  ;; check arguments

  (or (and (theory? theory)
	   (symbol? the-name)
	   (closed? defining-expr)
	   (theory-sort? theory defining-sort)
	   (contains-expression? (theory-language theory) defining-expr))
      (imps-error "BUILD-DEFINITION: bad arguments ~S ~S ~S ~S."
		  theory the-name defining-expr defining-sort))
  (and (eq? theory the-kernel-theory)
       (imps-error "BUILD-DEFINITION: The first argument may not be the
                    kernel theory."))

  ;; build definition

  (let* ((new-constant (or (find-constant (theory-language theory) the-name)
			   (make-formal-constant-in-new-language
			    (theory-language theory)
			    defining-sort
			    the-name)))
	 (axiom 
	  (make-axiom-for-definition theory the-name new-constant defining-expr usage-list))
	 (theorems (if (and (memq 'rewrite usage-list)
			    (higher-sort? (expression-sorting new-constant)))
		       (make-singleton 
			(make-theorem-for-definition 
			 theory the-name new-constant defining-expr usage-list))
		       the-empty-set))

	 ;; If USAGE-LIST contains 'REWRITE, the "applied equation" theorem 
	 ;; is installed as a rewrite rule and the "equation" is not installed 
	 ;; as a rewrite.  This is done because simplification tries to replace
	 ;; an expression with a constant equal to it.

	 (definition (make-definition)))
    (set (definition-home-theory definition) theory)
    (set (definition-constant definition) new-constant)
    (set (definition-defining-expr definition) defining-expr)
    (set (definition-axiom definition) axiom)
    (set (definition-theorems definition) theorems)
    (set (definition-usage-list definition) usage-list)
    definition))

(define (MAKE-AXIOM-FOR-DEFINITION theory the-name constant defining-expr usage-list)
  (build-theorem theory 
		 (equality constant defining-expr)
		 (concatenate-symbol the-name '-equation_ (name theory))
		 (delq 'rewrite usage-list)))

(define (MAKE-THEOREM-FOR-DEFINITION theory the-name constant defining-expr usage-list)
  (let* ((sort (expression-sorting constant))
	 (var-list (sorts->new-variables
		    (higher-sort-domains sort)
		    'a
		    (variables defining-expr)))
	 (beta-reduced-application (lambda (expr arg-list)
				     (beta-reduce 
				      (apply apply-operator expr arg-list)))))
    (build-theorem theory 
		   (universal-closure
		    (quasi-equality-or-equality
		     (beta-reduced-application constant var-list)
		     (beta-reduced-application defining-expr var-list)))
		   (concatenate-symbol the-name '-applied-equation_ (name theory))
		   usage-list)))

(define (DEFINITION->SEXP definition)
    (list
     (definition-home-theory definition)
     (name (definition-constant definition))
     (expression->sexp (definition-defining-expr definition))))

;;;(define (DEFINITION->SEXP definition)
;;;    (list
;;;     (definition-home-theory definition)
;;;     (name (definition-constant definition))
;;;     (qp (definition-defining-expr definition))))

(define (SEXP->DEFINITION sexp)
  (destructure (((theory the-name sexpr) sexp))
    (let ((expr 
	   (*destructure-theory-read-proc* (theory-language theory) sexpr)))
      (cond ((theory-get-definition theory the-name)
	     =>
	     (lambda (def)
	       (let ((old-defining-expr (definition-defining-expr def)))
		 (if (eq? expr old-defining-expr)
		     def
		     (imps-error "sexp->definition:  name ~S already in use with meaning~_~S."
			    the-name old-defining-expr)))))
	    (else
	     (build-definition theory the-name expr '#f))))))

(define (THEORY-ADD-DEFINITION theory definition)
  (let ((home-theory (definition-home-theory definition))
	(the-name (name (definition-constant definition)))
	(defining-expr (definition-defining-expr definition)))

    ;; Checks

    (and (eq? home-theory the-kernel-theory)
	 (imps-error "THEORY-ADD-DEFINITION: Definitions may not be added to 
                     the kernel theory."))
    (or (eq? home-theory theory)
	(structural-sub-theory? home-theory theory)
	(imps-error "THEORY-ADD-DEFINITION: ~A ~S ~A ~S"
		    "The home theory of" definition
		    "is neither a structural sub-theory of nor equal to" theory))
    (or (preserve-theory-language?)
	(and (find-constant (theory-language theory) the-name)
	     (imps-error "THEORY-ADD-DEFINITION: ~S ~A ~S ~A"
			 the-name "has been used in the home theory"
			 theory "of the definition")))
    (let ((super-theory (find-super-theory-having-constant theory the-name)))
      (and super-theory
	   (imps-error "THEORY-ADD-DEFINITION: ~S ~A ~S ~A ~S ~A"
		       the-name "has been used in the super-theory" super-theory 
		       "of the home theory" home-theory "of the definition")))

    ;; Sort definedness check

    (let ((definedness-formula (definedness-formula definition))
	  (sort-definedness-formula (sort-definedness-formula definition))
	  (constant-sort (expression-sorting (definition-constant definition))))
      (if (eq? (expression-sorting defining-expr) constant-sort)
	  (or (necessarily-defined? defining-expr)
	      (theory-entails-immediately? home-theory definedness-formula)
	      (theory-theorem? home-theory definedness-formula)
	      (theory-theorem? home-theory sort-definedness-formula)
	      (imps-error "THEORY-ADD-DEFINITION: ~S ~A ~S; ~A ~S."
			  defining-expr "is not known to be defined in" 
			  home-theory "prove" definedness-formula))
	  (or (necessarily-defined-in-sort? defining-expr constant-sort)
	      (theory-entails-immediately? home-theory sort-definedness-formula)
	      (theory-theorem? home-theory sort-definedness-formula)
	      (imps-error "THEORY-ADD-DEFINITION: ~S ~A ~S ~A ~S; ~A ~S."
			  defining-expr "is not known to be defined in sort" 
			  constant-sort "in" home-theory "prove" 
			  sort-definedness-formula))))


    ;; Add DEFINITION to HOME-THEORY and all super-theories of HOME-THEORY
    
    (let ((theories (add-set-element
		     home-theory
		     (find-structural-super-theories home-theory))))
      (set-walk
       (lambda (theory-x)
	 (theory-add-definition-aux theory-x definition))
       theories))

    ;; Add definition theorems to HOME-THEORY and all super-theories of its HOME-THEORY

    (let ((theorems (definition-theorems definition)))
      (or (empty-set? theorems)
	  (set-walk
	   (lambda (theorem)
	     (theory-add-theorem-without-event home-theory theorem nil))
	   theorems)))
    definition))

(define (THEORY-ADD-DEFINITION-AUX theory definition)
  (let ((new-constant  (definition-constant definition))
	(axiom (definition-axiom definition)))
    (theory-push-definition-event theory definition)
    (or (preserve-theory-language?)
	(extend-theory-language 
	 theory 
	 (make-singleton new-constant) 
	 null-sort-resolver))

    ;; The order in which definitions are made must be reflected in the order of 
    ;; (THEORY-DEFINITIONS THEORY).  Hence DEFINITION is installed as the last 
    ;; member of (THEORY-DEFINITIONS THEORY). 
    
    (set (theory-definitions theory)	        ; install definition
	 (append-item-to-end-of-list definition (theory-definitions theory)))
    (theory-add-theorem-aux theory axiom '#f)))	; install axiom as a theorem


;;; It may be necessary to call THEORY-BUILD-DEFINITION more than once
;;; in order to create a definition.  If THE-NAME has been used in
;;; any of the structural super-theories of THEORY, then the call will
;;; fail and the user must recall THEORY-BUILD-DEFINITION with a new name.
;;; If THE-NAME is accepted but DEFINING-EXPR is not known to be  
;;; defined in THEORY, then the call will fail.  In this case the user
;;; should prove that DEFINING-EXPR is defined, add the corresponding
;;; theorem to THEORY, and then recall THEORY-BUILD-DEFINITION.

(define (THEORY-BUILD-DEFINITION 
	 theory the-name defining-expr defining-sort usage-list)
  (or (get-existing-definition 
       theory the-name defining-expr defining-sort usage-list)
      (theory-add-definition 
       theory
       (apply build-definition 
	      theory the-name defining-expr defining-sort usage-list))))

(define (THEORY-BUILD-DEFINITION-WITHOUT-CHECKING-DEFINEDNESS
	 theory the-name defining-expr defining-sort usage-list)
  (or (get-existing-definition 
       theory the-name defining-expr defining-sort usage-list)
      (let ((def (apply build-definition 
			theory 
			the-name 
			defining-expr
			defining-sort
			usage-list)))
	(add-sort-definedness-theorem def)
	(theory-add-definition theory def))))

(define (GET-EXISTING-DEFINITION 
	 theory the-name defining-expr defining-sort usage-list)
  (let ((def (theory-get-definition theory the-name)))
    (and def 
	 (eq? (definition-defining-expr def) defining-expr)
	 (eq? (expression-sorting (definition-constant def)) defining-sort)
	 (equal? (definition-usage-list def) usage-list)
	 def)))

(define (THEORY-GET-CONSTANT-DEFINITION theory constant-name)
  (cond ((not (constant? (find-constant (theory-language theory)
					constant-name)))
	 '#f)
	((theory-get-definition theory constant-name))
	((theory-get-recursive-definition theory constant-name))
	(else '#f)))

(define (THEORY-DEFINED-CONSTANT? theory expr)
  (or (theory-directly-defined-constant? theory expr)
      (theory-recursively-defined-constant? theory expr)))

(define (THEORY-GET-DEFINITION theory the-name)
  (let ((defs (theory-definitions theory)))
    (iterate iter ((defs defs))
      (cond ((null? defs) '#f)
	    ((eq? the-name (name (definition-constant (car defs))))
	     (car defs))
	    (else
	     (iter (cdr defs)))))))

(define (THEORY-GET-CONSTANT-DEFINITION-FROM-DEFINING-EXPR theory defining-expr)
  (let ((defs (theory-definitions theory)))
    (iterate iter ((defs defs)) 
      (cond ((null? defs) '#f)
	    ((alpha-equivalent? defining-expr (definition-defining-expr (car defs)))
	     (car defs))
	    (else
	     (iter (cdr defs)))))))

(define (THEORY-DIRECTLY-DEFINED-CONSTANT? theory expr)
  (and (constant? expr)
       (true? (theory-get-definition theory (name expr)))))

;;;(define (THEORY-DEFINED-CONSTANTS theory)
;;;  (set-union (collect-set
;;;	      (lambda (def) (make-set (list (definition-constant def))))
;;;	      (theory-definitions theory))
;;;	     (collect-set
;;;	      (lambda (def) (make-set (recursive-definition-constant-list def)))
;;;	      (theory-recursive-definitions theory))))

(define (THEORY-DEFINED-CONSTANTS theory)
  (append (map (lambda (def) (definition-constant def)) (theory-definitions theory))
	  (apply append
		 (map (lambda (def) (recursive-definition-constant-list def))
		      (theory-recursive-definitions theory)))))

(define (THEORY-NONRECURSIVELY-DEFINED-CONSTANTS theory)
  (collect-set
   (lambda (def) (make-set (list (definition-constant def))))
   (theory-definitions theory)))

(define (THEORY-DEFINITIONS->SEXP theory)
  (map
   definition->sexp
   (theory-definitions theory))) 




;;; EXPANDING CONSTANTS AND CONTRACTING EXPRESSIONS

(define (EXPAND-CONSTANT constant defining-expr expr path)
    (imps-enforce (lambda (target)
		    (eq? target constant))
		  (follow-path expr path))
    (substitution-at-path expr defining-expr path))

(define (EXPAND-CONSTANT-OCCURRENCES constant defining-expr expr paths)
  (imps-enforce paths-disjoint? paths)
  (iterate iter ((new-expr expr)
		 (paths paths))
    (if (null? paths)
	new-expr
	(iter (expand-constant constant defining-expr new-expr (car paths))
	      (cdr paths)))))

(define (EXPAND-CONSTANT-OCCURRENCES-ONCE constant defining-expr expr)
  (let ((paths
	 (paths-to-occurrences expr constant -1)))
  (expand-constant-occurrences constant defining-expr expr paths)))

(define (EXPAND-ALL-DIRECTLY-DEFINED-CONSTANT-OCCURRENCES-IN-THEORY-ONCE theory expr)
  (let ((expr expr))
  (walk
   (lambda (x) 
     (let ((constant (definition-constant x))
	   (defining-expr (definition-defining-expr x)))
       (set expr (expand-constant-occurrences-once constant defining-expr expr))))
   (theory-definitions theory))
  expr))

(define (MATCH-DEFINING-EXPR defining-expr expr path)
  (let ((target (follow-path expr path)))
    (cond ((eq? defining-expr target) the-empty-substitution)
	  ((and (lambda-expression? defining-expr)
		(lambda-expression? target))
	   (match-under-exoscopes
	    target
	    (binding-body defining-expr)
	    (set-difference (variables defining-expr)
			    (newly-bound-variables defining-expr))))
	  ((lambda-expression? defining-expr)
	   (substitution-extend-with-identity
	    (match-under-exoscopes
	     target
	     (binding-body defining-expr)
	     (set-difference (variables defining-expr)
			     (newly-bound-variables defining-expr)))
	    (binding-variables defining-expr)))
	  (else (fail)))))

(define (CONTRACT-EXPRESSION constant defining-expr expr path)
  (let* ((subst (match-expression defining-expr expr path)))
    (cond ((fail? subst) (return expr nil))
	  ((null? subst)
	   (return (substitution-at-path expr constant path) nil))
	  (else
	   (return
	    (substitution-at-path
	     expr
	     (apply-operator-to-substitution constant subst
					     (binding-variables defining-expr))
	     path)
	    (substitution-definedness-conditions subst))))))

(define (CONTRACT-EXPRESSION-OCCURRENCES constant defining-expr expr paths)
  (imps-enforce paths-disjoint? paths)
  (iterate iter ((new-expr expr)
		 (paths paths)
		 (paths-reqs nil))
    (if (null? paths)
	(return new-expr paths-reqs)
	(receive (new-expr new-reqs)
	  (contract-expression constant defining-expr new-expr (car paths))
	  (iter new-expr
		(cdr paths)
		(cons (list (car paths)
			    new-reqs)
		      paths-reqs))))))

;;(define-predicate implicit-definition?)
;;(define-operation implicit-unfolding-macete)

(define (implicit-iota-&-unfolding-theorems?
	 source-theory
	 iota-theorem
	 unfolding-theorem)
  (and
   (theory-theorem? source-theory iota-theorem)
   (theory-theorem? source-theory unfolding-theorem)
   (let ((iota-thm-iota-term
	  (let ((body (binding-body iota-theorem)))
	    (and (convergence? body)
		 (iota-expression? (convergence-term body))
		 (convergence-term body))))
	 (unfolding-body (binding-body unfolding-theorem)))
     (and
      iota-thm-iota-term
      (implication? unfolding-body)
      (let ((ant (implication-antecedent unfolding-body))
	    (conseq (implication-consequent unfolding-body)))
	(and
	 (equation? ant)
	 (variable? (expression-lhs ant))
	 (alpha-equivalent? (expression-rhs ant)
			    iota-thm-iota-term)
	 (equation? conseq)
	 (eq? (expression-lhs ant)
	      (expression-lhs conseq))
	 (let ((iota-var (car (binding-variables (expression-rhs ant))))
	       (iota-body (binding-body (expression-rhs ant)))
	       (relacement-var (expression-lhs conseq)))
	   (alpha-equivalent?
	    (expression-rhs conseq)
	    (apply-substitution
	     (one-component-subst iota-var relacement-var)
	     (expression-rhs iota-body))))))))))

(define (make-implicit-definition-type source-theory
				       iota-theorem
				       unfolding-theorem)
  (if
   (not
    (implicit-iota-&-unfolding-theorems? source-theory iota-theorem unfolding-theorem))
   (imps-error "make-implicit-definition-type: failing theorems ~S~%~S"
	       iota-theorem
	       unfolding-theorem)
   (lambda (target-theory new-name terms)
     (let ((iota-theorem-instance
	    (auto-instantiate-theorem-with-terms target-theory iota-theorem terms)))
       (if iota-theorem-instance
	   (let* ((definition
		    (theory-add-definition
		     target-theory
		     (build-definition
		      target-theory
		      new-name
		      (convergence-term iota-theorem-instance)
		      (expression-sorting (convergence-term iota-theorem-instance)))))
		  (unfolding-theorem-instance
		   (auto-instantiate-theorem-with-terms
		    target-theory
		    unfolding-theorem
		    (append terms (list (definition-constant definition))))))
	     (if (and unfolding-theorem-instance
		      (alpha-equivalent?
		       (implication-antecedent unfolding-theorem-instance)
		       (equality (definition-constant definition)
				 (definition-defining-expr definition))))
		 (let ((unfolder
			(retrieve-macete-from-formula
			 (theory-add-theorem
			  target-theory
			  (apply-macete-without-minor-premises
			   *beta-reduce-repeatedly-macete*
			   (theory-null-context target-theory)
			   (implication-consequent unfolding-theorem-instance))
			  (concatenate-symbol
			   (name (definition-constant definition))
			   '-implicit-unfolding)))))
		   (set (definition-implicit? definition)  '#t)
		   (set (definition-implicit-unfolding-macete definition) unfolder)
		   definition)
		 (imps-error "make-implicit-definition-type: failing application to:~%~S ~S ~S~%in the relation to ~S ~S ~%~S."
			     target-theory new-name terms
			     source-theory iota-theorem unfolding-theorem)))
	   (imps-error "make-implicit-definition-type: failing application to:~%~S ~S ~S~%in the relation to ~S ~S ~%~S."
		       target-theory new-name terms
		       source-theory iota-theorem unfolding-theorem))))))

(define (implicit-definition? definition)
  (and (definition? definition)
       (definition-implicit? definition)))






;;;;;; EXPANDING AND CONTRACTING DEFINITIONS
;;;
;;;(define (EXPAND-DEFINITION definition expr path)
;;;  (let ((constant (definition-constant definition))
;;;	(defining-expr (definition-defining-expr definition)))
;;;    (imps-enforce (lambda (target)
;;;		    (eq? target constant))
;;;		  (follow-path expr path))
;;;    (substitution-at-path expr defining-expr path)))
;;;
;;;(define (EXPAND-DEFINITION-OCCURRENCES definition expr paths)
;;;  (imps-enforce paths-disjoint? paths)
;;;  (iterate iter ((new-expr expr)
;;;		 (paths paths))
;;;    (if (null? paths)
;;;	new-expr
;;;	(iter (expand-definition definition new-expr (car paths))
;;;	      (cdr paths)))))
;;;
;;;(define (EXPAND-DEFINITION-OCCURRENCES-ONCE definition expr)
;;;  (let ((paths
;;;	 (paths-to-occurrences
;;;	  expr
;;;	  (definition-constant definition)
;;;	  -1)))
;;;  (expand-definition-occurrences definition expr paths)))
;;;
;;;(define (EXPAND-ALL-DEFINITION-OCCURRENCES-IN-THEORY-ONCE theory expr)
;;;  (let ((expr expr))
;;;  (walk
;;;   (lambda (x) (set expr (expand-definition-occurrences-once x expr)))
;;;   (theory-definitions theory))
;;;  expr))
;;;
;;;(define (MATCH-DEFINITION definition expr path)
;;;  (let ((defining-expr (definition-defining-expr definition))
;;;	(target (follow-path expr path)))
;;;    (cond ((eq? defining-expr target) the-empty-substitution)
;;;	  ((and (lambda-expression? defining-expr)
;;;		(lambda-expression? target))
;;;	   (match-under-exoscopes
;;;	    target
;;;	    (binding-body defining-expr)
;;;	    (set-difference (variables defining-expr)
;;;			    (newly-bound-variables defining-expr))))
;;;	  ((lambda-expression? defining-expr)
;;;	   (substitution-extend-with-identity
;;;	    (match-under-exoscopes
;;;	     target
;;;	     (binding-body defining-expr)
;;;	     (set-difference (variables defining-expr)
;;;			     (newly-bound-variables defining-expr)))
;;;	    (binding-variables defining-expr)))
;;;	  (else (fail)))))
;;;
;;;(define (CONTRACT-DEFINITION definition expr path)
;;;  (let* ((defining-expr (definition-defining-expr definition))
;;;	 (subst (match-definition definition expr path))
;;;	 (constant (definition-constant definition)))	      
;;;    (cond ((fail? subst) (return expr nil))
;;;	  ((null? subst)
;;;	   (return (substitution-at-path expr constant path) nil))
;;;	  (else
;;;	   (return
;;;	    (substitution-at-path
;;;	     expr
;;;	     (apply-operator-to-substitution constant subst
;;;					     (binding-variables defining-expr))
;;;	     path)
;;;	    (substitution-definedness-conditions subst))))))
;;;
;;;(define (CONTRACT-DEFINITION-OCCURRENCES definition expr paths)
;;;  (imps-enforce paths-disjoint? paths)
;;;  (iterate iter ((new-expr expr)
;;;		 (paths paths)
;;;		 (paths-reqs nil))
;;;    (if (null? paths)
;;;	(return new-expr paths-reqs)
;;;	(receive (new-expr new-reqs)
;;;	  (contract-definition definition new-expr (car paths))
;;;	  (iter new-expr
;;;		(cdr paths)
;;;		(cons (list (car paths)
;;;			    new-reqs)
;;;		      paths-reqs))))))
;;;
;;;;;; (let ((expanded-context
;;;;;;	   ((lambda (target lc)
;;;;;;	  (cond ((formula? target) (context-add-assumption lc target))
;;;;;;		((term-or-fn? target) (context-add-assumption
;;;;;;				       lc (is-defined target)))
;;;;;;		(else lc)))
;;;;;;	(follow-path expr path)
;;;;;;	(local-context-at-path context expr path))))
;;;;;;   (context-entails-substitution-defined? expanded-context subst))
;;;
