;% Copyright (c) 1990-1994 The MITRE Corporation
;% 
;% Authors: W. M. Farmer, J. D. Guttman, F. J. Thayer
;%   
;% The MITRE Corporation (MITRE) provides this software to you without
;% charge to use, copy, modify or enhance for any legitimate purpose
;% provided you reproduce MITRE's copyright notice in any copy or
;% derivative work of this software.
;% 
;% This software is the copyright work of MITRE.  No ownership or other
;% proprietary interest in this software is granted you other than what
;% is granted in this license.
;% 
;% Any modification or enhancement of this software must identify the
;% part of this software that was modified, by whom and when, and must
;% inherit this license including its warranty disclaimers.
;% 
;% MITRE IS PROVIDING THE PRODUCT "AS IS" AND MAKES NO WARRANTY, EXPRESS
;% OR IMPLIED, AS TO THE ACCURACY, CAPABILITY, EFFICIENCY OR FUNCTIONING
;% OF THIS SOFTWARE AND DOCUMENTATION.  IN NO EVENT WILL MITRE BE LIABLE
;% FOR ANY GENERAL, CONSEQUENTIAL, INDIRECT, INCIDENTAL, EXEMPLARY OR
;% SPECIAL DAMAGES, EVEN IF MITRE HAS BEEN ADVISED OF THE POSSIBILITY OF
;% SUCH DAMAGES.
;% 
;% You, at your expense, hereby indemnify and hold harmless MITRE, its
;% Board of Trustees, officers, agents and employees, from any and all
;% liability or damages to third parties, including attorneys' fees,
;% court costs, and other related costs and expenses, arising out of your
;% use of this software irrespective of the cause of said liability.
;% 
;% The export from the United States or the subsequent reexport of this
;% software is subject to compliance with United States export control
;% and munitions control restrictions.  You agree that in the event you
;% seek to export this software or any derivative work thereof, you
;% assume full responsibility for obtaining all necessary export licenses
;% and approvals and for assuring compliance with applicable reexport
;% restrictions.
;% 
;% 
;% COPYRIGHT NOTICE INSERTED: Mon Apr 11 11:42:27 EDT 1994


(herald THEORY-ENSEMBLES)


;;;A THEORY-ENSEMBLE is a structure for organizing related theories.

(define-structure-type THEORY-ENSEMBLE
  base-theory				;the theory of which every theory in the ensemble is an instance.
  fixed-theories-set
  replica-renamer                       ;A renaming procedure for each integer n.
  theory-replica-alist                  ;an list of the theory replicas
  theory-multiple-alist		        ;an alist of entries (n . n-theory-multiple)
  canonical-translations-alist          ;an alist of entries ((m . n) . translations)
                                        ;where m,n are integers and 1<=m<n.
  initial-movable-sorts                 ;used for building constant alists for translations.
  initial-movable-constants             ;used for building sort alists for translations.

  nontranslating-defined-constants
  nontranslating-defined-sorts
  maximum-multiple                      ;The largest computed multiple.
  )

;;;Fix a theory ensemble:

;;;A resolved sort or constant of the base theory is MOVABLE iff it does not belong to any
;;;of the fixed theories of the ensemble.

;;;The INDEXING-NAMES for a theory ensemble consist of the names of undefined movable sorts.
;;;This list is used for determining appropriate names for translated constants.

;;;If k is an integer k>=0, the kth CANONICAL REPLICA R_k, is an identical copy of the
;;;base theory B with a renaming of the movable constants and sorts by subscripting
;;;k.

;;;There is a unique translation B --> R_k which renames each movable sort or constant 
;;;by subscripting with k. Constants or sorts which are not movable are translated by a
;;;default renaming procedure. This procedure renames a sort or constant by subscripting
;;;the name with the translated indexing-names.

;;;There is a translation R_j -->R_k obtained by composing the inverse
;;;of B-->R_j with B-->R_k. This is called the CANONICAL TRANSLATION C_{j,k} from R_j to R_k.

;;;If k is an integer k>=2, the kth MULTIPLE M_k is the theory union of the canonical
;;;replicas R_0, ..., R_{k-1}. The MULTIPLE M_1 is defined to be the base theory.

;;;Given a non-repeating list L={a_0,a_1,...,a_{m-1}} of non-negative integers <=n-1,
;;;the CANONICAL TRANSLATION associated to L is a translation M_m -->M_n which is the union
;;;of the canonical translations C_{i,j}  R_k --> R_{a_k}.

;;;A defined constant in M_n is NATIVELY DEFINED in M_n iff it is not the translation, under
;;;some canonical translation, of a constant defined in M_m for m<n.


(define find-theory-ensemble
  (let ((alist '()))
    (operation
	(lambda (th fixed-theories)
	  (let ((sublist (cdr (assq th alist))))
	    (if sublist
		(cdr (ass set-equal? fixed-theories sublist))
		'())))
      ((setter soi)
       (lambda (th fixed-theories new-value)
	 (let ((look-up (assq th alist)))
	   (if look-up
	       (set (cdr look-up)
		    (cons (cons fixed-theories new-value) (cdr look-up)))
	       (set alist (cons (cons th (list (cons fixed-theories new-value))) alist)))))))))

;;A table to locate ensemble given theory multiple.


(define (name->theory-ensemble symbol)
  (find-theory-ensemble (name->theory symbol) (fixed-theories-set)))

(define (BUILD-THEORY-ENSEMBLE base-theory . aux)
  (let* ((fixed (if aux (car aux) (fixed-theories-set)))
	 (replica-renamer (if (cdr aux) (cadr aux) subscripting-renamer))
	 (look-up (find-theory-ensemble base-theory (fixed-theories-set))))
    (or (theory? base-theory)
	(imps-error "BUILD-THEORY-ENSEMBLE: ~A is not a theory." base-theory))
    (or (every? theory? (fixed-theories-set))
	(imps-error "BUILD-THEORY-ENSEMBLE: ~A contains a non-theory." base-theory))
    (if look-up
	look-up
	(let ((obj (make-theory-ensemble)))

	  (set (theory-ensemble-base-theory obj) base-theory)
	  (set (theory-ensemble-fixed-theories-set obj) fixed)
	  (set (theory-ensemble-replica-renamer obj) replica-renamer)
	  (set (theory-ensemble-theory-replica-alist obj) '())
	  (set (theory-ensemble-theory-multiple-alist obj) (list (cons 1 base-theory)))
	  (set (theory-ensemble-initial-movable-sorts obj)
	       (theory-ensemble-movable-sorts obj))

	  (set (theory-ensemble-initial-movable-constants obj)
               (theory-ensemble-movable-constants obj))

	  (set (theory-ensemble-canonical-translations-alist obj) '())

	  (set (find-theory-ensemble base-theory fixed) obj)
	  (set (theory-ensemble-nontranslating-defined-constants obj)
	       '())
	  (set (theory-ensemble-nontranslating-defined-sorts obj)
	       '())
	  (set (theory-ensemble-maximum-multiple obj) 1)

	  obj))))

(define (THEORY-ENSEMBLE-MOVABLE-SORTS ensemble)
  (set-diff (theory-ind-sorts-resolved (theory-ensemble-base-theory ensemble))
	    (big-u (map theory-ind-sorts-resolved
			(theory-ensemble-fixed-theories-set ensemble)))))

(define (THEORY-ENSEMBLE-MOVABLE-CONSTANTS ensemble)
  (set-diff (theory-constants (theory-ensemble-base-theory ensemble))
	    (big-u (map theory-constants
			(theory-ensemble-fixed-theories-set ensemble)))))

(define (RETRIEVE-THEORY-MULTIPLE theory-ensemble n)
  (cdr (ass = n (theory-ensemble-theory-multiple-alist theory-ensemble))))

(define (THEORY-ENSEMBLE-NAME ensemble)
  (name (theory-ensemble-base-theory ensemble)))

;;;(define (THEORY-ENSEMBLE-INSTANCES ensemble n)
;;;  ((theory-ensemble-instances-settable-alist ensemble) n))

(define (interpretations-from-theory theory fixed-theories)
  (let ((fixed-theories (compress-theories fixed-theories))
	(successes the-empty-set))
    (walk
     (lambda (pair0)
       (if (eq? theory (car pair0))
	   (walk
	    (lambda (pair1)
	      (walk
	       (lambda (pair2)

		 (walk
		  (lambda (pair3)
		    
		    (if (set-equal? fixed-theories (car pair3))
			(walk
			 (lambda (pair4)

			   (if (translation-theory-interpretation? (cdr pair4))
			       (set successes 
				    (add-set-element (cdr pair4) successes))))
			 (cdr pair3))))
		  (cdr pair2)))
	       (cdr pair1)))
	    (cdr pair0))))
     (cdr *global-translation-alist*))
    successes))

(define (THEORY-ENSEMBLE-INSTANCES ensemble n)
  (let ((multiple (retrieve-theory-multiple ensemble n))
	(fixed-theories (theory-ensemble-fixed-theories-set ensemble)))
    (if (null? multiple) the-empty-set
	(interpretations-from-theory multiple fixed-theories))))

(define (THEORY-ENSEMBLE-EXPORT-TRANSPORTABLE-RWRS ensemble)
  (let ((n (theory-ensemble-maximum-multiple ensemble))
	(sources (theory-ensemble-computed-multiples ensemble)))
    (iterate loop ((index 1))
      (if (< n index) '#t
	  (block
	    (walk
	     (lambda (trans)
	       (if (translation-theory-interpretation? trans)
		   (theory-import-transportable-rewrite-rules
		    (translation-target-theory trans)
		    sources)))
	     (theory-ensemble-instances ensemble n))
	    (loop (1+ n)))))))

(define (theory-ensemble-computed-multiples ensemble)
  (let ((n (theory-ensemble-maximum-multiple ensemble)))
    (iterate loop ((index 1) (multiples '()))
      (if (< n index)
	  multiples
	  (loop (1+ index) (cons (retrieve-theory-multiple ensemble index) multiples))))))

(define (trivial-translation? translation)
  (and (sub-theory? (translation-source-theory  translation)
		    (translation-target-theory  translation))
       (every? (lambda (x) (eq? (car x) (cdr x)))
	       (append (translation-constant-alist translation)
		       (translation-sort-alist translation)))))


(define (THEORY-ENSEMBLE-INSTANCE? ensemble translation n)
  (or (memq? translation (theory-ensemble-instances ensemble n))
      (trivial-translation? translation)))

(define (DONT-TRANSLATE-CONSTANT ensemble constant)
  (or (formal-symbol? constant)
      (imps-error "~A must be a formal symbol" constant))
  (set (theory-ensemble-nontranslating-defined-constants ensemble)
       (add-set-element 
	constant 
	(theory-ensemble-nontranslating-defined-constants ensemble))))

(define (DONT-TRANSLATE-SORT ensemble sort)
  (or (sort? sort)
      (imps-error "~A must be a sort" sort))
  (set (theory-ensemble-nontranslating-defined-sorts ensemble)
       (add-set-element 
	sort 
	(theory-ensemble-nontranslating-defined-sorts ensemble))))

(define (DEFINED-CONSTANTS-IN-BASE-THEORY ensemble)
  (defined-constants-in-theory-multiple ensemble 1))

(define (TRANSLATABLE-DEFINED-CONSTANTS-IN-BASE-THEORY ensemble)
  (translatable-defined-constants-in-theory-multiple ensemble 1))

;;;(let ((base-theory (theory-ensemble-base-theory ensemble)))
;;;    (set-diff
;;;     (theory-defined-constants base-theory)
;;;     (big-u (map (lambda (x)
;;;		   (theory-defined-constants x))
;;;		 (theory-ensemble-fixed-theories-set ensemble)))))


(define (DEFINED-SORTS-IN-BASE-THEORY ensemble)
  (defined-sorts-in-theory-multiple ensemble 1))

(define (TRANSLATABLE-DEFINED-SORTS-IN-BASE-THEORY ensemble)
  (translatable-defined-sorts-in-theory-multiple ensemble 1))

(define (THEORY-ENSEMBLE-INDEXING-NAMES ensemble)
  (map name
       (set-diff (theory-ensemble-initial-movable-sorts ensemble)
		 (defined-sorts-in-base-theory ensemble))))


;;;;;;;;;;;;;;;;;;;;;;;;;;THEORY MULTIPLES;;;;;;;;;;;;;;;;;;;;;;;;


(define (THEORY-ENSEMBLE-FIND-THEORY-MULTIPLE theory-ensemble n)
  (or (theory-ensemble? theory-ensemble)
      (imps-error "THEORY-ENSEMBLE-FIND-THEORY-MULTIPLE: ~A is not a theory ensemble." theory-ensemble))
  (if (= n 1)
      (theory-ensemble-base-theory theory-ensemble)
      (let ((look-up (retrieve-theory-multiple theory-ensemble n)))
	(if look-up
	    look-up
	    (receive  (theory-multiple translations)
	      (compute-theory-multiple-and-canonical-translations theory-ensemble n)
	      (set (theory-ensemble-theory-multiple-alist theory-ensemble)
		   (cons (cons n theory-multiple)
			 (theory-ensemble-theory-multiple-alist theory-ensemble)))
	      (set (theory-ensemble-canonical-translations-alist theory-ensemble)
		   (cons (cons (cons 1 n) translations)
			 (theory-ensemble-canonical-translations-alist theory-ensemble)))
	      (set (theory-ensemble-maximum-multiple theory-ensemble)
		   (max n (theory-ensemble-maximum-multiple theory-ensemble)))
		  
	      theory-multiple)))))

(define (COMPUTE-THEORY-REPLICA-AND-CANONICAL-TRANSLATION theory-ensemble subscript)
  (let ((theory (theory-ensemble-base-theory theory-ensemble))
	(replica-alist (theory-ensemble-theory-replica-alist theory-ensemble)))
    (let ((look-up (ass = subscript replica-alist))) 
      (if look-up
	  (let ((translation (cdr look-up)))
	    (return (translation-target-theory translation) translation))
	  (let* ((translation-name-and-nickname
		  (retrieve-unused-name
		   name->translation
		   (name theory)
		   '-to-
		   (name theory)
		   '-
		   subscript))
		 (new-theory-name (retrieve-unused-name name->theory (name theory) '- subscript))
		 (translation
		  (transport-theory
		   the-kernel-translation
		   theory
		   the-kernel-theory
		   (theory-ensemble-fixed-theories-set theory-ensemble)
		   ((theory-ensemble-replica-renamer theory-ensemble) subscript)
		   translation-name-and-nickname
		   translation-name-and-nickname
		   new-theory-name)))
	    (set (theory-ensemble-theory-replica-alist theory-ensemble)
		 (cons (cons subscript translation)
		       (theory-ensemble-theory-replica-alist theory-ensemble)))
	  

	    (return (translation-target-theory translation) translation))))))


(define (COMPUTE-THEORY-MULTIPLE-AND-CANONICAL-TRANSLATIONS theory-ensemble n)
  (let* ((up-to-n (list-up-to n))
	 (translations 
	  (map
	   (lambda (subscript)
	     (receive (() translation)
	       (compute-theory-replica-and-canonical-translation theory-ensemble subscript)
	       translation))
	   up-to-n))

	 ;;This is done in this apparently screwy way so that the (n+1)-st multiple contain
	 ;;the n-th multiple for 2<=n. The 2-multiple does not contain the 1-multiple
	 ;;but rather an isomorphic copy of it.
	 
	 (theory-multiple
	  (theory-union
	   (if (= n 2)
	       (map translation-target-theory translations)
	       (list (theory-ensemble-find-theory-multiple theory-ensemble (- n 1))
		     (translation-target-theory (last translations))))		 
	   (retrieve-unused-name
	    name->theory
	    (theory-ensemble-name theory-ensemble)
	    '-
	    n
	    '-tuples))))

    (return theory-multiple translations)))

(define alist-rename-indexers 
  (let ((numerical-indexer-for-weird-indexers 0))
    (lambda(alist indexers)
      (map (lambda (x)
	     (let ((look-up (assq x alist)))
	       (if look-up
		   (if (symbol? (cadr look-up))
		       (cadr look-up)
		       (increment numerical-indexer-for-weird-indexers))
		   (imps-error "THEORY-ENSEMBLE-FIND-THEORY-MULTIPLE: alist missing renaming for sort ~A." x))))
	   indexers))))

;;I think a more useful renamer for theories built via translations.

(define (nearly-trivial-renamer alist)
  (lambda (name)
    (let ((look-up
	   (assq name alist)))
      (if (and look-up (symbol? (cadr look-up)))
	  (cadr look-up)
	  name))))

(define (ENSEMBLE-DEFAULT-RENAMER
	 sort-renaming-alist
	 constant-renaming-alist
	 default-indexer)
  (let ((alist (append sort-renaming-alist constant-renaming-alist)))
    (lambda (name)
      (let ((look-up
	     (assq name alist)))
	(if (and look-up (symbol? (cadr look-up)))
	    (cadr look-up)
	    (apply concatenate-symbol name '_ default-indexer))))))

(define (SUBSCRIPTING-RENAMER n)
  (lambda (name) (concatenate-symbol name '_ n)))

;;;(define (RENAME-CONSTANTS-IN-LANGUAGE renamer names language)
;;;  (map (lambda (x)
;;;	 (retrieve-unused-name
;;;	  (lambda (x) (find-constant language x))
;;;	  (renamer x)))
;;;       names))
;;;
;;;(define (RENAME-SORTS-IN-LANGUAGE renamer names language)
;;;  (map (lambda (x)
;;;	 (retrieve-unused-name
;;;	  (lambda (x) (name->sort language x))
;;;	  (renamer x)))
;;;       names))

(define (namer-from-suggested-namer suggester theory)
  (let ((theories (add-set-element theory (find-structural-super-theories theory))))
    (lambda (x)
      (retrieve-unused-name
       (lambda (x) (any? (lambda (th)
			   (let ((language (theory-language th)))
			     (or (name->sort language x)
				 (find-constant language x))))
			 theories))
       (suggester x)))))


(define (DEFINED-CONSTANTS-IN-THEORY-MULTIPLE theory-ensemble n)
  (let ((n-theory-multiple (theory-ensemble-find-theory-multiple theory-ensemble n)))
    (set-diff
     (theory-defined-constants n-theory-multiple)
     (big-u (map (lambda (x)
		   (theory-defined-constants x))
		 (theory-ensemble-fixed-theories-set theory-ensemble))))))

(define (TRANSLATABLE-DEFINED-CONSTANTS-IN-THEORY-MULTIPLE theory-ensemble n)
  (set-diff (defined-constants-in-theory-multiple theory-ensemble n)
	    (theory-ensemble-nontranslating-defined-constants theory-ensemble)))

(define (DEFINED-SORTS-IN-THEORY-MULTIPLE theory-ensemble n)
  (let ((n-theory-multiple (theory-ensemble-find-theory-multiple theory-ensemble n)))
    (set-diff
     (theory-defined-sorts n-theory-multiple)
     (big-u (map (lambda (x)
		   (theory-defined-sorts x))
		 (theory-ensemble-fixed-theories-set theory-ensemble))))))


(define (TRANSLATABLE-DEFINED-SORTS-IN-THEORY-MULTIPLE theory-ensemble n)
  (set-diff (defined-sorts-in-theory-multiple theory-ensemble n)
	    (theory-ensemble-nontranslating-defined-sorts theory-ensemble)))

;;;;;;;;;;;;;;;;;;;;;;;;;;CANONICAL TRANSLATIONS;;;;;;;;;;;;;;;;;;;;;;;;;

(define (CANONICAL-TRANSLATIONS-BETWEEN-THEORY-MULTIPLES theory-ensemble m n)
  
  ;;Returns a list of translations (tr1 .... trk) corresponding to the
  ;;kth permutation of n take m. 

  (if (or (< n m) (and (= m 1) (= n 1))) ;;treat these cases specially.
      '()
      (let ((look-up (ass equal? (cons m n) (theory-ensemble-canonical-translations-alist theory-ensemble))))
	(if look-up (cdr look-up)
	    (let ((translations
		   (compute-canonical-translations-between-theory-multiples theory-ensemble m n)))
	      (if (not (= m 1))

		  ;;exclude this case, because building the theory multiple
		  ;;already has done this. 

		  (set (theory-ensemble-canonical-translations-alist theory-ensemble)
		       (cons (cons (cons m n) translations)
			     (theory-ensemble-canonical-translations-alist theory-ensemble))))
	      translations)))))


(define (base-theory-to-nth-replica-canonical-translation-alists
	 ensemble
	 n
	 replica-renamer)
  (return (map (lambda (x) (list (name x) ((replica-renamer n) (name x))))
	       (theory-ensemble-initial-movable-sorts ensemble))
	  (map (lambda (x) (list (name x) ((replica-renamer n) (name x))))
	       (theory-ensemble-initial-movable-constants ensemble))))
      
(define (mth-to-nth-replica-canonical-translation-alists
	 ensemble
	 m
	 n
	 replica-renamer)
  (return (map (lambda (x) (list ((replica-renamer m) (name x))
				 ((replica-renamer n) (name x))))
	       (theory-ensemble-initial-movable-sorts ensemble))
	  (map (lambda (x) (list ((replica-renamer m) (name x))
				 ((replica-renamer n) (name x))))
	       (theory-ensemble-initial-movable-constants ensemble))))

(define (COMPUTE-CANONICAL-TRANSLATIONS-BETWEEN-THEORY-MULTIPLES theory-ensemble m n)
  (if (= m 1)

      ;;THEORY-ENSEMBLE-FIND-THEORY-MULTIPLE
      ;;already computes and caches
      ;;CANONICAL-TRANSLATIONS-BETWEEN-THEORY-MULTIPLES for m=1.

      (block
	(theory-ensemble-find-theory-multiple theory-ensemble n)
	(canonical-translations-between-theory-multiples theory-ensemble m n))

      (map (lambda (permutation)
	     (compute-translation-between-theory-multiples-from-permutation
	      theory-ensemble permutation n))
	   (permutations n m))))

;;;an m permutation of n elements is a list (without repetitions) of length m taken from
;;;the set {0,1, ... ,n-1}. Each m permutation defines a translation from the theory multiple
;;;of order m to the theory multiple of order n. 

;;The following procedure computes translations for permutations of length at
;;least two. The canonical translation corresponding to a permutation of length
;;one is computed when the theory multiple is built.

(define (COMPUTE-TRANSLATION-BETWEEN-THEORY-MULTIPLES-FROM-PERMUTATION theory-ensemble permutation n)
  (let* ((m (length permutation))
	 (theory-m (theory-ensemble-find-theory-multiple theory-ensemble m))
	 (theory-n (theory-ensemble-find-theory-multiple theory-ensemble n)))
    
    (if (= m 1)
	(imps-error "COMPUTE-TRANSLATION-BETWEEN-THEORY-MULTIPLES-FROM-PERMUTATION:
permutation ~A has length 1." permutation))
    
    (let*((language-m (theory-language theory-m))
	  (language-n (theory-language theory-n))

	  ;;;name the translation: 
	  (translation-name
	   (apply
	    retrieve-unused-name
	    name->translation
	    (name theory-m)
	    '-to-
	    (name theory-n)
	    '-by-
	    (alternate-insert '- permutation)))
	  (sequence (enumerate-permutation-entries permutation))
	  (sort-name-alist '())
	  (constant-name-alist '()))
      (walk
       (lambda (x)
	 (receive (sort-alist-x constant-alist-x)
	   (mth-to-nth-replica-canonical-translation-alists
	    theory-ensemble (car x) (cdr x) (theory-ensemble-replica-renamer theory-ensemble))
	   (set sort-name-alist (append sort-name-alist sort-alist-x))
	   (set constant-name-alist (append constant-name-alist constant-alist-x))))
       sequence)
    
      (let ((translation (build-translation
			  theory-m
			  theory-n
			  the-empty-set
			  (theory-ensemble-fixed-theories-set theory-ensemble)
			  (destructure-sort-pairs sort-name-alist language-m language-n)
			  (destructure-constant-pairs constant-name-alist language-m language-n)
			  translation-name
			  translation-name
			  '#t)))

	translation))))

;;utilities

(define (permutations n m)
  (labels ((intersperse
	    (lambda (x l)
	      (if (null? l)
		  (list (list x))
		  (cons (cons x l) (map (lambda (y) (cons (car l) y)) (intersperse x (cdr l))))))))
    (cond ((= n 0) (if (= m 0) (list '())) '())
	  ((= m 0) (list '()))
	  ((= m 1) (map list (list-up-to n)))
	  (else
	   (append (permutations (- n 1) m)
		   (apply append
			  (map (lambda(x) (intersperse (- n 1) x))
			       (permutations (- n 1) (- m 1)))))))))

(define (enumerate-permutation-entries permutation)
  
  ;;;given a list (a b c ... ) returns ((0 . a) (1 . b) (2 . c) ... )
  
  (iterate loop ((j 0) (sequence '()) (permutation permutation))
    (if (null? permutation) (reverse! sequence)
	(loop (1+ j) (cons (cons j (car permutation)) sequence)
	      (cdr permutation)))))

(define (list-up-to n)
  (iterate loop ((j 0) (subscripts '()))
    (if (>= j n)
	 (reverse! subscripts)
	 (loop (1+ j)
	       (cons j subscripts)))))

(define (NATIVELY-DEFINED-CONSTANTS-IN-THEORY-MULTIPLE-AUX theory-ensemble n)
  (iterate loop ((index 1)
		 (proper-constants
		  (defined-constants-in-theory-multiple theory-ensemble n)))
    (if (>= index n)
	proper-constants
	(let* ((translations
		(canonical-translations-between-theory-multiples theory-ensemble index n))
	       (constants
		(defined-constants-in-theory-multiple theory-ensemble index))
	       (translated-constants
		(big-u
		 (map (lambda (trans)
			(map (lambda (const)
			       (translate-expression trans const))
			     constants))
		      translations))))
		  
	  (loop (1+ index) (set-diff (set-diff proper-constants translated-constants) constants))))))

(define (translations-between-theory-multiples theory-ensemble m n)
  (let ((n-max (theory-ensemble-maximum-multiple theory-ensemble)))
    (if (or (< n-max n) (< n-max m)) '()
	(let ((accum '())
	      (multiple (theory-ensemble-find-theory-multiple theory-ensemble n)))
	  (walk
	   (lambda (trans) (if (subtheory? (translation-target-theory trans) multiple)
			       (push accum trans)))
	   (theory-ensemble-instances theory-ensemble m))
	  accum))))



(define (NATIVELY-DEFINED-CONSTANTS-IN-THEORY-MULTIPLE theory-ensemble n)
  (let ((n-max (theory-ensemble-maximum-multiple theory-ensemble)))
    (iterate loop ((index (+ n 1))
		   (proper-constants
		    (natively-defined-constants-in-theory-multiple-aux theory-ensemble n)))
      (if (> index n-max)
	  proper-constants
	  (let* ((translations
		  (translations-between-theory-multiples theory-ensemble index n))
		 (constants
		  (natively-defined-constants-in-theory-multiple-aux theory-ensemble index))
		 (translated-constants
		  (big-u
		   (map (lambda (trans)
			  (map (lambda (const)
				 (translate-expression trans const))
			       constants))
			translations))))
		  
	    (loop (1+ index) (set-diff proper-constants translated-constants)))))))

(define (TRANSLATABLE-NATIVELY-DEFINED-CONSTANTS-IN-THEORY-MULTIPLE theory-ensemble n)
  (set-diff (natively-defined-constants-in-theory-multiple theory-ensemble n)
	    (theory-ensemble-nontranslating-defined-constants theory-ensemble)))

(define (NATIVELY-DEFINED-SORTS-IN-THEORY-MULTIPLE-AUX theory-ensemble n)
  (iterate loop ((index 1)
		 (proper-sorts
		  (defined-sorts-in-theory-multiple theory-ensemble n)))
    (if (>= index n)
	proper-sorts
	(let* ((translations
		(canonical-translations-between-theory-multiples theory-ensemble index n))
	       (sorts
		(defined-sorts-in-theory-multiple theory-ensemble index))
	       (translated-sorts
		(big-u
		 (map (lambda (trans)
			(map (lambda (sort)
			       (translate-sort trans sort))
			     sorts))
		      translations))))
	  (loop (1+ index) (set-diff (set-diff proper-sorts translated-sorts) sorts))))))

(define (NATIVELY-DEFINED-SORTS-IN-THEORY-MULTIPLE theory-ensemble n)
  (let ((n-max (theory-ensemble-maximum-multiple theory-ensemble)))
    (iterate loop ((index (+ n 1))
		   (proper-sorts
		    (natively-defined-sorts-in-theory-multiple-aux theory-ensemble n)))
      (if (> index n-max)
	  proper-sorts
	  (let* ((translations
		  (translations-between-theory-multiples theory-ensemble index n))
		 (sorts
		  (natively-defined-sorts-in-theory-multiple-aux theory-ensemble index))
		 (translated-sorts
		  (big-u
		   (map (lambda (trans)
			  (map (lambda (const)
				 (translate-sort trans const))
			       sorts))
			translations))))
	 	  
	    (loop (1+ index) (set-diff proper-sorts translated-sorts)))))))

(define (TRANSLATABLE-NATIVELY-DEFINED-SORTS-IN-THEORY-MULTIPLE theory-ensemble n)
  (set-diff (natively-defined-sorts-in-theory-multiple theory-ensemble n)
	    (theory-ensemble-nontranslating-defined-sorts theory-ensemble)))

(define (FIND-OR-BUILD-ENCOMPASSING-TRANSLATIONS
	 source-theory target-theory assumptions fixed-theories sort-alist constant-alist translation-name)
  (let ((successes (find-encompassing-translations 
		    source-theory
		    target-theory
		    assumptions
		    fixed-theories
		    sort-alist
		    constant-alist)))
    (if successes
	successes
	(list
	 (build-translation
	  source-theory
	  target-theory
	  assumptions
	  fixed-theories
	  sort-alist
	  constant-alist
	  translation-name
	  translation-name
	  '#t)))))


;;;;;;;;;;;;;;;;;;;;;;;;;;INSTANTIATION;;;;;;;;;;;;;;;;;;;;;;;;

;;;(define (THEORY-ENSEMBLE-BUILD-TRANSLATION-FROM-BASE-THEORY
;;;	 ensemble
;;;	 target-theory
;;;	 assumptions
;;;	 sort-name-pairs
;;;	 constant-name-pairs)
;;;  (let* ((source-theory (theory-ensemble-base-theory ensemble))
;;;	 (source-language (theory-language source-theory))
;;;	 (target-language (theory-language target-theory))
;;;	 (fixed-theories (theory-ensemble-fixed-theories-set ensemble))
;;;	 (transl-name
;;;	  (retrieve-unused-name
;;;	   name->translation
;;;	   (theory-ensemble-name ensemble)
;;;	   '-to-
;;;	   (name target-theory)))
;;;	 (sort-alist	 
;;;	  (destructure-sort-pairs sort-name-pairs source-language target-language))
;;;	 (constant-alist
;;;	  (destructure-constant-pairs constant-name-pairs source-language target-language))
;;;	 (translation
;;;	  (car (find-or-build-encompassing-translations
;;;		source-theory
;;;		target-theory
;;;		assumptions
;;;		fixed-theories
;;;		sort-alist
;;;		constant-alist
;;;		transl-name))))
;;;
;;;    translation))
      
;;;(define (THEORY-AND-ASSUMPTIONS-AS-BASE-THEORY-INSTANCE 
;;;	 ensemble
;;;	 target-theory
;;;	 assumptions
;;;	 sort-name-pairs
;;;	 constant-name-pairs
;;;	 special-renamings)
;;;  (let ((translation
;;;	 (theory-ensemble-build-translation-from-base-theory
;;;	  ensemble
;;;	  target-theory
;;;	  assumptions
;;;	  sort-name-pairs
;;;	  constant-name-pairs))
;;;	(renamer
;;;	 (namer-from-suggested-namer
;;;	  (nearly-trivial-renamer special-renamings)
;;;	  target-theory)))
;;;    (if (empty-set? assumptions)
;;;	(let ((sort-const-set
;;;	       (append (translatable-natively-defined-sorts-in-base-theory ensemble)
;;;		       (translatable-natively-defined-constants-in-base-theory ensemble))))
;;;	  (set (translation-default-renamer translation) renamer)
;;;	  (transport-defined-sorts-and-constants
;;;	   translation
;;;	   sort-const-set
;;;	   renamer)))
;;;    translation))

;;;(define (THEORY-AS-BASE-THEORY-INSTANCE 
;;;	 ensemble
;;;	 target-theory
;;;	 sort-name-pairs
;;;	 constant-name-pairs
;;;	 special-renamings)
;;;  (theory-and-assumptions-as-base-theory-instance 
;;;   ensemble
;;;   target-theory
;;;   the-empty-set
;;;   sort-name-pairs
;;;   constant-name-pairs
;;;   special-renamings))

(define (multi-association-list->translation-alists al)
  
  ;;a MULTI-ASSOCIATION-LIST is a list of entries
  ;;            ((a a1 a2 .. an) (b b1 b2 .. bn) ...)
  ;;all of the same length n+1. 

  ;;;This procedure turns a multi-association-list into a list of alists
  ;;(((a a1) (b b1) (c c1)...) ... ((a ak) (b bk) (c ck) ...))
  
  (let ((source (map car al))
	(targets (iterate loop ((targets '()) (al (map cdr al)))
		   (if (null? (car al))
		       (reverse! targets)
		       (loop (cons (map car al) targets) (map cdr al))))))
    (map (lambda (target)
	   (map (lambda (x y) (list x y))
		source target))

	 targets)))

(define (multi-association-list->translation-alist-from-theory-multiple
	 al
	 replica-renamer)
  (if (= (length (car al)) 2)
      al
      (iterate loop ((index 0)
		     (accum '())
		     (translation-alists
		      (multi-association-list->translation-alists al)))
	(if (null? translation-alists)
	    (apply append (reverse! accum))
	    (loop (1+ index)
		  (cons (map (lambda (x)

			       (list ((replica-renamer index) (car x)) (cadr x)))
			     (car translation-alists))
			accum)
		  (cdr translation-alists))))))

(define (multi-association-list-length-check al n)
  (or (list? al)
      (every? list? al)
      (every? (lambda (x) (= (length x) n)) al)
      (imps-error "~A~%is a bad multiple association list for building a translation. All entries must be lists of length ~A." a1 n)))
  
(define (rename-indexers-for-translation-from-theory-multiple al indexers)
  (let* ((alists (multi-association-list->translation-alists al))
	 (renamed-indexers
	  (map (lambda (x) (alternate-insert '% (alist-rename-indexers x indexers)))
	       alists)))
    (apply append (alternate-insert (list '$) renamed-indexers))))

(define (THEORY-ENSEMBLE-BUILD-TRANSLATION-FROM-THEORY-MULTIPLE
	 ensemble
	 target-theories;;list (th0 ... thn)
	 sort-name-associations;;list ((s s0 ... sn) ...)
	 constant-name-associations;;list ((c c0 ... cn) ...)
	 )
  (multi-association-list-length-check
   (append sort-name-associations constant-name-associations)
   (length target-theories))
  (let* ((n (length target-theories))
	 (source-theory (theory-ensemble-find-theory-multiple ensemble n))
	 (source-language (theory-language source-theory))
	 (target-theory (theory-union target-theories))
	 (target-language (theory-language target-theory))
	 (sort-name-pairs
	  (multi-association-list->translation-alist-from-theory-multiple
	   sort-name-associations
	   (theory-ensemble-replica-renamer ensemble)))
	 (constant-name-pairs
	  (multi-association-list->translation-alist-from-theory-multiple
	   constant-name-associations
	   (theory-ensemble-replica-renamer ensemble)))
	 (assumptions the-empty-set)
	 (fixed-theories (theory-ensemble-fixed-theories-set ensemble))
	 (transl-name
	  (apply
	   retrieve-unused-name
	   name->translation
	   (name source-theory)
	   '-to-
	   (alternate-insert '- (remove-duplicates eq? (map name target-theories)))))
	 (sort-alist
	  (destructure-sort-pairs sort-name-pairs source-language target-language))
	 (constant-alist
	  (destructure-constant-pairs constant-name-pairs source-language target-language)))
    (let ((translation
	   (car
	    (find-or-build-encompassing-translations
	     source-theory
	     target-theory
	     assumptions
	     fixed-theories
	     sort-alist
	     constant-alist
	     transl-name))))

      translation)))


(define (THEORY-ENSEMBLE-TRANSPORT-DEFINITIONS-FROM-THEORY-MULTIPLE
	 ensemble
	 target-theories;;list (th0 ... thn)
	 sort-name-associations;;list ((s s0 ... sn) ...)
	 constant-name-associations;;list ((c c0 ... cn) ...)
	 special-renamings)
  (let* ((translation
	  (theory-ensemble-build-translation-from-theory-multiple
	   ensemble
	   target-theories
	   sort-name-associations
	   constant-name-associations))
	 (renamer
	  (namer-from-suggested-namer
	   (nearly-trivial-renamer special-renamings)
	   (translation-target-theory translation)))
	 (index (length target-theories))
	 (sort-const-set
	  (append
	   (translatable-natively-defined-constants-in-theory-multiple ensemble index)
	   (translatable-natively-defined-sorts-in-theory-multiple ensemble index))))
    (set (translation-default-renamer translation) renamer)
    (if (not (trivial-translation? translation))
	(transport-defined-sorts-and-constants
	 translation
	 sort-const-set
	 renamer))
    translation))


(define (THEORY-ENSEMBLE-TRANSPORT-DEFINITIONS-FROM-THEORY-MULTIPLE-USING-PERMUTATION
   ensemble
   target-theories
   permutation
   sort-name-associations
   constant-name-associations
   special-renamings)
  (let ((new-target-theories
	 (choose-list-entries target-theories permutation))
	(new-sort-name-associations
	 (map (lambda (x) (cons (car x) (choose-list-entries (cdr x) permutation)))
	      sort-name-associations))
	(new-constant-name-associations
	 (map (lambda (x) (cons (car x) (choose-list-entries (cdr x) permutation)))
	      constant-name-associations)))
    (theory-ensemble-transport-definitions-from-theory-multiple
     ensemble
     new-target-theories
     new-sort-name-associations
     new-constant-name-associations
     special-renamings)))

(define (THEORY-ENSEMBLE-TRANSPORT-DEFINITIONS-FROM-THEORY-MULTIPLES-USING-MULTIPLE-PERMUTATIONS
   ensemble
   target-theories
   permutations
   sort-name-associations
   constant-name-associations
   special-renamings)
  (map
   (lambda (permutation)
     (theory-ensemble-transport-definitions-from-theory-multiple-using-permutation
      ensemble
      target-theories
      permutation
      sort-name-associations
      constant-name-associations
      special-renamings))
   permutations))

(define (TRANSPORT-DEFINITIONS-FROM-THEORY-MULTIPLES
	 ensemble
	 source-indices
	 target-theories;;list (th0 ... thn)
	 sort-name-associations;;list ((s s0 ... sn) ...)
	 constant-name-associations;;list ((c c0 ... cn) ...)
	 special-renamings)
  (let ((n (length target-theories))
	(translations '()))
    (walk
     (lambda (index)
       (let ((perms (permutations n index)))
	 (walk (lambda (permutation)
		 (let ((trans
			(theory-ensemble-transport-definitions-from-theory-multiple-using-permutation
			 ensemble
			 target-theories
			 permutation
			 sort-name-associations
			 constant-name-associations
			 special-renamings)))
		   (push translations trans)))
	       perms)))		   
     source-indices)
    translations))

(define (TRANSPORT-DEFINED-SORTS-AND-CONSTANTS-TO-THEORY-MULTIPLE
	 ensemble
	 source-indices
	 target-index)
  (let ((target-theory
	  (theory-ensemble-find-theory-multiple ensemble target-index))
	(accumulated-translations the-empty-set))
    (walk
     (lambda (index)

       ;;BEWARE! translations and permutations are homologous lists, in the sense that
       ;;the permutation at offset i corresponds to the canonical translation at offset i.

       (let ((translations
	      (canonical-translations-between-theory-multiples ensemble index target-index))
	     (perms (permutations target-index index))
	     (sort-const-set
	      (append
	       (translatable-natively-defined-sorts-in-theory-multiple ensemble index)
	       (translatable-natively-defined-constants-in-theory-multiple ensemble index))))
	 (set accumulated-translations (set-union accumulated-translations translations)) 
	 (walk
	  (lambda (translation perm)
	    (let ((renamer
		   (namer-from-suggested-namer
		    (subscripting-renamer (apply concatenate-symbol perm))
		    target-theory)))

	      (set (translation-default-renamer translation) renamer)
	      (if (not (trivial-translation? translation))
		  (transport-defined-sorts-and-constants
		   translation
		   sort-const-set
		   renamer))))
	  translations perms)))
     source-indices)
    accumulated-translations))

(define (TRANSPORT-DEFINED-SORTS-AND-CONSTANTS-TO-THEORY-MULTIPLE-USING-PERMUTATIONS
	 ensemble
	 perms
	 target-index)
  (let ((source-indices (make-set (map length perms)))
	(target-theory
	 (theory-ensemble-find-theory-multiple ensemble target-index))
	(accumulated-translations the-empty-set))
    (walk
     (lambda (index)

       ;;BEWARE! translations and permutations are homologous lists, in the sense that
       ;;the permutation at offset i corresponds to the canonical translation at offset i.

       (let ((translations
	      (canonical-translations-between-theory-multiples ensemble index target-index))
	     (all-perms (permutations target-index index))
	     (sort-const-set
	      (append
	       (translatable-natively-defined-sorts-in-theory-multiple ensemble index)
	       (translatable-natively-defined-constants-in-theory-multiple ensemble index))))

	 (set accumulated-translations (set-union accumulated-translations translations)) 
	 (walk
	  (lambda (translation perm)
	    (if (mem? equal? perm perms)
		(let ((renamer
		       (namer-from-suggested-namer
			(subscripting-renamer (apply concatenate-symbol perm))
			target-theory)))
		  (set (translation-default-renamer translation) renamer)
		  (if (not (trivial-translation? translation))
		      (transport-defined-sorts-and-constants
		       translation
		       sort-const-set
		       renamer)))))
	  translations all-perms)))
     source-indices)
    accumulated-translations))

;;;(define (transport-defined-constants-to-theory-ensemble-instances theory constant-names)
;;;  (destructure (((ensemble . multiple) (table-entry *multiple-ensemble-table* theory)))
;;;    (let ((instances (theory-ensemble-instances ensemble multiple)))
;;;      (walk
;;;       (lambda (trans)
;;;	 (let ((renamer (translation-default-renamer trans)))
;;;	   (transport-defined-constants trans constant-names (map renamer constant-names))))
;;;       instances)
;;;      instances)))


;;;(define (transport-defined-constants-to-theory-ensemble-instances theory constant-names)
;;;  (let ((instances (interpretations-from-theory theory)))
;;;    (walk
;;;       (lambda (trans)
;;;	 (let ((renamer (or (translation-default-renamer trans)
;;;			    identity)))
;;;	   (transport-defined-constants trans constant-names (map renamer constant-names))))
;;;       instances)
;;;      instances))
;;;
