/******************************************************************************
  anf_ideal_2_z.c
******************************************************************************/
 
#include "kant.h"        

t_void
anf_ideal_2_z WITH_2_ARGS(
	order,		ord,
	anf_ideal,	id
)
/*******************************************************************************
 
Description:

        Computes a Z-basis presentation of an ideal if a 2-element presentation 
        is given. This is done by Hermite reduction of the (n,2*n)-matrix given
        by the two representation matrices of the generators. If the first 
        generator is a rational integer a modular technique can be applied since 
	in this case the first matrix is a diagonal matrix.
 
	The coefficient order must be Z, if this is not the case the reduction 
	method must be changed.
 
        If the "one-position" of the order is 1 this function also sets the
        minimum of the ideal.
 
  
Calling sequence:
 
        anf_ideal_2_z(ord, id);
 
        order   	ord:   order over which id is defined
        anf_ideal       id:    ideal (will get the Z-basis presentation add'lly)

 
History:
 
	92-09-22 JS    case module is 1 (again!)
	92-09-02 CO    changed || to && in line 68;
		       denominator must be 1 if we set the minimum
	92-03-17 JS    case one of the generators is 1
	91-09-01 JS    first version

*******************************************************************************/
{       
	block_declarations;
 
	integer_small	deg;
        order		ordcoef;
	matrix		mat, mat1, mat2, mat0, hnf;
	anf_elt		gen1, gen2, gen1mul, gen2mul;
	integer_big	denom, den1, den2, module, mult1, mult2;
 
	order_must_be_over_z(ord);
 
	if(!anf_ideal_is_2(id)) 
	   error_internal("anf_ideal_2_z: No 2-element presentation given.");
 
        ordcoef = order_coef_order(ord);
	deg = order_rel_degree(ord);
   
	gen1 = anf_ideal_gen1(id);
	gen2 = anf_ideal_gen2(id);
	den1 = anf_elt_is_integer(gen1) ? 1 : anf_elt_den(gen1);
	den2 = anf_elt_is_integer(gen2) ? 1 : anf_elt_den(gen2);
         
/*
    trivial case
*/
        if (gen1 == 1 && gen2 == 1)
        {
		anf_ideal_tran_den(id) = 1;
		anf_ideal_tran_hnf(id) = 1;
		anf_ideal_tran(id)     = mat_ring_create_id(ordcoef, deg);
		anf_ideal_min(id)      = 1;
                return;
        }
 
	if(!anf_elt_is_integer(gen1))
	{       
/*
    We paste together the two representation matrices and compute
    the Hermite normal form of the resulting (deg, deg+deg) matrix
 
    Before doing that we have to compute the common denominator
*/
 
		denom = integer_lcm(den1, den2);
 
		if(denom > 1)
		{
			mult1 = integer_div(denom, den1);
			mult2 = integer_div(denom, den2);
 
			gen1mul = anf_mult_z(ord, gen1, mult1);
			gen2mul = anf_mult_z(ord, gen2, mult2);
 
			mat1 = anf_rep_mat(ord, gen1mul);
			mat2 = anf_rep_mat(ord, gen2mul);
 
			anf_elt_delete(ord, &gen1mul);
			anf_elt_delete(ord, &gen2mul);
			integer_delref(mult1);
			integer_delref(mult2);
		}
		else
		{
			mat1 = anf_rep_mat(ord, gen1);
			mat2 = anf_rep_mat(ord, gen2);
		}

		mat0 = mat_new(deg, deg+deg);
		mat_ring_create_zero_sub(ordcoef, &mat0);
		mat  = mat_ring_insert(ordcoef, mat0, mat1, 1, 1);
		mat_delref(ordcoef, &mat0);     
    
		mat0 = mat;
		mat  = mat_ring_insert(ordcoef, mat0, mat2, 1, deg+1);
 
		mat_delref(ordcoef, &mat0);
		mat_delref(ordcoef, &mat1);
		mat_delref(ordcoef, &mat2);
 
		hnf = mat_ring_hnf_col_upper(ordcoef, mat);
 
		anf_ideal_tran(id) = 
			mat_ring_submat(ordcoef, hnf, 1, 1, deg, deg);
		anf_ideal_tran_den(id) = denom;
		anf_ideal_tran_hnf(id) = 1;
		if((order_one_position(ord) == 1) && (denom == 1))
		{
			anf_ideal_min(id) = 
			integer_incref(mat_elt(anf_ideal_tran(id), 1, 1));
		}
 
		mat_delref(ordcoef, &mat);
		mat_delref(ordcoef, &hnf);
	}
	else
	{
/*
    We compute the modular Hermite normal form of the representation
    matrix of the second generator. As module we use the product of
    first generator and the denominator of the second generator.
*/
		
		mat = anf_rep_mat(ord, gen2);
 
		module = integer_mult(gen1, den2);
 
                if (module == 1) 
                        hnf = mat_ring_create_id(ordcoef, deg);
                else
			hnf = mat_ring_hnf_mod_upper(ordcoef, mat, module);


		anf_ideal_tran(id)     = hnf;
		anf_ideal_tran_den(id) = integer_incref(den2);
		anf_ideal_tran_hnf(id) = 1; 
 
		if((order_one_position(ord) == 1) && (denom == 1))
		{
			anf_ideal_min(id) = 
			integer_incref(mat_elt(anf_ideal_tran(id), 1, 1));
		}
 
		mat_delref(ordcoef, &mat);
		integer_delref(module);

	}
 
	return;
}
