/******************************************************************************
  lat_sublattice_enlarge.c
******************************************************************************/

#include "kant.h"

matrix
lat_sublattice_enlarge WITH_1_ARG(
	vector,		m
)
/******************************************************************************
 
Description:	Let b_1,...,b_{k+1} be vectors of a k-dimensional lattice
		such that b_1,...,b_k are linearly independent and let
		m_1,...,m_{k+1} (|m_1|+...+|m_{k+1}|>0) be integers such
		that 0 = \sum_{i=1}^{k+1} m_i b_i .

		We compute a (k+1)x(k) matrix trans such that
		(b_1,...,b_{k+1})*trans
		is a basis of \sum_{i=1}^{k+1} ZZ b_i.

		See:	M. Pohst and H. Zassenhaus,
			"Algorithmic Algebraic Number Theory",
			Cambridge Univ. Press 1989, p.211.

Calling sequence:
 
	trans =  lat_sublattice_enlarge(m);

	trans	: matrix	= see above
	m	: vector	= see above (m = (m_1,...,m_{k+1})

History:
 
	92-12-09 KW	written
 
******************************************************************************/
{
	block_declarations;
                        
	integer_big	tempa,tempb,tempc;
	integer_small	i,j,jj,kk;
	matrix		tran1,tran2;
	t_handle	Z;
	vector		mm;

	Z = structure_z;

	kk = vec_length(m);
	mm = vec_new(kk);
	for (i=1;i<=kk;i++) vec_entry(mm,i) = integer_incref(vec_entry(m,i));

/*
**	Assure gcd(m_1,...,m_{kk}) = 1.
*/
	mat_z_simplify(Z,mm,&tempa);
	integer_delref(tempa);

	tran1 = mat_ring_create_id(Z,kk);
	do
	{
/*
**		Determine non-zero m_j of minimal absolute value
*/
		j = 0;
		tempa = 0;
		for (i=1;i<=kk;i++)
		{
			tempb = integer_abs(vec_entry(mm,i));
			if ( (tempb) && 
				((!j) || (integer_compare(tempb,tempa) != 1)) )
			{
				j = i;
				integer_delref(tempa);
				tempa = tempb;
			}
			else integer_delref(tempb);
		}
		integer_delref(tempa);

/*
**		Division with remainder
*/                                     
		jj = 0;
		tempc = vec_entry(mm,j);
		for (i=1;i<=kk;i++)
		{		
			integer_quot_rem(vec_entry(mm,i),tempc,&tempa,&tempb);
			if (tempb)
			{
				jj++;
				integer_delref(vec_entry(mm,i));
				vec_entry(mm,i) = tempb;
				mat_ring_col_add(Z,tran1,i,j,tempa,1,kk);
			}
			else integer_delref(tempb);
			integer_delref(tempa);
		}
	}
	while(jj);

/*
**	Delete column j of tran1
*/
	tran2 = mat_new(kk,kk-1);
	for (jj=1;jj<=kk;jj++)
	{
		for (i=1;i<j;i++)
		{
			mat_elt(tran2,jj,i) = integer_incref(mat_elt(tran1,jj,i));
		}
		for (i=j+1;i<=kk;i++)
		{
			mat_elt(tran2,jj,i-1) = integer_incref(mat_elt(tran1,jj,i));
		}
	}

	mat_delref(Z,&tran1);
	vec_delete(Z,&mm);

	return(tran2);
}
