/*  inthdl_power.c
*/

#include "defs.h"
#include "integer.e"
#include "inthdl.e"
#include "intbig.h"


void
inthdl_power	WITH_3_ARGS(
    inthdl_handle, ahdl,
    t_int, n,
    inthdl_handle, result
)
/*
Raise the integer in block with t_handle ahdl to the power n, where n is a
beta-digit, returning the result in the block with t_handle result, which must
be a pre-allocated integer block different from ahdl with room for at
least n * intbig_curr_size(bhdl) beta-digits.

Note that n must be non-negative, and if the integer in ahdl is zero then
n must be positive: otherwise, the resulting behaviour is not defined.
*/
{
    inthdl_length	alen;
    inthdl_handle	wrk;
    inthdl_handle	apower;
    inthdl_handle	total;
    inthdl_length	reslen;

    if (n == 0)
    {
	intbig_sign(result) = 1;
	intbig_digit(result, 0) = 1;
	intbig_curr_size(result) = 1;
	return;
    }

    alen = intbig_curr_size(ahdl);

    if (n == 1)
    {
	if (ahdl != result)
	{
	    intbig_sign(result) = intbig_sign(ahdl);
	    intbig_copy_digits(ahdl, 0, alen, result, 0);
	    intbig_curr_size(result) = alen;
	}
	return;
    }

    /*
    To calculate the result, we store the base integer (from ahdl)
    in apower, then repeatedly square apower, multiplying apower
    into the final total each time the relevant bit of n is one.

    We need three blocks for working variables: wrk, apower and total.
    Their handles will be permuted as we go to avoid copying digits about.
    We use result for wrk to start with; apower and total will be allocated
    when first needed.
    */

    wrk = result;
    apower = total = 0;
    reslen = n * alen;

#define swap(x,y)	{ register inthdl_handle temp = x; x = y; y = temp; }

    for (;;)
    {
	if (n & 1)
	{
	    if (total == 0)
	    {
		/* set total = apower (or ahdl if apower not yet calculated) */

		inthdl_handle	ap;
		inthdl_length	aplen;

		total = inthdl_buf_alloc(reslen);

		ap = apower ? apower : ahdl;
		aplen = intbig_curr_size(ap);

		intbig_sign(total) = intbig_sign(ap);
		intbig_copy_digits(ap, 0, aplen, total, 0);
		intbig_curr_size(total) = aplen;
	    }
	    else
	    {
		/* set total = total * apower (apower must exist by now) */

		inthdl_mult(total, apower, wrk);
		swap(wrk, total);
	    }
	}

	if ((n >>= 1) == 0)
	    break;

	/* square apower (or ahdl if apower not yet calculated) */

	if (apower == 0)
	{
	    apower = inthdl_buf_alloc(reslen);
	    inthdl_square(ahdl, apower);
	}
	else
	{
	    inthdl_square(apower, wrk);
	    swap(wrk, apower);
	}
    }

    /*
    By now both apower and total must exist since n started greater than 1.
    Note that t_handle result is equal to exactly one of wrk, apower and total.
    */

    if (total != result)
    {
	inthdl_length	rlen = intbig_curr_size(total);

	intbig_sign(result) = intbig_sign(total);
	intbig_copy_digits(total, 0, rlen, result, 0);
	intbig_curr_size(result) = rlen;

	inthdl_buf_delete(total);
    }

    if (wrk != result)
	inthdl_buf_delete(wrk);

    if (apower != result)
	inthdl_buf_delete(apower);
}
