#include "defs.h"
#include "mp.e"
#include "mp.h"


mp_float
mp_ei	WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Returns y = ei(x) = -e1(-x) 
	  = principal value of integral from - infinity to x of exp(u)/u du.

The time taken is O(t * M(t)).
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);
    mp_round_type	save_round = round;
    mp_sign_type	x_sign;
    mp_base_type	b;
    mp_length		t, new_t;
    mp_acc_float	temp1, temp2, temp3;
    mp_ptr_type		temp1_ptr, temp2_ptr, temp3_ptr;
    mp_int		log_t, i;


    mp_check_2("mp_ei", xp, yp);

    x_sign = mp_sign(xp);

    if (x_sign == 0)

	/*
	ei(0) is undefined.
	*/

	mp_error("mp_ei: x is zero");
    

    b = mp_b(xp);
    t = mp_t(xp);

    new_t = (6 * t + 9) / 5;

    round = MP_TRUNC;
    mp_acc_float_alloc_3(b, new_t, temp1, temp2, temp3);
    mp_change_up();

    mp_move(x, temp1);
    mp_abs(temp1, temp2);


    /*
    Restore t.
    */

    temp1_ptr = mp_acc_float_ptr(temp1);
    temp2_ptr = mp_acc_float_ptr(temp2);
    temp3_ptr = mp_acc_float_ptr(temp3);
    yp = mp_ptr(y);


#define fix_pointers()		if (mp_has_changed()) 			    \
				{					    \
				    temp1_ptr = mp_acc_float_ptr(temp1);    \
				    temp2_ptr = mp_acc_float_ptr(temp2);    \
				    temp3_ptr = mp_acc_float_ptr(temp3);    \
				    yp = mp_ptr(y);			    \
				}

    mp_t(temp1_ptr) = mp_t(temp2_ptr) = mp_t(temp3_ptr) = t;


    /*
    See if int_abs(x) is large enough to use the asymptotic series.
    Note that 7/10 > log(2).
    */

    log_t = mp_times_log2_b(t, b);


    if (mp_cmp_q(temp2, 7 * log_t, 10) > 0)
    {
	/*
	Use asymptotic series, without increasing t.  mp_exp() will give an
	error message if x is too large.
	*/

	mp_int	i;


	mp_rec(x, temp2);
	mp_exp(x, y);

	fix_pointers();

	if (!mp_is_zero(yp))
	{
	    mp_mul_eq(y, temp2);
	    mp_copy(y, temp1);

	    i = 0;


	    fix_pointers();

	    do
	    {
		/*
		Reduce t if possible for multiply.
		*/

		mp_length    mul_t = t + 2 + mp_expt(temp1_ptr) - mp_expt(yp);


		/*
		Return if terms small enough to be negligible.
		*/

		if (mul_t <= 2)
		    break;

		if (mul_t > t)
		    mul_t = t;

		mp_t(temp1_ptr) = mp_t(temp2_ptr) = mp_t(temp3_ptr) = mul_t;


		mp_copy(temp1, temp3);
		mp_mul_eq(temp1, temp2);
		mp_mul_int_eq(temp1, ++i);

		/*
		Return if terms increasing.
		*/

		if (mp_cmp_abs(temp1, temp3) >= 0)
		    break;

		/*
		Restore t of temp1 for addition.
		*/

		fix_pointers();
		mp_t(temp1_ptr) = t;

		mp_add_eq(y, temp1);

		fix_pointers();
	    } while (!mp_is_zero(temp1_ptr));
	}
    }


    /*
    See if x is negative and the continued fraction usable.  The constant
    1/15 was determinded empirically and may be decreased (but not increased)
    if desired.
    */

    else if (x_sign < 0 && mp_cmp_q(temp2, log_t, 15) > 0)
    {
	/*
	Here t * log(b) / 10 < -x <= t * log(b), so use the continued fraction.
	Use the equivalent of 6 decimal places for the forward recurrence.
	We could use single-precision real arithmetic if trusted.
	*/

	mp_int		j = 0, k = t;
	mp_length	t6;


	t6 = mp_change_base(b, 10, 5) + 1;

	if (t6 < 2)
	    t6 = 2;

	else if (t6 > t)
	    t6 = t;

	
	/*
	Temporarily change t of x also.
	*/

	xp = mp_ptr(x);
	mp_t(temp1_ptr) = mp_t(temp2_ptr) = mp_t(temp3_ptr) = mp_t(xp) = t6;

	mp_neg(x, temp1);
	mp_int_to_mp(1, temp2);

	do
	{
	    /*
	    Use the forward recurrence to find out how many terms are needed
	    in the backward recurrence.
	    */

	    mp_div_int(temp1, ++j, temp3);
	    mp_add_eq(temp2, temp3);
	    mp_mul(x, temp2, temp3);
	    mp_sub_eq(temp1, temp3);

	    fix_pointers();

	    if (!mp_is_zero(temp2_ptr))
		while (mp_expt(temp2_ptr) > 1)
		{
		    mp_expt(temp2_ptr)--;
		    mp_expt(temp1_ptr)--;

		    if ((k -= 2) <= 0)
			break;
		}

	} while (k > 0);


	/*
	Restore t for backward recurrence.
	*/

	xp = mp_ptr(x);
	fix_pointers();

	mp_t(temp1_ptr) = mp_t(temp2_ptr) = mp_t(temp3_ptr) = mp_t(xp) = t;

	mp_abs(x, temp2);


	/*
	Now use backward recurrence.
	*/

	mp_int_to_mp(1, temp1);

	do
	{
	    mp_div_int(temp2, j, temp3);
	    mp_add_eq(temp1, temp3);
	    mp_mul(x, temp1, temp3);
	    mp_sub_eq(temp2, temp3);

	    /*
	    Scale to avoid overflow.
	    */

	    fix_pointers();
	    mp_expt(temp1_ptr) -= mp_expt(temp2_ptr);
	    mp_expt(temp2_ptr) = 0;

	} while (--j > 0);

	mp_div_eq(temp1, temp2);

	mp_exp(x, y);
	mp_mul_eq(y, temp1);

	fix_pointers();
	mp_set_sign(yp, -mp_sign(yp));
    }

    else
    {
	/*
	Use the power series, but increase t if x is negative to
	compensate for cancellation.
	*/

	mp_length	new_t1, new_t2;
	mp_int		i;


	t++;

	if (x_sign < 0)
	{
	    /*
	    x is negative so the result is about b^-td and the terms are about
	    b^td so we need up to 2 * td extra digits to compensate for
	    cancellation.  mp_change_base(...)/3 underestimates log(b).
	    */

	    mp_length		td, upper2;


	    mp_mul_q(temp2, 3, mp_change_base(2, int_max(2, b / 2), 2), temp3);

	    td = mp_to_int(temp3) + 1;

	    new_t2 = t + td;
	    new_t1 = int_min(new_t2 + td, new_t);

	    upper2 = (11 * t + 19) / 10;
	    if (new_t2 > upper2)
		new_t2 = upper2;

	    /*
	    Use new_t2 for log and Euler's constant computation.
	    */

	    fix_pointers();
	    mp_t(temp1_ptr) = mp_t(temp2_ptr) = mp_t(temp3_ptr) = new_t2;
	}

	else
	    new_t1 = new_t2 = t;


	/*
	Now prepare to sum power series.
	*/

	mp_log(temp2, temp3);
	mp_eul(temp1);
	mp_add_eq(temp1, temp3);


	/*
	Now use new_t1 digits for summing power series.
	Restore sign of temp2.
	*/

	fix_pointers();
	mp_t(temp1_ptr) = mp_t(temp2_ptr) = mp_t(temp3_ptr) = new_t1;

	mp_set_sign(temp2_ptr, x_sign);
	mp_add_eq(temp1, temp2);

	fix_pointers();
	mp_copy_ptr(temp2_ptr, temp3_ptr);

	i = 0;

	do
	{
	    /*
	    Reduce t if possible for multiplications.
	    */

	    mp_bool	changed_t;


	    if (changed_t = x_sign > 0 || mp_expt(temp3_ptr) <= 0)
	    {
		register mp_length		mul_t;


		mul_t = mp_expt(temp3_ptr) + 2 +
			    (x_sign > 0? new_t1 - mp_expt(temp1_ptr): new_t2);

		if (mul_t <= 2)
		    break;

		if (mul_t > new_t1)
		    mul_t = new_t1;

		mp_t(temp2_ptr) = mp_t(temp3_ptr) = mul_t;
	    }
	    
	    mp_mul_eq(temp3, temp2);

	    i++;
	    mp_mul_double_q(temp3, i, 1, i + 1, i + 1);


	    if (changed_t)
	    {
		/*
		Restore t for addition.
		*/

		fix_pointers();
		mp_t(temp3_ptr) = new_t1;
	    }

	    mp_add_eq(temp1, temp3);

	    fix_pointers();
	} while (!mp_is_zero(temp3_ptr));


	mp_t(temp1_ptr) = t - 1;
	mp_copy(temp1, y);
    }

    round = save_round;

    mp_acc_float_delete(temp3);
    mp_acc_float_delete(temp2);
    mp_acc_float_delete(temp1);
    mp_change_down();

    return y;
}



mp_float
mp_li	WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Returns y = li(x) = logarithmic integral of x
	  = principal value of integral from 0 to x of du/log(u).

The time taken is O(t * M(t)).  The condition on x is that x >= 0 && x != 1.
Error in y could be induced by an O(b^(1 - t)) relative perturbation in
x followed by similar perturbation in y.  Thus relative error in y is
small unless x is close to 1 or to the zero 1.45136923488338105028...
of li(x).
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);
    mp_round_type	save_round;


    mp_check_2("mp_li", xp, yp);

    if (mp_is_neg(xp))
	mp_error("mp_li: x is negative");

    if (mp_is_zero(xp))
    {
	mp_set_sign(yp, 0);

	return y;
    }

    if (mp_cmp_int(x, 1) == 0)
	mp_error("mp_li: x is 1");

    /*
    x > 0 && x != 1, so use ei(log(x)) with truncated arithmetic.
    */

    save_round = round;
    round = MP_TRUNC;

    mp_log(x, y);
    mp_ei(y, y);

    round = save_round;

    return y;
}
