#include "defs.h"
#include "mp.e"
#include "mp.h"


mp_float
mp_erf		WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Returns y = erf(x) = sqrt(4/pi) * (integral from 0 to x of exp(-u^2) du).
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);
    mp_round_type	save_round = round;
    mp_base_type	b;
    mp_length		t, new_t;
    mp_sign_type	x_sign;
    mp_acc_float	temp1, temp2;
    mp_ptr_type		temp1_ptr, temp2_ptr;


    mp_check_2("mp_erf", xp, yp);

    x_sign = mp_sign(xp);

    if (x_sign == 0)
    {
	mp_set_sign(yp, 0);

	return y;
    }


    b = mp_b(xp);
    t = mp_t(xp);

    round = MP_TRUNC;
    new_t = t + 1 + mp_extra_guard_digits(mp_times_log2_b(t, b), b);

    mp_change_up();
    mp_acc_float_alloc_2(b, new_t, temp1, temp2);

    mp_move(x, temp1);

    temp1_ptr = mp_acc_float_ptr(temp1);
    temp2_ptr = mp_acc_float_ptr(temp2);


#define fix_pointers()		if (mp_has_changed())			    \
				{					    \
				    temp1_ptr = mp_acc_float_ptr(temp1);    \
				    temp2_ptr = mp_acc_float_ptr(temp2);    \
				    yp = mp_ptr(y);	    		    \
				}


    mp_set_sign(temp1_ptr, 1);

    mp_mul(temp1, temp1, temp2);

    /*
    Note that 7/10 > log(2).
    */

    if (mp_cmp_q(temp2, 7 * mp_times_log2_b(t, b), 10) >= 0)

	/*
	Here int_abs(x) is so large that erf(x) is +-1 to full accuracy.
	*/

	mp_int_to_mp(x_sign, temp2);

    else
    {
	/*
	Try to use the asymptotic series.  We can possibly reduce t temporarily.
	*/

	mp_length	working_t;
	mp_int		irx;
	mp_bool		erf3_worked;


	mp_mul_q_eq(temp2, 10, mp_change_base(2, b, 7));
	irx = mp_to_int(temp2);

	working_t = 4 * mp_guard_digits(100, b);

	if (working_t < new_t - irx)
	    working_t = new_t - irx;

	else if (working_t > new_t)
	    working_t = new_t;

	fix_pointers();
	mp_t(temp1_ptr) = mp_t(temp2_ptr) = working_t;


	erf3_worked = mp_erf3(temp1, temp1, FALSE);

	if (!erf3_worked)
	{
	    /*
	    The asymptotic series is insufficient, so use power series.
	    */

	    fix_pointers();
	    mp_t(temp1_ptr) = mp_t(temp2_ptr) = new_t;

	    mp_move(x, temp1);
	    mp_erf2(temp1, temp1);
	}


	/*
	In both cases multiply by sqrt(4/pi) * exp(-x^2).
	*/

	mp_move(x, temp2);
	mp_mul_eq(temp2, temp2);

	fix_pointers();
	mp_set_sign(temp2_ptr, -mp_sign(temp2_ptr));

	mp_exp(temp2, temp2);
	mp_mul_eq(temp1, temp2);
	mp_pi(temp2);
	mp_root(temp2, -2, temp2);
	mp_mul_eq(temp1, temp2);

	if (erf3_worked)
	{
	    mp_mul_int_eq(temp1, -2);

	    fix_pointers();
	    mp_set_digits_zero(mp_digit_ptr(temp1_ptr, working_t),
							new_t - working_t);

	    mp_t(temp1_ptr) = mp_t(temp2_ptr) = new_t;

	    mp_add_int(temp1, 1, temp2);

	    if (x_sign < 0)
	    {
		fix_pointers();
		mp_set_sign(temp2_ptr, -mp_sign(temp2_ptr));
	    }
	}

	else
	    /*
	    We used the power series.
	    */

	    mp_mul_int(temp1, 2, temp2);

    }

    round = save_round;
    mp_move_round(temp2, y);


    /*
    Ensure that result is in the range -1 to +1.
    */

    if (mp_has_changed())
	yp = mp_ptr(y);

    if (!mp_is_zero(yp) && mp_expt(yp) > 0)
	mp_int_to_mp(x_sign, y);

    mp_acc_float_delete(temp2);
    mp_acc_float_delete(temp1);
    mp_change_down();

    return y;
}



mp_float
mp_erfc		WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Returns y = erfc(x) = 1 - erf(x).
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);
    mp_round_type	save_round = round;
    mp_base_type	b;
    mp_length		t, new_t;
    mp_acc_float	temp1, temp2;


    mp_check_2("mp_erfc", xp, yp);

    if (!mp_is_pos(xp))
    {
	/*
	If x <= 0, there is no significant loss of accuracy in using erf(x).
	*/

	round = mp_fix_directed(-1, round);
	mp_erf(x, y);

	round = save_round;

	/*
	Set y = 1 - y.
	*/

	yp = mp_ptr(y);
	mp_set_sign(yp, -mp_sign(yp));

	mp_add_int_eq(y, 1);

	return y;
    }


    b = mp_b(xp);
    t = mp_t(xp);

    round = MP_TRUNC;
    new_t = t + 1 + mp_extra_guard_digits(mp_times_log2_b(t, b), b);

    mp_acc_float_alloc_2(b, new_t, temp1, temp2);

    mp_move(x, temp1);


    if (mp_erf3(temp1, temp1, FALSE))
    {
	/*
	The asymptotic series worked, so multiply by sqrt(4/pi) * exp(-x^2).
	*/

	mp_move(x, temp2);
	mp_mul_eq(temp2, temp2);

	mp_set_sign(mp_acc_float_ptr(temp2), -mp_sign(mp_acc_float_ptr(temp2)));

	mp_exp(temp2, temp2);
	mp_mul_eq(temp1, temp2);
	mp_pi(temp2);
	mp_root(temp2, -2, temp2);
	mp_mul_eq(temp1, temp2);
	mp_mul_int_eq(temp1, 2);
    }

    else
    {
	/*
	The asymptotic series is inaccurate, so we have to use mp_erf(),
	increasing the precision to compensate for cancellation.  An alternative
	method (possibly faster) would be to use the continued fraction for
	exp(x^2) * erfc(x).
	*/

	mp_t(mp_acc_float_ptr(temp1)) = t;
	mp_mul(x, x, temp1);


	/*
	log(b) > mp_change_base(2, b, 80) / 120.
	*/

	mp_mul_q_eq(temp1, 120, mp_change_base(2, b, 80));
	t += mp_to_int(temp1);

	mp_acc_float_delete(temp1);
	new_t = t + 1 + mp_extra_guard_digits(mp_times_log2_b(t, b), b);

	mp_acc_float_alloc(b, new_t, temp1);

	mp_move(x, temp1);
	mp_erf(temp1, temp1);

	mp_set_sign(mp_acc_float_ptr(temp1), -mp_sign(mp_acc_float_ptr(temp1)));

	mp_add_int_eq(temp1, 1);
    }

    round = save_round;
    mp_move_round(temp1, y);

    mp_acc_float_delete(temp2);
    mp_acc_float_delete(temp1);

    return y;
}


void
mp_priv_erf2		WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Returns y = erf(x) using the power series for small x.  Called by mp_erf().
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);
    mp_round_type	save_round = round;
    mp_base_type	b;
    mp_length		t, new_t;
    mp_acc_float	temp1, temp2;
    mp_sign_type	x_sign;


    if ((x_sign = mp_sign(xp)) == 0)
    {
	mp_set_sign(yp, 0);
	return;
    }

    b = mp_b(xp);
    t = mp_t(xp);

    round = MP_TRUNC;
    mp_acc_float_alloc_2(b, t, temp1, temp2);

    mp_mul(x, x, temp1);


    /*
    Note that 7/10 > log(2).
    */
    
    if (mp_cmp_q(temp1, 7 * mp_times_log2_b(t, b), 10) > 0)
    {
	/*
	int_abs(x) is large, so integral is +-sqrt(pi/4) to available accuracy.
	If int_abs(x) is too large, mp_exp() will give an error message.
	*/

	mp_exp(temp1, temp1);

	mp_pi(y);
	mp_sqrt(y, y);
	mp_div_int_eq(y, 2 * x_sign);
	mp_mul_eq(y, temp1);
    }
    else
    {
	/*
	Use the power series.
	*/

	mp_ptr_type	temp1_ptr, temp2_ptr;
	mp_int		i;


	mp_copy(x, y);

	mp_mul_int_eq(temp1, 2);
	mp_copy(x, temp2);

	temp1_ptr = mp_acc_float_ptr(temp1);
	temp2_ptr = mp_acc_float_ptr(temp2);
	yp = mp_ptr(y);


#define fix_pointers()		if (mp_has_changed())			    \
				{					    \
				    temp1_ptr = mp_acc_float_ptr(temp1);    \
				    temp2_ptr = mp_acc_float_ptr(temp2);    \
				    yp = mp_ptr(y);	    		    \
				}


	i = 1;

	do
	{
	    mp_length	mul_t = t + 2 + mp_expt(temp2_ptr) - mp_expt(yp);

	    if (mul_t <= 2)
		break;

	    if (mul_t > t)
		mul_t = t;

	    mp_t(temp1_ptr) = mp_t(temp2_ptr) = mul_t;

	    mp_mul_eq(temp2, temp1);
	    mp_div_int_eq(temp2, i += 2);

	    /*
	    Fix t of temp2 for addition.
	    */

	    fix_pointers();
	    mp_t(temp2_ptr) = t;

	    mp_add_eq(y, temp2);

	    fix_pointers();
	} while (!mp_is_zero(temp2_ptr));
    }

    round = save_round;

    mp_acc_float_delete(temp2);
    mp_acc_float_delete(temp1);
}


mp_bool
mp_priv_erf3	WITH_3_ARGS(
	mp_float,	x,
	mp_float,	y,
	mp_bool,	to_infinity
)
/*
If to_infinity is true, sets:

    y = exp(x^2) * integral from x to infinity of exp(u^2) du,

otherwise sets:

    y = exp(-x^2) * integral from 0 to x of exp(u^2) du.


In both cases the asymptotic series is used.  The return value of the function
is whether x is large enough for the asymptotic series to give full accuracy.
The condition on x for the return value to be true is approximately that
x > sqrt(t * log(b)).
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp;
    mp_round_type	save_round = round;
    mp_base_type	b;
    mp_length		t;
    mp_acc_float	temp1, temp2;
    mp_ptr_type		temp1_ptr, temp2_ptr;
    mp_int		i;


    /*
    Check that we can get at least t - 2 digits accuracy.
    */

    DEBUG_BEGIN(DEBUG_ERF);
    DEBUG_PRINTF_1("+erf3 {\n");
    DEBUG_1("x = ", xp);

    if (mp_cmp_int(x, 1) <= 0)
    {
	DEBUG_PRINTF_1("-} fail\n");
	DEBUG_END();

	return FALSE;
    }

    b = mp_b(xp);
    t = mp_t(xp);

    round = MP_TRUNC;
    mp_acc_float_alloc_2(b, t, temp1, temp2);

    xp = mp_ptr(x);

    if (mp_expt(xp) < t)
    {
	mp_mul(x, x, temp1);

	/*
	Note that 7/10 > log(2).
	*/

	if (mp_cmp_q(temp1, 7 * mp_times_log2_b(t - 2, b), 10) <= 0)
	{
	    mp_acc_float_delete(temp2);
	    mp_acc_float_delete(temp1);

	    DEBUG_PRINTF_1("-} fail\n");
	    DEBUG_END();
	    return FALSE;
	}
    }

    /*
    Now it is worth trying the asymptotic series.
    */

    mp_rec(x, y);
    mp_mul(y, y, temp1);
    mp_div_int_eq(temp1, 2);

    temp1_ptr = mp_acc_float_ptr(temp1);
    temp2_ptr = mp_acc_float_ptr(temp2);
    yp = mp_ptr(y);

#define fix_pointers()		if (mp_has_changed())			    \
				{					    \
				    temp1_ptr = mp_acc_float_ptr(temp1);    \
				    temp2_ptr = mp_acc_float_ptr(temp2);    \
				    yp = mp_ptr(y);	    		    \
				}

    if (!to_infinity)
	mp_set_sign(temp2_ptr, -mp_sign(temp2_ptr));

    
    mp_div_int_eq(y, 2);
    mp_copy(y, temp2);

    i = 1;

    fix_pointers();

    do
    {
	/*
	Sum series, reducing t if possible.
	*/

	mp_expt_type	temp2_expt = mp_expt(temp2_ptr);
	mp_length	mul_t = t + 2 + temp2_expt - mp_expt(yp);

	if (mul_t <= 2)
	    break;
	
	if (mul_t > t)
	    mul_t = t;

	mp_t(temp1_ptr) = mp_t(temp2_ptr) = mul_t;

	mp_mul_eq(temp2, temp1);
	mp_mul_int_eq(temp2, i);

	i += 2;

	/*
	Restore t of temp2 for addition.
	*/

	fix_pointers();
	mp_t(temp2_ptr) = t;

	/*
	Check if terms are getting larger - if so x is too small for
	the asymptotic series to be accurate.
	*/

	if (mp_expt(temp2_ptr) > temp2_expt)
	{
	    /*
	    Give up: return failure.
	    */

	    round = save_round;
	    mp_acc_float_delete(temp2);
	    mp_acc_float_delete(temp1);

	    DEBUG_PRINTF_1("-} fail\n");
	    DEBUG_END();

	    return FALSE;
	}

	mp_add_eq(y, temp2);

	fix_pointers();
    } while (!mp_is_zero(temp2_ptr));


    /*
    Return successfully.
    */

    round = save_round;
    mp_acc_float_delete(temp2);
    mp_acc_float_delete(temp1);

    DEBUG_1("-} works, y = ", yp);
    DEBUG_END();

    return TRUE;
}
