#include "defs.h"
#include "mp.e"
#include "mp.h"

mp_float
mp_priv_exp1		WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Sets y = exp(x) - 1 and returns y given that int_abs(x) < 1.
The time taken is O(sqrt(t)*M(t)).  Accumulator operations are performed.
*/
{
    /*
    The algorithm is described in: R.P. Brent, The Complexity of
    Multiple-Precision Arithmetic (in Complexity of Computational
    Problem Solving, Univ. of Queensland Press, Brisbane, 1976,
    126-165).  Asymptotically faster methods exist, but are not useful
    unless t is very large. See comments to mp_atan() and mp_pi_gl().
    */

    mp_ptr_type		xp = mp_ptr(x);
    mp_length		t, new_t;
    mp_round_type	save_round = round;
    mp_base_type	b;
    mp_int		i, q;
    mp_expt_type	expt;
    mp_acc_float	x_copy, term, sum;
    mp_ptr_type		x_copy_ptr, term_ptr, sum_ptr;


    DEBUG_BEGIN(DEBUG_OTHER);
    DEBUG_PRINTF_1("+exp1 {\n");
    DEBUG_1("x = ", xp);

    if (mp_is_zero(xp))
    {
	mp_set_sign(mp_ptr(y), 0);

	DEBUG_PRINTF_1("-} y = 1\n");
	DEBUG_END();

	return y;
    }

    t = mp_t(xp);
    b = mp_b(xp);
    expt = mp_expt(xp);

    if (expt > 0)
	mp_error("mp_exp1: int_abs(x) is not less than 1");

    /*
    Compute approximately optimal q (the scaling index).
    */

    q = 0;

    if (expt >= -t)
    {
	mp_int	la = mp_times_log2_b(t / 3, b);

	/*
	Compute integer square root.
	*/

	do
	{
	    q++;
	}
	while (q * q <= la);

	q += expt * mp_times_log2_b(1, b);

	if (q < 0)
	    q = 0;
    }

    new_t = t + 1 + mp_extra_guard_digits(q, b);


    /*
    Use truncation rather than rounding internally.
    */

    if (round == MP_RND)
	round = MP_TRUNC;


    mp_acc_float_alloc_3(b, new_t, x_copy, term, sum);
    mp_change_up();

    mp_move(x, x_copy);


    if (q > 0)
    {
	mp_int		multiplier = 1, limit = MAX_INT / 2;

	/*
	Halve x q times.
	*/

	for (i = 0; i < q; i++)
	{
	    multiplier <<= 1;

	    if (multiplier >= limit || multiplier == b || i == q - 1)
	    {
		mp_div_int_eq(x_copy, multiplier);
		multiplier = 1;
	    }
	}
    }

    x_copy_ptr = mp_acc_float_ptr(x_copy);
    term_ptr = mp_acc_float_ptr(term);
    sum_ptr = mp_acc_float_ptr(sum);

    mp_copy_ptr(x_copy_ptr, term_ptr);
    mp_copy_ptr(x_copy_ptr, sum_ptr);


#define fix_pointers()		if (mp_has_changed()) {			\
				    x_copy_ptr = mp_acc_float_ptr(x_copy); \
				    term_ptr = mp_acc_float_ptr(term);	\
				    sum_ptr = mp_acc_float_ptr(sum);	\
				}

    i = 1;

    do
    {
	if (round == MP_TRUNC)
	{
	    /*
	    Reduce t if possible.
	    */

	    mp_length mul_t = new_t + 2 + mp_expt(term_ptr) - mp_expt(sum_ptr);

	    if (mul_t <= 2)
		break;
	    
	    if (mul_t > new_t)
		mul_t = new_t;

	    mp_t(x_copy_ptr) = mp_t(term_ptr) = mul_t;
	}

	mp_mul_eq(term, x_copy);
	mp_div_int_eq(term, ++i);

	if (round == MP_TRUNC)
	{
	    /*
	    Restore t of term for addition.
	    */

	    fix_pointers();

	    mp_t(term_ptr) = new_t;
	}

	mp_add_eq(sum, term);

	fix_pointers();

    } while (mp_expt(sum_ptr) - mp_expt(term_ptr) <= new_t + 1);


    if (round == MP_RND_UP)
    {
	/*
	Rounding up is needed, so add int_abs(last term).
	*/

	mp_set_sign(term_ptr, int_abs(mp_sign(term_ptr)));
	mp_add_eq(sum, term);
    }

    if (q > 0)
    {
	/*
	Apply sum = (sum + 1)^2 - 1 [i.e. sum = sum(sum + 2)] for q iterations.
	Allocating the temporary float two is better than calling mp_add_int()
	with the integer argument 2 every time.
	*/

	mp_acc_float		two;

	mp_acc_float_alloc(b, new_t, two);
	mp_int_to_mp(2, two);

	for (i = 0; i < q; i++)
	{
	    /*
	    Use term as a temporary variable.
	    */

	    mp_add(sum, two, term);
	    mp_mul_eq(sum, term);
	}
	mp_acc_float_delete(two);
    }


    /*
    Move result into y.
    */

    mp_move(sum, y);


    /*
    Restore accumulator stack pointer and level and rounding type.
    */

    mp_acc_float_delete(term);
    mp_acc_float_delete(sum);
    mp_acc_float_delete(x_copy);
    mp_change_down();

    round = save_round;

    DEBUG_1("-} y = ", mp_ptr(y));
    DEBUG_END();

    return y;
}



mp_float
mp_exp		WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Sets y = exp(x) and returns y.  The time taken is O(sqrt(t)*M(t)).
*/
{
    mp_ptr_type		xp = mp_ptr(x);
    mp_length		t;
    mp_base_type	b;
    mp_expt_type	expt;
    mp_acc_float	temp;


    DEBUG_BEGIN(DEBUG_OTHER);
    DEBUG_PRINTF_1("+exp {\n");
    DEBUG_1("x = ", xp);

    if (mp_is_zero(xp))
    {
	mp_int_to_mp(1, y);

	DEBUG_PRINTF_1("-} y = 1\n");
	DEBUG_END();

	return y;
    }


    t = mp_t(xp);
    b = mp_b(xp);
    expt = mp_expt(xp);


    /*
    Check whether int_abs(x) < 1.
    */

    if (expt <= 0)
    {
	mp_acc_float_alloc(b, t + mp_guard_digits(100, b), temp);

	mp_move(x, temp);
    }
    else
    {
	/*
	Use truncation rather than rounding internally.
	*/

	if (round == MP_RND)
	    round = MP_TRUNC;

	mp_acc_float_alloc(b, t + 1 + expt + mp_guard_digits(100, b), temp);

	/*
	Copy x and divide by b^expt (i.e. set exponent to 0).
	*/

	mp_move(x, temp);
	mp_acc_expt(temp) = 0;
    }


    mp_exp1(temp, temp);
    mp_add_int_eq(temp, 1);


    /*
    Restore the exponent by raising the result to the b-th power expt times.
    Under/overflow will occur if x is out of allowable range.
    */

    while (expt-- > 0)
	mp_abs_int_power(temp, b, temp);
    

    mp_move(temp, y);
    mp_acc_float_delete(temp);

    DEBUG_1("-} y = ", mp_ptr(y));
    DEBUG_END();

    return y;
}
