#include "defs.h"
#include "mp.e"
#include "mp.h"


mp_float
mp_priv_atn1	WITH_2_ARGS(
	mp_int,		n,
	mp_float,	y
)
/*
Sets y = arctan(1 / n) and returns y.  The usual power series is used.
Accumulator operations are performed.
*/
{
    mp_ptr_type		yp = mp_ptr(y);
    mp_length		t, new_t;
    mp_base_type	b;
    mp_int		i, limit, squ_n;
    mp_acc_float	term, sum;
    mp_ptr_type		term_ptr, sum_ptr;
    mp_round_type	save_round = round;

    DEBUG_BEGIN(DEBUG_ATAN);
    DEBUG_PRINTF_1("+atn1 {\n");
    DEBUG_PRINTF_2("n = %d\n", n);

    if (n <= 1)
	mp_error("mp_atn1: n is not greater than 1");

    if (round == MP_RND)
	round = MP_TRUNC;

    t = mp_t(yp);
    b = mp_b(yp);

    new_t = t + 1 + mp_extra_guard_digits(mp_times_log2_b(t, b), b);

    mp_acc_float_alloc_2(b, new_t, term, sum);
    mp_change_up();

    mp_q_to_mp(1, n, term);

    term_ptr = mp_acc_float_ptr(term);
    sum_ptr = mp_acc_float_ptr(sum);
    mp_copy_ptr(term_ptr, sum_ptr);


#define fix_pointers()		if (mp_has_changed())			\
				{					\
				    term_ptr = mp_acc_float_ptr(term);	\
				    sum_ptr = mp_acc_float_ptr(sum);	\
				}

    limit = (MAX_INT / n) / n - 2;
    squ_n = n * n;

    i = 1;

    do
    {
	if (round == MP_TRUNC)
	{
	    mp_length mul_t = new_t + 2 + mp_expt(term_ptr) - mp_expt(sum_ptr);

	    if (mul_t < 2)
		break;

	    if (mul_t > new_t)
		mul_t = new_t;

	    mp_t(term_ptr) = mul_t;
	}

	if (i < limit)
	    mp_mul_q_eq(term, -i, (i + 2) * squ_n);

	else
	{
	    mp_mul_q_eq(term, -i, i + 2);
	    mp_mul_double_q(term, 1, 1, n, n);
	}

	i += 2;

	if (round == MP_TRUNC)
	{
	    /*
	    Fix t of term for addition.
	    */

	    fix_pointers();
	    mp_t(term_ptr) = new_t;
	}

	mp_add_eq(sum, term);

	fix_pointers();

    } while (mp_expt(sum_ptr) - mp_expt(term_ptr) <= new_t + 2); 

    mp_move_round(sum, y);

    mp_acc_float_delete(sum);
    mp_acc_float_delete(term);
    mp_change_down();

    round = save_round;

    DEBUG_1("-} y = ", yp);
    DEBUG_END();

    return y;

#undef fix_pointers
}


mp_float
mp_atan		WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Returns y = arctan(x) using an O(t * M(t)) method which could easily be
modified to an O(sqrt(t) * M(t)) method (as in mp_exp1()).  y is in the
range -pi/2 to pi/2.
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);
    mp_round_type	save_round = round;
    mp_base_type	b;
    mp_length		t, new_t;
    mp_int		q, i;
    mp_acc_float	x_copy, squ_x, atan, one;
    mp_ptr_type		x_copy_ptr, squ_x_ptr, atan_ptr;


    DEBUG_BEGIN(DEBUG_ATAN);
    DEBUG_PRINTF_1("+atan {\n");
    DEBUG_1("x = ", xp);

    mp_check_2("mp_atan", xp, yp);

    if (mp_is_zero(xp))
    {
	mp_set_sign(yp, 0);

	DEBUG_PRINTF_1("y = 0");
	DEBUG_END();
	return y;
    }


    b = mp_b(xp);
    t = mp_t(xp);

    new_t = t + 1 + mp_extra_guard_digits(mp_times_log2_b(t, b), b);

    mp_change_up();
    mp_acc_float_alloc_4(b, new_t, x_copy, squ_x, atan, one);

    if (round == MP_RND)
	round = MP_TRUNC;

    mp_move(x, x_copy);

    /*
    Set the temporary variable one to 1.  This saves calculating it every time
    in the following loop.
    */

    mp_int_to_mp(1, one);
    x_copy_ptr = mp_acc_float_ptr(x_copy);


    /*
    Reduce x so that its absolute value is <= 1/2.  This is done using the
    indentity:

	arctan(x) = 2 * arctan(x / (1 + sqrt(1 + x^2)))
    */

    for (q = 1;
	 mp_expt(x_copy_ptr) > 0 ||
	    mp_expt(x_copy_ptr) == 0 && 2 * (mp_digit(x_copy_ptr, 0) + 1) > b;
	 q <<= 1
	)
    {
	/*
	Set x = x / (1 + sqrt(1 + x^2)).  First account for directed rounding -
	change sense of directed rounding for the denominator.
	*/

	round = mp_fix_directed(-mp_sign(x_copy_ptr), round);

	mp_mul(x_copy, x_copy, squ_x);
	mp_add_eq(squ_x, one);
	mp_sqrt(squ_x, squ_x);
	mp_add_eq(squ_x, one);

	if (mp_has_changed())
	    x_copy_ptr = mp_acc_float_ptr(x_copy);

	/*
	Restore (directed) rounding for division.
	*/

	round = mp_fix_directed(-mp_sign(x_copy_ptr), round);

	mp_div_eq(x_copy, squ_x);

	if (mp_has_changed())
	    x_copy_ptr = mp_acc_float_ptr(x_copy);
    }


    /*
    Now x is in the range [-1/2, 1/2], so use the power series:

	arctan(x) = x - x^3/3 + x^5/5 - x^7/7 + ....

    */

    squ_x_ptr = mp_acc_float_ptr(squ_x);
    atan_ptr = mp_acc_float_ptr(atan);
    mp_copy_ptr(x_copy_ptr, atan_ptr);

    mp_mul(x_copy, x_copy, squ_x);

    i = 1;

#define fix_pointers()		if (mp_has_changed()) \
				{ 					    \
				    x_copy_ptr = mp_acc_float_ptr(x_copy);  \
				    squ_x_ptr = mp_acc_float_ptr(squ_x);    \
				    atan_ptr = mp_acc_float_ptr(atan);	    \
				}

    fix_pointers();

    do
    {
	if (round == MP_TRUNC)
	{
	    /*
	    Reduce t if possible.
	    */

	    mp_length	t = new_t + 2 + mp_expt(x_copy_ptr);

	    if (t <= 2)
		break;
	    
	    if (t > new_t)
		t = new_t;

	    mp_t(x_copy_ptr) = mp_t(squ_x_ptr) = t;
	}

	mp_mul_eq(x_copy, squ_x);
	mp_mul_q_eq(x_copy, -i, i + 2);

	i += 2;

	if (round == MP_TRUNC)
	{
	    fix_pointers();
	    mp_t(x_copy_ptr) = new_t;
	}

	mp_add_eq(atan, x_copy);

	fix_pointers();

    } while (mp_expt(atan_ptr) - mp_expt(x_copy_ptr) <= new_t + 1);


    /*
    Correct for argument reduction - q is 2^(number of times x was reduced).
    */

    mp_mul_int_eq(atan, q);


    mp_move_round(atan, y);

    mp_acc_float_delete(one);
    mp_acc_float_delete(atan);
    mp_acc_float_delete(squ_x);
    mp_acc_float_delete(x_copy);
    mp_change_down();

    round = save_round;

    DEBUG_1("-} y = ", yp);
    DEBUG_END();

    return y;
}


mp_float
mp_acos		WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Returns y = arccos(x), assuming that int_abs(x) <= 1.  y is in the range 0
to pi.  The method is to use mp_asin(), so the time taken is O(M(t) * t).
No rounding options are considered and no guard digits are used.
*/
{
    mp_ptr_type		xp = mp_ptr(x);
    mp_acc_float	temp;

    /*
    Set y = pi/2 - arcsin(x).
    */

    mp_pi(y);
    mp_div_int_eq(y, 2);

    mp_acc_float_alloc(mp_b(xp), mp_t(xp), temp);
    mp_asin(x, temp);
    
    mp_sub_eq(y, temp);
    mp_acc_float_delete(temp);

    return y;
}


mp_float
mp_asin		WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Returns y = arcsin(x), assuming that int_abs(x) <= 1.  y is in the range -pi/2
to pi/2.  The method is to use mp_atan(), so the time taken is O(M(t) * t).
No rounding options are considered and no guard digits are used.
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);
    mp_round_type	save_round = round;


    mp_check_2("mp_asin", xp, yp);

    if (mp_is_zero(xp))
    {
	mp_set_sign(yp, 0);

	return y;
    }

    if (mp_expt(xp) > 0)
    {
	/*
	Here int_abs(x) is >= 1, so see if x is +-1.
	*/

	mp_sign_type	sign = mp_sign(xp);

	if (mp_cmp_int(x, sign))
	    mp_error("mp_asin: int_abs(x) > 1");

	
	/*
	x is +-1, so return +-pi/2.
	*/

	mp_pi(y);
	mp_div_int_eq(y, 2 * sign);
    }

    else
    {

	/*
	Here 0 < int_abs(x) < 1, so use arctan(x/sqrt(1 - x^2)).
	*/

	mp_acc_float	temp1, temp2;


	mp_acc_float_alloc_2(mp_b(xp), mp_t(xp), temp1, temp2);
	mp_int_to_mp(1, temp1);

	/*
	To compute (1 - x^2) we multiply (1 - x) by (1 + x).  The following
	subtraction is exact if x is close to 1, and the following addition is
	exact if x is close to -1.  Thus (1 - x^2) is computed with small
	relative error.   temp1 is set to 1 + x, temp2 set to 1 - x.
	*/

	mp_sub(temp1, x, temp2);
	mp_add_eq(temp1, x);

	mp_mul_eq(temp1, temp2);
	mp_root(temp1, -2, temp1);

	mp_mul(x, temp1, y);


	/*
	Now y = x/sqrt(1 - x^2).
	*/

	mp_atan(y, y);

	mp_acc_float_delete(temp2);
	mp_acc_float_delete(temp1);
    }

    round = save_round;

    return y;
}


mp_float
mp_atan2	WITH_3_ARGS(
	mp_float,	y,
	mp_float,	x,
	mp_float,	z
)
/*
Returns z = arctan(y/x) in the appropriate quadrant.  If both x and y are zero,
an error occurs.  If x is zero and y non-zero, z is sign(y) * pi/2.  Otherwise
z is in the range (-pi, pi] according to the quadrant which x and y indicate.
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y), zp = mp_ptr(z);
    mp_base_type	b;
    mp_acc_float	temp1, temp2;
    mp_sign_type	x_sign, y_sign;

    
    mp_check_3("mp_atan2", xp, yp, zp);

    x_sign = mp_sign(xp);
    y_sign = mp_sign(yp);

    b = mp_b(xp);

    mp_acc_float_alloc_2(b, mp_t(xp) + 1 + mp_extra_guard_digits(1, b),
							temp1, temp2);

    if (x_sign == 0)
    {
	if (y_sign == 0)
	    mp_error("mp_atan2: x and y are both zero");

	mp_pi(temp1);
	mp_div_int_eq(temp1, y_sign * 2);
    }
    else
    {
	mp_move(x, temp1);
	mp_move(y, temp2);

	mp_div_eq(temp2, temp1);

	mp_atan(temp2, temp1);

	if (x_sign < 0)
	{
	    /*
	    Add/subtract pi to get angle in correct quadrant.  The angle is
	    currently in the range (-pi/2, pi/2).  If y < 0 (3rd quadrant),
	    subtract pi to get angle in range (-pi, -pi/2).  If y >= 0 (2nd
	    quadrant), add pi to get angle in range (pi/2, pi].
	    */

	    mp_pi(temp2);

	    if (y_sign < 0)
		mp_set_sign(mp_acc_float_ptr(temp2), -1);

	    mp_add_eq(temp1, temp2);
	}
    }

    mp_move_round(temp1, z);

    mp_acc_float_delete(temp2);
    mp_acc_float_delete(temp1);

    return z;
}
