#include "defs.h"
#include "mp.e"
#include "mp.h"


void
mp_priv_move	WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Copies the sign, exponent, and digits of x into y and either pads with zeros
or rounds appropriately.  x & y need not have the same number of digits, but
their base must be equal.  Rounding is determined by the rounding type of y.
This routine is mainly used to copy one mp number to another which has a
different precision (e.g. a temporary variable).  Accumulator operations are
performed if t of y is less than t of x.
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);
    mp_length		tx, ty, guard;
    mp_sign_type	sign;


    DEBUG_BEGIN(DEBUG_OTHER);
    DEBUG_PRINTF_1("+move {\n");
    DEBUG_1("x = ", xp);

    if (mp_b(xp) != mp_b(yp))
	mp_error("mp_move: x and y have different bases");

    tx = mp_t(xp);
    ty = mp_t(yp);

    sign = mp_sign(xp);
    guard = tx - ty;

    if (sign == 0)
	mp_set_sign(yp, 0);
    else if (guard <= 0)
    {
	/*
	x has less digits than y so pad with trailing zeros.
	*/

	mp_set_sign(yp, sign);
	mp_expt(yp) = mp_expt(xp);

	mp_dig_copy(mp_digit_ptr(yp, 0), mp_digit_ptr(xp, 0), tx);
	mp_set_digits_zero(mp_digit_ptr(yp, tx), -guard);
    }
    else
    {
	/*
	Move x to temporary storage and round.
	*/

	mp_acc_index		acc_index;
	mp_digit_ptr_type	p;

	mp_change_up();
	mp_acc_digit_alloc(tx, acc_index);

	p = mp_acc_digit_ptr(acc_index);

	if (mp_has_changed())
	{
	    xp = mp_ptr(x);
	    yp = mp_ptr(y);
	}

	mp_dig_copy(p, mp_digit_ptr(xp, 0), tx);
	mp_nzr(sign, mp_expt(xp), yp, p, guard);

	mp_acc_delete(acc_index);
	mp_change_down();
    }

    DEBUG_1("-} y = ", yp);
    DEBUG_END();
}



void
mp_priv_move_round	WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Moves x + s * (b^-(t of y)) * int_abs(x) into y using mp_move(), where s is:
	 0	if rounding is truncating or round to nearest,
	-1	if rounding is round down,
	+1	if rounding is round up.

The restrictions on x and y are the same as those in mp_move() (q.v.).
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);
    mp_length		tx, ty;
    mp_acc_float	temp;
    mp_ptr_type		temp_ptr;


    if (round == MP_TRUNC || round == MP_RND || mp_is_zero(xp))
    {
	mp_move(x, y);
	return;
    }

    DEBUG_BEGIN(DEBUG_OTHER);
    DEBUG_PRINTF_1("+rnd {\n");
    DEBUG_1("x = ", xp);

    if (mp_b(xp) != mp_b(yp))
	mp_error("mp_rnd: x and y have different bases");

    tx = mp_t(xp);
    ty = mp_t(yp);

    mp_acc_float_alloc(mp_b(xp), tx, temp);
    temp_ptr = mp_acc_float_ptr(temp);

    mp_copy(x, temp);


    /*
    Divide temp by b^ty.
    */

    mp_expt(temp_ptr) -= ty;


    /*
    Negate temp according to sign of x and rounding type.
    */

    mp_mul_int_eq(temp, (round == MP_RND_UP? 1: -1) * mp_sign(mp_ptr(x)));


    /*
    Set y = x + temp.
    */

    mp_add_eq(temp, x);
    mp_move(temp, y);

    mp_acc_float_delete(temp);

    DEBUG_1("-} y = ", mp_ptr(y));
    DEBUG_END();
}
