#include "defs.h"
#include "debug.e"
#include "integer.e"
#include "poly.h"
#include "poly_z_faclst.h"
#include "poly_mat.h"
#include "zm.e"

t_handle
poly_u_zm_fact WITH_3_ARGS(
    t_handle,         pring,
    t_int,    pdig,
    t_poly,        apoly
)
/*
** MODUPOLY_FACT: Modular univariate polynomial factorisation using
** the Berlekamp algorithm. ( MUPFBL ).
**
** pdig: prime beta-integer
** apoly: monic univariate polynomial over Zpdig.
** Returns: a list of the factors of apoly in polynomial form.
*/
{
    block_declarations;
    t_int    adeg;          /* degree of apoly                     */
    t_poly        aderivpoly;    /* derivative of apoly                 */
    t_poly        gcdpoly;       /* gcd at various places               */
    t_handle           resmh;         /* result t_handle                       */
    t_handle           Qmatrixh;      /* t_handle for the Q matrix constructed */
    t_handle           vectorsh;      /* t_handle for the matrix of vectors    */
                                    /* in the nullspace                    */
    t_int    i;             /* loop counter                        */


    DENY( m_poly_const( apoly ) );

    ASSERT( integer_is_single( pdig ));

    ASSERT(m_poly_univariate( m_poly_poly_to_handle (apoly)));

    ASSERT( m_poly_u_zm_is_monic( pdig, apoly ));

    IF_DEBUG_FLAG( DEBUG_MODUPOLY_FACT,
    {
	cay_print( "entering poly_u_zm_fact()\napoly = " );
	poly_z_write( pring, apoly );
	dmplin();
    }
    );

    /* apoly is non-constant */

    adeg = poly_deg( apoly );

    if ( adeg < 2 )
    {
	resmh = m_poly_z_faclst_alloc( 1 );
	m_poly_z_faclst_factor( resmh, 0 ) = m_modpoly_incref( pring, apoly );
	m_poly_z_faclst_power( resmh, 0 ) = 1;
	m_poly_z_faclst_len_put( resmh, 1 );

	IF_DEBUG_FLAG( DEBUG_MODUPOLY_FACT,
	{
	    cay_print( "linear polynomial, poly_u_zm_fact() returns\n" );
	    poly_z_faclst_write( pring, resmh );
	}
	);
        return  resmh;
    }

    /* Assume that apoly is not const as adeg >= 2 */
    /* Ensure that apoly is square free            */

    aderivpoly = poly_u_zm_deriv( pring, pdig, apoly );
    gcdpoly = poly_u_zm_gcd( pring, pdig, apoly, aderivpoly );
    m_modpoly_delref( pring, aderivpoly );

    if ( ! poly_z_is_one_poly (pring, gcdpoly))
    {
        /* if gcd(apoly,aderivpoly) != 1 then apoly is not sq free */

        resmh = poly_u_zm_fact_b1( pring, pdig, apoly, gcdpoly );
	m_modpoly_delref( pring, gcdpoly );

        return  resmh;
    }

    /* apoly is square free */
    m_modpoly_delref (pring, gcdpoly);

    Qmatrixh = poly_u_zm_fact_b2( pring, pdig, apoly );

    /* Set Qmatrix = Qmatrix - I where I is an identity matrix of */
    /* the same dimension as Qmatrix, in preparation for step b3. */

    for ( i = m_poly_mat_row( Qmatrixh ); i >= 1; i-- )
    {
        m_poly_mat_elt( Qmatrixh, i, i ) = modint_subtract( pdig, m_poly_mat_elt( Qmatrixh, i, i ), 1 );
    }

    IF_DEBUG_FLAG( DEBUG_MODUPOLY_FACT,
    {
	cay_print( "Qmatrix = Qmatrix - I is\n" );
	poly_z_Qmatrix_write( Qmatrixh );
    }
    );

    vectorsh = poly_u_zm_fact_b3( pring, pdig, Qmatrixh );

    IF_DEBUG_FLAG( DEBUG_MODUPOLY_FACT,
    {
	cay_print( "vectors are\n" );
	poly_z_Qmatrix_write( vectorsh );
    }
    );

    mem_delete_hptr( &Qmatrixh );

    if ( m_poly_mat_row( vectorsh ) == 1 )
    {
	resmh = m_poly_z_faclst_alloc( 1 );
	m_poly_z_faclst_factor( resmh, 0 ) = m_modpoly_incref( pring, apoly );
	m_poly_z_faclst_power( resmh, 0 ) = 1;
	m_poly_z_faclst_len_put( resmh, 1 );

        mem_delete_hptr( &vectorsh );

	IF_DEBUG_FLAG( DEBUG_MODUPOLY_FACT,
	{
	    cay_print( "apoly irreducible, poly_u_zm_fact() returns\n" );
	    poly_z_faclst_write( pring, resmh );
	}
	);

        return  resmh;
    }

    resmh = poly_u_zm_fact_b4( pring, pdig, apoly, vectorsh );

    IF_DEBUG_FLAG( DEBUG_MODUPOLY_FACT,
    {
        cay_print( "poly_u_zm_fact() returns\n" );
        poly_z_faclst_write( pring, resmh );
    }
    );

    mem_delete_hptr( &vectorsh );

    return  resmh;
}

