/*  integer.e

Specification of the public interface to the multi-precision integer arithmetic
package

The package is based on the SAC-2 Arithmetic System (REFERENCE?  ...  Loos,
Karlsruhe, 1980?).  This implementation was developed by Robyn Deed, Jim
Richardson and Graeme Woodward of the Department of Pure Mathematics, University
of Sydney.

There are three layers to the package:

    The integer_ interface: a convenient high-level interface allowing the
    caller to do arithmetic on large integers represented by a single datum
    which is either a small integer or a specially marked t_handle for a
    block containing a large integer.  The caller is shielded from all
    details of memory allocation, thus achieving ease of coding possibly at
    the expense of performance.  Deletion of large integers after use,
    however, *is* the responsibility of the caller.  The interface is specified
    in this include file (integer.e).

    The inthdl_ interface: the caller typically passes handles for
    arguments which are already existing large integers, and a t_handle for
    a preallocated block of size large enough for the result.  This should
    be faster than using integer_ calls if much arithmetic is being done
    in a loop.  Calling routines should include both integer.e and
    inthdl.e.

    The inner kernel intbig_: not to be called directly from outside the
    package.

Several low-level routines for performing arithmetic on beta-digits have names
with prefix ib_, but may be called at will as if they were part of the integer_
interface.
*/

/*
The remainder of this file specifies the public interface to the integer_
level of the package.

The following macros are for maintaining reference counters.  Their use
requires that "trbl_declarations;" appear in the declarations of automatic
variables: this provides a temporary variable, thus permitting the macros
to be called with complicated expressions as arguments.

Increment the reference counter of a big integer (unless it is single-
precision); return the integer itself as the value of the macro to allow
embedding in calculations:
*/

/*
 *	7 Aug, 1990, standard include file multiple inclusion stuff.
 */

#ifndef _integer_e_
#define _integer_e_

#include "defs.h"

#ifdef POSH
#define integer_incref(i)       \
    ( BLOCK_TEMP_VARIABLE = i, \
      integer_is_single( BLOCK_TEMP_VARIABLE ) || \
        block_incref( BLOCK_TEMP_VARIABLE - integer_BETA ), \
      BLOCK_TEMP_VARIABLE )
#else
#define integer_incref(i)       \
    ( BLOCK_TEMP_VARIABLE = i, \
      integer_is_single( BLOCK_TEMP_VARIABLE ) || \
        block_incref( integer_BETA - BLOCK_TEMP_VARIABLE ), \
      BLOCK_TEMP_VARIABLE )
#endif

/*
Decrement the reference counter of a big integer (unless single-precision)
and delete it if the reference counter goes to zero
*/

#ifdef POSH
#define integer_delref(i)       \
    (void)( BLOCK_TEMP_VARIABLE = i, \
            integer_is_single( BLOCK_TEMP_VARIABLE ) || \
                block_decref_delete( BLOCK_TEMP_VARIABLE - integer_BETA ) )
#else
#define integer_delref(i)       \
    (void)( BLOCK_TEMP_VARIABLE = i, \
            integer_is_single( BLOCK_TEMP_VARIABLE ) || \
                block_decref_delete( integer_BETA - BLOCK_TEMP_VARIABLE ) )
#endif

#define integer_delete(pi)	\
    (void)( integer_delref( *(pi) ), *(pi) = integer_BETA )


/*
Next we define the radix base, integer_BETA, for the representation of large
integers.  Note that the definition of integer_BETA is MACHINE-DEPENDENT (see
integer_internals.h for details).
*/

#define integer_BETA	536870912

/*
Type for a "beta-digit", that is, an ordinary machine integer with absolute
value less than integer_BETA:
*/
 
#if 0
typedef t_word	integer_small;
#endif

/*
The following type is for general single- or multi-precision integers,
represented either as a small integer, i.e., a beta-digit, or as an
inthdl_handle marked in a special way (see inthdl.e):
*/

#if 0
typedef t_word	integer_big;
#endif

/*
The following type is for the "action function" called by integer_break()
to return the successive characters of the decimal representation of a
big integer to the caller.  Note that because of limitations in C prototype
specification for functional arguments, the function will be called with and
must accept a long int argument, even though the value will always be a
char.
*/

#if 0
typedef void	(*integer_break_action)( /* long int */ );
#endif

/*
The following macro may be applied to an integer_big to determine if
it is an ordinary small integer, as opposed to a specially marked t_handle
for a block storing a multi-precision integer.  The macro returns TRUE
in the former case.
*/

#define integer_is_single(i)	((i) < integer_BETA)

/* Macro for transferring the contents of b to a. */
#define integer_reassign(a,b) \
    {integer_big intgr_tmp_mat420 = (a); (a) = (b); integer_delref(intgr_tmp_mat420);}

/*
Macros for debugging printout
*/

#ifdef DEVELOP

#ifdef	INTBIG_MASTER_DEBUG_DEF
/* GET A BRAIN!!!!!!!!! */
#if 0
t_word	intbig_debug_flag = 0;
#endif
#else
extern t_word	intbig_debug_flag;
#endif

#define DEBUG_INTEGER(x)	if(intbig_debug_flag) integer_debug x
#define DEBUG_INTEGER_0(m)	DEBUG_INTEGER( (m, 0) )
#define DEBUG_INTEGER_1(m,a)	DEBUG_INTEGER( (m, 1, a) )
#define DEBUG_INTEGER_2(m,a,b)	DEBUG_INTEGER( (m, 2, a, b) )

#else

#define DEBUG_INTEGER(x)
#define DEBUG_INTEGER_0(m)
#define DEBUG_INTEGER_1(m,a)
#define DEBUG_INTEGER_2(m,a,b)

#endif


/*
Testing evenness is done efficiently by the following:
*/

#define integer_is_even(a)	( ! integer_rem( a, 2 ) )

#include "integer_protos.h"

/*
 *	Useful comparison stuff
 */

#define integer_le(a,b)			(integer_compare(a,b)<=0)
#define integer_lt(a,b)			(integer_compare(a,b)<0)
#define integer_gt(a,b)			(integer_compare(a,b)>0)
#define integer_ge(a,b)			(integer_compare(a,b)>=0)
#define integer_eq(a,b)			(integer_compare(a,b)==0)
#define integer_ne(a,b)			(integer_compare(a,b)!=0)

/* 
** Protos for Allan's SMALL integer functions in mat/ifuncs.c
*/
extern t_handle int_max P_((t_int, t_int));
extern t_handle int_min P_((t_int, t_int));

#endif /* _integer_e_ */
