
#ifndef MAT_INTERNALS_INCLUDED
#define MAT_INTERNALS_INCLUDED


#include "mat.e"
#include "att.h"

/*
** Ordering of the flags inside a matrix.
*/

typedef enum {
	MAT_ATT_PACK = 0,
	MAT_ATT_TOTAL
	} mat_att_numbers;

/*
** Structure of the matrix header.
*/

typedef struct
	{
		t_block_header	trbl_hdr;
		integer_small	rows;
		integer_small	cols;
		integer_small	flgs[ att_req_length( MAT_ATT_TOTAL )];
	}
	mat_header;

#define MAT_HDR_LEN	(sizeof(mat_header) / sizeof(t_word))

/*
** Structure of the matrix block.
*/

typedef struct
	{
		mat_header	mat_hdr;
		t_word		mat_elts[ VARIABLE_LENGTH ];
	}
	mat_struct;

/*
** Size definitions and requirements.
*/

#define mat_req_size( m, n ) \
		( (m) * (n) + MAT_HDR_LEN )

#define mat_avail_size( h ) \
		( mem_words( h ) - MAT_HDR_LEN )

/*
** Access to information in the matrix header.
*/

#define mat_row( h ) \
		(((mat_header *) mem_access(h)) -> rows)

#define mat_col( h ) \
		(((mat_header *) mem_access(h)) -> cols)

#define mat_flgs( h ) \
		(((mat_header *) mem_access(h)) -> flgs)

/*
** Information contained in the flags in the header.
*/

#define mat_is_packed(h) \
		att2_is_true( mat_flgs(h), MAT_ATT_PACK )

#define mat_set_pack(h) \
		att2_set_true( mat_flgs(h), MAT_ATT_PACK )

#define mat_set_unpack(h) \
		att2_set_false( mat_flgs(h), MAT_ATT_PACK )

/*
** Access to elements of the matrix.
** mat_entry( mat, i ): i = 1, ..., m*n.
** mat_elt( mat, i, j ): i = 1, ..., m; j = 1, ..., n.
*/

#define mat_elt0_ptr( h ) \
		(&((((mat_struct *) mem_access(h)) -> mat_elts)[-1]))

#define mat_entry( h, i ) \
		(mat_elt0_ptr(h)[(i)])

#define mat_elt(h, i, j) \
		mat_entry((h), ((i) - 1)*mat_col(h) + (j))

/*
** Buffers.
*/

#define mat_buff_alloc mat_new

#define mat_buff_free mem_delete_hptr

/*
** Macros to hide allocation of unpacked versions of matrices versus
** actual use of matrices.
*/

#define mat_alloc_result_unpkd( respack, c, cprime, m, n ) \
	{ \
		if ( respack ) \
		{ \
			cprime = mat_buff_alloc( m, n ); \
		} \
		else \
		{ \
			if ( c == 0 ) \
			{ \
				c = mat_new( m, n ); \
			} \
			cprime = c; \
		} \
	}

#define mat_create_result( ring, respack, c, cprime ) \
	{ \
		if ( respack ) \
		{ \
			mat_pack( ring, cprime, &(c) ); \
			mat_buff_free( &(cprime) ); \
		} \
	}

#define mat_result_pkd( ring, c ) \
		( ring_is_packed( ring ) && (c == 0 || mat_is_packed( c )) )

#define mat_create_unpkd( ring, a, aprime, m, n ) \
	{ \
		if ( mat_is_packed( a ) ) \
		{ \
			mat_unpack( ring, a, &aprime ); \
		} \
		else \
		{ \
			aprime = a; \
		} \
	}

#define mat_create_unpkd_copy( ring, a, aprime, m, n ) \
	{ \
		if ( mat_is_packed( a ) ) \
		{ \
			mat_unpack( ring, a, &aprime ); \
		} \
		else \
		{ \
			integer_small	i_, *ptr_a_, *ptr_ap_; \
			if ( aprime == 0 ) \
			{ \
				aprime = mat_buff_alloc( m, n ); \
			} \
			ptr_a_ = mat_elt0_ptr( a ); \
			ptr_ap_ = mat_elt0_ptr( aprime ); \
			for ( i_=1; i_<=(m)*(n); ++i_ ) \
				ptr_ap_[i_] = ptr_a_[i_]; \
			mat_incref_entries( ring, aprime ); \
		} \
	}

#define mat_free_unpkd( a, aprime ) \
	{ \
		if ( a != aprime ) \
		{ \
			mat_buff_free( &(aprime) ); \
		} \
	}

/*
** Extension of matrix to new number of rows.
*/

#define mat_row_extend( h, nr ) \
	((void)((( mem_words( h ) <= mat_req_size( nr, mat_col(h)))? \
			(mem_extend_words_zero( h, mat_req_size( nr, mat_col(h)))) : 0) \
			, (mat_row( h ) = nr)))

/*
** Conversion to old memory management system.
*/

#define mat_stk_to_mat( cr, sn, it, pm, r, c ) \
		mat_stk_to_blk( cr, sn, it, pm, r, c )

/* Hack for excluding print_dmat in Kant Black box */
#define print_matrix(a, b, c)
 

#endif /* MAT_INTERNALS_INCLUDED */

