//
// LiDIA - a library for computational number theory
//   Copyright (c) 1995 by the LiDIA Group
//
// File        : bf_lattice.c 
// Author      : Werner Backes (WB), Thorsten Lauer (TL) 
// Last change : WB/TL, Feb 10 1995, initial version
//

#include <LiDIA/bigfloat_lattice.h>

// 
// variables for temporary use
//

// 
// For storing temporary values
//
bigfloat bigfloat_lattice::tempvz;
bigfloat bigfloat_lattice::tempmz0;
bigfloat bigfloat_lattice::tempmz1;
bigfloat bigfloat_lattice::tempmz2;
bigfloat bigfloat_lattice::tempmz3;
bigfloat bigfloat_lattice::tempmz4;
bigfloat bigfloat_lattice::ergmz;

//
// vector - size for arithmetic operations
//
long bigfloat_lattice::vectsize;

//
// vectors for temporary use
//
long bigfloat_lattice::tempvectsize;
bigfloat* bigfloat_lattice::tempvect0;
bigfloat* bigfloat_lattice::tempvect1;

//
// number of class bigfloat_lattice
// (the last one frees tempvect)
//
long bigfloat_lattice::last;

// 
// Constructor / Destructor
//

//
// The presentation of the lattice is optimized for exchanging 
// columns. The problem is that it takes more time exchanging 
// rows. Changing of columns is only changing of pointers  
// The constructor creates a static vectors with dynamic size.
// The last destructor that is called frees this vector.
//

//
// Simple constructor with dimension (m, n) 
// rows and columns 
// 
bigfloat_lattice::bigfloat_lattice(long m, long n)
{
  debug_handler("bigfloat_lattice","bigfloat_lattice(m, n)");
  last++;
  rows=m;
  cols=n;
  cols_rank=n;
  value=new bigfloat*[cols];
  memory_handler(value, "bigfloat_lattice","bigfloat_lattice(m, n) :: "
                        "Not enough memory !");
  for (register long i=0;i<cols;i++)
    {
      value[i]=new bigfloat[rows];
      memory_handler(value[i], "bigfloat_lattice","bigfloat_lattice(m, n) :: "
                               "Not enough memory !");
    }
  if (tempvectsize < rows)
    {
      if (tempvectsize != 0)
        {
          delete[] tempvect0;
          delete[] tempvect1;
        }
      tempvectsize=rows;
      tempvect0=new bigfloat[tempvectsize];
      memory_handler(tempvect0, "bigfloat_lattice","bigfloat_lattice(m, n) :: "
                                "Not enough memory !");
      tempvect1=new bigfloat[tempvectsize];
      memory_handler(tempvect1, "bigfloat_lattice","bigfloat_lattice(m, n) :: "
                                "Not enough memory !");
    }
}

//
// Constructor with dimension (m, n) 
// rows and columns and values bigfloat **abf
// prec is computed as quotient of bit length and log(10),  + 1
// 
bigfloat_lattice::bigfloat_lattice(long m, long n, bigfloat **abf)
{
  debug_handler("bigfloat_lattice","bigfloat_lattice(m, n, abf)");
  last++;
  rows=m;
  cols=n;
  cols_rank=n;
  value=new bigfloat*[cols];
  memory_handler(value, "bigfloat_lattice","bigfloat_lattice(m, n, abf) :: "
                        "Not enough memory !");
  for (register long i=0;i<cols;i++)
    {
      value[i]=new bigfloat[rows];
      memory_handler(value[i], "bigfloat_lattice","bigfloat_lattice(m, n, abf) :: "
                               "Not enough memory !");
      for (register long j=0;j<rows;j++)
        value[i][j].assign(abf[j][i]);
    }
  if (tempvectsize < rows)
    {
      if (tempvectsize != 0)
        {
          delete[] tempvect0;
          delete[] tempvect1;
        }
      tempvectsize=rows;
      tempvect0=new bigfloat[tempvectsize];
      memory_handler(tempvect0, "bigfloat_lattice","bigfloat_lattice(m, n, abf) :: "
                                "Not enough memory !");
      tempvect1=new bigfloat[tempvectsize];
      memory_handler(tempvect1, "bigfloat_lattice","bigfloat_lattice(m, n) :: "
                                "Not enough memory !");
    }
}

//
// Copy - constructor 
// creating a copy of bigfloat_lattice A
//
bigfloat_lattice::bigfloat_lattice(const bigfloat_lattice& A)
{
  debug_handler("bigfloat_lattice","bigfloat_lattice(A)");
  long i,j;
  last++;
  rows=A.rows;
  cols=A.cols;
  cols_rank=A.cols_rank;
  value=new bigfloat*[cols];
  memory_handler(value, "bigfloat_lattice","bigfloat_lattice(A) :: "
                        "Not enough memory !");
   for (i=0;i<cols;i++)
    {
      value[i]=new bigfloat[rows];
      memory_handler(value[i], "bigfloat_lattice","bigfloat_lattice(A) :: "
                               "Not enough memory !");
      for (j=0;j<rows;j++)
        (value[i][j]).assign(A.value[i][j]);
    }  
  if (tempvectsize < rows)
    {
      if (tempvectsize != 0)
        {
          delete[] tempvect0;
          delete[] tempvect1;
        }
      tempvectsize=rows;
      tempvect0=new bigfloat[tempvectsize];
      memory_handler(tempvect0, "bigfloat_lattice","bigfloat_lattice(A) :: "
                                "Not enough memory !");
      tempvect1=new bigfloat[tempvectsize];
      memory_handler(tempvect1, "bigfloat_lattice","bigfloat_lattice(A) :: "
                                "Not enough memory !");
     }
}
    
//
// The destructor that frees allocated storage
//
bigfloat_lattice::~bigfloat_lattice()
{
  debug_handler("bigfloat_lattice","~bigfloat_lattice()");
  while (cols--)
    delete[] value[cols];
  delete[] value;
  last--;
  if (last == 0)
   {
     tempvectsize=0;
     delete[] tempvect0;
     delete[] tempvect1;
   }
}

//
// Input / Output
//
// The lattice is read row by row, but the presentation
// of the lattice is column by column.
// The readed precision is set. 
//
istream& operator >> (istream& in, bigfloat_lattice& A)
{
  debug_handler("bigfloat_lattice","operator >> (in, A)");
  char st[200000];

  for (register long x=0;x<A.rows;++x)
    for (register long y=0;y<A.cols;++y)
      {
        in >> st;
        string_to_bigfloat(st, A.value[y][x]);
      }
  return (in);
}

//
// The lattice is written row by row to ostream 
//
ostream& operator << (ostream& out, const bigfloat_lattice& A)
{
  debug_handler("bigfloat_lattice","operator << (out, A)");
  long i,j;
  for (i=0;i<A.rows;++i)
    {   
      out << "(";
      for (j=0;j<A.cols-1;++j)
        out << A.value[j][i] << ", ";
      out << A.value[A.cols-1][i] << ")\n";
    }
  return (out);
}

//
// Assignment of two bigfloat_lattice
//
void bigfloat_lattice::assign(const bigfloat_lattice& A)
{
  debug_handler("bigfloat_lattice","assign(A)");
  long i,j;
  if ((cols == A.cols) && (rows == A.rows))
    {
      cols_rank=A.cols_rank;
      for (i=0;i<cols;i++)
        for (j=0;j<rows;j++)
          value[i][j].assign(A.value[i][j]);
      return;
    }
  lidia_error_handler("bigfloat_lattice","assign(A) :: "
                "tryed to assign two different sized lattice");
}

//
// Fills lattice with zero
//
void bigfloat_lattice::assign_zero()
{
  debug_handler("bigfloat_lattice","assign_zero()");
  long i,j;
  for (i=0;i<cols;i++)
    for (j=0;j<rows;j++)
      value[i][j].assign_zero();
}

//
// Overloaded operator, returns bigint_lattice
//
bigfloat_lattice bigfloat_lattice::operator = (const bigfloat_lattice& A)
{
  debug_handler("bigfloat_lattice","operator = (A)");
  assign(A);
  return (*this);
}

#ifndef __GNUC__

//
// Vector Operations
//

void assign(bigfloat* a, bigfloat* b)
{
  debug_handler("bigfloat_lattice","friend void assign(a, b)");
  long i;
  for (i=bigfloat_lattice::vectsize-1;i>=0;i--)
    a[i].assign(b[i]);
}

void assign_zero(bigfloat* a)
{
  debug_handler("bigfloat_lattice","friend void assign_zero(a)");
  long i;
  for (i=bigfloat_lattice::vectsize-1;i>=0;i--)
    a[i].assign_zero();
}

void add(bigfloat* c, bigfloat* a, bigfloat* b)
{
  debug_handler("bigfloat_lattice","friend void add(c, a, b)");
  long i;
  for (i=bigfloat_lattice::vectsize-1;i>=0;i--)
    add(c[i],a[i],b[i]);
}

void subtract(bigfloat* c, bigfloat* a, bigfloat* b)
{
  debug_handler("bigfloat_lattice","friend void subtract(c, a, b)");
  long i;
  for (i=bigfloat_lattice::vectsize-1;i>=0;i--)
    subtract(c[i],a[i],b[i]);
}

void scalmul(bigfloat* b, const bigfloat& d, bigfloat* a)
{
  debug_handler("bigfloat_lattice","friend scalmul(b, d, a)");
  long i;
  for (i=bigfloat_lattice::vectsize-1;i>=0;i--)
    multiply(b[i], a[i], d);
}

void scalprod(bigfloat& res, bigfloat* a, bigfloat* b)
{
  debug_handler("bigfloat_lattice","friend scalprod(res, a, b)");
  long i;
  res.assign_zero();
  for (i=bigfloat_lattice::vectsize-1;i>=0;i--) 
    {
      multiply(bigfloat_lattice::tempvz, a[i], b[i]);
      add(res, res, bigfloat_lattice::tempvz);
    }
}

void l2_norm(bigfloat& norm, bigfloat* v)
{
  debug_handler("bigfloat_lattice","l2_norm(norm, v)");
  long i;
  norm.assign_zero();		// Initialisation
  for (i=bigfloat_lattice::vectsize-1;i>=0;i--) 
    {
      square(bigfloat_lattice::tempvz,v[i]);
      add(norm,norm,bigfloat_lattice::tempvz);
    }
  sqrt(norm, norm);
}

void l1_norm(bigfloat& norm, bigfloat* v)
{
  debug_handler("bigfloat_lattice","l1_norm(norm, v)");
  long i;
  norm.assign_zero();	 // Initialisation
  for (i=bigfloat_lattice::vectsize-1;i>=0;i--) 
    {
      bigfloat_lattice::tempvz.assign(abs(v[i]));
      add(norm,norm,bigfloat_lattice::tempvz);
    }
}

#endif

//
// Algorithm - Subroutines
//

void bigfloat_lattice::gamma_compute(bigfloat& g, long l)
{
  debug_handler("bigfloat_lattice","gamma_compute(g, l)");
// Computation of the l-th Hermite - Constant gamma,
  switch(l)
    {
      case 1 : {
        g.assign_one(); 
      } break;
      case 2: {
        g.assign(4.0/3.0); 
      } break;
      case 3: { 
        g.assign(2); 
      } break;
      case 4: {
        g.assign(4); 
      } break;
      case 5: {
        g.assign(8); 
      } break;
      case 6: {
        g.assign(64.0/3.0); 
      } break;
      case 7: {
        g.assign(64); 
      } break;
      case 8: { 
        g.assign(256); 
      } break;
      default: {
        tempvz.assign(0.75);
        log(tempvz, tempvz);
        g.assign((long )(l * (l-1)));
        g.divide_by_2();
        multiply(tempvz, tempvz, g);
        exp(g, tempvz); 
      } break;
    }
}

void bigfloat_lattice::alpha_compute(bigfloat& alpha)
{
// alpha = max{vect[1].l2_norm(),...,vect[rows].l2_norm()}
// norm = vect[i].l2_norm(), 0 <= i < rows
  debug_handler("bigfloat_lattice","alpha_compute(alpha)");
  long i;
  vectsize=rows;
  l2_norm(alpha,value[0]);
  for (i=1;i<cols;++i)
    {
      l2_norm(tempmz0,value[i]);
      if (alpha.compare(tempmz0) < 0)
	alpha.assign(tempmz0);
    }
}

void bigfloat_lattice::zwei_pot_q_compute(bigfloat& zweipotq, long& n2, bigfloat& alpha)
{
  debug_handler("bigfloat_lattice","zwei_pot_q_compute(zwpq, n2, alpha)");
  long beta = cols;

// Computation of beta = min (A.rows, A.cols)
  if (rows < cols)
    beta=rows;
 
  tempmz0.assign(n2);
  sqrt(tempmz0,tempmz0);
  tempmz0.divide_by_2();
  multiply(tempmz0,cols,tempmz0);
  tempmz1.assign(cols);
  sqrt(tempmz1,tempmz1);
  add(tempmz0,tempmz0,tempmz1);
//  add(lambda,tempmz0,tempmz1);
 
  log(tempmz1,alpha);
  multiply(tempmz1,tempmz1,beta);
  exp(tempmz1,tempmz1);
// tempmz1 == potdummy
  gamma_compute(tempmz2 ,beta);
// gamma_compute(gammadummy ,beta);
 
  sqrt(tempmz2, tempmz2);
  multiply(tempmz2, tempmz2, tempmz1);
  multiply(tempmz2, tempmz2, tempmz0);
  tempmz0.assign(n2);
  multiply(tempmz0, tempmz0, cols);
  sqrt(tempmz0, tempmz0);
  add(tempmz0, tempmz0, 2);
  multiply(zweipotq, tempmz0, tempmz2);
  
  tempmz0.assign(beta + cols);
  tempmz0.divide_by_2();
  subtract(tempmz0, tempmz0, 1);
  tempmz0.multiply_by_2();
  exp(tempmz0, tempmz0);
  multiply(tempmz0, zweipotq, tempmz0);
 
  tempmz2.assign(beta + 1);
  tempmz2.divide_by_2();
  tempmz1.assign(rows);
  log(tempmz1,tempmz1);
  multiply(tempmz2, tempmz2, tempmz1);
  exp(tempmz2, tempmz2);
  divide(tempmz0, tempmz0, tempmz2); 
  ceil(zweipotq, tempmz0);
}

//
// Algorithms
//

//
// Schnorr - Euchner version of lll optimized for bigfloats
// result : lll - reduced lattice for parameter y
//
void bigfloat_lattice::lll(bigfloat_lattice& my, const bigfloat& schranke, double y, 
                           unsigned long& red_count)
{
  debug_handler("bigfloat_lattice", "lll(my, sch, y, prec, red_count) [1]");
  long i,j;
  long k = 1;
  red_count=0;
  vectsize=rows;
  scalprod(tempvect0[0],value[0], value[0]);
  bool Fc;
  bigfloat halb(0.5);
  while (k < cols)
    {
      Fc = true;
      while (Fc)
      {	
//
// begin of orthogonalization
//
        debug_handler("bigfloat_lattice", "lll(my, sch, y, prec, red_count) [2]");
        scalprod(tempvect0[k], value[k], value[k]);
        for (j = 0; j < k; ++j)
          {
            scalprod(my.value[k][j], value[k], value[j]);
            for (i = 0; i < j; ++i) 
              {
                multiply(tempmz0, my.value[j][i], my.value[k][i]);
                multiply(tempmz0, tempmz0, tempvect0[i]);
 	        subtract(my.value[k][j], my.value[k][j], tempmz0);
              }
     
            tempmz0.assign(my.value[k][j]);
            divide(my.value[k][j], my.value[k][j], tempvect0[j]);
            multiply(tempmz0, tempmz0, my.value[k][j]);    
            subtract(tempvect0[k], tempvect0[k], tempmz0);
          }
//
// end of orthogonalization
//
        Fc = false;
//
// begin of reduction
//
        debug_handler("bigfloat_lattice", "lll(my, sch, y, prec, red_count) [3]");
        for (j = k-1; j >= 0; j--)
          if (abs(my.value[k][j]).compare(halb) > 0)
            {
              ++red_count;
              round(tempmz0, my.value[k][j]);
              tempmz1.assign(abs(tempmz0));
	      if (tempmz1.compare(schranke) > 0)
	        Fc = true;

	      for (i = 0; i < j; ++i) 
                {
	          multiply(tempmz1, tempmz0, my.value[j][i]);
                  subtract(my.value[k][i], my.value[k][i], tempmz1);
                }

              subtract(my.value[k][j], my.value[k][j], tempmz0);
              scalmul(tempvect1, tempmz0, value[j]);
              subtract(value[k], value[k], tempvect1);
            }
//
// end of reduction
//
	if (Fc)
	  if (k > 1) 
	    --k;
      }
  
    debug_handler("bigfloat_lattice", "lll(my, sch, y, prec, red_count) [4]");
    square(tempmz1, my.value[k][k-1]); 
    multiply(tempmz1, tempmz1, tempvect0[k-1]);
    add(tempmz1, tempmz1, tempvect0[k]);
    tempmz0.assign(y);
    multiply(tempmz0,tempvect0[k-1], tempmz0); 
    if (tempmz0.compare(tempmz1) > 0)              // (y * tempmv[0].p[k-1] > tempmz1)
      {
	swap(value[k], value[k-1]);
	if (k > 1)
	  --k;
        if(k == 1) scalprod(tempvect0[0], value[0], value[0]);
      }
    else
      ++k;
  }
}   

//
// Schnorr - Euchner version of lll optimized for bigfloats
// result : transformation lattice
//
void bigfloat_lattice::lll_trans(bigfloat_lattice& T, bigfloat_lattice& my, const bigfloat& schranke, 
                                 double y, unsigned long& red_count)
{
  debug_handler("bigfloat_lattice", "lll_trans(T, my, sch, y, prec, red_count) [1]");
  long i, j;
  long k = 1;
  vectsize=rows;
  scalprod(tempvect0[0],value[0], value[0]);
  bool Fc;
  while (k < cols)
    {
      Fc = true;
      while (Fc)
      {	
//
// begin of orthogonalization
//
        debug_handler("bigfloat_lattice", "lll_trans(T, my, sch, y, prec, red_count) [2]");
        scalprod(tempvect0[k], value[k], value[k]);
        for (j = 0; j < k; ++j)
          { 
            scalprod(my.value[k][j], value[k], value[j]);
            for (i = 0; i < j; ++i) 
              {
                multiply(tempmz0, my.value[j][i], my.value[k][i]);
                multiply(tempmz0, tempmz0, tempvect0[i]);
	        subtract(my.value[k][j], my.value[k][j], tempmz0);
              }
     
            tempmz0.assign(my.value[k][j]);
            divide(my.value[k][j], my.value[k][j], tempvect0[j]);
            multiply(tempmz0, tempmz0, my.value[k][j]);    
            subtract(tempvect0[k], tempvect0[k], tempmz0);
          }
//
// end of orthogonalization
//
        Fc = false;
//
// begin of reduction
//
        debug_handler("bigfloat_lattice", "lll_trans(T, my, sch, y, prec, red_count) [3]");
        for (j = k-1; j >= 0; j--)
          if (abs(my.value[k][j]).compare(bigfloat(0.5)) > 0)
            {
              ++red_count;
              round(tempmz0, my.value[k][j]);
              tempmz1.assign(abs(tempmz0));
	      if (tempmz1.compare(schranke) > 0)
	        Fc = true;

	      for (i = 0; i < j; ++i) 
                {
	          multiply(tempmz1, tempmz0, my.value[j][i]);
                  subtract(my.value[k][i], my.value[k][i], tempmz1);
                }

              subtract(my.value[k][j], my.value[k][j], tempmz0);
              scalmul(tempvect1, tempmz0, value[j]);
              subtract(value[k], value[k], tempvect1);
              vectsize=cols;
              scalmul(tempvect1, tempmz0, T.value[j]);
              subtract(T.value[k], T.value[k], tempvect1);
              vectsize=rows;
            }
//
// end of reduction
//
	if (Fc)
	  if (k > 1) 
	    --k;
      }
  
    debug_handler("bigfloat_lattice", "lll_trans(T, my, sch, y, prec, red_count) [4]");
    square(tempmz1, my.value[k][k-1]); 
    multiply(tempmz1, tempmz1, tempvect0[k-1]);
    add(tempmz1, tempmz1, tempvect0[k]);
    tempmz2.assign(y);
    multiply(tempmz2,tempvect0[k-1], tempmz2); 
    if (tempmz2.compare(tempmz1) > 0)              // (y * tempmv[0].p[k-1] > tempmz1)
      {
	swap(value[k], value[k-1]);
        swap(T.value[k], T.value[k-1]);
	if (k > 1)
	  --k;
        if(k == 1) scalprod(tempvect0[0], value[0], value[0]);
      }
    else
      ++k;
  }
}   

//
// Buchmann Kessler version of lll for generating systems
// result : transformation lattice
//
void bigfloat_lattice::lin_gen_system(bigfloat_lattice& T, long prec, double y, 
                                      unsigned long& rc, timer& ti, long& rk)
{
  debug_handler("bigfloat_lattice","lin_gen_system(T, prec, y, rc, ti, rk) [1]");

  bigfloat_lattice Atilde(rows+cols,cols),	// The approximate lattice
                   Ahead(rows, cols);		// The weighted lll_matrix_bigfloat
  bigfloat *help=new bigfloat[rows],
           *rel=new bigfloat[cols],
           *temp=new bigfloat[rows];

  bigfloat zweipotq,
	   norm1,		// To store the l1_norm
           norm2,		// To store the l2_norm
           vor,
           rechts,
           alpha;
  long n2 = rows;
  long i,j;

  zweipotq.precision(prec);
  alpha_compute(alpha);
  zwei_pot_q_compute(zweipotq, n2, alpha);
  for (i=0;i<cols;++i)
    for (j=0;j< rows;++j)
      {
        multiply(tempmz0,zweipotq,value[i][j]);
        round(Ahead.value[i][j], tempmz0);
      }
  for (i=0;i<cols;++i)
    {
      Atilde.value[i][i].assign_one();  // 1, wenn i = j, 0 sonst
      for (j=cols;j<(cols+rows);++j)
        Atilde.value[i][j].assign(Ahead.value[i][j-cols]);
    }
  tempmz0.assign(prec);
  tempmz0.divide_by_2();
  tempmz1.assign(10);
  log(tempmz1,tempmz1);
  multiply(tempmz0,tempmz0,tempmz1);
  exp(ergmz,tempmz0);   

  debug_handler("bigfloat_lattice","lin_gen_system(T, prec, y, rc, ti, rk) [2]");
  ti.start_timer();
  Atilde.lll(T,ergmz,y,rc);
  ti.stop_timer();
  debug_handler("bigfloat_lattice","lin_gen_system(T, prec, y, rc, ti, rk) [3]");

  long l = 0;
  do
    {
      vectsize=rows;
      ::assign_zero(help);	// Initializes help with the zero - vector
       for (long j = 0; j < cols; ++j)
        {
          rel[j].assign(Atilde.value[l][j]);
          scalmul(temp,rel[j], Ahead.value[j]);
          add(help,help,temp);
        }
      l2_norm(norm2,help);
      vectsize=cols;
      l1_norm(norm1,rel);
      ++l;
      divide(vor, n2, rows);
      vor.multiply_by_2();
      sqrt(vor, vor);
      multiply(vor, vor, norm1);
      divide(rechts, (bigfloat) sqrt(n2), (bigfloat )2.0);
      multiply(rechts, rechts, norm1);
    } 
  while ((zweipotq > vor) && (norm2 <= rechts) && (l < cols));
  debug_handler("bigfloat_lattice","lin_gen_system(T, prec, y, rc, ti, rk) [4]");

  if (l >= cols)
    warning_handler("bigfloat_lattice","lin_gen_system(T, prec, y, rc, ti, rank) :: "
                    "lattice of dimension 1");

  T.cols_rank = cols - l + 1;	// r is the dimension of the lattice
  rk=T.cols_rank;
  for (i=0;i<cols-T.cols_rank;i++)
    for (j=0;j<cols;j++)
      T.value[i][j].assign_zero();    
  for (i=cols-T.cols_rank;i<cols;++i)
    for (j=0;j<cols;++j)
      T.value[i][j].assign(Atilde.value[i][j]);
//
// Freee allocated storage
//
  delete[] help;
  delete[] rel;
  delete[] temp;
} 

//
// Computes lll - reduced basis by multiplikation of A and T 
// T transformation lattice
//
void bigfloat_lattice::compute_basis(const bigfloat_lattice& A,
                                     const bigfloat_lattice& T)
{
  debug_handler("bigfloat_lattice","compute_basis(A, T)");
  bigfloat_lattice temp(A.rows, A.cols);
  long i, j, cx;
// Computing the basis - vectors
  
  for (i=0;i<A.rows;++i)
    for (j=0;j<A.cols;++j) 
      for (cx=0;cx<A.cols;cx++)
        {
          multiply(tempmz0, A.value[cx][i], T.value[j][cx]);
          add(temp.value[j][i],temp.value[j][i],tempmz0);
        }
  assign(temp);
}

//
// Gram - Schmidt - Orthogonalisation for lattice basis
// returns 0 if lattice is no basis 1 otherwise
// 
long bigfloat_lattice::gram_schmidt_orth(bigfloat_lattice& gso, bigfloat_lattice& my, long prec)
{
  debug_handler("bigfloat_lattice","gram_schmidt_orth(gso, my, prec)");
  long i,j;
  
  vectsize=rows;
  tempvect0[0].precision(prec);

  for (i=0;i<cols;i++) 
    {
      ::assign(gso.value[i], value[i]);
      for(j=0;j<i;j++) 
        {
          scalmul(tempvect0, my.value[j][i], gso.value[j]);
          ::subtract(gso.value[i], gso.value[i], tempvect0);
        }
        
      scalprod(tempvect1[i], gso.value[i], gso.value[i]);    
      for (j=0;j<i;j++) 
        {
          scalprod(tempmz0, value[i], gso.value[j]);
          if (tempvect1[j].is_approx_zero())
            return (0);
          divide(my.value[j][i], tempmz0, tempvect1[j]);
        }
    }
  return (1);
}

//
// check if reduction is good enough for parameter y 
//
int bigfloat_lattice::lll_check(double y, long prec)
{
  debug_handler("bigfloat_lattice","lll_check(y, prec)");
  bigfloat_lattice my(cols,cols);
  bigfloat_lattice gso(rows, cols);
  long i, j;

  if ((gram_schmidt_orth(gso,my, prec)) == 0) 
    lidia_error_handler("bigfloat_lattice","lll_check(y, prec) :: "
                  "lattice is no basis !");
  for (i=0;i<cols;i++)
    for (j=0;j<i;j++)
      if (abs(my.value[j][i]).compare(bigfloat(0.5)) > 0)
        return(0);
  vectsize=rows;
  for (i=1;i<cols;i++)
    {
      scalmul(tempvect0, my.value[i][i-1], gso.value[i-1]);
      ::add(tempvect0,tempvect0,gso.value[i]); 
      scalprod(tempmz0, tempvect0, tempvect0);
      scalprod(tempmz1, gso.value[i-1], gso.value[i-1]);
      tempmz2.assign(y);
      multiply(tempmz2, tempmz2, tempmz1);
      if (tempmz0.compare(tempmz2) < 0)
        return(0);
    } 
  return(1);
}

// 
// Searches paramter the lattice is reduced for
// returns 0 by non - reduction
//
double bigfloat_lattice::lll_check(long prec)
{
  debug_handler("bigfloat_lattice","lll_check(prec)");
  bigfloat_lattice my(cols,cols);
  bigfloat_lattice gso(rows, cols);
  double y=(1.0 + 0.25)/2;
  double old_y = 0;

  while((fabs(old_y-y) > 0.001) && (old_y < 1.0)) {
    if(lll_check(y, prec)) {
      old_y = y;
      y = y + 0.2;
    }
    else y = old_y + (y - old_y) / 2;
  };

  return ((old_y>1.0)?1.0:old_y);
}

void bigfloat_lattice::cholesky(bigfloat& det)
{
  debug_handler("bigfloat_lattice","cholesky(det)");
  bigfloat_lattice Q(cols, cols),   //Q = (<Q[j], Q[i]>) the Fmatrix of the inner products.
                                    // <.,.> denotes the ordinary inner product
		                    // Q is stored only up to the diagonal elements because
		                    // it 's a symmetric matrix
                                    // Q wird als obere Dreiecksmatrix abgespeichert.
                   A(cols, cols);   // Cholesky - matrix
                                    // Matrix fuer die Cholesky - Zerlegung.
  long i, j, l;
  
  det = 1;	//Initialisation
  vectsize=rows;
  for (i=0;i<cols;++i)
    {
      for (j=0;j<=i;++j)
        scalprod(Q.value[i][j],value[j], value[i]);

      for (j=0;j<i;++j)
        {
          A.value[i][j].assign(Q.value[i][j]/A.value[j][j]);
          for (l=j+1;l<=i;++l) 
            {
              multiply(tempmz0, A.value[l][j], A.value[i][j]);
              multiply(tempmz0, tempmz0, A.value[j][j]);
              subtract(Q.value[i][l], Q.value[i][l], tempmz0);
            }
        }
      A.value[i][j].assign(Q.value[i][i]);

      if (A.value[i][i].is_zero())
        { 
          for (l=i;l<cols-1;++l)
            ::assign(value[l],value[l+1]);
          cols--;
          --i;		// L.vect[i] has become new values, so it must be
                        // reregarded
        }
      else
         multiply(det, det, A.value[i][i]);
      // Now A is the Cholesky - matrix
    }
  det.assign(abs(det));	// The determinant of the lattice is |det(A)|
  sqrt(det, det);	        // => det must have a positive value
}

// Tests if the resulting vectors are short enough. When using this function 
// there is premised that the resulting vectors form a basis of the lattice
//
// Literatur: Johannes Buchmann and Volker Kessler: "Computing a reduced 
//	      lattice basis from a generating system", Corollary 4.2
//
// The return - code is
//	1 if the vectors are short enough
//	0 if there is at least one vector which is to long

int bigfloat_lattice::is_reduced()
{
  debug_handler("bigfloat_lattice","is_reduced()");
  long i;

  cholesky(ergmz);

  tempmz0.assign(log(ergmz));
  tempmz1.assign(cols_rank);
  divide(tempmz0,tempmz0,tempmz1);
  exp(tempmz0,tempmz0);
//  bound = exp(log(det) / rank);   // bound = d(L)^(1/r)	
  
  tempmz1.assign(rows+2);
  multiply(tempmz0,tempmz0,tempmz1);
//  bound = bound * (rows + 2);	// bound = (rows + 2) * d(L)^(1/r)

  tempmz1.assign(cols_rank-1);
  tempmz1.divide_by_2();
  tempmz2.assign(2);
  log(tempmz2,tempmz2);
  multiply(tempmz2,tempmz2,tempmz1);
  exp(tempmz2,tempmz2);
//  bounddummy = exp(log(2.0) * (rank - 1)/2.0);	// bounddummy = 2^((r-1)/2)

  multiply(tempmz0,tempmz0,tempmz2);
//  bound = bound * bounddummy;
  vectsize=rows;
  l2_norm(tempmz4, value[0]);
  if (tempmz4.compare(tempmz0) > 0)
    return(0);

  tempmz0.assign(rows+2);
  log(tempmz0, tempmz0);
  tempmz1.assign(cols_rank);
  multiply(tempmz1, tempmz1, tempmz0);
  exp(tempmz1, tempmz1);
//  bound = exp(rank * log(rows + 2));	// bound = (B.rows + 2)^r

  tempmz0.assign(cols_rank-1);
  tempmz0.divide_by_2();
  tempmz2.assign(cols_rank);
  multiply(tempmz2,tempmz2,tempmz0);
  tempmz0.assign(2);
  log(tempmz0,tempmz0);
  multiply(tempmz0, tempmz0, tempmz2);
  exp(tempmz0,tempmz0);
//  bounddummy = exp(log(2.0) * rank * (rank - 1)/2.0);	// bounddummy = 2^(r * (r - 1)/2)

  multiply(tempmz2,ergmz,tempmz0);
  multiply(tempmz2,tempmz2,tempmz1);
//  bound = bound * bounddummy * det;

  tempmz3.assign_one();
  for (i=0;i<cols_rank;++i)
    {
     l2_norm(tempmz4, value[i]);
     multiply(tempmz3, tempmz3, tempmz4);	// prod = l2_norm(B.value[0])
	                                        //      * ...
		                                //	* l2_norm(B.value[r])
    }
  if (tempmz3.compare(tempmz2) > 0)
    return(0);
  return(1);
}

