//
// LiDIA - a library for computational number theory
//   Copyright (c) 1995 by the LiDIA Group
//
// File        : do_lattice.c 
// Author      : Werner Backes (WB), Thorsten Lauer (TL) 
// Last change : WB/TL, Feb 10 1995, initial version
//

#include <LiDIA/double_lattice.h>

// 
// variables for temporary use
//

// 
// For storing temporary values
//
double double_lattice::tempmz0;
double double_lattice::tempmz1;

//
// vector - size for arithmetic operations
//
long double_lattice::vectsize;

//
//vectors for temporary use
//
long double_lattice::tempvectsize;
double* double_lattice::tempvect0;
double* double_lattice::tempvect1;

//
// number of class double_lattice
// (the last one frees tempvect)
//
long double_lattice::last;

// 
// Constructor / Destructor
//

//
// The presentation of the lattice is optimized for exchanging 
// columns. The problem is that it takes more time exchanging 
// rows. Changing of columns is only changing of pointers  
// The constructor creates a static vectors with dynamic size.
// The last destructor that is called frees this vector.
//

//
// Simple constructor with dimension (m, n) 
// rows and columns 
// 
double_lattice::double_lattice(long m, long n)
{
  debug_handler("double_lattice","double_lattice(m, n)");
  long i;
  last++;
  rows=m;
  cols=n;
  cols_rank=n;
  value=new double*[cols];
  memory_handler(value, "double_lattice","double_lattice(m, n) :: "
                        "Not enough memory !");
  for (i=0;i<cols;i++)
    {
      value[i]=new double[rows];
      memory_handler(value[i], "double_lattice","double_lattice(m, n) :: "
                               "Not enough memory !");
    }
  if (tempvectsize < rows)
    {
      if (tempvectsize != 0)
        {
          delete[] tempvect0;
          delete[] tempvect1;
        }
      tempvectsize=rows;
      tempvect0=new double[tempvectsize];
      memory_handler(tempvect0, "double_lattice","double_lattice(m, n) :: "
                                "Not enough memory !");
      tempvect1=new double[tempvectsize];
      memory_handler(tempvect1, "double_lattice","double_lattice(m, n) :: "
                                "Not enough memory !");
    }
}

//
// Constructor with dimension (m, n) 
// rows and columns and values double **ado
// 
double_lattice::double_lattice(long m, long n, double **ado)
{
  debug_handler("double_lattice","double_lattice(m, n, ado)");
  long i, j;
  last++;
  rows=m;
  cols=n;
  cols_rank=n;
  value=new double*[cols];
  memory_handler(value, "double_lattice","double_lattice(m, n, ado) :: "
                        "Not enough memory !");
  for (i=0;i<cols;i++)
    {
      value[i]=new double[rows];
      memory_handler(value[i], "double_lattice","double_lattice(m, n, ado) :: "
                               "Not enough memory !");
      for (j=0;j<rows;j++)
        value[i][j]=ado[j][i];
    }
  if (tempvectsize < rows)
    {
      if (tempvectsize != 0)
        {
          delete[] tempvect0;
          delete[] tempvect1;
        }
      tempvectsize=rows;
      tempvect0=new double[tempvectsize];
      memory_handler(tempvect0, "double_lattice","double_lattice(m, n, ado) :: "
                                "Not enough memory !");
      tempvect1=new double[tempvectsize];
      memory_handler(tempvect1, "double_lattice","double_lattice(m, n, ado) :: "
                                "Not enough memory !");
    }
}

//
// Copy - constructor 
// creating a copy of double_lattice A
//
double_lattice::double_lattice(const double_lattice& A)
{
  debug_handler("double_lattice","double_lattice(A)");
  long i, j;
  last++;
  rows=A.rows;
  cols=A.cols;
  cols_rank=A.cols_rank;
  value=new double*[cols];
  memory_handler(value, "double_lattice","double_lattice(A) :: "
                        "Not enough memory !");
   for (i=0;i<cols;i++)
    {
      value[i]=new double[rows];
      memory_handler(value[i], "double_lattice","double_lattice(A) :: "
                               "Not enough memory !");
      for (j=0;j<rows;j++)
         value[i][j]=A.value[i][j];
    }  
  if (tempvectsize < rows)
    {
      if (tempvectsize != 0)
        {
          delete[] tempvect0;
          delete[] tempvect1;
        }
      tempvectsize=rows;
      tempvect0=new double[tempvectsize];
      memory_handler(tempvect0, "double_lattice","double_lattice(A) :: "
                                "Not enough memory !");
      tempvect1=new double[tempvectsize];
      memory_handler(tempvect1, "double_lattice","double_lattice(A) :: "
                                "Not enough memory !");
     }
}
    
//
// The destructor that frees allocated storage
//
double_lattice::~double_lattice()
{
  debug_handler("double_lattice","~double_lattice()");
  while (cols--)
    delete[] value[cols];
  delete[] value;
  last--;
  if (last == 0)
   {
     tempvectsize=0;
     delete[] tempvect0;
     delete[] tempvect1;
   }
}

//
// Input / Output
//
// The lattice is read row by row, but the presentation
// of the lattice is column by column.
// The readed precision is set. 
//
istream& operator >> (istream& in, double_lattice& A)
{
  debug_handler("double_lattice","operator >> (in, A)");
  long i, j;
  for (i=0;i<A.rows;++i)
    for (j=0;j<A.cols;++j)
      in >> A.value[j][i];
  return (in);
}

//
// The lattice is written row by row to ostream 
//
ostream& operator << (ostream& out, const double_lattice& A)
{
  debug_handler("double_lattice","operator << (out, A)");
  long i, j;
  for (i=0;i<A.cols;++i)
    {   
      out << "(";
      for (j=0;j<A.rows-1;++j)
        out << A.value[j][i] << ", ";
      out << A.value[A.rows-1][i] << ")\n";
    }
  return (out);
}

//
// Assignment of two double_lattice
//
void double_lattice::assign(const double_lattice& A)
{
  debug_handler("double_lattice","assign(A)");
  long i, j;
  if ((cols == A.cols) && (rows == A.rows))
    {
      cols_rank=A.cols_rank;
      for (i=0;i<cols;i++)
        for (j=0;j<rows;j++)
          value[i][j]=A.value[i][j];
      return;
    }
  lidia_error_handler("double_lattice","assign(A) :: "
                "tryed to assign two different sized lattice");
}

//
// Fills lattice with zero
//
void double_lattice::assign_zero()
{
  debug_handler("double_lattice","assign_zero()");
  long i, j;
  for (i=0;i<cols;i++)
    for (j=0;j<rows;j++)
      value[i][j]=0;
}

//
// Overloaded operator, returns bigint_lattice
//
double_lattice double_lattice::operator = (const double_lattice& A)
{
  debug_handler("double_lattice","operator = (A)");
  assign(A);
  return (*this);
}

#ifndef __GNUC__

//
// Vector Operations
//

void assign(double* a, double* b)
{
  debug_handler("double_lattice","friend void assign(a, b)");
  long i;
  for (i=double_lattice::vectsize-1;i>=0;i--)
    a[i]=b[i];
}

void assign_zero(double* a)
{
  debug_handler("double_lattice","friend void assign_zero(a)");
  long i;
  for (i=double_lattice::vectsize-1;i>=0;i--)
    a[i]=0;
}

void add(double* c, double* a, double* b)
{
  debug_handler("double_lattice","friend void add(c, a, b)");
  long i;
  for (i=double_lattice::vectsize-1;i>=0;i--)
    c[i]=a[i]+b[i];
}

void subtract(double* c, double* a, double* b)
{
  debug_handler("double_lattice","friend void subtract(c, a, b)");
  for(register long i=double_lattice::vectsize-1;i>=0;i--)
    c[i]=a[i]-b[i];
}

void scalmul(double* b, const double& d, double* a)
{
  debug_handler("double_lattice","friend scalmul(b, d, a)");
  long i;
  for (i=double_lattice::vectsize-1;i>=0;i--)
    b[i]=a[i]*d;
}


void scalprod(double& res, double* a, double* b)
{
  debug_handler("double_lattice","friend scalprod(res, a, b)");
  long i;
  res=0;
  for (i=double_lattice::vectsize-1;i>=0;i--)
    res+=a[i]*b[i];
}

void l2_norm(double& norm, double* v)
{
  debug_handler("double_lattice","l2_norm(norm, v)");
  long i;
  norm=0;		// Initialisation
  for (i=double_lattice::vectsize-1;i>=0;i--)
    norm+=v[i]*v[i];
  norm=sqrt(norm);
}

void l1_norm(double& norm, double* v)
{
  debug_handler("double_lattice","l1_norm(norm, v)");
  long i;
  norm=0;	 // Initialisation
  for (i=double_lattice::vectsize-1;i>=0;i--)
    norm+=fabs(v[i]);
}

#endif

//
// Algorithm - Subroutines
//

void double_lattice::gamma_compute(double& g, long l)
{
  debug_handler("double_lattice","gamma_compute(g, l)");

// Computation of the l-th Hermite - Constant gamma,
  switch(l)
    {
      case 1: {
        g = 1; 
      } break;
      case 2: {
        g = 4.0/3.0; 
      } break;
      case 3: {
        g = 2; 
      } break;
      case 4: { 
        g = 4; 
      } break;
      case 5: { 
        g = 8; 
      } break;
      case 6: {
        g = 64.0/3.0; 
      } break;
      case 7: {
        g = 64; 
      } break;
      case 8: {
        g = 256; 
      } break;
      default: {
        g=exp(((l*(l-1))/2.0)*log(4.0/3.0));
      } break;
    }
}

void double_lattice::alpha_compute(double& alpha)
{
// alpha = max.value[1].l2_norm(),....value[rows].l2_norm()}
// norm =.value[i].l2_norm(), 0 <= i < rows
  debug_handler("double_lattice","alpha_compute(alpha)");
  long i;
  vectsize=rows;
  l2_norm(alpha,value[0]);
  for (i=1;i<cols;++i)
    {
      l2_norm(tempmz0,value[i]);
      if (alpha < tempmz0)
	alpha=tempmz0;
    }
}

void double_lattice::zwei_pot_q_compute(double& zweipotq, long& n2, double& alpha)
{
  debug_handler("double_lattice","zwei_pot_q_compute(zwpq, n2, alpha)");
  long beta = cols;
  double potdummy,	// dummy - variable for computing powers
	 gammadummy;
  double lambda;     //lambda = rows * sqrt(n2)/2 + sqrt(cols)


   // Computation of beta = min (A.rows, A.cols)
  if (rows < cols)
    beta = rows;
 
  lambda= sqrt(cols)+cols*sqrt(n2)/2.0;

//  tempmz[0].assign(n2);
//  sqrt(tempmz[1],tempmz[0]);
//  tempmz[1].divide_by_2();
//  multiply(tempmz[2],cols,tempmz[1]);
//  tempmz[1].assign(cols);
//  sqrt(tempmz[0],tempmz[1]);
//  add(lambda,tempmz[2],tempmz[0]);
 
  potdummy=exp(log(alpha)*beta);

//  log(tempmz[0],alpha);
//  multiply(tempmz[1],tempmz[0],beta);
//  exp(potdummy,tempmz[1]);

  gamma_compute(gammadummy,beta);
 
  zweipotq=sqrt(gammadummy)*potdummy*lambda*(sqrt(n2*cols)+2);
  
//  sqrt(tempmz[0], gammadummy);
//  multiply(tempmz[1], tempmz[0], potdummy);
//  multiply(tempmz[2], tempmz[1], lambda);
//  tempmz[0].assign(n2);
//  multiply(tempmz[1], tempmz[0], cols);
//  sqrt(tempmz[0], tempmz[1]);
//  add(tempmz[1], tempmz[0], 2);
//  multiply(zweipotq, tempmz[1], tempmz[2]);
  

  zweipotq*=exp((((beta+cols)/2.0)-1)*2);

//  tempmz[0].assign(beta + cols);
//  tempmz[0].divide_by_2();
//  subtract(tempmz[1], tempmz[0], 1);
//  tempmz[0].assign(2);   
//  multiply(tempmz[2], tempmz[1], tempmz[0]);
//  exp(tempmz[0], tempmz[2]);
//  multiply(tempmz[2], zweipotq, tempmz[0]);
 
  zweipotq=ceil(zweipotq/exp((beta+1)/2.0*log(rows)));

//  tempmz[0].assign(beta + 1);
//  tempmz[0].divide_by_2();
//  tempmz[1].assign(rows);
//  log(tempmz[3],tempmz[1]);
//  multiply(tempmz[1], tempmz[0], tempmz[3]);
//  exp(potdummy, tempmz[1]);
//  divide(tempmz[1], tempmz[2], potdummy); 
//  ceil(zweipotq, tempmz[1]);
  
}

//
// Algorithms
//

//
// Schnorr - Euchner version of lll optimized for doubles
// result : lll - reduced lattice for parameter y
//
void double_lattice::lll(double_lattice& my, const double& schcols_ranke, double y, 
                           unsigned long& red_count)
{
  debug_handler("double_lattice", "lll(my, sch, y, prec, red_count) [1]");
  long i,j;
  long k = 1;
  vectsize=rows;
  scalprod(tempvect0[0],value[0], value[0]);
  bool Fc;
  while (k < cols)
    {
      Fc = true;
      while (Fc)
      {	
//
// begin of orthogonalization
//
        debug_handler("double_lattice", "lll(my, sch, y, prec, red_count) [2]");
        scalprod(tempvect0[k], value[k], value[k]);
        for (j=0;j<k;++j)
          {
            scalprod(my.value[k][j], value[k], value[j]);
            for (i=0;i<j;++i) 
              {
                my.value[k][j]-=my.value[j][i]*my.value[k][i]*tempvect0[i];
//                multiply(tempmz[0], my.value[j][i], my.value[k][i]);
//                multiply(tempmz[1], tempmz[0], tem.value[0][i]);
// 	          subtract(my.value[k][j], my.value[k][j], tempmz[1]);
              }
              tempmz0=my.value[k][j];     
              my.value[k][j]/=tempvect0[j];
              tempvect0[k]-=tempmz0*my.value[k][j];
              
//            tempmz[0].assign((my.value[k])[j]);
//            divide((my.value[k])[j], (my.value[k])[j], tem.value[0][j]);
//            multiply(tempmz[1], tempmz[0], (my.value[k])[j]);    
//            subtract(tem.value[0][k], tem.value[0][k], tempmz[1]);
          }
//
// end of orthogonalization
//
        Fc = false;
//
// begin of reduction
//
        debug_handler("double_lattice", "lll(my, sch, y, prec, red_count) [3]");
        for (j=k-1;j>=0;j--)
          if (fabs(my.value[k][j]) > 0.5)
            {
              ++red_count;
              tempmz0=rint(my.value[k][j]);
	      if (fabs(tempmz0) > schcols_ranke)
	        Fc = true;
 
	      for (i=0;i<j;++i) 
                my.value[k][i]-=tempmz0*my.value[j][i];

              my.value[k][j]-=tempmz0;
              scalmul(tempvect1, tempmz0, value[j]);
              subtract(value[k], value[k], tempvect1);
 
            }
//
// end of reduction
//
	if (Fc)
	  if (k > 1)
	    --k;
      }
  
    debug_handler("double_lattice", "lll(my, sch, y, prec, red_count) [4]");

//    square(tempmz[1], (my.value[k])[k-1]); 
//    multiply(tempmz[0], tempmz[1], tem.value[0][k-1]);
//    add(tempmz[1], tempmz[0], tem.value[0][k]);
//    tempmz[0].assign(y);
//    multiply(tempmz[2],tem.value[0][k-1], tempmz[0]); 

    if ((y*tempvect0[k-1]) >
       (my.value[k][k-1]*my.value[k][k-1]*tempvect0[k-1]+tempvect0[k]))
      {
	swap(value[k], value[k-1]);
	if (k > 1)
	  --k;
        if(k == 1) scalprod(tempvect0[0], value[0], value[0]);
      }
    else
      ++k;
  }
}   

//
// Schnorr - Euchner version of lll optimized for doubles
// result : transformation lattice
//
void double_lattice::lll_trans(double_lattice& T, double_lattice& my, const double& schcols_ranke, 
                               double y, unsigned long& red_count)
{
  debug_handler("double_lattice", "lll_trans(T, my, sch, y, prec, red_count) [1]");
  long i,j;
  long k = 1;
  vectsize=rows;
  scalprod(tempvect0[0],value[0], value[0]);
  bool Fc;
  while (k < cols)
    {
      Fc = true;
      while (Fc)
      {	
//
// begin of orthogonalization
//
        debug_handler("double_lattice", "lll_trans(T, my, sch, y, prec, red_count) [2]");
        scalprod(tempvect0[k], value[k], value[k]);
        for (j=0;j<k;++j)
          {
            scalprod(my.value[k][j], value[k], value[j]);
            for (i=0;i<j;++i) 
              {
                my.value[k][j]-=my.value[j][i]*my.value[k][i]*tempvect0[i];
//                multiply(tempmz[0], (my.value[j])[i], (my.value[k])[i]);
//                multiply(tempmz[1], tempmz[0], tem.value[0][i]);
// 	          subtract((my.value[k])[j], (my.value[k])[j], tempmz[1]);
              }
              tempmz0=my.value[k][j];     
              my.value[k][j]/=tempvect0[j];
              tempvect0[j]-=tempmz0*my.value[k][j];
              
//            tempmz[0].assign((my.value[k])[j]);
//            divide((my.value[k])[j], (my.value[k])[j], tem.value[0][j]);
//            multiply(tempmz[1], tempmz[0], (my.value[k])[j]);    
//            subtract(tem.value[0][k], tem.value[0][k], tempmz[1]);
          }
//
// end of orthogonalization
//
        Fc = false;
//
// begin of reduction
//
        debug_handler("double_lattice", "lll_trans(T, my, sch, y, prec, red_count) [3]");
        for (j=k-1;j>=0;j--)
          if (fabs(my.value[k][j]) > 0.5)
            {
              ++red_count;
              tempmz0=rint(my.value[k][j]);
	      if (fabs(tempmz0) > schcols_ranke)
	        Fc = true;

	      for (i=0;i<j;++i) 
                my.value[k][i]-=tempmz0*my.value[j][i];

              my.value[k][j]-=tempmz0;
              scalmul(tempvect1, tempmz0, value[j]);
              subtract(value[k], value[k], tempvect1);
	      vectsize=cols;
              scalmul(tempvect1, tempmz0, T.value[j]);
              subtract(T.value[k], T.value[k], tempvect1);
	      vectsize=rows;

//              ++red_count;
//              round(tempmz[0], (my.value[k])[j]);
//              tempmz[1].assign(abs(tempmz[0]));
//	        if (tempmz[1].compare(schcols_ranke) > 0)
//	          Fc = true;
//
//	        for (i = 0; i < j; ++i) 
//                {
//	            multiply(tempmz[1], tempmz[0], (my.value[j])[i]);
//                  subtract((my.value[k])[i], (my.value[k])[i], tempmz[1]);
//                }
//
//              subtract((my.value[k])[j], (my.value[k])[j], tempmz[0]);
//              scalmul(tem.value[1], tempmz[0], value[j]);
//              subtract(value[k], value[k], tem.value[1]);

            }
//
// end of reduction
//
	if (Fc)
	  if (k > 1) 
	    --k;
      }
  
    debug_handler("double_lattice", "lll_trans(T, my, sch, y, prec, red_count) [4]");

//    square(tempmz[1], (my.value[k])[k-1]); 
//    multiply(tempmz[0], tempmz[1], tem.value[0][k-1]);
//    add(tempmz[1], tempmz[0], tem.value[0][k]);
//    tempmz[0].assign(y);
//    multiply(tempmz[2],tem.value[0][k-1], tempmz[0]); 

    if (y*tempvect0[k-1] > 
       (my.value[k][k-1]*my.value[k][k-1]*tempvect0[k-1]+tempvect0[k]))
      {
	swap(value[k], value[k-1]);
	swap(T.value[k], T.value[k-1]);
	if (k > 1)
	  --k;
        if(k == 1) scalprod(tempvect0[0], value[0], value[0]);
      }
    else
      ++k;
  }
}   

//
// Buchmann Kessler version of lll for generating systems
// result : transformation lattice
//
void double_lattice::lin_gen_system(double_lattice& T, double y, 
                                    unsigned long& rc, timer& ti, long& rk)
{
  debug_handler("double_lattice","lin_gen_system(T, y, rc, ti, rk) [1]");


  double_lattice Atilde(rows+cols,cols),	// The approximate lattice
                 Ahead(rows, cols);		// The weighted lattice_double
  double *help=new double[rows],
         *rel=new double[cols],
         *temp=new double[rows];

  double zweipotq,
         norm1,		// To store the l1_norm
         norm2,		// To store the l2_norm
         vor,
         rechts,
         alpha;
  long n2 = rows;
  long i, j, l;

  alpha_compute(alpha);
  zwei_pot_q_compute(zweipotq, n2, alpha);
  for (i=0;i<cols;++i)
    for (j=0;j<rows;++j)
      {
        Ahead.value[i][j]=rint(zweipotq*value[i][j]);
//        multiply(tempmz[0],zweipotq,value[i][j]);
//        round(Ahead.value[i][j], tempmz0);
      }
  for (i=0;i<cols;++i)
    {
      Atilde.value[i][i]=1;  // 1, wenn i = j, 0 sonst
      for (j=cols;j<(cols+rows);++j)
        {
      	  Atilde.value[i][j]=Ahead.value[i][j-cols];
        }
    }

  debug_handler("double_lattice","lin_gen_system(T, prec, y, rc, ti, rk) [2]");
  ti.start_timer();
  Atilde.lll(T,exp(7.5*log(2)),y,rc);
  ti.stop_timer();
  debug_handler("double_lattice","lin_gen_system(T, prec, y, rc, ti, rk) [3]");

  l=0;
  do
    {
      vectsize=rows;
      ::assign_zero(help);	// Initializes help with the zero -.valueor
       for (j=0;j<cols;++j)
        {
          rel[j]=Atilde.value[l][j];
          scalmul(temp,rel[j], (Ahead.value[j]));
          add(help,help,temp);
        }
      l2_norm(norm2,help);
      vectsize=cols;
      l1_norm(norm1,rel);
      ++l;
      vor=norm1*sqrt((n2/rows)*2.0);
      rechts=sqrt(n2)/2.0*norm1;
    } 
  while ((zweipotq > vor) && (norm2 <= rechts) && (l < cols));
  debug_handler("double_lattice","lin_gen_system(T, prec, y, rc, ti, rk) [4]");

  if (l >= cols)
    warning_handler("double_lattice","lin_gen_system(T, prec, y, rc, ti, cols_rank) :: "
                    "lattice of dimension 1");

  T.cols_rank = cols - l + 1;	// r is the dimension of the lattice
  rk=T.cols_rank;
  for (i=0;i<cols-T.cols_rank;i++)
    for (j=0;j<cols;j++)
      T.value[i][j]=0;
  for (i=cols-T.cols_rank;i<cols;++i)
    for (j=0;j<cols;++j)
      T.value[i][j]=Atilde.value[i][j];
// 
// Free allocated storage
//
  delete[] help;
  delete[] rel;
  delete[] temp;
} 

//
// Gram - Schmidt - Orthogonalisation for lattice basis
// returns 0 if lattice is no basis 1 otherwise
// 
long double_lattice::gram_schmidt_orth(double_lattice& gso, double_lattice& my)
{
  debug_handler("double_lattice","gram_schmidt_orth(gso, my)");
  long i, j;
  vectsize=rows;
  for (i=0;i<cols;i++) 
    {
      ::assign(gso.value[i], value[i]);
      for (j=0;j<i;j++) 
        {
          scalmul(tempvect0, my.value[j][i], gso.value[j]);
          ::subtract(gso.value[i], gso.value[i], tempvect0);
        }
      scalprod(tempvect1[i], gso.value[i], gso.value[i]);    
      for(j=0;j<i;j++) 
        {
          scalprod(tempmz0, value[i], gso.value[j]);
          if (fabs(tempvect1[j]) < 10E-16)
            return (0);
          my.value[j][i]=tempmz0/tempvect1[j];
        }
    }
  return (1);
}

//
// check if reduction is good enough for parameter y 
//
int double_lattice::lll_check(double y)
{
  debug_handler("double_lattice","lll_ckeck(y)");
  double_lattice my(cols,cols);
  double_lattice gso(rows, cols);
  long i, j;

  if ((gram_schmidt_orth(gso,my)) == 0) 
    lidia_error_handler("double_lattice","lll_check(y) :: "
                  "lattice is no basis !");
  for (i=0;i<cols;i++)
    for (j=0;j<i;j++)
      if (fabs(my.value[j][i]) > 0.5)
        return(0);
  vectsize=rows;
  for (i=1;i<cols;i++)
    {
      scalmul(tempvect0, my.value[i][i-1], gso.value[i-1]);
      ::add(tempvect0,tempvect0,gso.value[i]); 
      scalprod(tempmz0, tempvect0, tempvect0);
      scalprod(tempmz1, gso.value[i-1], gso.value[i-1]);
      if (tempmz0 < y*tempmz1)
        return(0);
    } 
  return(1);
}

// 
// Searches paramter the lattice is reduced for
// returns 0 by non - reduction
//
double double_lattice::lll_check()
{
  debug_handler("double_lattice","lll_ckeck()");
  double_lattice my(cols,cols);
  double_lattice gso(rows, cols);
  double y=(1.0 + 0.25)/2;
  double old_y = 0;

  while((fabs(old_y-y) > 0.001) && (old_y < 1.0)) {
    if(lll_check(y)) {
      old_y = y;
      y = y + 0.2;
    }
    else y = old_y + (y - old_y) / 2;
  };

  return ((old_y>1.0)?1.0:old_y);
}

void double_lattice::cholesky(double& det)
{
  debug_handler("double_lattice","cholesky(det)");
  double_lattice Q(cols, cols),   // Q = (<Q[j], Q[i]>) the Fmatrix of the inner products.
                                  // <.,.> denotes the ordinary inner product
		                  // Q is stored only up to the diagonal elements because
                                  // it 's a symmetric matrix
                                  // Q wird als obere Dreiecksmatrix abgespeichert.
                 A(cols, cols);   // Cholesky - matrix
                                  // Matrix fuer die Cholesky - Zerlegung.
  long i, j, l;
  
  det = 1;	//Initialisation
  vectsize=rows;
  for (i=0;i<cols;++i)
    {
      for (j=0;j<=i;++j)
        {
         scalprod(Q.value[i][j],value[j], value[i]);
        }
      for (j=0;j<i;++j)
        {
          A.value[i][j] = Q.value[i][j] / A.value[j][j];
          for (l=j+1;l<=i;++l)
            Q.value[i][l] = Q.value[i][l] - A.value[l][j] * 
                            A.value[i][j] * A.value[j][j];
        }
      A.value[i][j] = Q.value[i][i];
      if (A.value[i][i] == 0)
        { 
          for (l=i;l<cols-1;++l)
            ::assign(value[l],value[l+1]);
          cols--;
          --i;		// L.value[i] has become new values, so it must be
                        // reregarded
        }
      else
        det = det * A.value[i][i];
      // Now A is the Cholesky - matrix
    }
  det=fabs(det);		// The determinant of the lattice is |det(A)|
  det=sqrt(det);	        // => det must have a positive value
}

int double_lattice::is_reduced()
{
  debug_handler("double_lattice","is_reduced()");
  double det,	     // det is the determinant of B 
         bound,	     // the bound to be tested
	 bounddummy, // a temporary bound - variable
	 prod;	     // the product of the l2 - norm of the basis -vectors
  long i;

  cholesky(det);

  vectsize=rows;

  bound = exp(log(det) / cols_rank); // bound = d(L)^(1/r)	
  bound = bound * (rows + 2);	// bound = (rows + 2) * d(L)^(1/r)
  bounddummy = exp(log(2.0) * (cols_rank - 1)/2.0);	// bounddummy = 2^((r-1)/2)
  bound = bound * bounddummy;
  l2_norm(tempmz0,value[0]);
  if (tempmz0 > bound)
    return(0);

  bound = exp(cols_rank * log(rows + 2));	// bound = (B.rows + 2)^r
  bounddummy = exp(log(2.0) * cols_rank * (cols_rank - 1)/2.0);	// bounddummy = 2^(r * (r - 1)/2)
  bound = bound * bounddummy * det;

  prod = 1;
  for (i=0;i<cols_rank;++i)
    {
      l2_norm(tempmz0, value[i]);
      prod = prod * tempmz0;	        // prod = l2_norm(B.value[0])
	                                //        * ...
    } 					//	  * l2_norm(B.value[r])
  if (prod > bound)
    return(0);
  return(1);
}

//
// Computes lll - reduced basis by multiplikation of A and T 
// T transformation lattice
//
void double_lattice::compute_basis(const double_lattice& A,
                                     const double_lattice& T)
{
  debug_handler("double_lattice","compute_basis(A, T)");
  double_lattice temp(A.rows, A.cols);
  long i, j, cx;

// Computing the basis - vectors
  for (i=0;i<A.rows;++i)
    for (j=0;j<A.cols;++j) 
      for (cx=0;cx<A.cols;cx++)
        temp.value[j][i]+=A.value[cx][i]*T.value[j][cx];
  assign(temp);
}

