/** 
 ** This  include  file  supports basic polynomial operations over 
 ** all LiDIA-types
 **
 ** Most   of   this   code ( +, -, *, /, gcd, xgcd, cin, cout )  was
 ** originally  written by Victor Shoup; Thomas Papanikolaou changed
 ** his implementation to use templates and the names of the functions
 ** to our standard. Then I extended this code to include the case of
 ** polynomials over the integers.
 ** 
 **						Stefan Neis
 **/

#ifndef LIDIA_POLYNOMIAL_H
#define LIDIA_POLYNOMIAL_H

#ifdef __GNUG__
#define LiDIA_INLINE inline
#else
#define LiDIA_INLINE
#endif

#include <iostream.h>
#include <LiDIA/lidia.h>
#include <LiDIA/base_vector.h>
#include <LiDIA/bigint.h>
#include <LiDIA/bigrational.h>
#include <LiDIA/bigfloat.h>
#include <LiDIA/bigcomplex.h>

//
// The first class in this file (base_polynomial) is for internal use only !!!
//
// The user is supposed to always use "polynomial < T >".
//

template < class T > class base_polynomial
{
protected:

  /**
   ** base_polynomial<T> a = c[0] + c[1]*X + ... + c[deg]*X^deg
   **/

  T * coeff;
  lidia_size_t deg;

public:

  /**
   ** constructors and destructor
   **/

  base_polynomial()
  {
    deg = 0;
    coeff = new T[1];
    coeff[0] = 0;
  }

  base_polynomial(T x)
  {
    deg = 0;
    coeff = new T[1];
    coeff[0] = x;
  }

  T * copy_data(T * d, const T * vd, lidia_size_t al)
  {
    debug_handler_l("base_polynomial", "copy_data()", -1);
    for (lidia_size_t i = 0; i <= al; i++)
      d[i] = vd[i];
    return d;
    debug_handler_l("base_polynomial", "end copy_data()", -1);
  }
 
  base_polynomial(const T * v, lidia_size_t d)
  {
    deg = d;
    coeff = new T[d + 1];
    coeff = copy_data(coeff, v, deg);
    remove_leading_zeros();
  }

  base_polynomial(const base_vector < T > v)
  {
    deg = v.size()-1;
    coeff = v.get_data();
    remove_leading_zeros();
  }

  base_polynomial(const base_polynomial < T > &p)
  {
    deg = p.deg;
    coeff = new T[deg + 1];
    coeff = copy_data(coeff, p.coeff, deg);
  }

  ~base_polynomial()
  {
    delete[] coeff;
  }

  /**
   ** inline friend functions
   **/

  /**  comparisons  **/
  LiDIA_INLINE friend bool operator == (const base_polynomial < T > &a,
                            const base_polynomial < T > &b)
  {
    T * ap, *bp;
    lidia_size_t i, da = a.deg, db = b.deg;
    if (da != db)
      return false;
    for (i = da + 1, ap = a.coeff, bp = b.coeff; i; i--, ap++, bp++)
      if (*ap != *bp)
	return false;
    return true;
  }

  LiDIA_INLINE friend bool operator != (const base_polynomial < T > &a,
			  const base_polynomial < T > &b)
  {
    T *ap, *bp;
    lidia_size_t i, da = a.deg, db = b.deg;
    if (da != db)
      return true;
    for (i = da + 1, ap = a.coeff, bp = b.coeff; i; i--, ap++, bp++)
      if (*ap != *bp)
	return true;
    return false;
  }

  /**  some operators  **/
  LiDIA_INLINE friend base_polynomial < T > operator - (const base_polynomial < T > &a)
  {
    lidia_size_t i, deg = a.deg;
    base_polynomial < T > c;
    c.set_degree(deg);
    for (i = 0; i <= deg; i++)
      c.coeff[i] = -a.coeff[i];
    return c;
  }

  LiDIA_INLINE friend base_polynomial < T > operator * (const T & b,
					   const base_polynomial < T > &a)
  {
    T *ap, *cp;
    lidia_size_t deg_a = a.deg;
    lidia_size_t i;

    base_polynomial < T > c;
    c.set_degree(deg_a);

    for (i = 0, ap = a.coeff, cp = c.coeff; i <= deg_a; i++, ap++, cp++)
      *cp = b * (*ap);
    return c;
  }

  LiDIA_INLINE friend base_polynomial < T > operator * (const base_polynomial < T > &a,
				      const T & b)
  {
    T *ap, *cp;
    lidia_size_t deg_a = a.deg;
    lidia_size_t i;

    base_polynomial < T > c;
    c.set_degree(deg_a);

    for (i = 0, ap = a.coeff, cp = c.coeff; i <= deg_a; i++, ap++, cp++)
      *cp = (*ap) * b;
    return c;
  }

  /** others **/
  LiDIA_INLINE friend T lead_coeff(const base_polynomial < T > &a)
  {return a.coeff[a.deg];}

  LiDIA_INLINE friend T const_term(const base_polynomial < T > &a)
  {return a.coeff[0];}

  /**
   ** member functions
   **/

  int set_data ( T * d, lidia_size_t l ) 
  {
    debug_handler_l("base_polynomial", "set_data ( T*, lidia_size_t )" , -1);

    delete[] coeff;
    coeff = new T[l];
    deg=l-1;

    for (register lidia_size_t i = 0; i < l; i++)
      coeff[i] = d[i];
    remove_leading_zeros();
    return 0;
  }


  T* get_data () const
  {
    debug_handler_l("base_polynomial", "get_data ()", -1 ) ;

    T * d;

    d = new T [deg+1] ;
    for (register lidia_size_t i = 0; i <= deg; i++)
      d[i] = coeff[i] ;
    return d ;
  }

  lidia_size_t degree() const
  {
    return deg;
  }

  void set_degree(lidia_size_t);

  bool is_zero() const
  {
    return (deg == 0 && coeff[0] == T(0)); 
  }

  bool is_one() const
  {
    return (deg == 0 && coeff[0] == T(1)); 
  }

  bool is_x() const
  {
    return (deg == 1 && coeff[1] == T(1) && coeff[0] == T(0)); 
  }

  void assign_zero()
  {
    deg = 0;
    delete[] coeff;
    coeff = new T[1];
    coeff[0] = 0;
  }

  void assign_one()
  {
    deg = 0;
    delete[] coeff;
    coeff = new T[1];
    coeff[0] = 1;
  }

  void assign_x()
  {
    deg = 1;
    delete[] coeff;
    coeff = new T[2];
    coeff[0] = 0;
    coeff[1] = 1;
  }

  void remove_leading_zeros();

  /**
   ** assignment
   **/

  base_polynomial < T > &operator = (const base_polynomial < T > &a);

  /**
   ** operator overloading
   **/

  T & operator[] (lidia_size_t i) const
  {
    if ((i < 0) || (i > deg))
      lidia_error_handler("base_polynomial", "operator []::out of range");
    return coeff[i];
  }

  T operator() (T value) const;

  friend base_polynomial < T > operator + (const base_polynomial < T > &a,
				      const base_polynomial < T > &b);

  friend base_polynomial < T > operator - (const base_polynomial < T > &a,
				      const base_polynomial < T > &b);

  friend base_polynomial < T > operator * (const base_polynomial < T > &a,
				      const base_polynomial < T > &b);
  friend void power(base_polynomial < T > & c, 
		    const base_polynomial < T > & a, const bigint & b);

  void operator += (const base_polynomial < T > &a)
  {*this = *this + a;}

  void operator -= (const base_polynomial < T > &a)
  {*this = *this - a;}

  void operator *= (const base_polynomial < T > &a)
  {*this = *this * a;}


  /**
   ** functions
   **/

  friend base_polynomial < T > derivation(const base_polynomial < T > &a);


  /**
   ** input / output
   **/

  friend istream & operator >> (istream &, base_polynomial < T > &);
  istream & read_verbose(istream &);
  friend ostream & operator << (ostream &, const base_polynomial < T > &);

};

//
// Now the general class for polynomials:
//

template < class T > class polynomial: public base_polynomial < T >
{
public:
  /**
   ** constructors and destructor
   **/

  polynomial(): base_polynomial < T > ()
  { }

  polynomial(T x): base_polynomial < T > (x)
  { }

  polynomial(T * v, lidia_size_t d): base_polynomial < T > (v,d)
  { }

  polynomial(const base_vector < T > v): base_polynomial < T > (v)
  { }

  polynomial(const base_polynomial < T > &p): base_polynomial < T > (p)
  { }

  ~polynomial()
  { }

  polynomial < T > &operator = (const base_polynomial < T > &a);
};

class polynomial <bigint> : public base_polynomial <bigint>
{
  // For some routines, I would like to use polynomial<bigmod>, which is
  // not yet finished...
public:

  /**
   ** constructors and destructor
   **/

  polynomial(): base_polynomial <bigint> ()
  { }

  polynomial(bigint x): base_polynomial <bigint> (x)
  { }

  polynomial(const bigint * v, lidia_size_t d): base_polynomial <bigint> (v,d)
  { }

  polynomial(const base_vector <bigint> v): base_polynomial <bigint> (v)
  { }

  polynomial(const base_polynomial <bigint> &p): base_polynomial <bigint> (p)
  { }

  ~polynomial()
  { }

  polynomial <bigint> &operator = (const base_polynomial <bigint> &a);

  /**
   ** Pseudo - division and related stuff
   **/

  friend void div_rem(polynomial <bigint> &q, polynomial <bigint> &r,
		      const base_polynomial <bigint> &a, const base_polynomial <bigint> &b);

  friend polynomial <bigint> operator / (const base_polynomial <bigint> &a,
					 const bigint & b);

  friend polynomial <bigint> operator / (const base_polynomial <bigint> &a,
					 const base_polynomial <bigint> &b);

  friend polynomial <bigint> operator % (const base_polynomial <bigint> &a,
					 const base_polynomial <bigint> &b);

  void operator /= (const base_polynomial <bigint> &a);

  void operator /= (const bigint &a);

  void operator %= (const base_polynomial <bigint> &a)
  {*this = *this % a;}

  friend void power_mod(polynomial <bigint> & c, 
			const base_polynomial <bigint> & a, const bigint & b,
			const base_polynomial <bigint> & f);

  /**
   ** Gcd's and related stuff, i.e. content and primitive part.
   **/

  friend bigint cont(const base_polynomial <bigint> &a);

  friend polynomial <bigint> pp(const base_polynomial <bigint> &a);

  friend polynomial <bigint> gcd(const base_polynomial <bigint> &aa,
			      const base_polynomial <bigint> &bb);
    
  friend polynomial <bigint> xgcd(polynomial <bigint> &x,
			       polynomial <bigint> &y,
			       const base_polynomial <bigint> &aa,
			       const base_polynomial <bigint> &bb);
  
  /**
   ** Number of real roots
   **/

  friend lidia_size_t no_of_real_roots(const base_polynomial<bigint>& poly_T);  


  /**
   ** Modular Computations needed for factoring modulo p
   **/

  friend void div_rem(polynomial <bigint> &q, polynomial <bigint> &r,
		      const base_polynomial <bigint> &a,
		      const base_polynomial <bigint> &b, const bigint & p);

  friend polynomial <bigint> operator % (const base_polynomial <bigint> &a,
					 const bigint &b);

  friend void  operator %= (polynomial <bigint> &a, const bigint &b);

  friend void power(polynomial <bigint> & c,
		    const base_polynomial <bigint> & a,
		    const bigint & b, const bigint & p);

  friend void power_mod(polynomial <bigint> & c,
			const base_polynomial <bigint> & a, const bigint & b,
			const base_polynomial <bigint> & f, const bigint & p);

  friend polynomial <bigint> gcd(const base_polynomial <bigint> &aa,
				 const base_polynomial <bigint> &bb,
				 const bigint & p);

  /**
   ** Factoring modulo p
   **/

  friend void squarefree_factor(const base_polynomial <bigint> &f,
				polynomial <bigint> * fa,
				const bigint & p);

  friend int ddf(const base_polynomial <bigint> &f0,
		 polynomial <bigint> * fa, 
		 const bigint & p);

  friend void can_zass(const base_polynomial <bigint> &f,
		       polynomial < bigint> * fa,
		       lidia_size_t d, const bigint & p,
		       lidia_size_t count = 0);

  friend void factor(const base_polynomial <bigint> &f,
		     polynomial <bigint> * fa,  lidia_size_t * exponent, 
		     const bigint & p);
};

LiDIA_INLINE
polynomial <bigint> operator / (const base_polynomial <bigint> &a,
				const base_polynomial <bigint> &b)
{
  polynomial <bigint> q, r;
  div_rem(q, r, a, b);
  return q;
}

LiDIA_INLINE
void operator %= (polynomial <bigint> &a, const bigint &b)
{
  a = a % b;
}

LiDIA_INLINE
bigint cont(const base_polynomial <bigint> &a)
{
    debug_handler_l("polynomial<bigint>", "cont(...)", -1);
    lidia_size_t d = a.degree();
    bigint g = a[d];
    for (lidia_size_t i = d - 1; i >= 0; g = gcd(a[i--],g));
    return g;
    debug_handler_l("polynomial<bigint>", "end cont(...)", -1);
}

LiDIA_INLINE
polynomial <bigint> pp(const base_polynomial <bigint> &a)
{
    debug_handler_l("polynomial<bigint>", "pp(...)", -1);
    if (a.is_zero())
	return a;
    else
	return (a / cont(a));
    debug_handler_l("polynomial<bigint>", "end pp(...)", -1);
}

#endif
