.TH RKF45 L LOCAL "1st UNM Distribution" "LOBO.LIB User's Manual"
.hy 0
.if t .ds ST \v'.2m'*\v'-.2m'
.if n .ds ST *
.tr ^\|
.tr ~
.SH NAME
RKF45
.SH DECLARATIONS
.nf
subroutine RKF45(F,neqn,y,t,tout,relerr,abserr,iflag,work,iwork)
external F
integer neqn,iflag, iwork(5)
double precision y(neqn), t, tout, relerr, abserr, work(3+6\*(STneqn)
.fi
.SH SYNOPSIS
.PP
FORTRAN subroutine primarily designed to solve initial value problems of
non-stiff and mildly stiff differential equations.
RKF45 should generally not be used when the user is demanding
high accuracy.
.PP
RKF45  uses the Runge-Kutta-Fehlberg (4,5) method developed
by Fehlberg and described in Shampine[1,2,3].  RKF45 is
further described in Forsythe, Malcolm and Moler[4].
.SH PARAMETERS
.TP 10
F
User supplied subroutine to evaluate derivatives of \fIy(t)\fR,
i.e., the differential equation being solved.
The user must declare F to be external and must have
the form:
.RS
.ns
.IP
.nf
subroutine F(t,y,yp)
double precision t, y(*), yp(*)
Evaluate each of the derivatives
   yp(i)~=~dy(i)/dt, i~=~1,...,neqn
return
end
.RE
.fi
.TP 10
neqn
is the number of equations, one or more, to be integrated.
.TP 10
y(neqn)
On entry \fIy\fR
is the vector of initial conditions.
.br
On return \fIy\fR
is the solution at \fIt\fR.
.TP 10
t
On entry \fIt\fR
is the starting point of integration.
.br
On return \fIt\fR
is the last point reached in integration.
.TP 10
tout
is the output point at which the solution is desired, \fItout\fR
may be less than \fIt\fR.
The condition \fIt\fR~=~\fItout\fR
is allowed on the first call only, in which case RKF45 performs
a single step and returns.
.TP 10
abserr
.br
.ns
.TP 10
relerr
are the absolute and relative local error tolerances.
Relative error should not be smaller than about 1.0e-8.
Specifically, \fIrelerr\fR must be larger than machine error.
If a smaller than
allowable value of \fIrelerr\fR is attempted, RKF45 increases
\fIrelerr\fR appropriately and returns with \fIiflag\fR = 3 without
continuing the integration,
see INTERPRETATION OF IFLAG below.
.TP 10
iflag
Input and output parameter containing information about
the status of integration.
.RS
.TP 10
1
Input only on the first call to RKF45 for each new problem.
Instructs the code to initialize itself then to integrate
the interval from \fIt\fR to \fItout\fR if no error
conditions are encountered.
.TP 10
2
Normal return in that the integration has reached \fItout\fR
with no error conditions.  As input on subsequent calls, instructs
the code to continue integration with a newly defined \fItout\fR.
.TP 10
3,...,8
Returns of these values indicate a warning or error condition
was encountered and some action must be taken before the
integration can be continued, if at all.  See INTERPRETATION
OF IFLAG below.  See the EXAMPLE for the recommended
response.
.RE
.IP
Single step mode is also available.  This is indicated by
setting \fIiflag\fR = -1 on the first call to RKF45.
The expected return in this mode is \fIiflag\fR = -2
indicating a single successful step in the direction of
\fItout\fR has been taken.
The return of \fIiflag\fR = 2
indicates that \fItout\fR has been reached with the
completion of this step.
To continue with single step integration, input \fIiflag\fR = -2
on subsequent calls to RKF45.
.TP 10
work(m)
array used internally by RKF45; \fIm\fR >= 3+6\*(ST\fIneqn\fR.
.TP 10
iwork(k)
integer array used internally by RKF45; \fIk\fR >= 5.


.SH INTERPRETATION OF IFLAG
RKF45 returns with all the information needed to continue the
integration.
If the integration reached \fItout\fR successfully, (\fIiflag\fR~=~2)
the user need only
define a new \fItout\fR and call RKF45 again.  In the one-step integrator
mode, if \fIiflag\fR~=~-2, the user need only re-call RKF45 to
continue the integration.
In the one-step mode, the user must keep in mind that each step taken is
in the direction of the current \fItout\fR.
Specifically, upon reaching \fItout\fR, \fIiflag\fR~=~2 is returned,
the user must then define a new \fItout\fR and
reset \fIiflag\fR~=~-2 to continue in the one-step integrator mode.
.PP
The return of \fIflag\fR values 3 and 4 are warnings and integration
has been halted before this step was completed.
The value \fIiflag\fR~=~3 indicates relative error
tolerance was too small, RKF45 has increased \fIrelerr\fR
appropriately for continuing.
The value \fIiflag\fR~=~4 indicates 500 steps have been taken
meaning more than 3000 derivative evaluations have been made.
Continuation will allow 500 more steps before this warning
is issued again.
To continue, simply call
RKF45 again, no changes are necessary.
.PP
The case \fIiflag\fR~=~5 occurs only with \fIabserr\fR~=~0.
Integration cannot be continued
as the solution is near
zero making a pure relative error test impossible.  To continue
the user must reset
\fIabserr\fR to a positive value before calling RKF45 again.
Using the one-step integration mode may be
a good way to proceed in this situation.
.PP
The occurrence of \fIiflag\fR~=~6 indicates
integration was not completed because the requested
accuracy could not be achieved using the smallest
allowable step size.  This usually
indicates a trouble spot, i.e.
the solution is changing rapidly, therefore a
singularity may be present, and it
often is inadvisable to continue.
If the user insists on continuing the integration,
it is necessary to increase either \fIabserr\fR or \fIrelerr\fR or
both
and to reset \fIiflag\fR~=~2 (or -2).
If this is not done, execution will
be terminated.
.PP
If \fIiflag\fR~=~7, it is likely that RKF45 is inefficient for solving
this problem.  Too much output is restricting the
natural step size choice.
Using the one-step integrator
mode may be helpful.
The user should
consider switching to the Adams codes DE/STEP, INTRP.  If the user
insists upon continuing the integration with RKF45, he must reset
\fIiflag\fR to 2 before calling RKF45 again.
.PP
The return of \fIiflag\fR~=~8 indicates invalid
input parameters.  This may include one or more of
the following:
.nf
	\fIneqn\fR~<=~0
	\fIt~=~tout\fR  and  \fIiflag\fR~<>~1~~or~~-1
	\fIrelerr\fR or \fIabserr\fR~<~0
	\fIiflag\fR~=~0~~or~~<~-2~~or~~>~8
.fi
The user should check all assignments to input parameters then
try again.
.SH DETAILS
RKF45 integrates a system of \fIneqn\fR first order
ordinary differential equations
\fIdy(i)/dt\fR~=~\fRf(t,y(1),y(2),...,y(neqn))\fR
where the \fIy(i)\fR are given at \fIt\fR.
.sp
Typically the subroutine is used to integrate from \fIt\fR to
\fItout\fR but it can be used as a one-step integrator
to advance the solution a single step in the direction of \fItout\fR.
Actually, RKF45 is an interfacing routine which calls subroutine
RKFS for the solution.  RKFS in turn calls subroutine FEHL which
computes an approximate solution over one step.
.PP
.SH EXAMPLE
.ta .13i .38i .5i 1i
.nf
C		Sample main program using RKF45
C		This sample demonstrates an appropriate use of the
C		parameter IFLAG.
C
C		Declare the parameters, variables and subroutine.
C		The EXTERNAL F indicates that F is the name of a subroutine.
C
			EXTERNAL F
			DOUBLE PRECISION T,Y(2),T0,Y0,YP0,TFINAL,TPRINT,TOUT
			DOUBLE PRECISION RELERR,ABSERR,WORK(15)
			INTEGER IWORK(5),IFLAG,NEQN
C
C		Initialize the parameters.
C		Setting TOUT = T0 causes the initial data to be printed.
C
			NEQN = 2
			T0 = 0.0D0
			Y0 = 1.0D0
			YP0 = 0.0D0
			TFINAL = 2.0D0
			TPRINT = 0.1D0
			RELERR = 1.0D-5
			ABSERR = 0.0D0
			IFLAG = 1
			TOUT = T0
C
C		Start the main loop.
C
			T = T0
			Y(1) = Y0
			Y(2) = YP0
	01		IF (TOUT .GT. TFINAL) TOUT = TFINAL
			CALL RKF45(F,NEQN,Y,T,TOUT,RELERR,ABSERR,IFLAG,WORK,IWORK)
C
			WRITE(6,02) T, Y(1), Y(2)
	02		FORMAT(1X, F9.5, 2(3X,F11.5))
C
			GO TO (10,20,30,40,50,60,70,80), IFLAG
C
C		IFLAG = 1, which can't be returned
C
	10		WRITE(6,11)
	11		FORMAT(/20H THIS IS IMPOSSIBLE.)
			STOP
C
C		IFLAG = 2, everything is normal.
C
	20		TOUT = T + TPRINT
			IF (T .LT. TFINAL) GO TO  01
			STOP
C
C		These are warning and error messages.
C
	30		WRITE(6,31) RELERR
	31		FORMAT('WARNING. IFLAG = 3, RELERR INCREASED TO , 1PE12.3')
			GO TO  01
C
	40		WRITE(6,41)
	41		FORMAT('NOTICE. IFLAG = 4, RKF45 HAS TAKEN 500 STEPS.')
			GO TO  01
C
	50		ABSERR = RELERR
			WRITE(6,51) ABSERR
	51		FORMAT('WARNING. IFLAG = 5, SOLUTION IS NEAR 0,'/,
		$		' ABSERR = ,1PE12.3')
			GO TO  01
C
	60		WRITE(6,61)
	61		FORMAT('IFLAG = 6, RKF45 CANNOT PROCEED,'/,
		$		' SINGULARITY LIKELY.')
			STOP
C
	70		WRITE(6,71)
	71		FORMAT('IFLAG = 7'/,
		$		' PRINT IS SMALLER THAN EFFICIENT STEP SIZE.'/,
		$		' SEE RKF45 COMMENTS.')
			IFLAG = 2
			GO TO  01
C
	80		WRITE(6,81)
	81		FORMAT('ERROR. IFLAG = 8, IMPROPER CALL TO RKF45.')
			STOP
			END

			SUBROUTINE F(T,Y,YP)
			DOUBLE PRECISION T,Y,YP
			YP(1) = Y(2)
			YP(2) = -Y(1)
			END
.fi
.SH REFERENCES
.IP [1]
Shampine, L. F., H. A. Watts, S. M. Davenport, \fISolving Nonstiff
Ordinary Differential Equations -- the State of the Art\fR,
SIAM Review, Vol. 18, pp. 376-411, 1976.
.IP [2]
Shampine, L. F., H. A. Watts, \fIThe Art of Writing a Runge-Kutta Code,
Part I\fR, \fBMathematical Software III\fR, J. R. Rice (editor),
pp. 257-275, Academic Press, 1977.
.IP [3]
Shampine, L. F., H. A. Watts, \fIThe Art of Writing a Runge-Kutta Code,
Part II\fR, Applied Mathematics and Computation, Vol. 5, 1979,
pp. 93-121.
.IP [4]
Forsythe, G. E., M. A. Malcolm, C. B. Moler, \fBComputer Methods
for Mathematical Computations\fR. Prentice-Hall, 1977.
.SH AUTHORS
.nf
H. A. Watts and L. F. Shampine
Sandia Laboratories
Albuquerque, New Mexico
.SH FURTHER INFORMATION
.nf
Contact:
Mathematical Software Project
Computer Science Department
University of New Mexico
Albuquerque, NM  87131
