#############################################################################
##
#A  Matrix Group and G-module library                   Derek Holt
#A                                                      Charles Leedham-Green
#A                                                      Eamonn O'Brien
#A                                                      Sarah Rees 
##
#A  @(#)$Id$
##
#Y  Copyright 1994 -- School of Mathematical Sciences, ANU   
##
#H  $Log$
##
############################################################################
##
#F  InfoTensorProd (...)  . . . . . . . . . . . for debugging assistance
##
##
if not IsBound(InfoTensorProd)  then InfoTensorProd := Ignore;  fi;
#############################################################################
##
#F  TensorProductDecomp( module,basis,dim1,dim2)  . . test to see if we
## have a tensor product, of modules of dimensions dim1, dim2.
##
## 
TensorProductDecomp := function ( module,basis,dim1,dim2) 
   local F,invbasis,
         g,x,factors,matrices1,matrices2;

   #check added by EOB
   if dim1 * dim2 <> DimFlag (module) then return false; fi;

   F := FieldFlag(module);
   invbasis := basis^-1;
   matrices1 := [];
   matrices2 := [];
   for g in module.matrices do
     x := basis*g*invbasis;
     factors := KroneckerFactors(x,dim1,dim2,F);
     if factors = false then return false;
     else
       Add(matrices1,factors[1]);
       Add(matrices2,factors[2]);
     fi;
   od;
   SetTensorProdFlag(module,true);
   SetTensorBasisFlag(module,basis);
   SetTensorFactorsFlag(module,[GModule(matrices1,F),GModule(matrices2,F)]);
   InfoTensorProd("Module is a tensor prod. of modules of dim ",dim1," and ",dim2,".\n");
   return true;

end;
       
#############################################################################
##
#F  KroneckerFactors( x,dim1,dim2 [,F])  . . test to see x is a Kronecker
## product of 2 matrices of dimensions dim1, dim2 resp., over F.  More precisely,
## we try to find A, a dim1 x dim1 matrix, and B, a dim2 x dim2 matrix, so
## that x decomposes into dim1xdim1 blocks, with the k,l-th block equal to 
## A[k][l]*B, i.e. x is the Kronecker product of A and B;
## If we can find such matrices we return the pair [A,B], otherwise we
## return false.
##
## 
KroneckerFactors := function ( arg ) 
   local x, dim1, dim2, F,
         r,s,r0,s0,i,j,k,l,y,A,B;

   x := arg[1];
   dim1 := arg[2];
   dim2 := arg[3];
   if Number(arg)=3 then F:=Field(Flat(x));
   else F := arg[4]; fi;
   A := [];
   B := [];
# first find a position where there's a non-zero entry
   i:=1;j:=1;
   while  x[i][j] = F.zero do
     if j<dim2 then j := j+1;
     else j:= 1; i:= i+1; fi;
   od;
# so x[i][j]<>0;
   y := x[i][j];
   r := (i-1) mod dim2 + 1; r0 := i - r;
   s := (j-1) mod dim2 + 1; s0 := j - s;
   for i in [1..dim2] do
     B[i] := [];
     for j in [1..dim2] do
       B[i][j] := x[r0+i][s0+j];
     od;
   od;
   for k in [1..dim1] do
     A[k] := [];
     for l in [1..dim1] do
       A[k][l] := x[(k-1)*dim2 + r][(l-1)*dim2 + s]/y;  
     od;
   od;
   if x <> KroneckerProduct(A,B) then return false; 
   else return [A,B];
   fi;

end;
       
#############################################################################
##
#F  UndoTensorProdFlags( module)  . . undo flags set by TensorProd 
##
## 
UndoTensorProdFlags := function (module) 

   UndoTensorProdFlag(module);
   UndoTensorBasisFlag(module);
   UndoTensorFactorsFlag(module);

end;

#############################################################################
##
#F  SymTensorProductDecomp( module,Smodule)  . . test to see if have a 
## symmetric tensor product
##
## module is a module for a finite matrix group G over a finite field.
## and Smodule is the module corresponding to the action of a subgroup 
## <S> of G on the same vector space. 
## G and <S> are assumed to act absolutely irreducibly.
## The function returns true if Smodule can be decomposed as a 
## tensor product of spaces all of the same dimension, and if further 
## these tensor factors 
## are permuted by the action of G. In that case components of
## the record module record the tensor decomposition and the action of
## G permuting the factors. If no such decomposition is found the
## function returns false.
## The function uses random elements to try and find the tensor
## decomposition. Thus a negative answer cannot be 100% reliable.
## 
SymTensorProductDecomp := function ( module,Smodule) 

  local d,F,matrices,S,ngens,permutes,numTries,
        divisors,poss,pair,dd,n,seed,tenpow,basis,invbasis,permaction,
        pi_ij,pi_g,factors,
        g,h, i,j,k;


  numTries := 20;  

  d := DimFlag(module);
  F := FieldFlag(module);
  matrices := MatricesFlag(module);
  S := MatricesFlag(Smodule);
  ngens := Length(matrices);
  permaction := [];


#  divisors := Reversed(DivisorsInt(d));
  divisors := DivisorsInt(d);
  poss :=[];
  for dd in divisors do
    if dd<>1 and dd<> d then
      n :=  IntPower(d,dd);
      if n<>false then Add(poss,[dd,n]); fi;
    fi;
  od;
  if poss=[] then 
    InfoTensorProd("Dimension is not a proper power.\n");
    return false; 
  fi;

  seed := InitialiseSeed(S,10,50);
  for pair in poss do
    InfoTensorProd("Trying pair ",pair," in SymTensorProductDecomp.\n");
    dd := pair[1]; n := pair[2];
    tenpow := MultipleTensorProductDecomp(Smodule,dd,n,seed,numTries);
    if tenpow <> false then
      InfoTensorProd("Found a tensor power decomposition.\n");
      InfoTensorProd("Module is ",n,"-th power of a ",dd,"-dim module.\n");
      basis := tenpow[1];
      invbasis := basis^-1;
      k:= 1;
      permutes := true;
      while  permutes=true and k<= ngens do
        g := basis*matrices[k]*invbasis;
        pi_g := ();
        i:= 1;
        while permutes=true and i<n do
          j:= i;
          factors := KroneckerFactors(g,dd^(n-i),dd,F);
          if factors=false then
            repeat
              j := j+1; 
              if j<=n then
                pi_ij := SwapFactors(1,j+1-i,dd,n+1-i,F);  
                factors := KroneckerFactors(g*pi_ij,dd^(n-i),dd,F);
                if factors<> false then pi_g := (i,j)*pi_g; fi;
              fi;
            until j>n or factors <> false;
          fi;
          if factors=false then permutes := false;
          else g:= factors[1]; i:= i+1; fi;
        od;
        if permutes=true then 
          InfoTensorProd(k,"-th generator acts as permutation ",pi_g," on factors.\n");
          Add(permaction,pi_g); 
          k := k+1;
        else InfoTensorProd(k,"-th generator does not permute factors.\n");
        fi; 
      od;
      if permutes=true then 
        SetSymTensorProdFlag(module,true);
        SetSymTensorBasisFlag(module,basis);
        SetSymTensorFactorsFlag(module,tenpow[2]);
        SetSymTensorPermFlag(module,permaction);
        return true;
      fi; 
    fi;
  od;

  return false;

end;

#############################################################################
##
#F  MultipleTensorProductDecomp( module,d,n,seed,numTries)  . . 
##
## The function uses random methods to try and decompose the module
## `module' as a tensor product of n spaces of dimension d.
## The method is iterative; at each stage of a successful
## decomposition a space W of dimension a
## power of d is written as tensor product of two such spaces W1, W2 of lower
## dimension. (This is done using a randomly generated element of an appropriate
## order as input for a call of the function SmashGMod.)
## Up to NumTries random elements are tried at each stage.)
## seed is a seed for the random process, and numTries a parameter
## which determines the random elements which are tried at each
## level of the tensor decomposition.
## 
MultipleTensorProductDecomp := function ( module,d,n,seed,numTries) 
  
  local h,pair,po,poprimes,r,hh,SS,nn,sseed,
        F,q,N,GLprimes,i, 
        basis, P, factors,
        module1,module2,P1,P2,dim1,dim2,
        tenpow1,tenpow2,factors1,factors2, 
        try, numTries;
        

  InfoTensorProd("Trying to decompose as a ",n,"-th tensor power of a ",d,"-dimensional module.\n");

   #check added by EOB
   if d^n <> DimFlag (module) then return false; fi;

  F := FieldFlag(module);
  q := Size(F);
  if IsIrredGMod(module)=false or IsAbsIrredGMod(module)=false then
    return false;
  fi;

  GLprimes := [F.char];
  for i in [1..d] do 
    if q <> 2 or i <> 1 then  # exclude q^i-1=1
      GLprimes := Concatenation(GLprimes,Set(FactorsInt(q^i - 1))); 
      GLprimes  := Set(GLprimes);
    fi;
  od;


  try := 1;
  while try <= numTries do
    InfoTensorProd("In MultipleTensorProductDecomp loop, try ",try," of ",numTries,".\n");
    h := RandomElement(seed);
    pair := MatrixProjectiveOrder(h); po := pair[1];
    if po<>1 then
      poprimes := Set(FactorsInt(po));
    else poprimes := [];
    fi;
    for r in poprimes do
      if not r in GLprimes then
        InfoTensorProd("Projective order ",po," of element incompatible with ",n,"-fold tensor power of V(",d,",",q,").\n");
        InfoTensorProd("GLprimes = ",GLprimes,"\n");
        InfoTensorProd("poprimes = ",poprimes,"\n");
        return false;
      fi;
    od;
        
    for r in poprimes do
      hh := h^(po/r);
      SS := [hh];
      UndoTensorProdFlags(module);
      if SmashGMod(module,SS,"tensorprod")=true and TensorProdFlag(module)=true then 
        basis := TensorBasisFlag(module);
        module1 := TensorFactorsFlag(module)[1];
        module2 := TensorFactorsFlag(module)[2];
        dim1 := DimFlag(module1);
        dim2 := DimFlag(module2);
        nn := IntPower(dim1,d);
        if nn<>false then
          if nn<>1 then
            sseed := InitialiseSeed(MatricesFlag(module1),10,50);
            tenpow1:= MultipleTensorProductDecomp(module1,d,nn,sseed,numTries);
# We may want to change this: return false if recursive call to 
#MultipleTensorProductDecomp returns false
            if tenpow1 = false then return false; fi;
          else tenpow1 := [ IdentityMat(dim1,F),[ module1]]; 
          fi;
          nn := n - nn;
          if nn<>1 and tenpow1<>false then
            sseed := InitialiseSeed(MatricesFlag(module2),10,50);
            tenpow2 := MultipleTensorProductDecomp(module2,d,nn,sseed,numTries);
# We may want to change this: return false if recursive call to 
#MultipleTensorProductDecomp returns false
            if tenpow2 = false then return false; fi;
          else tenpow2 := [ IdentityMat(dim2,F),[ module2]]; 
          fi;
          if tenpow1 <> false and tenpow2 <>false then 
            P1 := tenpow1[1];
            P2 := tenpow2[1];
            P := KroneckerProduct(P1,P2);
            factors1 := tenpow1[2];
            factors2 := tenpow2[2];
            factors := Concatenation(factors1,factors2);
# After debugging this bit of code can be shortened, and a lot of variables thrown away.
            return [P*basis,factors]; 
          fi;
        fi;
      fi;
    od;
    try := try+1;
  od;

  InfoTensorProd("Failed to decompose as a ",n,"-th tensor power of a ",d,"-dimensional module.\n");
  return false;


end;

#############################################################################
##
#F  IntPower( A,a)  . . if A=a^k, for k>=1 return k
## otherwise return false
## Simliarly return false if either A or a is not a positive integer.
##
## 
IntPower := function ( A,a) 

  local N,k;

  if A<0  or a<0 or A<a then return false; fi;
  N := A; k:= 0;
  repeat N := QuoInt(N,a); k:=k+1; until RemInt(N,a)<>0;  
  if N=1 then return k; else return false; fi;

end;

#############################################################################
##
#F  SwapFactors(i,j,dd,n,F)  . . set pi_ij to be a d=dd^n by d matrix over F
## that swaps the i-th and j-th factors in the tensor product of n
## dd-dimensional spaces. We assume that i<j.
##
## 
SwapFactors := function ( i,j,dd,n,F) 

   local  d,k,l,A,B,C,a,b,c,Mb,Mc,r,s,Mr,Ms,pi_ij;

   d := dd^n;
   pi_ij:= [];
   for k in [1..d] do pi_ij[k]:= []; od;
   for k in [1..d] do
     for l in [1..d] do
       pi_ij[k][l]:= F.zero;
     od;
   od;

   Mb := dd^i;
   Mc := dd^j;
   A := dd^(i-1)-1;
   B := dd^(j-i-1)-1;
   C := dd^(n-j)-1;
   Mr := dd^(i-1);
   Ms := dd^(j-1);

   for a in [0..A] do
     for b in [0..B] do
       for c in [0..C] do
         for r in [0..dd-1] do
           for s in [0..dd-1] do
             k := a + Mr *r + Mb*b + Ms*s + Mc*c + 1;
             l := a + Mr *s + Mb*b + Ms*r + Mc*c + 1;
             pi_ij[k][l] := F.one; pi_ij[l][k] := F.one;
           od;
         od;
       od;
     od;
   od;    
   return pi_ij;

end;
