

/***********************************************************************

   This software is for research and educational purposes only.

************************************************************************/



#ifndef ZZX__H
#define ZZX__H

#include "vec_ZZ.h"

#include "zz_pX.h"

#include "ZZ_pX.h"


class ZZX {

public:

vector(ZZ) rep;

private:

static ZZX _zero;

public:

/***************************************************************

          Constructors, Destructors, and Assignment

****************************************************************/


ZZX()
//  initial value 0

   { }


ZZX(INIT_SIZE_TYPE, long n) 
// initial value 0, but space is pre-allocated for n coefficients

   { rep.SetMaxLength(n); }

ZZX(const ZZX& a) : rep(a.rep) { }
// initial value is a

ZZX(INIT_VAL_TYPE, long);
ZZX(INIT_VAL_TYPE, const ZZ&);



void operator=(const ZZX& a) 
   { rep = a.rep; }

~ZZX() { }

void normalize();
// strip leading zeros

void SetMaxLength(long n) 
// pre-allocate space for n coefficients.
// Value is unchanged

   { rep.SetMaxLength(n); }


void kill() 
// free space held by this polynomial.  Value becomes 0.

   { rep.kill(); }

static const ZZX& zero() { return _zero; }

};




/********************************************************************

                           input and output

I/O format:

   [a_0 a_1 ... a_n],

represents the polynomial a_0 + a_1*X + ... + a_n*X^n.

On output, all coefficients will be integers between 0 and p-1,
amd a_n not zero (the zero polynomial is [ ]).
Leading zeroes are stripped.

*********************************************************************/


istream& operator>>(istream& s, ZZX& x);
ostream& operator<<(ostream& s, const ZZX& a);




/**********************************************************

                   Some utility routines

***********************************************************/


inline long deg(const ZZX& a) { return a.rep.length() - 1; }
// degree of a polynomial.
// note that the zero polynomial has degree -1.

const ZZ& coeff(const ZZX& a, long i);
// zero if i not in range

void GetCoeff(ZZ& x, const ZZX& a, long i);
// x = a[i], or zero if i not in range

const ZZ& LeadCoeff(const ZZX& a);
// zero if a == 0

const ZZ& ConstTerm(const ZZX& a);
// zero if a == 0

void SetCoeff(ZZX& x, long i, const ZZ& a);
// x[i] = a, error is raised if i < 0

void SetCoeff(ZZX& x, long i);
// x[i] = 1, error is raised if i < 0

void SetX(ZZX& x);
// x is set to the monomial X

long IsX(const ZZX& a);
// test if x = X

inline void clear(ZZX& x) 
// x = 0

   { x.rep.SetLength(0); }

inline void set(ZZX& x)
// x = 1

   { x.rep.SetLength(1); set(x.rep[0]); }

inline void swap(ZZX& x, ZZX& y)
// swap x & y (only pointers are swapped)

   { swap(x.rep, y.rep); }

void trunc(ZZX& x, const ZZX& a, long m);
// x = a % X^m

void RightShift(ZZX& x, const ZZX& a, long n);
// x = a/X^n

void LeftShift(ZZX& x, const ZZX& a, long n);
// x = a*X^n

void diff(ZZX& x, const ZZX& a);
// x = derivative of a




/*******************************************************************

                        conversion routines

********************************************************************/



void operator<<(ZZX& x, long a);
void operator<<(ZZX& x, const ZZ& a);
void operator<<(ZZX& x, const vector(ZZ)& a);

void operator<<(zz_pX& x, const ZZX& a);
void operator<<(ZZX& x, const zz_pX& a);

void operator<<(ZZ_pX& x, const ZZX& a);
void operator<<(ZZX& x, const ZZ_pX& a);





/*************************************************************

                        Comparison

**************************************************************/

long IsZero(const ZZX& a); 

long IsOne(const ZZX& a);

long operator==(const ZZX& a, const ZZX& b);

long operator!=(const ZZX& a, const ZZX& b);





/***************************************************************

                         Addition

****************************************************************/

void add(ZZX& x, const ZZX& a, const ZZX& b);
// x = a + b

void sub(ZZX& x, const ZZX& a, const ZZX& b);
// x = a - b

void negate(ZZX& x, const ZZX& a);
// x = -a

// scalar versions

void add(ZZX & x, const ZZX& a, const ZZ& b); // x = a + b
void add(ZZX& x, const ZZX& a, long b); // x = a + b
void add(ZZX& x, const ZZ& a, const ZZX& b); // x = a + b
void add(ZZX& x, long a, const ZZX& b); // x = a + b

void sub(ZZX & x, const ZZX& a, const ZZ& b); // x = a - b
void sub(ZZX& x, const ZZX& a, long b); // x = a - b
void sub(ZZX& x, const ZZ& a, const ZZX& b); // x = a - b
void sub(ZZX& x, long a, const ZZX& b); // x = a - b


/*****************************************************************

                        Multiplication

******************************************************************/


void mul(ZZX& x, const ZZX& a, const ZZX& b);
// x = a * b

void sqr(ZZX& x, const ZZX& a);
// x = a^2

void PlainMul(ZZX& x, const ZZX& a, const ZZX& b);
void PlainSqr(ZZX& x, const ZZX& a);

void KarMul(ZZX& x, const ZZX& a, const ZZX& b);
void KarSqr(ZZX& x, const ZZX& a);

void HomMul(ZZX& x, const ZZX& a, const ZZX& b);
void HomSqr(ZZX& x, const ZZX& a);

void SSMul(ZZX& x, const ZZX& a, const ZZX& b);
void SSSqr(ZZX& x, const ZZX& a);

void mul(ZZX & x, const ZZX& a, const ZZ& b); // x = a * b
void mul(ZZX& x, const ZZX& a, long b); // x = a * b
void mul(ZZX& x, const ZZ& a, const ZZX& b); // x = a * b
void mul(ZZX& x, long a, const ZZX& b); // x = a * b







/*************************************************************

                      Division

**************************************************************/



// "plain" versions
void PlainDivRem(ZZX& q, ZZX& r, const ZZX& a, const ZZX& b);
void PlainDiv(ZZX& q, const ZZX& a, const ZZX& b);
void PlainRem(ZZX& r, const ZZX& a, const ZZX& b);

// "homomorphic imaging" versions
void HomDivRem(ZZX& q, ZZX& r, const ZZX& a, const ZZX& b);
void HomDiv(ZZX& q, const ZZX& a, const ZZX& b);
void HomRem(ZZX& r, const ZZX& a, const ZZX& b);

inline void DivRem(ZZX& q, ZZX& r, const ZZX& a, const ZZX& b)
// performs pseudo-division:  computes q and r
// with deg(r) < deg(b), and LeadCoeff(b)^(deg(a)-deg(b)+1) a = b q + r.
// current implementation always defaults to "plain"

   { PlainDivRem(q, r, a, b); }

inline void div(ZZX& q, const ZZX& a, const ZZX& b)

   { PlainDiv(q, a, b); }

inline void rem(ZZX& r, const ZZX& a, const ZZX& b)

   { PlainRem(r, a, b); }


// Modular arithemtic---f must be monic, and other args
// must have degree less than that of f
void MulMod(ZZX& x, const ZZX& a, const ZZX& b, const ZZX& f);
void SqrMod(ZZX& x, const ZZX& a, const ZZX& f);
void MulByXMod(ZZX& x, const ZZX& a, const ZZX& f);


// these always use "plain" division
long PlainDivide(ZZX& q, const ZZX& a, const ZZX& b);
long PlainDivide(const ZZX& a, const ZZX& b);

// these always use "homomorphic imaging"
long HomDivide(ZZX& q, const ZZX& a, const ZZX& b);
long HomDivide(const ZZX& a, const ZZX& b);

inline long divide(ZZX& q, const ZZX& a, const ZZX& b)
// if b | a, sets q = a/b and returns 1; otherwise returns 0
// current implementation always defaults to HomDivide

   { return HomDivide(q, a, b); }

inline long divide(const ZZX& a, const ZZX& b)
// if b | a, returns 1; otherwise returns 0
// current implementation always defaults to HomDivide

   { return HomDivide(a, b); }



long divide(ZZX& q, const ZZX& a, const ZZ& b);
// if b | a, sets q = a/b and returns 1; otherwise returns 0

long divide(const ZZX& a, const ZZ& b);
// if b | a, returns 1; otherwise returns 0

//single-precision versions
long divide(ZZX& q, const ZZX& a, long b);
long divide(const ZZX& a, long b);




void content(ZZ& d, const ZZX& f);
// d = content of f, sign(d) == sign(LeadCoeff(f))

void PrimitivePart(ZZX& pp, const ZZX& f);
// pp = primitive part of f, LeadCoeff(pp) >= 0

void GCD(ZZX& d, const ZZX& a, const ZZX& b);
// d = gcd(a, b), LeadCoeff(d) >= 0

long MaxBits(const ZZX& f);
// returns max NumBits of coefficients of f

long CharPolyBound(const ZZX& a, const ZZX& f);



/***************************************************************

                      traces, norms, resultants

****************************************************************/

void TraceVec(vector(ZZ)& S, const ZZX& f);
// S[i] = Trace(X^i mod f), for i = 0..deg(f)-1.
// f must be a monic polynomial.

void trace(ZZ& res, const ZZX& a, const ZZX& f);
// res = trace of (a mod f)
// f must be monic


void resultant(ZZ& res, const ZZX& a, const ZZX& b, long deterministic=0);
// res = resultant of a and b
// if !deterministic, then it may use a randomized strategy
//    that errs with probability no more than 2^{-80}.

void norm(ZZ& res, const ZZX& a, const ZZX& f, long deterministic=0);
// res = norm of (a mod f)
// f must be monic
// if !deterministic, then it may use a randomized strategy
//    that errs with probability no more than 2^{-80}.


void discriminant(ZZ& d, const ZZX& a, long deterministic=0);
// d = discriminant of a
//   = (-1)^{m(m-1)/2} resultant(a, a')/lc(a),
//     where m = deg(a)
// if !deterministic, then it may use a randomized strategy
//    that errs with probability no more than 2^{-80}.


void CharPoly(ZZX& g, const ZZX& a, const ZZX& f, long deterministic=0);
// g = char poly of (a mod f)
// f must be monic
// if !deterministic, then it may use a randomized strategy
//    that errs with probability no more than 2^{-80}.


void MinPoly(ZZX& g, const ZZX& a, const ZZX& f);
// g = min poly of (a mod f)
// f must be monic
// may use a probabilistic strategy that errs with
//   probability no more than 2^{-80}

void XGCD(ZZ& r, ZZX& s, ZZX& t, const ZZX& a, const ZZX& b, 
          long deterministic=0);
// r = resultant of a and b;
// if r != 0, then computes s and t such that:
//   a*s + b*t = r;
// otherwise s and t not affected.
// if !deterministic, then resultant computation may use a randomized strategy
//    that errs with probability no more than 2^{-80}.



/******************************************************

      Incremental Chinese Remaindering

*******************************************************/

long CRT(ZZX& a, ZZ& prod, const zz_pX& A);
long CRT(ZZX& a, ZZ& prod, const ZZ_pX& A);
// If p is the current modulus with (p, prod) = 1;
// Computes b such that b = a mod prod and b = A mod p,
//    with coefficients in the interval (-p*prod/2, p*prod/2];
// Sets a = b, prod = p*prod, and returns 1 if a's value changed.




// vectors

vector_decl(ZZX)

vector_eq_decl(ZZX)

vector_io_decl(ZZX)


#endif
