;;; -*- Mode:Lisp; Package:User; Base:10; Lowercase:T; Syntax:Common-Lisp -*-
;;; ===========================================================================
;;;			    System Maintenance
;;; ===========================================================================
;;; (c) Copyright 1989, 1991 Cornell University

;;; $Id: maintenance.lisp,v 1.18 1993/05/16 19:47:20 rz Exp $

(in-package #-CCL "USER" #+CCL "CL-USER")

(defun load-weyl-version ()
  (cond ((probe-file *weyl-version-file*)
	 (load *weyl-version-file* :verbose nil))
	(t (setq *weyl-version* (list 1 0)))))

(defun dump-weyl-version ()
  (with-open-file (file *weyl-version-file*
			:direction :output
			:if-exists :supersede)
    (format file ";; Weyl version ~D.~D~%~
                  (in-package #-CCL \"USER\" #+CCL \"CL-USER\")~
                  (setq *weyl-version* (list ~D ~D))~3%"
	    (first *weyl-version*) (second *weyl-version*)
	    (first *weyl-version*) (second *weyl-version*))))

(defun next-major-weyl-version ()
  (load-weyl-version)
  (incf (first *weyl-version*))
  (setf (second *weyl-version*) 0)
  (dump-weyl-version))

(defun next-minor-weyl-version ()
  (load-weyl-version)
  (incf (second *weyl-version*))
  (dump-weyl-version))

(defun ensure-sufficient-memory-for-weyl ()
  #+Lucid
  (declare (special lucid::*external-growth-limit*))
  #+Lucid
  (when (> 1024. lucid::*external-growth-limit*)
    (format t "~%;;; Increasing memory growth limit to 1024 segements, 64MB~%")
    (change-memory-management :growth-limit 1024.)))

(defun load-weyl ()
  (ensure-sufficient-memory-for-weyl)
  (defsys::load-system 'weyl)
  (load-weyl-version)
  (pushnew :weyl *features*)
  (funcall (intern "INITIALIZE-CONTEXTS" 'weyli))
  (funcall (intern "RESET-DOMAINS" 'weyli))
  (format t ";;; Weyl ~D.~D loaded. ~%"
	  (first *weyl-version*) (second *weyl-version*))
  (values))

(defun compile-weyl ()
  (ensure-sufficient-memory-for-weyl)
  (flet ((compile-if-necessary (file)
	   (let ((src (make-pathname
		       :name file
		       :type (first (eval (intern "*SUFFIXES*" 'defsys)))
		       :device (pathname-device *weyl-directory*)
		       :directory (pathname-directory *weyl-directory*)))
		 (obj (make-pathname
		       :name file
		       :type (rest (eval (intern "*SUFFIXES*" 'defsys)))
		       :device (pathname-device *weyl-directory*)
		       :directory (pathname-directory *weyl-directory*))))
	     (when (or (not (probe-file obj))
		       (> (file-write-date src) (file-write-date obj)))
	       (compile-file src)
	       (load obj)))))
    (compile-if-necessary "sysdef")
    (compile-if-necessary "defsystem")
    (compile-if-necessary "maintenance")
    (defsys:compile-system 'weyl)
    (next-minor-weyl-version)))

#+Lucid
(defun dump-weyl (&optional (name "weyl"))
  (load-weyl)
  (multiple-value-bind (seconds minutes hour date month year d-o-w d-s-t t-z)
      (decode-universal-time (get-universal-time))
    (declare (ignore seconds minutes hour d-o-w d-s-t t-z))
    (let ((file (make-pathname
		 :name (cond ((member :mips *features*) "weyl-mips")
			     ((member :sparc *features*) "weyl-sparc")
			     (t "Weyl-Unknown"))
		 :directory (pathname-directory *weyl-directory*)))
	  (archive (format nil
			   (cond ((member :mips *features*)
				  "~A/~A-mips-~D-~D-~D-~D")
				 ((member :sparc *features*)
				  "~A/~A-sparc-~D-~D-~D-~D")
				 (t "~A/~A-unknown-~D-~D-~D-~D"))
			   *weyl-archive-directory*
			   name month date year
			   (+ minutes (* 100 hour))))
	  (banner (weyl-banner)))
      (declare (special system::*enter-top-level-hook*))
      (when (probe-file file)
	(delete-file file))
      ;; Comment the following line to store binaries in the source directory.
      (user::shell (format nil "ln -s ~A ~A" archive file))
      (setq system::*enter-top-level-hook* 
	    #'(lambda ()
		(format t ";;; ~A~2%"  banner)
		(lucid::default-enter-top-level-hook)))
      (disksave file :full-gc t)
      (format t ";;; Weyl ~D.~D successfully dumped into ~A~%~
                 ;;; and link was created to it from ~A" 
	      (first *weyl-version*) (second *weyl-version*)
	      archive file))))

#+Lucid
(defun weyl-banner ()
  (multiple-value-bind (second minute hour date month year day-of-week)
      (decode-universal-time (get-universal-time))
    (declare (ignore second))
    (format nil "Weyl Version ~D.~D, saved ~2D:~2D ~A, ~A ~D, ~D"
	    (first *weyl-version*) (second *weyl-version*)
	    hour minute
	    (second (assoc day-of-week
			   '((0 "Monday") (1 "Tuesday") (2 "Wednesday")
			     (3 "Thursday") (4 "Friday") (5 "Saturday")
			     (6 "Sunday"))))
	    (second (assoc month
			   '((1 "January") (2 "February") (3 "March")
			     (4 "April") (5 "May") (6 "June") (7 "July")
			     (8 "August") (9 "September") (10 "October")
			     (11 "November") (12 "December"))))
	    date
	    year)))

#+CCL
(defun dump-weyl (&optional (name "Weyl"))
  (load-weyl)
  (let ((file (make-pathname
               :name (format nil "~A ~Db~D" 
                             name
                             (first *weyl-version*) (second *weyl-version*))
               :directory (pathname-directory (user-homedir-pathname)))))
    (when (probe-file file)
      (delete-file file))
    (format t ";;; Weyl ~D.~D successfully being dumped into ~A.~%"
            (first *weyl-version*) (second *weyl-version*) file)
    (save-application  file :init-file "init")))

(defsys:defsystem weyl
    (:default-pathname #.*weyl-directory*
     :default-package weyl)
  lisp-support
  (domain-support :load-after (lisp-support)
		  :load-before-compile (lisp-support))
  (algebraic-domains :load-after (domain-support)
		     :load-before-compile (domain-support))
  (avl :load-after (algebraic-domains)
       :load-before-compile (algebraic-domains))
  (lisp-numbers :load-after (algebraic-domains)
		:load-before-compile (algebraic-domains))
  (sets :load-after (algebraic-domains)
	:load-before-compile (algebraic-domains))
  (morphisms :load-after (algebraic-domains lisp-numbers)
	     :load-before-compile (algebraic-domains avl))
  (general :load-after (algebraic-domains)
	   :load-before-compile (algebraic-domains))
  (fourier :load-after (general)
	   :load-before-compile (general))
  (direct-sums :load-before-compile (algebraic-domains sets))
  (numbers :load-after (algebraic-domains bigfloat)
	   :load-before-compile (algebraic-domains bigfloat))
  #+ignore
  (rational-integers :load-after (algebraic-domains)
		     :load-before-compile (algebraic-domains))
  (gfp :load-after (algebraic-domains)
       :load-before-compile (algebraic-domains))
  (bigfloat :load-after (algebraic-domains lisp-numbers)
	    :load-before-compile (algebraic-domains lisp-numbers))
  #+ignore
  (real-numbers :load-after (rational-integers bigfloat)
		:load-before-compile (morphisms rational-integers bigfloat))
  #+ignore
  (complex-numbers :load-after (morphisms)
		   :load-before-compile (morphisms real-numbers))
  (quotient-fields :load-after (morphisms algebraic-domains)
		   :load-before-compile (morphisms algebraic-domains))
  #+ignore
  (rational-numbers :load-after (morphisms)
		    :load-before-compile (morphisms rational-integers))
  (poly-tools :load-after (morphisms general)
	      :load-before-compile (morphisms))
  (mpolynomial :load-after (poly-tools)
	       :load-before-compile (poly-tools))
  (upolynomial :load-after (poly-tools)
	       :load-before-compile (poly-tools))  
  (epolynomial :load-after (poly-tools)
	       :load-before-compile (poly-tools mpolynomial))
  (sparsegcd :load-after (poly-tools)
             :load-before-compile (poly-tools mpolynomial))
  (grobner :load-after (poly-tools)
	   :load-before-compile (poly-tools mpolynomial epolynomial))
  (rational-functions
     :load-after (general quotient-fields poly-tools mpolynomial)
     :load-before-compile (general quotient-fields poly-tools mpolynomial))
  (differential-domains :load-after (mpolynomial)
			:load-before-compile (mpolynomial))
  (algebraic-extension :load-after (mpolynomial)
		       :load-before-compile (mpolynomial))
  (vector :load-after (sets)
	  :load-before-compile (sets))
  (projective-space :load-after (vector)
		    :load-before-compile (vector))
  (quaternions :load-after (vector)
	       :load-before-compile (vector))
  (matrix :load-before-compile (morphisms))
  (topology :load-before-compile (avl mpolynomial vector)))

