;;; -*- Mode:Lisp; Package:Weyli; Base:10; Lowercase:T; Syntax:Common-Lisp -*-
;;; ===========================================================================
;;;			         Numbers
;;; ===========================================================================
;;; (c) Copyright 1991 Cornell University

;;; $Id: numbers.lisp,v 1.24 1993/05/11 15:11:44 rz Exp $

(in-package "WEYLI")

;; The Rational Integers  (Z)

(define-domain-element-classes rational-integers
    rational-integer)

(defmethod print-object ((d rational-integers) stream)
  #+Genera
  (format stream "~'bZ~")
  #-Genera
  (princ "Z"  stream))

(define-domain-creator rational-integers ()
  (make-instance 'rational-integers)
  :predicate (lambda (d) (eql (class-name (class-of d)) 'rational-integers)))

(defmethod initialize-instance :after ((d rational-integers) &rest plist)
  (declare (ignore plist))
  (with-slots (zero one) d
    (setq zero (make-element d 0))
    (setq one (make-element d 1))))


(defmethod print-object ((n rational-integer) stream)
  (princ (integer-value n) stream))


(defmethod make-element ((domain rational-integers) (x integer) &rest ignore)
  (declare (ignore ignore))
  (make-instance 'rational-integer :domain domain :value x))

(defmethod weyl:make-element ((domain rational-integers) (x integer)
			      &rest args)
  (when (not (null args))
    (error "Too many arguments to MAKE-ELEMENT of ~S: ~S"
	   domain (cons x args)))
  (make-instance 'rational-integer :domain domain :value x))


(defmethod coerce ((x integer) (domain rational-integers))
  (make-element domain x))

;; The Rational numbers  (Q)

(define-domain-element-classes rational-numbers
    rational-number rational-integer)

(defmethod print-object ((d rational-numbers) stream)
  #+Genera
  (format stream "~'bQ~")
  #-Genera
  (princ "Q"  stream))

(define-domain-creator rational-numbers ()
  (let ((domain (make-instance 'rational-numbers))
	(Z (get-rational-integers)))
    (make-homomorphism Z
		       (lambda (x)
			 (make-element domain (integer-value x)))
		       domain)
    domain)
  :predicate (lambda (d) (eql (class-name (class-of d)) 'rational-numbers)))

(define-domain-creator quotient-field ((ring rational-integers))
  (make-rational-numbers))

(defmethod get-quotient-field ((ring rational-integers))
  (get-rational-numbers))


(defmethod print-object ((ratfun rational-number) stream)
  (with-numerator-and-denominator (numerator denominator) ratfun
    (cond ((1? denominator)
	   (prin1 numerator stream))
	  (t (prin1 numerator stream)
	     (princ "/" stream)
	     (prin1 denominator stream)))))


(defmethod make-element ((qf rational-numbers) (num integer) &rest ignore)
  (declare (ignore ignore))
  (make-instance 'rational-integer :domain qf :value num))

(defmethod weyl:make-element ((qf rational-numbers) (num integer) &rest args)
  (when (not (null args))
    (error "Too many arguments to MAKE-ELEMENT of ~S: ~S"
	   qf (cons num args)))
  (make-instance 'rational-integer :domain qf :value num))

(defmethod coerce ((elt integer) (domain rational-numbers))
  (make-element domain elt))


(defmethod make-element ((qf rational-numbers) (num ratio) &rest ignore)
  (declare (ignore ignore))
  (make-instance 'rational-number :domain qf
		 :numerator (lisp:numerator num)
		 :denominator (lisp:denominator num)))

(defmethod weyl:make-element ((qf rational-numbers) (num ratio) &rest args)
  (when (not (null args))
    (error "Too many arguments to MAKE-ELEMENT of ~S: ~S"
	   qf (cons num args)))
  (make-instance 'rational-number :domain qf
		 :numerator (lisp:numerator num)
		 :denominator (lisp:denominator num)))

(defmethod coerce ((elt ratio) (domain rational-numbers))
  (make-element domain elt))


(defmethod make-quotient-element
    ((qf rational-numbers) (numerator integer) (denominator integer))
  (cond ((lisp:= denominator 1)
	 (make-instance 'rational-integer :domain qf :value numerator))
	(t (make-instance 'rational-number :domain qf
			  :numerator numerator
			  :denominator denominator))))

;; The Real numbers (R)

(define-domain-element-classes real-numbers
    floating-point-number bigfloat rational-integer rational-number)

(defmethod print-object ((d real-numbers) stream)
  #+Genera
  (format stream "~'bR~")
  #-Genera
  (princ "R"  stream))

;; For these two there is only one domain.  (We could also implement
;; several integer domains if we wanted later.)

(define-domain-creator real-numbers ()
  (let ((domain (make-instance 'real-numbers))
	(Q (get-rational-numbers)))
    (make-homomorphism Q
		       (lambda (x)
			 (make-element domain
				       (if (typep x 'rational-integer)
					   (integer-value x)
					   (lisp:/ (qo-numerator x)
						   (qo-denominator x)))))
		       domain)
    domain)
  :predicate (lambda (d) (eql (class-name (class-of d)) 'real-numbers)))
	
(defvar *floating-point-precision* 16.
  "Precision below which to use the machine floating point representation")

(defvar *real-precision* *floating-point-precision*
  "The default precision when creating a Real number")


(defmethod print-object ((z floating-point-number) stream)
  (format stream "~D" (fp-value z)))


(defmethod make-element ((domain real-numbers) (x integer) &rest ignore)
  (declare (ignore ignore))
  (make-instance 'rational-integer :domain domain :value x))

(defmethod weyl:make-element ((domain real-numbers) (x integer) &rest args)
  (when (not (null args))
    (error "Too many arguments to MAKE-ELEMENT of ~S: ~S"
	   domain (cons x args)))
  (make-instance 'rational-integer :domain domain :value x))

(defmethod coerce ((elt integer) (domain real-numbers))
  (make-element domain elt))


(defmethod make-element ((domain real-numbers) (x ratio) &rest ignore)
  (declare (ignore ignore))
  (make-instance 'rational-number :domain domain
		 :numerator (lisp:numerator x)
		 :denominator (lisp:denominator x)))

(defmethod weyl:make-element ((domain real-numbers) (x ratio) &rest args)
  (when (not (null args))
    (error "Too many arguments to MAKE-ELEMENT of ~S: ~S"
	   domain (cons x args)))
  (make-instance 'rational-number :domain domain
		 :numerator (lisp:numerator x)
		 :denominator (lisp:denominator x)))

(defmethod coerce ((elt ratio) (domain real-numbers))
  (make-element domain elt))

;; The following method is needed so that routines that return ratios
;; will work.
(defmethod make-quotient-element
    ((domain real-numbers) (numerator integer) (denominator integer))
  (cond ((lisp:= denominator 1)
	 (make-instance 'rational-integer :domain domain :value numerator))
	(t (make-instance 'rational-number :domain domain
			  :numerator numerator
			  :denominator denominator))))


(defmethod make-element ((domain real-numbers) (x float) &rest ignore)
  (declare (ignore ignore))
  (make-instance 'floating-point-number :domain domain
		 :value x))

(defmethod weyl:make-element ((domain real-numbers) (x float) &rest args)
  (when (not (null args))
    (error "Too many arguments to MAKE-ELEMENT of ~S: ~S"
	   domain (cons x args)))
  (make-instance 'floating-point-number :domain domain
		 :value x))

(defmethod coerce ((elt float) (domain real-numbers))
  (make-element domain elt))


(defmethod zero ((domain real-numbers))
  (weyl:make-element domain 0))

(defmethod one ((domain real-numbers))
  (weyl:make-element domain 1))

;; The Complex numbers (C)

(define-domain-element-classes complex-numbers
    complex-number)

(defmethod print-object ((d complex-numbers) stream)
  #+Genera
  (format stream "~'bC~")
  #-Genera
  (princ "C"  stream))

;; For these two there is only one domain.  (We could also implement
;; several integer domains if we wanted later.)

(define-domain-creator complex-numbers ()
  (let ((domain (make-instance 'complex-numbers))
	(R (get-real-numbers)))
    (make-homomorphism R
		       (lambda (x)
			 (make-element domain
				       (cond ((typep x 'rational-integer)
					      (integer-value x))
					     ((typep x 'rational-number)
					      (lisp:/ (qo-numerator x)
						      (qo-denominator x)))
					     ((typep x 'floating-point-number)
					      (fp-value x))
					     (t x))))
		       domain)
    domain)
  :predicate (lambda (d) (eql (class-name (class-of d)) 'complex-numbers)))
	

(defmethod print-object ((z complex-number) stream)
  (with-slots ((x real) (y imag)) z
    (cond ((0? y)
	   (princ x stream))
	  ((0? x)
	   (if (1? y)
	       (format stream "i")
	       (format stream "~S i" y)))
	  (t (if (1? y)
		 (format stream "~S + i" x)
		 (format stream "~S + ~S i" x y))))))


(defmethod make-element ((domain complex-numbers) (x integer) &rest args)
  (if (or (null args) (0? (first args)))
      (make-instance 'rational-integer :domain domain :value x)
      (make-instance 'complex-number :domain domain
		     :realpart x
		     :imagpart (first args))))

(defmethod weyl:make-element ((domain complex-numbers) (x integer) &rest args)
  (cond ((or (null args)
	     (and (0? (first args)) (null (rest args))))
	 (make-instance 'rational-integer :domain domain :value x))
	((null (rest args))
	 (if (and (lisp:numberp (first args))
		  (not (lisp:complexp (first args))))
	     (make-instance 'complex-number :domain domain
			    :realpart x
			    :imagpart (first args))
	     (error "Wrong type of arguments to MAKE-ELEMENT of ~S: ~S"
		    domain (cons x args))))
	(t (error "Too many arguments to MAKE-ELEMENT of ~S: ~S"
		  domain (cons x args)))))
  

(defmethod make-element ((domain complex-numbers) (x ratio) &rest args)
  (if (or (null args) (0? (first args)))
      (make-instance 'rational-number :domain domain
		     :numerator (lisp:numerator x)
		     :denominator (lisp:denominator x))
      (make-instance 'complex-number :domain domain
		     :realpart x
		     :imagpart (first args))))

(defmethod weyl:make-element ((domain complex-numbers) (x ratio) &rest args)
  (cond ((or (null args)
	     (and (0? (first args))
		  (null (rest args))))
	 (make-instance 'rational-number :domain domain
			:numerator (lisp:numerator x)
			:denominator (lisp:denominator x)))
	((null (rest args))
	 (if (and (lisp:numberp (first args))
		  (not (lisp:complexp (first args))))
	     (make-instance 'complex-number :domain domain
			    :realpart x
			    :imagpart (first args))
	     (error "Wrong type of arguments to MAKE-ELEMENT of ~S: ~S"
		    domain (cons x args))))
	(t (error "Too many arguments to MAKE-ELEMENT of ~S: ~S"
		  domain (cons x args)))))


(defmethod make-element ((domain complex-numbers) (x float) &rest args)
  (if (or (null args) (0? (first args)))
      (make-instance 'floating-point-number :domain domain
		     :value x)
      (make-instance 'complex-number :domain domain
		     :realpart x
		     :imagpart (first args))))

(defmethod weyl:make-element ((domain complex-numbers) (x float) &rest args)
  (cond ((or (null args)
	     (and (0? (first args))
		  (null (rest args))))
	 (make-instance 'floating-point-number :domain domain
			:value x))
	((null (rest args))	 
	 (if (and (lisp:numberp (first args))
		  (not (lisp:complexp (first args))))
	     (make-instance 'complex-number :domain domain
			    :realpart x
			    :imagpart (first args))
	     (error "Wrong type of arguments to MAKE-ELEMENT of ~S: ~S"
		    domain (cons x args))))
	(t (error "Too many arguments to MAKE-ELEMENT of ~S: ~S"
		  domain (cons x args)))))

(defmethod make-element
    ((domain complex-numbers) (x lisp:complex) &rest ignore)
  (declare (ignore ignore))
  (let ((real (lisp:realpart x))
	(imag (lisp:imagpart x)))
    (if (0? imag)
	(make-element domain real)
	(make-instance 'complex-number :domain domain
		       :realpart real :imagpart imag))))

(defmethod weyl:make-element
    ((domain complex-numbers) (x lisp:complex) &rest args)
  (cond ((null args)
	 (make-element domain x))
	(t (error "Too many arguments to MAKE-ELEMENT of ~S: ~S"
		  domain (cons x args)))))

(defmethod realpart ((x number))
  (lisp:realpart x))

(defmethod realpart ((x rational-integer))
  (cond ((or (typep (domain-of x) 'complex-numbers)
	     (typep (domain-of x) 'non-strict-domain))
	 (make-element (get-real-numbers) (integer-value x)))
	(t (error "Don't know what \"realpart\" means in ~S"
		  (domain-of x)))))

(defmethod realpart ((x rational-number))
  (cond ((or (typep (domain-of x) 'complex-numbers)	     
	     (typep (domain-of x) 'non-strict-domain))
	 (make-element (get-real-numbers)
		       (lisp:/ (qo-numerator x) (qo-denominator x))))
	(t (error "Don't know what \"realpart\" means in ~S"
		  (domain-of x)))))

(defmethod realpart ((x floating-point-number))
  (cond ((or (typep (domain-of x) 'complex-numbers)
	     (typep (domain-of x) 'non-strict-domain))
	 (make-element (get-real-numbers) (fp-value x)))
	(t (error "Don't know what \"realpart\" means in ~S"
		  (domain-of x)))))

(defmethod realpart ((x bigfloat))
  (cond ((or (typep (domain-of x) 'complex-numbers)
	     (typep (domain-of x) 'non-strict-domain))
	 (make-bigfloat (get-real-numbers)
			(bigfloat-mantissa x)
			(bigfloat-exponent x)))
	(t (error "Don't know what \"realpart\" means in ~S"
		  (domain-of x)))))

(defmethod realpart ((x complex-number))
  (let ((real (cn-realpart x)))
    (cond ((typep real 'number)
	   (make-element (get-real-numbers) real))
	  (t (make-bigfloat (get-real-numbers)
			    (bigfloat-mantissa real)
			    (bigfloat-exponent real))))))

(defmethod imagpart ((x number))
  (lisp:imagpart x))

(defmethod imagpart ((x rational-integer))
  (cond ((or (typep (domain-of x) 'complex-numbers)
	     (typep (domain-of x) 'non-strict-domain))
	 (zero (get-real-numbers)))
	(t (error "Don't know what \"imagpart\" means in ~S"
		  (domain-of x)))))

(defmethod imagpart ((x rational-number))
  (cond ((or (typep (domain-of x) 'complex-numbers)
	     (typep (domain-of x) 'non-strict-domain))
	 (zero (get-real-numbers)))
	(t (error "Don't know what \"imagpart\" means in ~S"
		  (domain-of x)))))

(defmethod imagpart ((x floating-point-number))
  (cond ((or (typep (domain-of x) 'complex-numbers)
	     (typep (domain-of x) 'non-strict-domain))
	 (zero (get-real-numbers)))
	(t (error "Don't know what \"imagpart\" means in ~S"
		  (domain-of x)))))

(defmethod imagpart ((x bigfloat))
  (cond ((or (typep (domain-of x) 'complex-numbers)
	     (typep (domain-of x) 'non-strict-domain))
	 (zero (get-real-numbers)))
	(t (error "Don't know what \"imagpart\" means in ~S"
		  (domain-of x)))))

(defmethod imagpart ((x complex-number))
  (let ((imag (cn-imagpart x)))
    (cond ((typep imag 'number)
	   (make-element (get-real-numbers) imag))
	  (t (make-bigfloat (get-real-numbers)
			    (bigfloat-mantissa imag)
			    (bigfloat-exponent imag))))))

(defmethod conjugate  ((x number))
  (lisp:conjugate x))

(defmethod conjugate ((x rational-integer))
  (cond ((or (typep (domain-of x) 'complex-numbers)
	     (typep (domain-of x) 'non-strict-domain))
	 x)
	(t (error "Don't know what \"conjugate\" means in ~S"
		  (domain-of x)))))

(defmethod conjugate ((x rational-number))
  (cond ((or (typep (domain-of x) 'complex-numbers)
	     (typep (domain-of x) 'non-strict-domain))
	 x)
	(t (error "Don't know what \"conjugate\" means in ~S"
		  (domain-of x)))))

(defmethod conjugate ((x floating-point-number))
  (cond ((or (typep (domain-of x) 'complex-numbers)
	     (typep (domain-of x) 'non-strict-domain))
	 x)
	(t (error "Don't know what \"conjugate\" means in ~S"
		  (domain-of x)))))

(defmethod conjugate ((x bigfloat))
  (cond ((or (typep (domain-of x) 'complex-numbers)
	     (typep (domain-of x) 'non-strict-domain))
	 x)
	(t (error "Don't know what \"conjugate\" means in ~S"
		  (domain-of x)))))

(defmethod conjugate ((x complex-number))
  (make-instance 'complex-number :domain (domain-of x)
		 :realpart (cn-realpart x)
		 :imagpart (- (cn-imagpart x))))

(defmethod abs ((x number))
  (lisp:abs x))

(defmethod abs ((x rational-integer))
  (make-element (domain-of x) (lisp:abs (integer-value x))))

(defmethod abs ((x rational-number))
  (make-instance 'rational-number
		 :domain (domain-of x)
		 :numerator (abs (qo-numerator x))     
		 :denominator (qo-denominator x)))

(defmethod abs ((z floating-point-number))
  (make-element (domain-of z) (lisp:abs (fp-value z))))

(defmethod abs ((number bigfloat))
  (bind-domain-context (domain-of number)
    (bf-abs number)))

(defmethod abs ((z complex-number))
  (let ((x (cn-realpart z))
	(y (cn-imagpart z)))
    (make-element (domain-of z) (sqrt (+ (* x x) (* y y))))))

(defmethod phase ((x number))
  (lisp:phase x))

(defmethod phase ((x rational-integer))
  (zero (domain-of x)))

(defmethod phase ((x rational-number))
  (zero (domain-of x)))

(defmethod phase ((z floating-point-number))
  (zero (domain-of z)))

(defmethod phase ((number bigfloat))
  (zero (domain-of number)))

(defmethod phase ((z complex-number))
  (let ((x (cn-realpart z))
	(y (cn-imagpart z)))
    (make-element (domain-of z) (atan y x))))

(defmethod random-constant ((domain numeric-domain) &optional height)
  (random domain height))

(defvar *default-random-height* (lisp:expt 10 9))

(defmethod random ((domain rational-integers) 
                   &optional (height *default-random-height*))
  (make-element domain (lisp:random height)))

(defmethod random ((domain rational-numbers) 
                   &optional (height *default-random-height*))
  (make-element domain (/ (if (zerop (lisp:random 2))
                              (lisp:random height)
                              (lisp:- (lisp:random height)
                          (lisp:random height))))))

(defun random-floating-number (height)
  (let ((num (lisp:+ (float (lisp:random height))
                     (lisp:/ (float (lisp:random height))
                             (float (lisp:random height))))))
    (if (zerop (lisp:random 2)) num (lisp:- num))))

(defmethod random ((domain real-numbers) 
                   &optional (height *default-random-height*))
  (make-element domain (random-floating-number height)))

(defmethod random ((domain complex-numbers)
                   &optional (height *default-random-height*))
  (make-instance 'complex-number :domain domain
                 :realpart (random-floating-number height)
                 :imagpart (random-floating-number height)))

(defmethod height ((x number))
  (lisp:abs x))

(defmethod height ((x rational-integer))
  (make-element (get-real-numbers) (lisp:abs (integer-value x))))

(defmethod height ((x rational-number))
  (make-element (get-real-numbers) (lisp:max (lisp:abs (qo-numerator x))     
                                             (qo-denominator x))))

(defmethod height ((z floating-point-number))
  (make-element (get-real-numbers) (lisp:abs (fp-value z))))

;; FIXTHIS I think this is buggy!
(defmethod height ((number bigfloat))
  (bind-domain-context (domain-of number)
    (bf-abs number)))

(defmethod height ((z complex-number))
  (let ((x (cn-realpart z))
	(y (cn-imagpart z)))
    (make-element (get-real-numbers) (lisp:max (lisp:abs x) (lisp:abs y)))))

(defmethod convert-to-lisp-number ((x number))
  x)

(defmethod convert-to-lisp-number ((x rational-integer))
  (integer-value x))

(defmethod convert-to-lisp-number ((x rational-number))
  (lisp:/ (qo-numerator x) (qo-denominator x)))

(defmethod convert-to-lisp-number ((x floating-point-number))
  (fp-value x))

(defmethod convert-to-lisp-number ((x bigfloat))
  x)

(defun parse-numeric-obj (num)
  ;;(declare (values num type domain))
  (cond ((typep num 'number)
	 (values num
		 (if (typep num 'integer) 'integer (lisp:type-of num))
		 nil))
	((typep num 'rational-integer)
	 (values (integer-value num) 'rational-integer (domain-of num)))
	((typep num 'rational-number)
	 (values (lisp:/ (qo-numerator num)
			 (qo-denominaTor num))
		 'rational-number
		 (domain-of num)))
	((typep num 'numeric)
	 (values num (class-name (class-of num)) (domain-of num)))
	(t (error "~S is not a numeric object" num))))

(defmethod numerator ((n rational-integer))
  (cond ((or (typep (domain-of n) 'field)
	     (typep (domain-of n) 'non-strict-domain))
	 n)
	(t (error "Don't know what \"numerator\" means in ~S"
		  (domain-of n)))))

(defmethod denominator ((n rational-integer))
  (cond ((or (typep (domain-of n) 'field)
	     (typep (domain-of n) 'non-strict-domain))
	 (one (domain-of n)))
	(t (error "Don't know what \"denominator\" means in ~S"
		  (domain-of n)))))

(defmethod 0? (x)
  (declare (ignore x))
  nil)

(defmethod 0? ((x number))
  (lisp:zerop x))

(defmethod 0? ((x rational-integer))
  (lisp:zerop (integer-value x)))

(defmethod 0? ((x rational-number))
  nil)

(defmethod 0? ((x floating-point-number))
  (lisp:zerop (fp-value x)))

(defmethod 0? ((number bigfloat))
  (equal (bigfloat-mantissa number) 0))

(defmethod 0? ((x complex-number))
  (and (0? (realpart x)) (0? (imagpart x))))

(defmethod 1? (x)
  (declare (ignore x))
  nil)

(defmethod 1? ((x number))  
  (= x 1))

(defmethod 1? ((x rational-integer))  
  (eql (integer-value x) 1))

(defmethod 1? ((x rational-number))
  nil)

(defmethod 1? ((x floating-point-number))
  (lisp:= 1.0 (fp-value x)))

(defmethod 1? ((number bigfloat))
  (and (equal (bigfloat-mantissa number) 1)
       (eql (bigfloat-exponent number) 0)))

(defmethod 1? ((x complex-number))
  (and (1? (realpart x)) (0? (imagpart x))))

(defmethod minus ((x number))
  (lisp:- x))

(defmethod minus ((x rational-integer))
  (make-element (domain-of x) (lisp:- (integer-value x))))

(defmethod minus ((x rational-number))
  (make-quotient-element (domain-of x)
			 (lisp:- (qo-numerator x))
			 (qo-denominator x)))

(defmethod minus ((x floating-point-number))
  (make-element (domain-of x) (lisp:- (fp-value x))))

(defmethod minus ((number bigfloat))
  (bind-domain-context (domain-of number)
    (bf-minus number)))

(defmethod minus ((x complex-number))
  (make-element (domain-of x) (- (cn-realpart x)) (- (cn-imagpart x))))


(defmethod minus? ((x number))
  (lisp:minusp x))

(defmethod minus? ((x lisp:complex))
  nil)

(defmethod minus? ((x rational-integer))
  (lisp:minusp (integer-value x)))

(defmethod minus? ((x rational-number))
  (lisp:minusp (qo-numerator x)))

(defmethod minus? ((x floating-point-number))
  (lisp:minusp (fp-value x)))

(defmethod minus? ((x bigfloat))
  (lisp:minusp (bigfloat-mantissa x)))

(defmethod plus? ((x number))
  (lisp:plusp x))

(defmethod plus? ((x lisp:complex))
  (not (lisp:zerop x)))

(defmethod plus? ((x rational-integer))
  (lisp:plusp (integer-value x)))

(defmethod plus? ((x rational-number))
  (lisp:plusp (qo-numerator x)))

(defmethod plus? ((x floating-point-number))
  (lisp:plusp (fp-value x)))

(defmethod plus? ((x bigfloat))
  (lisp:plusp (bigfloat-mantissa x)))

(defmethod integer? ((x number))
  (lisp:integerp x))

(defmethod integer? ((x numeric))
  nil)

(defmethod integer? ((x rational-integer))
  t)

(defmethod recip ((x number))
  (lisp:/ x))

(defmethod recip ((x rational-integer))
  (let ((x-val (integer-value x))
	(domain (domain-of x)))
    (cond ((or (eql x-val 1) (eql x-val -1))
	   x)
	  ((typep domain 'field)
	   (make-element domain (lisp:/ 1 x-val)))
	  (t 
	   (error "Trying to take the reciprocal of the rational integer ~S"
		  x)))))

;; recip of a rational integer is covered by QUOTIENT-ELEMENT

(defmethod recip ((x floating-point-number))
  (when (0? x)
    (error "Error: Attempt take reciprocal of zero: ~S" x))
    (make-element (domain-of x) (lisp:/ (fp-value x))))

(defmethod recip ((z complex-number))
  (when (0? z)
    (error "Error: Attempt take reciprocal of zero: ~S" z))
  (let ((x (realpart z))
	(y (imagpart z))
	denom)
    (setq denom (+ (* x x) (* y y)))
    (make-element (domain-of z)
		  (convert-to-lisp-number (/ x denom))
		  (convert-to-lisp-number (/ (- y) denom)))))

(defmethod sqrt ((x number))
  (lisp:sqrt  x))

(defmethod sqrt ((x integer))
  (let* ((n (lisp:abs x))
	 (root (faster-isqrt n)))
    (unless (lisp:= n (lisp:* root root))
      (setq root (lisp:sqrt n)))
    (if (minus? x) (lisp:complex 0 root)
	root)))

(defmethod sqrt ((x rational-integer))
  (let ((domain (domain-of x)))
    (cond ((or (typep domain 'complete-set)
	       (typep domain 'non-strict-domain))
	   (make-element domain (lisp:sqrt (integer-value x))))
	  ((minus? (integer-value x))
	   (error "Can't take the sqrt of a negative number: ~S" x))
	  (t (let* ((n (integer-value x))
		    (root (faster-isqrt n)))
	       (cond ((lisp:= n (lisp:* root root))
		      (make-element domain root))
		     (t (error "~S does not have a sqrt in ~S"
			       x domain))))))))

(defmethod sqrt ((x rational-number))
  (let ((domain (domain-of x)))
    (cond ((or (typep domain 'complete-set)
	       (typep domain 'non-strict-domain))
	   (make-element domain (lisp:sqrt (lisp:/ (qo-numerator x)
						   (qo-denominator x)))))
	  (t (let* ((n (qo-numerator x))
		    (d (qo-denominator x))
		    (n-root (faster-isqrt n))
		    (d-root (faster-isqrt d)))
	       (cond ((and (lisp:= n (lisp:* n-root n-root))
			   (lisp:= d (lisp:* d-root d-root)))
		      (make-quotient-element domain n-root d-root))
		     (t (error "~S does not have a sqrt in ~S"
			       x domain))))))))

(defmethod sqrt ((x floating-point-number)) 
  (make-element (domain-of x) (lisp:sqrt (fp-value x))))

(defmethod sqrt ((number bigfloat))
  (bind-domain-context (domain-of number)
    (bf-sqrt number *REAL-PRECISION*)))

(defmethod sqrt ((number complex-number))
  (let* ((x (cn-realpart number))
	 (y (cn-imagpart number))
	 (mag (/ (sqrt (+ (* x x) (* y y))) 2)))
    (make-instance 'complex-number :domain (domain-of number)
		   :realpart (sqrt (+ mag (/ x 2)))
		   :imagpart (if (plus? y)
				 (sqrt (- mag (/ x 2)))
				 (- (sqrt (- mag (/ x 2))))))))

(defmethod binary= ((x number) (y number))
  (lisp:= x y))

(defmethod-sd binary= ((x rational-integer) (y rational-integer))
  (eql (integer-value x) (integer-value y)))

(defmethod-sd binary= ((x rational-number) (y rational-number))
  (and (lisp:= (qo-numerator x) (qo-denominator y))
       (lisp:= (qo-numerator y) (qo-denominator x))))

(defmethod-sd binary= ((x rational-number) (y rational-integer))
  nil)

(defmethod-sd binary= ((x rational-integer) (y rational-number))
  nil)

(defmethod-sd binary= ((x rational-integer) (y floating-point-number))
  nil)

(defmethod-sd binary= ((x floating-point-number) (y floating-point-number) )
  (lisp:= (fp-value x) (fp-value y)))

(defmethod binary= ((x float) (y floating-point-number))
  (lisp:= x (fp-value y)))

(defmethod binary=  ((x floating-point-number) (y float))
  (lisp:= (fp-value x) y))

(defmethod binary=  ((x float) (y bigfloat))
  (lisp:= x (lisp:* (lisp:float (bigfloat-mantissa y))
		    (lisp:expt 10.0 (bigfloat-exponent y)))))

(defmethod binary=  ((x bigfloat) (y float))
  (lisp:= (lisp:* (lisp:float (bigfloat-mantissa x))
		  (lisp:expt 10.0 (bigfloat-exponent x)))
	  y))

(defmethod-sd binary= ((x floating-point-number) (y bigfloat))
  (setq y (decprec! y *floating-point-precision*))
  (lisp:= (fp-value x)
	  (lisp:* (lisp:float (bigfloat-mantissa y))
		  (lisp:expt 10.0 (bigfloat-exponent y)))))

(defmethod-sd binary= ((x bigfloat) (y floating-point-number))
  (setq x (decprec! x *floating-point-precision*))
  (lisp:= (lisp:* (lisp:float (bigfloat-mantissa x))
		  (lisp:expt 10.0 (bigfloat-exponent x)))
	  (fp-value y)))

(defmethod-sd binary= ((x bigfloat) (y bigfloat))
  (bf-binary= x y))

(defmethod-sd binary= ((x complex-number) (y complex-number))
  (and (= (cn-realpart x) (cn-realpart y))
       (= (cn-imagpart x) (cn-imagpart y))))

(defmethod binary> ((x number) (y number))
  (lisp:> x y))

(defmethod-sd binary> ((x rational-integer) (y rational-integer))
  (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
      (lisp:> (integer-value x) (integer-value y))
      (call-next-method x y)))

(defmethod-sd binary> ((x rational-number) (y rational-number))
  (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
      (lisp:> (lisp:* (qo-numerator x) (qo-denominator y))
	      (lisp:* (qo-numerator y) (qo-denominator x)))
      (call-next-method x y)))

(defmethod-sd binary> ((x rational-number) (y rational-integer))
  (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
      (lisp:> (qo-numerator x) (lisp:* (integer-value y) (qo-denominator x)))      
      (call-next-method x y)))

(defmethod-sd binary> ((x rational-integer) (y rational-number))
  (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
      (lisp:> (lisp:* (integer-value x) (qo-denominator y)) (qo-numerator y))
      (call-next-method x y)))

(defmethod-sd binary> ((x floating-point-number) (y floating-point-number))
  (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
      (lisp:> (fp-value x) (fp-value y))
      (call-next-method x y)))

(defmethod binary> :around ((x float) (y floating-point-number))
  (let ((domain (domain-of y)))
    (if (or (typep domain 'ordered-set)
	    (typep domain 'non-strict-domain))
	(lisp:> x (fp-value y))
	(call-next-method x y))))

(defmethod binary> :around ((x floating-point-number) (y float))
  (let ((domain (domain-of x)))
    (if (or (typep domain 'ordered-set)
	    (typep domain 'non-strict-domain))
	(lisp:> (fp-value x) y)
	(call-next-method x y))))

(defmethod-sd binary> ((x floating-point-number) (y rational-integer))
  (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
      (lisp:> (fp-value x) (integer-value y))
      (call-next-method x y)))

(defmethod-sd binary> ((x rational-integer) (y floating-point-number))
  (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
      (lisp:> (integer-value x) (fp-value y))
      (call-next-method x y)))

(defmethod-sd binary> ((x floating-point-number) (y rational-number))
  (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
      (lisp:> (lisp:* (qo-denominator y) (fp-value x)) (qo-numerator y))
      (call-next-method x y)))

(defmethod-sd binary> ((x rational-number) (y floating-point-number))
  (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
      (lisp:> (qo-numerator x) (lisp:* (qo-denominator x) (fp-value y)))
      (call-next-method x y)))

(defmethod binary> :around ((x float) (y bigfloat))
  (let ((domain (domain-of y)))
    (if (or (typep domain 'ordered-set)
	    (typep domain 'non-strict-domain))
	(lisp:> x (lisp:* (lisp:float (bigfloat-mantissa y))
			  (lisp:expt 10.0 (bigfloat-exponent y))))
	(call-next-method x y))))

(defmethod binary> :around ((x bigfloat) (y float))
  (let ((domain (domain-of x)))
    (if (or (typep domain 'ordered-set)
	    (typep domain 'non-strict-domain))
	(lisp:> (lisp:* (lisp:float (bigfloat-mantissa x))
			(lisp:expt 10.0 (bigfloat-exponent x)))
		y)
	(call-next-method x y))))

(defmethod-sd binary> ((x floating-point-number) (y bigfloat))
  (cond ((or (typep domain 'ordered-set)
	     (typep domain 'non-strict-domain))
	 (setq y (decprec! y *floating-point-precision*))
	 (lisp:> (fp-value x)
		 (lisp:* (lisp:float (bigfloat-mantissa y))
			 (lisp:expt 10.0 (bigfloat-exponent y)))))
	(t (call-next-method x y))))

(defmethod-sd binary> ((x bigfloat) (y floating-point-number))
  (cond ((or (typep domain 'ordered-set)
	     (typep domain 'non-strict-domain))
	 (setq x (decprec! x *floating-point-precision*))
	 (lisp:> (lisp:* (lisp:float (bigfloat-mantissa x))
			 (lisp:expt 10.0 (bigfloat-exponent x)))
		 (fp-value y)))
	(t (call-next-method x y))))

(defmethod binary>= ((x number) (y number))
  (lisp:>= x y))

(defmethod-sd binary>= ((x rational-integer) (y rational-integer))
  (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
      (lisp:>= (integer-value x) (integer-value y))
      (call-next-method x y)))

(defmethod-sd binary>= ((x rational-number) (y rational-number))
  (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
      (lisp:>= (lisp:* (qo-numerator x) (qo-denominator y))
	      (lisp:* (qo-numerator y) (qo-denominator x)))
      (call-next-method x y)))

(defmethod-sd binary>= ((x rational-number) (y rational-integer))
  (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
      (lisp:>= (qo-numerator x) (lisp:* (integer-value y) (qo-denominator x)))      
      (call-next-method x y)))

(defmethod-sd binary>= ((x rational-integer) (y rational-number))
  (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
      (lisp:>= (lisp:* (integer-value x) (qo-denominator y)) (qo-numerator y))
      (call-next-method x y)))

(defmethod-sd binary>= ((x floating-point-number) (y floating-point-number))
  (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
      (lisp:>= (fp-value x) (fp-value y))
      (call-next-method x y)))

(defmethod binary>= :around  ((x float) (y floating-point-number))
  (let ((domain (domain-of y)))
    (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
	(lisp:>= x (fp-value y))
	(call-next-method x y))))

(defmethod binary>= :around  ((x floating-point-number) (y float))
  (let ((domain (domain-of x)))
    (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
	(lisp:>= (fp-value x) y)
	(call-next-method x y))))

(defmethod-sd binary>= ((x floating-point-number) (y rational-integer))
  (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
      (lisp:>= (fp-value x) (integer-value y))
      (call-next-method x y)))

(defmethod-sd binary>= ((x rational-integer) (y floating-point-number))
  (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
      (lisp:>= (integer-value x) (fp-value y))
      (call-next-method x y)))

(defmethod-sd binary>= ((x floating-point-number) (y rational-number))
  (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
      (lisp:>= (lisp:* (qo-denominator y) (fp-value x)) (qo-numerator y))
      (call-next-method x y)))

(defmethod-sd binary>= ((x rational-number) (y floating-point-number))
  (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
      (lisp:>= (qo-numerator x) (lisp:* (qo-denominator x) (fp-value y)))
      (call-next-method x y)))

(defmethod binary>= :around ((x float) (y bigfloat))
  (let ((domain (domain-of y)))
    (if (or (typep domain 'ordered-set)
	    (typep domain 'non-strict-domain))
	(lisp:>= x (lisp:* (lisp:float (bigfloat-mantissa y))
			   (lisp:expt 10.0 (bigfloat-exponent y))))
	(call-next-method x y))))

(defmethod binary>= :around ((x bigfloat) (y float))
  (let ((domain (domain-of x)))
    (if (or (typep domain 'ordered-set)
	    (typep domain 'non-strict-domain))
	(lisp:>= (lisp:* (lisp:float (bigfloat-mantissa x))
			 (lisp:expt 10.0 (bigfloat-exponent x)))
		 y)
	(call-next-method x y))))

(defmethod-sd binary>= ((x floating-point-number) (y bigfloat))
  (cond ((or (typep domain 'ordered-set)
	     (typep domain 'non-strict-domain))
	 (setq y (decprec! y *floating-point-precision*))
	 (lisp:>= (fp-value x)
		 (lisp:* (lisp:float (bigfloat-mantissa y))
			 (lisp:expt 10.0 (bigfloat-exponent y)))))
	(t (call-next-method x y))))

(defmethod-sd binary>= ((x bigfloat) (y floating-point-number))
  (cond ((or (typep domain 'ordered-set)
	     (typep domain 'non-strict-domain))
	 (setq x (decprec! x *floating-point-precision*))
	 (lisp:>= (lisp:* (lisp:float (bigfloat-mantissa x))
			 (lisp:expt 10.0 (bigfloat-exponent x)))
		 (fp-value y)))
	(t (call-next-method x y))))

(defmethod max-pair ((x number) (y number))
  (if (lisp:> x y) x y))

(defmethod-sd max-pair ((x rational-integer) (y rational-integer))
  (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
      (if (lisp:>= (integer-value x) (integer-value y))
	  x y)
      (call-next-method x y)))

(defmethod-sd max-pair ((x rational-number) (y rational-number))
  (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
      (if (lisp:>= (lisp:* (qo-numerator x) (qo-denominator y))
		   (lisp:* (qo-numerator y) (qo-denominator x)))
	  x y)
      (call-next-method x y)))

(defmethod-sd max-pair ((x rational-number) (y rational-integer))
  (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
      (if (lisp:>= (qo-numerator x)
		   (lisp:* (integer-value y) (qo-denominator x)))
	  x y)
      (call-next-method x y)))

(defmethod-sd max-pair ((x rational-integer) (y rational-number))
  (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
      (if (lisp:>= (lisp:* (integer-value x)
			   (qo-denominator y)) (qo-numerator y))
	  x y)
      (call-next-method x y)))

(defmethod-sd max-pair ((x floating-point-number) (y floating-point-number))
  (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
      (if (lisp:>= (fp-value x) (fp-value y))
	  x y)
      (call-next-method x y)))

(defmethod max-pair :around ((x float) (y floating-point-number))
  (let ((domain (domain-of y)))
    (if (or (typep domain 'ordered-set)
	    (typep domain 'non-strict-domain))
	(if (lisp:>= x (fp-value y))
	    (make-element domain x)
	    y)
	(call-next-method x y))))

(defmethod max-pair :around ((x floating-point-number) (y float))
  (let ((domain (domain-of x)))
    (if (or (typep domain 'ordered-set)
	    (typep domain 'non-strict-domain))
	(if (lisp:>= (fp-value x) y)
	    x (make-element domain y))
      (call-next-method x y))))

(defmethod max-pair :around ((x float) (y bigfloat))
  (let ((domain (domain-of y)))
    (if (or (typep domain 'ordered-set)
	    (typep domain 'non-strict-domain))
	(if (lisp:>= x (lisp:* (lisp:float (bigfloat-mantissa y))
			       (lisp:expt 10.0 (bigfloat-exponent y))))
	    (make-element domain x)
	    y)
	(call-next-method x y))))

(defmethod-sd max-pair  ((x bigfloat) (y float))
  (let ((domain (domain-of x)))
    (if (or (typep domain 'ordered-set)
	    (typep domain 'non-strict-domain))
	(if (lisp:>= (lisp:* (lisp:float (bigfloat-mantissa x))
			     (lisp:expt 10.0 (bigfloat-exponent x)))
		     y)
	    x (make-element domain y))
	(call-next-method x y))))

(defmethod-sd max-pair ((x floating-point-number) (y bigfloat))
  (cond ((or (typep domain 'ordered-set)
	     (typep domain 'non-strict-domain))
	 (let ((yy (decprec! y *floating-point-precision*)))
	   (if (lisp:>= (fp-value x)
			(lisp:* (lisp:float (bigfloat-mantissa yy))
				(lisp:expt 10.0 (bigfloat-exponent yy))))
	       x y)))
	(t (call-next-method x y))))

(defmethod-sd max-pair ((x bigfloat) (y floating-point-number))
  (cond ((or (typep domain 'ordered-set)
	     (typep domain 'non-strict-domain))
	 (let ((xx (decprec! x *floating-point-precision*)))
	   (if (lisp:>= (lisp:* (lisp:float (bigfloat-mantissa xx))
				(lisp:expt 10.0 (bigfloat-exponent xx)))
			(fp-value y))
	       x y)))
	(t (call-next-method x y))))

(defmethod min-pair ((x number) (y number))
  (if (lisp:> x y) y x))

(defmethod-sd min-pair ((x rational-integer) (y rational-integer))
  (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
      (if (lisp:<= (integer-value x) (integer-value y))
	  x y)
      (call-next-method x y)))

(defmethod-sd min-pair ((x rational-number) (y rational-number))
  (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
      (if (lisp:<= (lisp:* (qo-numerator x) (qo-denominator y))
		   (lisp:* (qo-numerator y) (qo-denominator x)))
	  x y)
      (call-next-method x y)))

(defmethod-sd min-pair ((x rational-number) (y rational-integer))
  (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
      (if (lisp:<= (qo-numerator x)
		   (lisp:* (integer-value y) (qo-denominator x)))
	  x y)
      (call-next-method x y)))

(defmethod-sd min-pair ((x rational-integer) (y rational-number))
  (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
      (if (lisp:<= (lisp:* (integer-value x)
			   (qo-denominator y)) (qo-numerator y))
	  x y)
      (call-next-method x y)))

(defmethod-sd min-pair ((x floating-point-number) (y floating-point-number))
  (if (or (typep domain 'ordered-set)
	  (typep domain 'non-strict-domain))
      (if (lisp:<= (fp-value x) (fp-value y))
	  x y)
      (call-next-method x y)))

(defmethod min-pair :around ((x float) (y floating-point-number))
  (let ((domain (domain-of y)))
    (if (or (typep domain 'ordered-set)
	    (typep domain 'non-strict-domain))
	(if (lisp:<= x (fp-value y))
	    (make-element domain x)
	    y)
	(call-next-method x y))))

(defmethod min-pair :around ((x floating-point-number) (y float))
  (let ((domain (domain-of x)))
    (if (or (typep domain 'ordered-set)
	    (typep domain 'non-strict-domain))
	(if (lisp:<= (fp-value x) y)
	    x (make-element domain y))
      (call-next-method x y))))

(defmethod min-pair :around ((x float) (y bigfloat))
  (let ((domain (domain-of y)))
    (if (or (typep domain 'ordered-set)
	    (typep domain 'non-strict-domain))
	(if (lisp:<= x (lisp:* (lisp:float (bigfloat-mantissa y))
			       (lisp:expt 10.0 (bigfloat-exponent y))))
	    (make-element domain x)
	    y)
	(call-next-method x y))))

(defmethod-sd min-pair  ((x bigfloat) (y float))
  (let ((domain (domain-of x)))
    (if (or (typep domain 'ordered-set)
	    (typep domain 'non-strict-domain))
	(if (lisp:<= (lisp:* (lisp:float (bigfloat-mantissa x))
			     (lisp:expt 10.0 (bigfloat-exponent x)))
		     y)
	    x (make-element domain y))
	(call-next-method x y))))

(defmethod-sd min-pair ((x floating-point-number) (y bigfloat))
  (cond ((or (typep domain 'ordered-set)
	     (typep domain 'non-strict-domain))
	 (let ((yy (decprec! y *floating-point-precision*)))
	   (if (lisp:<= (fp-value x)
			(lisp:* (lisp:float (bigfloat-mantissa yy))
				(lisp:expt 10.0 (bigfloat-exponent yy))))
	       x y)))
	(t (call-next-method x y))))

(defmethod-sd min-pair ((x bigfloat) (y floating-point-number))
  (cond ((or (typep domain 'ordered-set)
	     (typep domain 'non-strict-domain))
	 (let ((xx (decprec! x *floating-point-precision*)))
	   (if (lisp:<= (lisp:* (lisp:float (bigfloat-mantissa xx))
				(lisp:expt 10.0 (bigfloat-exponent xx)))
			(fp-value y))
	       x y)))
	(t (call-next-method x y))))

(defmethod plus ((x number) (y number))
  (lisp:+ x y))

(defmethod plus  ((x integer) (y rational-integer))
  (make-element (domain-of y) (lisp:+ x (integer-value y))))

(defmethod plus ((x integer) (y rational-number))
  (make-element (domain-of y)
		(lisp:+ x (lisp:/ (qo-numerator y) (qo-denominator y)))))

(defmethod plus ((x integer) (y floating-point-number))
  (make-element (domain-of y) (lisp:+ (lisp:float x) (fp-value y))))

(defmethod plus ((x integer) (y bigfloat))
  (setq y (decprec! y *floating-point-precision*))
  (make-element (domain-of y)
		(lisp:+ (lisp:float x)
			(lisp:* (lisp:float (bigfloat-mantissa y))
				(lisp:expt 10.0 (bigfloat-exponent y))))))

(defmethod plus ((x integer) (y complex-number))
  (make-element (domain-of y) (+ x (cn-realpart y)) (cn-imagpart y)))

(defmethod plus :around ((x ratio) (y rational-integer))
  (let ((domain (domain-of y)))
    (if (typep domain 'field)
	(make-element domain (lisp:+ x (integer-value y)))
	(call-next-method x y))))

(defmethod plus ((x ratio) (y rational-number))
  (let ((x-num (lisp:numerator x))
	(x-den (lisp:denominator x))
	(y-num (qo-numerator y))
	(y-den (qo-denominator y)))
    (make-element (domain-of y)
		  (lisp:/ (lisp:+ (lisp:* x-num y-den)
				  (lisp:* x-den y-num))
			  (lisp:* x-den y-den)))))

(defmethod plus ((x ratio) (y floating-point-number))
  (make-element (domain-of y) (lisp:+ (lisp:float x) (fp-value y))))

(defmethod plus ((x ratio) (y complex-number))
  (make-element (domain-of y)
		(+ x (cn-realpart y))
		(cn-imagpart y)))

(defmethod plus :around ((x float) (y rational-integer))
  (let ((domain (domain-of y)))
    (if (typep domain 'complete-set)
	(make-element domain (lisp:+ x (integer-value y)))
	(call-next-method x y))))

(defmethod plus ((x float) (y rational-number))
  (let ((domain (domain-of y))
	(y-num (qo-numerator y))
	(y-den (qo-denominator y)))
    (if (typep domain 'complete-set)
	(make-element domain (lisp:/ (lisp:+ y-num (lisp:* x y-den))
				     y-den))
	(call-next-method x y))))

(defmethod plus ((x float) (y floating-point-number))
  (make-element (domain-of y) (lisp:+  x (fp-value y))))

(defmethod plus ((x float) (y bigfloat))
  (setq y (decprec! y *floating-point-precision*))
  (make-element (domain-of y)
		(lisp:+ x
			(lisp:* (lisp:float (bigfloat-mantissa y))
				(lisp:expt 10.0 (bigfloat-exponent y))))))

(defmethod plus ((x float) (y complex-number))
  (make-element (domain-of y) (+ x (cn-realpart y)) (cn-imagpart y)))

(defmethod plus ((x rational-integer) (y integer))
  (make-element (domain-of x) (lisp:+ (integer-value x) y)))

(defmethod plus :around ((x rational-integer) (y ratio))
  (let ((domain (domain-of x)))
    (if (or (typep domain 'field)
	    (typep domain 'non-strict-domain))
	(make-element domain (lisp:+ (integer-value x) y))
	(call-next-method x y))))

(defmethod plus :around ((x rational-integer) (y float))
  (let ((domain (domain-of x)))
    (if (typep domain 'complete-set)
	(make-element domain (lisp:+ (lisp:float (integer-value x))
				     y))
	(call-next-method x y))))

(defmethod-sd plus ((x rational-integer) (y rational-integer))
  (make-element domain (lisp:+ (integer-value x) (integer-value y))))

(defmethod-sd plus ((x rational-integer) (y rational-number))
  (if (or (typep domain 'field)
	  (typep domain 'non-strict-domain))
      (make-element domain (lisp:+ (integer-value x)
				   (lisp:/ (qo-numerator y) (qo-denominator y))))
      (call-next-method x y)))

(defmethod-sd plus ((x rational-integer) (y floating-point-number))
  (if (typep domain 'complete-set)
      (make-element domain (lisp:+ (lisp:float (integer-value x))
				   (fp-value y)))
      (call-next-method x y)))

(defmethod-sd plus ((x rational-integer) (y bigfloat))
  (setq y (decprec! y *floating-point-precision*))
  (make-element domain
		(lisp:+ (lisp:float (integer-value x))
			(lisp:* (lisp:float (bigfloat-mantissa y))
				(lisp:expt 10.0 (bigfloat-exponent y))))))

(defmethod plus :around ((x rational-integer) (y lisp:complex))
  (let ((domain (domain-of x)))
    (if (typep domain 'complex-numbers)
	(make-element domain
		      (lisp:+ (integer-value x) (lisp:realpart y))
		      (lisp:imagpart y))
	(call-next-method x y))))

(defmethod-sd plus ((x rational-integer) (y complex-number))
  (make-element domain
		(+ (integer-value x) (cn-realpart y))
		(cn-imagpart y)))

(defmethod plus ((x rational-number) (y integer))
  (make-element (domain-of x)
		(lisp:+ (lisp:/ (qo-numerator x) (qo-denominator x))
			y)))

(defmethod-sd plus ((x rational-number) (y rational-integer))
  (let ((x-num (qo-numerator x))
	(x-den (qo-denominator x)))
    (make-element domain
		  (lisp:+ (lisp:/ x-num x-den) (integer-value y)))))

(defmethod plus :around ((x rational-number) (y ratio))
  (make-element (domain-of x)
		(lisp:+ (lisp:/ (qo-numerator x) (qo-denominator x))
			  y)))

(defmethod-sd plus ((x rational-number) (y rational-number))
  (make-element domain
		(lisp:+ (lisp:/ (qo-numerator x) (qo-denominator x))
			(lisp:/ (qo-numerator y) (qo-denominator y)))))

(defmethod plus :around ((x rational-number) (y float))
  (let ((domain (domain-of x)))
    (if (typep domain 'complete-set)
	(make-element domain (lisp:+ (lisp:/ (lisp:float (qo-numerator x))
					     (lisp:float (qo-denominator x)))
				     y))
	(call-next-method x y))))

(defmethod-sd plus ((x rational-number) (y floating-point-number))
  (if (typep domain 'complete-set)
      (make-element domain (lisp:+ (lisp:/ (lisp:float (qo-numerator x))
					   (lisp:float (qo-denominator x)))
				   (fp-value y)))
      (call-next-method x y)))


(defmethod-sd plus ((x rational-number) (y bigfloat))
  (setq y (decprec! y *floating-point-precision*))
  (make-element domain
		(lisp:+ (lisp:/ (lisp:float (qo-numerator x))
				(lisp:float (qo-denominator x)))
			(lisp:* (lisp:float (bigfloat-mantissa y))
				(lisp:expt 10.0 (bigfloat-exponent y))))))

(defmethod plus :around ((x rational-number) (y lisp:complex))
  (let ((domain (domain-of x)))
    (if (typep domain 'complex-numbers)
	(make-element domain
		      (lisp:+ (lisp:/ (qo-numerator x) (qo-denominator x))
			      (lisp:realpart y))
		      (lisp:imagpart y))
	(call-next-method x y))))

(defmethod-sd plus ((x rational-number) (y complex-number))
  (make-element domain
		(+ (lisp:/ (qo-numerator x) (qo-denominator x))
		   (cn-realpart y))
		(cn-imagpart y)))


(defmethod plus ((x floating-point-number) (y integer))
  (make-element (domain-of x)
		(lisp:+ (fp-value x) (lisp:float y))))

(defmethod-sd plus ((x floating-point-number) (y rational-integer))
  (make-element domain
		(lisp:+ (fp-value x) (integer-value y))))

(defmethod plus :around ((x floating-point-number) (y ratio))
  (make-element (domain-of x)
		(lisp:+ (fp-value x) (lisp:float y))))

(defmethod-sd plus ((x floating-point-number) (y rational-number))
  (let ((y-num (qo-numerator y))
	(y-den (qo-denominator y)))
    (make-element domain
		  (lisp:/ (lisp:+ (lisp:* (fp-value x) y-den) y-num)
			  y-den))))

(defmethod plus  ((x floating-point-number) (y float))
  (make-element (domain-of x)
		(lisp:+ (fp-value x) y)))

(defmethod-sd plus ((x floating-point-number) (y floating-point-number))
  (make-element domain (lisp:+ (fp-value x) (fp-value y))))


(defmethod-sd plus ((x floating-point-number) (y bigfloat))
  (setq y (decprec! y *floating-point-precision*))
  (make-element domain
		(lisp:+ (lisp:* (lisp:float (bigfloat-mantissa y))
				(lisp:expt 10.0 (bigfloat-exponent y)))
			(fp-value x))))

(defmethod plus :around ((x floating-point-number) (y lisp:complex))
  (let ((domain (domain-of x)))
    (if (typep domain 'complex-numbers)
	(make-element domain
		      (lisp:+ (fp-value x) (lisp:realpart y))
		      (lisp:imagpart y))
	(call-next-method x y))))

(defmethod-sd plus ((x floating-point-number) (y complex-number))
  (make-element domain
		(+ (fp-value x) (cn-realpart y))
		(cn-imagpart y)))

(defmethod plus ((x bigfloat) (y integer))
  (setq x (decprec! x *floating-point-precision*))
  (make-element (domain-of x)
		(lisp:+ (lisp:* (lisp:float (bigfloat-mantissa x))
				(lisp:expt 10.0 (bigfloat-exponent x)))
			(lisp:float y))))

(defmethod-sd plus ((x bigfloat) (y rational-integer))
  (setq x (decprec! x *floating-point-precision*))
  (make-element domain
		(lisp:+ (lisp:* (lisp:float (bigfloat-mantissa x))
				(lisp:expt 10.0 (bigfloat-exponent x)))
			(lisp:float (integer-value y)))))

(defmethod plus :around ((x bigfloat) (y ratio))
  (setq x (decprec! x *floating-point-precision*))
  (make-element (domain-of x)
		(lisp:+ (lisp:* (lisp:float (bigfloat-mantissa x))
				(lisp:expt 10.0 (bigfloat-exponent x)))
			(lisp:float y))))

(defmethod-sd plus ((x bigfloat) (y rational-number))
  (setq x (decprec! x *floating-point-precision*))
  (make-element domain
		(lisp:+ (lisp:* (lisp:float (bigfloat-mantissa x))
				(lisp:expt 10.0 (bigfloat-exponent x)))
			(lisp:/ (lisp:float (qo-numerator y))
				(lisp:float (qo-denominator y))))))

(defmethod plus  ((x bigfloat) (y float))
  (setq x (decprec! x *floating-point-precision*))
  (make-element (domain-of x)
		(lisp:+ (lisp:* (lisp:float (bigfloat-mantissa x))
				(lisp:expt 10.0 (bigfloat-exponent x)))
			y)))

(defmethod-sd plus ((x bigfloat) (y floating-point-number))
  (setq x (decprec! x *floating-point-precision*))
  (make-element domain
		(lisp:+ (lisp:* (lisp:float (bigfloat-mantissa x))
				(lisp:expt 10.0 (bigfloat-exponent x)))
			(fp-value y))))

(defmethod-sd plus ((x bigfloat) (y bigfloat))
  (bind-domain-context domain
    (round!mt (bf-plus x y) *REAL-PRECISION*)))

(defmethod plus :around ((x bigfloat) (y lisp:complex))
  (let ((domain (domain-of x)))
    (if (typep domain 'complex-numbers)
	(make-element domain (+ x (lisp:realpart y)) (lisp:imagpart y))
	(call-next-method x y))))

(defmethod-sd plus ((x bigfloat) (y complex-number))
  (make-element domain (+ x (lisp:realpart y)) (lisp:imagpart y)))


(defmethod plus :around ((x lisp:complex) (y rational-integer))
  (let ((domain (domain-of y)))
    (if (typep domain 'complex-numbers)
	(make-element domain
		      (+ (lisp:realpart x) (integer-value y))
		      (lisp:imagpart x))
	(call-next-method x y))))

(defmethod plus :around ((x lisp:complex) (y rational-number))
  (let ((domain (domain-of y)))
    (if (typep domain 'complex-numbers)
	(make-element domain
		      (lisp:+ (lisp:realpart x)
			      (lisp:/ (qo-numerator y) (qo-denominator y)))
		      (lisp:imagpart y))
	(call-next-method x y))))

(defmethod plus :around ((x lisp:complex) (y floating-point-number))
  (let ((domain (domain-of y)))
    (if (typep domain 'complex-numbers)
	(make-element domain
		      (lisp:+ (lisp:realpart x) (fp-value y))
		      (lisp:imagpart x))
	(call-next-method x y))))

(defmethod plus :around ((x lisp:complex) (y bigfloat))
  (let ((domain (domain-of y)))
    (if (typep domain 'complex-numbers)
	(make-element domain (+ (cn-realpart x) y) (cn-imagpart x))
	(call-next-method x y))))

(defmethod plus ((x lisp:complex) (y complex-number))
  (make-element (domain-of y)
		(+ (lisp:realpart x) (cn-realpart y))
		(+ (lisp:imagpart x) (cn-imagpart y))))

(defmethod plus ((x complex-number) (y integer))
  (make-element (domain-of x) (+ (cn-realpart x) y) (cn-imagpart x)))

(defmethod-sd plus ((x complex-number) (y rational-integer))
  (make-element domain
		(+ (cn-realpart x) (integer-value y))
		(cn-imagpart x)))

(defmethod plus :around ((x complex-number) (y ratio))
  (make-element (domain-of x) (+ (cn-realpart x) y) (cn-imagpart x)))

(defmethod-sd plus ((x complex-number) (y rational-number))
  (make-element domain
		(+ (cn-realpart x)
		   (lisp:/ (qo-numerator y) (qo-denominator y)))
		(cn-imagpart x)))

(defmethod plus  ((x complex-number) (y float))
  (make-element (domain-of x) (+ (cn-realpart x) y) (cn-imagpart x)))

(defmethod-sd plus ((x complex-number) (y floating-point-number))
  (make-element domain
		(+ (cn-realpart x) (fp-value y))
		(cn-imagpart x)))

(defmethod-sd plus ((x complex-number) (y bigfloat))
  (make-element domain (+ (cn-realpart x) y) (cn-imagpart x)))

(defmethod-sd plus ((x complex-number) (y complex-number))
  (make-element domain
		(+ (cn-realpart x) (cn-realpart y))
		(+ (cn-imagpart x) (cn-imagpart y))))

(defmethod plus ((x complex-number) (y lisp:complex))
  (make-element (domain-of x)
		(+ (cn-realpart x) (lisp:realpart y))
		(+ (cn-imagpart x) (lisp:imagpart y))))

(defmethod difference ((x number) (y number))
  (lisp:- x y))

(defmethod difference  ((x integer) (y rational-integer))
  (make-element (domain-of y) (lisp:- x (integer-value y))))

(defmethod difference ((x integer) (y rational-number))
  (make-element (domain-of y)
		(lisp:- x (lisp:/ (qo-numerator y) (qo-denominator y)))))

(defmethod difference ((x integer) (y floating-point-number))
  (make-element (domain-of y) (lisp:- (lisp:float x) (fp-value y))))

(defmethod difference ((x integer) (y bigfloat))
  (setq y (decprec! y *floating-point-precision*))
  (make-element (domain-of y)
		(lisp:- (lisp:float x)
			(lisp:* (lisp:float (bigfloat-mantissa y))
				(lisp:expt 10.0 (bigfloat-exponent y))))))

(defmethod difference ((x integer) (y complex-number))
  (make-element (domain-of y)
		(- x (cn-realpart y))
		(- (cn-imagpart y))))

(defmethod difference :around ((x ratio) (y rational-integer))
  (let ((domain (domain-of y)))
    (if (or (typep domain 'field)
	    (typep domain 'non-strict-domain))
	(make-element domain (lisp:- x (integer-value y)))
	(call-next-method x y))))

(defmethod difference ((x ratio) (y rational-number))
  (let ((x-num (lisp:numerator x))
	(x-den (lisp:denominator x))
	(y-num (qo-numerator y))
	(y-den (qo-denominator y)))
    (make-element (domain-of y)
		  (lisp:/ (lisp:- (lisp:* x-num y-den)
				  (lisp:* x-den y-num))
			  (lisp:* x-den y-den)))))

(defmethod difference ((x ratio) (y floating-point-number))
  (make-element (domain-of y) (lisp:- (lisp:float x) (fp-value y))))

(defmethod difference ((x ratio) (y complex-number))
  (make-element (domain-of y)
		(- x (cn-realpart y))
		(- (cn-imagpart y))))

(defmethod difference :around ((x float) (y rational-integer))
  (let ((domain (domain-of y)))
    (if (typep domain 'complete-set)
	(make-element domain (lisp:- x (integer-value y)))
	(call-next-method x y))))

(defmethod difference ((x float) (y rational-number))
  (let ((domain (domain-of y))
	(y-num (qo-numerator y))
	(y-den (qo-denominator y)))
    (if (typep domain 'complete-set)
	(make-element domain (lisp:- x (lisp:/ (lisp:float y-num)
					       (lisp:float y-den))))
	(call-next-method x y))))

(defmethod difference ((x float) (y floating-point-number))
  (make-element (domain-of y) (lisp:-  x (fp-value y))))

(defmethod difference ((x float) (y bigfloat))
  (setq y (decprec! y *floating-point-precision*))
  (make-element (domain-of y)
		(lisp:- x
			(lisp:* (lisp:float (bigfloat-mantissa y))
				(lisp:expt 10.0 (bigfloat-exponent y))))))

(defmethod difference ((x float) (y complex-number))
  (make-element (domain-of y) (- x (cn-realpart y)) (- (cn-imagpart y))))

(defmethod difference ((x rational-integer) (y integer))
  (make-element (domain-of x) (lisp:- (integer-value x) y)))

(defmethod difference :around ((x rational-integer) (y ratio))
  (let ((domain (domain-of x)))
    (if (or (typep domain 'field)
	    (typep domain 'non-strict-domain))
	(make-element domain (lisp:- (integer-value x) y))
	(call-next-method x y))))

(defmethod difference :around ((x rational-integer) (y float))
  (let ((domain (domain-of x)))
    (if (typep domain 'complete-set)
	(make-element domain (lisp:- (lisp:float (integer-value x))
				     y))
	(call-next-method x y))))

(defmethod-sd difference ((x rational-integer) (y rational-integer))
  (make-element domain (lisp:- (integer-value x) (integer-value y))))

(defmethod-sd difference ((x rational-integer) (y rational-number))
  (if (or (typep domain 'field)
	  (typep domain 'non-strict-domain))
      (make-element domain
		    (lisp:- (integer-value x)
			    (lisp:/ (qo-numerator y) (qo-denominator y))))
      (call-next-method x y)))

(defmethod-sd difference ((x rational-integer) (y floating-point-number))
  (if (typep domain 'complete-set)
      (make-element domain (lisp:- (lisp:float (integer-value x))
				   (fp-value y)))
      (call-next-method x y)))

(defmethod-sd difference ((x rational-integer) (y bigfloat))
  (setq y (decprec! y *floating-point-precision*))
  (make-element domain
		(lisp:- (lisp:float (integer-value x))
			(lisp:* (lisp:float (bigfloat-mantissa y))
				(lisp:expt 10.0 (bigfloat-exponent y))))))

(defmethod difference :around ((x rational-integer) (y lisp:complex))
  (let ((domain (domain-of x)))
    (if (typep domain 'complex-numbers)
	(make-element domain
		      (lisp:- (integer-value x) (lisp:realpart y))
		      (lisp:- (lisp:imagpart y)))
	(call-next-method x y))))

(defmethod-sd difference ((x rational-integer) (y complex-number))
  (make-element domain
		(- (integer-value x) (cn-realpart y))
		(- (cn-imagpart y))))

(defmethod difference ((x rational-number) (y integer))
  (make-element (domain-of x)
		(lisp:- (lisp:/ (qo-numerator x) (qo-denominator x))
			y)))

(defmethod-sd difference ((x rational-number) (y rational-integer))
  (let ((x-num (qo-numerator x))
	(x-den (qo-denominator x)))	
    (make-element domain
		  (lisp:- (lisp:/ x-num x-den) (integer-value y)))))

(defmethod difference :around ((x rational-number) (y ratio))
  (make-element (domain-of x)
		(lisp:- (lisp:/ (qo-numerator x) (qo-denominator x))
			  y)))

(defmethod-sd difference ((x rational-number) (y rational-number))
  (make-element domain
		(lisp:- (lisp:/ (qo-numerator x) (qo-denominator x))
			(lisp:/ (qo-numerator y) (qo-denominator y)))))

(defmethod difference :around ((x rational-number) (y float))
  (let ((domain (domain-of x)))
    (if (typep domain 'complete-set)
	(make-element domain
		      (lisp:- (lisp:/ (lisp:float (qo-numerator x))
				      (lisp:float (qo-denominator x)))
			      y))
	(call-next-method x y))))

(defmethod-sd difference ((x rational-number) (y floating-point-number))
  (if (typep domain 'complete-set)
      (make-element domain
		    (lisp:- (lisp:/ (lisp:float (qo-numerator x))
				    (lisp:float (qo-denominator x)))
			    (fp-value y)))
      (call-next-method x y)))


(defmethod-sd difference ((x rational-number) (y bigfloat))
  (setq y (decprec! y *floating-point-precision*))
  (make-element domain
		(lisp:- (lisp:/ (lisp:float (qo-numerator x))
				(lisp:float (qo-denominator x)))
			(lisp:* (lisp:float (bigfloat-mantissa y))
				(lisp:expt 10.0 (bigfloat-exponent y))))))

(defmethod difference :around ((x rational-number) (y lisp:complex))
  (let ((domain (domain-of x)))
    (if (typep domain 'complex-numbers)
	(make-element domain
		      (lisp:- (lisp:/ (qo-numerator x) (qo-denominator x))
			      (lisp:realpart y))
		      (lisp:- (lisp:imagpart y)))
	(call-next-method x y))))

(defmethod-sd difference ((x rational-number) (y complex-number))
  (make-element domain
		(- (lisp:/ (qo-numerator x) (qo-denominator x))
		   (cn-realpart y))
		(- (cn-imagpart y))))


(defmethod difference ((x floating-point-number) (y integer))
  (make-element (domain-of x)
		(lisp:- (fp-value x) (lisp:float y))))

(defmethod-sd difference ((x floating-point-number) (y rational-integer))
  (make-element domain
		(lisp:- (fp-value x) (integer-value y))))

(defmethod difference :around ((x floating-point-number) (y ratio))
  (make-element (domain-of x)
		(lisp:- (fp-value x) (lisp:float y))))

(defmethod-sd difference ((x floating-point-number) (y rational-number))
  (let ((y-num (qo-numerator y))
	(y-den (qo-denominator y)))
    (make-element domain
		  (lisp:/ (lisp:- (lisp:* (fp-value x) y-den) y-num)
			  y-den))))

(defmethod difference  ((x floating-point-number) (y float))
  (make-element (domain-of x)
		(lisp:- (fp-value x) y)))

(defmethod-sd difference ((x floating-point-number) (y floating-point-number))
  (make-element domain (lisp:- (fp-value x) (fp-value y))))


(defmethod-sd difference ((x floating-point-number) (y bigfloat))
  (setq y (decprec! y *floating-point-precision*))
  (make-element domain
		(lisp:- (lisp:* (lisp:float (bigfloat-mantissa y))
				(lisp:expt 10.0 (bigfloat-exponent y)))
			(fp-value x))))

(defmethod difference :around ((x floating-point-number) (y lisp:complex))
  (let ((domain (domain-of x)))
    (if (typep domain 'complex-numbers)
	(make-element domain
		      (lisp:- (fp-value x) (lisp:realpart y))
		      (lisp:- (lisp:imagpart y)))
	(call-next-method x y))))

(defmethod-sd difference ((x floating-point-number) (y complex-number))
  (make-element domain
		(- (fp-value x) (cn-realpart y))
		(- (cn-imagpart y))))

(defmethod difference ((x bigfloat) (y integer))
  (setq x (decprec! x *floating-point-precision*))
  (make-element (domain-of x)
		(lisp:- (lisp:* (lisp:float (bigfloat-mantissa x))
				(lisp:expt 10.0 (bigfloat-exponent x)))
			(lisp:float y))))

(defmethod-sd difference ((x bigfloat) (y rational-integer))
  (setq x (decprec! x *floating-point-precision*))
  (make-element domain
		(lisp:- (lisp:* (lisp:float (bigfloat-mantissa x))
				(lisp:expt 10.0 (bigfloat-exponent x)))
			(lisp:float (integer-value y)))))

(defmethod difference :around ((x bigfloat) (y ratio))
  (setq x (decprec! x *floating-point-precision*))
  (make-element (domain-of x)
		(lisp:- (lisp:* (lisp:float (bigfloat-mantissa x))
				(lisp:expt 10.0 (bigfloat-exponent x)))
			(lisp:float y))))

(defmethod-sd difference ((x bigfloat) (y rational-number))
  (setq x (decprec! x *floating-point-precision*))
  (make-element domain
		(lisp:- (lisp:* (lisp:float (bigfloat-mantissa x))
				(lisp:expt 10.0 (bigfloat-exponent x)))
			(lisp:/ (lisp:float (qo-numerator y))
				(lisp:float (qo-denominator y))))))

(defmethod difference  ((x bigfloat) (y float))
  (setq x (decprec! x *floating-point-precision*))
  (make-element (domain-of x)
		(lisp:- (lisp:* (lisp:float (bigfloat-mantissa x))
				(lisp:expt 10.0 (bigfloat-exponent x)))
			y)))

(defmethod-sd difference ((x bigfloat) (y floating-point-number))
  (setq x (decprec! x *floating-point-precision*))
  (make-element domain
		(lisp:- (lisp:* (lisp:float (bigfloat-mantissa x))
				(lisp:expt 10.0 (bigfloat-exponent x)))
			(fp-value y))))

(defmethod-sd difference ((x bigfloat) (y bigfloat))
  (bind-domain-context domain
    (round!mt (bf-difference x y) *REAL-PRECISION*)))

(defmethod difference :around ((x bigfloat) (y lisp:complex))
  (let ((domain (domain-of x)))
    (if (typep domain 'complex-numbers)
	(make-element domain
		      (- x (lisp:realpart y))
		      (lisp:- (lisp:imagpart y)))
	(call-next-method x y))))

(defmethod-sd difference ((x bigfloat) (y complex-number))
  (make-element domain (- x (lisp:realpart y)) (- (lisp:imagpart y))))


(defmethod difference :around ((x lisp:complex) (y rational-integer))
  (let ((domain (domain-of y)))
    (if (typep domain 'complex-numbers)
	(make-element domain
		      (lisp:- (lisp:realpart x) (integer-value y))
		      (lisp:imagpart x))
	(call-next-method x y))))

(defmethod difference :around ((x lisp:complex) (y rational-number))
  (let ((domain (domain-of y)))
    (if (typep domain 'complex-numbers)
	(make-element domain
		      (lisp:- (lisp:realpart x)
			      (lisp:/ (qo-numerator y) (qo-denominator y)))
		      (lisp:imagpart y))
	(call-next-method x y))))

(defmethod difference :around ((x lisp:complex) (y floating-point-number))
  (let ((domain (domain-of y)))
    (if (typep domain 'complex-numbers)
	(make-element domain
		      (lisp:- (lisp:realpart x) (fp-value y))
		      (lisp:imagpart x))
	(call-next-method x y))))

(defmethod difference :around ((x lisp:complex) (y bigfloat))
  (let ((domain (domain-of y)))
    (if (typep domain 'complex-numbers)
	(make-element domain (- (cn-realpart x) y) (cn-imagpart x))
	(call-next-method x y))))

(defmethod difference ((x lisp:complex) (y complex-number))
  (make-element (domain-of y)
		(- (lisp:realpart x) (cn-realpart y))
		(- (lisp:imagpart x) (cn-imagpart y))))

(defmethod difference ((x complex-number) (y integer))
  (make-element (domain-of x) (- (cn-realpart x) y) (cn-imagpart x)))

(defmethod-sd difference ((x complex-number) (y rational-integer))
  (make-element domain
		(- (cn-realpart x) (integer-value y))
		(cn-imagpart x)))

(defmethod difference :around ((x complex-number) (y ratio))
  (make-element (domain-of x) (- (cn-realpart x) y) (cn-imagpart x)))

(defmethod-sd difference ((x complex-number) (y rational-number))
  (make-element domain
		(- (cn-realpart x)
		   (lisp:/ (qo-numerator y) (qo-denominator y)))
		(cn-imagpart x)))

(defmethod difference  ((x complex-number) (y float))
  (make-element (domain-of x) (- (cn-realpart x) y) (cn-imagpart x)))

(defmethod-sd difference ((x complex-number) (y floating-point-number))
  (make-element domain
		(- (cn-realpart x) (fp-value y))
		(cn-imagpart x)))

(defmethod-sd difference ((x complex-number) (y bigfloat))
  (make-element domain (- (cn-realpart x) y) (cn-imagpart x)))

(defmethod-sd difference ((x complex-number) (y complex-number))
  (make-element domain
		(- (cn-realpart x) (cn-realpart y))
		(- (cn-imagpart x) (cn-imagpart y))))

(defmethod difference ((x complex-number) (y lisp:complex))
  (make-element (domain-of x)
		(- (cn-realpart x) (lisp:realpart y))
		(- (cn-imagpart x) (lisp:imagpart y))))

(defmethod times ((x number) (y number))
  (lisp:* x y))

(defmethod times  ((x integer) (y rational-integer))
  (make-element (domain-of y) (lisp:* x (integer-value y))))

(defmethod times ((x integer) (y rational-number))
  (make-element (domain-of y)
		(lisp:* x (lisp:/ (qo-numerator y) (qo-denominator y)))))

(defmethod times ((x integer) (y floating-point-number))
  (make-element (domain-of y) (lisp:* (lisp:float x) (fp-value y))))

(defmethod times ((x integer) (y bigfloat))
  (setq y (decprec! y *floating-point-precision*))
  (make-element (domain-of y)
		(lisp:* (lisp:float x)
			(lisp:* (lisp:float (bigfloat-mantissa y))
				(lisp:expt 10.0 (bigfloat-exponent y))))))

(defmethod times ((x integer) (y complex-number))
  (make-element (domain-of y)
		(* x (cn-realpart y))
		(* x (cn-imagpart y))))

(defmethod times :around ((x ratio) (y rational-integer))
  (let ((domain (domain-of y)))
    (if (or (typep domain 'field)
	    (typep domain 'non-strict-domain))
	(make-element domain (lisp:* x (integer-value y)))
	(call-next-method x y))))

(defmethod times ((x ratio) (y rational-number))
  (make-element (domain-of y)
		(lisp:* x (lisp:/ (qo-numerator y) (qo-denominator y)))))

(defmethod times ((x ratio) (y floating-point-number))
  (make-element (domain-of y) (lisp:* (lisp:float x) (fp-value y))))

(defmethod times ((x ratio) (y complex-number))
  (make-element (domain-of y)
		(* x (cn-realpart y))
		(* x (cn-imagpart y))))

(defmethod times :around ((x float) (y rational-integer))
  (let ((domain (domain-of y)))
    (if (typep domain 'complete-set)
	(make-element domain (lisp:* x (integer-value y)))
	(call-next-method x y))))

(defmethod times ((x float) (y rational-number))
  (let ((domain (domain-of y))
	(y-num (qo-numerator y))
	(y-den (qo-denominator y)))
    (if (typep domain 'complete-set)
	(make-element domain (lisp:* x (lisp:/ (lisp:float y-num)
					       (lisp:float y-den))))
	(call-next-method x y))))

(defmethod times ((x float) (y floating-point-number))
  (make-element (domain-of y) (lisp:*  x (fp-value y))))

(defmethod times ((x float) (y bigfloat))
  (setq y (decprec! y *floating-point-precision*))
  (make-element (domain-of y)
		(lisp:* x
			(lisp:* (lisp:float (bigfloat-mantissa y))
				(lisp:expt 10.0 (bigfloat-exponent y))))))

(defmethod times ((x float) (y complex-number))
  (make-element (domain-of y)
		(* x (cn-realpart y))
		(* x (cn-imagpart y))))

(defmethod times ((x rational-integer) (y integer))
  (make-element (domain-of x) (lisp:* (integer-value x) y)))

(defmethod times :around ((x rational-integer) (y ratio))
  (let ((domain (domain-of x)))
    (if (or (typep domain 'field)
	    (typep domain 'non-strict-domain))
	(make-element domain (lisp:* (integer-value x) y))
	(call-next-method x y))))

(defmethod times :around ((x rational-integer) (y float))
  (let ((domain (domain-of x)))
    (if (typep domain 'complete-set)
	(make-element domain (lisp:* (lisp:float (integer-value x))
				     y))
	(call-next-method x y))))

(defmethod-sd times ((x rational-integer) (y rational-integer))
  (make-element domain (lisp:* (integer-value x) (integer-value y))))

(defmethod-sd times ((x rational-integer) (y rational-number))
  (if (or (typep domain 'field)
	  (typep domain 'non-strict-domain))
      (make-element domain
		    (lisp:* (integer-value x)
			    (lisp:/ (qo-numerator y) (qo-denominator y))))
      (call-next-method x y)))

(defmethod-sd times ((x rational-integer) (y floating-point-number))
  (if (typep domain 'complete-set)
      (make-element domain (lisp:* (lisp:float (integer-value x))
				   (fp-value y)))
      (call-next-method x y)))

(defmethod-sd times ((x rational-integer) (y bigfloat))
  (setq y (decprec! y *floating-point-precision*))
  (make-element domain
		(lisp:* (lisp:float (integer-value x))
			(lisp:* (lisp:float (bigfloat-mantissa y))
				(lisp:expt 10.0 (bigfloat-exponent y))))))

(defmethod times :around ((x rational-integer) (y lisp:complex))
  (let ((domain (domain-of x))
	(x-int (integer-value x)))
    (if (typep domain 'complex-numbers)
	(make-element domain
		      (lisp:* x-int (lisp:realpart y))
		      (lisp:* x-int (lisp:imagpart y)))
	(call-next-method x y))))

(defmethod-sd times ((x rational-integer) (y complex-number))
  (let ((x-int (integer-value x)))
    (make-element domain
		  (* x-int (cn-realpart y))
		  (* x-int (cn-imagpart y)))))

(defmethod times ((x rational-number) (y integer))
  (make-element (domain-of x)
		(lisp:* (lisp:/ (qo-numerator x) (qo-denominator x))
			y)))

(defmethod-sd times ((x rational-number) (y rational-integer))
  (make-element domain
		(* (lisp:/ (qo-numerator x) (qo-denominator x))
		   (integer-value y))))

(defmethod times :around ((x rational-number) (y ratio))
  (make-element (domain-of x)
		(lisp:* (lisp:/ (qo-numerator x) (qo-denominator x))
			  y)))

(defmethod-sd times ((x rational-number) (y rational-number))
  (make-element domain
		(lisp:* (lisp:/ (qo-numerator x) (qo-denominator x))
			(lisp:/ (qo-numerator y) (qo-denominator y)))))

(defmethod times :around ((x rational-number) (y float))
  (let ((domain (domain-of x)))
    (if (typep domain 'complete-set)
	(make-element domain (lisp:* (lisp:/ (lisp:float (qo-numerator x))
					     (lisp:float (qo-denominator x)))
				     y))
	(call-next-method x y))))

(defmethod-sd times ((x rational-number) (y floating-point-number))
  (if (typep domain 'complete-set)
      (make-element domain (lisp:* (lisp:/ (lisp:float (qo-numerator x))
					   (lisp:float (qo-denominator x)))
				   (fp-value y)))
      (call-next-method x y)))


(defmethod-sd times ((x rational-number) (y bigfloat))
  (setq y (decprec! y *floating-point-precision*))
  (make-element domain
		(lisp:* (lisp:/ (lisp:float (qo-numerator x))
				(lisp:float (qo-denominator x)))
			(lisp:* (lisp:float (bigfloat-mantissa y))
				(lisp:expt 10.0 (bigfloat-exponent y))))))

(defmethod times :around ((x rational-number) (y lisp:complex))
  (let ((domain (domain-of x)))
    (if (typep domain 'complex-numbers)
	(let ((x-rat (lisp:/ (qo-numerator x) (qo-denominator x))))
	  (make-element domain
			(lisp:* x-rat (lisp:realpart y))
			(lisp:* x-rat (lisp:imagpart y))))
	(call-next-method x y))))

(defmethod-sd times ((x rational-number) (y complex-number))
  (let ((x-rat (lisp:/ (qo-numerator x) (qo-denominator x))))
    (make-element domain
		  (* x-rat (cn-realpart y))
		  (* x-rat (cn-imagpart y)))))


(defmethod times ((x floating-point-number) (y integer))
  (make-element (domain-of x)
		(lisp:* (fp-value x) (lisp:float y))))

(defmethod-sd times ((x floating-point-number) (y rational-integer))
  (make-element domain
		(lisp:* (fp-value x) (integer-value y))))

(defmethod times :around ((x floating-point-number) (y ratio))
  (make-element (domain-of x)
		(lisp:* (fp-value x) (lisp:float y))))

(defmethod-sd times ((x floating-point-number) (y rational-number))
  (let ((y-num (qo-numerator y))
	(y-den (qo-denominator y)))
    (make-element domain
		  (lisp:/ (lisp:* (fp-value x) y-num) y-den))))

(defmethod times  ((x floating-point-number) (y float))
  (make-element (domain-of x)
		(lisp:* (fp-value x) y)))

(defmethod-sd times ((x floating-point-number) (y floating-point-number))
  (make-element domain (lisp:* (fp-value x) (fp-value y))))

(defmethod-sd times ((x floating-point-number) (y bigfloat))
  (setq y (decprec! y *floating-point-precision*))
  (make-element domain
		(lisp:* (lisp:* (lisp:float (bigfloat-mantissa y))
				(lisp:expt 10.0 (bigfloat-exponent y)))
			(fp-value x))))

(defmethod times :around ((x floating-point-number) (y lisp:complex))
  (let ((domain (domain-of x))
	(x-val (fp-value x)))
    (if (typep domain 'complex-numbers)
	(make-element domain
		      (lisp:* x-val (lisp:realpart y))
		      (lisp:* x-val (lisp:imagpart y)))
	(call-next-method x y))))

(defmethod-sd times ((x floating-point-number) (y complex-number))
  (let ((x-val (fp-value x)))
    (make-element domain
		  (* x-val (cn-realpart y))
		  (* x-val (cn-imagpart y)))))

(defmethod times ((x bigfloat) (y integer))
  (setq x (decprec! x *floating-point-precision*))
  (make-element (domain-of x)
		(lisp:* (lisp:* (lisp:float (bigfloat-mantissa x))
				(lisp:expt 10.0 (bigfloat-exponent x)))
			(lisp:float y))))

(defmethod-sd times ((x bigfloat) (y rational-integer))
  (setq x (decprec! x *floating-point-precision*))
  (make-element domain
		(lisp:* (lisp:* (lisp:float (bigfloat-mantissa x))
				(lisp:expt 10.0 (bigfloat-exponent x)))
			(lisp:float (integer-value y)))))

(defmethod times :around ((x bigfloat) (y ratio))
  (setq x (decprec! x *floating-point-precision*))
  (make-element (domain-of x)
		(lisp:* (lisp:* (lisp:float (bigfloat-mantissa x))
				(lisp:expt 10.0 (bigfloat-exponent x)))
			(lisp:float y))))

(defmethod-sd times ((x bigfloat) (y rational-number))
  (setq x (decprec! x *floating-point-precision*))
  (make-element domain
		(lisp:* (lisp:* (lisp:float (bigfloat-mantissa x))
				(lisp:expt 10.0 (bigfloat-exponent x)))
			(lisp:/ (lisp:float (qo-numerator y))
				(lisp:float (qo-denominator y))))))

(defmethod times  ((x bigfloat) (y float))
  (setq x (decprec! x *floating-point-precision*))
  (make-element (domain-of x)
		(lisp:* (lisp:* (lisp:float (bigfloat-mantissa x))
				(lisp:expt 10.0 (bigfloat-exponent x)))
			y)))

(defmethod-sd times ((x bigfloat) (y floating-point-number))
  (setq x (decprec! x *floating-point-precision*))
  (make-element domain
		(lisp:* (lisp:* (lisp:float (bigfloat-mantissa x))
				(lisp:expt 10.0 (bigfloat-exponent x)))
			(fp-value y))))

(defmethod-sd times ((x bigfloat) (y bigfloat))
  (bind-domain-context domain
    (round!mt (bf-times x y) *REAL-PRECISION*)))

(defmethod times :around ((x bigfloat) (y lisp:complex))
  (let ((domain (domain-of x)))
    (if (typep domain 'complex-numbers)
	(make-element domain
		      (* x (lisp:realpart y))
		      (* x (lisp:imagpart y)))
	(call-next-method x y))))

(defmethod-sd times ((x bigfloat) (y complex-number))
  (make-element domain (* x (lisp:realpart y)) (* x (lisp:imagpart y))))

(defmethod times :around ((x lisp:complex) (y rational-integer))
  (let ((domain (domain-of y))
	(y-int (integer-value y)))
    (if (typep domain 'complex-numbers)
	(make-element domain
		      (lisp:* (lisp:realpart x) y-int)
		      (lisp:* (lisp:imagpart x) y-int))
	(call-next-method x y))))

(defmethod times :around ((x lisp:complex) (y rational-number))
  (let ((domain (domain-of y)))
    (if (typep domain 'complex-numbers)
	(let ((y-val (lisp:/ (qo-numerator y) (qo-denominator y))))
	  (make-element domain
			(lisp:* (lisp:realpart x) y-val)
			(lisp:* (lisp:imagpart y) y-val)))
	(call-next-method x y))))

(defmethod times :around ((x lisp:complex) (y floating-point-number))
  (let ((domain (domain-of y))
	(y-val (fp-value y)))
    (if (typep domain 'complex-numbers)
	(make-element domain
		      (lisp:* (lisp:realpart x) y-val)
		      (lisp:* (lisp:imagpart x) y-val))
	(call-next-method x y))))

(defmethod times :around ((x lisp:complex) (y bigfloat))
  (let ((domain (domain-of y)))
    (if (typep domain 'complex-numbers)
	(make-element domain (* (cn-realpart x) y) (* (cn-imagpart x) y))
	(call-next-method x y))))

(defmethod times ((x lisp:complex) (y complex-number))
  (let ((x-real (lisp:realpart x))
	(x-imag (lisp:imagpart x))
	(y-real (cn-realpart y))
	(y-imag (cn-imagpart y)))
    (make-element (domain-of y)
		  (- (* x-real y-real) (* x-imag y-imag))
		  (+ (* x-real y-imag) (* x-imag y-real)))))

(defmethod times ((x complex-number) (y integer))
  (make-element (domain-of x)
		(* (cn-realpart x) y)
		(* (cn-imagpart x) y)))

(defmethod-sd times ((x complex-number) (y rational-integer))
  (let ((y-int (integer-value y)))
    (make-element domain (* (cn-realpart x) y-int) (* (cn-imagpart x) y-int))))

(defmethod times :around ((x complex-number) (y ratio))
  (make-element (domain-of x)
		(* (cn-realpart x) y)
		(* (cn-imagpart x) y)))

(defmethod-sd times ((x complex-number) (y rational-number))
  (let ((y-val (lisp:/ (qo-numerator y) (qo-denominator y))))
    (make-element domain (* (cn-realpart x) y-val) (* (cn-imagpart x) y-val))))

(defmethod times  ((x complex-number) (y float))
  (make-element (domain-of x) (* (cn-realpart x) y) (* (cn-imagpart x) y)))

(defmethod-sd times ((x complex-number) (y floating-point-number))
  (let ((y-val (fp-value y)))
    (make-element domain (* (cn-realpart x) y-val) (* (cn-imagpart x) y-val))))

(defmethod-sd times ((x complex-number) (y bigfloat))
  (make-element domain (* (cn-realpart x) y) (* (cn-imagpart x) y)))

(defmethod-sd times ((x complex-number) (y complex-number))
  (let ((x-real (cn-realpart x))
	(x-imag (cn-imagpart x))
	(y-real (cn-realpart y))
	(y-imag (cn-imagpart y)))
  (make-element domain
		(- (* x-real y-real) (* x-imag y-imag))
		(+ (* x-real y-imag) (* x-imag y-real)))))

(defmethod times ((x complex-number) (y lisp:complex))
  (let ((x-real (cn-realpart x))
	(x-imag (cn-imagpart x))
	(y-real (lisp:realpart y))
	(y-imag (lisp:imagpart y)))
  (make-element (domain-of x)
		(- (* x-real y-real) (* x-imag y-imag))
		(+ (* x-real y-imag) (* x-imag y-real)))))

(defmethod expt ((n number) (e number))
  (lisp:expt n e))

(defmethod expt ((n integer) (e ratio))
  (let* ((num (lisp:numerator e))
	 (den (lisp:denominator e))
	 (nn (abs n))
	 (root (integer-nth-root nn den)))
    (setq root
	  (if (and root (lisp:= nn (lisp:expt root den)))
	      (lisp:expt root num)
	      (lisp:expt nn e)))
    (cond ((lisp:minusp n)
	   (if (lisp:evenp den)
	       (lisp:complex 0 root)
	       (lisp:- root)))
	  (t root))))

(defmethod expt ((n ratio) (e ratio))
  (lisp:/ (expt (lisp:numerator n) e)
	  (expt (lisp:denominator n) e)))

(defmethod expt ((n rational-integer) (e integer))
  (let ((domain (domain-of n)))
    (cond ((1? n) (one domain))
	  ((lisp:minusp e)
	   (if (or (typep domain 'field)
		   (typep domain 'non-strict-domain))
	       (make-quotient-element domain 1 (integer-value n))
	       (error "Raising ~D to a negative power ~D" n e)))
	  (t (if (eql (integer-value n) -1)
		 (if (oddp e) (- (one domain)) (one domain))
		 (make-element (domain-of n)
			       (lisp:expt (integer-value n) e)))))))

(defmethod-sd expt ((n rational-integer) (e rational-integer))
  (cond ((1? n) (one domain))
	((lisp:minusp (integer-value e))
	 (if (or (typep domain 'field)
		 (typep domain 'non-strict-domain))
	     (make-quotient-element domain 1 (integer-value n))
	     (error "Raising ~D to a negative power ~D" n e)))
	(t (if (eql (integer-value n) -1)
	       (if (oddp (integer-value e)) (- (one domain)) (one domain))
	       (make-element (domain-of n)
		 (lisp:expt (integer-value n) (integer-value e)))))))

(defmethod expt ((n rational-integer) (e ratio))
  (let* ((domain (domain-of n))
	 (nn (integer-value n))
	 (abs-nn (lisp:abs nn))
	 (num (lisp:numerator e))
	 (den (lisp:denominator e))
	 (root (integer-nth-root abs-nn den)))
    (setq root 
	  (cond ((lisp:= abs-nn (lisp:expt root den))
		 (lisp:expt root num))
		((or (typep domain 'complete-set)
		     (typep domain 'non-strict-domain))
		 (lisp:expt nn e))
		(t (error "Can't compute ~S to the ~S power in ~S"
			  n e domain))))
    (cond ((lisp:minusp nn)
	   (cond ((lisp:oddp den)
		  (make-element domain (lisp:- root)))
		 ((or (typep domain 'complete-set)
		      (typep domain 'non-strict-domain))
		  (make-element domain (lisp:complex 0 root)))
		 (t (error "Can't compute ~S to the ~S power in ~S"
			   n e domain))))
	  (t (make-element domain root)))))

(defmethod expt ((n rational-integer) (e rational-number))
  (let* ((domain (domain-of n))
	 (nn (integer-value n))
	 (abs-nn (lisp:abs nn))
	 (num (qo-numerator e))
	 (den (qo-denominator e))
	 (root (integer-nth-root abs-nn den)))
    (setq root 
	  (cond ((lisp:= abs-nn (lisp:expt root den))
		 (lisp:expt root num))
		((or (typep domain 'complete-set)
		     (typep domain 'non-strict-domain))
		 (lisp:expt nn e))
		(t (error "Can't compute ~S to the ~S power in ~S"
			  n e domain))))
    (cond ((lisp:minusp nn)
	   (cond ((lisp:oddp den)
		  (make-element domain (lisp:- root)))
		 ((or (typep domain 'complete-set)
		      (typep domain 'non-strict-domain))
		  (make-element domain (lisp:complex 0 root)))
		 (t (error "Can't compute ~S to the ~S power in ~S"
			   n e domain))))
	  (t (make-element domain root)))))

(defmethod expt ((x rational-number) (y ratio))
  (/ (expt (numerator x) y)
     (expt (denominator x) y)))

(defmethod expt ((x rational-number) (y rational-number))
  (/ (expt (numerator x) y)
     (expt (denominator x) y)))


(defmethod expt ((x floating-point-number) (y number))
  (make-element (domain-of x) (lisp:expt (fp-value x) y)))

(defmethod expt ((x floating-point-number) (y rational-integer))
  (make-element (domain-of x) (lisp:expt (fp-value x) (integer-value y))))

(defmethod expt ((x floating-point-number) (y rational-number))
  (make-element (domain-of x)
		(lisp:expt (fp-value x)
			   (lisp:/ (numerator y) (denominator y)))))

(defmethod expt ((x floating-point-number) (y floating-point-number))
  (make-element (domain-of x) (lisp:expt (fp-value x) (fp-value y))))

(defmethod expt ((number bigfloat) (k integer))
  (cond ((eql k 0) (make-bigfloat (domain-of number) 1 0))
	((eql k 1) number)
	(t (let ((domain (domain-of number)))
	     (bind-domain-context domain
	       (bf-expt number k *REAL-PRECISION*))))))

(defmethod quotient ((a number) (b number))
  (lisp:/ a b))

(defmethod quotient :around ((x integer) (y rational-integer))
  (let ((domain (domain-of y)))
    (cond ((1? y) (make-element domain x))
	  ((or (typep domain 'field)
	       (typep domain 'non-strict-domain))
	   (make-element domain (lisp:/ x (integer-value y))))
	  (t (call-next-method x y)))))

(defmethod quotient ((x integer) (y rational-number))
  (make-element (domain-of y)
		(lisp:* x (lisp:/ (qo-denominator y) (qo-numerator y)))))

(defmethod quotient ((x integer) (y floating-point-number))
  (make-element (domain-of y) (lisp:/ (lisp:float x) (fp-value y))))

(defmethod quotient ((x integer) (y bigfloat))
  (setq y (decprec! y *floating-point-precision*))
  (make-element (domain-of y)
		(lisp:/ (lisp:float x)
			(lisp:* (lisp:float (bigfloat-mantissa y))
				(lisp:expt 10.0 (bigfloat-exponent y))))))

(defmethod quotient ((x number) (y complex-number))
  (* x (recip y)))

(defmethod quotient ((x domain-element) (y complex-number))
  (* x (recip y)))

(defmethod quotient :around ((x ratio) (y rational-integer))
  (let ((domain (domain-of y)))
    (if (or (typep domain 'field)
	    (typep domain 'non-strict-domain))
	(make-element domain (lisp:/ x (integer-value y)))
	(call-next-method x y))))

(defmethod quotient ((x ratio) (y rational-number))
  (make-element (domain-of y)
		(lisp:/ x (lisp:/ (qo-numerator y) (qo-denominator y)))))

(defmethod quotient ((x ratio) (y floating-point-number))
  (make-element (domain-of y) (lisp:/ (lisp:float x) (fp-value y))))

#+ignore
(defmethod quotient ((x ratio) (y complex-number))
  )

(defmethod quotient :around ((x float) (y rational-integer))
  (let ((domain (domain-of y)))
    (if (typep domain 'complete-set)
	(make-element domain (lisp:/ x (integer-value y)))
	(call-next-method x y))))

(defmethod quotient :around ((x float) (y rational-number))
  (let ((domain (domain-of y))
	(y-num (qo-numerator y))
	(y-den (qo-denominator y)))
    (if (typep domain 'complete-set)
	(make-element domain (lisp:/ x (lisp:/ (lisp:float y-num)
					       (lisp:float y-den))))
	(call-next-method x y))))

(defmethod quotient ((x float) (y floating-point-number))
  (make-element (domain-of y) (lisp:/  x (fp-value y))))

(defmethod quotient ((x float) (y bigfloat))
  (setq y (decprec! y *floating-point-precision*))
  (make-element (domain-of y)
		(lisp:/ x
			(lisp:* (lisp:float (bigfloat-mantissa y))
				(lisp:expt 10.0 (bigfloat-exponent y))))))

#+ignore
(defmethod quotient ((x float) (y complex-number))
  )

(defmethod quotient :around ((x rational-integer) (y integer))
  (let ((domain (domain-of x)))
    (cond ((1? y) x)
	  ((or (typep domain 'field)
	       (typep domain 'non-strict-domain))
	   (make-element domain (lisp:/ (integer-value x) y)))
	  (t (call-next-method x y)))))

(defmethod quotient :around ((x rational-integer) (y ratio))
  (let ((domain (domain-of x)))
    (if (or (typep domain 'field)
	    (typep domain 'non-strict-domain))
	(make-element domain (lisp:/ (integer-value x) y))
	(call-next-method x y))))

(defmethod quotient :around ((x rational-integer) (y float))
  (let ((domain (domain-of x)))
    (if (typep domain 'complete-set)
	(make-element domain (lisp:/ (lisp:float (integer-value x))
				     y))
	(call-next-method x y))))

(defmethod-sd quotient ((x rational-integer) (y rational-integer))
  (cond ((1? y) x)
	((lisp:= -1 (integer-value y))
	 (make-element domain (lisp:- (integer-value x))))
	((or (typep domain 'field)
	     (typep domain 'non-strict-domain))
	 (make-element domain (lisp:/ (integer-value x) (integer-value y))))
	(t (call-next-method x y))))

(defmethod-sd quotient ((x rational-integer) (y rational-number))
  (if (or (typep domain 'field)
	  (typep domain 'non-strict-domain))
      (make-element domain
		    (lisp:/ (integer-value x)
			    (lisp:/ (qo-numerator y) (qo-denominator y))))
      (call-next-method x y)))

(defmethod-sd quotient ((x rational-integer) (y floating-point-number))
  (if (typep domain 'complete-set)
      (make-element domain (lisp:/ (lisp:float (integer-value x))
				   (fp-value y)))
      (call-next-method x y)))

(defmethod-sd quotient ((x rational-integer) (y bigfloat))
  (setq y (decprec! y *floating-point-precision*))
  (make-element domain
		(lisp:/ (lisp:float (integer-value x))
			(lisp:* (lisp:float (bigfloat-mantissa y))
				(lisp:expt 10.0 (bigfloat-exponent y))))))

(defmethod quotient :around ((x rational-integer) (y lisp:complex))
  (let ((domain (domain-of x)))
    (if (typep domain 'complex-numbers)
	(* x (lisp:/ y))
	(call-next-method x y))))

#+ignore
(defmethod-sd quotient ((x rational-integer) (y complex-number))
  )

(defmethod quotient ((x rational-number) (y integer))
  (make-element (domain-of x)
		(lisp:/ (lisp:/ (qo-numerator x) (qo-denominator x))
			y)))

(defmethod-sd quotient ((x rational-number) (y rational-integer))
  (make-element domain
		(/ (lisp:/ (qo-numerator x) (qo-denominator x))
		   (integer-value y))))

(defmethod quotient :around ((x rational-number) (y ratio))
  (make-element (domain-of x)
		(lisp:/ (lisp:/ (qo-numerator x) (qo-denominator x))
			  y)))

(defmethod-sd quotient ((x rational-number) (y rational-number))
  (make-element domain
		(lisp:/ (lisp:/ (qo-numerator x) (qo-denominator x))
			(lisp:/ (qo-numerator y) (qo-denominator y)))))

(defmethod quotient :around ((x rational-number) (y float))
  (let ((domain (domain-of x)))
    (if (typep domain 'complete-set)
	(make-element domain
		      (lisp:/ (lisp:/ (lisp:float (qo-numerator x))
				      (lisp:float (qo-denominator x)))
			      y))
	(call-next-method x y))))

(defmethod-sd quotient ((x rational-number) (y floating-point-number))
  (if (typep domain 'complete-set)
      (make-element domain
		    (lisp:/ (lisp:/ (lisp:float (qo-numerator x))
				    (lisp:float (qo-denominator x)))
			    (fp-value y)))
      (call-next-method x y)))


(defmethod-sd quotient ((x rational-number) (y bigfloat))
  (setq y (decprec! y *floating-point-precision*))
  (make-element domain
		(lisp:/ (lisp:/ (lisp:float (qo-numerator x))
				(lisp:float (qo-denominator x)))
			(lisp:* (lisp:float (bigfloat-mantissa y))
				(lisp:expt 10.0 (bigfloat-exponent y))))))

(defmethod quotient :around ((x rational-number) (y lisp:complex))
  (let ((domain (domain-of x)))
    (if (typep domain 'complex-numbers)
	(* x (lisp:/ y))
	(call-next-method x y))))

#+ignore
(defmethod-sd quotient ((x rational-number) (y complex-number))
  )

(defmethod quotient ((x floating-point-number) (y integer))
  (make-element (domain-of x)
		(lisp:/ (fp-value x) (lisp:float y))))

(defmethod-sd quotient ((x floating-point-number) (y rational-integer))
  (make-element domain
		(lisp:/ (fp-value x) (integer-value y))))

(defmethod quotient :around ((x floating-point-number) (y ratio))
  (make-element (domain-of x)
		(lisp:/ (fp-value x) (lisp:float y))))

(defmethod-sd quotient ((x floating-point-number) (y rational-number))
  (let ((y-num (qo-numerator y))
	(y-den (qo-denominator y)))
    (make-element domain
		  (lisp:/ (lisp:* (fp-value x) y-den) y-num))))

(defmethod quotient  ((x floating-point-number) (y float))
  (make-element (domain-of x)
		(lisp:/ (fp-value x) y)))

(defmethod-sd quotient ((x floating-point-number) (y floating-point-number))
  (make-element domain (lisp:/ (fp-value x) (fp-value y))))

(defmethod-sd quotient ((x floating-point-number) (y bigfloat))
  (setq y (decprec! y *floating-point-precision*))
  (make-element domain
		(lisp:/ (lisp:* (lisp:float (bigfloat-mantissa y))
				(lisp:expt 10.0 (bigfloat-exponent y)))
			(fp-value x))))

(defmethod quotient :around ((x floating-point-number) (y lisp:complex))
  (let ((domain (domain-of x))
	(x-val (fp-value x)))
    (if (typep domain 'complex-numbers)
	(make-element domain (* x-val (lisp:/ y)))
	(call-next-method x y))))

#+ignore
(defmethod-sd quotient ((x floating-point-number) (y complex-number))
  )

(defmethod quotient ((x bigfloat) (y integer))
  (setq x (decprec! x *floating-point-precision*))
  (make-element (domain-of x)
		(lisp:/ (lisp:* (lisp:float (bigfloat-mantissa x))
				(lisp:expt 10.0 (bigfloat-exponent x)))
			(lisp:float y))))

(defmethod-sd quotient ((x bigfloat) (y rational-integer))
  (setq x (decprec! x *floating-point-precision*))
  (make-element domain
		(lisp:/ (lisp:* (lisp:float (bigfloat-mantissa x))
				(lisp:expt 10.0 (bigfloat-exponent x)))
			(lisp:float (integer-value y)))))

(defmethod quotient :around ((x bigfloat) (y ratio))
  (setq x (decprec! x *floating-point-precision*))
  (make-element (domain-of x)
		(lisp:/ (lisp:* (lisp:float (bigfloat-mantissa x))
				(lisp:expt 10.0 (bigfloat-exponent x)))
			(lisp:float y))))

(defmethod-sd quotient ((x bigfloat) (y rational-number))
  (setq x (decprec! x *floating-point-precision*))
  (make-element domain
		(lisp:/ (lisp:* (lisp:float (bigfloat-mantissa x))
				(lisp:expt 10.0 (bigfloat-exponent x)))
			(lisp:/ (lisp:float (qo-numerator y))
				(lisp:float (qo-denominator y))))))

(defmethod quotient  ((x bigfloat) (y float))
  (setq x (decprec! x *floating-point-precision*))
  (make-element (domain-of x)
		(lisp:/ (lisp:* (lisp:float (bigfloat-mantissa x))
				(lisp:expt 10.0 (bigfloat-exponent x)))
			y)))

(defmethod-sd quotient ((x bigfloat) (y floating-point-number))
  (setq x (decprec! x *floating-point-precision*))
  (make-element domain
		(lisp:/ (lisp:* (lisp:float (bigfloat-mantissa x))
				(lisp:expt 10.0 (bigfloat-exponent x)))
			(fp-value y))))

(defmethod-sd quotient ((x bigfloat) (y bigfloat))
  (bind-domain-context domain
    (round!mt (bf-quotient x y *REAL-PRECISION*)
	      *REAL-PRECISION*)))

(defmethod quotient :around ((x bigfloat) (y lisp:complex))
  (let ((domain (domain-of x)))
    (if (typep domain 'complex-numbers)
	(* x (lisp:/ y))
	(call-next-method x y))))

#+ignore
(defmethod-sd quotient ((x bigfloat) (y complex-number))
  )

(defmethod quotient :around ((x lisp:complex) (y rational-integer))
  (let ((domain (domain-of y))
	(y-int (integer-value y)))
    (if (typep domain 'complex-numbers)
	(make-element domain
		      (lisp:/ (lisp:realpart x) y-int)
		      (lisp:/ (lisp:imagpart x) y-int))
	(call-next-method x y))))

(defmethod quotient :around ((x lisp:complex) (y rational-number))
  (let ((domain (domain-of y)))
    (if (typep domain 'complex-numbers)
	(let ((y-val (lisp:/ (qo-numerator y) (qo-denominator y))))
	  (make-element domain
			(lisp:/ (lisp:realpart x) y-val)
			(lisp:/ (lisp:imagpart y) y-val)))
	(call-next-method x y))))

(defmethod quotient :around ((x lisp:complex) (y floating-point-number))
  (let ((domain (domain-of y))
	(y-val (fp-value y)))
    (if (typep domain 'complex-numbers)
	(make-element domain
		      (lisp:/ (lisp:realpart x) y-val)
		      (lisp:/ (lisp:imagpart x) y-val))
	(call-next-method x y))))

(defmethod quotient :around ((x lisp:complex) (y bigfloat))
  (let ((domain (domain-of y)))
    (if (typep domain 'complex-numbers)
	(make-element domain (/ (cn-realpart x) y) (/ (cn-imagpart x) y))
	(call-next-method x y))))

#+ignore
(defmethod quotient ((x lisp:complex) (y complex-number))
  )

(defmethod quotient ((x complex-number) (y integer))
  (make-element (domain-of x)
		(/ (cn-realpart x) y)
		(/ (cn-imagpart x) y)))

(defmethod-sd quotient ((x complex-number) (y rational-integer))
  (let ((y-int (integer-value y)))
    (make-element domain (/ (cn-realpart x) y-int) (/ (cn-imagpart x) y-int))))

(defmethod quotient :around ((x complex-number) (y ratio))
  (make-element (domain-of x)
		(/ (cn-realpart x) y)
		(/ (cn-imagpart x) y)))

(defmethod-sd quotient ((x complex-number) (y rational-number))
  (let ((y-val (lisp:/ (qo-numerator y) (qo-denominator y))))
    (make-element domain (/ (cn-realpart x) y-val) (/ (cn-imagpart x) y-val))))

(defmethod quotient  ((x complex-number) (y float))
  (make-element (domain-of x) (/ (cn-realpart x) y) (/ (cn-imagpart x) y)))

(defmethod-sd quotient ((x complex-number) (y floating-point-number))
  (let ((y-val (fp-value y)))
    (make-element domain (/ (cn-realpart x) y-val) (/ (cn-imagpart x) y-val))))

(defmethod-sd quotient ((x complex-number) (y bigfloat))
  (make-element domain (/ (cn-realpart x) y) (/ (cn-imagpart x) y)))

(defmethod-sd quotient ((x complex-number) (y complex-number))
  (let* ((x-real (cn-realpart x))
	 (x-imag (cn-imagpart x))
	 (y-real (cn-realpart y))
	 (y-imag (cn-imagpart y))
	 (norm (+ (* y-real y-real) (* y-imag y-imag))))
    (make-element (domain-of x)
		  (/ (+ (* x-real y-real) (* x-imag y-imag)) norm)
		  (/ (- (* x-imag y-real) (* x-real y-imag)) norm))))

(defmethod quotient ((x complex-number) (y lisp:complex))
  (let* ((x-real (cn-realpart x))
	 (x-imag (cn-imagpart x))
	 (y-real (lisp:realpart y))
	 (y-imag (lisp:imagpart y))
	 (norm (+ (* y-real y-real) (* y-imag y-imag))))
  (make-element (domain-of x)
		(/ (+ (* x-real y-real) (* x-imag y-imag)) norm)
		(/ (- (* x-imag y-real) (* x-real y-imag)) norm))))

(defmethod remainder ((a number) (b number))
  (lisp:rem a b))

(defmethod-sd remainder ((a rational-integer) (b rational-integer))
  (make-element domain (lisp:rem (integer-value a) (integer-value b))))

;; The first value returned by TRUNCATE is an integer for numbers, and
;; is returned in the domain of the first argument.  The second value
;; is returned in the domain of the second argument.

(defmethod truncate1 ((a number))
  (lisp:truncate a))

(defmethod truncate1 ((a rational-integer))
  (values a (zero (domain-of a))))

(defmethod truncate1 ((a rational-number))
  (multiple-value-bind (q r)
      (lisp:truncate (qo-numerator a) (qo-denominator a))
    (values (make-element (domain-of a) q)
	    (make-element (domain-of a) r))))

(defmethod truncate1 ((a floating-point-number))
  (multiple-value-bind (q r) (lisp:truncate (fp-value a))
    (values (make-element (domain-of a) q)
	    (make-element (domain-of a) r))))

(defmethod truncate1 ((a complex-number))
  (error "Improper numeric argument"))

(defmethod truncate2 ((a number) (b number))
  (lisp:truncate a b))

(defmethod truncate2 ((a numeric) (b numeric))
  (multiple-value-bind (q r) (lisp:truncate (convert-to-lisp-number a)
					    (convert-to-lisp-number b))
    (values (make-element (domain-of a) q)
	    (make-element (domain-of b) r))))

(defmethod truncate2 ((a numeric) (b complex-number))
  (error "Improper numeric argument"))

(defmethod truncate2 ((a complex-number) (b numeric))
  (error "Improper numeric argument"))

(defmethod floor1 ((a number))
  (lisp:floor a))

(defmethod floor1 ((a rational-integer))
  (values a (zero (domain-of a))))

(defmethod floor1 ((a rational-number))
  (multiple-value-bind (q r)
      (lisp:floor (qo-numerator a) (qo-denominator a))
    (values (make-element (domain-of a) q)
	    (make-element (domain-of a) r))))

(defmethod floor1 ((a floating-point-number))
  (multiple-value-bind (q r) (lisp:floor (fp-value a))
    (values (make-element (domain-of a) q)
	    (make-element (domain-of a) r))))

(defmethod floor1 ((a complex-number))
  (error "Improper numeric argument"))

(defmethod floor2 ((a number) (b number))
  (lisp:floor a b))

(defmethod floor2 ((a numeric) (b numeric))
  (multiple-value-bind (q r) (lisp:floor (convert-to-lisp-number a)
					 (convert-to-lisp-number b))
    (values (make-element (domain-of a) q)
	    (make-element (domain-of b) r))))

(defmethod floor2 ((a numeric) (b complex-number))
  (error "Improper numeric argument"))

(defmethod floor2 ((a complex-number) (b numeric))
  (error "Improper numeric argument"))

(defmethod ceiling1 ((a number))
  (lisp:ceiling a))

(defmethod ceiling1 ((a rational-integer))
  (values a (zero (domain-of a))))

(defmethod ceiling1 ((a rational-number))
  (multiple-value-bind (q r)
      (lisp:ceiling (qo-numerator a) (qo-denominator a))
    (values (make-element (domain-of a) q)
	    (make-element (domain-of a) r))))

(defmethod ceiling1 ((a floating-point-number))
  (multiple-value-bind (q r) (lisp:ceiling (fp-value a))
    (values (make-element (domain-of a) q)
	    (make-element (domain-of a) r))))

(defmethod ceiling1 ((a complex-number))
  (error "Improper numeric argument"))

(defmethod ceiling2 ((a number) (b number))
  (lisp:ceiling a b))

(defmethod ceiling2 ((a numeric) (b numeric))
  (multiple-value-bind (q r) (lisp:ceiling (convert-to-lisp-number a)
					   (convert-to-lisp-number b))
    (values (make-element (domain-of a) q)
	    (make-element (domain-of b) r))))

(defmethod ceiling2 ((a numeric) (b complex-number))
  (error "Improper numeric argument"))

(defmethod ceiling2 ((a complex-number) (b numeric))
  (error "Improper numeric argument"))

(defmethod round1 ((a number))
  (lisp:round a))

(defmethod round1 ((a rational-integer))
  (values a (zero (domain-of a))))

(defmethod round1 ((a rational-number))
  (multiple-value-bind (q r)
      (lisp:round (qo-numerator a) (qo-denominator a))
    (values (make-element (domain-of a) q)
	    (make-element (domain-of a) r))))

(defmethod round1 ((a floating-point-number))
  (multiple-value-bind (q r) (lisp:round (fp-value a))
    (values (make-element (domain-of a) q)
	    (make-element (domain-of a) r))))

(defmethod round1 ((a complex-number))
  (error "Improper numeric argument"))

(defmethod round2 ((a number) (b number))
  (lisp:round a b))

(defmethod round2 ((a numeric) (b numeric))
  (multiple-value-bind (q r) (lisp:round (convert-to-lisp-number a)
					 (convert-to-lisp-number b))
    (values (make-element (domain-of a) q)
	    (make-element (domain-of b) r))))

(defmethod round2 ((a numeric) (b complex-number))
  (error "Improper numeric argument"))

(defmethod round2 ((a complex-number) (b numeric))
  (error "Improper numeric argument"))

(defmethod gcd ((a integer) (b integer))
  (lisp:gcd a b))

;; Do we really need this???
(defmethod gcd ((a float) (b float))
  a)

(defmethod-sd gcd ((a rational-integer) (b rational-integer))
  (make-element domain (lisp:gcd (integer-value a) (integer-value b))))

(defmethod-sd gcd ((a rational-integer) (b rational-number))
  (one domain))

(defmethod-sd gcd ((a rational-integer) (b floating-point-number))
  (one domain))

(defmethod-sd gcd ((a rational-number) (b rational-integer))
  (one domain))

(defmethod-sd gcd ((a rational-number) (b rational-number))
  (one domain))

(defmethod-sd gcd ((a rational-number) (b floating-point-number))
  (one domain))

(defmethod-sd gcd ((a floating-point-number) (b rational-integer))
  (one domain))

(defmethod-sd gcd ((a floating-point-number) (b rational-number))
  (one domain))

(defmethod-sd gcd ((a floating-point-number) (b floating-point-number))
  (one domain))


(defmethod lcm ((a integer) (b integer))
  (lisp:* (lisp:/ a (lisp:gcd a b)) b))

(defmethod-sd lcm ((a rational-integer) (b rational-integer))
  (let ((a (integer-value a))
	(b (integer-value b)))
    (make-element domain (lisp:* (lisp:/ a (lisp:gcd a b)) b))))

(defmethod-sd lcm ((a rational-integer) (b rational-number))
  (* a b))

(defmethod-sd lcm ((a rational-integer) (b floating-point-number))
  (* a b))

(defmethod-sd lcm ((a rational-number) (b rational-integer))
  (* a b))

(defmethod-sd lcm ((a rational-number) (b rational-number))
  (* a b))

(defmethod-sd lcm ((a rational-number) (b floating-point-number))
  (* a b))

(defmethod-sd lcm ((a floating-point-number) (b rational-integer))
  (* a b))

(defmethod-sd lcm ((a floating-point-number) (b rational-number))
  (* a b))

(defmethod-sd lcm ((a floating-point-number) (b floating-point-number))
  (* a b))

