;;; -*- Mode:Lisp; Package:Weyli; Base:10; Lowercase:T; Syntax:Common-Lisp -*-
;;; ===========================================================================
;;;			  Algebraic Domains
;;; ===========================================================================
;;; (c) Copyright 1989, 1992 Cornell University

;;; $Id: algebraic-domains.lisp,v 2.21 1992/12/07 22:03:26 rz Exp $

(in-package "WEYLI")

(defclass set (domain)
     ((equal-function :initform #'binary=
		      :initarg :equality
		      :accessor set-elt-equal))
  (:documentation "A class for finite, unordered sets"))

(defgeneric binary= (x y))
(defgeneric make-element (domain obj &rest rest))

;; Also used by AVL trees
(defclass has-comparison ()
     ((compare-function :initform #'binary>
			:initarg :compare-function
			:accessor set-elt-greaterp)))

;; Ordered set's presume the existence of a > predicate.  We provide a
;; default for >= but that is likely to also be provided primitively.
;; < and <= are handled via macros since they just involve changing
;; the order of the arguments.

(defclass ordered-set (set has-comparison)
     ())

(defgeneric binary> (x y))

(defgeneric binary>= (x y))
(defmethod binary>= (x y)
  (or (binary> x y) (binary= x y)))

(defsubst binary< (x y) (binary> y x))

(defsubst binary<= (x y) (binary>= y x))


;; The following is used to determine if floating point numbers can be
;; elements of the domain.  
(defclass complete-set (set)
     ())

;; Define-opeartions for sets is in sets.lisp

(defclass SemiGroup (set)
  ())

(define-operations semigroup
  (times (element self) (element self)) -> (element self)
  (expt (element self) positive-integer) -> (element self))

(defgeneric times (x y))
(defgeneric expt (x y))

(defclass Monoid (semigroup)
  ())

(define-operations Monoid
  (one self) -> (element monoid)
  (1? (element self)) -> Boolean
  (expt (element self) integer) -> (element self))

(defgeneric one (x))
(defmethod one ((domain domain))
  (coerce 1 domain))

(defgeneric 1? (x))

(defclass group (monoid)
  ())

(define-operations group
  (recip (element self)) -> (element self)
  (expt (element self) Integer) -> (element self))

(defgeneric recip (x))

(defclass abelian-semigroup (set)
  ())

(define-operations abelian-semigroup
  (plus (element self) (element self)) -> (element self)
  (times Integer (element self)) -> (element self))

(defgeneric plus (x y))

(defclass abelian-monoid (abelian-semigroup)
  ())

(define-operations abelian-monoid
  (zero self) -> (element self)
  (0? (element self)) -> Boolean
  (times integer (element self)) -> (element self))

(defgeneric zero (domain))
(defmethod zero ((domain domain))
  (coerce 0 domain))

(defgeneric 0? (x))

(defclass ordered-abelian-monoid (abelian-monoid ordered-set)
     ())

(defclass abelian-group (abelian-monoid)
  ())

(define-operations abelian-group
  (minus (element self)) -> (element self)
  (difference (element self) (element self)) -> (element self)
  (times integer (element self)) -> (element self))

(defgeneric minus (x))
(defgeneric difference (x y))

(defclass ordered-abelian-group (abelian-group ordered-set)
     ())

;; This is the mathematical definition of a RING.  It has just the
;; operations plus and times, and times distributes over plus.  In
;; most cases however we usually mean somewhat more.
(defclass rng (semigroup abelian-group)
  ())

#+IGNORE
(defaxiom rng ()
  (leftdistributive times plus)
  (rightDistributive times plus))


(Defclass simple-ring (rng monoid)
     ())

(define-operations simple-ring
  (characteristic self) -> Integer
  (one self) -> (element self)
  (1? (element self)) -> (element self)
  (recip (element self)) -> (element self))

(defclass has-coefficient-domain ()
  ((coefficient-domain :initform nil
		       :initarg :coefficient-domain
		       :reader coefficient-domain)))

(defvar *coefficient-domain* ()
  "Within the context of a polynomial operation, the coefficient domain")

(defmethod %bind-dynamic-domain-context :around
     ((domain has-coefficient-domain) function)
  (with-slots (coefficient-domain) domain
    (let ((*coefficient-domain* coefficient-domain))
      (call-next-method domain function))))

(defclass module (abelian-group has-coefficient-domain)
     ())

(defmethod characteristic ((domain module))
  (characteristic (coefficient-domain domain)))

;; The coefficient domain of an algebra should be a SIMPLE-RING 
(defclass algebra (module semigroup)
  ())

(defclass ring (algebra simple-ring)
     ()
  ;; Also has the distributive law
  )

(defclass ordered-ring (ring ordered-set)
     ())

(defmethod-sd max-pair ((x domain-element) (y domain-element))
  (if (> x y) x y))

(defmethod-sd min-pair ((x domain-element) (y domain-element))
  (if (> x y) y x))


(defclass integral-domain (ring)
     ()
  ;; No zero divisors
  )

(define-operations integral-domain
  ;; Unit coefficient associate
  (unit-normal (element self)) ->
    (values (element self) (element self) (element self))
  (associates? (element self) (element self)) -> Boolean
  (unit? (element self)) -> Boolean)

(defclass gcd-domain (integral-domain)
  ())

(define-operations gcd-domain
  (gcd (element self) (element self)) -> (element self)
  (lcm (element self) (element self)) -> (element self))

(defgeneric gcd (x y))
(defgeneric lcm (x y))

(defclass unique-factorization-domain (gcd-domain)
  ())

(define-operations unique-factorization-domain
  (prime? (element self)) -> Boolean
  (square-free (element self)) -> (element (factored-form self))
  (factor (element self)) -> (element (factored-form self)))

(defclass euclidean-domain (gcd-domain)
  ())

(define-operations euclidean-domain
  (sizelp (element self) (element self)) -> boolean
  (divide (element self) (element self)) -> (values (element self) (element self))
  (quotient (element self) (element self)) -> (element self)
  (remainder (element self) (element self)) -> (element self))

(defclass skew-field (ring)
  ())

(defclass field (skew-field)
  ())

(define-operations field
  (quotient (element self) (element self)) -> (element self)
  (recip (element self)) -> (element self))

(defclass finite-field (field finite-set)
  ())

(defclass ordered-field (field ordered-set)
     ())

(defclass algebraic-extension (ring)
     ())

(defclass simple-field-extension (algebraic-extension field)
     ())

;; A domain that has a dimension
(defclass dimensional-domain (domain)
     ((dimension :initform nil
		 :initarg :dimension
		 :reader dimension)))

(defclass free-module (module dimensional-domain)
     ())

(defclass vector-space (free-module)
     ()
  ;; Coefficient domain must be a field
  )

(defclass projective-space (free-module)
	 ())

(defclass differential-ring (ring)
     ())

(define-operations differential-ring
  (deriv (element self)) -> (element self))

(defclass quotient-ring (domain)
  ())

;; Quotient Fields

(defclass Quotient-Field (field)  
  ((ring :initform nil :initarg :ring
	 :reader QF-ring)
   (zero :initform nil)
   (one :initform nil)))

(defmethod characteristic ((domain quotient-field))
  (characteristic (QF-ring domain)))

;; The accessors here must not be numerator and denominator because
;; sometimes the internal structure is not a domain element and we
;; actually want to get our hands on the internal structure.
;; NUMERATOR and DENOMINATOR always return domain elements.

(defclass quotient-element (domain-element)
  ((numerator :accessor qo-numerator
	      :initarg :numerator)
   (denominator :accessor qo-denominator
		:initarg :denominator)))

#+Genera
(defmacro with-numerator-and-denominator
    ((num denom) quotient-element &body body &environment env)
  (scl:once-only (quotient-element &environment env)
    `(let ((,num (qo-numerator ,quotient-element))
	   (,denom (qo-denominator ,quotient-element)))
       ,@body)))

#-Genera
(defmacro with-numerator-and-denominator
    ((num denom) quotient-element &body body)
  `(let ((,num (qo-numerator ,quotient-element))
	 (,denom (qo-denominator ,quotient-element)))
     ,@body))

;; Non strict domains!!  I.e. if its syntactically possible to do an
;; operation on the operands, then you should go ahead and do it.

(defclass non-strict-domain ()
     ())

;;; Concrete classes

;; Sets

(defclass mutable-set (set)
     ()
  (:documentation "Sets built from this class can be modified"))


(defclass finite-set (set)
	 ())

(defclass set-element (domain-element)
     ((key :reader element-key
	   :initarg :key)))

(defclass set-element1 (set-element)
     ())

(defclass set-element2 (set-element)
     ((value :accessor element-value
	     :initarg :value)))

(defclass set-elements-as-singletons (set)
     ())

(defclass set-elements-as-pairs (set)
     ())

(defclass set-with-element-list (set)
     ((elements :accessor set-element-list
		:initform (list nil)
		:initarg :elements)))

(defclass mutable-set-with-element-list (set-with-element-list mutable-set)
     ())

(defclass set-with-sorted-element-list (ordered-set set-with-element-list)
     ())

(defclass mutable-set-with-sorted-element-list (ordered-set mutable-set-with-element-list)
     ())

;; The intiable sets classes

(defclass simple-set (mutable-set-with-element-list set-elements-as-singletons)
  ())

(defclass set-of-pairs (mutable-set-with-element-list set-elements-as-pairs)
  ())

(defclass ordered-simple-set
    (mutable-set-with-sorted-element-list set-elements-as-singletons)
  ())

(defclass ordered-set-of-pairs
    (mutable-set-with-sorted-element-list set-elements-as-pairs) 
  ())


;; Numbers of all sorts

;; All numeric quantities are built from this class (including
;; transcendental elements like e and pi when they exist).
(defclass numeric (domain-element)
     ())

(defsubst number? (x)
  (or (typep x 'lisp:number)
      (typep x 'numeric)))

;; Rational integers

(defclass rational-integers (gcd-domain caching-zero-and-one ordered-set)
  ())

(defmethod characteristic ((domain rational-integers))
  0)

(defclass rational-integer (numeric)
     ((value :initarg :value
	     :reader integer-value)))

;; Real numbers 

(defclass real-numbers (ordered-field complete-set)
     ())

(defmethod characteristic ((domain real-numbers))
  0)

(defclass real-number (numeric)
     ())

(defclass floating-point-number (real-number)
     ((value :initarg :value
	     :reader fp-value)))

(defclass bigfloat (real-number)
     ((mantissa :reader bigfloat-mantissa
		:initarg :mantissa)
      (exponent :reader bigfloat-exponent
		:initarg :exponent)))


(defclass complex-numbers (algebraic-extension field complete-set)
     ())

(defmethod characteristic ((domain complex-numbers))
  0)

(defclass complex-number (numeric)
     ((real :initarg :realpart
	    :reader cn-realpart)
      (imag :initarg :imagpart
	     :reader cn-imagpart)))

;;  Rational Numbers

(defclass rational-numbers (field ordered-set)
     ())

(defmethod characteristic ((domain rational-numbers))
  0)

(defclass rational-number (quotient-element numeric)
     ())


;; Finite fields 

(defclass GFp (field)
     ((characteristic :initform 0
		      :initarg :characteristic
		      :reader characteristic)))

(defclass GFq (GFp)
     ((degree :initarg :degree
	      :reader field-degree)))

(defclass GFp-element (numeric)
  ((value :reader gfp-value
	  :initarg :value)))

(defclass GF2^n (GFq)
     ((reduction-table :initarg :reduction-table
		       :reader GFp-reduction-table)))

(defclass GFm (rng)
  ())

(defclass GFm-element (numeric)
  ((value :initarg :value)
   (modulus :initarg :modulus)))

;; Polynomials

;; These are the pieces that are common to all polynomial domains and
;; polynomial representations.
(defclass has-ring-variables ()
     ((variables :initform nil
		 :initarg :variables
		 :reader ring-variables)))

;;FIXTHIS I don't think this is quite right.  I.e. Its not a GCD
;;domain for any coefficient domain.
(defclass polynomial-ring (gcd-domain module has-ring-variables)
  ())

;; Multivariate Polynomial rings need some structure to manage the their
;; variables.  This class provides hash tables and accessor methods of
;; this purpose.  This class is implementational.
(defclass variable-hash-table (has-ring-variables)  
  ((variable-hash-table :initform nil
			:accessor variable-hash-table)
   (variable-table :initform nil
		   :accessor variable-index-table)))

;; It is often useful to cache the values of zero and one since they are
;; often needed.  Need to include the class domain here to make
;; caching... more specific than just domain.
(defclass caching-zero-and-one (domain)
     ((zero)
      (one)))

;; Multivariate polynomials

(defclass multivariate-polynomial-ring
	  (polynomial-ring variable-hash-table caching-zero-and-one)
  ())


;; This is just the root of the polynomial structural type hierarchy.
(defclass polynomial (domain-element)
     ())

;; The following are the two different representation that are used.
;; An mpolynomial uses a recursive structure in the variables, while a
;; epolynomial is an expanded representation that uses exponent vectors.

(defclass mpolynomial (polynomial)
  ((form :reader poly-form
	 :initarg :form)))

(defclass epolynomial (polynomial)  
  ((form :reader poly-form
	 :initarg :form)
   (compare-function :reader compare-function
		     :initarg :compare-function)))

;; Univariate polynomials

(defclass upolynomial (polynomial)
  ((coef-list :reader poly-form
              :initarg :form)))

;; Rational functions

(defclass rational-function-field (quotient-field)  
  ())

(defclass rational-function (quotient-element)
  ())

;; Morphisms

(defclass morphism ()
     ((domain :reader morphism-domain
	      :initarg :domain)
      (map :reader morphism-map
	   :initarg :map)
      (range :reader morphism-range
	     :initarg :range))
  )

(defclass homomorphism (morphism)
     ())

(defclass automorphism (homomorphism)
     ())

;; Differential domains

(defclass differential-polynomial-ring
    (multivariate-polynomial-ring differential-ring)
  ())

;; Algebraic Extensions

(defclass algebraic-extension-ring 
     (algebraic-extension multivariate-polynomial-ring)
     ())

(defclass algebraic-object (mpolynomial)
     ())


;; Direct Sums

;; These are the root classes.  Classes like DIRECT-SUM-SEMIGROUP are
;; created in the direct-sum.lisp file along with several support
;; methods.

(defclass direct-sum (domain tuple) ())

(defclass direct-sum-element (domain-element tuple) ())


;; Vector Spaces

(defclass free-module-element (domain-element tuple)
     ())

(defclass vector-space-element (free-module-element)
     ())

;; This optimization is included because lisp vectors are used as
;; exponents in the expanded polynomial representation.
(defclass lisp-vector-space (vector-space)
  ())

(defclass lisp-vector (vector-space-element)
     ())

;; Projective spaces

(defclass projective-space-element (vector-space-element)
     ())

;; Matrices

;; This is is the domain of all matrices over a given ring.
(defclass matrix-space (module) ())

(defclass real-matrix-space (matrix-space) ())
(defclass complex-matrix-space (matrix-space) ())

(defclass GL-n (group has-coefficient-domain dimensional-domain) 
     ()
  (:documentation "General linear group"))

(defclass PSL-n (GL-n)
     ())

(defclass SL-n (PSL-n)
     ())

(defclass O-n (GL-n)
     ())

(defclass SO-n (O-n)
     ())

(defclass matrix-element (domain-element)
     ((value :initarg :value
	     :reader matrix-value)))

(defclass matrix-space-element (matrix-element)
     ((dimension1 :initarg :dimension1)
      (dimension2 :initarg :dimension2)))

;; These two classes are for efficiency
(defclass real-matrix-space-element (matrix-space-element) ())
(defclass complex-matrix-space-element (matrix-space-element) ())

(defclass GL-n-element (matrix-element)
     ())

(defclass PSL-n-element (GL-n-element)
     ())

(defclass SL-n-element (PSL-n-element)
     ())

(defclass O-n-element (GL-n-element)
     ())

(defclass SO-n-element (O-n-element)
     ())


;; Topoogical Domains

;; Abstract spaces don't necessarily have a well defined dimension
(defclass abstract-space (domain) ())

(defclass space (abstract-space dimensional-domain) ())

(defclass euclidean-space (space vector-space) ())


(defvar *point-counter* 0
  "Counter for all points.  This is used to establish an ordering.")

(defclass point (domain-element)
     ((order-number :initform (incf *point-counter*)
		    :reader point-order-number)))

(defclass abstract-point (point)
    ((name :initarg :name :reader point-name)))

(defclass euclidean-space-element (point vector-space-element)
     ())



