package
global(input,int2str,compile,mean,stddev,Tr,FFT2,invFFT2,fib,docview,kron)
{

function y = input(;s)
// y = input() returns a value entered by user.
// y = input("prompt") writes a prompt before asking.
local(str)
{
	if (isstr(s)) format(s);
	str = input_string();
	if (streq(str,"")) y = : else y = evalexpr(str)
};

function s = int2str(x)
// int2str(x) converts an integer x to a string.
{
	s = sformat("``",x)
};

function [] = compile(...)
// compile("file1.t",...) compiles given t-file(s) to object files.
{
	for (i=1; i<=Nargin(); i++) {
		tfile = argin(i);
		L = length(tfile);
		if (tfile[L]=='t' && tfile[L-1]=='.')
			ctfile = #(tfile[1:L-1],"ct")
		else
			ctfile = #(tfile,".ct");
		t2ct(tfile);
		system(#("telakka -c -O1 ",ctfile));
	}
};

function y = mean(x)
// mean(x) computes the arithmetic mean of a numeric array x.
{
	y=sum(x)/length(x)
};

function y = stddev(x)
// stddev(x) computes the standard deviation of a numeric array x.
// See also: mean.
{
	y = sqrt(mean( (x-mean(x))^2 ))
};

function y = Tr(x)
// Tr(x) returns the trace (diagonal sum) of a matrix.
{
	y = sum(diag(x))
};

function f = FFT2(u)
// FFT2(u) gives a 2D discrete Fourier transform of 2D array u.
// See also: invFFT2, FFT.
{
	f=FFT(FFT(u,1),2)
};

function f = invFFT2(u)
// invFFT2(u) gives the inverse 2D Fourier transform of u.
// See also: FFT2.
{
	f=invFFT(invFFT(u,1),2)
};

function f = fib(n)
// fib(n) returns the nth Fibonacci number.
// Recursive (==> very slow) implementation, useful for
// benchmarking only!
{
	if (n<2) f=n else f=fib(n-1)+fib(n-2)
};

function docview()
// docview() is an interactive program for viewing various
// kinds of Tela documentation.
local(directory,choicViewing,choiceDoc,editor,env)
{
	directory = "/usr/local/lib/tela";
	editor = "emacs";
	pager = "less -Ms";
	env = getenv("EDITOR");
	if (isstr(env)) editor = env;
	env = getenv("PAGER");
	if (isstr(env)) pager = env;
	
	format("Viewing Tela documentation:\n");
	choiceViewing = smenu("Choose viewing method",
						 "ASCII",
						 "HTML/Xmosaic (requires X)",
						 "Xdvi (requires X)",
						 "Xemacs (requires X)",
						 "Cancel");
	if (strstarteq(choiceViewing,"Cancel")) return;
	choiceDoc = smenu("Choose document to view",
					 "Basic help file",
					 "Builtin functions alphabetically sorted",
					 "Builtin functions sorted by sections",
					 "Cancel");
	if (strstarteq(choiceDoc,"Cancel")) return;
	
	if (strstarteq(choiceDoc,"Basic help"))
		file = "telahelp"
	else if (strstarteq(choiceDoc,"Builtin functions alpha"))
		file = "telafuncs"
	else if (strstarteq(choiceDoc,"Builtin functions sorted"))
		file = "telafuncsSectioned";

	if (strstarteq(choiceViewing,"ASCII"))
		system(#(pager," ",directory,"/",file,".txt"))
	else if (strstarteq(choiceViewing,"HTML"))
		system(#("cd ",directory,"/html; xmosaic ",file,".html&"))
	else if (strstarteq(choiceViewing,"Xdvi"))
		system(#("xdvi -geometry 847x570+0+0 -s 3 ",directory,"/",file,".dvi&"))
	else if (strstarteq(choiceViewing,"Xemacs"))
		system(#(editor," ",directory,"/",file,".txt&"));
};

function K = kron(A,B)
//  kron(X,Y) is the Kronecker tensor product of X and Y.
//	The result is a large matrix formed by taking all possible
//	products between the elements of X and those of Y.
//	For example, if X is 2 by 3, then KRON(X,Y) is
//
//	  #( X(1,1)*Y,  X(1,2)*Y,  X(1,3)*Y;
//	     X(2,1)*Y,  X(2,2)*Y,  X(2,3)*Y )
local(ma,na, mb,nb,i,j,jk,ik)
{

	if (rank(A)!=2 || rank(B)!=2) {K=0; return};
	
	[ma,na] = size(A);
	[mb,nb] = size(B);

	if (isint(A) && isint(B))
		K = izeros(ma*mb,na*nb)
	else if (iscomplex(A) || iscomplex(B))
		K = czeros(ma*mb,na*nb)
	else
		K = zeros(ma*mb,na*nb);

	if (ma*na <= mb*nb) {
		for (i=1; i<=ma; i++) {
			ik = 1+(i-1)*mb:i*mb;
			for (j=1; j<=na; j++) {
				jk = 1+(j-1)*nb:j*nb;
				K[ik,jk] = A[i,j]*B
			}
		}
	} else {
		for (i=1; i<=mb; i++) {
			ik=i:mb:(ma-1)*mb+i;
			for (j=1; j<=nb; j++) {
				jk = j:nb:(na-1)*nb+j;
				K[ik,jk] = B[i,j]*A;
			}
		}
	}

};

};
