/*****************************************************************************
* 
*			  NCSA HDF version 3.10r5
*			     October 24, 1991
*
* NCSA HDF Version 3.10r5 source code and documentation are in the public
* domain.  Specifically, we give to the public domain all rights for future
* licensing of the source code, all resale rights, and all publishing rights.
* 
* We ask, but do not require, that the following message be included in all
* derived works:
* 
* Portions developed at the National Center for Supercomputing Applications at
* the University of Illinois at Urbana-Champaign.
* 
* THE UNIVERSITY OF ILLINOIS GIVES NO WARRANTY, EXPRESSED OR IMPLIED, FOR THE
* SOFTWARE AND/OR DOCUMENTATION PROVIDED, INCLUDING, WITHOUT LIMITATION,
* WARRANTY OF MERCHANTABILITY AND WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE
* 
*****************************************************************************/
/*
** FILE
**	hdfpack.c
** USAGE
**	hdfpack <infile> <outfile>
** DESCRIPTION
**	This program will compact an HDF file by reading in all the data
**	elements in the file and writing them out to a new file.
** COMMENTS, BUGS, ASSUMPTIONS
**	Both arguments must be supplied to the program and they cannot be
**	identical.
**	You must have enough additional disk space for the compacted file.
** AUTHOR
**	Doug Ilg
*/

#include <stdio.h>
#include <string.h>
#include "df.h"


#if defined __STDC__ || defined PC
int main(int, char **);
int usage(char *);
int error(int);
int32 desc_comp(DFdesc *d1, DFdesc *d2);
#else
int main();
int usage();
int error();
int32 desc_comp();
#endif /* __STDC__ || PC */

#if defined __STDC__ || defined PC
main(int argc, char *argv[])
#else
main(argc, argv)
int argc;
char *argv[];
#endif /* __STDC__ || PC */
{
    int i, num_desc, desc_size, ret;
    DF *infile, *outfile;
    DFdesc *dlist;
    uint16 tag, ref;
    char *data;
    int32 data_size, len, oldoff;


/*
**   Check arguments
*/
    if (argc < 3) {
	usage(argv[0]);
	exit(255);
    }

    if (strcmp(argv[1], argv[2]) == 0) {
	usage(argv[0]);
	exit(255);
    }
    
/*
**   Check to make sure input file is HDF
*/
    ret = DFishdf(argv[1]);
    if (ret == -1) {
	printf("ERROR - input file is not an HDF file.\n");
	exit(255);
    }

/*
**   Open input and output files
*/
    infile = DFopen(argv[1], DFACC_READ, 0);

    if (infile == NULL)
	error(DFerror);

    outfile = DFopen(argv[2], DFACC_CREATE | DFACC_WRITE, 0);

    if (outfile == NULL)
	error(DFerror);

/*
**   See how many data elements there
**   are and make room for their DD's
*/
    desc_size = DFnumber(infile, DFTAG_WILDCARD);
    if (desc_size == -1)
	error(DFerror);

    dlist = (DFdesc *)malloc(desc_size * sizeof(*dlist));

/*
**   Allocate initial 1Kb for data
*/
    data_size = 1024;
    data = (char *)malloc(data_size);

/*
**   Get all DD's for data elements
*/
    num_desc = DFdescriptors(infile, dlist, 0, desc_size);
    if (num_desc == -1)
	error(DFerror);

/*
**   Sort DD's by offset
**   This id not strictly necessary, but is done "just in case"
*/
    qsort((char *)dlist, num_desc, sizeof(*dlist), desc_comp);

    oldoff = -1;
    for (i=0; i<num_desc; i++) {
	if (dlist[i].tag != DFTAG_NULL) {

/*
**   get more room for data, if necessary
*/
	    if (dlist[i].length > data_size) {
		free(data);
		data_size = dlist[i].length;
		data = (char*)malloc(data_size);
	    }

/*
**   if this DD points to different data, read the data
**   from the old file and write it to the new file
*/
	    if (dlist[i].offset != oldoff) {
	        len = DFgetelement(infile, dlist[i].tag, dlist[i].ref, data);
                if (len == -1)
	            error(DFerror);
	        ret= DFputelement(outfile, dlist[i].tag, dlist[i].ref,data,len);
                if (ret == -1)
	            error(DFerror);
	    }
	    else {
/*
**   otherwise, just make a new DD for same data
*/
		ret = DFdup(outfile, dlist[i].tag, dlist[i].ref, dlist[i-1].tag,
		      dlist[i-1].ref);
                if (ret == -1)
	            error(DFerror);
	    }

/*
**   save offset of data to check against next DD
*/
	    oldoff = dlist[i].offset;
	}
    }

/*
**   done; close files
*/
    DFclose(infile);
    DFclose(outfile);

    exit(0);
}

/*
** NAME
**	usage -- print out usage template
** USAGE
**	int usage()
** RETURNS
**	none
** DESCRIPTION
**	Print hdfpack's usage template to stdout.
** GLOBAL VARIABLES
** COMMENTS, BUGS, ASSUMPTIONS
** EXAMPLES
*/
#if defined __STDC__ || defined PC
int usage(char *name)
#else
int usage(name)
char *name;
#endif /* __STDC__ || PC */
{
    printf("Usage:  %s <infile> <outfile>\n", name);
    printf("        (file names must be different)\n");
}

/*
** NAME
**	error -- print out HDF error number
** USAGE
**	int error()
** RETURNS
**	none
** DESCRIPTION
**	Print an HDF error number to stdout.
** GLOBAL VARIABLES
** COMMENTS, BUGS, ASSUMPTIONS
**	This routine terminates the program with code 255.
** EXAMPLES
*/
#if defined __STDC__ || defined PC
int error(int num)
#else
int error(num)
int num;
#endif /* __STDC__ || PC */
{
    printf("HDF ERROR: %4d\n", num);
    exit(255);
}

/*
** NAME
**	desc_comp -- compare two DD's by offset
** USAGE
**	int desc_cmp(d1, d2)
**	DFdesc *d1		IN: pointer to a DD
**	DFdesc *d2		IN: pointer to a DD
** RETURNS
**	A number less than, greater than, or equal to 0, according to
**	whether d1's offset is less than, greater than, or equal to
**	d2's offset.
** DESCRIPTION
**	A pointer to this routine is passed to qsort() as the comparison
**	function for the sort.
** GLOBAL VARIABLES
** COMMENTS, BUGS, ASSUMPTIONS
** EXAMPLES
*/
#if defined __STDC__ || defined PC
int32 desc_comp(DFdesc *d1, DFdesc *d2)
#else
int32 desc_comp(d1, d2)
DFdesc *d1, *d2;
#endif /* __STDC__ || PC */
{
    return(d1->offset - d2->offset);
}
