#include "system.h"
#include "config.h"
#include "mem.h"
#include "mfileio.h"
#include "error.h"
#include <stdlib.h>
#include <limits.h>
#ifdef HAVE_ZLIB
#include <zlib.h>
#endif /* HAVE_ZLIB */
#include "filter.h"

static int is_white_space (char c);

int
is_white_space (char c) {

  return (c == ' ' || c ==  '\t' || c == '\f' || c == '\v' ||
	  c == '\b' || c == '\r' || c == '\n' || c == '\0');
}

unsigned long
decode_stream(char *dest, unsigned long destlen,
	      char *src, unsigned long srclen, char *filter)
{
  int status;
  unsigned long length;

  length = destlen;
#ifdef DEBUG
  fprintf(stderr, "applying filter: %s\n", filter);
#endif /* DEBUG */
  if (strcmp(filter, "ASCII85Decode") == 0) {
    status = decode_a85(dest, &length, src, srclen);
  } else if (strcmp(filter, "ASCIIHexDecode") == 0) {
    status = decode_ahx(dest, &length, src, srclen);
#ifdef HAVE_ZLIB
  } else if (strcmp(filter, "FlateDecode") == 0) {
    status = decode_flate(dest, &length, src, srclen);
#endif /* HAVE_ZLIB */
  } else {
    fprintf(stderr, "decode filter unsupported/unknown: %s\n", filter);
    status = F_FILTER_UNKNOWN;
  }

  if (status != F_OK) length = 0;
#ifdef DEBUG
  fprintf(stderr, "decoded stream lenght: %lu\n", length);
#endif /* DEBUG */
  
  return length;
}

int
decode_ahx (char *dest, unsigned long *destlen, char *src, unsigned long srclen)
{
  unsigned long filtered_length;
  char c, *end;
  int val, count;

  filtered_length = 0;
  end = src + srclen;
  if ((srclen % 2) && (*(end-1) != '>') ) return F_DATA_ERROR;

  count = 0;
  val = 0;
  while (src < end) {
    c = *src++;
    if (is_white_space(c)) continue;
    count++;

    if ( c == '>' && src == end - 1 ) { /* EOD */
      if (count == 1) {
	if ( filtered_length < *destlen ) {
	  *dest = (val & 0x0f) << 4;
	  filtered_length++;
	} else {
	  return F_BUF_ERROR;
	}
      }
      break;
    } else if ( c >= '0' && c <= '9' ) {
      val = (val << 4) + (c - '0');
    } else if (c >= 'a' && c <= 'f') {
      val = (val << 4) + (c - 'a' + 10);
    } else {
      return F_DATA_ERROR;
    }

    if (count == 2) {
      if ( filtered_length < *destlen ) {
	*dest++ = val & 0xff;
	filtered_length++;
	count = 0;
	val = 0;
      } else {
	return F_BUF_ERROR;
      }
    }
  }

  *destlen = filtered_length;
  return F_OK;
}

/*
  0000 = 0x7a (z)
  pqrs
   = (c1-33)*85**4+(c2-33)*85**3+(c3-33)*85**2+(c4-33)*85**1+(c5-33)*85**0
*/
int decode_a85(char *dest, unsigned long *destlen, char *src, unsigned long srclen)
{
  unsigned long filtered_length;
  char c, *end;
  int i, count;
  unsigned long l;

  end = src+srclen;
  filtered_length = 0;
  count = 0;
  l = 0;
  while(src < end) {
    c = *src++;
    if (is_white_space(c)) continue; /* skip "white space" characters */
    if ((c == 0x7e) && (src == end-1) &&
	(*src == 0x3e) && count != 1 ) { /* EOD */
      if ( count > 1 ) {
	if ( filtered_length < (*destlen) - (count-1) ) {
	  /* We must pad with 0x75 to avoid rounding error */
	  for (i=0;i<5-count;i++) l = l*85 + 0x75;
	  for (i=0;i<count-1;i++) {
	    *dest++ = (l >> (8*(3-i))) & 0xff;
	  }
	  filtered_length += count - 1;
	  count = 0;
	} else {
	  return F_BUF_ERROR;
	}
      }
      break;
    } else if (c >= 0x7a && count == 0) { /* z */
      if ( filtered_length < (*destlen)-4 ) {
	memset(dest, 0, 4);
	dest += 4;
	filtered_length += 4;
      } else {
	return F_BUF_ERROR;
      }
    } else if (c >= 0x21 && c <= 0x75) {
      count++;
      l = l * 85 + (c - 0x21);
      if (count == 5) { /* flush */
	if ( filtered_length < (*destlen)-4 ) {
	  for (i=0;i<4;i++) {
	    *dest++ = (l >> (8*(3-i))) & 0xff;
	  }
	  filtered_length += 4;
	  count = 0;
	  l = 0;
	} else {
	  return F_BUF_ERROR;
	}
      }
    } else {
      return F_DATA_ERROR;
    }
  }

  if (count != 0) {
    return F_DATA_ERROR;
  }

  *destlen = filtered_length;
  return F_OK;
}

#ifdef HAVE_ZLIB
int
decode_flate(char *dest, unsigned long *destlen, char *src, unsigned long srclen)
{
  int status;

  status = uncompress(dest, destlen, src, srclen);

  switch (status) {
  case Z_OK:
    status = F_OK;
    break;
  case Z_MEM_ERROR:
    status = F_MEM_ERROR;
    break;
  case Z_BUF_ERROR:
    status = F_BUF_ERROR;
    break;
  case Z_DATA_ERROR:
    status = F_DATA_ERROR;
    break;
  default:
    status = F_UNKNOWN_ERROR;
  }

  return status;
}
#endif /* HAVE_ZLIB */

