/* updmap.c

   Copyright 2000, 2015 Akira Kakuto.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with this library; if not, see <http://www.gnu.org/licenses/>.


   updmap for W32TeX (2006/04/08 --ak)
   (for one-byte fonts only)

   main map file names:
   dvips      --- psfonts.map
   pdftex     --- pdftex.map
   dvipdfmo   --- dvipdfm.map
   dvipdfmx   --- dvipdfm.map
   ps2pk      --- pspksupp.map
   mpost      --- psfonts.map

   First, default system files psfonts-base.map, pdftex-base.map,
   dvipdfm-base.map and pspksupp-base.map are copied to the
   main map files.

   Next, map files defined in texmf-dist/web2c/updmap.cfg like
   #
   # updmap.cfg (W32TeX 2004/02/03 --ak)
   #
   Map monotype.map
   ... ...

   are appended to the main map files. The map files described in
   updmap.cfg must be written by using dvips format.
   Allowed command in updmap.cfg is 'Map' only.
   'MixedMap' is also allowed, but the effect is the same as 'Map'.
   Lines containing other commands are ignored.

   The resulting lines in pdftex.map must be unique, but the uniqueness
   is not checked.

   usage:

   (1) updmap
         Recreate map files according to texmf-dist/web2c/updmap.cfg
   (2) updmap --help
         Prints help message and exits
   (3) updmap --nomkmap
         Do nothing
   (4) updmap --nomkmap --add foo1.map foo2.map ...
         If foo1.map ... are not written in updmap.cfg, add them to updmap.cfg
         without recreating map files
   (5) updmap --add foo1.map foo2.map ...
         If foo1.map ... are not written in updmap.cfg, add them to updmap.cfg
         and recreate map files
   (6) updmap --nomkmap --sub foo1.map foo2.map ...
         If foo1.map ... are written in updmap.cfg, erase them from updmap.cfg
         without recreating map files
   (7) updmap --sub foo1.map foo2.map ...
         If foo1.map ... are written in updmap.cfg, erase them from updmap.cfg
         and recreate map files

   supplementary options:
       --cnffile filename
         change configuration file (default: updmap.cfg) to 'filename'
       --quiet
         reduce verbosity
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <malloc.h>
#include <direct.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <io.h>
#include <fcntl.h>
#include <kpathsea/kpathsea.h>
#include <windows.h>

#define PSFONTS         "psfonts.map"
#define KANJI_BASE      "kanji-base.map"
#define PSFONTS_BASE    "psfonts-base.map"
#define PDFTEX          "pdftex.map"
#define PDFTEX_BASE     "pdftexdl14-base.map"
#define PDFTEX_BASEND14 "pdftexndl14-base.map"
#define DVIPDFM         "dvipdfm.map"
#define DVIPDFM_BASE    "dvipdfm-base.map"
#define PSPKSUPP        "pspksupp.map"
#define PSPKSUPP_BASE   "pspksupp-base.map"
#define UPDMAPCFG       "updmap.cfg"

#define LGBUF         2048
#define MDBUF         256
#define SMBUF         128
#define TIBUF         64

#define BBUFF_FUL     253
#define FFILE_END     254
#define LLINE_END     255

/* Global variables */
int FileName = 0;
int Verbose = 1;
int Bluesky = 0;
int Ndl14   = 0;
char Programname[256];

/* calculated but not used at present */
char Apk[4];
char Bpk[4];
char Spk[4];


/* error message */
void
fatal (char *s)
{
  fprintf (stderr, "%s\n", s);
}

/* check a directory */
static int
is_dir (char *buff)
{
  struct stat stats;

  return stat (buff, &stats) == 0 && S_ISDIR (stats.st_mode);
}

/* make a directory */
int
make_dir (char *buff)
{
  if (_mkdir (buff)) {
    fprintf(stderr, "mkdir %s error.\n", buff);
    return (1);
  }
  if (_chmod (buff, _S_IREAD | _S_IWRITE)) {
    fprintf(stderr, "chmod %s failed.\n", buff);
    return (1);
  }
  return (0);
}

int make_dir_p(char *buff)
{
  int  ret = 0;
  int  i = 0;
  char *p = buff;

  while (1) {
    if(*p == '\0') {
      ret = 0;
      if(!is_dir(buff)) {
        if(make_dir(buff)) {
          ret = 1;
        }
      }
      break;
    }
    if(*p == '/' && (i > 0 && *(p-1) != ':')) {
      *p = '\0';
      if(!is_dir(buff)) {
        if(make_dir(buff)) {
          ret = 1;
          *p = '/';
          break;
        }
      }
      *p = '/';
    }
    p++;
    i++;
  }
  return ret;
}

void
usage(void)
{
  fprintf(stdout,"Usage:\n");
  fprintf(stdout,"(1) updmap\n");
  fprintf(stdout,"    : recreate map files\n");
  fprintf(stdout,"(2) updmap --add foo1.map foo2.map ...\n");
  fprintf(stdout,"    : add foo1.map ... to configuration file and recreate map files\n");
  fprintf(stdout,"(3) updmap --nomkmap --add foo1.map foo2.map ...\n");
  fprintf(stdout,"    : add foo1.map ... to configuration file without recreating map files\n");
  fprintf(stdout,"(4) updmap --sub foo1.map foo2.map ...\n");
  fprintf(stdout,"    : erase foo1.map ... from configuration file and recreate map files\n");
  fprintf(stdout,"(5) updmap --nomkmap --sub foo1.map foo2.map ...\n");
  fprintf(stdout,"    : erase foo1.map ... from configuration file without recreating map files\n");
  fprintf(stdout,"(6) updmap --help\n");
  fprintf(stdout,"    : print this message and exit\n\n");
  fprintf(stdout,"Supplementary options:\n");

  fprintf(stdout,"    --cnffile filename\n");
  fprintf(stdout,"    : change configuration file (default: updmap.cfg) ");
  fprintf(stdout,"to 'filename'\n");
  fprintf(stdout,"    --ndl14\n");
  fprintf(stdout,"    : base14 fonts are not downloaded in pdfTeX.\n");
  fprintf(stdout,"    --quiet\n");
  fprintf(stdout,"    : reduce verbosity\n");
}

/* FILE *fi should be in binary mode */
int
ffgets(char *buf, int size, FILE *fi)
{
  char *p;
  int  c;
  int  n;

  n = 0;
  p = buf;
  *p = '\0';

  while(1) {
    c = getc(fi);
    if(c == 26)
      continue;
    n++;
    if(c == EOF) {
      n++;
      *p++ = '\n';
      *p = '\0';
      return FFILE_END;
    }
    else if(c == 0x0d || c == 0x0a) {
      n++;
      *p++ = '\n';
      *p = '\0';
      if(c == 0x0d) {
        c = getc(fi);
        if(c != 0x0a)
          ungetc(c, fi);
      }
      return LLINE_END;
    }
    else if(n == (size - 3)) {
      *p++ = c; *p = '\0';
      return BBUFF_FUL;
    }
    else *p++ = c;
  }
}

int
isskip(char c)
{
  if((c == ' ') || (c == '\t'))
    return 1;
  else if((c == '<') || (c == '[')) {
    if(FileName == 0)
      FileName = 1;
    else if(FileName == 1)
      FileName = 2;
    return 1;
  }
  else {
    return 0;
  }
}

void
skipchar(char **p)
{
  while(isskip(**p)) {
    if(**p == '<' || **p == '[') {
      char *ptr = Spk;
      while(**p == '<' || **p == '[') {
        *ptr++ = *(*p)++;
      }
      *ptr = '\0';
    } else {
      (*p)++;
    }
  }
  if(Spk[0]) {
    if(Apk[0] == 0) {
      strcpy(Apk, Spk);
    } else if(Bpk[0] == 0) {
      strcpy(Bpk, Spk);
    }
    Spk[0] = '\0';
  }
}

void
copy_base(FILE *dst, FILE *src)
{
  int ret;
  char buf[LGBUF];
  for( ; ; ) {
    ret = ffgets(buf, LGBUF - 4, src);
    fputs(buf, dst);
    if(ret == FFILE_END)
      break;
  }
}

char *
get_map_name(char *name)
{
  char *fullpath = NULL;
  fullpath = kpse_find_file(name, kpse_fontmap_format, 0);
  return fullpath;
}

char *
get_cfg_name(char *name)
{
  return kpse_find_file(name, kpse_web2c_format, 0);
}

void
do_update(int flag, char *mapname, FILE *mapfile, FILE *psfonts,
          FILE *pdftex, FILE *dvipdfm, FILE *pspksupp)
{
  char buf[LGBUF];
  char tfmname[SMBUF];
  char psname[SMBUF];
  char pscommand[SMBUF];
  char a[SMBUF];
  char b[SMBUF];
  char ab[SMBUF];
  char slantstr[TIBUF];
  char extendstr[TIBUF];
  char encname[SMBUF];
  char pfbname[SMBUF];
  char pfaname[SMBUF];
  char otfname[SMBUF];
  char ttfname[SMBUF];
  char t42name[SMBUF];
  char *p, *q;
  double slant, extend;
  int len;
  int ret = 0;

  do {
    ret = ffgets(buf, LGBUF - 4, mapfile);
    if(ret == BBUFF_FUL) {
      fprintf(stdout, "A line in %s seems to be too long.\n", mapname);
      fprintf(stdout, "I try to make an entry. But it may be wrong.\n");
    }
    tfmname[0] = psname[0] = pscommand[0] = a[0] = b[0] = '\0';
    slantstr[0] = extendstr[0] = encname[0] = pfbname[0] = '\0';
    pfaname[0] = otfname[0] = ttfname[0] = t42name[0] = '\0';
    FileName = 0; Spk[0] = Apk[0] = Bpk[0] = '\0';
    p = buf;
    skipchar(&p);
    if((*p == '%') || (*p == '#') || (*p == '\n'))
      continue;
    q = tfmname;
    while(!isskip(*p) && (*p != '\n'))
      *q++= *p++;
    *q = '\0';
    skipchar(&p);
    if((*p == '%') || (*p == '#') || (*p == '\n')) {
      fprintf(stdout, "I ignore an incorrect line in \"%s\".\n", mapname);
      continue;
    }
    if (FileName == 1) {
      q = a;
    } else
      q = psname;
    while(!isskip(*p) && (*p != '\n'))
      *q++ = *p++;
    *q = '\0';
    skipchar(&p);
/*
skip flags in other cases than pdftex and dvips
*/
    if(FileName == 0) {
      while(isdigit(*p))
        p++;
    }
    skipchar(&p);
    if((*p == '%') || (*p == '#') || (*p == '\n'))
      goto pprt;
    if(*p == '\"') {
      q = pscommand;
      *q++ = *p++;
      while(*p != '\"')
        *q++ = *p++;
      *q++ = *p++;
      *q = '\0';
      skipchar(&p);
      if((*p == '%') || (*p == '#') || (*p == '\n'))
        goto pprt;
    }
    if (FileName == 1 && a[0] == '\0') {
      q = a;
    } else if (FileName == 2 && b[0] == '\0') {
      q = b;
    } else {
      fprintf(stdout, "I ignore an incorrect line in \"%s\".\n", mapname);
      continue;
    }
    while(!isskip(*p) && (*p != '\n'))
      *q++ = *p++;
    *q = '\0';
    skipchar(&p);
    if((*p == '%') || (*p == '#') || (*p == '\n'))
      goto pprt;

/* allow ps command here also */

    if(*p == '\"') {
      q = pscommand;
      *q++ = *p++;
      while(*p != '\"')
        *q++ = *p++;
      *q++ = *p++;
      *q = '\0';
      skipchar(&p);
      if((*p == '%') || (*p == '#') || (*p == '\n'))
        goto pprt;
    }

    if (FileName == 1 && a[0] == '\0') {
      q = a;
    } else if (FileName == 2 && b[0] == '\0') {
      q = b;
    } else {
      fprintf(stdout, "I ignore an incorrect line in \"%s\".\n", mapname);
      continue;
    }

    while(!isskip(*p) && (*p != '\n'))
      *q++ = *p++;
    *q = '\0';

    skipchar(&p);
    if((*p == '%') || (*p == '#') || (*p == '\n'))
      goto pprt;

/* allow ps command in the final part also */

    if(*p == '\"') {
      q = pscommand;
      *q++ = *p++;
      while(*p != '\"')
        *q++ = *p++;
      *q++ = *p++;
      *q = '\0';
      skipchar(&p);
      if((*p == '%') || (*p == '#') || (*p == '\n'))
        goto pprt;
    }

    skipchar(&p);
    if((*p == '%') || (*p == '#') || (*p == '\n'))
      goto pprt;
    else {
      fprintf(stdout, "I ignore an incorrect line in \"%s\".\n", mapname);
      continue;
    }
  pprt:
    if(pscommand[0]) {
      p = strstr(pscommand, "SlantFont");
      if(p) {
        p--;
        while(*p == ' ' || *p == '\t') p--;
        while(*p != ' ' && *p != '\t' && *p != '\"') p--;
        p++;
        sscanf(p, "%lf SlantFont", &slant);
        sprintf(slantstr, "%lf", slant);
        p = slantstr + strlen(slantstr) - 1;
        while(*p == '0') {
          *p = '\0';
          p--;
        }
      }
      p = strstr(pscommand, "ExtendFont");
      if(p) {
        p--;
        while(*p == ' ' || *p == '\t') p--;
        while(*p != ' ' && *p != '\t' && *p != '\"') p--;
        p++;
        sscanf(p, "%lf ExtendFont", &extend);
        sprintf(extendstr, "%lf", extend);
        p = extendstr + strlen(extendstr) - 1;
        while(*p == '0') {
          *p = '\0';
          p--;
        }
      }
    }

    if(a[0]) {
      strcpy(ab, a);
      p = strrchr(ab, '.');
      if(p && !stricmp(p, ".enc")) {
        *p = '\0';
        strcpy(encname, ab);
      } else if(p && !stricmp(p, ".pfb")) {
        *p = '\0';
        strcpy(pfbname, ab);
      } else if(p && !stricmp(p, ".pfa")) {
        *p = '\0';
        strcpy(pfaname, ab);
      } else if(p && !stricmp(p, ".otf")) {
        *p = '\0';
        strcpy(otfname, ab);
      } else if(p && !stricmp(p, ".ttf")) {
        *p = '\0';
        strcpy(ttfname, ab);
      } else if(p && !stricmp(p, ".t42")) {
        *p = '\0';
        strcpy(t42name, ab);
      } else {
        strcpy(pfbname, ab);
      }
    }
    if(b[0]) {
      strcpy(ab, b);
      p = strrchr(ab, '.');
      if(p && !stricmp(p, ".enc")) {
        *p = '\0';
        strcpy(encname, ab);
      } else if(p && !stricmp(p, ".pfb")) {
        *p = '\0';
        strcpy(pfbname, ab);
      } else if(p && !stricmp(p, ".pfa")) {
        *p = '\0';
        strcpy(pfaname, ab);
      } else if(p && !stricmp(p, ".otf")) {
        *p = '\0';
        strcpy(otfname, ab);
      } else if(p && !stricmp(p, ".ttf")) {
        *p = '\0';
        strcpy(ttfname, ab);
      } else if(p && !stricmp(p, ".t42")) {
        *p = '\0';
        strcpy(t42name, ab);
      } else {
        strcpy(pfbname, ab);
      }
    }

/* dvips */
    if(flag & 1) {
      if(psname[0] != '\0' && otfname[0] == '\0' && ttfname[0] == '\0') {
        fputs(buf, psfonts);
      }
    }

/* pdfTeX */
    if(flag & 2) {
      /* simply output a line */
      fputs(buf, pdftex);
#if 0
      if(tfmname[0])
        fprintf(pdftex, "%s", tfmname);
      /* pdftex bug: erase PSname in SlantFont or ExtendFont cases */
      /* This bug seems to be fixed in the recent pdfTeX
      if((slantstr[0] == '\0') && (extendstr[0] == '\0') && psname[0])
      */
      if(psname[0])
        fprintf(pdftex, " %s", psname);
      if(pscommand[0])
        fprintf(pdftex, " %s", pscommand);
      if(a[0]) {
        fprintf(pdftex, " %s%s", Apk, a);
      }
      if(b[0]) {
        fprintf(pdftex, " %s%s", Bpk, b);
      }
      fprintf(pdftex, "\n");
#endif /* 0 */
    }

/* dvipdfmx */
    if((flag & 4) && !Bluesky) {
      if((pfbname[0] || ttfname[0] || otfname[0] || t42name[0]) && tfmname[0]) {
        len = (int)strlen(tfmname);
        if (tfmname[len-1] != '@') {
          fprintf(dvipdfm, "%s", tfmname);
          if(encname[0] == '\0')
            fprintf(dvipdfm, " default");
          else
            fprintf(dvipdfm, " %s", encname);
          if(pfbname[0]) {
            fprintf(dvipdfm, " %s", pfbname);
          } else if(ttfname[0]) {
            fprintf(dvipdfm, " %s.ttf", ttfname);
          } else if(otfname[0]) {
            fprintf(dvipdfm, " %s.otf", otfname);
          } else if(t42name[0]) {
            fprintf(dvipdfm, " %s.ttf", t42name);
          }
          if(slantstr[0])
            fprintf(dvipdfm, " -s %s", slantstr);
          if(extendstr[0])
            fprintf(dvipdfm, " -e %s", extendstr);
          fprintf(dvipdfm, "\n");
        }
      }
    }

/* ps2pk */
    if((flag & 8) && !Bluesky) {
      p = kpse_find_file(pfbname, kpse_afm_format, 0);
      if(p) {
        free(p);
        if(pfbname[0]) {
          fprintf(pspksupp, "%s", tfmname);
          fprintf(pspksupp, " %s.pfb", pfbname);
          if(slantstr[0] == '\0')
            fprintf(pspksupp, " 0");
          else
            fprintf(pspksupp, " %s", slantstr);
          if(extendstr[0] == '\0')
            fprintf(pspksupp, " 1");
          else
            fprintf(pspksupp, " %s", extendstr);
          if(encname[0])
            fprintf(pspksupp, " %s.enc", encname);
          fprintf(pspksupp, "\n");
        }
      }
    }
  } while(ret != FFILE_END);
}

#define __TMPDIR     "."
#define TEMPLATE     "/updmap.XXXXXX"

int main(int argc, char *argv[])
{
  int mkmap = 1;
  int addok;
  int subtracted;
  char *_tmpd;
  char tmp[MDBUF];
  char addmap[SMBUF];
  char submap[SMBUF];
  char cfgname[TIBUF];
  char fullcfgname[MDBUF];
  int i;

  int ifirst, ilast;
  int addflag, subflag;

  int flag = 0;
/* 
   1 : dvips
   2 : pdftex
   4 : dvipdfmx
   8 : ps2pk
*/
  FILE *psfonts=NULL, *psfonts_base=NULL;
  FILE *pdftex=NULL,   *pdftex_base=NULL;
  FILE *dvipdfm=NULL,  *dvipdfm_base=NULL;
  FILE *pspksupp=NULL, *pspksupp_base=NULL;
  FILE *updmapcfg=NULL;
  FILE *tmpupdmapcfg;
  char *p;
  FILE *mapfile=NULL;
  char mapname[SMBUF];

  char buffer[LGBUF];
  char texmfmainfontsmap[MDBUF];
  char mainmapname[MDBUF];

  kpse_set_program_name(argv[0], NULL);
  strcpy (Programname, kpse_program_name);

/*
  Check that the TeX system is W32TeX or not.
*/

  p = kpse_var_value("jtex_filetype");
  if(p == NULL) {
    fprintf(stderr, "This updmap should be used only on W32TeX.\n");
    fprintf(stderr, "I exit with doing nothing.\n");
    return 1;
  }
  free(p);
  strcpy(cfgname, UPDMAPCFG);
  addflag = 0;
  subflag = 0;

  if(argc > 1) {
    for(i=1; i < argc; i++) {
      if(!strncmp(argv[i], "-h", 2) ||
         !strncmp(argv[i], "--h", 3)) {
        usage();
        return 0;
      }
      else if(!strncmp(argv[i], "-nd", 3) ||
              !strncmp(argv[i], "--nd", 4)) {
        Ndl14 = 1;
      }
      else if(!strncmp(argv[i], "-no", 3) ||
              !strncmp(argv[i], "--no", 4)) {
        mkmap = 0;
      }
      else if(!strncmp(argv[i], "-q", 2) ||
              !strncmp(argv[i], "--q", 3)) {
        Verbose = 0;
      }
      else if(!strncmp(argv[i], "-a", 2) ||
              !strncmp(argv[i], "--a", 3)) {
        if(i >= argc - 1) {
          fprintf(stderr,"Argument error\n");
          usage();
          return 100;
        }
        i++;
        addflag = 1;
        ifirst = i;
        ilast = argc;
        i = argc;
      }
      else if(!strncmp(argv[i], "-s", 2) ||
              !strncmp(argv[i], "--s", 3)) {
        if(i >= argc - 1) {
          fprintf(stderr,"Argument error\n");
          usage();
          return 100;
        }
        i++;
        subflag = 1;
        ifirst = i;
        ilast = argc;
        i = argc;
      }
      else if(!strncmp(argv[i], "-c", 2) ||
              !strncmp(argv[i], "--c", 3)) {
        if(i >= argc - 1) {
          fprintf(stderr,"Argument error\n");
          usage();
          return 100;
        }
        i++;
        strcpy(cfgname, argv[i]);
      }
      else {
        fprintf(stderr,"Argument error\n");
        usage();
        return 100;
      }
    }
  }

  if(p = get_cfg_name(cfgname)) {
    strcpy(fullcfgname, p);
    free(p);
  }
  else {
    fprintf(stderr, "I cannot find %s\n", cfgname);
    return (100);
  }

/* add map file names to updmap.cfg */

  if(addflag) {
    for(i=ifirst; i < ilast; i++) {
      addok = 1;
      strcpy(addmap, argv[i]);
      p = get_map_name(addmap);
      if(p) {
        free(p);
        updmapcfg = fopen(fullcfgname, "ab+");
        if(!updmapcfg) {
          fprintf(stderr, "I cannot open configuration file\n");
          return 100;
        }
        while(fgets(buffer, LGBUF - 4, updmapcfg)) {
          p = buffer;
          while((*p == ' ') || (*p == '\t')) p++;
          if((*p == '%') || (*p == '#') || (*p == '\n'))
            continue;
          if(!strnicmp(p, "Map", 3))
            p += 3;
          else if(!strnicmp(p, "MixedMap", 8))
            p += 8;
          else
            continue;
          while((*p == ' ') || (*p == '\t')) p++;
          sscanf(p, "%s", mapname);
          if(!stricmp(mapname, addmap)) {
            addok = 0;
            if(Verbose)
              fprintf(stdout, "%s exists in the configuration file\n", addmap);
            break;
          }
        }
        if(addok) {
          strcpy(buffer, "Map ");
          strcat(buffer, addmap);
          strcat(buffer, "\n");
          fputs(buffer, updmapcfg);
          if(Verbose)
            fprintf(stdout, "I added %s to the configuration file\n", addmap);
        }
        fclose(updmapcfg);
      }
      else if(Verbose) {
        fprintf(stdout, "I don't add %s to the configuration file, ", addmap);
        fprintf(stdout, "because I cannot find it.\n");
      }
    }
  }


/* remove map file names from updmap.cfg */

  if(subflag) {
    _tmpd = getenv("TEMP");
    if(!_tmpd) _tmpd = getenv("TMP");
    if(!_tmpd) _tmpd = getenv("TMPDIR");
    if(_tmpd)
      _tmpd = xstrdup(_tmpd);
    else
      _tmpd = xstrdup((char *) __TMPDIR);
    for(p = _tmpd; *p; p++) {
      if(IS_KANJI(p)) {
        p++;
        continue;
      }
      if(*p == '\\') *p='/';
    }
    i = (int)strlen(_tmpd);
    if(_tmpd[i-1] == '/')
      _tmpd[i-1] = '\0';
    strcpy(tmp, _tmpd);
    free(_tmpd);
    strcat(tmp, TEMPLATE);
    _tmpd = mktemp(tmp);

    for(i=ifirst; i < ilast; i++) {
      subtracted = 0;
      strcpy(submap, argv[i]);
      tmpupdmapcfg = fopen(_tmpd, "wb");
      if (!tmpupdmapcfg) {
        fprintf(stdout,
               "I cannot open temporary file. I neglect the option --sub\n");
        goto Next;
      }
      updmapcfg = fopen(fullcfgname, "r");
      if(!updmapcfg) {
        fprintf(stdout, "I cannot open configuration file\n");
        fprintf(stdout, "I neglect the option --sub\n");
        if (tmpupdmapcfg) {
          fclose(tmpupdmapcfg);
          remove(_tmpd);
        }
        goto Next;
      }
      while(fgets(buffer, LGBUF - 4, updmapcfg)) {
        p = buffer;
        while((*p == ' ') || (*p == '\t')) p++;
        if((*p == '%') || (*p == '#') || (*p == '\n')) {
          fputs(buffer, tmpupdmapcfg);
          continue;
        }
        if(!strnicmp(p, "Map", 3))
          p += 3;
        else if(!strnicmp(p, "MixedMap", 8))
          p += 8;
        else {
          fputs(buffer, tmpupdmapcfg);
          continue;
        }
        while((*p == ' ') || (*p == '\t')) p++;
        sscanf(p, "%s", mapname);
        if(!stricmp(mapname, submap)) {
          subtracted = 1;
          continue;
        } else {
          fputs(buffer, tmpupdmapcfg);
          continue;
        }
      }
      fclose(updmapcfg);
      fclose(tmpupdmapcfg);
      CopyFile(_tmpd, fullcfgname, FALSE);
      remove(_tmpd);

      if(Verbose) {
        if(subtracted) {
          fprintf(stdout, "I erased the entry %s from configuration file\n", submap);
        } else if (submap[0]) {
          fprintf(stdout, "The entry for %s is not found in configuration file.\n", submap);
        }
      }
    }
  }


 Next:
  if(mkmap == 0) {
    if(Verbose)
      fprintf(stdout, "Map files are not updated because of the option --nomkmap\n");
    return 0;
  }

  updmapcfg = fopen(fullcfgname, "r");
  if(!updmapcfg) {
    fprintf(stderr, "I cannot open the configuration file\n");
    return 100;
  }

/* find $TEXMFVAR/fonts/map
   or $TEXMFMAIN/fonts/map */

  p = kpse_var_value("TEXMFVAR");
  if(!p)
    p = kpse_var_value("TEXMFMAIN");
  if(!p) {
    fprintf(stderr, "I cannot find TEXMFMAIN.\n");
    return 100;
  }

  strcpy (texmfmainfontsmap, p);
  free(p);

  for(p = texmfmainfontsmap; *p; p++) {
    if(IS_KANJI(p)) {
      p++;
      continue;
    }
    if(*p == '\\') *p = '/';
  }

  i = (int)strlen(texmfmainfontsmap);
  while(texmfmainfontsmap[i-1] == '/')
    i--;
  texmfmainfontsmap[i] = '\0';

  if(!is_dir(texmfmainfontsmap)) {
    if(make_dir_p(texmfmainfontsmap)) {
      fprintf(stderr, "Failed to access %s.\n", texmfmainfontsmap);
      return (100);
    }
  }

  strcat (texmfmainfontsmap, "/fonts");
  if (!is_dir(texmfmainfontsmap)) {
    if(make_dir(texmfmainfontsmap)) {
      return 100;
    }
  }
  strcat (texmfmainfontsmap, "/map");
  if (!is_dir(texmfmainfontsmap)) {
    if(make_dir(texmfmainfontsmap)) {
      return 100;
    }
  }

/* dvips */
  strcpy(mainmapname, texmfmainfontsmap);
  strcat(mainmapname, "/dvips");
  if (!is_dir(mainmapname)) {
    if(make_dir(mainmapname)) {
      return 100;
    }
  }
  strcat(mainmapname, "/updmap");
  if (!is_dir(mainmapname)) {
    if(make_dir(mainmapname)) {
      return 100;
    }
  }
  strcat(mainmapname, "/");
  strcat(mainmapname, PSFONTS);
  psfonts = fopen(mainmapname, "wb");
  if(psfonts) flag |= 1;

/* pdftex */
  strcpy(mainmapname, texmfmainfontsmap);
  strcat(mainmapname, "/pdftex");
  if (!is_dir(mainmapname)) {
    if(make_dir(mainmapname)) {
      return 100;
    }
  }
  strcat(mainmapname, "/updmap");
  if (!is_dir(mainmapname)) {
    if(make_dir(mainmapname)) {
      return 100;
    }
  }
  strcat(mainmapname, "/");
  strcat(mainmapname, PDFTEX);
  pdftex = fopen(mainmapname, "wb");
  if(pdftex) flag |= 2;

/* dvipdfmx */
  strcpy(mainmapname, texmfmainfontsmap);
  strcat(mainmapname, "/dvipdfmx");
  if (!is_dir(mainmapname)) {
    if(make_dir(mainmapname)) {
      return 100;
    }
  }
  strcat(mainmapname, "/updmap");
  if (!is_dir(mainmapname)) {
    if(make_dir(mainmapname)) {
      return 100;
    }
  }
  strcat(mainmapname, "/");
  strcat(mainmapname, DVIPDFM);
  dvipdfm = fopen(mainmapname, "wb");
  if(dvipdfm) flag |= 4;

/* ps2pk */
  strcpy(mainmapname, texmfmainfontsmap);
  strcat(mainmapname, "/ps2pk");
  if (!is_dir(mainmapname)) {
    if(make_dir(mainmapname)) {
      return 100;
    }
  }
  strcat(mainmapname, "/updmap");
  if (!is_dir(mainmapname)) {
    if(make_dir(mainmapname)) {
      return 100;
    }
  }
  strcat(mainmapname, "/");
  strcat(mainmapname, PSPKSUPP);
  pspksupp = fopen(mainmapname, "wb");
  if(pspksupp) flag |= 8;

/*
 First, I copy system map files foo-base.map to foo.map, if exists.
*/
/* dvips */
  if(flag & 1) {
    if(p = get_map_name(KANJI_BASE)) {
      psfonts_base = fopen(p, "rb");
      free(p);
      if(psfonts_base) {
        copy_base(psfonts, psfonts_base);
        fclose(psfonts_base);
      }
    }
    if(p = get_map_name(PSFONTS_BASE)) {
      psfonts_base = fopen(p, "rb");
      free(p);
      if(psfonts_base) {
        copy_base(psfonts, psfonts_base);
        fclose(psfonts_base);
      }
    }
  }

/* pdftex */
  if(flag & 2) {
    if(Ndl14 == 0)
      p = get_map_name(PDFTEX_BASE);
    else
      p = get_map_name(PDFTEX_BASEND14);
    if(p) {
      pdftex_base = fopen(p, "rb");
      free(p);
      if(pdftex_base) {
        copy_base(pdftex, pdftex_base);
        fclose(pdftex_base);
      }
    }
  }

/* dvipdfmx */
  if(flag & 4) {
     kpse_reset_program_name("dvipdfmx");
     if(p = get_map_name(DVIPDFM_BASE)) {
        dvipdfm_base = fopen(p, "rb");
        free(p);
        if(dvipdfm_base) {
           copy_base(dvipdfm, dvipdfm_base);
           fclose(dvipdfm_base);
        }
     }
     kpse_reset_program_name(Programname);
  }

/* ps2pk */
  if(flag & 8) {
    if(p = get_map_name(PSPKSUPP_BASE)) {
      pspksupp_base = fopen(p, "rb");
      free(p);
      if(pspksupp_base) {
        copy_base(pspksupp, pspksupp_base);
        fclose(pspksupp_base);
      }
    }
  }

/*
 Work with texmf/web2c/updmap.cfg
*/
  while(fgets(buffer, LGBUF - 4, updmapcfg)) {
    p = buffer;
    while((*p == ' ') || (*p == '\t')) p++;
    if((*p == '%') || (*p == '#') || (*p == '\n'))
      continue;
    if(!strnicmp(p, "Map", 3))
      p += 3;
    else if(!strnicmp(p, "MixedMap", 8))
      p += 8;
    else
      continue;
    while((*p == ' ') || (*p == '\t')) p++;
    sscanf(p, "%s", mapname);
    if(strnicmp(mapname, "bluesky-dl", 10) == 0)
      Bluesky = 1;
    else
      Bluesky = 0;
    p = get_map_name(mapname);
    if(p) {
      mapfile = fopen(p, "rb");
      free(p);
      if(mapfile) {
        if(Verbose)
          fprintf(stdout, "Working with %s ...\n", mapname);
        do_update(flag, mapname, mapfile, psfonts, pdftex, dvipdfm, pspksupp);
        fclose(mapfile);
      }
      else
        fprintf(stdout, "I cannot open %s. I ignore this map file.\n", mapname);
    }
    else
      fprintf(stdout, "I cannot find %s. I ignore this map file.\n", mapname);
  }

  fclose(updmapcfg);

  if(flag & 1) {
    fclose(psfonts);
    if(Verbose)
      fprintf(stdout, "I updated psfonts.map.\n");
  }
  if(flag & 2) {
    fclose(pdftex);
    if(Verbose)
      fprintf(stdout, "I updated pdftex.map.\n");
  }
  if(flag & 4) {
    fclose(dvipdfm);
    if(Verbose)
      fprintf(stdout, "I updated dvipdfm.map.\n");
  }
  if(flag & 8) {
    fclose(pspksupp);
    if(Verbose)
      fprintf(stdout, "I updated pspksupp.map.\n");
  }

  return (0);
}
