/* -*-C++-*-
 * ###################################################################
 *	Cpptcl - Integrating C++ with Tcl
 * 
 *	FILE: "tcl_base.cc"
 *					created: 16/10/97 {2:08:39 pm}	
 *				   last update: 09/05/98 {23:36:27 PM}	
 *	Author:	Vince Darley
 *	E-mail:	<mailto:darley@fas.harvard.edu>
 *	  mail:	Division of	Applied	Sciences, Harvard University
 *			Oxford Street, Cambridge MA	02138, USA
 *	   www: <http://www.fas.harvard.edu/~darley/>
 *	
 *	See	header file	for	further	information
 * ###################################################################
 */

#include "tcl_base.icc"
#include "cpptcl_init.h"
#include "cpptcl_metaobject.h"
#include "meta_object.h"
#include "cpptcl_members.h"
#include "cpptcl_config_mem.h"

DLL_EXPORT
cpptcl_metaobject* tcl_interaction::metaobject = 0;

#ifdef CPPTCL_USE_SCOPED_OBJECTS
extern "C" {
	char* Cpptcl_ObjectScopeVarDeleted(ClientData clientData,
		Tcl_Interp *interp, CONST char *name1, CONST char *name2, int flags);
}
#endif

DLL_EXPORT
const meta_object tcl_interaction::_type("type-hierarchy");

//virtual
int tcl_interaction::parse_meta_commands(tcl_obj& tcl_, tcl_args& arg){
	if(arg("","gets class of the object")=="getClass") {
		arg >> done;
		NO_EXCEPTIONS(arg);
		tcl_ << type() << result;
		return tcl_;
	} else {
		return arg.no_match_yet();
	}
}

Cpptcl_IBaseClass(tcl_base,"cpp-object",tcl_interaction);

//extern "C"
int Cpptcl_ParseObjectCommand(ClientData clientData, Tcl_Interp* interp,
			      int objc, struct Tcl_Obj * const objv[]){
    // verify that this command was called on the same interpreter clientData
    // was created on

    tcl_base* o = (tcl_base*)clientData;
    if (interp != o->get_interp()){
	tcl_obj t (interp);
	t << objv[0] << ": Object called from wrong interpreter!!" << tcl_error;
	return TCL_ERROR;
    } else {
	tcl_obj& tcl_ = o->get_tcl_obj();
	// synchronise with the result (aids backwards compatibility)
	tcl_ << get_result;// << discard ??
	tcl_.reset();
#ifndef NO_EXCEPTION_HANDLING      	
	try {
#endif
	    tcl_args arg(tcl_,objc -1, (Tcl_Obj**)(objv +1),o);
	    return o->tcl_command_entry_point(arg);
#ifndef NO_EXCEPTION_HANDLING      	
	} catch (int err) {
	    if(err != TCL_ERROR) {
		// My code only uses one exception type so far
		o->get_tcl_obj() << o << " : Unknown exception thrown" << result;
	    }
	    // the error message is stored in the tcl interpreter
	    return TCL_ERROR;
	}
#endif
    }
}

/* 
 * -------------------------------------------------------------------------
 * 
 * "tcl_command_entry_point" --
 * 
 *  Currently can only expand abbreviations of toplevel arguments.
 * -------------------------------------------------------------------------
 */
int tcl_interaction::tcl_command_entry_point(tcl_args& arg) {
	int res;
	int count = arg.args_left();
	while (1) {
		if((res = parse_meta_commands(get_tcl_obj(),arg)) != CPPTCL_NOT_HANDLED) {
			return res;
		}
		arg.haveErr = 0;
		// if we failed at the first arg, try this:
		if (count != arg.args_left()) break;
		if((res = parse_tcl_command(arg)) != CPPTCL_NOT_HANDLED) {
			return res;
		}
		arg.haveErr = 0;
		// else try and find an abbreviation for what we got.
		// if we can't return
		if(!arg.try_abbreviation())
			return arg.no_match();
	}
	// else no match
	return arg.no_match();
}

//extern "C"
void Cpptcl_DeleteObject(ClientData clientData){
	// this may be called either from tcl or from ~tcl_base indirectly
  	delete (tcl_base*) clientData;
}

//virtual
tcl_base::~tcl_base(void){
    if(!embedded()) {
#ifdef CPPTCL_USE_SCOPED_OBJECTS
	char * command = char_tcl_command();
	if(tracing_for_scope) {
	    tracing_for_scope = false;
	    // remove the trace
	    remove_a_trace(get_interp(),command);
	}	
#endif
	if(cmd_info_) {
	    Cpptcl_DeletedObject(this);
	    Tcl_DeleteCommandFromToken(get_interp(), cmd_info_);
	}
	cmd_info_ = 0;		
    } else {
	if(_name) {
	    Tcl_DecrRefCount(_name);
	    _name = 0;
	}
    }
}

#ifdef USE_TCL_STUBS
#include <tclInt.h>
#else
extern "C" int TclRenameCommand(Tcl_Interp* , char* , char*);
#endif

bool tcl_base::change_name_to(Tcl_Obj* newname) {
    if(!embedded()) {
#ifdef CPPTCL_USE_SCOPED_OBJECTS
	if(!could_make_a_trace(tcl_,newname)) {
	    return false;
	}
	// remove old trace
	char * command = char_tcl_command();
	if(tracing_for_scope) {
	    // remove the trace
	    remove_a_trace(get_interp(),command);
	}	
#endif
	int length;
	Cpptcl_DeletedObject(this);
	TclRenameCommand(tcl_, command, Tcl_GetStringFromObj(newname, &length));
	Cpptcl_CreatedNewObject(tcl_,this);
#ifdef CPPTCL_USE_SCOPED_OBJECTS
	// create scoped variable
	if(tracing_for_scope) {
	    make_a_trace();
	} 
#endif
    }
    return true;
}

void tcl_base::remove_a_trace(Tcl_Interp* interp, char* command) {
    if(command) {
	if(Tcl_VarTraceInfo(interp, command, TCL_TRACE_UNSETS,
			    (Tcl_VarTraceProc*) Cpptcl_ObjectScopeVarDeleted, NULL) != NULL) {
	    Tcl_UntraceVar(interp, command, TCL_TRACE_UNSETS,
			   (Tcl_VarTraceProc*) Cpptcl_ObjectScopeVarDeleted, (ClientData) this);
	    Tcl_UnsetVar(interp, command, 0);
	}
    }
}

bool tcl_base::could_make_a_trace(tcl_obj& tcl_, Tcl_Obj* name) {
    if (Tcl_ObjGetVar2(tcl_, name, NULL, 0) != NULL) {
	tcl_ << "Illegal name for object command \"" << name 
	     << "\": local variable of same name is active" << tcl_error;
	return false;
    }
    return true;
}

bool tcl_base::make_a_trace(void) {
    tracing_for_scope = true;
    static Tcl_Obj *elt = 0;
    if(elt == NULL) {
   	// create the object, and increment the ref count so it doesn't later vanish on us.
	elt = Tcl_NewStringObj("cpptcl-trace",12);
	Tcl_IncrRefCount(elt);
	// if we were ever able to 'unload' the Cpptcl library, we'd want to decrement
	// this ref count then.
    }
    Tcl_Obj *tmp = Tcl_NewStringObj(char_tcl_command(),-1);
    Tcl_IncrRefCount(tmp);
    if (Tcl_ObjSetVar2(tcl_, tmp, NULL, elt, 0) == NULL) {
	tcl_ << "Weird, I thought I could set a new trace-variable \""
	     << name() << "\", but then I couldn't." << tcl_error;
	Tcl_DecrRefCount(tmp);
	return false;
    }
    Tcl_DecrRefCount(tmp);
    Tcl_TraceVar(tcl_, char_tcl_command(), TCL_TRACE_UNSETS, 
		 (Tcl_VarTraceProc*) Cpptcl_ObjectScopeVarDeleted, (ClientData) this);
    return true;
}

//virtual
int tcl_base::parse_meta_commands(tcl_obj&, tcl_args& arg){
	if (arg("newName","changes the Tcl command name of the object")=="rename") {
  		Tcl_Obj* newName;
  		arg >> newName >> done;
  		NO_EXCEPTIONS(arg);
  		int length = 0;
  		Tcl_GetStringFromObj(newName,&length);
  		if(length == 0) {
  		    tcl_ << "can't delete or set name to empty string" << tcl_error;
  		    //delete this;
  		} else {
		    if(!change_name_to(newName)) {
			tcl_ << "rename failed!" << tcl_error;		    	
		    }
		}
		return tcl_;
  	} else if(has_members() && arg("?type?","lists registered object members")=="listMembers") {
		ometa<cpp_mem> m;
		arg >> optional >> m >> done;
  		NO_EXCEPTIONS(arg);
		metaobject->list_members(&meta_info(),tcl_,m.type());
		tcl_ << result;
		return tcl_;
  	} else if(has_members() && arg("?member-name?",
		"gets type of the object or one of its members")=="getType") {
		if(arg.empty()) {
			arg >> done;
			NO_EXCEPTIONS(arg);
			tcl_ << type() << result;
		} else {
			// member
			member<cpp_config_mem> i(this);
			arg >> i >> done;
			NO_EXCEPTIONS(arg);
			cpp_config_mem* mm = i;
			tcl_ << mm->data_type() << result;
		}
		return tcl_;
  	} else if(has_members() && (arg("member ?args?","query a member")=="memberconfigure" 
		|| arg("member ?args?","query a member")=="mconfig")) {
		member<cpp_mem> i(this);
		arg >> i;
		NO_EXCEPTIONS(arg);
		return i.obj->parse_meta_commands(tcl_,arg);
  	} else {
		return tcl_interaction::parse_meta_commands(tcl_,arg);
  	}
}

///
tcl_obj& tcl_base::tcl_command(tcl_obj& o) const {
	if(embedded()) {
		tcl_obj t;
		t << list_mode_on << objcontainer() << name() << list_mode_off;
		return o << t;
	} else {
		return o << Tcl_GetCommandName(tcl_,cmd_info_);
	}
	
}

tcl_base* Cpptcl_getObjectByName(Tcl_Interp* interp, const char* name){
  	Tcl_CmdInfo info;
  	if (!Tcl_GetCommandInfo(interp, (char*) name, &info))
    	return (tcl_base*) NULL;
  	else
		return (tcl_base*) (info.objClientData);
}

tcl_base* Cpptcl_getObjectByName(Tcl_Interp* interp, Tcl_Obj* name){
	int len;
  	return Cpptcl_getObjectByName(interp,Tcl_GetStringFromObj(name,&len));
}

void tcl_base::update_metaobject(tcl_obj& i, cpptcl_metaobject_fn f) {
	static char cppMeta[] = "cppmeta";
	Tcl_Obj* cppM = Tcl_NewStringObj(cppMeta,7);
	tcl_args t(i);
	t.setName(cppM);
	if(metaobject ==0) {
		if(f) {
			metaobject = (*f)(t,0);
		} else {
			metaobject = new cpptcl_metaobject(t);
		}
	} else {
		/* This does all that's needed */
		metaobject = (*f)(t,metaobject);
	}
	
}
tcl_args& operator>> (tcl_args& arg, tcl_base*& into){
	arg.set_conversion_type("tcl_base");
	const char* t;
	arg.const_string_read(t);
	if(!t){
		into=0;
	} else {
		into = Cpptcl_getObjectByName(arg,t);
		if(into) {
			arg.parsed_so_far << t;
		} else {
			arg.signal_error(tcl_args::Conversion);
		}
	}
	arg.read_done();
	return arg;
}

