
#ifndef lint
static char    *copyright = "Copyright (C) 1994, Steve Cumming";
#endif

/*
 * Copyright (C) 1994 by Steve Cumming (stevec@geog.ubc.ca)
 *
 * Permission to use, copy, modify, and distribute this software 
 * for any purpose and without fee is hereby granted, provided
 * that the above copyright notices appear in all copies and that both the
 * copyright notice and this permission notice appear in supporting
 * documentation.  This software is provided "as is" without express or
 * implied warranty.
 */



#include <sys/param.h>
#include <assert.h>
#include <string.h>
#include <math.h>
#include "lib.h"
#include "gp.h"
#include "search.h"
#include "array.h"
#include "mlib.h"
#ifdef MAPS
#include "nr.h"
#include "mapgen.h"
#include "map.h"
#include "zone.h"
#endif
#include "histgen.h"


/*
  Count an array into n-bins. 
*/

struct array * 
    binify(struct array * a, int n, double mn, double mx){
	int i;
	int bin;
	double x,y;
	double step = n / (mx - mn);
	struct array * res = new_array(FLOAT,n);
	if (a) {
	    for (i = 0; i < a->n; i++){
		x = get_double_array(a,i);
		bin =  (int) ((x - mn) * step);
		if (bin < 0)
		    bin = 0;
		if (bin >= n)
		    bin = n - 1;
		((float *)res->a)[bin]++;
	    }
	} else   /* make an axis */{
	    y = mn;
	    step = (mx - mn) / (double )n;
	    for (i = 0; i < n; i++){
		((float *)res->a)[i] = (float) y;
		y += step;
	    }
	}


	return res;
    }


struct array * 
    log_binify(struct array * a, double mn, double mx){
	int i,n;
	int bin, offset;
	double  x,y;

	struct array * res;
	int  lmn = log_b(mn,2.0);
	int  lmx = log_b(mx,2.0);
	
	if (mn >= mx){
	    error("log_binify(): Bad Range\n");
	    return NULL;
	}

	if (lmn < 0)
	    lmn--;
	lmx++;
	n = 10 * (lmx - lmn);
	res = new_array(FLOAT,n);

	if (a) {
	    for (i = 0; i < a->n; i++){
		x = get_double_array(a,i);
		y = log_b(x,2.0);
/*
		if (y < 0) y-= 1.0;
*/
		bin = 10 * (int)(y - lmn);
		if (y < 0){
		    y = -y;
		    while (y >= 1.0)y--;
		    y = 1.0 - y;
		}
		else while (y >= 1.0)y--;
		offset = (y * 10.0);
		((float *)res->a)[bin+offset]++;

	    }
	} else   /* make an axis */{
	    
	    y = lmn;
	    for (i = 0; i < n; i++){
		((float *)res->a)[i] = pow(2.0,y);
		y += 0.1;
	    }
	}


	return res;
    }
		

/* 
  These are the same except that instead of a count,
  we sum the values in the parellel array b 
*/

struct array * 
    binify_sum(struct array * a, struct array * b, int n, double mn, double mx){
	int i;
	int bin;
	double x,y;
	double step = n / (mx - mn);
	struct array * res = new_array(FLOAT,n);
	if (a) {
	    for (i = 0; i < a->n; i++){
		x = get_double_array(a,i);
		bin =  (int) ((x - mn) * step);
		if (bin < 0)
		    bin = 0;
		if (bin >= n)
		    bin = n - 1;
		((float *)res->a)[bin] += (float)get_double_array(b,i);
	    }
	} else   /* make an axis */{
	    y = mn;
	    step = (mx - mn) / (double )n;
	    for (i = 0; i < n; i++){
		((float *)res->a)[i] = (float) y;
		y += step;
	    }
	}


	return res;
    }


struct array * 
    log_binify_sum(struct array * a, struct array * b, double mn, double mx){
	int i,n;
	int bin, offset;
	double  x,y;

	struct array * res;
	int  lmn = log_b(mn,2.0);
	int  lmx = log_b(mx,2.0);
	
	if (mn >= mx){
	    error("log_binify(): Bad Range\n");
	    return NULL;
	}

	if (lmn > 0)
	    lmn++;
	else 	
	    lmn--;   
	if (lmx > 0)
	    lmx++;
	else 	
	    lmx--;

	n = 10 * (lmx - lmn);
	res = new_array(FLOAT,n);

	if (a) {
	    for (i = 0; i < a->n; i++){
		x = get_double_array(a,i);
		y = log_b(x,2.0);
		if (y < 0) y-= 1.0;
		bin = 10 * (int)(y - lmn);
		if (y < 0){
		    y = -y;
		    while (y >= 1.0)y--;
		    y = 1.0 - y;
		}
		else while (y >= 1.0)y--;
		offset = (y * 10.0);
		((float *)res->a)[bin+offset] += (float)get_double_array(b,i);
	    }
	} else   /* make an axis */{
	    
	    y = lmn;
	    for (i = 0; i < n; i++){
		((float *)res->a)[i] = pow(2.0,y);
		y += 0.1;
	    }
	}


	return res;
    }
		
	

struct array * csum_array_in_place(struct array * a){

    int i;
    double x;
    double sum = 0.0;

    switch (a->type) {
    case DOUBLE:
	for (i = 0; i < a->n; i++){
	    x = ((double *)a->a)[i];
	    sum += x;
	    ((double *)a->a)[i] = sum;
	}
	break;
    case FLOAT:
	for (i = 0; i < a->n; i++){
	    x = (double)((float *)a->a)[i];
	    sum += x;
	    ((float *)a->a)[i] = (float)sum;
	}
	break;
    default:
	fatal("csum_array_in_place(): Illegal array type\n");
	break;
    }
    return a;
}


struct array * cdf_array_in_place(struct array * a, int n){

    int i;
    double x,xn;
    double sum = 0.0;
    xn = (double) n;

    switch (a->type) {
    case DOUBLE:
	for (i = 0; i < a->n; i++){
	    x = ((double *)a->a)[i];
	    sum += x;
	    ((double *)a->a)[i] = sum / xn;
	}
	break;
    case FLOAT:
	for (i = 0; i < a->n; i++){
	    x = (double)((float *)a->a)[i];
	    sum += x;
	    ((float *)a->a)[i] = (float)(sum / xn);
	}
	break;
    default:
	fatal("cdf_array_in_place(): Illegal array type\n");
	break;
    }
    return a;
}


struct array * pdf_array_in_place(struct array * a, int n){

    int i;
    double x,xn;
    xn = (double) n;
    switch (a->type){
    case DOUBLE:
	for (i = 0; i < a->n; i++){
	    x = ((double *)a->a)[i];
	    ((double *)a->a)[i]  = x / xn;
	}
	break;
    case FLOAT:
	for (i = 0; i < a->n; i++){
	    x = (double)((float *)a->a)[i];
	    ((float *)a->a)[i] = (float)(x / xn);
	}
	break;
    default:
	fatal("cdf_array_in_place(): Illegal array type\n");
	break;
    }
    return a;
}
    


#ifdef MAPS

#define LOG_TICS 2

static char mgbuf[256];

/* This is support for 2-dimensional distributions
   which requires a whole bunch of other stuff that I am
   not able to make availebl at this time

/*
  Builds a 2-ry distribution into m1 and m2
  using stand to select stands and pi1 and pi2 to
  deliver the two attributes.

  If used for 2d KS tests on large input data sets,
  p should be small, or you'll  wait a long time.

  OW, if you just want to make histograms, or zonify and
  do 1d KS tests, it doen's matter much.

*/

void
    make_arrays(struct map * files,
	       int (*filter)(struct stand *),
	       struct value * (*pi1)(struct stand *),
	       struct value * (*pi2)(struct stand *),
	       struct array ** x1,
	       struct array ** x2,
	       double p,
	       char * dir)
{

    int n,m,y,x;
    char * str;
    FILE *f;
    struct array * a1, * a2, *tmpa;
    char * savedir;
    
    if (*x1){
	free_array(*x1);
	*x1 = 0x0;
    }
    if (*x2){
	free_array(*x2);
	*x2 = 0x0;
    }

    savedir = pushdir(dir);

    get_map_size(files,&n,&m);
	
    if (files->type != STR){
	error("dist_map2(): I need a map of file names!");
	return;

    }

    for (y = 0; y < n; y++){
	for (x = 0; x < m; x++){

	    if  (is_null_point(files,y,x))
		continue;

	    str = get_map_s(files,y,x);
	    if ((f = fopen(str,"r")) == NULL){
		fprintf(stderr,"Could not open %s\n",str);
		continue;
	    }
		
	    get_dist2(f, filter, pi1, pi2, &a1, &a2,p);
	    if (Debug)
		fprintf(stderr,"dist_map2: %s [%d]\n",str,a1->n);
	    fclose(f);
	
/*
  need an append_array (in place);
*/
	    tmpa = cat_array(*x1,a1);
	    free_array(*x1);
	    free_array(a1);
	    *x1 = tmpa;
	
	    tmpa = cat_array(*x2,a2);
	    free_array(*x2);
	    free_array(a2);
	    *x2 = tmpa;
	}
    }

    popdir(savedir);
    return;
}


void
    make_array(struct map * files,
	       int (*filter)(struct stand *),
	       struct value * (*pi1)(struct stand *),
	       struct array ** x1,
	       char * dir)
{

    int n,m,y,x;
    char * str;
    FILE *f;
    struct array * a1, * a2, *tmpa;
    char * savedir;
    
    if (*x1){
	free_array(*x1);
	*x1 = 0x0;
    }

    savedir = pushdir(dir);

    get_map_size(files,&n,&m);
	
    if (files->type != STR){
	error("make_array(): I need a map of file names!");
	return;

    }

    for (y = 0; y < n; y++){
	for (x = 0; x < m; x++){

	    if  (is_null_point(files,y,x))
		continue;

	    str = get_map_s(files,y,x);
	    if ((f = fopen(str,"r")) == NULL){
		fprintf(stderr,"Could not open %s\n",str);
		continue;
	    }
		
	    get_dist1(f, filter, pi1, &a1);
	    if (Debug)
		fprintf(stderr,"make_array: %s [%d]\n",str,a1->n);
	    fclose(f);
	
/*
  need an append_array (in place);
*/
	    tmpa = cat_array(*x1,a1);
	    free_array(*x1);
	    free_array(a1);
	    *x1 = tmpa;
	
	}
    }

    popdir(savedir);
    return;
}


void
    dist_map2(struct map * files,
	      struct map ** m1, struct map ** m2,
	      int (*filter)(struct stand *),
	      struct value * (*pi1)(struct stand *),
	      struct value * (*pi2)(struct stand *),
	      double p, char * dir)
{

    int n,m,y,x;
    struct map * n1;	
    struct map * n2;
    char * str;
    FILE *f;
    struct array * a1, * a2;
    char * savedir;

    get_map_size(files,&n,&m);
    n1 = new_map(n,m, "Distribution (x)", POINTER);
    n2 = new_map(n,m, "Distribution (y)", POINTER);
    set_map_free(n1, (void (*)(void *))free_array);
    set_map_free(n2, (void (*)(void *))free_array);
    
    savedir = pushdir(dir);
    
    if (files->type != STR){
	error("dist_map2(): I need a map of file names!");
	return;

    }

    for (y = 0; y < n; y++){
	for (x = 0; x < m; x++){

	    if  (is_null_point(files,y,x)){
		put_map_null(n1,y,x);
		put_map_null(n2,y,x);
		continue;
	    }

	    str = get_map_s(files,y,x);
	    if ((f = fopen(str,"r")) == NULL){
		fprintf(stderr,"Could not open %s\n",str);
		put_map_null(n1,y,x);
		put_map_null(n2,y,x);
		continue;
	    }
		
	    get_dist2(f, filter, pi1, pi2, &a1, &a2,p);
	    if (Debug)
		fprintf(stderr,"dist_map2: %s [%d]\n",str,a1->n);
	    fclose(f);
	    put_map_ptr(n1,y,x,(void *)a1);
	    put_map_ptr(n2,y,x,(void *)a2);
	}
    }

    popdir(savedir);

    *m1 = n1;
    *m2 = n2;
    return;
}


/*
  This is designed for 2d- KS test
*/

void get_dist2(FILE * f, 
	       int (*filter)(struct stand *),
	       struct value * (*pi1)(struct stand *),
	       struct value * (*pi2)(struct stand *),	
	       struct array ** x, struct array ** y,
	       double p)
{
    static int iseed = -1;
    int n;
    struct stand stand;
    struct array * ax, *ay;
    int sz = 100;

    ax = new_array((*pi1)(NULL)->type,sz);
    ay = new_array((*pi2)(NULL)->type,sz);
    
    n = 0;
    while (!get_nxt_line(f,mgbuf)){
	parse_stand_record(&stand,mgbuf);
	if (filter(&stand) && (ran2(&iseed) < p)) {
	    put_array_value(ax,n,(*pi1)(&stand));
	    put_array_value(ay,n,(*pi2)(&stand));
	    n++;
	    if (n == sz){
		sz <<= 2;
		ax = resize_array(ax,sz);		
		ay = resize_array(ay,sz);
	    }
	}
    }

    /* resize the arrays */

    ax = resize_array(ax,n);
    ay = resize_array(ay,n);
    *x = ax;
    *y = ay;
    
    return;
}



void
    dist_map1(struct map * files,
	      struct map ** m1,
	      int (*filter)(struct stand *),
	      struct value * (*pi1)(struct stand *),
	      char * dir)
{

    int n,m,y,x;
    struct map * n1;	
    char * str;
    FILE *f;
    struct array * a1;
    char *pathname = xalloc(MAXPATHLEN,"mapgen");

    get_map_size(files,&n,&m);
    n1 = new_map(n,m, "Distribution (x)", POINTER);
    set_map_free(n1, (void (*)(void *))free_array);

    if (getwd(pathname) == 0)
	fatal(pathname);
    if (dir != NULL ? chdir(dir): 0)
	fatal("could not change directory");
	
    if (files->type != STR){
	error("dist_map1(): I need a map of file names!");
	return;

    }

    for (y = 0; y < n; y++){
	for (x = 0; x < m; x++){

	    if  (is_null_point(files,y,x)){
		put_map_null(n1,y,x);
		continue;
	    }

	    str = get_map_s(files,y,x);
	    if ((f = fopen(str,"r")) == NULL){
		fprintf(stderr,"Could not open %s\n",str);
		put_map_null(n1,y,x);
		continue;
	    }
		
	    get_dist1(f, filter, pi1, &a1);
	    if (Debug)
		fprintf(stderr,"dist_map2: %s [%d]\n",str,a1->n);
	    fclose(f);
	    put_map_ptr(n1,y,x,(void *)a1);
	}
    }

    if (dir != NULL ? chdir(pathname): 0)
	fatal("could not restore directory");
    free(pathname);

    *m1 = n1;
    return;
}

/*
  This is designed for 1d- KS test
*/


void get_dist1(FILE * f, 
	       int (*filter)(struct stand *),
	       struct value * (*pi1)(struct stand *),
	       struct array ** x)
{
    int n;
    struct stand stand;
    struct array * ax;
    int sz = 100;

    ax = new_array((*pi1)(NULL)->type,sz);
    n = 0;
    while (!get_nxt_line(f,mgbuf)){
	parse_stand_record(&stand,mgbuf);
	if (filter(&stand)){
	    put_array_value(ax,n,(*pi1)(&stand));
	    n++;
	    if (n == sz){
		sz <<= 2;
		ax = resize_array(ax,sz);		
	    }
	}
    }

    /* resize the arrays */

    ax = resize_array(ax,n);
    *x = ax;   
    return;
}
    

/*
  This code is used to make splots input data
  on a Regular grid.
*/



struct bin2d * new_bin2d(int n, int nx, int ny)
{
    int i;
    struct value v;
    struct bin2d * r = (struct bin2d *)xalloc(sizeof(struct bin2d),"new_bin2d()");
    r->n = n;
    r->xtics = new_array(DOUBLE,nx);
    r->xtics1 = new_array(DOUBLE,nx);
    r->ytics = new_array(DOUBLE,ny);
    r->ytics1 = new_array(DOUBLE,ny);
    r->s = (char **)xalloc(n * sizeof(char *),"new_bin2d()");
    r->m = (struct map **)xalloc(n * sizeof(struct map *),"new_bin2d()");
    for (i = 0; i < n; i++){
	r->m[i] = new_map(nx,ny,"Bin2d element",DOUBLE);
	set_map_null(r->m[i],Double(&v, -1.0));		/* all non neg counts */
	init_map(r->m[i], Double(&v,0.0));
    }
    return r;
}

void free_bin2d(struct bin2d * b){
    int i;
    free_array(b->xtics);
    free_array(b->xtics1);
    free_array(b->ytics);
    free_array(b->ytics1);
    for (i = 0; i < b->n; i++){
	free_map((b->m)[i]);
	free(b->s[i]);
    }
    free(b->s);
    free(b->title);
    free(b->m);
    free(b);
    return;
}

/*
  This is designed to binify pairs of (x,y) points
  for plotting by gnuplot's splot command.

  Input arrays can not be sorted.
*/


struct bin2d * make_bin2d(void * hx, void * hy, char * title, int xdim, int ydim, int pdf)
{
    
    double minx, maxx, miny, maxy;	/* absolute bounds */
    double mn, mx;
    struct array * vx, * vy;
    ENTRY *ex, *ey, nex, ney;
    struct array * tx, *ty;		/* totals          */
    struct array *ax, *ay;		/* scratch arrays  */
    int dimx, dimy;
    struct bin2d * res;
    double log_step;
    double stepx, stepy, z;
    int lminx, lminy, lmaxx, lmaxy;

    int ix,iy,offset;

    int i,j,k,n;

    vx = array_from_hash(hx);
    vy = array_from_hash(hy);
    
    /* cruise arrays, finding global mins and max's */

    tx = NULL;
    ty = NULL;

    assert(vx->n == vy->n);

    for (i = 0; i < vx->n; i++){
	ex = (ENTRY *) ((void **)vx->a)[i];
	ey = (ENTRY *) ((void **)vy->a)[i];
	ax = (struct array *)ex->data;
	ay = (struct array *)ey->data;
	assert(ax->n == ay->n);
	tx = cat_array(tx, ax);
	ty = cat_array(ty, ay);
    }

    array_bounds(tx,&minx,&maxx);
    array_bounds(ty,&miny,&maxy);

    /*   make room for the totals */

    if (vx->n > 1){		  /* redundant if only one set */
	nex.key = "Total";
	nex.data = (void *)tx;
	ney.key = "Total";
	ney.data = (void *)ty;

	vx = shift_array(vx,1);
	vy = shift_array(vy,1);    
	put_ptr_array(vx,0,&nex);
	put_ptr_array(vy,0,&ney);
    }


    log_step = 1.0/LOG_TICS;

    /* determine dimensions */

    if (ydim == LOG_BIN){
	lminy = log_b(miny,2.0);
	lmaxy = log_b(maxy,2.0);
	if (lminy > 0) lminy++;
	else lminy--;
	if (lmaxy > 0) lmaxy++;
	else lmaxy--;
	dimy = LOG_TICS * (lmaxy - lminy);
    }
    else {
	dimy = ydim;
	stepy = ydim  / (maxy - miny);
    }

    if (xdim == LOG_BIN){
	lminx = log_b(minx,2.0);
	lmaxx = log_b(maxx,2.0);
	if (lminx > 0) lminx++;
	else lminx--;
	if (lmaxx > 0) lmaxx++;
	else lmaxx--;
	dimx = LOG_TICS * (lmaxx - lminx);
    }
    else{
	dimx = xdim;
	stepx = xdim  / (maxx - minx);
    }

    res = new_bin2d(vx->n, dimx, dimy);		/* including the total */
    res->title = xstrdup(title,"Nah. Can't happen\n");

    /* build the tic arrays	*/
    
    if (xdim == LOG_BIN){
	z = lminx;
	for (i = 0; i < dimx; i++){
	    ((double *)(res->xtics->a))[i] = pow(2.0,z);
	    ((double *)(res->xtics1->a))[i] = z;
	    z += log_step;
	}
    } else {
	int q = (maxx - minx) / (double)xdim;
	z = minx;
	for (i = 0; i < dimx; i++) {
	    ((double *)(res->xtics->a))[i] = z;
      	    ((double *)(res->xtics1->a))[i] = i;
	    z += q;
	}
    }

    if (ydim == LOG_BIN){
	z = lminy;
	for (i = 0; i < dimy; i++){
	    ((double *)(res->ytics->a))[i] = pow(2.0,z);
	    ((double *)(res->ytics1->a))[i] = z;
	    z += log_step;
	}
    } else {
	int q = (maxy - miny) / (double)ydim;
	z = miny;
	for (i = 0; i < dimy; i++) {
	    ((double *)(res->ytics->a))[i] = z;
	    ((double *)(res->ytics1->a))[i] = i;
	    z += q;
	}
    }
    
    for (i = 0; i < vx->n; i++){
	static struct value v;
	struct map * m = res->m[i];
	double  N = 0.0;
	ex = (ENTRY *) ((void **)vx->a)[i];
	ey = (ENTRY *) ((void **)vy->a)[i];
	ax = (struct array *)ex->data;
	ay = (struct array *)ey->data;
	
	if (i == 0)
	    res->s[i] = xstrdup("Total","make_bins2()");
	else
	    res->s[i] = xstrdup(ex->key,"make_bins2()");	/* both have same key */

/*
	if (Debug)
	    fprintf(stderr,"Binning %d elements from %s\n",ax->n, ex->key);
*/

	for (j = 0; j < ax->n; j++){
	    static double x,y;
	    x = get_double_array(ax,j);
	    y = get_double_array(ay,j);

	    /* compute buckets         */

	    if (xdim == LOG_BIN){
		z = log_b(x,2.0);
		if (z < 0) z-= 1.0;
		ix = LOG_TICS * (int)(z - lminx);
		if (z < 0){
		    z = - z;
		    while (z >= 1.0) z--;
		    z = 1.0 - z;
		} else while (z >= 1.0) z--;
		ix += (z * LOG_TICS);

	    } else {
		ix = (int)((x - minx) * stepx);
	    }

	    if (ydim == LOG_BIN){
		z = log_b(y,2.0);
		if (z < 0) z-= 1.0;
		iy = LOG_TICS * (int)(z - lminy);
		if (z < 0){
		    z = - z;
		    while (z >= 1.0) z--;
		    z = 1.0 - z;
		} else while (z >= 1.0) z--;
		iy += (z * LOG_TICS);

	    } else {
		iy = (int)((y - miny) * stepy);
	    }

	    if (iy < 0) iy = 0;
	    if (ix < 0) ix = 0;
	    if (iy >= dimy) iy = dimy - 1;
	    if (ix >= dimx) ix = dimx - 1;
	    

	    inc_map_element(m,ix,iy);
	    N++;
	}
	if (pdf) 
	    scalar_op_map(m, Double(&v,N), GP_DIVIDE);
    }
	    
    
    free_array(vx);
    free_array(vy);
    free_array(tx);
    free_array(ty);
    return res;
}



void write_bin2d(struct bin2d * b, FILE * f)
{
 
    
    int i,j,k;
    struct map * m;
    char * str;
    struct array *x = b->xtics;
    struct array *x1 = b->xtics1;
    struct array *y = b->ytics;
    struct array *y1 = b->ytics1;

/*
  all that is needed is to write out a header of some kind
*/
    fprintf(f,"# %s\n#\n",b->title);
    fprintf(f,"# Automatically generated  2-d binned, aggregated distributions\n#\n");
    fprintf(f,"# (xval, xord), (yval, yord) (data1, ... data%d\n",b->n);
    for (i = 0; i < b->n; i++)
	fprintf(f,"# %d \t %s\n",i + 5, b->s[i]);
    fprintf(f,"# \n");
    
    for (i = 0; i < x->n; i++){
	for (j = 0; j < y->n; j++){
	    write_array_element(x,i,f);
	    write_array_element(x1,i,f);
	    write_array_element(y,j,f);
	    write_array_element(y1,j,f);
	    for (k = 0; k < b->n; k++){
		str =  value_to_string(get_map_value(b->m[k],i,j));
		fprintf(f," %s ",str);
	    }
	    fprintf(f,"\n");
	}
	fprintf(f,"\n");
    }
    fprintf(f,"\n");
    return;
}




/*
  This passes FILE to callout one line at a time.
  callout is supposed to turn the line into a value
  which we then put into the array

  callout must initialise itself when passed a NULL,
  and return the type of it's value.
*/

struct array * get_dist(FILE * f, struct value * (*callout)(char *, int *)){

    struct value *stat;
    struct array * dist;
    int index;
    int n = 0;
    int sz = 100;
    
    stat = (*callout)(NULL,0);			/* ask callout what it returns */
    dist = new_array(stat->type,sz);

    while (!get_nxt_line(f,mgbuf)){	
	index = n;
	stat = (*callout)(mgbuf,&index);
	if (index != HIST_NULL){		/* this record is in the dist. */
	    put_array_value(dist,n,stat);
	    n++;
		if (n == sz){
		    sz <<= 2;
		    dist = resize_array(dist,sz);
		}
	}
    }
    dist = resize_array(dist,n);

    return dist;
}

/*
  from a map of file names, derive a new map of array, entrywise, using callout
*/

struct map *
dist_map(struct map * map, struct value * (*callout)(char *, int *), char * dir){

    int y,x,n,m;
    char * str;
    struct array * a;
    struct map * new;
    FILE * f;
    int Fuck = 0;

    char *pathname = xalloc(MAXPATHLEN,"mapgen");
    
    if (getwd(pathname) == 0)
	fatal(pathname);
    if (dir != NULL ? chdir(dir): 0)
	fatal("could not change directory");
	
    if (map->type != STR){
	error("dist_map(): I need a map of file names!");
	return(NULL);

    }
    get_map_size(map,&n,&m);
    new = new_map(n,m,"Distribution", POINTER);
    set_map_free(new, (void (*)(void *))free_array);

    for (y = 0; y < n; y++){
	for (x = 0; x < m; x++){
	    if (is_null_point(map,y,x)){
		put_map_null(new,y,x);
		continue;
	    }
	    str = get_map_s(map,y,x);
	    if ((f = fopen(str,"r")) == NULL){
		fprintf(stderr,"Could not open %s\n",str);
		put_map_null(new,y,x);
		continue;
	    }
	    if (Debug)
		fprintf(stderr,"Gen %s",str);

	    a = get_dist(f,callout);
	    fclose(f);

	    if (Debug)
		fprintf(stderr,"\n");
	    
	    put_map_ptr(new,y,x, (void *)a);
	}
    }		    

    if (dir != NULL ? chdir(pathname): 0)
	fatal("could not restore directory");
    free(pathname);

    return new;
}

#endif
