/* -*- c++ -*- */
/*
 * Copyright 2002 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#ifndef _GRWEAVERMODTAIL_H_
#define _GRWEAVERMODTAIL_H_

#include <assert.h>

/*!
 * \brief Tail  block of a Weaver SSB/VSB modulator
 *
 * Complex mix with cos/sin of IF_freq & sum components.
 * See also GrSSBMod, which is basically the same as this
 * except it expects I & Q to to be a single VrComplex stream
 *
 * This is typically used like this:
 *
 *                      |---> matched filter --->|
 * ---> GrWeaverModHead |                        | GrWeaverTail --->
 *	  	        |---> matched filter --->|
 */

template<class iType, class oType>
class GrWeaverModTail : public VrSigProc {

public:
  GrWeaverModTail (float freq, float gain);

  virtual const char *name () { return "GrWeaverModTail"; }

  virtual int work (VrSampleRange output, void *o[],
		    VrSampleRange inputs[], void *i[]);

  //! frequency is in Hz
  void set_freq (float frequency) { freq = frequency; }
  void set_gain (float g)    { gain = g; }

protected:
  gr_nco<float,float>	nco;
  float			gain;
  float			freq;
};

template<class iType, class oType> 
GrWeaverModTail<iType,oType>::GrWeaverModTail (float freq, float gain)
  : VrSigProc(1, sizeof(iType), sizeof(oType)), gain(gain), freq(freq)
{
}

template<class iType, class oType> int
GrWeaverModTail<iType, oType>::work(VrSampleRange output, void *ao[],
				    VrSampleRange inputs[], void *ai[])
{
  assert (numberInputs == 2);

  iType *i = ((iType **) ai)[0];
  iType *q = ((iType **) ai)[1];
  oType *o = ((oType **) ao)[0];
  long size = output.size;

  sync (output.index);

  nco.set_freq (freq / getSamplingFrequency () * 2 * M_PI);

  for (long n = 0; n < size; n++) {
    float	cosv, sinv;
    nco.cossin (cosv, sinv);	// compute them at the same time (faster!)
    o[n] = (oType) (gain * (i[n] * cosv + q[n] * sinv));
    nco.step();
  }
  return output.size;
}

#endif /* _GRWEAVERMODTAIL_H_ */
