/* -*- c++ -*- */
/*
 * Copyright 2005 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#ifndef INCLUDED_GR_TIMER_H
#define INCLUDED_GR_TIMER_H

#include <gr_types.h>

class gr_timer;

typedef boost::shared_ptr<gr_timer> gr_timer_sptr;

typedef void (*gr_timer_hook)(gr_timer *, void *);

/*!
 * \brief create a timeout.
 *
 * gr_timer_hook is called when timer fires.
 */
gr_timer_sptr gr_make_timer (gr_timer_hook, void *);

/*!
 * \brief implement timeouts
 */
class gr_timer {
  double	d_expiry;
  double	d_period;
  gr_timer_hook	d_hook;
  void	       *d_hook_arg;

  friend gr_timer_sptr gr_make_timer (gr_timer_hook, void *);

  gr_timer (...);

public:
  ~gr_timer ();

  //! return absolute current time (seconds since the epoc).
  static double now ();
  
  /*!
   * \brief schedule timer to fire at abs_when
   * \param abs_when	absolute time in seconds since the epoc.
   */
  void schedule_at (double abs_when);

  /*!
   * \brief schedule timer to fire rel_when seconds from now.
   * \param rel_when	relative time in seconds from now.
   */
  void schedule_after (double rel_when);	// relative time in seconds

  /*!
   * \brief schedule a periodic timeout.
   * \param abs_when	absolute time to fire first time
   * \param period	time between firings
   */
  void schedule_periodic (double abs_when, double period);

  //! cancel timer
  void unschedule ();
};

#endif /* INCLUDED_GR_TIMER_H */
