;;; ecb-help.el --- online help for ECB and bug reporting

;; Copyright (C) 2001 Jesper Nordenberg
;; Copyright (C) 2001 Free Software Foundation, Inc.
;; Copyright (C) 2001 Klaus Berndl <klaus.berndl@sdm.de>

;; Author: Klaus Berndl <klaus.berndl@sdm.de>
;; Maintainer: Klaus Berndl <klaus.berndl@sdm.de>
;; Keywords: java, class, browser

;; This program is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by the Free
;; Software Foundation; either version 2, or (at your option) any later
;; version.

;; This program is distributed in the hope that it will be useful, but WITHOUT
;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;; FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
;; more details.

;; You should have received a copy of the GNU General Public License along
;; with GNU Emacs; see the file COPYING. If not, write to the Free Software
;; Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

;;; Commentary:
;;
;; Contains all online-help for ECB (stolen something from recentf.el)

;; $Id: ecb-help.el,v 1.79 2002/08/09 11:33:44 berndl Exp $

;;; Code

(require 'ecb-layout)
(require 'ecb-util)

(defconst ecb-help-info-start-file "ecb.info")
(defconst ecb-help-html-start-file "ecb.html")

(defgroup ecb-help nil
  "Settings for the ECB online help"
  :group 'ecb)

(defcustom ecb-show-help-format 'info
  "*The format `ecb-show-help' shows its online help. Allowed values are 'info
\(for the Info format) and 'html \(for HTML format). If the value is 'html
then `browse-url-browser-function' says which browser is used."
  :group 'ecb-help
  :type '(choice :tag "Online-help format" :menu-tag "Online-help format"
                 (const :tag "Info" :value info)
                 (const :tag "Html" :value html)))


(defcustom ecb-help-info-path (concat
                               (if running-xemacs
                                   (if (file-exists-p
                                        (concat ecb-ecb-dir
                                                ecb-help-info-start-file))
                                       "./"
                                     "../../info/")
                                 "./")
                               ecb-help-info-start-file)
  "*Path where the ECB online help in info format resides.
This must be the location of the file \"ecb.info\" which comes with the ECB
distribution. If is installed by unpacking the archive available on the ECB
website then this is the installation directory of ECB, i.e. where the elisp
files of ECB reside. If it is installed as XEmacs-package \(e.g. via the
package manager of XEmacs) then this is probably not the directory where the
elisp files of ECB reside but the directory \"../../info/\" \(relativ to the
elisp directory of ECB).

The path can either be an absolute path or a path relativ to the directory
where the elisp files of ECB are.

Normally there should be no need to change this option!"
  :group 'ecb-help
  :type 'file)

(defcustom ecb-help-html-path (concat
                               (if running-xemacs
                                   (cond ((file-exists-p
                                           (concat ecb-ecb-dir
                                                   ecb-help-html-start-file))
                                          "./")
                                         ((file-exists-p
                                           (concat ecb-ecb-dir
                                                   "../../html/"
                                                   ecb-help-html-start-file))
                                          "../../html/")
                                         (t
                                          "../../etc/ecb/html/"))
                                 "./")
                               ecb-help-html-start-file)
  "**Path where the ECB online help in HTML format resides.
This must be the location of the file \"ecb.html\" which comes with the ECB
distribution. If is installed by unpacking the archive available on the ECB
website then this is the installation directory of ECB, i.e. where the elisp
files of ECB reside. If it is installed as XEmacs-package \(e.g. via the
package manager of XEmacs) then this is probably not the directory where the
elisp files of ECB reside but either the directory \"../../html/\" or
\"../../etc/ecb/html/\" \(both relativ to the elisp directory of ECB).

The path can either be an absolute path or a path relativ to the directory
where the elisp files of ECB are.

Normally there should be no need to change this option!"
  :group 'ecb-help
  :type 'file)


(defun ecb-info (info-file &optional no-file-not-exist-err)
  "Starts `info' with INFO-FILE. If INFO-FILE does not exists then nil is
returned otherwise true. If NO-FILE-NOT-EXIST-ERR is not nil then just nil is
returned if INFO-FILE does not exist otherwise an error is reported."
  (if (file-exists-p info-file)
      (prog1 t
        (info info-file))
    (unless no-file-not-exist-err
      (ecb-error "Info file %s does not exists!" info-file))
    nil))

(defun ecb-browse-html-file (html-file &optional no-file-not-exist-err)
  "Opens HTML-FILE in the standard-webbrowser with `browse-url'. If INFO-FILE
does not exists then nil is returned otherwise true. If NO-FILE-NOT-EXIST-ERR
is not nil then just nil is returned if HTML-FILE does not exist otherwise an
error is reported."
  (if (file-exists-p html-file)
      (prog1 t
        (if (and (locate-library "browse-url")
                 (require 'browse-url)
                 (fboundp 'browse-url))
            (browse-url (concat "file://" html-file)
                        (if (boundp 'browse-url-new-window-flag)
                            browse-url-new-window-flag
                          browse-url-new-window-p))
          (ecb-error "Function 'browse-url needed for displaying HTML!")))
    (unless no-file-not-exist-err
      (ecb-error "HTML file %s does not exists!" html-file))
    nil))

;;;###autoload
(defun ecb-show-help (&optional format)
  "Shows the online help of ECB in Info or HTML-format depending on the value
of the option `ecb-show-help-format'. If called with prefix argument, i.e. if
FORMAT is not nil then the user is prompted to choose the format of the help
\(Info or Html).

If an error about not finding the needed help-file occurs please take a look
at the options `ecb-help-info-start-file' and `ecb-help-html-start-file'!"
  (interactive "P")
  (let ((f (if format
               (intern (ecb-query-string "Choose format of online-help:"
                                         (if (equal 'ecb-show-help-format
                                                    'html)
                                             '("info" "html")
                                           '("html" "info"))))
             ecb-show-help-format))
        (info-path-abs (expand-file-name
                        (if (or (string-match "^\\." ecb-help-info-path)
                                (string-match (concat "^"
                                                      (regexp-quote
                                                       ecb-help-info-start-file))
                                              ecb-help-info-path))
                            (concat ecb-ecb-dir ecb-help-info-path)
                          ecb-help-info-path)))
        (html-path-abs (ecb-fix-filename
                        (if (or (string-match "^\\." ecb-help-html-path)
                                (string-match (concat "^"
                                                      (regexp-quote
                                                       ecb-help-html-start-file))
                                              ecb-help-html-path))
                            (concat ecb-ecb-dir ecb-help-html-path)
                          ecb-help-html-path))))
    (if (equal f 'info)
        (ecb-info info-path-abs)
      (message "Opening ECB online-help in a web-browser...")
      (ecb-browse-html-file html-path-abs))))

;;
;; Problem reporting functions stolen from JDE
;;
(defvar ecb-problem-report-mail-address "ecb-list@lists.sourceforge.net" )

(defconst ecb-problem-report-message
  "Please enter the details of your bug report here")

(defun ecb-submit-problem-report()
  "Submit a problem report for the ECB to the ECB mailing-list. This command
generates in the edit-window a problem-report which contains already the
current values of all ECB options, the current backtrace-buffer if there is
any and the current message-buffer. You will be asked for a problem-report
subject and then you must insert a description of the problem. Please describe
the problem as detailed as possible!"
  (interactive)
  (if (not (ecb-point-in-edit-window))
      (ecb-select-edit-window))
  (if (not (locate-library "reporter"))
      (ecb-error "You need the reporter.el package to submit a bugreport for ECB!")
    (require 'reporter)
    (and 
     (y-or-n-p "Do you want to submit a problem report on the ECB? ")
     (progn
       (message "Preparing problem report...")
       ;;prepare the basic buffer
       (reporter-submit-bug-report
        ecb-problem-report-mail-address
        (format "ECB: %s, Semantic: %s, JDE: %s"
                ecb-version
                (if (boundp 'semantic-version)
                    semantic-version
                  "<=1.3.3")
                (if (boundp 'jde-version)
                    jde-version
                  "No JDE"))
        (ecb-problem-report-list-all-variables)
        nil
        'ecb-problem-report-post-hook
        ecb-problem-report-message)
       (ecb-redraw-layout)
       (mail-subject)
       (insert (read-string "Problem report subject: "
                            (format "ECB-%s -- " ecb-version)))
       (mail-text)
       (search-forward ecb-problem-report-message)
       (end-of-line)
       (message "Preparing bug report...done")))))

(defun ecb-problem-report-post-hook()
  "Function run the reporter package done its work. It looks for a message- and
a backtrace-buffer and inserts the contents of that."
  (save-excursion
    (beginning-of-buffer)
    ;; if the mail-packages has already inserted a signature we must not go to
    ;; the buffer-end but just before the signature
    (if (re-search-forward "^--[ \t]*$" nil t)
        (progn
          (beginning-of-line)
          (insert-string "\n\n\n")
          (forward-line -2))
      (goto-char (point-max))
      (insert-string "\n\n")) 
    (let* ((messages-buffer 
	    (get-buffer
	     (if running-xemacs " *Message-Log*" "*Messages*")))
	   (backtrace-buffer (get-buffer "*Backtrace*")))

      ;;insert the contents of the backtrace buffer if it is there. 
      (insert-string "\n\n-----------------------------------------------------\n")
      (if backtrace-buffer
          (progn
            (insert-string "The contents of the *Backtrace* buffer were\n\n")
	    (insert-buffer backtrace-buffer)
            ;; we must force the mark
	    (goto-char (mark t))
            (insert-string "\nEnd Insert *Backtrace* buffer" ))
        (insert-string "There was no *Backtrace* buffer" ))
      (insert-string "\n-----------------------------------------------------\n\n")

      ;;insert the contents of the messages buffer if it is there. 
      (insert-string "-----------------------------------------------------\n")
      (if messages-buffer
          (progn
            (insert-string "The contents of the *Messages* buffer were\n\n")
	    (insert-buffer messages-buffer)
	    (goto-char (mark t))
            (insert-string "\nEnd Insert *Messages* buffer" ))
        (insert-string "There was no *Messages* buffer" ))
      (insert-string  "\n-----------------------------------------------------\n\n"))))


(defun ecb-problem-report-list-all-variables()
  "List all variables starting with `ecb-' and some other variables which
could be interesting for support."
  (let ((emacs-vars (sort `(semantic-after-toplevel-cache-change-hook
                            semantic-after-partial-cache-change-hook
                            pre-command-hook
                            post-command-hook
                            after-save-hook
                            help-mode-hook
                            compilation-mode-hook
                            ,(if (boundp 'ediff-quit-hook)
                                 'ediff-quit-hook))
                          (function (lambda (l r)
                                      (string< (symbol-name l) (symbol-name r))))))
        (ecb-options (mapcar
                      'intern
                      (sort
                       (let (completion-ignore-case)
                         (all-completions "ecb-" obarray 'user-variable-p))
                       'string-lessp)))
        (ecb-internal-vars (sort '(ecb-path-selected-directory
                                   ecb-path-selected-source
                                   ecb-use-semantic-grouping
                                   ecb-idle-timer-alist
                                   ecb-post-command-hooks
                                   ecb-old-compilation-window-height
                                   ecb-toggle-layout-state)
                                 (function (lambda (l r)
                                             (string< (symbol-name l)
                                                      (symbol-name r)))))))
    (append emacs-vars ecb-internal-vars ecb-options)))


(provide 'ecb-help)

;; ecb-help.el ends here
