/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991,1992 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 * This code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY. No author or distributor accepts
 * responsibility to anyone for the consequences of using this code
 * or for whether it serves any particular purpose or works at all,
 * unless explicitly stated in a written agreement.
 *
 * Everyone is granted permission to copy, modify and redistribute
 * this code, except that the original author(s) must be given due credit,
 * and this copyright notice must be preserved on all copies.
 *
 *	Author:  Alan M. Carroll (carroll@cs.uiuc.edu)
 *      Modified:  Doug Bogia (bogia@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
 */

/*	shell.c: Normal toplevel shells (the shell widget) */
/* widget server top level shell support */
/* $Source: /export/kaplan/stable/sun4.os4.1/cb-2.0/src/data/node120.text,v $ */

static char rcsid[] = "shell.c $Revision: 1.0.1.2.1.3 $ $Date: 92/05/12 15:51:31 $ $State: Exp $ $Author: CBmgr $";

/* ------------------------------------------------------------------------ */
#include "header.h"
#include <Xm/DialogS.h>
#include <Xm/Protocols.h>
/* ------------------------------------------------------------------------ */
extern void
  ShellRemove(), ShellChildDestroyed(), ShellUpdate(), SendDestroy(),
  ShellReply(), ShellRealized(), ShellTransmit(), ShellEditable();

extern t_generic_widget ShellHandle();

struct widget_class_struct ShellClassRecord =
{
  WS_SHELL, "shell",
  ShellHandle, ShellRemove, ShellChildDestroyed, ShellUpdate,
  ShellReply, ShellRealized, ShellTransmit, ShellEditable,
} ;
/* ------------------------------------------------------------------------ */
ShellDestroyCallback(w, shell, data)
     Widget w;
     t_shell_widget shell;
     caddr_t data;
{
  ShellRemove(shell);
}
/* ------------------------------------------------------------------------ */
ShellUnmapCallback(w, shell, data)
     Widget w;
     t_shell_widget shell;
     caddr_t data;
{
  XtUnmapWidget(shell->widget);
}
/* ------------------------------------------------------------------------ */
ShellPopupCallback(w, shell, data)
     Widget w;
     t_shell_widget shell;
     caddr_t data;
{
  Dimension width, height;
  XtVaGetValues(shell->widget,
			XmNwidth, &width, XmNheight, &height, NULL);
  /* Now, actually move it there. For a prompt, we're using a
   * dialog widget, so we need to move the child instead of the
   * shell itself
   */
  WidgetAtMouse(((t_shell_widget)shell)->child->widget);
}
/* ------------------------------------------------------------------------ */
static struct keyword_entry_struct ShellKeywords[] =
{
  { ":title", "Shell", KEYWORD_COOKED },
  { ":header", NULL, KEYWORD_COOKED },
  { ":context", NULL, KEYWORD_GET_SEXP },
  { ":tag", NULL, KEYWORD_RAW },
  { ":wrap", (void *)KEYWORD_TRUE, KEYWORD_FLAG },
  { ":editable", (void *)KEYWORD_NONE, KEYWORD_FLAG },
  { ":type", NULL, KEYWORD_SEXP },
  { ":values", NULL, KEYWORD_SEXP },
  { ":popup-for", NULL, KEYWORD_SEXP },
  { ":destroy", NULL, KEYWORD_RAW },
  { ":args", NULL, KEYWORD_GET_SEXP },
  { ":transmit", (void *)KEYWORD_TRUE, KEYWORD_FLAG },
  { ":min-width", NULL, KEYWORD_SEXP },
  { ":min-height", NULL, KEYWORD_SEXP },
  { ":max-width", NULL, KEYWORD_SEXP },
  { ":max-height", NULL, KEYWORD_SEXP },
};

t_generic_widget
ShellHandle(parent,sexp)
     t_generic_widget parent;
     t_sexp sexp;
{
  t_generic_widget sw, child = NULL, tmp;
  t_shell_widget shell = NULL;
  t_sexp id, s_type, s_popup_for;
  t_sexp s_min_width, s_min_height, s_max_width, s_max_height;
  t_keyword_flag f_edit;
  t_keyword_flag f_wrap;
  char *title, *str;
  int n;
  Dimension cur_width, cur_height;
  Arg argl[8];
  char *class_name = "shell";
  WidgetClass class = topLevelShellWidgetClass; /* default */
  Widget w_parent = toplevel_widget;	/* default parent */
  t_shell_type prev_type;	/* previous type of shell */

  /* shell's shouldn't have parents, if we do, just ignore the request. */
  if (NULL == sexp || NULL != parent) return NULL;

  /* see if there's already a shell of this name */
  id = MBnth(sexp,1);
  for ( sw = toplevel_shells, tmp = NULL
       ; NULL != sw
       ; tmp = sw, sw = sw->next )
    {
      if (MBequal(id, sw->id))
	{
	  /* remove from top level list */
	  if (NULL == tmp) toplevel_shells = toplevel_shells->next;
	  else tmp->next = sw->next;
	  shell = (t_shell_widget)sw;
	  break;
	}
    }

  if (NULL == shell)
    {
      shell = NEW_STRUCT(shell_widget_struct);
      shell->type = WS_SHELL;
      shell->class = &ShellClassRecord;
      shell->parent = NULL;
      shell->child = NULL;		/* important - existance flag */
      shell->widget = NULL;
      shell->id = Getnth(sexp,1);
      shell->top = (t_generic_widget)shell;	/* we are our own top level */
      shell->stype = SHELL_NORMAL;
      shell->transmit_values = NULL;
    }
  else
    {
      /* free things that are going to get over-written */
      MBfree(shell->context);
      MBfree(shell->args);
      XtFree(shell->tag);
      XtFree(shell->destroy);
      XtFree(shell->mbus_header);
      XtFree(shell->values);
      /* Now, set the values to an initial state */
      shell->mbus_header = NULL;
      shell->context = NULL;
      prev_type = shell->stype;
    }

  /* sexp should be (shell "id" <items> &key ...) */
  MBparse_keywords(MBnthcdr(sexp,3), ShellKeywords, ARRAY_SIZE(ShellKeywords));
  title = (char *)ShellKeywords[0].result;
  shell->mbus_header = (char *)ShellKeywords[1].result;
  shell->context = (t_sexp)ShellKeywords[2].result;
  shell->tag = (char *)ShellKeywords[3].result;
  shell->mbus_wrap = (t_keyword_flag)ShellKeywords[4].result == KEYWORD_TRUE;
  f_edit = (t_keyword_flag)ShellKeywords[5].result;
  s_type = (t_sexp)ShellKeywords[6].result;
  shell->values = ContextString((t_sexp)ShellKeywords[7].result);
  s_popup_for = (t_sexp)ShellKeywords[8].result;
  shell->destroy = (char *)ShellKeywords[9].result;
  shell->args = (t_sexp)ShellKeywords[10].result;
  shell->transmit = (t_keyword_flag)ShellKeywords[11].result == KEYWORD_TRUE;
  s_min_width = (t_sexp)ShellKeywords[12].result;
  s_min_height = (t_sexp)ShellKeywords[13].result;
  s_max_width = (t_sexp)ShellKeywords[14].result;
  s_max_height = (t_sexp)ShellKeywords[15].result;
  
  if (!MBcompare_Cstring(s_type, ":prompt"))
    {
      class = xmDialogShellWidgetClass;
      shell->stype = SHELL_PROMPT;
    }
  else if (!MBcompare_Cstring(s_type, ":persistent"))
    shell->stype = SHELL_PERSISTENT;

  if (NULL == shell->widget)
    {
      n = 0;
      XtSetArg(argl[n], XmNallowShellResize, True), ++n;
      XtSetArg(argl[n], XmNwindowGroup, XtWindow(toplevel_widget)), ++n;
      /* To make things a little easier, we'll turn this off. This means
       * that we can realize and manage the widget before it's mapped.
       * We'll turn this back on after calling the realized() chain, unless
       * it's a persistent shell.
       */
      XtSetArg(argl[n], XmNmappedWhenManaged, False), ++n;

      /* If it's persistent, we have to intercept the delete request */
      if (SHELL_PERSISTENT == shell->stype)
	XtSetArg(argl[n], XmNdeleteResponse, XmUNMAP), ++n;
      else if (SHELL_PROMPT == shell->stype)
	{
	  Widget w;
	  class_name = "prompt";
	  if (NULL != s_popup_for
	      && NULL != (sw = FindToplevelShellID(s_popup_for)))
	    w_parent = sw->widget;
	}

      shell->widget = XtCreatePopupShell(class_name, class,
					 w_parent, argl, n);

      XtAddCallback(shell->widget, XmNdestroyCallback,
		    ShellDestroyCallback, shell);

      /* Well, dude-boys, it seems that the XmUNMAP delete response
       * resource doesn't work as advertised, it just ignores the DELETE
       * message, so we'll just roll our own. Yow.
       */
      if (SHELL_PERSISTENT == shell->stype)
	XmAddWMProtocolCallback(shell->widget, xa_wm_delete_window,
				(void *)ShellUnmapCallback, (caddr_t)shell);
      else if (SHELL_PROMPT == shell->stype)
	XtAddCallback(shell->widget, XmNpopupCallback,
		      ShellPopupCallback, shell);
    }
  else				/* there was already a shell */
    {
      if (prev_type == SHELL_PROMPT && NULL != shell->child)
	{
	  /* Have to get rid of the child now, instead of after parsing
	   * the new one; otherwise, the widget server dies with an
	   * error.
	   */
	  XtUnmanageChild(shell->child->widget);
	  shell->child->class->remove(shell->child);
	  XtDestroyWidget(shell->child->widget);
	  shell->child = NULL;	/* mark it as gone */
	  /* Need to sync things up here or else the new shell will
	   * take a long time to appear the first time, won't appear
	   * the second time, and will appear the third time.
	   */
	  XSync(XtDisplay(toplevel_widget), False);
	}
    }

  /* now, process the child widget and see if it's valid */
  child = HandleBusItem(shell, MBnth(sexp,2));

  if (NULL == child)			/* nope, didn't succeed */
    {
      if (MBLogLevel > 1) fprintf(stderr, "Shell got a bad child\n");
      if (NULL == shell->child)		/* nothing, evaporate */
	{
	  XtDestroyWidget(shell->widget);
	  shell = NULL;
	}
    }
  else					/* successfully got a child */
    {
      if (NULL == shell->child)
	shell->child = child;
      else				/* pre-existing child, remove */
	{
	  XtUnmanageChild(shell->child->widget);
	  shell->child->class->remove(shell->child);
	  XtDestroyWidget(shell->child->widget);
	  shell->child = child;
	}

      if (SHELL_PROMPT == shell->stype)
	{
	  if (KEYWORD_NONE == f_edit) f_edit = KEYWORD_TRUE;
	  /* We have to set this in order to get the dialog to show up where
	   * the mouse is, and not on top of the control panel
	   */
	  XtSetArg(argl[0], XmNdefaultPosition, False);
	  XtSetValues(shell->child->widget, argl, 1);
	}
      /* This apparently maps prompt (popup) shells, so XtPopup never
       * actually gets called, but the popup callback gets invoked.
       * Very wierd.
       */
      XtManageChild(shell->child->widget);

      if (NULL != shell->child && KEYWORD_NONE != f_edit)
	shell->class->editable(shell, KEYWORD_TRUE == f_edit);

      if (NULL != title)
	{
	  n = 0;
	  XtSetArg(argl[n], XmNtitle, title), ++n;
	  XtSetArg(argl[n], XmNiconName, title), ++n;
	  XtSetValues(shell->widget, argl, n);
	}
      XtRealizeWidget(shell->widget);
      shell->class->realized(shell);	/* do final adjustments */
      if (s_min_width || s_min_height || s_max_width || s_max_height)
      {
	/* Now set the min/max sizes according the keys given.
	 * Here we read the values of the current window sizes in
	 * case we need them later.  Seemed easier to do it this way.
	 */
	n = 0;
	XtSetArg(argl[n], XmNwidth, &cur_width), ++n;
	XtSetArg(argl[n], XmNheight, &cur_height), ++n;
	XtGetValues(shell->widget, argl, n);
	n = 0;
	if (s_min_width)
	{
	  if (!MBcompare_Cstring(s_min_width, ":fixed"))
	      XtSetArg(argl[n], XmNminWidth, cur_width), ++n;
	  else
	  {
	    str = MBCstring(s_min_width);
	    XtSetArg(argl[n], XmNminWidth, atoi(str)), ++n;
	    FREE(str);
	  }
	}
	if (s_min_height)
	{
	  if (!MBcompare_Cstring(s_min_height, ":fixed"))
	      XtSetArg(argl[n], XmNminHeight, cur_height), ++n;
	  else
	  {
	    str = MBCstring(s_min_height);
	    XtSetArg(argl[n], XmNminHeight, atoi(str)), ++n;
	    FREE(str);
	  }
	}
	if (s_max_width)
	{
	  if (!MBcompare_Cstring(s_max_width, ":fixed"))
	      XtSetArg(argl[n], XmNmaxWidth, cur_width), ++n;
	  else
	  {
	    str = MBCstring(s_max_width);
	    XtSetArg(argl[n], XmNmaxWidth, atoi(str)), ++n;
	    FREE(str);
	  }
	}
	if (s_max_height)
	{
	  if (!MBcompare_Cstring(s_max_height, ":fixed"))
	      XtSetArg(argl[n], XmNmaxHeight, cur_height), ++n;
	  else
	  {
	    str = MBCstring(s_max_height);
	    XtSetArg(argl[n], XmNmaxHeight, atoi(str)), ++n;
	    FREE(str);
	  }
	}
	XtSetValues(shell->widget, argl, n);
      }
      
    }

  XtFree(title);
  return (t_generic_widget) shell;
}
/* ------------------------------------------------------------------------ */
/* Called by someone else when they want us to remove all of our data
 * structures. Destruction of the widgets is the caller's job.
 */
void
ShellRemove(self) t_shell_widget self;
{
  t_generic_widget list, tmp;

  SendDestroy(self);

  MBfree(self->id);
  MBfree(self->context);
  MBfree(self->args);
  XtFree(self->values);
  XtFree(self->tag);
  XtFree(self->destroy);
  XtFree(self->mbus_header);
  if (NULL != self->child) self->child->class->remove(self->child);

  /* remove ourselves from the top level list */
  for ( tmp = NULL, list = toplevel_shells
       ; NULL != list
       ; tmp = list, list = list->next )
    if (list == (t_generic_widget)self)
      {
	if (NULL == tmp)		/* first item */
	  toplevel_shells = toplevel_shells->next;
	else
	  tmp->next = list->next;	/* clip self out */
	break;
      }

  XtFree(self);
}
/* ------------------------------------------------------------------------ */
/* Sends destroy tag if one exists */
void
SendDestroy(self) t_shell_widget self;
{
  if (NULL != self->destroy)
    {
      t_mbus_reply reply = NewReply();

      reply->tag = self->destroy;
      ShellReply(self, reply);
      SendReply(reply);
      FreeReply(reply);
    }
}
/* ------------------------------------------------------------------------ */
void
ShellActionReply(shell, s_reply)
     t_shell_widget shell;
     t_sexp s_reply;
{
  t_mbus_reply reply = NewReply();

  if (MB_CONSP(s_reply))
    {
      reply->tag = MBprint_Cstring(MB_CAR(s_reply));
      reply->value = ContextString(MB_CDR(s_reply));
    }
  else
    reply->tag = MBprint_Cstring(s_reply);
  shell->class->reply(shell, reply);

  SendReply(reply);
  FreeReply(reply);
}

/* ------------------------------------------------------------------------ */
/* Called with an update sexp */
static struct keyword_entry_struct sukw [] =
{
  { ":editable", KEYWORD_NONE, KEYWORD_FLAG },
  { ":delete", NULL, KEYWORD_SEXP },
  { ":action", NULL, KEYWORD_SEXP },
  { ":reply", NULL, KEYWORD_SEXP },
  { ":title", NULL, KEYWORD_COOKED },
  { ":id", NULL, KEYWORD_GET_SEXP },
  { ":transmit", KEYWORD_NONE, KEYWORD_FLAG },
} ;

void
ShellUpdate(self,sexp)
     t_shell_widget self;
     t_sexp sexp;
{
  t_sexp s_delete, s_action, s_reply, s_id;
  t_keyword_flag f_edit, f_transmit;
  char *title;
  int n;

  if (NULL == self->child || NULL == sexp) return;	/* nothing to do! */

  if (!MBequal(MBnth(sexp,1), self->id)) /* not ours, pass on */
    self->child->class->update(self->child, sexp);
  else
    {
      MBparse_keywords(sexp, sukw, ARRAY_SIZE(sukw));
      n = 0;
      f_edit = (t_keyword_flag) sukw[n++].result;
      s_delete = (t_sexp) sukw[n++].result;
      s_action = (t_sexp)sukw[n++].result;
      s_reply = (t_sexp)sukw[n++].result;
      title = (char *) sukw[n++].result;
      s_id = (t_sexp) sukw[n++].result;
      f_transmit = (t_keyword_flag) sukw[n++].result;

      if (KEYWORD_NONE != f_edit)
	self->class->editable(self, KEYWORD_TRUE == f_edit);

      if (NULL != s_action) Act(self, s_action, 1);

      if (NULL != s_reply) ShellActionReply(self, s_reply);

      if (!MBcompare_Cstring(s_delete, ":self"))
	XtDestroyWidget(self->widget);

      if (NULL != title)
	XtVaSetValues(self->widget, XmNtitle, title, XmNiconName, title, NULL);

      if (NULL != s_id)
	{
	  /* When a shell id is updated, we transmit the destroy tag for the */
          /* the object represented by the old id, if one exists. */
	  if (self->id != s_id) SendDestroy(self);
	  MBfree(self->id);
	  self->id = s_id;
	}
      if (KEYWORD_NONE != f_transmit)
	  self->transmit = KEYWORD_TRUE == f_transmit;
    }
}
/* ------------------------------------------------------------------------ */
/* Called by a child when it decides to destroy itself for whatever reason.
 * We should update our internal data to remove references to the child.
 * Assume that all of the childs internal and widgets stuff is taken care
 * of by the child.
 */
void
ShellChildDestroyed(self, child)
     t_shell_widget self;
     t_generic_widget child;
{
}
/* ------------------------------------------------------------------------ */
/* Fill in a reply record to be sent to the MBus */
void
ShellReply(self, reply)
     t_shell_widget self;
     t_mbus_reply reply;
{
  if (NULL == reply->context) reply->context = ContextString(self->context);
  if (NULL == reply->header) reply->header = strdup(self->mbus_header);
  if (NULL == reply->tag) reply->tag = strdup(self->tag);
  if (NULL == reply->args) reply->args = ContextString(self->args);
  reply->wrap = self->mbus_wrap;
}
/* ------------------------------------------------------------------------ */
/* Called after the widget has been realized, for final fiddling */
void
ShellRealized(self) t_shell_widget self;
{

  if (NULL != self->child) self->child->class->realized(self->child);
  XRaiseWindow(XtDisplay(self->widget), XtWindow(self->widget));
}
/* ------------------------------------------------------------------------ */
void
ShellTransmit(message, orig_reply, self)
  t_sexp message;
  t_mbus_reply orig_reply;
  t_shell_widget self;
{
  if (NULL != self->child)
      self->child->class->transmit(message, orig_reply, self->child);
  if (NULL != self->values) MBput_Cstring(message, self->values);
}
/* ------------------------------------------------------------------------ */
void
ShellEditable(self, flag)
     t_shell_widget self;
     int flag;
{
  if (NULL != self->child) self->child->class->editable(self->child, flag);
}
/* ------------------------------------------------------------------------ */
void
ShellTransmitHook(message, orig_reply, self)
  t_sexp message;
  t_mbus_reply orig_reply;
  t_shell_widget self;
{
  char *s;
  if (NULL != self->transmit_values)
    {
      s = ContextString(self->transmit_values);
      MBput_Cstring(message, s);
      XtFree(s);
    }
  self->class->transmit(message, orig_reply, self);
}
/* ------------------------------------------------------------------------ */
/* Extra things on the transmit can be a tag, or a list of keyword / value
 * pairs for sending along.
 */
void
ShellPrepareTransmit(self, sexp)
     t_shell_widget self;
     t_sexp sexp;
{
  /* sexp is of the form (transmit id [tag] <values>) */
  t_mbus_reply reply;
  t_sexp tag;

  reply = NewReply();

  tag = MBnth(sexp, 2);
  /* See if the tag has been specified.  We will know that it is a tag
   * if it is an atom and not a keyword or if it is a list.
   */
  if (MB_CHUNKP(tag) && !MB_KEYWORDP(tag))
  {
    /* The tag is an atom */
    reply->status = TAG_PROVIDED;
    reply->tag = MBprint_Cstring(tag);
    self->transmit_values = MBnthcdr(sexp, 3);
  }
  else if (MB_CONSP(tag))
  {
    /* The tag is a list.  Assume it contains the tag and the new
     * args to be used
     */
    reply->status = TAG_AND_ARGS_PROVIDED;
    reply->tag = ContextString(tag);
    /* In this case we don't want the args from the shell.  We will
     * assume they were sent with the tag.  Thus, we set the args to
     * the null string.
     */
    reply->args = strdup("");
    self->transmit_values = MBnthcdr(sexp, 3);
  }    
  else if (NULL != self->tag)
      /* tag will be set by the ShellReply */
      self->transmit_values = MBnthcdr(sexp, 2);
  else
      self->transmit_values = NULL;

  reply->transmit = self->transmit;
  reply->hook = ShellTransmitHook;
  reply->hook_data = (caddr_t) self;

  ShellReply(self, reply);

  SendReply(reply);
  FreeReply(reply);
}
/* ------------------------------------------------------------------------ */
/* Take a message like (terminate <tag> <value>), and delete the shell
 * if tag and value are in the context
 */
void
ShellTerminate(shell, sexp)
     t_shell_widget shell;
     t_sexp sexp;
{
  t_sexp s_tag, s_value;
  t_sexp context, tag, value;

  tag = MBnth(sexp, 1);
  value = MBnth(sexp, 2);

  if (NULL == tag || NULL == value) return;

  for (context = shell->context
       ; NULL != context
       ; context = MBnthcdr(context, 2))
    {
      s_tag = MB_CAR(context);
      s_value = MBnth(context, 1);

      if (MBequal(tag, s_tag) && MBequal(value, s_value))
	{
	  XtDestroyWidget(shell->widget);
	  return;
	}
    }
}
/* ------------------------------------------------------------------------ */
