/*
 * readhead.c
 *
 * This file contains functions for reading parts of a ZIP-archive.
 * Gratefully received from a person who wishes to remain anonymous.
 *
 * $Id: readhead.c,v 1.3 1996/08/21 18:09:30 conrad Release1_2 $
 *
 * $Log: readhead.c,v $
 * Revision 1.3  1996/08/21 18:09:30  conrad
 * Some cleanups to suppress some warnings...
 *
 * Revision 1.2  1996/08/21 17:38:14  conrad
 * Added noskip parameter to read_{local,dir,end}
 *
 */

/******************************************************************************/
/* Functions to read signature and headers                                    */
/******************************************************************************/
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include "headers.h"

static char RCSID[]="$Id: readhead.c,v 1.3 1996/08/21 18:09:30 conrad Release1_2 $";

local lh;
dirtype dir;
enddirtype enddir;
ddesctype ddesc;
char		*filename=NULL;
static int	fnlen;

/******************************************************************************/
/* READ Integer                                                               */
/******************************************************************************/
static int read2int( FILE *fp, unsigned short *dest )
{
unsigned char	dummy[2];

    if( fread( dummy, 2, 1, fp ) != 1 )
	return 0;

    *dest = (((unsigned short)dummy[1])<<8)+dummy[0];

    return 1;
}

static int read4int( FILE *fp, unsigned long *dest )
{
unsigned short	d1, d2;

    if( !read2int( fp, &d1 ) || !read2int( fp, &d2 ) )
	return 0;

    *dest = (((unsigned int)d2)<<16) + d1;

    return 1;
}

/******************************************************************************/
/* READ SIGNATURE                                                             */
/******************************************************************************/
int read_sig(FILE *fp)
{
unsigned long sig;
int ret;

  ret = read4int( fp, &sig );
  if(ret != 1)
  {
    printf("Error in signature reading\n");
    return(-1);
  }
  else;

  switch(sig)
  {
    case(0x04034b50):
      return(LOCAL);
      break;
    case(0x02014b50):
      return(DIR);
      break;
    case(0x06054b50):
      return(END);
      break;
    default:
      return(-1);
      break;
  }
}


/******************************************************************************/
/* READ FILE HEADER                                                           */
/* Returns: -1 generic error                                                  */
/*           1 found                                                          */
/*           0 skipped                                                        */
/******************************************************************************/
int read_local(FILE *fp,char *name, int noskip)
{
int ret;

  ret = fread( lh.version, 2, 1, fp );
  if( ret == 1 )
	ret = fread( lh.gpb, 2, 1, fp );
  if( ret == 1 )
	ret = fread( lh.compr, 2, 1, fp );
  if( ret == 1 )
	ret = fread( lh.time, 2, 1, fp );
  if( ret == 1 )
	ret = fread( lh.date, 2, 1, fp );
  if( ret == 1 )
	ret = read4int( fp, &lh.crc );
  if( ret == 1 )
	ret = read4int( fp, &lh.csize );
  if( ret == 1 )
	ret = read4int( fp, &lh.uncsize );
  if( ret == 1 )
	ret = read2int( fp, &lh.flen );
  if( ret == 1 )
	ret = read2int( fp, &lh.extralen );
  if(ret != 1)
  {
    printf("Error in local header\n");
    return(-1);
  }
  else;

#ifdef DEBUG
  printf("Version       = %02x %02x\n", lh.version[0], lh.version[1]);
  printf("General purp. = %02x %02x\n", lh.gpb[0], lh.gpb[1]);
  printf("Compression   = %02x %02x\n", lh.compr[0], lh.compr[1]);
  printf("Time          = %02x %02x\n", lh.time[0], lh.time[1]);
  printf("Date          = %02x %02x\n", lh.date[0], lh.date[1]);
  printf("CRC           = %08lx\n", lh.crc);
  printf("Compr. size   = %lu\n", lh.csize);
  printf("Uncomp. size  = %lu\n", lh.uncsize);
  printf("Filename len. = %d\n", lh.flen);
  printf("Extra field   = %d\n", lh.extralen);
#endif

  if( !filename )
  {
    filename = malloc( 2*lh.flen+1 );
    fnlen = 2*lh.flen;
  }
  else if( lh.flen > fnlen )
  {
    filename = realloc( filename, 2*lh.flen+1 );
    fnlen = 2*lh.flen;
  }
  memset(filename, '\0', fnlen+1);
  ret = fread(filename, lh.flen, 1, fp);
  if(ret != 1)
  {
    printf("Error in reading filename\n");
    return(-1);
  }
  else;

#ifdef DEBUG
  printf("Filename = %s\n", filename);
#endif

  if(lh.extralen != 0 && !noskip)
  {
#ifdef DEBUG
    printf("File contains extra data\n");
#endif
    ret = fseek(fp, lh.extralen, SEEK_CUR);
  }
  else;

  /**************************************************************************/
  /* If the file is the one we are searching, returns                       */
  /**************************************************************************/
  if(strcmp(name, filename) == 0)
  {
    return(1);
  }
  else if( !noskip )
  {
    /************************************************************************/
    /* This is to skip the file data                                        */
    /************************************************************************/
    ret = fseek(fp, lh.csize, SEEK_CUR);
    if(ret != 0)
    {
      perror("Error seeking file");
      return(-1);
    }
    else;
#ifdef DEBUG
    printf("File data skipped...\n");
#endif
  
    if((lh.gpb[0] & 0x08) == 0x08)
    {
#ifdef DEBUG
      printf("File contains data descriptor\n");
#endif
      /*ret = fread((char *)&ddesc, sizeof(ddesc), 1, fp);*/
      ret = read4int( fp, &ddesc.crc );
      if( ret == 1 )
	ret = read4int( fp, &ddesc.csize );
      if( ret == 1 )
	ret = read4int( fp, &ddesc.uncsize );
      if(ret != 1)
      {
        perror("Error reading data descriptor");
        return(-1);
      }
      else;
#ifdef DEBUG
      printf("CRC           = %08lx\n", ddesc.crc);
      printf("Compr. size   = %lu\n", ddesc.csize);
      printf("Uncomp. size  = %lu\n", ddesc.uncsize);
#endif
    }
    else;
  }

  return(0);
}

/******************************************************************************/
/* READ DIRECTORY ENTRY                                                       */
/******************************************************************************/
int read_dir(FILE *fp, int noskip)
{
int ret;

  /*ret = fread((char *)&dir, sizeof(dir), 1, fp);*/
  ret = fread( dir.version, 2, 1, fp );
  if( ret == 1 )
	ret = fread( dir.verneed, 2, 1, fp );
  if( ret == 1 )
	ret = fread( dir.gpb, 2, 1, fp );
  if( ret == 1 )
	ret = fread( dir.compr, 2, 1, fp );
  if( ret == 1 )
	ret = fread( dir.time, 2, 1, fp );
  if( ret == 1 )
	ret = fread( dir.date, 2, 1, fp );
  if( ret == 1 )
	ret = read4int( fp, &dir.crc );
  if( ret == 1 )
	ret = read4int( fp, &dir.csize );
  if( ret == 1 )
	ret = read4int( fp, &dir.uncsize );
  if( ret == 1 )
	ret = read2int( fp, &dir.flen );
  if( ret == 1 )
	ret = read2int( fp, &dir.extralen );
  if( ret == 1 )
	ret = read2int( fp, &dir.commlen );
  if( ret == 1 )
	ret = read2int( fp, &dir.disk );
  if( ret == 1 )
	ret = fread( dir.attr, 2, 1, fp );
  if( ret == 1 )
	ret = read4int( fp, &dir.extattr );
  if( ret == 1 )
	ret = read4int( fp, &dir.locoffset );
  if(ret != 1)
  {
    printf("Error in dir header\n");
    return(-1);
  }
  else;

#ifdef DEBUG
  printf("Version       = %02x %02x\n", dir.version[0], dir.version[1]);
  printf("Version need  = %02x %02x\n", dir.verneed[0], dir.verneed[1]);
  printf("General purp. = %02x %02x\n", dir.gpb[0], dir.gpb[1]);
  printf("Compression   = %02x %02x\n", dir.compr[0], dir.compr[1]);
  printf("Time          = %02x %02x\n", dir.time[0], dir.time[1]);
  printf("Date          = %02x %02x\n", dir.date[0], dir.date[1]);
  printf("CRC           = %08lx\n", dir.crc);
  printf("Compr. size   = %lu\n", dir.csize);
  printf("Uncomp. size  = %lu\n", dir.uncsize);
  printf("Filename len. = %d\n", dir.flen);
  printf("Extra field   = %d\n", dir.extralen);
  printf("Comment len   = %d\n", dir.commlen);
  printf("Disk          = %d\n", dir.disk);
  printf("Attributes    = %02x %02x\n", dir.attr[0], dir.attr[1]);
  printf("Ext. attr.    = %08lx\n", dir.extattr);
  printf("Offset of loc = %lu\n", dir.locoffset);
#endif

  if( !filename )
  {
    filename = malloc( 2*lh.flen+1 );
    fnlen = 2*lh.flen;
  }
  else if( lh.flen > fnlen )
  {
    filename = realloc( filename, 2*lh.flen+1 );
    fnlen = 2*lh.flen;
  }
  memset(filename, '\0', fnlen+1);
  ret = fread(filename, dir.flen, 1, fp);
  if(ret != 1)
  {
    printf("Error in reading filename\n");
    return(-1);
  }
  else;

#ifdef DEBUG
  printf("Filename = %s\n", filename);
#endif

  if(dir.extralen != 0 && !noskip)
  {
#ifdef DEBUG
    printf("File contains extra data (skipped)\n");
#endif
    ret = fseek(fp, dir.extralen, SEEK_CUR);
  }
  else;

  if(dir.commlen != 0 && !noskip)
  {
#ifdef DEBUG
    printf("File contains comment data (skipped)\n");
#endif
    ret = fseek(fp, dir.commlen, SEEK_CUR);
  }
  else;

  return 0;
}

/******************************************************************************/
/* READ END OF DIRECTORY                                                      */
/******************************************************************************/
int read_end(FILE *fp, int noskip)
{
int ret;

  /*ret = fread((char *)&enddir, sizeof(enddir), 1, fp);*/
  ret = read2int( fp, &enddir.diskno );
  if( ret == 1 )
	ret = read2int( fp, &enddir.centrdiskno );
  if( ret == 1 )
	ret = read2int( fp, &enddir.diskentries );
  if( ret == 1 )
	ret = read2int( fp, &enddir.totentries );
  if( ret == 1 )
	ret = read4int( fp, &enddir.centrsize );
  if( ret == 1 )
	ret = read4int( fp, &enddir.centroffset );
  if( ret == 1 )
	ret = read2int( fp, &enddir.commlen );
  if(ret != 1)
  {
    printf("Error in end of dir\n");
    return(-1);
  }
  else;

#ifdef DEBUG
  printf("Disk          = %d\n", enddir.diskno);
  printf("Central disk  = %d\n", enddir.centrdiskno);
  printf("Disk entries  = %d\n", enddir.diskentries);
  printf("Total  entr.  = %d\n", enddir.totentries);
  printf("Central size  = %lu\n", enddir.centrsize);
  printf("Central offset= %lu\n", enddir.centroffset);
  printf("Comment len   = %d\n", enddir.commlen);
#endif

  if(enddir.commlen != 0 && !noskip)
  {
#ifdef DEBUG
    printf("Directory contains comment data (skipped)\n");
#endif
    ret = fseek(fp, enddir.commlen, SEEK_CUR);
  }
  else;

  return 0;
}
