/*  sun-sp64.c -- 6400 bps coder based on switched prediction.
 *
 *  The author of this software is William Dorsey.
 *  Copyright (c) 1994, 1995 by William Dorsey.  All rights reserved.
 *
 * THIS SOFTWARE IS BEING PROVIDED "AS IS", WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTY.  IN PARTICULAR, THE AUTHOR DOES NOT MAKE ANY CLAIM OR
 * WARRANTY OF ANY KIND CONCERNING THE MERCHANTABILITY OF THIS SOFTWARE OR
 * ITS FITNESS FOR ANY PARTICULAR PURPOSE.
 */

/* REVISION HISTORY
 *
 * DATE      RESPONSIBLE PARTY  DESCRIPTION
 * -------------------------------------------------------------------------
 * 95/06/08  W. Dorsey		Creation
 */

#include <stdio.h>
#include <math.h>

#include "machine.h"

/* Conversion functions */
#define Irint(X)	((int)((X) < 0. ? (X) - 0.5 : (X) + 0.5))
#define audio_s2f(X)	(((UINT16)(X)) == 0x8000 ? -1. :	\
			((float)((INT16)(X))) / 32767.)
#define audio_f2s(X)	((X) >= 1. ? 32767 : (X) <= -1. ? -32767 :	\
			(INT16)(Irint((X) * 32767.)))

/* Operating Parameters */
#define IBLKSIZE	160
#define	OBLKSIZE	(IBLKSIZE*3/4)
#define	YPMIN		0.001
#define YPMAX		0.175
#define YPSIZE		6
#define SLOPE_MULT	0.80

static struct dmsp_state_t {
	float	ypdelta;
	float	y[2];
} handle;

static float filt[4][2] = {
	{  1.53, -0.72, },	/* filter coefficients */
	{  0.95, -0.34, },
	{  0.48, -0.21, },
	{ -0.63, -0.36  }
};

/* Local functions */
void downsample(float *x, int nx, float *y);
void upsample(float *x, int nx, float *y);

void
sp64_init(void)
{
    /* initializations */
    handle.y[0] = 0.0;
    handle.y[1] = 0.0;
    handle.ypdelta = (YPMAX - YPMIN) / ((float) (1<<YPSIZE) - 1);
}

int
sp64_encode(INT16 *x, UINT8 *bits)
{
    int		i, state, step;
    float	yy, y, phi0, phi1, step_size;
    float	*xp, xf[IBLKSIZE], xr[OBLKSIZE];
    UINT8	mask = 0x80, data = 0;

    /* convert input to floating point */
    for (i = 0; i < IBLKSIZE; i++)
	xf[i] = audio_s2f(x[i]);

    /* resample input */
    downsample(xf, IBLKSIZE, xr);

    /*
     * Compute normalized autocorrelation at lag 0 & 1
     * Compute step sized based on RMS value
     */
    xp = xr;
    yy = *xp++;				/* priming the loop */
    phi0 = yy * yy + 1.0e-20;		/* 1.0e-20 to prevent divide by 0 */
    step_size = phi1 = 0.0;

    for (i = 1; i < OBLKSIZE; i++, xp++) {
        phi0 += *xp * *xp;		/* autocorr at lag 0 */
        phi1 += *xp * yy;		/* autocorr at lag 1 */
        step_size += (*xp - yy) * (*xp - yy);	/* rms calc */
        yy = *xp;
    }
    phi1 /= phi0;

    /* select predictor state */
    if (phi1 > 0.7)
	state = 0;
    else if (phi1 > 0.4)
	state = 1;
    else if (phi1 > 0.0)
	state = 2;
    else
	state = 3;

    /* compute step size based on RMS value of input */
    step_size = SLOPE_MULT * sqrt(step_size / (OBLKSIZE-1));

    /* check step size for bounds */
    if (step_size < YPMIN)
	step_size = YPMIN;
    else if (step_size > YPMAX)
	step_size = YPMAX;
	
    /* quantize step size to YPSIZE bits */
    step = Irint((step_size - YPMIN) / handle.ypdelta);
    step_size = YPMIN + (float) step * handle.ypdelta;

    /* save predictor state and quantized step size in output */
    *bits++ = state + (step << 2);

    /* main loop */
    for (i = 0; i < OBLKSIZE; i++) {
	/* apply linear predictive filter */
	y = filt[state][0] * handle.y[0] + filt[state][1] * handle.y[1];

	handle.y[1] = handle.y[0];

	if (xr[i] > y) {
	    y += step_size;
	    data |= mask;
	}
	else
	    y -= step_size;

	if (!(mask >>= 1)) {
	    *bits++ = data;
	    data = mask;
	    mask = 0x80;
	}

	handle.y[0] = y;
    }
}

int
sp64_decode(UINT8 *bits, INT16 *x)
{
    int		i, state, step;
    float	y, step_size;
    float	xf[IBLKSIZE], xr[OBLKSIZE];

    /* get state & step size */
    state = *bits & 0x3;	/* ls 2 bits */
    step = *bits++ >> 2;	/* ms 6 bits */
    
    /* decode step size from quantized step size */
    step_size = YPMIN + (float) step * handle.ypdelta;

    /* compute output from input bits */
    for (i = 0; i < OBLKSIZE; i++) {
	/* apply linear predictive filter */
	y = filt[state][0] * handle.y[0] + filt[state][1] * handle.y[1];

	handle.y[1] = handle.y[0];

	if (bits[i>>3] & 0x80>>(i&7))
	    y += step_size;
	else
	    y -= step_size;

	/* save output */
	xr[i] = handle.y[0] = y;
    }

    /* resample */
    upsample(xr, OBLKSIZE, xf);

    /* convert output to 16-bit signed integers */
    for (i = 0; i < IBLKSIZE; i++)
	x[i] = audio_f2s(xf[i]);
}
