// === Starter Application 1.0.0 === //

/*  ================================================================
    Credit card verification functions
    ================================================================ */


/*  ================================================================
    FUNCTION:  IsCC()
 
    INPUT:     st - a string representing a credit card number

    RETURNS:  true, if the credit card number passes the Luhn Mod-10
		    test.
	      false, otherwise
    ================================================================ */

function IsCC(st) {
  // Encoding only works on cards with less than 19 digits
  if (st.length > 19)
    return (false);

  sum = 0; mul = 1; l = st.length;
  for (i = 0; i < l; i++) {
    digit = st.substring(l-i-1,l-i);
    tproduct = digit*mul;
    if (tproduct >= 10)
      sum += (tproduct % 10) + 1;
    else
      sum += tproduct;
    if (mul == 1)
      mul++;
    else
      mul--;
  }
// Uncomment the following line to help create credit card numbers
// 1. Create a dummy number with a 0 as the last digit
// 2. Examine the sum written out
// 3. Replace the last digit with the difference between the sum and
//    the next multiple of 10.

//  document.writeln("<BR>Sum      = ",sum,"<BR>");

  if ((sum % 10) == 0)
    return (true);
  else
    return (false);

} // END FUNCTION IsCC()



/*  ================================================================
    FUNCTION:  IsVISA(), a.k.a. IsVisa()
 
    INPUT:     cc - a string representing a credit card number

    RETURNS:  true, if the credit card number is a valid VISA number.
		    
	      false, otherwise
    ================================================================ */

function IsVISA(cc)
{
  if (((cc.length == 16) || (cc.length == 13)) &&
      (cc.substring(0,1) == 4))
    return IsCC(cc);
  return false;
}  // END FUNCTION IsVISA()


function IsVisa(cc)
{
  return IsVISA(cc);
}  // END FUNCTION IsVisa


/*  ================================================================
    FUNCTION:  IsMasterCard(), a.k.a. IsMastercard(), IsMC()
 
    INPUT:     cc - a string representing a credit card number

    RETURNS:  true, if the credit card number is a valid MasterCard
		    number.
		    
	      false, otherwise
    ================================================================ */

function IsMasterCard(cc)
{
  firstdig = cc.substring(0,1);
  seconddig = cc.substring(1,2);
  if ((cc.length == 16) && (firstdig == 5) &&
      ((seconddig >= 1) && (seconddig <= 5)))
    return IsCC(cc);
  return false;

} // END FUNCTION IsMasterCard()


function IsMastercard(cc)
{
  return IsMasterCard(cc);
}


function IsMC(cc)
{
  return IsMasterCard(cc);
}



/*  ================================================================
    FUNCTION:  IsAmericanExpress(), a.k.a. IsAmEx()
 
    INPUT:     cc - a string representing a credit card number

    RETURNS:  true, if the credit card number is a valid American
		    Express number.
		    
	      false, otherwise
    ================================================================ */

function IsAmericanExpress(cc)
{
  firstdig = cc.substring(0,1);
  seconddig = cc.substring(1,2);
  if ((cc.length == 15) && (firstdig == 3) &&
      ((seconddig == 4) || (seconddig == 7)))
    return IsCC(cc);
  return false;

} // END FUNCTION IsAmericanExpress()


function IsAmEx(cc)
{
  return IsAmericanExpress(cc);
}



/*  ================================================================
    FUNCTION:  IsDinersClub(), a.k.a. IsDC(), IsDiners()
 
    INPUT:     cc - a string representing a credit card number

    RETURNS:  true, if the credit card number is a valid Diner's
		    Club number.
		    
	      false, otherwise
    ================================================================ */

function IsDinersClub(cc)
{
  firstdig = cc.substring(0,1);
  seconddig = cc.substring(1,2);
  if ((cc.length == 14) && (firstdig == 3) &&
      ((seconddig == 0) || (seconddig == 6) || (seconddig == 8)))
    return IsCC(cc);
  return false;
}

function IsDC(cc)
{
  return IsDinersClub(cc);
}

function IsDiners(cc)
{
  return IsDinersClub(cc);
}


/*  ================================================================
    FUNCTION:  IsCarteBlanche(), a.k.a. IsCB()
 
    INPUT:     cc - a string representing a credit card number

    RETURNS:  true, if the credit card number is a valid Carte
		    Blanche number.
		    
	      false, otherwise
    ================================================================ */

function IsCarteBlanche(cc)
{
  return IsDinersClub(cc);
}

function IsCB(cc)
{
  return IsCarteBlanche(cc);
}


/*  ================================================================
    FUNCTION:  IsDiscover()
 
    INPUT:     cc - a string representing a credit card number

    RETURNS:  true, if the credit card number is a valid Discover
		    card number.
		    
	      false, otherwise
    ================================================================ */

function IsDiscover(cc)
{
  first4digs = cc.substring(0,4);
  if ((cc.length == 16) && (first4digs == "6011"))
    return IsCC(cc);
  return false;

} // END FUNCTION IsDiscover()





/*  ================================================================
    FUNCTION:  IsenRoute(), a.k.a. IsEnRoute()
 
    INPUT:     cc - a string representing a credit card number

    RETURNS:  true, if the credit card number is a valid enRoute
		    card number.
		    
	      false, otherwise
    ================================================================ */

function IsenRoute(cc)
{
  first4digs = cc.substring(0,4);
  if ((cc.length == 15) &&
      ((first4digs == "2014") ||
       (first4digs == "2149")))
    return IsCC(cc);
  return false;
}

function IsEnRoute(cc)
{
  return IsenRoute(cc);
}


/*  ================================================================
    FUNCTION:  IsJCB()
 
    INPUT:     cc - a string representing a credit card number

    RETURNS:  true, if the credit card number is a valid JCB
		    card number.
		    
	      false, otherwise
    ================================================================ */

function IsJCB(cc)
{
  first4digs = cc.substring(0,4);
  if ((cc.length == 16) &&
      ((first4digs == "3088") ||
       (first4digs == "3096") ||
       (first4digs == "3112") ||
       (first4digs == "3158") ||
       (first4digs == "3337") ||
       (first4digs == "3528")))
    return IsCC(cc);
  return false;

} // END FUNCTION IsJCB()



/*  ================================================================
    FUNCTION:  IsAnyCard()
 
    INPUT:     cc - a string representing a credit card number

    RETURNS:  true, if the credit card number is any valid credit
		    card number for any of the accepted card types.
		    
	      false, otherwise
    ================================================================ */

function IsAnyCard(cc)
{
  if (!IsCC(cc))
    return false;
  if (!IsMC(cc) && !IsVISA(cc) && !IsAmEx(cc) && !IsDiners(cc) &&
      !IsDiscover(cc) && !IsEnRoute(cc) && !IsJCB(cc)) {
    return false;
  }
  return true;

} // END FUNCTION IsAnyCard()



/*  ================================================================
    FUNCTION:  IsCardMatch()
 
    INPUT:    cardType - a string representing the credit card type
	      cardNumber - a string representing a credit card number

    RETURNS:  true, if the credit card number is valid for the particular
	      credit card type given in "cardType".
		    
	      false, otherwise
    ================================================================ */

function IsCardMatch (cardType, cardNumber)
{

	cardType = cardType.toUpperCase();
	var doesMatch = true;

	if ((cardType == "VISA") && (!IsVisa(cardNumber)))
		doesMatch = false;
	if ((cardType == "MASTERCARD") && (!IsMasterCard(cardNumber)))
		doesMatch = false;
	if ((cardType == "AMEX") && (!IsAmEx(cardNumber)))
		doesMatch = false;
	if ((cardType == "DISCOVER") && (!IsDiscover(cardNumber)))
		doesMatch = false;
	if ((cardType == "JCB") && (!IsJCB(cardNumber)))
		doesMatch = false;
	if ((cardType == "DINERS") && (!IsDiners(cardNumber)))
		doesMatch = false;
	if ((cardType == "CARTEBLANCHE") && (!IsCarteBlanche(cardNumber)))
		doesMatch = false;

	return doesMatch;

}  // END FUNCTION CardMatch()

