// === Starter Application 1.0.0 === //

/* ======================================================

FUNCTION:  AuthToCapturing()

INPUT:  batch 	   - the batch object associated with the transaction
		     to be captured.
      	purchaseid - the unique id of the transaction to capture
	all 	   - a boolean flag that tells whether or not to
		   capture ALL "AUTHORIZED" transactions.  If 
		   TRUE, all "AUTHORIZED" transactions will be
		   changed to "CAPTURING".  If FALSE, only the
		   
RETURN:    none

DESCRIPTION:  This function essentially prepares an AUTHORIZED transaction
	for capturing, but it does NOT perform the actual capture.  The 
	function can do one of two things, depending on the
	boolean parameter "all".  

		If all is TRUE, then every transaction in the database 
	that has a status of AUTHORIZED has its status changed to 
	CAPTURING and is prepared for capturing.  If "all" is
	FALSE, then only the transaction indicated by "purchaseid" 
	has it's status changed to CAPTURING.

	      The "all" parameter is usually only set to TRUE when the
	user performs an auto-capture just before settling a batch.
	      

STATE CHANGES:  

	Changes the state of a transaction from AUTHORIZED to CAPTURING.
  
====================================================== */


//  You MUST call "GetCurrentBatch()" before calling this function
// to make sure you have the current, open batch.
//
// If "all" is true, that indicates that ALL AUTHORIZED transactions
// are to be updated to "CAPTURING"
//
function AuthToCapturing(batch, purchaseid, all)
{
   database.beginTransaction();

   //
   //  Make sure the batch we were passed is valid and status is OPENED.
   //
   batchCursor = database.cursor("select ID from LP_BATCH where ID = " + batch.ID + " AND status='OPENED'");
     
   if (!batchCursor.next())
   {
		
      batchCursor.close();
      database.rollbackTransaction();
      PrintError("No OPENED batch found.", "No batch with ID " + batch.ID +
                 " and status OPENED was found in the database.  Either the batch " + 
		 " is currently SETTLING or there is no batch with a matching ID in the database.  " + 
		"No captures or credits may take place while a batch is SETTLING.");

   }
   batchCursor.close();

   if (all)
   {
    	//
   	// Update all authorized transactions to indicate "CAPTURING"
	//
	   error = database.execute("update LP_PURCHASE set status = 'CAPTURING', batchID = " + batch.ID + " where status = 'AUTHORIZED'");
   }
   else 
   {
 	//
	// Update the indicated authorized transaction to indicate "CAPTURING"
	//
   	error = database.execute("update LP_PURCHASE set status = 'CAPTURING', batchID = " + batch.ID + " where id = " + purchaseid + " AND status = 'AUTHORIZED'");
   }
   

   if (error)
   {
      database.rollbackTransaction();
	if ((error == 5) || (error == 7))
		PrintError("Database Error.", "Could not update authorization in LP_PURCHASE table to CAPTURING  " +
			"Error code " + error + ": " + database.majorErrorMessage());
	else if (error)
		PrintError("Database Error.", "Could not update authorization in LP_PURCHASE table to CAPTURING  " +
				"Error code " + error + " returned from the database.");
   }
     
   database.commitTransaction();

} // END AuthToCapturing()


/* ======================================================

FUNCTION:  Capture()

INPUT:  id	   - the id of the purchase as it relates to the LP_PURCHASE table.
 	batch 	   - the batch object associated with the transaction
		     to be captured.
	merchant   - the merchant object, previously created
	terminal   - the terminal object, previously created
	processor  - the processor object, previously created
	      
	merchantReference - the merchant reference string for the capture.
	all	   - boolean variable.  If TRUE, capture *all* transactions in the "CAPTURING" state.
			Otherwise, capture only the transaction indcated by "id" parameter.
		   
RETURN:    TRUE if the capture was successful, FALSE if it was not.

DESCRIPTION:  This function attempts to capture all transactions that
	are currently in the "CAPTURING" state.  AuthToCapturing() should
	be called before calling this function to capture new transactions.
	This function also may be used without or in conjunction with
 	AuthToCapturing() to re-try captures that got "stuck" in the 
	"CAPTURING" state.
		If the capture succeeds, the database is updated with
	the resulting information.  If it fails, the database record
	remains in the state "CAPTURING."  

STATE CHANGES:  

	Changes the state of a transaction from CAPTURING to CAPTURED.

SPECIAL NOTES:  This function takes advantage of and relies upon idempotency (see LivePayment documentation)

  
====================================================== */

function Capture(id, batch, merchant, terminal, processor, merchantReference, all)
{
    var returnval = false;
	
      database.beginTransaction();

      if (all)
	purchaseCursor = database.cursor("select * from LP_PURCHASE where status = 'CAPTURING'", true);
      else
	purchaseCursor = database.cursor("select * from LP_PURCHASE where ID = " +
			 id + " AND status = 'CAPTURING'", true);

      while (purchaseCursor.next())
      {
	var eventID = GetNextEventID(); 

	var slipID = purchaseCursor.slipID;
        
         //
         // construct slip object
         //
         slipCursor = database.cursor("select * from LP_SLIP where ID = " + slipID);
         if (slipCursor.next())
         {
            asciiDER = slipCursor.slipder; 
            slip = new Slip(asciiDER);
            if (slip.bad())
            {
               slipCursor.close();
               purchaseCursor.close();
               database.rollbackTransaction();
               PrintFmtError("Failed to construct slip object from DER.",
                             slip.getStatusMessage());
            }
         } 
         else
         {
            slipCursor.close();
            purchaseCursor.close();
            database.rollbackTransaction();
            PrintError("Slip not found", "No slip with ID " + slipID + " in database.  Could not perform capture with the slip.");
         }
         slipCursor.close();
   

         //
         // construct payevent object
         //

	 if (merchantReference == null)
		merchantReference = slip.merchantReference;

         payevent = new PayEvent(merchantReference);
         payevent.amount = purchaseCursor.amount; 
         payevent.authCode = purchaseCursor.authCode;
         payevent.paySvcData = purchaseCursor.paySvcData;
         payevent.avsResp = purchaseCursor.avsResp;
         payevent.eventID = eventID;
         
         //
         // Perform the capture
         //
         if (processor.capture(terminal, merchant, payevent, slip, batch))
         {
            purchaseCursor.eventID = eventID;
            purchaseCursor.eventTime = payevent.eventTime;
            purchaseCursor.status = "CAPTURED";
            if ((error = purchaseCursor.updateRow("LP_PURCHASE")))  
            {
               purchaseCursor.close();
               database.rollbackTransaction();
	if ((error == 5) || (error == 7))
		PrintError("Database Error.", "Could not update LP_PURCHASE table to CAPTURED after successful capture  " +
			"Error code " + error + ": " + database.majorErrorMessage());
	else if (error)
		PrintError("Database Error.", "Could not update LP_PURCHASE table to CAPTURED after successful capture  " +
				"Error code " + error + " returned from the database.");
            }  // END IF
	    returnval = true;	// At this point, the capture was successful.
         }
         else
         {
            purchaseCursor.close();
            database.rollbackTransaction();
            PrintFmtError("Failed to capture.", processor.getStatusMessage());
         }  // END IF-ELSE from "processor.capture(...)"
    
      }  // END WHILE

      purchaseCursor.close();
      database.commitTransaction();

   return returnval;

}  // END Capture()
