// === Starter Application 1.0.0 === //

//
// COMMON FUNCTIONS used in more than one page
//


/*  ================================================================
    FUNCTION:  	SaleObject()
 
    INPUT:     	amount - the amount of the sale, as a non-zero, non-negative
			integer (e.g. must be a value in cents, for US currency).
	  	currency - currency for the transaction (3-letter code)
		orderDesc - a description of the order
		cardHolderName - the first and last name of the credit card holder, 
				 as it appears on the card.
		cardType - the type of credit card (e.g. Visa, MasterCard, etc.)
		cardNumber - credit card number (must contain only digits 0-9, 
						no spaces or dashes)
		cardExpDate - card expiration date (must be of the form YYYYMM)
		address - billing address for the card
		zip - billing zip/postal code for the card
		merchantReference - a string for future reference (e.g. an invoice number)

    RETURNS:   	a new instance of the object

    DESCRIPTION:  This object constructor is used to contain and pass
	values associated with a purchase to the Authorize() function or for a manual 
	credit.   It is intended to be called from a form handler (e.g. auth.html)
	which formats user input before calling Authorize() or GenerateSlip().

    SPECIAL NOTES:  This function is a JavaScript object constructor

    ================================================================ */

function SaleObject ( amount, currency, orderDesc, cardHolderName, cardType, cardNumber, cardExpDate, address, zip, merchantReference )
{
	this.amount = amount;
	this.currency = currency;
	this.orderDesc = orderDesc;
	this.cardHolderName = cardHolderName;
	this.cardType = cardType;
	this.cardNumber = cardNumber;
	this.cardExpDate = cardExpDate;
	this.address = address;
	this.zip = zip;
	this.merchantReference = merchantReference;

} // END OBJECT CONSTRUCTOR SaleObject



/* ======================================================

FUNCTION:  GenerateSlip()

INPUT:    slipData - the slipData object (of type SaleObject) contains the 
		    following properties used in this procedure:

		amount - amount of the transaction
		currency - type of currency involved
		orderDesc - a description of the order
		cardNumber - credit card number
		cardExpDate - expiration date
		cardType - credit card type
		address - address of card holder
		zip - zipcode of card holder

		merchantReference - a reference number that the merchant
			may assign to the transaction
 
          processor - the processor object, previously created

RETURN:   the constructed slip object

DESCRIPTION:  Constructs a slip object to be used in performing
	an authorization or a manual credit.  Also stores the slip 
	information in the database for future reference.  
====================================================== */

var slipID = 0;
function GenerateSlip(slipData, processor)
{

  var asciiDER = "";

  // 
  // Create an initial slip object
  //
  slip                   = new Slip(slipData.cardNumber, slipData.cardExpDate, 
                                    slipData.amount, slipData.currency);

  slip.billingStreet     = slipData.address;
  slip.billingZip        = slipData.zip;
  slip.cardType          = slipData.cardType;

  // if no merchantReference is supplied, fill it in for
  // the merchant.
  //

  if ((slipData.merchantReference == null) || (slipData.merchantReference == "null"))
	slipData.merchantReference = GenerateMerchantReference();
  slip.merchantReference = slipData.merchantReference;

  slip.appendOrderDesc(slipData.orderDesc);

  // 
  // Encode the slip
  //
  if (!slip.encode(processor))
  {
     PrintFmtError("Failed to encode slip",
                   slip.getStatusMessage());
  }

  asciiDER = slip.getDER();

  //
  // Save the slip in database
  //
  var nextSlipID = GetNextSlipID();
  database.beginTransaction();

  cursor = database.cursor("select * from LP_SLIP", true);
  cursor.ID = nextSlipID;
  cursor.slipDER = asciiDER;
  if ((error = cursor.insertRow("LP_SLIP")))
  {
     cursor.close();
     database.rollbackTransaction();
	if ((error == 5) || (error == 7))
		PrintError("Database Error.", "Failed to insert slip into database.  Could not complete the transaction.  " +
			"Error code " + error + ": " + database.majorErrorMessage());
	else if (error)
		PrintError("Database Error.", "Failed to insert slip into database.  Could not complete the transaction.  " +
				"Error code " + error + " returned from the database.");
  }
  cursor.close();
 
  database.commitTransaction();

  //
  // Recreate the slip from the extracted original slip (in DER format)
  //
  slip2 = new Slip(asciiDER);
  if (slip2.bad())
  {
     PrintFmtError("Failed to construct slip object from DER.",
                   slip2.getStatusMessage());
  }
  slip2.initMerchantOrderDesc(slipData.amount, slipData.currency);
  slip2.appendMerchantOrderDesc(slipData.orderDesc);

  slipID = nextSlipID;	// set GLOBAL var slipID
  return (slip2);

} // END GenerateSlip()



/* ======================================================

FUNCTION:  GetCurrentBatch()

INPUT:     merchant,
	   terminal,
	   processor - objects required for obtaining a new batch
		number from acquirer (previously created).

RETURN:   the current batch number according to the acquirer.  

DESCRIPTION:  This function should not be confused with the "getCurrentBatch"
	method of the PROCESSOR object.  This function uses that method
	in certain circumstances, but it goes beyond the functionality
	of that method and deals with our database as well.  
	      This function first checks to see if we already have a
	batch that is currently in the OPENED or SETTLING state/status.
	If so, this function does not bother to send a request across
	the Internet to the acquirer to obtain the current batch number
	because we already have that batch number stored locally in
	our database.  So instead, the function performs a much faster
	operation by pulling it from the local database and returning
	it to the user.  
		If no batch is currently OPENED or SETTLING, then we
	will have to start a new batch, so we will need the next batch number
	from the acquirer.  In this case, the function does send a request	
	across the Internet to the acquirer and obtains the new batch number.
	In addition, the function creates a new record in the LP_BATCH 
	table for the new batch, sets it's status to "OPENED" and then
	returns the batch object (containing the new ID and batch number)
	to the caller.
====================================================== */

function GetCurrentBatch(merchant, terminal, processor)
{
   database.beginTransaction();

   batchCursor = database.cursor("select * from LP_BATCH where status = 'OPENED' OR status = 'SETTLING'", true);

   if (batchCursor.next())
   {
      //
      // use current batch
      //
      batch = new Batch(batchCursor.batchNumber);

      // 
      // ID and status are not built-in properties of the batch object.
      // They are created here to remember the ID and status for future use
      //
      batch.ID = batchCursor.ID; 
      batch.status = batchCursor.status;
   }
   else
   {
      //
      // get new batch
      //
      if ((batch = processor.getCurrentBatch(terminal, merchant)) == null)
      {
         batchCursor.close();
         database.rollbackTransaction();
         PrintFmtError("Get current batch failed.",
                       proc.getStatusMessage());
      }
      else
      {
         // 
         // ID and status are not built-in properties of the batch object.
         // They are created here to remember the ID and status for future use
         //
         batch.ID = batchCursor.ID = GetNextBatchID();
         batch.status = batchCursor.status = "OPENED";
         batchCursor.batchNumber = batch.batchNumber;

	 // Currenly, the "currency" type is set in "start.html"
	 //
	 batchCursor.currency = project.currency;
	 
	 server.lock();
	   server.eventID = 0;	// Reset Event ID for a new batch
	 server.unlock();

         if ((error = batchCursor.insertRow("LP_BATCH")))
         {
            batchCursor.close();
            database.rollbackTransaction();
	    if ((error == 5) || (error == 7))
		PrintError("Database Error.", "Failed to insert record for new batch into database.  " +
			"Error code " + error + ": " + database.majorErrorMessage());
	    else if (error)
		PrintError("Database Error.", "Failed to insert record for new batch into database.  " +
				"Error code " + error + " returned from the database.");
 
        }
      }
   }
   batchCursor.close();

   database.commitTransaction();
   return (batch);

} // END GetCurrentBatch()



/* ======================================================

FUNCTION:  GenerateMerchantReference()

INPUT:     none.

RETURN:    Create a merchantReference number based on the purchase
	  ID.  This function is called when the merchant fails
	  to supply a merchantReference value of their own.

====================================================== */

function GenerateMerchantReference()
{

  var acquirer = project.acquirer;

  server.lock();
    var nextID = server.lastPurchaseID;
  server.unlock();

  // certain acquirers have restrictions on the length of the
  // merchantReference string - so clip of characters that
  // extend beyond the limit for a given acquirer.
  //
  nextID = MakeValidMerchantReference(acquirer, nextID);
	
  return(nextID);

}  // END FUNCTION GenerateMerchantReference()



/* ======================================================

FUNCTION:  MakeValidMerchantReference()

INPUT:     acquirer - a string representing the acquirer

	   merchantReference - a string representing a merchantReference
		identifier to be trimmed, if necessary

RETURN:    a merchantReference string that is to be trimmed, if 
	   necessary, to the acceptable number of characters for 
	   the given acquirer

====================================================== */

function MakeValidMerchantReference( acquirer, merchantReference )
{
   var maxchars;

   if (acquirer == "FDC")
	maxchars = 10;

   //
   //  You may add different acquirer restrictions here
   //
   // e.g.  else if (acquirer = ...)

   // Trim down merchantReference to an acceptable number of chars
   //
   merchantReference = merchantReference.substring(0, (maxchars-1));

   return(merchantReference);

}  // END FUNCTION MakeValidMerchantReference()






