#ifndef __FIST_CRYPTFS_H_
#define __FIST_CRYPTFS_H_

#include <linux/ioctl.h>
#include "blowfish.h"

#define FIST_HASH_SHIFT  4
#define FIST_HASH_SIZE  (1 << FIST_HASH_SHIFT)
#define FIST_HASH_MASK  (FIST_HASH_SIZE - 1)

typedef struct fist_key {
  int uid;
  BF_KEY key;
  struct fist_key *next;
} fist_key_t;

/*
 * cryptfs inode data in memory
 */
struct cryptfs_inode_info {
  struct inode *hidden_inode;
};

/*
 * cryptfs super-block data in memory
 */
struct cryptfs_sb_info {
  struct super_block *hidden_super;
  struct inode *s_rootinode;
  fist_key_t *fwi_user[FIST_HASH_SIZE];
};

#ifdef FIST_DEBUG

extern void fist_dprint_internal(int level, char *str,...);
extern int fist_get_debug_value(void);
extern int fist_set_debug_value(int val);
#define print_entry_location() \
	fist_dprint(4, "IN:  %s %s:%d\n", __FUNCTION__, __FILE__, __LINE__)
#define print_exit_location() \
	fist_dprint(4, "OUT: %s %s:%d\n", __FUNCTION__, __FILE__, __LINE__)
#define print_exit_status(status) \
	fist_dprint(4, "OUT: %s %s:%d, STATUS: %d\n", __FUNCTION__, __FILE__, __LINE__, status)
#define dprintk(format, args...) printk(KERN_DEBUG format, ##args)
#define fist_dprint(level, str, args...) \
	fist_dprint_internal(level, KERN_DEBUG str, ## args)
#define ASSERT(EX) ((void)((EX) || printk(KERN_CRIT "ASSERTION FAILED: %s at %s:%d\n", #EX, __FILE__, __LINE__)))

#else /* not FIST_DEBUG */

#define fist_dprint(level, format, args...)
#define dprintk(format, args...)
#define print_entry_location()
#define print_exit_location()
#define print_exit_status(status)
#define fist_print_vnode(a, b)
#define fist_print_vfs(a, b)
#define fist_print_uap(a, b)
#define ASSERT(foo)

#endif /* not FIST_DEBUG */

#define FIST_IOCTL_GET_DEBUG_VALUE	_IOR(0x15, 1,  int)
#define FIST_IOCTL_SET_DEBUG_VALUE	_IOW(0x15, 2,  int)
#define FIST_IOCTL_GET_NUM_VMA		_IOW(0x15, 3,  int)
#define FIST_IOCTL_SET_KEY		_IOW(0x15, 10, char [16])

     /* 13 bits -> 8192 bytes for the block size */
#define CRYPT_BLOCK_SHIFT	12
#define CRYPT_BLOCK_SIZE	(1 << CRYPT_BLOCK_SHIFT)
#define CRYPT_BLOCK_MASK	(CRYPT_BLOCK_SIZE - 1)

     /* VMArea TO Hidden VMArea */
#define vmatohvma(vma) ((struct vm_area_struct *)((vma)->vm_private_data))
     /* File TO Hidden File */
#define ftohf(file) ((struct file *)((file)->private_data))
     /* Inode TO Private Data */
#define itopd(ino) ((struct cryptfs_inode_info *)(ino)->u.generic_ip)
     /* Inode TO Hidden Inode */
#define itohi(ino) (itopd(ino)->hidden_inode)
     /* Superblock TO Private Data */
#define stopd(super) ((struct cryptfs_sb_info *)(super)->u.generic_sbp)
     /* Superblock TO Hidden Superblock */
#define stohs(super) (stopd(super)->hidden_super)

  /*
   * Flags for crypt_{en,de}code_filename
   * DO_DOTS means the special entries . and .. should be encoded (for symlink)
   * SKIP_DOTS means they should be preserved intact
   */
#define DO_DOTS   0
#define SKIP_DOTS 1

#define bcopy(s, d, l) memcpy((d), (s), (l))
#define bzero(s, c)    memset((s), 0, (c))
#define MIN(a,b) (((a) < (b)) ? (a) : (b))
#define MAX(a,b) (((a) > (b)) ? (a) : (b))

void *fist_get_userpass(const struct super_block *super);
void fist_set_userpass(const struct super_block *super, unsigned char *key);
void fist_free_userpass(const struct super_block *super);

int crypt_encode_block(char *func, int line, char *from, char *to, int len, void *key);
int crypt_decode_block(char *func, int line, char *from, char *to, int len, void *key);
int fist_crypt_encodefilename(const char *name, int length, char **uuencoded_name, void *key, int skip_dots);
int fist_crypt_decodefilename(const char *name, int length, char **decrypted_name, void *key, int skip_dots);

#endif	/* __FIST_CRYPTFS_H_ */

/*
 * Local variables:
 * c-basic-offset: 4
 * End:
 */
