/*
 *  $Id: vm_area.c,v 1.2 1999/01/27 15:29:37 ezk Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#include <fist.h>
#include <cryptfs.h>


STATIC void
cryptfs_vm_open(vm_area_t *vma)
{
    vm_area_t *hidden_vma = vmatohvma(vma), *hidden_vma2;
    file_t *file = vma->vm_file;
    file_t *hidden_file = ftohf(file);

    print_entry_location();
    fist_dprint(6, "VM_OPEN: file 0x%x, hidden_file 0x%x, file->f_count %d, hidden_file->f_count %d\n",
		(int) file, (int) hidden_file,
		(int) file->f_count, (int) hidden_file->f_count);

    if (hidden_vma->vm_ops->open)
	hidden_vma->vm_ops->open(hidden_vma);
    hidden_file->f_count++;

    /* we need to duplicate the private data */
    hidden_vma2 = kmalloc(sizeof(vm_area_t), GFP_KERNEL);
    if (!hidden_vma2) {
	printk("VM_OPEN: Out of memory\n");
	goto out;
    }
    memcpy(hidden_vma2, hidden_vma, sizeof(vm_area_t));
    vmatohvma(vma) = hidden_vma2;

 out:
    fist_dprint(6, "VM_OPEN2: file 0x%x, hidden_file 0x%x, file->f_count %d, hidden_file->f_count %d\n",
		(int) file, (int) hidden_file,
		(int) file->f_count, (int) hidden_file->f_count);
    print_exit_location();
}


STATIC void
cryptfs_vm_close(vm_area_t *vma)
{
    vm_area_t *hidden_vma = vmatohvma(vma);
    file_t *file = vma->vm_file;
    file_t *hidden_file = ftohf(file);

    print_entry_location();
    fist_dprint(6, "VM_CLOSE1: file 0x%x, hidden_file 0x%x, file->f_count %d, hidden_file->f_count %d\n",
		(int) file, (int) hidden_file,
		(int) file->f_count, (int) hidden_file->f_count);

    ASSERT(hidden_vma != NULL);

    if (hidden_vma->vm_ops->close)
	hidden_vma->vm_ops->close(hidden_vma);
    fput(hidden_file);
    kfree_s(hidden_vma, sizeof(vm_area_t));
    vmatohvma(vma) = NULL;

    fist_dprint(6, "VM_CLOSE2: file 0x%x, hidden_file 0x%x, file->f_count %d, hidden_file->f_count %d\n",
		(int) file, (int) hidden_file,
		(int) file->f_count, (int) hidden_file->f_count);
    print_exit_location();
}


STATIC void
cryptfs_vm_shared_unmap(vm_area_t *vma, unsigned long start, size_t len)
{
    vm_area_t *hidden_vma = vmatohvma(vma);
    file_t *file = vma->vm_file;
    file_t *hidden_file = ftohf(file);

    print_entry_location();
    fist_dprint(6, "VM_S_UNMAP1: file 0x%x, hidden_file 0x%x, file->f_count %d, hidden_file->f_count %d\n",
		(int) file, (int) hidden_file,
		(int) file->f_count, (int) hidden_file->f_count);

    /*
     * call sync (filemap_sync) because the default filemap_unmap
     * calls it too.
     */
    filemap_sync(vma, start, len, MS_ASYNC);

    if (hidden_vma->vm_ops->unmap)
	hidden_vma->vm_ops->unmap(hidden_vma, start, len);

    fist_dprint(6, "VM_S_UNMAP2: file 0x%x, hidden_file 0x%x, file->f_count %d, hidden_file->f_count %d\n",
		(int) file, (int) hidden_file,
		(int) file->f_count, (int) hidden_file->f_count);
    print_exit_location();
}


/*
 * Shared mappings need to be able to do the right thing at
 * close/unmap/sync. They will also use the private file as
 * backing-store for swapping..
 */
struct vm_operations_struct cryptfs_shared_vmops = {
    cryptfs_vm_open,		/* open */
    cryptfs_vm_close,		/* close */
    cryptfs_vm_shared_unmap,	/* unmap */
    NULL,			/* protect */
    filemap_sync,		/* sync */
    NULL,			/* advise */
    filemap_nopage,		/* nopage */
    NULL,			/* wppage */
    filemap_swapout,		/* swapout */
    NULL			/* swapin */
};

/*
 * Private mappings just need to be able to load in the map.
 *
 * (This is actually used for shared mappings as well, if we
 * know they can't ever get write permissions..)
 */
struct vm_operations_struct cryptfs_private_vmops = {
    NULL,			/* open */
    NULL,			/* close */
    NULL,			/* unmap */
    NULL,			/* protect */
    NULL,			/* sync */
    NULL,			/* advise */
    filemap_nopage,		/* nopage */
    NULL,			/* wppage */
    NULL,			/* swapout */
    NULL			/* swapin */
};

/*
 * Local variables:
 * c-basic-offset: 4
 * End:
 */
