/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)skip_keystore.c	1.19 95/11/22 Sun Microsystems"

/*
 * System includes
 */
#ifdef SYSV
#include <skip_sysv.h>
#endif
#ifdef SUNOS
#include <skip_sunos.h>
#endif
#if !defined(SYSV) && !defined(SUNOS)
#include <skip_os.h>
#endif

/*
 * SKIP includes
 */
#include <bdcmod.h>
#include <skip_types.h>
#include <skip_proto.h>
#include <skip_keymgrmsgs.h>
#include <skip_crypt.h>
#include <skip_key.h>
#include <skip_ioctl.h>

/*
 * SKIP bulk data cryptor.  Implements skip_encrypt()/skip_decrypt() for
 * kernel clients.
 */


/*
 * hash table information
 */
static MUTEX_T		*skip_encrypt_hashlocks, *skip_decrypt_hashlocks;
static skip_keycb_t	**skip_encrypt_hashtable, **skip_decrypt_hashtable;

/*
 * max number of packets to hold while waiting for a key, max wait time (secs)
 */
int			skip_key_max_pkts = SKIP_KEY_MAX_PKTS;
int			skip_key_max_km_wait = SKIP_KEY_MAX_KM_WAIT;

/*
 * maximum number of bytes to transmit before changing key
 */
int			skip_key_max_bytes = SKIP_KEY_MAX_BYTES;

/*
 * delete an unused key control block after this time (secs)
 */
int			skip_key_max_idle = SKIP_KEY_MAX_IDLE;

/*
 * maximum time to use an encrypt key before changing it
 */
int			skip_key_max_use = SKIP_KEY_MAX_USE;

/*
 * maximum number of cipher states to have open at one time
 */
static const unsigned long	skip_key_max_entries = SKIP_KEY_MAX_ENTRIES;

/*
 *  Ioctl stats interface
 */
ioctl_key_stat_t	skip_keys_stats = { 0 };
int			skip_params_debug = 0;

/*
 * Supported Kij algorithms by the key manager
 */
boolean_t	skip_kmgr_kij_alg[SKIP_MAXCRYPTORS] = { 0 };
boolean_t	skip_kmgr_kij_alg_v1[SKIP_MAXCRYPTORS] = { 0 };


/*
 * Forward declarations for static functions
 */
static int	skip_key_destroy(skip_keycb_t *, void *);
static skip_keycb_t *
		skip_encrypt_hash_locate(skip_param_t *, unsigned int); 
static skip_keycb_t *
		skip_decrypt_hash_locate(skip_param_t *, char *,
					int, unsigned int); 
static skip_keycb_t *
		skip_encrypt_hash_add(unsigned int);
static skip_keycb_t *
		skip_decrypt_hash_add(unsigned int);
static void
		skip_hash_remove(skip_keycb_t *, skip_keycb_t **,
					unsigned long *);
static void	queue_pkt(skip_keycb_t *, SKIP_BUF_T *, SKIP_BUF_T *,
					skip_client_t *, void *);



/* skip_key_initstore()
 *
 * Initialise the key store
 *
 * Returns: none
 */
void
skip_key_initstore()
{
	register int	i;

	skip_encrypt_hashtable = (skip_keycb_t **)
		MEM_ALLOC(sizeof (skip_keycb_t *) * SKIP_HASH_TABLESZ);
	if (skip_encrypt_hashtable == NULL) {
		PANIC("skip: no encrypt hash table memory");
	}

	skip_decrypt_hashtable = (skip_keycb_t **)
		MEM_ALLOC(sizeof (skip_keycb_t *) * SKIP_HASH_TABLESZ);
	if (skip_decrypt_hashtable == NULL) {
		PANIC("skip: no decrypt hash table memory");
	}

	skip_encrypt_hashlocks = (MUTEX_T *)
		MEM_ALLOC(sizeof (MUTEX_T) * SKIP_HASH_TABLESZ);
	if (skip_encrypt_hashlocks == NULL) {
		PANIC("skip: no encrypt lock table memory");
	}

	skip_decrypt_hashlocks = (MUTEX_T *)
		MEM_ALLOC(sizeof (MUTEX_T) * SKIP_HASH_TABLESZ);
	if (skip_decrypt_hashlocks == NULL) {
		PANIC("skip: no decrypt lock table memory");
	}

	for (i = 0; i < SKIP_HASH_TABLESZ; i++) {
		MUTEX_INIT(&skip_encrypt_hashlocks[i], "SKIP encrypt hash");
		MUTEX_INIT(&skip_decrypt_hashlocks[i], "SKIP decrypt hash");
	}
}

/* skip_key_uninitstore()
 *
 * Uninitialise the key store
 *
 * Returns: none
 */
void
skip_key_uninitstore()
{
	register int	i;

	skip_key_iterate(skip_key_destroy, NULL);

	for (i = 0; i < SKIP_HASH_TABLESZ; i++) {
		MUTEX_DESTROY(&skip_encrypt_hashlocks[i]);
		MUTEX_DESTROY(&skip_decrypt_hashlocks[i]);
	}

	MEM_FREE(skip_encrypt_hashlocks,
		sizeof (MUTEX_T) * SKIP_HASH_TABLESZ);
	skip_encrypt_hashlocks = NULL;

	MEM_FREE(skip_decrypt_hashlocks,
		sizeof (MUTEX_T) * SKIP_HASH_TABLESZ);
	skip_decrypt_hashlocks = NULL;

	MEM_FREE(skip_encrypt_hashtable, sizeof (skip_keycb_t *) * 
		SKIP_HASH_TABLESZ);
	skip_encrypt_hashtable = NULL;

	MEM_FREE(skip_decrypt_hashtable, sizeof (skip_keycb_t *) * 
		SKIP_HASH_TABLESZ);
	skip_decrypt_hashtable = NULL;
}

/* skip_open()
 *
 * Give a SKIP client access to the bulk data crypt service
 *
 * Returns: skip descriptor
 */
void *
skip_open(
	void *client_data,
	void (*encrypt_done)(),
	void (*decrypt_done)()
)
{
	register skip_client_t	*sd;

	if (encrypt_done == NULL) {
		PANIC("skip_open: NULL encrypt_done");
	}

	if (decrypt_done == NULL) {
		PANIC("skip_open: NULL decrypt_done");
	}

	sd = (skip_client_t *) MEM_ALLOC(sizeof(skip_client_t));
	if (sd == NULL) {
		return (NULL);
	}

	sd->client_data = client_data;
	sd->encrypt_done = encrypt_done;
	sd->decrypt_done = decrypt_done;

	return(sd);
}

/* skip_client_close()
 *
 * Remove all references to a SKIP client who is closing
 *
 * Returns: none
 */
static int
skip_client_close(skip_keycb_t *cp, void *sd)
{
	skip_queue_t		*callback;

	for (callback = cp->queue; callback; callback = callback->next) {
		if (callback->client == sd) {
			/*
			 * callback for this client - zap it
			 */
			return (1);
		}
	}
	return (0);
}

/* skip_close()
 *
 * Remove bulk data crypt user
 *
 * Returns: none
 */
void
skip_close(void *sd)
{
	skip_key_iterate(skip_client_close, sd);
	MEM_FREE(sd, sizeof(skip_client_t));
}

/* skip_encryptpkt()
 *
 * Encrypt a packet using the supplied key information.
 * The original packet is unchanged.
 *
 * Returns: 0 on success, non-zero on error
 */
static int
skip_encryptpkt(skip_keycb_t *cp, SKIP_BUF_T *in, SKIP_BUF_T *out)
{
	unsigned int		msize;
	register skip_cryptor_t	*alg;
	skip_param_t		*params;
	unsigned int		hdr_offset = 0;
	unsigned int		payload_offset;
	unsigned char		*ah_start;

	msize = BUFLEN(in);

	params = &cp->hdr.params;

	alg = GETCRYPTMOD(params->version, params->kp_alg);

	hdr_offset = skip_hdr_encode(&cp->hdr, BUFEND(out));
	BUFGROW(out, hdr_offset);

	ah_start = BUFEND(out);

	if (params->mac_alg) {

		SKIP_ESP_DEBUG("skip_encryptpkt: AH SKIP_V2 packet\n");

		return(EPROTO);
	}

	if (params->kp_alg) {

		SKIP_ESP_DEBUG("skip_encryptpkt: ESP SKIP_V2 packet\n");
		/*
		 * The packet must be encrypted...
		 */
		payload_offset = skip_iv_encode(&cp->hdr, BUFEND(out));
		BUFGROW(out, payload_offset);

		/*
		 * note how many bytes we have sent this key
		 */
		cp->obytes += msize;
	
		/*
		 * encrypt the packet...
		 */
		(*alg->encrypt)(cp->cs,
				(char *) BUFSTART(in),
				&msize,
				(char *) BUFEND(out),
				cp->hdr.mid.buf,
				cp->hdr.mid.len,
				&params->payload_type);
		if (msize) {
			/*
			 * set the encrypted packet's size and destination
			 */
			BUFGROW(out, msize);
			alg->encrypts++;
		} else {
			/*
			 * XXX something went wrong... trap to management
			 */
			if (alg->encrypterrs % 5 == 0) {
				SKIP_DEBUG1(
					"skip_encryptpkt: %s encryption failed",
					alg->name);
			}
			alg->encrypterrs++;
			return (EINVAL);
		}

	}
	return (0);
}

/* skip_update_encrypt_key()
 *
 * The key manager has responded with a key. Update the associated key block 
 * and encrypt any waiting packets.
 *
 * Returns: none
 */
void
skip_update_encrypt_key(struct Encrypted_key_rsp *er)
{
	register skip_keycb_t	*cp;
	register skip_queue_t	*callback = NULL, *nextcallback;
	register unsigned int	h;
	skip_param_t		*params;
	register skip_cryptor_t	*alg;

	SKIP_DEBUG("skip_update_encrypt_key: updating key control block\n");

	params = &(er->params);
	/*
	 * lock the corresponding entry in the hash table
	 */
	h = SKIP_ENCRYPT_HASH(params->r_mkeyid.buf);

	MUTEX_ENTER(&skip_encrypt_hashlocks[h]);

	cp = skip_encrypt_hash_locate(params, h);

	if (cp) {

		alg = GETCRYPTMOD(params->version, 
					cp->hdr.params.kp_alg);
		if (cp->cs) {
			/*
			 * close existing key handle
			 */
			(alg->close)(cp->cs);
		}

		/*
		 * initialise kp in the key control block
		 */
		cp->kp.len = er->kp_keylen;
		bcopy((caddr_t) er + er->kp_offset, (caddr_t) cp->kp.buf,
								er->kp_keylen);
		/*
		 * initialise the message indicator in the key control block
		 * to the initialisation vector supplied by the key manager 
		 */
		cp->hdr.mid.len = er->iv_len;
		bcopy((caddr_t) er + er->iv_offset, (caddr_t) cp->hdr.mid.buf,
								er->iv_len);
		/*
		 * initialise ekp in the key control block
		 */
		cp->hdr.ekp.len = er->ekp_keylen;
		bcopy((caddr_t) er + er->ekp_offset, (caddr_t) cp->hdr.ekp.buf,
								er->ekp_keylen);

		/*
		 * ESP/AH mode
		 */
		if (params->version == SKIP_V2) {
			cp->hdr.params.counter_n	= params->counter_n;
			cp->hdr.params.ttl		= params->ttl;	

			if (er->E_kp_keylen) {
				cp->E_kp.len = er->E_kp_keylen;
				bcopy((caddr_t) er + er->E_kp_offset,
							(caddr_t) cp->E_kp.buf,
							er->E_kp_keylen);
			}

			if (er->E_kp_keylen) {
				cp->A_kp.len = er->A_kp_keylen;
				bcopy((caddr_t) er + er->A_kp_offset,
							(caddr_t) cp->A_kp.buf,
							er->A_kp_keylen);
			}
		}
		
		/*
		 * Pre-compute the fixed part of SKIP header
		 */
		skip_hdr_init(cp);

		/*
		 * open new handle to crypt engine
		 */
		if (params->version != SKIP_V2) {
			SKIP_ESP_DEBUG("skip_update_encrypt_key: SKIP_V1\n");
			cp->cs = (alg->open)(
				cp->kp.buf, cp->kp.len,	
				(char *) er + er->iv_offset, er->iv_len);
		} else {
			SKIP_ESP_DEBUG1("skip_update_encrypt_key: SKIP_V2"
				" E_kp.len=%d\n", cp->E_kp.len);
			cp->cs = (alg->open)(
				cp->E_kp.buf, cp->E_kp.len,	
				(char *) er + er->iv_offset, er->iv_len);
		}

		if (cp->cs == NULL) {
			/*
			 * open of crypto algorithm failed
			 */
			SKIP_ESP_DEBUG1("skip_update_encrypt_key: crytpo "
			"algorithm %d open failed\n", cp->hdr.params.kp_alg);

			MUTEX_EXIT(&skip_encrypt_hashlocks[h]);
			return;
		}

		/*
		 * reset the bytes transmitted
		 */
		cp->max_obytes 	= skip_key_max_bytes;
		cp->obytes	= 0;
		cp->last_obytes	= 0;

		/*
		 * reset the encrypt key's time to live value
		 */
		if (params->version == SKIP_V2) {
			cp->ttl = skip_key_max_idle > params->ttl ? 
					params->ttl : skip_key_max_idle;
		} else {
			cp->ttl = skip_key_max_idle;
		}

		/*
		 * encrypt any packets which were waiting for the 
		 * key information. We queue them so that we can send them
		 * all in one go without holding any locks.
		 */
		SKIP_DEBUG("skip_update_encrypt_key: encrypting queued pkts");

		/*
		 * make a note of head of encrypted packet list
		 */

		callback = cp->queue;

		for (; cp->queue; cp->queue = cp->queue->next) {
			/*
			 * encrypt the packet
			 */
			cp->queue->rc = skip_encryptpkt(cp,
							cp->queue->in,
							cp->queue->out);
		}
	}
	MUTEX_EXIT(&skip_encrypt_hashlocks[h]);

	/*
	 *  send the encrypted packets
	 */
	for (; callback; callback = nextcallback) {

		nextcallback = callback->next;

		/*
		 * send encrypted packet on its way...
		 */
		(*callback->client->encrypt_done)(callback->client->client_data,
						callback->in,
						callback->out,
						callback->arg,
						callback->rc);
		MEM_FREE(callback, sizeof (*callback));
		SKIP_DEBUG(".");
	}
	SKIP_DEBUG("\n");
}

/* skip_encrypt()
 *
 * Find a key control block  given a key id and kp_alg then
 * encrypt the packet. "plain" contains the original packet
 * "encrypted" is the buffer to write the encrypted packet into
 *
 * Returns: none
 */
void
skip_encrypt(void *sd,
		SKIP_BUF_T	*plain,
		SKIP_BUF_T	*encrypted,
		skip_param_t	*skip_parms,
		void		*arg
)
{
	register skip_client_t	*client = sd;
	register int		rc;
	register skip_keycb_t	*cp;
	register unsigned int	h;

	skip_keys_stats.skip_key_lookups++;

	/*
	 * lock the corresponding entry in the hash table
	 */
	h = SKIP_ENCRYPT_HASH(skip_parms->r_mkeyid.buf);

	MUTEX_ENTER(&skip_encrypt_hashlocks[h]);

	cp = skip_encrypt_hash_locate(skip_parms, h);

	if (cp && cp->cs) {

		/*
		 * holding the key information - encrypt and go...
		 */
		rc = skip_encryptpkt(cp, plain, encrypted);

		MUTEX_EXIT(&skip_encrypt_hashlocks[h]);

		/*
		 * send encrypted packet on its way...
		 */
		(*client->encrypt_done)(client->client_data,
					plain,
					encrypted,
					arg,
					rc);
		return;
	}

	/*
	 * cp may or may not be NULL depending on whether we have
	 * already hit the key manager yet for this key
	 */
	if (cp == NULL) {
		cp = skip_encrypt_hash_add(h);
		if (cp == NULL) {
			MUTEX_EXIT(&skip_encrypt_hashlocks[h]);
			(*client->encrypt_done)(client->client_data,
						plain,
						encrypted,
						arg,
						ENOMEM);
			return;
		}

		cp->hash	= h;
		cp->max_obytes	= SKIP_IGNORE_VAL;
		bcopy((caddr_t) skip_parms,
			(caddr_t) &cp->hdr.params, sizeof(skip_param_t));

		/*
		 * this allows the key manager skip_key_max_km_wait seconds
		 * to respond, after which the request is deleted
		 */
		cp->ttl		= skip_key_max_km_wait;

		/*
		 * hit the key manager for the key
		 */
		skip_key_tellkm(SKIP_ENCRYPTED_PKT_KEY_REQ, cp);
	}
	/*
	 * queue the packets up to a defined maximum
	 */
	queue_pkt(cp, plain, encrypted, client, arg);
	MUTEX_EXIT(&skip_encrypt_hashlocks[h]);
}

/* skip_decryptpkt()
 *
 * decrypt a packet using the supplied key information. Returns the 
 * decrypted packet on sucess or NULL on failure. The original packet is
 * unchanged.
 */
static int
skip_decryptpkt(skip_keycb_t *cp,
		SKIP_BUF_T *in,
		SKIP_BUF_T *out,
		int hdr_offset
)
{
	register int		payload_offset, iv_size;
	register skip_param_t	*params;
	register skip_cryptor_t	*alg;
	unsigned int		msize;

	params = &cp->hdr.params;

	/*
	 * key was used - reset the time to live
	 */
	cp->ttl = skip_key_max_idle;

	alg = GETCRYPTMOD(params->version, params->kp_alg);

	/*
	 * Get the MID/IV value as well as the payload offset
	 */
	if (params->version != SKIP_V2) {
		iv_size = skip_iv_v1_decode(&cp->hdr,
						BUFSTART(in) + hdr_offset,
						BUFEND(in));
	} else {
		iv_size = skip_iv_esp_decode(&cp->hdr,
						BUFSTART(in) + hdr_offset,
						BUFEND(in));
	}

	if (iv_size < 0) {
		return(EPROTO);
	}

	payload_offset = hdr_offset + iv_size;

	SKIP_ESP_DEBUG2("skip_decryptpkt: hdr_offset=%d, payload_offset=%d\n",
						hdr_offset, payload_offset)

	msize = BUFLEN(in) - payload_offset;

	/*
	 * decrypt the packet
	 */
	(*alg->decrypt)(cp->cs, 
			(char *) BUFSTART(in) + payload_offset,
			&msize,
			(char *) BUFSTART(out),
			cp->hdr.mid.buf,
			cp->hdr.mid.len,
			&params->payload_type);
	if (msize) {
		BUFGROW(out, msize);
		alg->decrypts++;
	} else {
		/*
		 * XXX something went wrong... trap to management
		 */
		if (alg->decrypterrs % 1000 == 0) {
			SKIP_DEBUG1(
				"skip_decryptpkt: %s decryption failed",
				alg->name);
		}
		alg->decrypterrs++;
		return(EINVAL);
	}
	return (0);
}

/* skip_update_decrypt_key()
 *
 * The key manager has responded with a key. Update the associated key block 
 * and decrypt any waiting packets.
 *
 * Returns: none
 */
void
skip_update_decrypt_key(struct Decrypted_key_rsp *dr)
{
	register skip_keycb_t	*cp;
	register skip_queue_t	*callback = NULL, *nextcallback;
	register unsigned int	h;
	skip_param_t		*params;
	register skip_cryptor_t	*alg;

	params = &(dr->params);

	h = SKIP_DECRYPT_HASH((unsigned char *) dr + dr->ekp_offset);

	MUTEX_ENTER(&skip_decrypt_hashlocks[h]);

	cp = skip_decrypt_hash_locate(params,
			(caddr_t) dr + dr->ekp_offset, dr->ekp_keylen, h);
	if (cp) {
		/*
		 * found it...
		 */
		alg = GETCRYPTMOD(cp->hdr.params.version,
					cp->hdr.params.kp_alg);
		if (cp->cs) {
			/*
			 * close existing key handle
			 */
			(alg->close)(cp->cs);
		}

		/*
		 * initialise kp in the key control block
		 */
		cp->kp.len = dr->kp_keylen;
		bcopy((caddr_t) dr + dr->kp_offset, (caddr_t) cp->kp.buf,
								dr->kp_keylen);

		if (params->version == SKIP_V2) {
			/*
			 * IPSP mode
			 */
			cp->hdr.params.counter_n	= params->counter_n;
			cp->hdr.params.ttl		= params->ttl;	

			if (dr->E_kp_keylen) {
				cp->E_kp.len = dr->E_kp_keylen;
				bcopy((caddr_t) dr + dr->E_kp_offset,
							(caddr_t) cp->E_kp.buf,
							dr->E_kp_keylen);
			}

			if (dr->A_kp_keylen) {
				cp->A_kp.len = dr->A_kp_keylen;
				bcopy((caddr_t) dr + dr->A_kp_offset,
							(caddr_t) cp->A_kp.buf,
							dr->A_kp_keylen);
			}
			/*
			 * open the crypto algorithm for decrypt
			 */
			SKIP_ESP_DEBUG1("skip_update_decrypt_key: SKIP_V2"
				" E_kp.len=%d\n", cp->E_kp.len);
			cp->cs = (alg->open)(
				cp->E_kp.buf, cp->E_kp.len,	
				cp->hdr.mid.buf, cp->hdr.mid.len);
		} else {
			/*
			 * SKIP V1/SunScreen mode
			 */
			cp->cs = (alg->open)(
				cp->kp.buf, cp->kp.len,	
				cp->hdr.mid.buf, cp->hdr.mid.len);
		}

		if (cp->cs == NULL) {
			/*
			 * open failed
			 */
			SKIP_DEBUG1("skip_update_decrypt_key: crypto "
			"algorithm %d open failed\n", cp->hdr.params.kp_alg);

			MUTEX_EXIT(&skip_decrypt_hashlocks[h]);
			return;
		}

		/*
		 * initialise the key's time to live value
		 */
		cp->ttl = skip_key_max_idle;

		/*
		 * decrypt any packets which were waiting for the 
		 * key information. We queue them so that we can send them
		 * all in one go without holding the mutex.
		 */
		SKIP_DEBUG("skip_update_decrypt_key: decrypting queued pkts");

		callback = cp->queue;

		for (; cp->queue; cp->queue = cp->queue->next) {

			if (params->mac_alg) {
				SKIP_ESP_DEBUG("skip_update_decrypt_key: "
					"dropping AH packet\n");
				cp->queue->rc = EPROTO;
				continue;
			}
			
			if (params->kp_alg) {
				/*
				 * decrypt the packet
				 */
				cp->queue->rc = skip_decryptpkt(cp,
							cp->queue->in,
							cp->queue->out,
							cp->data_offset);
			}
		}
	}

	MUTEX_EXIT(&skip_decrypt_hashlocks[h]);

	/*
	 *  send the decrypted packets
	 */
	for (; callback; callback = nextcallback) {

		nextcallback = callback->next;

		/*
		 * send decrypted packet on its way...
		 */
		(*callback->client->decrypt_done)(
						callback->client->client_data,
						callback->in,
						callback->out,
						callback->arg,
						callback->rc);
		MEM_FREE(callback, sizeof(*callback));
		SKIP_DEBUG(".");
	}
	SKIP_DEBUG("\n");
}

/* skip_decrypt()
 *
 * Find a key control block and decrypt the packet. 
 *
 * Returns: none
 */
void
skip_decrypt(void * sd,
	     SKIP_BUF_T *encrypted,
	     SKIP_BUF_T *plain,
	     skip_hdr_t	*hdr,
	     int	data_offset,
	     void 	*arg)
{
	register skip_client_t	*client = sd;
	register skip_keycb_t	*cp;
	register int		rc;
	register unsigned int	h;

	skip_keys_stats.skip_key_lookups++;

	/*
	 * lock the corresponding entry in the hash table
	 */
	h = SKIP_DECRYPT_HASH(hdr->ekp.buf);

	SKIP_DEBUG1("skip_decrypt: version = %d\n", hdr->params.version);

	MUTEX_ENTER(&skip_decrypt_hashlocks[h]);

	cp = skip_decrypt_hash_locate(&hdr->params,
					hdr->ekp.buf, hdr->ekp.len, h);

	if (cp && cp->cs) {

		/*
		 * update header information
		 */
		bcopy((caddr_t) hdr, (caddr_t) &cp->hdr, sizeof(skip_hdr_t));
		cp->data_offset = data_offset;

		if (hdr->params.mac_alg) {

			MUTEX_EXIT(&skip_decrypt_hashlocks[h]);
			/*
			 * send decrypted packet on its way...
			 */
	
			(*client->decrypt_done)(client->client_data,
						encrypted,
						plain,
						arg,
						EPROTO);
		}

		/*
		 * holding valid key information - decrypt and go.
		 */
		if (hdr->params.kp_alg) {

			rc = skip_decryptpkt(cp, encrypted, plain, data_offset);

			MUTEX_EXIT(&skip_decrypt_hashlocks[h]);
	
			/*
			 * send decrypted packet on its way...
			 */
	
			(*client->decrypt_done)(client->client_data,
						encrypted,
						plain,
						arg,
						rc);
	
		}
		return;
	}

	/*
	 * not currently holding valid decrypt key information
	 */
	if (cp == NULL) {
		/*
		 * This is the first time we have had to decrypt a packet
		 * like this.  Make a new key control block.
		 */
		cp = skip_decrypt_hash_add(h);
		if (cp == NULL) {
			MUTEX_EXIT(&skip_decrypt_hashlocks[h]);
			/*
			 * memory allocation failure?
			 */
			(*client->decrypt_done)(
					client->client_data,
					encrypted,
					plain,
					arg,
					ENOMEM);
			return;
		}

		cp->hash	= h;
		cp->data_offset = data_offset;
		cp->max_obytes	= SKIP_IGNORE_VAL;
		bcopy((caddr_t) hdr, (caddr_t) &cp->hdr, sizeof(skip_hdr_t));

		/*
		 * this allows the key manager skip_key_max_km_wait seconds
		 * to respond, after which the request is deleted
		 */
		cp->ttl		= skip_key_max_km_wait;

		/*
		 * hit the key manager for the key
		 */
		skip_key_tellkm(SKIP_DECRYPTED_PKT_KEY_REQ, cp);
	}
	/*
	 * queue the packet up to a defined maximum
	 */
	queue_pkt(cp, encrypted, plain, client, arg);
	MUTEX_EXIT(&skip_decrypt_hashlocks[h]);
}

/* skip_key_check()
 *
 * When iterated over the key list, decides which keys need attention -
 * either they should be removed or they need updated.
 *
 * Returns: 1 if entry should be removed, 0 otherwise
 */
int
skip_key_check(skip_keycb_t *cp)
{
	if (cp->max_obytes != SKIP_IGNORE_VAL) {
		/*
		 * check to see if we need to change encryption key
		 */
		if ((cp->obytes >= (unsigned) cp->max_obytes) ||
			((cp->last_obytes != cp->obytes) &&
			 (cp->ttl <= skip_key_tick))) {
			/*
			 * time to change encryption keys. Ask for it
			 * now and give the key manager a short time to
			 * respond
			 */
			cp->max_obytes	= SKIP_IGNORE_VAL;
			cp->ttl		= skip_key_max_km_wait;
			skip_key_tellkm(SKIP_ENCRYPTED_PKT_KEY_REQ, cp);
			return (0);
		}
		cp->last_obytes = cp->obytes;
	}

	/*
	 * check to see if key has expired
	 */
	if ((cp->ttl -= skip_key_tick) <= 0) {
		return (1);
	}

	return (0);
}

/* skip_key_destroy()
 *
 * When iterated over the key list, destroys all keys.
 *
 * Returns: 1
 */
/*ARGSUSED*/
static int
skip_key_destroy(skip_keycb_t *cp, void *arg)
{
	return (1);
}

/* skip_key_iterate()
 *
 * apply a function to all entries. The function may specify the removal of
 * the current element by returning a non-zero return code.
 *
 * Returns: none
 */
void
skip_key_iterate(int (*f)(), void *arg)
{
	skip_keycb_t	*cp, *ncp;
	int		i, rc;

	for (i = 0; i < SKIP_HASH_TABLESZ; i++) {

		/*
		 * iterate over the encryption key control blocks
		 */
		MUTEX_ENTER(&skip_encrypt_hashlocks[i]);
		for (cp = skip_encrypt_hashtable[i]; cp; cp = ncp) {
			ncp = cp->next;

			rc = (*f)(cp, arg);

			if (rc) {
				skip_hash_remove(cp, skip_encrypt_hashtable,
				&skip_keys_stats.skip_encrypt_keys_active);
			}
		}
		MUTEX_EXIT(&skip_encrypt_hashlocks[i]);
	}
	for (i = 0; i < SKIP_HASH_TABLESZ; i++) {

		/*
		 * iterate over the decryption key control blocks
		 */
		MUTEX_ENTER(&skip_decrypt_hashlocks[i]);
		for (cp = skip_decrypt_hashtable[i]; cp; cp = ncp) {
			ncp = cp->next;

			rc = (*f)(cp);

			if (rc) {
				skip_hash_remove(cp, skip_decrypt_hashtable,
				&skip_keys_stats.skip_decrypt_keys_active);
			}
		}
		MUTEX_EXIT(&skip_decrypt_hashlocks[i]);
	}
}

/* queue_pkt()
 *
 * add a pair of {in, out} packets plus details of who to call back
 * to the key control block (up to the specified limit)
 */
static void
queue_pkt(skip_keycb_t *cp,
		SKIP_BUF_T *in,
		SKIP_BUF_T *out,
		skip_client_t *client,
		void *arg)
{
	register int			pktcnt;
	register skip_queue_t		*queue, *prevqueue;

	pktcnt = 0;
	prevqueue = NULL;
	for (queue = cp->queue; queue; queue = queue->next) {
		prevqueue = queue;
		pktcnt++;
	}

	if (pktcnt >= skip_key_max_pkts) {
		/*
		 * holding queue overflow
		 */
		/*
		 * XXX call client's error handler
		 */
		BUFFREE(in);
		BUFFREE(out);
		return;
	}

	queue = (skip_queue_t *) MEM_ALLOC(sizeof (*queue));

	if (queue == NULL) {
		/*
		 * XXX call client's error handler
		 */
		BUFFREE(in);
		BUFFREE(out);
		return;
	}

#ifdef KERNEL
	/*
	 * XXX 4.x only
	 * we have knowledge that arg is a pointer to a struct sockaddr which
	 * needs to be preserved across function calls
	 */
	if (arg) {
		queue->dst = * (struct sockaddr *) arg;
		arg = & queue->dst;
	}
#endif
	 
	queue->next	= NULL;
	queue->in	= in;
	queue->out	= out;
	queue->client	= client;
	queue->arg	= arg;

	if (prevqueue == NULL) {
		cp->queue = queue;
	} else {
		prevqueue->next = queue;
	}
}

/* skip_encrypt_hash_locate()
 *
 * Find a hash table entry for an encrypt key
 *
 * Returns: the entry or NULL if not found
 */
static skip_keycb_t *
skip_encrypt_hash_locate(skip_param_t *params, unsigned int hash)
{
	register skip_keycb_t		*cp;

	cp = skip_encrypt_hashtable[hash];

	while (cp) {
		/*
		 * check collision entries...
		 */
		if (PARAMSMATCH(params, &cp->hdr.params)) {
			/*
			 * found it...
			 */
			break;
		} else {
			skip_diffs("skip_encrypt_hash_locate",
					params, &cp->hdr.params);
		}
		cp = cp->next;
	}
	return (cp);
}

/* skip_encrypt_hash_add()
 *
 * Add a hash table entry for an encryt key
 *
 * Returns: the entry or NULL on error
 */
static skip_keycb_t *
skip_encrypt_hash_add(unsigned int hash)
{
	register skip_keycb_t	*cp;

	if (skip_keys_stats.skip_encrypt_keys_active == skip_key_max_entries) {
		return (NULL);
	}

        cp = (skip_keycb_t *) MEM_ALLOC(sizeof(*cp));

	if (cp == NULL) {
		return (NULL);
	}

	skip_keys_stats.skip_encrypt_keys_active++;

	if (skip_encrypt_hashtable[hash] == NULL) {
		skip_encrypt_hashtable[hash] = cp;
	} else {
		/*
		 * hash collision found - insert in collision chain...
		 */
		skip_keys_stats.skip_hash_collisions++;
		cp->next = skip_encrypt_hashtable[hash];
		skip_encrypt_hashtable[hash]->prev = cp; 
		skip_encrypt_hashtable[hash] = cp; 
	}
	return(cp);
}

/* skip_decrypt_hash_locate()
 *
 * Find a hash table entry for a decrypt key
 *
 * Returns: the entry or NULL if not found
 */
static skip_keycb_t *
skip_decrypt_hash_locate(skip_param_t *params, 
				char *ekpbuf,
				int ekplen,
				unsigned int hash
)
{
	register skip_keycb_t		*cp;

	cp = skip_decrypt_hashtable[hash];

	while (cp) {
		/*
		 * check collision entries...
		 */
		if (!bcmp(ekpbuf, cp->hdr.ekp.buf, ekplen) &&
				PARAMSMATCH(params, &cp->hdr.params)) {
			/*
			 * found it...
			 */
			break;
		}
		cp = cp->next;
	}

	return (cp);
}

/* skip_decrypt_hash_add()
 *
 * Add a hash table entry for a decryt key
 *
 * Returns: the entry or NULL on error
 */
static skip_keycb_t *
skip_decrypt_hash_add(unsigned int hash)
{
	register skip_keycb_t	*cp;


	if (skip_keys_stats.skip_decrypt_keys_active == skip_key_max_entries) {
		return (NULL);
	}

        cp = (skip_keycb_t *) MEM_ALLOC(sizeof(*cp));

	if (cp == NULL) {
		return (NULL);
	}

	skip_keys_stats.skip_decrypt_keys_active++;

	if (skip_decrypt_hashtable[hash] == NULL) {
		skip_decrypt_hashtable[hash] = cp;
	} else {
		/*
		 * hash collision found - insert in collision chain...
		 */
		skip_keys_stats.skip_hash_collisions++;
		cp->next = skip_decrypt_hashtable[hash];
		skip_decrypt_hashtable[hash]->prev = cp; 
		skip_decrypt_hashtable[hash] = cp; 
	}
	return(cp);
}


/* skip_hash_remove()
 *
 * Returns: remove a decryption hash table entry
 */
static void
skip_hash_remove(skip_keycb_t * cp, skip_keycb_t *table[], unsigned long *countp)
{
	register skip_queue_t	*callback, *nextcallback;
	register skip_cryptor_t	*alg;

	if (cp->next != NULL) {
		cp->next->prev = cp->prev;
	}
	if (cp->prev != NULL) {
		cp->prev->next = cp->next;
	} else {
		table[cp->hash] = cp->next;
	}

	/*
	 * drop any waiting packets
	 */
	for (callback = cp->queue; callback; callback = nextcallback) {

		nextcallback = callback->next;

		BUFFREE(callback->in);
		BUFFREE(callback->out);
		MEM_FREE(callback, sizeof(*callback));

	}

	if (cp->cs) {
		/*
		 * close key handle
		 */
		alg = GETCRYPTMOD(cp->hdr.params.version,
					cp->hdr.params.kp_alg);
		(alg->close)(cp->cs);
	}

	/*
	 * XXX anything else that should be zapped?
	 */

	MEM_FREE(cp, sizeof(*cp));

	(*countp)--;

}


/*
 * skip_get_key_status()
 *
 *  fill in key status ioctl
 */
int
skip_get_key_status(ioctl_key_status_t *p_status)
{

	register skip_keycb_t	*cp;
	register unsigned int	hash;

	p_status->count_out = 0;
	p_status->count_in = 0;


	/*
	 * Get encrypt keys status
	 */
	for (hash = 0; hash < SKIP_HASH_TABLESZ; hash++) {

		MUTEX_ENTER(&skip_encrypt_hashlocks[hash]);

		cp = skip_encrypt_hashtable[hash];

		while (cp) {
			if ((p_status->params.r_nsid ==
					cp->hdr.params.r_nsid) &&
				KEYVAREQUAL(p_status->params.r_mkeyid,
					cp->hdr.params.r_mkeyid) &&
				!bcmp((caddr_t) &p_status->params.ip_addr,
					(caddr_t) &cp->hdr.params.ip_addr,
					sizeof(struct in_addr))) {
				/*
				 * found it...
				 */
				p_status->params	= cp->hdr.params;
				p_status->count_out	= 1;
				p_status->obytes	= cp->obytes;
				p_status->max_obytes	= cp->max_obytes;
	
				p_status->ekp_len = cp->hdr.ekp.len;
				bcopy((caddr_t) cp->hdr.ekp.buf,
					(caddr_t) p_status->ekp_buf,
					cp->hdr.ekp.len);
				break;
			}
			cp = cp->next;
		}

		MUTEX_EXIT(&skip_encrypt_hashlocks[hash]);
	}


	/*
	 * Get decrypt keys status
	 */
	for (hash = 0; hash < SKIP_HASH_TABLESZ; hash++) {

		MUTEX_ENTER(&skip_decrypt_hashlocks[hash]);

		cp = skip_decrypt_hashtable[hash];
	
		while (cp) {
			if ((p_status->params.r_nsid ==
					cp->hdr.params.r_nsid) &&
				KEYVAREQUAL(p_status->params.r_mkeyid,
					cp->hdr.params.r_mkeyid) &&
				!bcmp((caddr_t) &p_status->params.ip_addr,
					(caddr_t) &cp->hdr.params.ip_addr,
					sizeof(struct in_addr))) {
				/*
				 * found it...
				 */
				p_status->count_in++;
			}
			cp = cp->next;
		}

		MUTEX_EXIT(&skip_decrypt_hashlocks[hash]);
	}
	return(0);
}

/*
 * skip_diffs()
 *
 * debugging function for printing diffs of two skip_param_t structs
 */
int
skip_diffs(char *who, skip_param_t *in, skip_param_t *ref)
{
	int i;

	if (!skip_params_debug) {
		return(0);
	}

	printf("%s: PARAMS mismatch\n", who);

	if (in->version != ref->version) {
		printf("in->version=%d, ref->version=%d\n",
					in->version, ref->version);
	}
	if (in->r_nsid != ref->r_nsid) {
		printf("in->r_nsid=%d, ref->r_nsid=%d\n",
					in->r_nsid, ref->r_nsid);
	}
	if (in->s_nsid != ref->s_nsid) {
		printf("in->s_nsid=%d, ref->s_nsid=%d\n",
					in->s_nsid, ref->s_nsid);
	}
	if (in->kij_alg != ref->kij_alg) {
		printf("in->kij_alg=%d, ref->kij_alg=%d\n",
					in->kij_alg, ref->kij_alg);
	}
	if (in->kp_alg != ref->kp_alg) {
		printf("in->kp_alg=%d, ref->kp_alg=%d\n",
					in->kp_alg, ref->kp_alg);
	}
	if (in->mac_alg != ref->mac_alg) {
		printf("in->mac_alg=%d, ref->mac_alg=%d\n",
					in->mac_alg, ref->mac_alg);
	}
	if (in->comp_alg != ref->comp_alg) {
		printf("in->comp_alg=%d, ref->comp_alg=%d\n",
					in->comp_alg, ref->comp_alg);
	}
	if (in->r_mkeyid.len != ref->r_mkeyid.len) {
		printf("in->r_mkeyid.len=%d, ref->r_mkeyid.len=%d\n",
					in->r_mkeyid.len, ref->r_mkeyid.len);
	}

	if (!KEYVAREQUAL(in->r_mkeyid, ref->r_mkeyid)) {
		printf("Master key in  [");
		for (i = 0; i < 4; i++ ) {
			printf("0x%x ", (unsigned char) in->r_mkeyid.buf[i]);

		}
		printf("]\n");
		printf("Master key ref [");
		for (i = 0; i < 4; i++ ) {
			printf("0x%x ", (unsigned char) ref->r_mkeyid.buf[i]);
		}
		printf("]\n");
	}

	if (bcmp((caddr_t) &in->ip_addr, (caddr_t) &ref->ip_addr,
						sizeof(in->ip_addr))) {
		printf("in->ip_addr=0x%x, ref->ip_addr=0x%x\n",
			in->ip_addr.s_addr, ref->ip_addr.s_addr);
	}

	return(0);
}
