/* opiepasswd.c: Add/change an OTP password in the key database.

Portions of this software are Copyright 1996 by Craig Metz, All Rights
Reserved. The Inner Net Copyright Notice and License Agreement applies to
these portions of the software.

Portions of this software are Copyright 1995 by Randall Atkinson and Dan
McDonald, All Rights Reserved. All Rights under this copyright are assigned
to the U.S. Naval Research Laboratory (NRL). The NRL Copyright Notice and
License Agreement applies to this software.

	History:

	Modified by cmetz for OPIE 2.22. Finally got rid of the lock
	        filename kluge by implementing refcounts for locks.
		Use opiepasswd() to update key file. Error if we can't
		write to the key file. Check for minimum seed length.
        Modified at NRL for OPIE 2.2. Changed opiestrip_crlf to
                opiestripcrlf. Check opiereadpass() return value.
                Minor optimization. Change calls to opiereadpass() to
                use echo arg. Use opiereadpass() where we can.
                Make everything static. Ifdef around some headers.
                Changed use of gethostname() to uname(). Got rid of
                the need for buf[]. Properly check return value of
                opieatob8. Check seed length. Always generate proper-
                length seeds.
	Modified at NRL for OPIE 2.1. Minor autoconf changes.
        Modified heavily at NRL for OPIE 2.0.
	Written at Bellcore for the S/Key Version 1 software distribution
		(skeyinit.c).
*/
/* The implications of this program needing to run setuid are not entirely
   clear. We believe it to be safe, but more proactive measures need to be
   taken to reduce the risks of being setuid (such as discarding priveleges
   as quickly as possible. More thought needs to be given to this at some
   future date. */

#include "opie_cfg.h"

#if HAVE_PWD_H
#include <pwd.h>
#endif /* HAVE_PWD_H */
#include <stdio.h>
#if HAVE_STRING_H
#include <string.h>
#endif /* HAVE_STRING_H */
#include <stdio.h>
#if TM_IN_SYS_TIME
#include <sys/time.h>
#else /* TM_IN_SYS_TIME */
#include <time.h>
#endif /* TM_IN_SYS_TIME */
#include <sys/types.h>
#if HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif /* HAVE_STDLIB_H */
#include <ctype.h>

#include "opie.h"

extern int optind;
extern char *optarg;

static VOIDRET usage FUNCTION((myname), char *myname)
{
  fprintf(stderr, "usage: %s [-v] [-h] [-c] [-n initial_sequence_number]\n                            [-s seed] [username]\n", myname);
  exit(1);
}

int main FUNCTION((argc, argv), int argc AND char *argv[])
{
  struct opie opie;
  int rval, n = 499, nn, i, consolemode = 0;
  char seed[18];
  char tmp[OPIE_RESPONSE_MAX + 2];	/* extra space for \n */
  char key[8], key2[8];
  struct passwd *pp;
  char passwd[OPIE_PASS_MAX + 1], passwd2[OPIE_PASS_MAX + 1];
  time_t now;
  struct tm *tm;
  char tbuf[30];
  char lastc;
  int l;

  memset(seed, 0, sizeof(seed));
  memset(tmp, 0, sizeof(tmp));
  memset(key, 0, sizeof(key));
  memset(key2, 0, sizeof(key2));

  if (!(pp = getpwuid(getuid()))) {
    fprintf(stderr, "Who are you?");
    return 1;
  }

  while ((i = getopt(argc, argv, "hvcn:s:")) != EOF) {
    switch (i) {
    case 'v':
      opieversion();
    case 'c':
      consolemode = 1;
      break;
    case 'n':
      nn = atoi(optarg);
      if (!(nn > 0 && nn < 10000)) {
	printf("Sequence numbers must be > 0 and < 10000\n");
	exit(1);
      }
      n = nn;
      break;
    case 's':
      nn = strlen(optarg);
      if ((nn > OPIE_SEED_MAX) || (nn < OPIE_SEED_MIN)) {
	printf("Seeds must be between %d and %d characters long.\n",
	       OPIE_SEED_MIN, OPIE_SEED_MAX);
	exit(1);
      }
      strncpy(seed, optarg, sizeof(seed));
      seed[sizeof(seed) - 1] = 0;
      break;
    default:
      usage(argv[0]);
    }
  }

  if (argc - optind >= 1) {
    if (strcmp(argv[optind], pp->pw_name)) {
      if (getuid()) {
	printf("Only root can change others' passwords.\n");
	return (1);
      }
      if ((pp = getpwnam(argv[optind])) == NULL) {
	printf("%s: user unknown.\n", argv[optind]);
	return 1;
      }
    }
  }

  rval = opiechallenge(&opie, pp->pw_name, tmp);
  opielock(pp->pw_name);

  switch (rval) {
  case 0:
    printf("Updating %s:\n", pp->pw_name);
    break;
  case 1:
    printf("Adding %s:\n", pp->pw_name);
    break;
  default:
    perror("Error opening key database");
    return 1;
  }

  if (seed[0]) {
    int i = strlen(seed);
    if (i > OPIE_SEED_MAX) {
      fprintf(stderr, "Seeds must be less than %d characters long.", OPIE_SEED_MAX);
      exit(1);
    }
    if (i < OPIE_SEED_MIN) {
      fprintf(stderr, "Seeds must be greater than %d characters long.", OPIE_SEED_MIN);
      exit(1);
    }
  } else {
    if (!rval)
      strcpy(seed, opie.seed);

    if (opienewseed(seed) < 0) {
      fprintf(stderr, "Error updating seed.\n");
      exit(1);
    }
  }

  if (opie.seed && opie.seed[0] && !strcmp(opie.seed, seed)) {
    fprintf(stderr, "You must use a different seed for the new OTP sequence.\n");
    exit(1);
  }

  if (!consolemode) {
    printf("Reminder: You need the response from your OPIE calculator.\n");
    if (!rval && getuid()) {
      printf("Old secret pass phrase:\n\t%s\n\tResponse: ", tmp);
      if (!opiereadpass(tmp, sizeof(tmp), 1)) {
	fprintf(stderr, "Error!\n");
	exit(1);
      }
      if (nn = opieverify(&opie, tmp)) {
	fprintf(stderr, "Error!\n");
	exit(1);
      }
    }
    printf("New secret pass phrase:");
    for (i = 0;; i++) {
      if (i >= 2)
	exit(1);
      printf("\n\totp-md%d %d %s\n\tResponse: ", MDX, n, seed);
      if (!opiereadpass(tmp, sizeof(tmp), 1)) {
	fprintf(stderr, "Error!\n");
	exit(1);
      }
      if (tmp[0] == '?') {
	printf("Enter the response from your OTP calculator: \n");
	continue;
      }
      if (tmp[0] == '\0') {
        fprintf(stderr, "Secret pass phrase unchanged.\n");
	exit(1);
      }
      if ((opieetob(key, tmp) == 1) || !opieatob8(key, tmp))
	break;	/* Valid format */
      printf("Invalid format, try again with 6 English words.\n");
    }
  } else {
    /* Get user's secret password */
    fprintf(stderr, "Reminder - Only use this method from the console; NEVER from remote. If you\n");
    fprintf(stderr, "are using telnet, xterm, or a dial-in, type ^C now or exit with no password.\n");
    fprintf(stderr, "Then run opiepasswd without the -c parameter.\n");
    if (opieinsecure()) {
      fprintf(stderr, "Sorry, but you don't seem to be on the console or a secure terminal.\n");
      exit(1);
    };
    printf("Using MD%d to compute responses.\n", MDX);
    if (!rval && getuid()) {
      printf("Enter old secret pass phrase: ");
      if (!opiereadpass(passwd, sizeof(passwd), 0)) {
        fprintf(stderr, "Error reading secret pass phrase!\n");
        exit(1);
      }
      if (!passwd[0]) {
        fprintf(stderr, "Secret pass phrase unchanged.\n");
	exit(1);
      }
      if (opiekeycrunch(MDX, key, opie.seed, passwd) != 0) {
	fprintf(stderr, "%s: key crunch failed -- secret pass phrase unchanged\n", argv[0]);
	exit(1);
      }
      memset(passwd, 0, sizeof(passwd));
      nn = opie.n - 1;
      while (nn-- != 0)
	opiehash(key, MDX);
      opiebtoe(tbuf, key);
      nn = opieverify(&opie, tbuf);
      if (nn) {
	fprintf(stderr, "Sorry.\n");
	exit(1);
      }
    }
    for (i = 0;; i++) {
      if (i >= 2)
	exit(1);
      printf("Enter new secret pass phrase: ");
      if (!opiereadpass(passwd, sizeof(passwd), 0)) {
	fprintf(stderr, "Error reading secret pass phrase.\n");
	exit(1);
      }
      if (!passwd[0] || feof(stdin)) {
	fprintf(stderr, "Secret pass phrase unchanged.\n");
	exit(1);
      }
      printf("Again new secret pass phrase: ");
      if (!opiereadpass(passwd2, sizeof(passwd2), 0)) {
	fprintf(stderr, "Error reading secret pass phrase.\n");
	exit(1);
      }
      if (feof(stdin)) {
	fprintf(stderr, "Secret pass phrase unchanged.\n");
	exit(1);
      }
      if (!passwd[0] || !strcmp(passwd, passwd2))
        break;
      fprintf(stderr, "Sorry, no match.\n");
    }
    memset(passwd2, 0, sizeof(passwd2));
    if (opiepasscheck(passwd)) { 
      memset(passwd, 0, sizeof(passwd));
      fprintf(stderr, "Secret pass phrases must be between %d and %d characters long.\n", OPIE_PASS_MIN, OPIE_PASS_MAX);
      exit(1);
    }

    /* Crunch seed and password into starting key */
    if (opiekeycrunch(MDX, key, seed, passwd) != 0) {
      memset(passwd, 0, sizeof(passwd));
      fprintf(stderr, "%s: key crunch failed\n", argv[0]);
      return 1;
    }
    memset(passwd, 0, sizeof(passwd));
    nn = n;
    while (nn-- != 0)
      opiehash(key, MDX);
  }

  if (opiepasswd(&opie, pp->pw_name, n, seed, key)) {
    fprintf(stderr, "Error updating key database.\n");
    return 1;
  }

  printf("\nID %s OTP key is %d %s\n", pp->pw_name, n, seed);
  printf("%s\n", opiebtoe(tbuf, key));
  return 0;
}
