%{

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <netinet/in.h>
#include <linux/if.h>
#include <sys/ioctl.h>
#include <fcntl.h>

#include <linux/ipsec.h>

int yylex(void);
int yyparse(void);

#include "pk_access.h"
#include "ipsec_access.h"

static int sock;
static int attsock;
static int ipsec_dev;

#define YYDEBUG	1

static struct pk_message *curr_msg = NULL;

#define next_message(s)	if (1) { if (pkmsg_send(sock, curr_msg) < 0)	\
					perror(s);			\
				 pkmsg_free(curr_msg);			\
				 curr_msg = pkmsg_new(); } else (void)NULL

#define dump_message	if (1) { pkmsg_free(curr_msg);			\
				 curr_msg = pkmsg_new(); } else (void)NULL

static int make_key(int ktype, int ksize, char *kmat);
static void test_and_print(int s);
static struct sadb_sa *get_sa(void);
static int put_bundle_command(void *, int);
static int put_addr(int type, struct sockaddr_in *sin);

%}

%union {
	struct sockaddr_in			saddr;
	struct ipsec_payload_bundle		bundle;
	struct ipsec_payload_policy		policy;
	struct { __u32 addr; __u32 mask; }	amask;
	unsigned long long			lval;
	unsigned char				kval[256];
	char					str[256];
	int					ival;
}

%token DUMP ACQUIRE GETSPI GET ADD UPDATE REGISTER FLUSH DELETE QUIT
%token ATTACH DETACH CREATEDEV DESTROYDEV
%token <str> ID
%token ESP AH UNSPEC
%token HARD SOFT BYTES PACKETS USE TIME
%token SOURCE DEST PROXY SPI SPIRANGE SPIANY
%token AKEY EKEY
%token MD5HMAC SHA1 MD5KPDK
%token DESCBC DES3CBC DESIV32 DESIV64 IDEA
%token REPLAY
%token DOT SEMI ERROR
%token MATURE LARVAL DYING DEAD
%token <lval> INT
%token <kval> KEYMAT
%token BUNDLE SLASH PROTOCOL POSITION POLICY NONE ACQUIRE INPUT OUTPUT
%token SPORT DPORT UID
%token COPY CLEAR FLAGS

%type <ival> auth_algo encrypt_algo state
%type <saddr> addr
%type <bundle> sa_list
%type <amask> source_mask dest_mask proto_mask sport_mask dport_mask uid_mask
%type <policy> policy_addr policy_extra
%type <ival> mask position ptype spi_type pdir
%type <ival> df_flags

%%
command :
		command dump_command
	|	command acquire_command
	|	command getspi_command
	|	command get_command
	|	command add_command
	|	command delete_command
	|	command update_command
	|	command register_command
	|	command flush_command
	|	command attach_command
	|	command detach_command
	|	command create_command
	|	command destroy_command
	|	command bundle_command
	|	command policy_command
	|	command flags_command
	|	command error SEMI
		{ dump_message; }
	|	command QUIT
		{ exit(0); }
	|	SEMI
	|

dump_command :
		DUMP type SEMI
			{
				curr_msg->hdr.sadb_msg_type = SADB_DUMP;
				next_message("dump");
				test_and_print(sock);
			}

delete_command :
		DELETE type contents SEMI
			{
				curr_msg->hdr.sadb_msg_type = SADB_DELETE;
				next_message("delete");
				test_and_print(sock);
			}

acquire_command :
		ACQUIRE type contents SEMI
			{ 
				curr_msg->hdr.sadb_msg_type = SADB_GETSPI;
				next_message("acquire");
				test_and_print(sock);
			}

getspi_command :
		GETSPI type contents SEMI
			{
				curr_msg->hdr.sadb_msg_type = SADB_GETSPI;
				next_message("getSPI");
				test_and_print(sock);
			}

get_command :
		GET type contents SEMI
			{
				curr_msg->hdr.sadb_msg_type = SADB_GET;
				next_message("get");
				test_and_print(sock);
			}

add_command :
		ADD type contents SEMI
			{
				curr_msg->hdr.sadb_msg_type = SADB_ADD;
				next_message("add");
				test_and_print(sock);
			}

update_command :
		UPDATE type contents SEMI
			{
				curr_msg->hdr.sadb_msg_type = SADB_UPDATE;
				next_message("update");
				test_and_print(sock);
			}

register_command :
		REGISTER type SEMI
			{
				curr_msg->hdr.sadb_msg_type = SADB_REGISTER;
				next_message("register");
				test_and_print(sock);
			}

flush_command :
		FLUSH type SEMI
			{
				curr_msg->hdr.sadb_msg_type = SADB_FLUSH;
				next_message("flush");
				test_and_print(sock);
			}

attach_command :
		ATTACH ID ID SEMI
		{
			struct ifreq ifr;

			memset(&ifr, 0, sizeof ifr);
			strcpy(ifr.ifr_name, $2);
			strcpy(ifr.ifr_slave, $3);
			if (ioctl(attsock, IPSEC_ATTACH, &ifr))
				perror("attach");
		}

detach_command :
		DETACH ID SEMI
		{
			struct ifreq ifr;

			memset(&ifr, 0, sizeof ifr);
			strcpy(ifr.ifr_name, $2);
			if (ioctl(attsock, IPSEC_DETACH, &ifr))
				perror("attach");
		}

create_command :
		CREATEDEV INT SEMI
		{
			struct ifreq ifr;

			memset(&ifr, 0, sizeof ifr);
			strcpy(ifr.ifr_name, "ipsec0");
			*(int *)&ifr.ifr_data = $2;
			if (ioctl(attsock, IPSEC_GETNEW, &ifr))
				perror("attach");
		}

destroy_command :
		DESTROYDEV ID SEMI
		{
			struct ifreq ifr;

			memset(&ifr, 0, sizeof ifr);
			strcpy(ifr.ifr_name, $2);
			if (ioctl(attsock, IPSEC_RELEASE, &ifr))
				perror("attach");
		}

flags_command :
		FLAGS ID df_flags
		{
			struct ifreq ifr;

			memset(&ifr, 0, sizeof ifr);
			strcpy(ifr.ifr_name, $2);
			*(int *)(ifr.ifr_data) = $3;
			if (ioctl(attsock, IPSEC_SETFLAGS, &ifr))
				perror("attach");
		}

df_flags :	COPY	{ $$ = IPSEC_COPY_DF; }
	|	CLEAR	{ $$ = 0; }

bundle_command:
		bundle_add
	|	bundle_delete
	|	bundle_flush

bundle_add :
		BUNDLE ADD INT sa_list SEMI
		{
			struct {
				struct ipsec_netlink_hdr inh;
				struct ipsec_payload_hdr iph;
				struct ipsec_payload_bundle ipc;
			} tval;

			tval.inh.len = sizeof tval;
			tval.inh.version = IPSEC_NL_VERSION;
			tval.iph.len = sizeof tval - sizeof tval.inh;
			tval.iph.payload = IPSEC_PAYLOAD_BUNDLE;
			tval.iph.command = IPSEC_NL_BUNDLE_ADD;
			tval.ipc = $4;
			tval.ipc.bundle_id = $3;
			put_bundle_command(&tval, sizeof tval);
		}

bundle_delete :
		BUNDLE DELETE INT SEMI
		{
			struct {
				struct ipsec_netlink_hdr inh;
				struct ipsec_payload_hdr iph;
				__u32 id;
			} tval;

			tval.inh.len = sizeof tval;
			tval.inh.version = IPSEC_NL_VERSION;
			tval.iph.len = sizeof tval - sizeof tval.inh;
			tval.iph.payload = IPSEC_PAYLOAD_ID;
			tval.iph.command = IPSEC_NL_BUNDLE_REMOVE;
			tval.id = $3;
			put_bundle_command(&tval, sizeof tval);
		}

bundle_flush :
		BUNDLE FLUSH SEMI
		{
			struct {
				struct ipsec_netlink_hdr inh;
				struct ipsec_payload_hdr iph;
			} tval;

			tval.inh.len = sizeof tval;
			tval.inh.version = IPSEC_NL_VERSION;
			tval.iph.len = sizeof tval - sizeof tval.inh;
			tval.iph.payload = IPSEC_PAYLOAD_EMPTY;
			tval.iph.command = IPSEC_NL_BUNDLE_FLUSH;
			put_bundle_command(&tval, sizeof tval);
		}


policy_command:
		policy_add
	|	policy_delete
	|	policy_flush

policy_add :
		POLICY ADD pdir policy_addr position ptype INT SEMI

		{
			struct {
				struct ipsec_netlink_hdr inh;
				struct ipsec_payload_hdr iph;
				struct ipsec_payload_policy ipp;
			} tval;

			tval.inh.len = sizeof tval;
			tval.inh.version = IPSEC_NL_VERSION;
			tval.iph.len = sizeof tval - sizeof tval.inh;
			tval.iph.payload = IPSEC_PAYLOAD_POLICY;
			tval.iph.command = IPSEC_NL_POLICY_ADD;
			tval.ipp = $4;
			tval.ipp.policy_list = $3;
			tval.ipp.position = $5;
			tval.ipp.policy_type = $6;
			tval.ipp.id = $7;
			put_bundle_command(&tval, sizeof tval);
		}

policy_delete :
		POLICY DELETE pdir policy_addr SEMI
		{
			struct {
				struct ipsec_netlink_hdr inh;
				struct ipsec_payload_hdr iph;
				struct ipsec_payload_policy ipp;
			} tval;

			tval.inh.len = sizeof tval;
			tval.inh.version = IPSEC_NL_VERSION;
			tval.iph.len = sizeof tval - sizeof tval.inh;
			tval.iph.payload = IPSEC_PAYLOAD_POLICY;
			tval.iph.command = IPSEC_NL_POLICY_REMOVE;
			tval.ipp = $4;
			tval.ipp.policy_list = $3;
			put_bundle_command(&tval, sizeof tval);
		}


policy_addr :	source_mask dest_mask policy_extra
		{
			$$ = $3;
			$$.saddr = $1.addr;
			$$.smask = $1.mask;
			$$.daddr = $2.addr;
			$$.dmask = $2.mask;
		}

policy_extra:
		{ memset(&($$), 0, sizeof $$); }
	|	policy_extra proto_mask
		{ $$ = $1; $$.protocol = $2.addr; $$.pmask = $2.mask; }
	|	policy_extra sport_mask
		{ $$ = $1; $$.sport = $2.addr; $$.sportmask = $2.mask; }
	|	policy_extra dport_mask
		{ $$ = $1; $$.dport = $2.addr; $$.dportmask = $2.mask; }
	|	policy_extra uid_mask
		{ $$ = $1; $$.uid = $2.addr; $$.uidmask = $2.mask; }

pdir :		INPUT
		{ $$ = IPSEC_INPUT_POLICY; }
	|	OUTPUT
		{ $$ = IPSEC_OUTPUT_POLICY; }

ptype :		BUNDLE
		{ $$ = IPSEC_PT_BUNDLE; }
	|	NONE
		{ $$ = IPSEC_PT_NONE; }
	|	ERROR
		{ $$ = IPSEC_PT_ERROR; }
	|	ACQUIRE
		{ $$ = IPSEC_PT_ACQUIRE; }

policy_flush :	POLICY FLUSH SEMI
		{
			struct {
				struct ipsec_netlink_hdr inh;
				struct ipsec_payload_hdr iph;
			} tval;

			tval.inh.len = sizeof tval;
			tval.inh.version = IPSEC_NL_VERSION;
			tval.iph.len = sizeof tval - sizeof tval.inh;
			tval.iph.payload = IPSEC_PAYLOAD_EMPTY;
			tval.iph.command = IPSEC_NL_POLICY_FLUSH;
			put_bundle_command(&tval, sizeof tval);
		}

source_mask :	SOURCE addr mask
		{
			$$.addr = $2.sin_addr.s_addr;
			$$.mask = $3;
			if (($$.addr) & ~($$.mask)) {
				fprintf(stderr,
					"invalid source address/mask %x/%x\n",
					$$.addr, $$.mask);
				goto yyerrlab;
			}
		}

dest_mask :	DEST addr mask
		{
			$$.addr = $2.sin_addr.s_addr;
			$$.mask = $3;
			if (($$.addr) & ~($$.mask)) {
				fprintf(stderr,
					"invalid dest address/mask %x/%x\n",
					$$.addr, $$.mask);
				goto yyerrlab;
			}
		}

proto_mask :	PROTOCOL INT
		{ $$.addr = $2; $$.mask = 0xffffffff; }

sport_mask :	SPORT INT
		{ $$.addr = htons($2); $$.mask = 0xffffffff; }

dport_mask :	DPORT INT
		{ $$.addr = htons($2); $$.mask = 0xffffffff; }

uid_mask :	UID INT
		{ $$.addr = $2; $$.mask = 0xffffffff; }


position :	POSITION INT
		{ $$ = $2; }
	|
		{ $$ = 0; }

mask :
		SLASH INT
		{
			if ($2 > 32)
				goto yyerrlab;
			if ($2 > 0)
				$$ = (0xffffffff << (32 - $2)) & 0xffffffff;
			else
				$$ = 0;
		}
	|
		{ $$ = 0xffffffff; }

sa_list :	spi_type INT addr
		{
			memset(&($$), 0, sizeof $$);
			$$.bundle_id = 0;
			$$.protocol[0] = $1;
			$$.spi[0] = htonl($2);
			$$.proxy[0] = $3.sin_addr.s_addr;
		}
	|	sa_list spi_type INT addr
		{
			$$ = $1;
			$$.bundle_id++;
			if ($$.bundle_id > IPSEC_MAX_SPIS)
				goto yyerrlab;
			$$.protocol[$$.bundle_id] = $2;
			$$.spi[$$.bundle_id] = htonl($3);
			$$.proxy[$$.bundle_id] = $4.sin_addr.s_addr;
		}

spi_type :	AH
		{ $$ = IPPROTO_AH; }
	|	ESP
		{ $$ = IPPROTO_ESP; }

type:		
		ESP
			{ curr_msg->hdr.sadb_msg_satype = SADB_SATYPE_ESP; }
	|	AH
			{ curr_msg->hdr.sadb_msg_satype = SADB_SATYPE_AH; }
	|	UNSPEC
			{ curr_msg->hdr.sadb_msg_satype = SADB_SATYPE_UNSPEC; }

contents :
		contents entry
	|	entry

entry :
		address
	|	range
	|	spi
	|	sa
	|	auth_key
	|	encrypt_key
	|	timeout

timeout:
		HARD	{ get_life(SADB_EXT_LIFETIME_HARD); }
			limits { get_life(-1); }
	|	SOFT	{ get_life(SADB_EXT_LIFETIME_SOFT); }
			limits { get_life(-1); }

limits: 	limits limit
	|	limit

limit:
		PACKETS INT
		{
			struct sadb_lifetime *life;

			if (!(life = get_life(0)))
				goto yyerrlab;
			life->sadb_lifetime_allocations = $2;
		}
	|	BYTES INT
		{
			struct sadb_lifetime *life;

			if (!(life = get_life(0)))
				goto yyerrlab;
			life->sadb_lifetime_bytes = $2;
		}
	|	TIME INT
		{
			struct sadb_lifetime *life;

			if (!(life = get_life(0)))
				goto yyerrlab;
			life->sadb_lifetime_addtime = $2;
		}
	|	USE INT
		{
			struct sadb_lifetime *life;

			if (!(life = get_life(0)))
				goto yyerrlab;
			life->sadb_lifetime_usetime = $2;
		}

sa :		auth_algo
		{
			struct sadb_sa *sa = get_sa();

			if (!sa)
				goto yyerrlab;
			sa->sadb_sa_auth = $1;
		}
	|	encrypt_algo
		{
			struct sadb_sa *sa = get_sa();

			if (!sa)
				goto yyerrlab;
			sa->sadb_sa_encrypt = $1;
		}
	|	state
		{
			struct sadb_sa *sa = get_sa();

			if (!sa)
				goto yyerrlab;
			sa->sadb_sa_state = $1;
		}
	|	REPLAY INT
		{
			struct sadb_sa *sa = get_sa();

			if (!sa)
				goto yyerrlab;
			sa->sadb_sa_replay = $2;
		}

auth_key :
		AKEY INT KEYMAT
		{
			if (make_key(SADB_EXT_KEY_AUTH, $2, $3))
				goto yyerrlab;
		}

encrypt_key :
		EKEY INT KEYMAT
		{
			if (make_key(SADB_EXT_KEY_ENCRYPT, $2, $3))
				goto yyerrlab;
		}

auth_algo :
		MD5HMAC { $$ = SADB_AALG_MD5HMAC; }
	|	SHA1	{ $$ = SADB_AALG_SHA1HMAC; }
	|	MD5KPDK	{ $$ = SADB_AALG_X_MD5KPDK; }

encrypt_algo :
		DESCBC	{ $$ = SADB_EALG_DESCBC; }
	|	DES3CBC	{ $$ = SADB_EALG_3DESCBC; }
	|	DESIV32	{ $$ = SADB_EALG_X_DESIV32; }
	|	DESIV64	{ $$ = SADB_EALG_X_DESIV64; }
	|	IDEA	{ $$ = SADB_EALG_X_IDEA; }
	|	NONE	{ $$ = SADB_EALG_NONE; }

state :
		LARVAL	{ $$ = SADB_SASTATE_LARVAL; }
	|	MATURE	{ $$ = SADB_SASTATE_MATURE; }
	|	DYING	{ $$ = SADB_SASTATE_DYING; }
	|	DEAD	{ $$ = SADB_SASTATE_DEAD; }

spi :		SPI INT
		{
			struct sadb_sa *sa = get_sa();

			if (!sa)
				goto yyerrlab;
			sa->sadb_sa_spi = htonl($2);
		}

range :		SPIRANGE INT INT
		{
			struct sadb_spirange *spir;

			spir = (struct sadb_spirange *)
			       pkmsg_add(curr_msg,
					 SADB_EXT_SPIRANGE, sizeof spir[0]);
			if (!spir)
				goto yyerrlab;
			spir->sadb_spirange_min = htonl($2);
			spir->sadb_spirange_max = htonl($3);
		}
	|	SPIANY
		{
			struct sadb_spirange *spir;

			spir = (struct sadb_spirange *)
			       pkmsg_add(curr_msg,
					 SADB_EXT_SPIRANGE, sizeof spir[0]);
			if (!spir)
				goto yyerrlab;
			spir->sadb_spirange_min = 0;
			spir->sadb_spirange_max = 0xffffffff;
		}


address :
		DEST addr
			{
				if (put_addr(SADB_EXT_ADDRESS_DST, &($2)))
					goto yyerrlab;
			}
	|	SOURCE addr
			{
				if (put_addr(SADB_EXT_ADDRESS_SRC, &($2)))
					goto yyerrlab;
			}
	|	PROXY addr
			{
				if (put_addr(SADB_EXT_ADDRESS_PROXY, &($2)))
					goto yyerrlab;
			}

addr:
		INT DOT INT DOT INT DOT INT
			{
				memset(&($$), 0, sizeof(struct sockaddr));
				$$.sin_family = AF_INET;
				$$.sin_addr.s_addr = ntohl((($1)<<24)
							   | (($3) << 16)
							   | (($5) << 8)
							   | ($7));
			}
%%

static struct sadb_lifetime *
get_life(int type)
{
	struct sadb_lifetime *life;
	static int last_type;

	if (type)
	{
		last_type = type;
		if (type == -1)
			return NULL;
		life = (struct sadb_lifetime *)pkmsg_add(curr_msg, type,
							 sizeof *life);
		return life;
	}
	if (type >= 0)
		return (struct sadb_lifetime *)curr_msg->n_ex[last_type];
	return NULL;
}

static int
put_addr(int type, struct sockaddr_in *sin)
{
	struct sadb_address *saddr;

	saddr = (struct sadb_address *)
		pkmsg_add(curr_msg, type, (sizeof *saddr
					   + sizeof(struct sockaddr_in)));
	if (!saddr)
		return -1;
	saddr->sadb_address_proto = 0;
	memcpy(&saddr[1], sin, sizeof(struct sockaddr_in));
	return 0;
}

static int
put_bundle_command(void *inh, int len)
{
	int count;

	count = write(ipsec_dev, inh, len);
	if (count != len)
		perror("/dev/ipsec");
	return count;
}

static struct sadb_sa *
get_sa()
{
	struct sadb_sa *sa;

	if ((sa = (struct sadb_sa *)curr_msg->n_ex[SADB_EXT_SA]) != NULL)
		return sa;
	sa = (struct sadb_sa *) pkmsg_add(curr_msg, SADB_EXT_SA, sizeof *sa);
	sa->sadb_sa_state = SADB_SASTATE_MATURE;
	return sa;
}

static void
test_and_print(int s)
{
    fd_set rd;
    struct timeval tv;

    FD_ZERO(&rd);
    FD_SET(s, &rd);
    memset(&tv, 0, sizeof tv);
    while (select(s + 1, &rd, NULL, NULL, &tv) > 0)
    {
	struct pk_message *pm;

	pm = pkmsg_receive(s);
	pkmsg_print(pm);
	pkmsg_free(pm);
    }
}

static int
make_key(int ktype, int ksize, char *kmat)
{
    int mesg_size;
    struct sadb_ext *se;
    struct sadb_key *sk;

    if (kmat[0] < (ksize + 7)>>3) {
	fprintf(stderr, "too little key material entered\n");
	return -1;
    }
    mesg_size = (ksize + 63) >> 6;	/* multiple of 64 */
    se = pkmsg_add(curr_msg, ktype, sizeof(struct sadb_key) + mesg_size * 8);
    sk = (struct sadb_key *)se;
    if (!se)
	return -1;
    if (kmat[0] > (ksize + 7)>>3)
	fprintf(stderr, "excess key material entered\n");
	
    memcpy(&sk[1], kmat, (ksize+7)>>3);
    sk->sadb_key_bits = ksize;
    return 0;
}


int
main()
{
    sock = socket(AF_KEY, SOCK_RAW, PF_KEY_V2);
    if (sock < 0)
    {
	perror("pfkey socket");
	return 1;
    }
    attsock = socket(AF_INET, SOCK_STREAM, 0);
    if (attsock < 0)
    {
	perror("af_inet socket");
	return 1;
    }
    ipsec_dev = open("/dev/ipsec", O_RDWR);
    if (ipsec_dev < 0)
    {
	perror("/dev/ipsec");
	return 1;
    }
    curr_msg = pkmsg_new();
    yyparse();
    return 0;
}

int
yyerror()
{
    fflush(stdout);
    fprintf(stderr, "HELP!: syntax error, enter a semicolon to clear\n");
    return 0;
}
