//
// LiDIA - a library for computational number theory
//   Copyright (c) 1995 by the LiDIA Group
//
// File        : bigint.h 
// Author      : Thomas Papanikolaou (TP)
// Last change : TP, Feb 7 1995, initial version
//

#ifndef LIDIA_BIGINT_H
#define LIDIA_BIGINT_H

/**
**-------------------------------------------------------------------
** include here the include file of the C multiple precision integers
** and define type_name to be the integer type we have. 
** 
** #include "integer_include_file.h"
** #define integer_type_name  	your_integer_type_name 
**
**/

#include <LiDIA/gmp.h>
#include <LiDIA/gmp-impl.h>
#include <LiDIA/longlong.h>
#include <LiDIA/gmp-mparam.h>

#define integer_type_name       MP_INT
#define base_digit      	DigitType 

/**
** You must now write the implementation of the following in the file
** bigint.c. Please, change nothing after this point.
**-------------------------------------------------------------------
**/

#include <LiDIA/lidia.h>

class bigint
{

  /**
  ** the C type we use to represent a bigint
  **/

  integer_type_name I;

            public:

  /**
  ** constructors and destructor; we could leave out some of these
  **/

            bigint();
            bigint(int i);
            bigint(long l);
            bigint(unsigned long ul);
            bigint(double d);
            bigint(const bigint & a);
           ~bigint();

  /**
  ** inline member functions
  **/
  
  int bit(unsigned int i) const;
  int length() const;
  int bit_length() const;
  int sign() const;
  int is_odd() const;
  int is_even() const;
  int is_positive() const;
  int is_negative() const;
  int is_zero() const;
  int is_gt_zero() const;
  int is_ge_zero() const;
  int is_lt_zero() const;
  int is_le_zero() const;
  int is_one() const;
  int intify(int &i) const;
  int longify(long &i) const;
  int abs_compare(const bigint & a) const;
  int compare(const bigint & a) const;
  unsigned long most_significant_digit() const;
  unsigned long least_significant_digit() const;

  /**
  ** the next two definitions are needed by bigfloat
  **/

  //static const unsigned long radix();
  static const double radix();
  static const int bits_per_digit();

  void absolute_value();
  void negate();
  void assign_zero();
  void assign_one();
  void assign(int i);
  void assign(long ui);
  void assign(unsigned long ui);
  void assign(const bigint & a);
  void multiply_by_2();
  void divide_by_2();

  /**
  ** type checking
  **/

  friend int is_char(const bigint & a);
  friend int is_uchar(const bigint & a);
  friend int is_short(const bigint & a);
  friend int is_ushort(const bigint & a);
  friend int is_int(const bigint & a);
  friend int is_uint(const bigint & a);
  friend int is_long(const bigint & a);
  friend int is_ulong(const bigint & a);

  /**
  ** assignments
  **/

  int operator = (int i);
  long operator = (long l);
  unsigned long operator = (unsigned long ul);
  double operator = (double d);
  bigint & operator = (const bigint & a);

  /**
  ** comparisons
  **/

  friend int operator == (const bigint & a, const bigint & b);
  friend int operator != (const bigint & a, const bigint & b);
  friend int operator > (const bigint & a, const bigint & b);
  friend int operator >= (const bigint & a, const bigint & b);
  friend int operator < (const bigint & a, const bigint & b);
  friend int operator <= (const bigint & a, const bigint & b);

  /**
  ** operator overloading
  **/

  friend bigint operator - (const bigint & a);
  friend bigint operator + (const bigint & a, const bigint & b);
  friend bigint operator - (const bigint & a, const bigint & b);
  friend bigint operator *(const bigint & a, const bigint & b);
  friend bigint operator / (const bigint & a, const bigint & b);
  friend bigint operator % (const bigint & a, const bigint & b);
  friend bigint operator << (const bigint & a, unsigned long u);
  friend bigint operator >> (const bigint & a, unsigned long u);
  friend bigint operator & (const bigint & a, const bigint & b);
  friend bigint operator | (const bigint & a, const bigint & b);
  friend bigint operator ^ (const bigint & a, const bigint & b);

  bigint & operator += (const bigint & a);
  bigint & operator -= (const bigint & a);
  bigint & operator *= (const bigint & a);
  bigint & operator /= (const bigint & a);
  bigint & operator %= (const bigint & a);
  bigint & operator <<= (unsigned long ui);
  bigint & operator >>= (unsigned long ui);
  bigint & operator &= (const bigint & a);
  bigint & operator |= (const bigint & a);
  bigint & operator ^= (const bigint & a);

  bigint operator~ ();
  bigint & operator++ ();
  bigint & operator-- ();
  int operator ! ();

  /**
  ** Procedural versions
  **/

  friend void negate(bigint & a, const bigint & b);
  friend void add(bigint & c, const bigint & a, const bigint & b);
  friend void subtract(bigint & c, const bigint & a, const bigint & b);
  friend void multiply(bigint & c, const bigint & a, const bigint & b);
  friend void divide(bigint & c, const bigint & a, const bigint & b);
  friend void remainder(bigint & c, const bigint & a, const bigint & b);
  friend void div_rem(bigint & q, bigint & r, const bigint & a, const bigint & b);
  friend void shift_left(bigint & c, const bigint & a, long ui);
  friend void shift_right(bigint & c, const bigint & a, long ui);
  friend void power(bigint & c, const bigint & a, const bigint & b);
  friend void power(bigint & c, const bigint & a, long i);
  friend void and(bigint & c, const bigint & a, const bigint & b);
  friend void or(bigint & c, const bigint & a, const bigint & b);
  friend void xor(bigint & c, const bigint & a, const bigint & b);
  friend void not(bigint & c, const bigint & a);
  friend void inc(bigint & c);
  friend void dec(bigint & c);

  friend void add(bigint & c, const bigint & a, long i);
  friend void subtract(bigint & c, const bigint & a, long i);
  friend void multiply(bigint & c, const bigint & a, long i);
  friend void divide(bigint & c, const bigint & a, long i);
  friend void remainder(long &r, const bigint & a, long i);
  friend long remainder(const bigint & a, long i);
  friend void div_rem(bigint & q, long &r, const bigint & a, long i);

  /**
  ** gcd's
  **/

  friend bigint gcd(const bigint & a, const bigint & b);
  friend bigint bgcd(const bigint & a, const bigint & b);
  friend bigint dgcd(const bigint & a, const bigint & b);
  friend bigint xgcd(bigint & u, bigint & v, const bigint & a, const bigint & b);
  friend bigint xgcd_left(bigint & u, const bigint & a, const bigint & b);
  friend bigint xgcd_right(bigint & v, const bigint & a, const bigint & b);

  /**
  ** functions
  **/

  friend bigint abs(const bigint & a);
  friend void seed(const bigint & a);
  friend bigint randomize(const bigint & a);
  friend double dbl(const bigint & a);

  friend void sqrt(bigint & a, const bigint & b);
  friend void square(bigint & a, const bigint & b);
  friend void swap(bigint & a, bigint & b);

  /**
  ** input / output
  **/

  friend istream & operator >> (istream & in, bigint & a);
  friend ostream & operator << (ostream & out, const bigint & a);

  friend int string_to_bigint(char *s, bigint & a);
  friend int bigint_to_string(const bigint & a, char *s);

  /**
  ** using fread/fwrite
  **/

  void read_from_file(FILE * fp);
  void write_to_file(FILE * fp);

  /**
  ** using fscanf/fprintf
  **/

  void scan_from_file(FILE * fp);
  void print_to_file(FILE * fp);

};

#include <LiDIA/interface_lib.h>

#endif
