//
// LiDIA - a library for computational number theory
//   Copyright (c) 1994, 1995 by the LiDIA Group
//
// File        : bm_base.c
// Author      : Patrick Theobald (PT)
// Last change : PT, Jun 25 1995, initial version
//

/*
$Id: bm_base.c,v 1.5 1995/08/07 08:30:53 lidiaadm Exp $
*/

#include <LiDIA/bigint_matrix.h>

/**
 ** constructors
 **/

 bigint_matrix::bigint_matrix()
{
  debug_handler("bigint_matrix", "in constructor bigint_matrix()");
  
  print_mode = BEAUTY_MODE;
  rows = 1;
  columns = 1;
  
  value = new bigint *[1];
  memory_handler(value, "bigint_matrix", "bigint_matrix() :: "
		 "Error in memory allocation (value)");
  value[0] = new bigint[1];
  memory_handler(value[0], "bigint_matrix", "bigint_matrix() :: "
		 "Error in memory allocation (value[0])");
}

 bigint_matrix::bigint_matrix(long r)
{
  debug_handler("bigint_matrix", "in constructor bigint_matrix(long)");
  
  if (r < 1)
    lidia_error_handler("bigint_matrix", "bigint_matrix(long) :: "
		  "Parameter must be greater than zero !");
  
  print_mode = BEAUTY_MODE;
  rows = r;
  columns = 1;

  register long i;
  value = new bigint *[r];
  memory_handler(value, "bigint_matrix", "bigint_matrix(long) :: "
		 "Error in memory allocation (value)");
  for (i = 0; i < r; i++)
    {
      value[i] = new bigint[1];
      memory_handler(value[i], "bigint_matrix", "bigint_matrix(long) :: "
		     "Error in memory allocation (value[i])");
    }
}

 bigint_matrix::bigint_matrix(long r, long c)
{
  debug_handler("bigint_matrix", "in constructor "
		"bigint_matrix(long, long)");

  if (r < 1 || c < 1)
    lidia_error_handler("bigint_matrix", "bigint_matrix(long, long) :: "
		  "Parameter must be greater than zero !");
  
  print_mode = BEAUTY_MODE;
  rows = r;
  columns = c;
  
  register long i;
  value = new bigint *[r];
  memory_handler(value, "bigint_matrix", "bigint_matrix(long, long) :: "
		 "Error in memory allocation (value)");
  for (i = 0; i < r; i++)
    {
      value[i] = new bigint[c];
      memory_handler(value[i], "bigint_matrix", "bigint_matrix(long, long) :: "
		     "Error in memory allocation (value[i])");
    }
}

 bigint_matrix::bigint_matrix(long r, long c, bigint ** A)
{
  debug_handler("bigint_matrix", "in constructor "
		"bigint_matrix(long, long, bigint **)");
  
  if (r < 1 || c < 1 || A==NULL)
    lidia_error_handler("bigint_matrix", "bigint_matrix(long, long, bigint **) :: "
		  "Error in parameter !");

  print_mode = BEAUTY_MODE;
  rows = r;
  columns = c;

  register long i, j;
  bigint *tmp, *Atmp;
  value = new bigint *[r];
  memory_handler(value, "bigint_matrix", 
		 "bigint_matrix(long, long, bigint **) :: "
		 "Error in memory allocation (value)");
  for (i = 0; i < r; i++)
    {
      tmp = new bigint[c];
      Atmp = A[i];
      memory_handler(tmp, "bigint_matrix",
		     "bigint_matrix(long, long, bigint **) :: "
		     "Error in memory allocation (tmp)");
      for (j = 0; j < c; j++)
	tmp[j].assign(Atmp[j]);
      value[i] = tmp;
    }
}

 bigint_matrix::bigint_matrix(long r, long c, long **A)
{
  debug_handler("bigint_matrix", "in constructor "
		"bigint_matrix(long, long, long **)");
  
  if (r < 1 || c < 1 || A==NULL)
    lidia_error_handler("bigint_matrix", "constructor :: Error in parameter !");
  
  print_mode = BEAUTY_MODE;
  rows = r;
  columns = c;
  
  register long i, j;
  long *Atmp;
  bigint *tmp;
  value = new bigint *[r];
  memory_handler(value, "bigint_matrix",
		 "bigint_matrix(long, long, long **) :: "
		 "Error in memory allocation (value)");
  for (i = 0; i < r; i++)
    {
      tmp = new bigint[c];
      Atmp = A[i];
      memory_handler(tmp, "bigint_matrix",
		     "bigint_matrix(long, long, long **) :: "
		     "Error in memory allocation (tmp)");
      for (j = 0; j < c; j++)
	tmp[j].assign(Atmp[j]);
      value[i] = tmp;
    }
}

 bigint_matrix::bigint_matrix(const bigint_matrix & M)
{
  debug_handler("bigint_matrix", "in constructor "
		"bigint_matrix(const bigint_matrix &)");
  
  print_mode = M.print_mode;
  rows = M.rows;
  columns = M.columns;
  
  register long i, j;
  bigint *tmp, *Mtmp;
  value = new bigint *[rows];
  memory_handler(value, "bigint_matrix", 
		 "bigint_matrix(const bigint_matrix &) :: "
		 "Error in memory allocation (value)");
  for (i = 0; i < rows; i++)
    {
      tmp = new bigint[columns];
      Mtmp = M.value[i];
      memory_handler(tmp, "bigint_matrix",
		     "bigint_matrix(const bigint_matrix &) :: "
		     "Error in memory allocation (tmp)");
      for (j = 0; j < columns; j++)
	tmp[j].assign(Mtmp[j]);
      value[i] = tmp;
    }
}

/**
 ** destructor
 **/

 bigint_matrix::~bigint_matrix()
{
  debug_handler("bigint_matrix", "in destructor ~bigint_matrix()");
  
  register long i;
  for (i = 0; i < rows; i++)
    delete[] value[i];
  delete[] value;
}

/**
 ** Operator Input / Output
 **/

ostream & operator << (ostream & out, const bigint_matrix & M)
{
  /**
   ** DESCRIPTION: Output to stream 
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", 
		  "in operator <<(ostream &, const bigint_matrix &)", 2);
  
  switch(M.print_mode)
    {
    case BEAUTY_MODE:   
      register long i, j;
      bigint *Mtmp;
      
      for (i = 0; i < M.rows; i++)
	{
	  Mtmp = M.value[i];
	  out << "\n( " << flush;
	  for (j = 0; j < M.columns; j++)
	    out << Mtmp[j] << " " << flush;
	  out << ")" << flush;
	}
      out << "\n" << flush;
      break;
    case LIDIA_MODE:
      M.write_to_stream(out);
      break;
    case GP_MODE:
      M.write_to_gp(out);
      break;
    case MAPLE_MODE:
      M.write_to_maple(out);
      break;
    case MATHEMATICA_MODE:
      M.write_to_mathematica(out);
      break;
    case KANT_MODE:
      M.write_to_kash(out);
      break;
    }
  return out;
}

istream & operator >> (istream & in, bigint_matrix & M)
{
  /**
   ** DESCRIPTION: Input from stream
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix",
		  "in operator >>(istream &, bigint_matrix &)", 2);
  
  char c;
  in >> c;
  in.putback(c);
  
  switch(c)
    {
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
      bigint *Mtmp;
      long i, j;
      in >> i >> j;
      if (i != M.rows)
	M.set_no_of_rows(i);
      if (j != M.columns)
	M.set_no_of_columns(j);
      for (i = 0; i < M.rows; i++)
	{
	  Mtmp = M.value[i];
	  for (j = 0; j < M.columns; j++)
	    in >> Mtmp[j];
	}
      break;
    case '[':
      M.read_from_gp(in);
      break;
    case 'a':
      M.read_from_maple(in);
      break;
    case '{':
      M.read_from_mathematica(in);
      break;
    case 'M':
      M.read_from_kash(in);
      break;
    default:
      lidia_error_handler("bigint_matrix", "operator >> :: "
		    "Unknown matrix format !!");
    }
  return in;
}

/** 
 ** set print_mode / get print_mode
 **/

void bigint_matrix::
set_print_mode(int art)
{
  /**
   ** DESCRIPTION: A.set_print_mode(art);
   **              => A.print_mode = art 
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix","in member - function "
		  "set_print_mode(int)", 2);

  print_mode = (art < 0 || art > 5) ? 0 : art;
}

int bigint_matrix::
get_print_mode()
{
  /**
   ** DESCRIPTION: A.get_print_mode() returns the print_mode 
   **              of matrix A. 
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix","in member - function "
		  "get_print_mode()", 2);
  
  return print_mode;
}

/**
 ** stream handling - LIDIA 
 **/

void bigint_matrix::
write_to_stream(ostream & out) const
{
  /**
   ** DESCRIPTION: A.write_to_stream(out);
   **              => writes matrix A to stream out in LiDIA - format.
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member - function "
		  "write_to_stream(ostream &)", 2);
  
  register long i, j, l = columns - 1;
  bigint *tmp;

  out << rows << " " << columns << "\n" << flush;
  for (i = 0; i < rows; i++)
    {
      tmp=value[i];
      for (j = 0; j < l; j++)
	out << tmp[j] << " " << flush;
      out << tmp[j] << "\n" << flush;
    }
}

void
write_to_stream( const bigint_matrix & A, ostream & out)
{
  debug_handler_l("bigint_matrix", "in function "
		  "write_to_stream(const bigint_matrix &, ostream &)", 2);
  
  A.write_to_stream(out);
}

void bigint_matrix::
read_from_stream(istream & in)
{
  /**
   ** DESCRIPTION: A.read_from_stream(in);
   **              => reads matrix A in LiDIA - format from stream in .
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member - function "
		  "read_from_stream(istream &)", 2);
  
  long i, j;
  bigint *tmp;

  in >>  i >> j;
  if (i != rows)
    set_no_of_rows(i);
  if (j != columns)
    set_no_of_columns(j);
  
  for (i = 0; i < rows; i++)
    {
      tmp = value[i];
      for (j = 0; j < columns; j++)
	in >> tmp[j];
    }
}

void
read_from_stream(bigint_matrix & A, istream &in)
{
  debug_handler_l("bigint_matrix", "in function "
		  "read_from_stream(bigint_matrix &, istream &)", 2);
  
  A.read_from_stream(in);
}

/**
 ** stream handling - MATHEMATICA
 **/

void bigint_matrix::
write_to_mathematica(ostream &out) const
{
  /**
   ** DESCRIPTION: A.write_to_mathematica(out);
   **              => writes matrix A to stream out in mathematica format
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in member - function "
		  "write_to_mathematica(ostream &)", 2);
  
  register long i, j, l=columns-1, k=rows-1;
  bigint *tmp;

  out << "{" << flush;
  for (i = 0; i < rows; i++)
    {
      tmp = value[i];
      out << "{" << flush;
      for (j = 0; j < columns; j++)
	{
	  out << tmp[j] << flush;
	  if (j != l)
	    out << "," << flush;
          else
	    if (i != k)
	      out << "},\n" << flush;
        }
    }
  out << "}}" << endl << flush;
}

void bigint_matrix::
read_from_mathematica(istream & dz)
{
  /**
   ** DESCRIPTION: A.read_from_mathematica(dz);
   **              => reads matrix A from stream dz in mathematica format
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member - function "
		  "read_from_mathematica(istream &dz)", 2);
  
  char c = 0;

  streampos p = dz.tellg();  /* read position */
  
  dz >> ws >> c;
  if (c != '{')
    lidia_error_handler("bigint_matrix", "read_from_mathematica :: "
		  "Error in format !");
  
  register long i1 = 0, i2 = 0, i3 = 1, i, j;
  while (!dz.eof() && i3 != 0)
    {
      dz >> ws >> c; 
      if (c == ',')
	i1++;
      if (c == '{')
	{
	  i2++;i3++;
	}
      if (c == '}')
	i3--;
    }
  i1 += i2;
  i1 = i1 / i2;
  
  if (rows != i2)
    set_no_of_rows(i2);
  if (columns != i1)
    set_no_of_columns(i1);
  
  dz.seekg(p,ios::beg);
  
  dz >> ws >> c >> ws >> c;
  for (i = 0; i < i2; i++)
    {
      for (j = 0; j < i1; j++)
	dz >> ws >> value[i][j] >> ws >> c;
      dz >> ws >> c >> ws >> c;
    }
}

/**
 ** stream handling - MAPLE
 **/

void bigint_matrix::
write_to_maple(ostream &out) const
{
  /**
   ** DESCRIPTION: A.write_to_maple(out);
   **              => writes matrix A to stream out in maple format
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in member - function "
		  "write_to_maple(ostream &)", 2);
  
  register int i, j, l=(int)columns-1, k=(int)rows-1;
  bigint *tmp;
  
  out << "array(1 .. " << rows << ", 1 .. " << columns << ",[" << flush;
  
  for (i = 0; i < rows; i++)
    {
      tmp = value[i];
      for (j = 0; j < columns; j++)
	{
	  out << "(" << i+1 << ", " << j+1 << ")=" << tmp[j] << flush;
	  if (!(j == l && i == k))
	    out << "," << flush;
        }
    }
  out << "]); \n" << flush;
}

void bigint_matrix::
read_from_maple(istream &dz)
{
  /**
   ** DESCRIPTION: A.read_from_maple(dz);
   **              => reads matrix A from stream dz in maple format
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member - function "
		  "read_from_maple(istream &dz)", 2);
  
  char c = 0;
  while(c != '(' && dz.good())
    dz >> ws >> c;
  
  if (dz.bad())
    lidia_error_handler("bigint_matrix","read_from_maple :: "
		  "Error in format !");
  
  register long i, j;
  long startr, startc, endr, endc, x, y;
  
  /* read row dimension */
  dz >> startr >> ws >> c >> ws >> c >> endr >> ws >> c;
  
  /* read column dimension */ 
  dz >> startc >> ws >> c >> ws >> c >> endc >> ws >> c;
  
  endr = endr-startr+1;
  if (endr < 0)
    endr = -endr;
  endc = endc-startc+1;
  if (endc < 0)
    endc = -endc;

  if (columns != endc)
    set_no_of_columns(endc);
  if (rows != endr)
    set_no_of_rows(endr);
  
  dz >> ws >> c; 
		      
  bigint TMP;
  for (i=0;i<endc;i++)
    for (j=0;j<endr;j++) 
      {
	dz >> ws >> c >> x >> ws >> c >> y; 
	dz >> ws >> c >> ws >> c >> ws >> TMP >> ws >> c;
	value[x-startr][y-startc].assign(TMP);
      }
  dz >> ws >> c >> ws >> c;
}

/**
 ** stream handling - PARI
 **/

void bigint_matrix::
write_to_gp(ostream & out) const
{
  /**
   ** DESCRIPTION: A.write_to_gp(out);
   **              => writes matrix A to stream out in PARI format
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in member - function "
		  "write_to_gp(ostream &)",2);
  
  register long i, j, l=columns-1, k=rows-1;
  bigint *tmp;

  out << "[" << flush;
  for (i = 0; i < rows; i++)
    {
      tmp = value[i];
      for (j = 0; j < columns; j++)
	{
	  out << tmp[j] << flush;
	  if (j != l)
	    out << ", " << flush;
	}
      if (i != k)
	out << "; " << flush;
    }
  out << "]" << flush;
}

void bigint_matrix::
read_from_gp(istream &dz)
{
  /**
   ** DESCRIPTION: A.read_from_gp(dz);
   **              => reads matrix A from stream dz in PARI format
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member - function "
		  "read_from_gp(istream &)",2);
  
  char c = 0;
  
  streampos p = dz.tellg();
  
  dz >> ws >> c;
  if (c != '[')
    lidia_error_handler("bigint_matrix", "read_from_gp :: Error in format !");
  
  register long i, j, i1 = 0, i2 = 0, i3 = 1;  
  while (!dz.eof() && i3 != 0)
    {
      dz >> ws >> c; 
      if (c == ',')
	i1++;
      if (c == ';')
	i2++;
      if (c == ']')
	i3--;
    }
  i2++;
  i1 += i2;
  i1 = i1 / i2;
  
  if (rows != i2)
    set_no_of_rows(i2);
  if (columns != i1)
    set_no_of_columns(i1);

  dz.seekg(p,ios::beg);
  
  dz >> ws >> c;
  
  for (i = 0; i < i2; i++)
    for (j = 0; j < i1; j++)
      dz >> ws >> value[i][j] >> ws >> c;
}

/**
 ** stream handling - KANT
 **/

void bigint_matrix::
write_to_kash(ostream &out) const
{
  /**
   ** DESCRIPTION: A.write_to_kash(out);
   **              => writes matrix A to stream out in kash format
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in member - function "
		  "write_to_kash(ostream &)", 2);
  
  register long i, j, l=columns-1, k=rows-1;
  bigint *tmp;

  out << "Mat(Z,[" << flush;
  for (i = 0; i < rows; i++)
    {
      tmp = value[i];
      out << "[" << flush;
      for (j = 0; j < columns; j++)
	{
	  out << tmp[j] << flush;
	  if (j != l)
	    out << "," << flush;
          else
	    if (i != k)
	      out << "],\n" << flush;
        }
    }
  out << "]]);" << endl << flush;
}

void bigint_matrix::
read_from_kash(istream & dz)
{
  /**
   ** DESCRIPTION: A.read_from_kash(dz);
   **              => reads matrix A from stream dz in kash format
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member - function "
		  "read_from_kash(istream & dz)", 2);
  
  char c = 0;
  
  dz >> ws >> c;
  if (c != 'M')
    lidia_error_handler("bigint_matrix", "read_from_mathematica :: "
		  "Error in format !");

  dz >> ws >> c >> ws >> c >> ws >> c >> ws >> c >> ws >> c >> ws >> c;

  streampos p = dz.tellg();  /* read position */

  register long i1 = 0, i2 = 0, i3 = 1, i, j;
  while (!dz.eof() && i3 != 0)
    {
      dz >> ws >> c; 

      if (c == ',')
	i1++;
      if (c == '[')
	{
	  i2++;i3++;
	}
      if (c == ']')
	i3--;
    }
  i1 += i2;
  i1 = i1 / i2;

  if (rows != i2)
    set_no_of_rows(i2);
  if (columns != i1)
    set_no_of_columns(i1);
  
  dz.seekg(p,ios::beg);
  
  dz >> ws >> c;

  for (i = 0; i < i2; i++)
    {
      for (j = 0; j < i1; j++)
	dz >> ws >> value[i][j] >> ws >> c;
      dz >> ws >> c >> ws >> c;
    }
}

/**
 ** access functions
 **/

void bigint_matrix::
sto(long x, long y, const bigint & e)
{
  /**
   ** DESCRIPTION: A.sto(x,y,e) stores e at position (x,y) in matrix A.
   ** ERROR: x < 0 or x >= rows or y < 0 or y >= columns
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in member - function "
		  "sto(long, long, const bigint &)", 2);
  
  if (x<0 || x>=rows || y < 0 || y >= columns)
    lidia_error_handler("bigint_matrix", "sto :: Parameter out of range !");
  value[x][y].assign(e);
}

void bigint_matrix::
sto_column(const bigint * v, long l, long j)
{
  /**
   ** DESCRIPTION: A.sto_column(v,l,j) stores v[0],...,v[l-1] in column j 
   **              of matrix A.
   ** ERROR: j > A.columns or l > A.rows or j < 0 or l < 0
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member - function "
		  "sto_column(const bigint *, long, long)", 2);
  
  register long k;
  if (j >= columns || rows < l || j < 0 || l < 0)
    lidia_error_handler("bigint_matrix", "sto_column :: Parameter out of range !");
  for (k = 0; k < l; k++)
    value[k][j].assign(v[k]);
}

void bigint_matrix::
sto_row(const bigint * v, long l, long i)
{
  /**
   ** DESCRIPTION: A.sto_row(v,l,j) stores v[0],...,v[l-1] in row i of matrix A.
   ** ERROR: i > A.rows or l > A.columns or i < 0 or l < 0
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member - function "
		  "sto_row(const bigint *, long, long)", 2);
  
  register long k;
  if (i >= rows || l > columns || i < 0 || l < 0)
    lidia_error_handler("bigint_matrix", "sto_row :: Parameter out of range !");
  bigint *tmp = value[i];
  for (k = 0; k < l; k++)
    tmp[k].assign(v[k]);
}

bigint bigint_matrix::
operator () (long x, long y) const
{
  /**
   ** DESCRIPTION: A(x,y) = value at position (x,y) in matrix A
   ** ERROR: x < 0 or y < 0 or x >= A.rows or y >= A.columns
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in operator () (long, long)", 2);
  
  if (x < 0 || y < 0 || x >= rows || y >= columns)
    lidia_error_handler("bigint_matrix", "operator () :: Parameter out of range !");
  return value[x][y];
}

bigint bigint_matrix::
member(long x, long y) const
{
  /**
   ** DESCRIPTION: A.member(x,y) = value at position (x,y) in matrix A
   ** ERROR: x < 0 or y < 0 or x >= A.rows or y >= A.columns
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member - function "
		  "member(long, long)", 2);
  
  if (x < 0 || y < 0 || x >= rows || y >= columns)
    lidia_error_handler("bigint_matrix", "member :: Parameter out of range !");
  return value[x][y];
}

bigint *bigint_matrix::
operator() (long i) const
{
  /**
   ** DESCRIPTION: RES = A(i);
   **              => RES[0] = A.value[0][i],...,RES[A.rows-1] 
   **                        = A.value[A.rows-1][i]
   ** ERROR: 0 > i or i >= A.columns
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in operator ()(long)",2);
  
  return column(i);
}

bigint *bigint_matrix::
column(long i) const
{
  /**
   ** DESCRIPTION: RES = A.column(i);
   **              => RES[0] = A.value[0][i],...,
   **                 RES[A.rows-1] = A.value[A.rows-1][i]
   ** ERROR: 0 > i or i >= A.columns
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in member - function column(long)", 2);
  
  if (i >= columns || i < 0)
    lidia_error_handler("bigint_matrix", " column :: Parameter out of range");
  bigint *RES = new bigint[rows];
  memory_handler(RES, "bigint_matrix", "column :: "
		 "Error in memory allocation (RES)");
  register long k;
  for (k = 0; k < rows; k++)
    RES[k].assign(value[k][i]);
  return RES;
}

bigint *bigint_matrix::
operator[] (long i) const
{
  /**
   ** DESCRIPTION: RES = A[i];
   **              => RES[0] = A.value[i][0],...,RES[A.columns-1] 
   **                        = A.value[i][A.columns-1]
   ** ERROR: 0 > i or i >= A.rows
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in operator [](long)", 2);
  
  return row(i);
}

bigint *bigint_matrix::
row(long i) const
{
  /**
   ** DESCRIPTION: RES = A.row(i);
   **              => RES[0] = A.value[i][0],...,RES[A.columns-1] 
   **                        = A.value[i][A.columns-1]
   ** ERROR: 0 > i or i >= A.rows
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in member - function row(long)", 2);
  
  if (i >= rows || i < 0)
    lidia_error_handler("bigint_matrix", "row :: Parameter out of range !");
  bigint *RES = new bigint[columns];
  memory_handler(RES, "bigint_matrix", "row :: "
		 "Error in memory allocation (RES)");
  register long k;
  bigint *tmp = value[i];
  for (k = 0; k < columns; k++)
    RES[k] = tmp[k];
  return RES;
}

/**
 ** divide functions
 **/

void bigint_matrix::
divide(bigint_matrix & A, bigint_matrix & B, bigint_matrix & C, bigint_matrix & D) const
{
  /**
   ** DESCRIPTION: E.divide(A,B,C,D);
   **              => divides matrix E longo the submatrices A,B,C,D
   **                     (A B)
   **              => E = (C D)
   ** ERROR: A.columns > columns or A.rows > rows or
   **        B.columns > columns or B.rows > rows or 
   **        C.columns > columns or C.rows > rows or
   **        D.columns > columns or D.rows > rows 
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in member - function "
		  "divide(bigint_matrix &, bigint_matrix &, "
		  "bigint_matrix &, bigint_matrix &)", 2);
  
  register long i,j,diff,diff1;
  bigint *tmp, *tmp1;
  if (A.rows > rows || A.columns > columns ||
      B.rows > rows || B.columns > columns ||
      C.rows > rows || C.columns > columns ||
      D.rows > rows || D.columns > columns)
    lidia_error_handler("bigint_matrix", "divide :: Error in matrix dimension !");
  
  for (i=0;i<A.rows;i++)
    {
      tmp = value[i];
      tmp1 = A.value[i];
      for (j=0;j<A.columns;j++)
	tmp1[j].assign(tmp[j]);
    }
  
  diff = columns - B.columns;
  for (i=0;i<B.rows;i++)
    {
      tmp = value[i];
      tmp1 = B.value[i];
      for (j=0;j<B.columns;j++)
	tmp1[j].assign(tmp[diff+j]);
    }
  
  diff1 = rows-C.rows;
  for (i=0;i<C.rows;i++)
    {
      tmp = value[diff1+i];
      tmp1 = C.value[i];
      for (j=0;j<C.columns;j++)
	tmp1[j].assign(tmp[j]);
    }
  
  diff = columns-D.columns;
  diff1 = rows-D.rows;
  for (i=0;i<D.rows;i++)
    {
      tmp = value[diff1+i];
      tmp1 = D.value[i];
      for (j=0;j<B.columns;j++)
	tmp1[j].assign(tmp[diff+j]);
    }
}

void bigint_matrix::
divide_h(bigint_matrix & A, bigint_matrix & B) const
{
  /**
   ** DESCRIPTION: C.divide_h(A,B);
   **              => divides matrix C longo the submatrices A,B
   **              => C = (A B)
   ** ERROR: A.columns > columns or A.rows > rows or
   **        B.columns > columns or A.rows > rows
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member - function "
		  "divide_h(bigint_matrix &, bigint_matrix &)", 2);

  register long i,j,diff;
  bigint *tmp, *tmp1;
  if (A.rows > rows || A.columns > columns ||
      B.rows > rows || B.columns > columns)
    lidia_error_handler("bigint_matrix", "divide_h :: Error in matrix dimension !");
  
  for (i=0;i<A.rows;i++)
    {
      tmp = value[i];
      tmp1 = A.value[i];
      for (j=0;j<A.columns;j++)
	tmp1[j].assign(tmp[j]);
    }
	
  diff = columns-B.columns;
  for (i=0;i<B.rows;i++)
    {
      tmp = value[i];
      tmp1 = B.value[i];
      for (j=0;j<B.columns;j++)
	tmp1[j].assign(tmp[diff+j]);
    }
}

void bigint_matrix::
divide_v(bigint_matrix & A, bigint_matrix & B) const
{
  /**
   ** DESCRIPTION: C.divide_v(A,B);
   **              => divides matrix C longo the submatrices A,B
   **                     (A)
   **              => C = (B)
   ** ERROR: A.columns > columns or A.rows > rows or
   **        B.columns > columns or A.rows > rows 
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in member - function "
		  "divide_v(bigint_matrix&, bigint_matrix&)", 2);

  register long i,j,diff;
  bigint *tmp, *tmp1;
  if (A.rows > rows || A.columns > columns ||
      B.rows > rows || B.columns > columns)
    lidia_error_handler("bigint_matrix", "divide_v :: Error in matrix dimension !");
  
  for (i=0;i<A.rows;i++)
    {
      tmp = value[i];
      tmp1 = A.value[i];
      for (j=0;j<A.columns;j++)
	tmp1[j].assign(tmp[j]);
    }
  
  diff = rows-B.rows;
  for (i=0;i<B.rows;i++)
    {
      tmp = value[diff+i];
      tmp1 = B.value[i];
      for (j=0;j<B.columns;j++)
	tmp1[j].assign(tmp[j]);
    }
}

/**
 ** compose functions
 **/

void bigint_matrix::
compose(const bigint_matrix & A, const bigint_matrix & B, const bigint_matrix & C, const bigint_matrix & D)
{
  /**
   ** DESCRIPTION: E.compose(A,B,C,D)
   **              => compose the submatrices A,B,C,D to E
   **                     (A B)
   **              => E = (C D)
   ** ERROR: A.columns > columns or A.rows > rows or
   **        B.columns > columns or B.rows > rows or 
   **        C.columns > columns or C.rows > rows or 
   **        D.columns > columns or D.rows > rows
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in member - function "
		  "compose(const bigint_matrix &, const bigint_matrix &, "
		  "const bigint_matrix &, const bigint_matrix &)", 2);
  
  register long i,j,diff,diff1;
  bigint *tmp, *tmp1;
  if (A.rows > rows || A.columns > columns ||
      B.rows > rows || B.columns > columns ||
      C.rows > rows || C.columns > columns ||
      D.rows > rows || D.columns > columns)
    lidia_error_handler("bigint_matrix", "compose :: Error in matrix dimension !");
  
  for (i=0;i<A.rows;i++)
    {
      tmp = value[i];
      tmp1 = A.value[i];
      for (j=0;j<A.columns;j++)
	tmp[j].assign(tmp1[j]);
    }

  diff = columns - B.columns;
  for (i=0;i<B.rows;i++)
    {
      tmp = value[i];
      tmp1 = B.value[i];
      for (j=0;j<B.columns;j++)
	tmp[diff+j].assign(tmp1[j]);
    }
  
  diff1 = rows-C.rows;
  for (i=0;i<C.rows;i++)
    {
      tmp = value[diff1+i];
      tmp1 = C.value[i];
      for (j=0;j<C.columns;j++)
	tmp[j].assign(tmp1[j]);
    }
  
  diff = columns-D.columns;
  diff1 = rows-D.rows;
  for (i=0;i<D.rows;i++)
    {
      tmp = value[diff1+i];
      tmp1 = D.value[i];
      for (j=0;j<B.columns;j++)
	tmp[diff+j].assign(tmp1[j]);
    }
}

void bigint_matrix::
compose_h(const bigint_matrix & A, const bigint_matrix & B)
{
  /**
   ** DESCRIPTION: C.compose_h(A,B);
   **              => compose the submatrices A,B to C
   **              => C = (A B)
   ** ERROR: A.columns > columns or A.rows > rows or
   **        B.columns > columns or B.rows > rows
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member-function "
		  "compose_h(const bigint_matrix &, const bigint_matrix &)", 2);

  register long i,j,diff;
  bigint *tmp, *tmp1;
  if (A.rows > rows || A.columns > columns ||
      B.rows > rows || B.columns > columns)
    lidia_error_handler("bigint_matrix", "compose_h :: Error in matrix dimension !");
  
  for (i=0; i<A.rows; i++)
    {
      tmp = value[i];
      tmp1 = A.value[i];
      for (j=0;j<A.columns;j++)
	tmp[j].assign(tmp1[j]);
    }

  diff = columns - B.columns;
  for (i=0; i<B.rows; i++)
    {
      tmp = value[i];
      tmp1 = B.value[i];
      for (j=0;j<B.columns;j++)
	tmp[diff+j].assign(tmp1[j]);
    }
}

void bigint_matrix::
compose_v(const bigint_matrix & A, const bigint_matrix & B)
{
  /**
   ** DESCRIPTION: C.compose_v(A,B);
   **              => compose the submatrices A,B to C
   **                     (A)
   **              => C = (B)
   ** ERROR: A.columns > columns or A.rows > rows or
   **        B.columns > columns or B.rows > rows
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in member - function "
		  "compose_v(const bigint_matrix &, const bigint_matrix &)",2);

  register long i,j,diff;
  bigint *tmp, *tmp1;
  if (A.rows > rows || A.columns > columns ||
      B.rows > rows || B.columns > columns)
    lidia_error_handler("bigint_matrix", "compose_v :: Error in matrix dimension !");
  
  for (i=0; i<A.rows; i++)
    {
      tmp = value[i];
      tmp1 = A.value[i];
      for (j=0;j<A.columns;j++)
	tmp[j].assign(tmp1[j]);
    }
  
  diff = rows-B.rows;
  for (i=0;i<B.rows;i++)
    {
      tmp = value[diff+i];
      tmp1 = B.value[i];
      for (j=0;j<B.columns;j++)
	tmp[j].assign(tmp1[j]);
    }
}

/**
 ** exchange functions / swap functions
 **/

void 
swap(bigint_matrix &A, bigint_matrix &B)
{
  /**
   ** DESCRIPTION: swap(A,B) exchanges matrix A with matrix B.
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in function "
		  "swap(bigint_matrix &, bigint_matrix &)", 2);
  
  long TMP;
  bigint **tmp;
  
  /* swap no_of_columns */
  TMP = A.columns;
  A.columns = B.columns;
  B.columns = TMP;

  /* swap no_of_rows */
  TMP = A.rows;
  A.rows = B.rows;
  B.rows = TMP;

  /* swap values */
  tmp = A.value;
  A.value = B.value;
  B.value = tmp;
}

void bigint_matrix::
swap_columns(long i, long j)
{
  /**
   ** DESCRIPTION: A.swap_columns(i,j) exchanges column i with column j 
   **              in matrix A.
   ** ERROR: i < 0 or j < 0 or i >= A.columns or j >= A.columns
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member - function "
		  "swap_columns(long , long)", 2);
  
  if (i < 0 || i >= columns || j < 0 || j >= columns)
    lidia_error_handler("bigint_matrix", "swap_columns :: Error in arguments");
  
  register long z;
  bigint *tmp;
  for (z = 0; z < rows; z++)
  {
    tmp = value[z];
    swap(tmp[i], tmp[j]);
  }
}

void bigint_matrix::
swap_rows(long i, long j)
{
  /**
   ** DESCRIPTION: A.swap_rows(i,j) exchanges row i with row j in matrix A.
   ** ERROR: i < 0 or j < 0 or i >= A.rows or j >= A.rows
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in member - function "
		  "swap_rows(long , long)", 2);
  
  if (i < 0 || i >= rows || j < 0 || j >= rows)
    lidia_error_handler("bigint_matrix", "swap_rows :: Error in arguments");

  bigint *tmp = value[i];
  value[i] = value[j];
  value[j] = tmp;
}

/**
 ** structur functions
 **/

long bigint_matrix::
get_no_of_rows() const
{
  /**
   ** DESCRIPTION: A.get_no_of_rows() = number of rows of matrix A
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in member - function "
		  "get_no_of_rows()", 2);
  
  return rows;
}

long bigint_matrix::
get_no_of_columns() const
{
  /**
   ** DESCRIPTION: A.get_no_of_columns() = number of columns of matrix A
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in member - function "
		  "get_no_of_columns()", 2);

  return columns;
}

void bigint_matrix::
set_no_of_rows(long r)
{
  /**
   ** DESCRIPTION: A.set_no_of_rows(r) sets number of rows of matrix A to r.
   ** ERROR: r < 1
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in member - function "
		  "set_no_of_rows(long)", 2);
  
  if (r < 1)
    lidia_error_handler("bigint_matrix", "set_no_of_rows :: "
		  "Error in parameter !");
  
  if (r == rows)
    return;

  bigint **tmp = value;
  register long i, j;
  if (r < rows)
    {
      for (i = r; i < rows; i++)
	delete[] value[i];
      value = new bigint *[r];
      for (i = 0; i < r; i++)
	value[i] = tmp[i];
    }
  else
    {
      bigint *tmp1;
      value = new bigint *[r];
      memory_handler(value, "bigint_matrix", "set_no_of_rows :: "
		     "Error in memory allocation (value)");
      for (i = 0; i < rows; i++)
	value[i] = tmp[i];
      for (i = rows; i < r; i++)
	{
	  tmp1 = new bigint[columns];
	  memory_handler(tmp1, "bigint_matrix", "set_no_of_rows :: "
			 "Error in memory allocation (tmp1)");
	  for (j = 0; j < columns; j++)
	    tmp1[j].assign_zero();
	  value[i] = tmp1;
	}
    }
  rows = r;
  delete[] tmp;
}

void bigint_matrix::
set_no_of_columns(long c)
{
  /**
   ** DESCRIPTION: A.set_no_of_columns(c) sets number of columns of matrix A to c.
   ** ERROR: c < 1
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member - function "
		  "set_no_of_columns(long)", 2);

  if (c < 1)
    lidia_error_handler("bigint_matrix", "set_no_of_columns :: "
		  "Error in parameter !");

  if (columns == c)
    return;
  
  register long i, j;
  bigint *tmp, *tmp1;
  if (c < columns)
    {
      for (i = 0; i < rows; i++)
	{
	  tmp1 = value[i];
	  tmp = new bigint[c];
	  memory_handler(tmp, "bigint_matrix", "set_no_of_columns :: "
			 "Error in memory allocation (tmp)");
	  for (j = 0; j < c; j++)
	    tmp[j].assign(tmp1[j]);
	  value[i] = tmp;
	  delete[] tmp1;
	}
    }
  else
    {
      for (i = 0; i < rows; i++)
	{
	  tmp1 = value[i];
	  tmp = new bigint[c];
	  memory_handler(tmp, "bigint_matrix", "set_no_of_columns :: "
			 "Error in memory allocation (tmp)");
	  for (j = 0; j < columns; j++)
	    tmp[j].assign(tmp1[j]);
	  for (j = columns; j < c; j++)
	    tmp[j].assign_zero();
	  value[i] = tmp;
	  delete[] tmp1;
	}
    }
  columns = c;
}

/**
 ** randomize
 **/

void bigint_matrix::
randomize(const bigint & S)
{
  /**
   ** DESCRIPTION: RES.random(S);
   **              => 0 <= RES.value[i][j] <= S, i=0,...,RES.rows-1,
   **                 j=0,...,RES.columns-1
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member - function "
		  "randomize(const bigint &)", 2);
  
  register long i, j;
  bigint *tmp;
  seed(S);
  for (i = 0; i < rows; i++)
    {
      tmp = value[i];
      for (j = 0; j < columns; j++)
	tmp[j].assign(::randomize(S));
    }
}

bigint_matrix
randomize(long r, long c, const bigint & S)
{
  debug_handler_l("bigint_matrix", "in function "
		  "randomize(long, long, const bigint &)", 2);
  
  bigint_matrix RES(r, c);
  RES.randomize(S);
  return RES;
}
 
/**
 ** assign operator
 **/

bigint_matrix & bigint_matrix::
operator = (const bigint_matrix & M)
{
  debug_handler_l("bigint_matrix", "in operator =(const bigint_matrix &)", 2);
  
  assign(M);
  return *this;
}

void bigint_matrix::
assign(const bigint_matrix & M)
{
  /**
   ** DESCRIPTION: value[x][y] = M.value[x][y],
   **              x=0,...,rows-1, y=0,...,columns-1
   ** ERROR: rows != M.rows or columns != M.columns
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member - function "
		  "assign(const bigint_matrix &)", 2);
  
  register long i, j;
  if (rows != M.rows)
    set_no_of_rows(M.rows);
  if (columns != M.columns)
    set_no_of_columns(M.columns);
  
  bigint *tmp, *Mtmp;
  for (i = 0; i < rows; i++)
    {
      tmp = value[i];
      Mtmp = M.value[i];
      for (j = 0; j < columns; j++)
	tmp[j].assign(Mtmp[j]);
    }
}

void
assign(bigint_matrix & RES, const bigint_matrix & M)
{
  debug_handler_l("bigint_matrix", "in function "
		  "assign(bigint_matrix &, const bigint_matrix &)", 2);
  
  RES.assign(M);
}

/**
 ** transpose function
 **/

bigint_matrix bigint_matrix::
trans() const
{
  /**
   ** DESCRIPTION: AT = A.trans();
   **              => AT.value[x][y] = value[y][x],
   **              x=0,...,columns-1, y=0,...,rows-1
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in member - function "
		  "trans()", 2);

  register long i, j;
  bigint *tmp;
  bigint_matrix TRANS(columns, rows);
  for (i = 0; i < rows; i++)
    {
      tmp = value[i];
      for (j = 0; j < columns; j++)
	TRANS.value[j][i].assign(tmp[j]);
    }
  return TRANS;
}

bigint_matrix
trans(const bigint_matrix & A)
{
  debug_handler_l("bigint_matrix", "in function "
		  "trans(const bigint_matrix &)", 2);
  
  return A.trans();
}

void bigint_matrix::
trans(const bigint_matrix & A)
{
  /**
   ** DESCRIPTION: AT.trans(A);
   **              => AT.value[x][y] = A.value[y][x],
   **              x=0,...,columns-1, y=0,...,rows-1
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member - function "
		  "trans(const bigint_matrix &)",2);
  
  register long i, j;
  bigint *Atmp;
  
  if (columns != A.rows)
    set_no_of_columns(A.rows);
  if (rows != A.columns)
    set_no_of_rows(A.columns);
  
  for (i = 0; i < A.rows; i++)
    {
      Atmp = A.value[i];
      for (j = 0; j < A.columns; j++)
	value[j][i].assign(Atmp[j]);
    }
}

void 
trans(bigint_matrix & AT, const bigint_matrix & A)
{
  debug_handler_l("bigint_matrix", "in function "
		  "trans(bigint_matrix &, const bigint_matrix &)", 2);
  
  AT.trans(A);
}










