/*
Copyright (C) 1992,1993,1994,1995 Trusted Information Systems, Inc.

Export of this software from the United States of America or
Canada requires a specific license from the United States
Government.  This version of this software is not suitable for
export.

WITHIN THAT CONSTRAINT, the full text of the license agreement
that specifies the conditions under which this software may be
used is published in the file license.txt in the same directory
as that containing the TIS/MOSS source.

Trusted Information Systems makes no representation about the
suitability of this software for any purpose.  It is provided
"as is" without express or implied warranty.
*/


/* validate_user - return a list of userlist structures forming a path
 * from the argument user to a root.
 * 
 * Root is defined to be those user records identified by the alias
 * "me".  One of these user records must have been used to create a
 * "trusted" tag for the subordinate user.  The root record must be
 * self-signed with a trusted line.
 * 
 * Certificates will be followed if they exist, although the "final"
 * certificate must contained a trusted tag created by "root".
 */

#include "config.h"
#include "general.h"

#include "bbuf.h"
#include "crypto.h"
#include "cryptocb.h"
#include "key.h"
#include "new.h"
#include "userlist.h"
#include "util.h"


static int verify_trusted(user, issuer)
char  **user,
      **issuer;
{
    int             ret = BUNKNOWN;
    char           *trustedline = NULLCP,
                   *micalgstr = NULLCP,
                   *sigalgstr = NULLCP,
                   *sigvalstr = NULLCP,
	           *pubkey = NULLCP,
                  **av,
                   *cp;
    
    struct bbuf    *signbb = NULLBB,
                   *mic = NULLBB,
                   *emic = NULLBB,
                   *key = NULLBB,
                   *dsigbb = NULLBB;
    struct bbuf    bb;
    struct algent  *mic_ae = NULL_ALGENT;
    struct algid   *mic_aid = NULL_ALGID;

    /* get the trusted line */

    if ((av = tag_user(user, "trusted")) == NULLVP) {
	ret =  BNOSIG;
	goto cleanup;
    }

    /* Make a copy with which we can work */

    if ((trustedline = add2cp(NULLCP, *(av+1))) == NULLCP) {
	ret = BBADSIG;
	goto cleanup;
    }

    /* Find the signature components of the trusted line */

    if ((cp = INDEX(trustedline, ',')) == NULLCP) {
	ret = BBADSIG;
	goto cleanup;
    }
    micalgstr = ++cp;
    if ((cp = INDEX(micalgstr, ',')) == NULLCP) {
        ret = BBADSIG;
        goto cleanup;
    }
    *cp = NULLC;
    sigalgstr = ++cp;
    if ((cp = INDEX(sigalgstr, ',')) == NULLCP) {
        ret = BBADSIG;
        goto cleanup;
    }
    *cp = NULLC;
    sigvalstr = ++cp;

    /* flatten user record for signing */

    signbb = trustbb_user(user);

    /* get the hash/mic algorithm */

    if ((mic_ae = getalghdrstr(algorithms, micalgstr, MIC))
        == NULL_ALGENT) {
	ret = BBADSIG;
        goto cleanup;
    }

    /* hash the data */

    (void) gen_mic(mic_ae->code, NULLBB, signbb, &mic);

    /* encode the MIC */

    mic_aid = alloc_algid();
    mic_aid->alg = mic_ae->code;

    if (encode_mic(mic, mic_aid, &emic) != OK) {
	ret = BBADSIG;
	goto cleanup;
    }

    /* get issuer's public key and setup for use */

    if ((pubkey = key_user(issuer)) == NULLCP) {
	ret = BNOISS;
	goto cleanup;
    }

    bb.data = (unsigned char *)pubkey;
    bb.length = strlen(pubkey);
    (void) bdecode(&bb, &key);

    if (set_key(key) != OK) {
	ret = BBADSIG;
	goto cleanup;
    }

    /* verify the signature */

    bb.data = (unsigned char *)sigvalstr;
    bb.length = strlen(sigvalstr);
    (void) bdecode(&bb, &dsigbb);

    if (verify(dsigbb, emic) != OK) { /* XXX FIRST ARGUMENT WRONG */
	ret = BBADSIG;
	goto cleanup;
    }

    ret = OK;

 cleanup:

    FREE(trustedline);
    FREE(pubkey);
    FREE_ALGID(mic_aid);
    FREE_BBUF(dsigbb);
    FREE_BBUF(emic);
    FREE_BBUF(key);
    FREE_BBUF(mic);
    FREE_BBUF(signbb);

    return ret;
}


static int verify_cert(user, issuer)
char  **user,
      **issuer;
{
    int             ret = BUNKNOWN;
    struct bbuf     bbuf,
                   *dcert = NULLBB,
                   *dicert = NULLBB,
                   *dcrl = NULLBB;
    char          **av;

    /* get the user's and issuer's certificate and the crl */

    if ((av = tag_user(user, "certificate")) == NULLVP)
	goto cleanup;
    bbuf.data = (unsigned char *) *(av+1);
    bbuf.length = strlen(*(av+1));
    (void) bdecode(&bbuf, &dcert);

    if ((av = tag_user(issuer, "certificate")) == NULLVP)
	goto cleanup;
    bbuf.data = (unsigned char *) *(av+1);
    bbuf.length = strlen(*(av+1));
    (void) bdecode(&bbuf, &dicert);

    if ((av = tag_user(issuer, "crl")) != NULLVP) {
	bbuf.data = (unsigned char *) *(av+1);
	bbuf.length = strlen(*(av+1));
	(void) bdecode(&bbuf, &dcrl);
    }

    if (validate_cert(dcert, dicert, dcrl) != OK)
	goto cleanup;

    ret = dcert->status & ~(BBADCHN | BNOTOK);

 cleanup:

    FREE_BBUF(dcert);
    FREE_BBUF(dcrl);
    FREE_BBUF(dicert);

    return ret;
}


struct user_list *validate_user (user,  roots,  trusted)
char **user;
struct user_list *roots;
int *trusted;

{
    struct user_list *retpath = NULLUL;
    struct user_list *root;
    char **issuer = NULLVP;
    char **trustedav;
    char **issuerav;
    char *issuer_alias = NULLCP;
    char *comma, *userpk, *isspk;
    int atroot, attop;
       
    /* Check args */
    
    if (trusted == (int *)0) 
	goto cleanup;

    *trusted = FALSE;
    
    if (user == NULLVP)
        goto cleanup;

    /* prepare link in trust path */

    retpath = alloc_ul();
    retpath->user = dupav(user);

    /* Follow trusted path */ 

    if ((trustedav = tag_user(user, "trusted")) != NULLVP) {

	/* break out the issuer alias */

	if ((comma = INDEX(*(trustedav+1), ',')) == NULLCP) {
	    retpath->status = BBADSIG;
	    goto cleanup;
	}
	*comma = NULLC;
	issuer_alias = add2cp(NULLCP, *(trustedav+1));
	*comma = ',';

	/*  see if the issuer is on the root list */

	atroot = FALSE;
	for (root = roots; root != NULLUL; root = root->next)
	    if (tagval_user(root->user, "alias", issuer_alias) != NULLVP) {
		issuer = dupav(root->user);
		atroot = TRUE;
		break;
	    }

	/* If not, see of the issuer is in the database */

	if (!atroot) {
	    rewind_indexfile();
	    if ((issuer = get_tv_user("alias", issuer_alias)) == NULLVP) {
		retpath->status = BNOISS;
		goto cleanup;
	    }
	}

	/* Is the trusted line good? */

	if (verify_trusted(user, issuer) != OK) {
	    retpath->status = BBADSIG;
	    goto cleanup;
	}

    }

    /* Follow certificate path */
	
    else if (tag_user(user, "certificate") != NULLVP) {

	if ((issuerav = tag_user(user, "issuer-name")) == NULLVP) {
	    retpath->status = BNOISS;
	    goto cleanup;
	}

        /*  see if the issuer is on the root list */

        atroot = FALSE;
        for (root = roots; root != NULLUL; root = root->next)
            if (tagval_user(root->user,"subject-name",*(issuerav+1)) != 
		NULLVP) {
		issuer = dupav(root->user);
                atroot = TRUE;
                break;
            }
	
	/* If not, see of the issuer is in the database */

        if (!atroot) {
            rewind_indexfile();
            if ((issuer = get_tv_user("subject-name", *(issuerav+1))) ==
		NULLVP) {
                retpath->status = BNOISS;
                goto cleanup;
            }
        }

        /* Is the certificate ok? */

	if ((retpath->status = verify_cert(user, issuer)) != OK)
            goto cleanup;

    }
    else
	goto cleanup;


    /*  See if we're at the top (self signed) */

    userpk = key_user(user);
    isspk = key_user(issuer);
    attop = (!strcmp(userpk, isspk));
		     
    FREE(userpk);
    FREE(isspk);

    if (attop && !atroot) {
	retpath->status = BSLFSGN;
	goto cleanup;
    }

    /* Are we done? */
    
    retpath->status = OK;

    if (atroot) { 
	*trusted = TRUE;
	retpath->next = alloc_ul();
	retpath->next->user = issuer;
	issuer = NULLVP;
	retpath->next->status = OK;
    }
    else 
	retpath->next = validate_user (issuer, roots, trusted);

 cleanup:

    FREE(issuer_alias);
    FREE_AV(issuer);

    return retpath;
}
