#!/bin/sh
#
# James M. Galvin <galvin@tis.com>				1/94
# 
#  4/94 Pierre Pasturel <pasturel@la.tis.com> - XMH/PEM6.2 Integration
# 
#  4/94 Jim Galvin <galvin@tis.com> - add "streamline" feature
# 
#  6/94 Jim Galvin <galvin@tis.com> - sed requires NEWLINE, so new tools
# 
#  9/94 Jim Galvin <galvin@tis.com> - drop support for 1421 PEM
# 
# 10/94 Jim Galvin <galvin@tis.com> - provide support for generic PEM


# This script is suitable for use as an MH user's sendproc or as a filter
# for other applications that want a convenient way to apply security
# services to data.  However, even as a filter, it requires the presence of
# MH 6.8.3's mhn program.
# 
# For an MH user to use this script the following line must be included in
# the user's .mh_profile:
#
# 	sendproc: pemsend.sh
# 
# The variable MHLPROC, which specifies the location of the MHL program, may
# need to be tailored, especially if you are not an MH user.  See the
# subroutine "domhl" that appears after these comments.


# This script assumes it will be called with a filename argument.  Although
# MH "send" does allow alternative command line arguments for identifying
# messages to be submitted for delivery, the most popular programmatic
# invocations of "send" do not use these arguments.  Thus, this script is
# most suitable for those purposes and is not intended to replace "send".
# If you are not using MH send, always invoke this script with a filename
# argument.


# The command line arguments, which override default arguments specified in
# an MH user's .mh_profile, are examined to determine if the message to be
# submitted for delivery is to have privacy enhancements applied first.
# 
# The following command line arguments are supported:
# 
# 	-privacy	-- perform -sign and -encrypt
# 	-noprivacy	-- do not perform -privacy
# 
# 	-encrypt	-- apply encryption to message, recip= are required
# 	-noencrypt	-- do not perform -encrypt
# 	recip=ALIAS	-- indicate confidential recipients of message; as many
#			   of these as there are recipients may be specified
# 	-norecip	-- delete any previously specified recipients
#	alias ALIAS	-- argument is required; provided for compatibility
#			   with TIS/PEM command line conventions
# 
# 	-sign		-- apply a digital signature to message, orig= optional
# 	-nosign		-- do not perform -sign
# 	orig=ALIAS	-- indicate digital signature originator of message;
#			   only one may be specified
# 	-noorig		-- delete any previously specified originator
#	sig-alias ALIAS	-- argument is required; provided for compatibility
#			   with TIS/PEM command line conventions
# 
#	micalg=HASH	-- the hash algorithm to use to generate the hash value
#			   that is signed to create the digital signature
# 
#	protocol=application/pem-signature
# 			-- recognized for interface compatibility
#	protocol=application/pem-keys
# 			-- recognized for interface compatibility
# 
# 	-keydata	-- include keydata in message
# 	-nokeydata	-- do not perform -keydata
#
# 	-msg		-- data to be enhanced is an RFC 822 message
# 	-nomsg		-- data to be enhanced is a file; -type and -stype
#			   are also required
# 
# 	-send		-- invoke "send" after enhancements
# 	-nosend		-- do not invoke "send" after enhancements
#
# 	-verbose	-- print informative error messages
#	-noverbose	-- do not print informative error messages
# 
# 	-type TYPE	-- indicates MIME content-type of data to be
# 			   enhanced; valid only with -nomsg
# 	-stype STYPE	-- indicates MIME sub content-type of data to be
#			   enhanced; valid only with -nomsg
# 
# 	-*		-- all other '-' arguments are passed to SEND
# 	*		-- all other arguments are assumed to be files
# 
# If there are multiple occurances of any option, except as explicitly
# specified, the last occurance will override all other occurances.


status=99
trap 'rm -f /tmp/b$$ /tmp/h$$ /tmp/m$$ /tmp/#m$$ /tmp/#m$$.orig; exit $status' 0 1 2 3 9 15

umask 0077


# this routine is used throughout; do not change

msg() {
	if [ $VERBOSE -eq 1 ]; then
		echo $0: $1 1>&2
	fi
}


# this routine is a convenience used only by domhl(); if you are not an MH
# user you will probably want to change this routine so that it does not
# print out the warning message

setupcat() {
	if [ $# -ne 0 ]; then
		msg "missing \"$1\"\; continuing with \"cat\"..."
	fi
	MHLPROC=cat
	MHLARGS=
}


# The following routine may need to be tailored appropriately.  If you
# change it we recommend you add another "elif" rather than changing entries
# that are here.  Also, please tell us and we'll add it to the distribution.

domhl() {
	MHLARGS="-form mhl.pemheaders"
	if [ -r /usr/lib/mh/mhl -a -x /usr/lib/mh/mhl ]; then
		if [ -r /usr/lib/mh/mhl.pemheaders ]; then
			MHLPROC=/usr/lib/mh/mhl
		else
			setupcat /usr/lib/mh/mhl.pemheaders
		fi
	elif [ -r /usr/local/lib/mh/mhl -a -x /usr/local/lib/mh/mhl ]; then
		if [ -r /usr/local/lib/mh/mhl.pemheaders ]; then
			MHLPROC=/usr/local/lib/mh/mhl
		else
			setupcat /usr/local/lib/mh/mhl.pemheaders
		fi
	elif [ -r /usr/local/tmhlib/mhl -a -x /usr/local/tmhlib/mhl ]; then
		if [ -r /usr/local/tmhlib/mhl.pemheaders ]; then
			MHLPROC=/usr/local/tmhlib/mhl
		else
			setupcat /usr/local/tmhlib/mhl.pemheaders
		fi
	elif [ $MSG -eq 1 ]; then
		setupcat mhl
	else
		setupcat
	fi
}


# parse all the arguments; this routine is also used to parse just PEM
# arguments so we reset them upon invocation

doargs() {

	DOENC=0
	DOSIG=0

	ORIG=
	RECIPS=

	HASH=

	while [ $# -gt 0 ]; do
		case $1 in
		-priv*)
			DOENC=1 ; DOSIG=1 ; shift ;;
		-nopriv*)
			DOENC=0 ; DOSIG=0 ; shift ;;

		-enc*)
			DOENC=1 ; shift ;;
		-noenc*)
			DOENC=0 ; shift ;;
		recip=*)
			RECIPS="$RECIPS $1" ; shift ;;
		-norecip)
			RECIPS= ; shift ;;
		alias)
			if [ $# -eq 1 ]; then
				msg "Missing argument for option alias."
			fi
			shift
			RECIPS="$RECIPS recip=$1" ; shift ;;

		-sign)
			DOSIG=1
			if [ $DOENC -eq 1 ]; then
				msg "Signing will occur before encrypting."
			fi
			shift ;;
		-nosign)
			DOSIG=0 ; shift ;;
		orig=*)
			ORIG="$1" ; shift ;;
		-noorig)
			ORIG= ; shift ;;
		sig-alias)
			if [ $# -eq 1 ]; then
				msg "Missing argument for option sig-alias."
			fi
			shift
			ORIG="orig=$1" ; shift ;;

		micalg=md5)
			HASH=$1 ; shift ;;
		hashalg=md5)
			HASH=$1 ; shift ;;
		micalg=*)
			echo $0: unknown option \"$1\" 1>&2
			echo $0: message preserved 1>&2
			status=1
			exit ;;

		protocol=application/pem-signature)
			shift ;;
		protocol=application/pem-keys)
			shift ;;
		protocol=*)
			echo $0: unknown option \"$1\" 1>&2
			echo $0: message preserved 1>&2
			status=1
			exit ;;

		-keydata)
			DOKEY=1 ; shift ;;
		-nokeydata)
			DOKEY=0 ; shift ;;

		-msg)
			MSG=1 ; shift ;;
		-nomsg)
			DOSND=0
			MSG=0 ; shift ;;
		-send)
			MSG=1
			DOSND=1 ; shift ;;
		-nosend)
			DOSND=0 ; shift ;;

		-verbose)
			VERBOSE=1 ; shift ;;
		-noverbose)
			VERBOSE=0 ; shift ;;

		-type)
			if [ $# -eq 1 ]; then
				msg "Missing argument for option -type."
			fi
			shift
			TYPE=$1 ; shift ;;
		-stype)
			if [ $# -eq 1 ]; then
				msg "Missing argument for option -stype."
			fi
			shift
			STYPE=$1 ; shift ;;

		-*)
			ARGS="$ARGS $1" ; shift ;;

		*)
			FILES="$FILES $1" ; shift ;;
		esac
	done
}


# this provides a pass through mode so the script is effective at all times

nopem() {
	if [ ! -z "$ORIG" ]; then
		msg "option orig= ignored without pem options"
	fi
	if [ ! -z "$RECIPS" ]; then
		msg "option recip= ignored without pem options"
	fi
	if [ ! -z "$HASH" ]; then
		msg "option micalg= ignored without pem options"
	fi
	if [ $DOSND -eq 1 ]; then
		send $ARGS $1
		if [ $? -ne 0 ]; then
			echo $0: ERROR sending \"$1\" 1>&2
			status=1
			exit
		fi
	fi
}


# this routine checks the arguments for self-consistency

chkpemargs() {
	if [ $DOENC -eq 0 ]; then
		if [ ! -z "$RECIPS" ]; then
			msg "option recip= ignored without -encrypt"
		fi
	else
		if [ -z "$RECIPS" ]; then
			echo $0: missing recip= for -encrypt 1>&2
			echo $0: message preserved in $FILES 1>&2
			status=1
			exit
		fi
		if [ $DOSIG -eq 1 ]; then
			msg "signature will be applied first"
		fi
	fi
	if [ $DOSIG -eq 0 ]; then
		if [ ! -z "$ORIG" ]; then
			msg "option orig= ignored without -sign"
		fi
	fi
	if [ $MSG -eq 0 ]; then
		if [ -z "$TYPE" -o -z "$STYPE" ]; then
			echo $0: missing -type or -stype for -nomsg 1>&2
			echo $0: data preserved in $FILES 1>&2
			status=1
			exit
		fi
	else
		if [ ! -z "$TYPE" -o ! -z "$STYPE" ]; then
			msg "options -type and -stype ignored without -nomsg"
		fi
	fi
}


# this routine applies the PEM enhancements to the message in the argument
# filename

dopemmsg() {
	# sigh -- data on stdout and headers on stderr; normal UNIX
	# utilities don't work because they assume lines are terminated
	# with a NEWLINE character; therefore we "invented" msgsplit and
	# headskip; note that the headers are always NEWLINE terminated but
	# the body may be binary data
	#
	# "msgsplit" separates the headers and body of a message
	# 
	# "headskip -#" copies all but the first # lines

	msgsplit < $1			>  /tmp/b$$	2> /tmp/h$$

	cp /tmp/h$$ 					   /tmp/m$$

	if [ $DOENC -eq 1 ]; then
		echo -n "#encrypt"			>> /tmp/m$$
		for r in $RECIPS ; do
			echo -n "; $r"			>> /tmp/m$$
		done
		echo ""					>> /tmp/m$$
	fi

	if [ $DOSIG -eq 1 ]; then
		echo -n "#sign"				>> /tmp/m$$
		if [ ! -z "$ORIG" ]; then
			echo -n "; $ORIG"		>> /tmp/m$$
		fi
		if [ ! -z "$HASH" ]; then
			echo -n "; $HASH"		>> /tmp/m$$
		fi
		echo ""					>> /tmp/m$$
	fi

	echo '#<message/rfc822'				>> /tmp/m$$
	$MHLPROC $MHLARGS /tmp/h$$			>> /tmp/m$$
	echo ""						>> /tmp/m$$
	cat /tmp/b$$					>> /tmp/m$$

	if [ $DOKEY -ne 0 ]; then
		echo "#application/key-data"		>> /tmp/m$$
	fi

	mhdraft="/tmp/m$$" ; export mhdraft
	mhn /tmp/m$$
	if [ $? -ne 0 ]; then
		echo $0: message preserved in $1 1>&2
		status=1
		exit
	fi

	cp $1 $1.orig
	cp /tmp/m$$ $1

	if [ $DOSND -eq 1 ]; then
		send $ARGS $1
		if [ $? -ne 0 ]; then
			echo $0: ERROR sending \"$1\" 1>&2
			status=1
			exit
		fi
	fi
}


# this routine applies the PEM enhancements to the data in the argument
# filename

dopemfile() {
	# sigh -- data on stdout and headers on stderr; normal UNIX
	# utilities don't work because they assume lines are terminated
	# with a NEWLINE character; therefore we "invented" msgsplit and
	# headskip; note that the headers are always NEWLINE terminated but
	# the body may be binary data
	#
	# "msgsplit" separates the headers and body of a message
	# 
	# "headskip -#" copies all but the first # lines

	cp /dev/null 					   /tmp/m$$

	echo ""						>> /tmp/m$$
	if [ $DOENC -eq 1 ]; then
		echo -n "#encrypt"			>> /tmp/m$$
		for r in $RECIPS ; do
			echo -n "; $r"			>> /tmp/m$$
		done
		echo ""					>> /tmp/m$$
	fi

	if [ $DOSIG -eq 1 ]; then
		echo -n "#sign"				>> /tmp/m$$
		if [ ! -z "$ORIG" ]; then
			echo -n "; $ORIG"		>> /tmp/m$$
		fi
		if [ ! -z "$HASH" ]; then
			echo -n "; $HASH"		>> /tmp/m$$
		fi
		echo ""					>> /tmp/m$$
	fi

	echo -n '#<'					>> /tmp/m$$
	echo "$TYPE/$STYPE"				>> /tmp/m$$
	cat $1						>> /tmp/m$$

	if [ $DOKEY -ne 0 ]; then
		echo "#application/key-data"		>> /tmp/m$$
	fi

	cp /tmp/m$$ /tmp/M
	mhdraft="/tmp/m$$" ; export mhdraft
	mhn /tmp/m$$
	if [ $? -ne 0 ]; then
		echo $0: message preserved in $1 1>&2
		status=1
		exit
	fi

	cp $1 $1.orig
	cp /tmp/m$$ $1

	if [ $DOSND -eq 1 ]; then
		send $ARGS $1
		if [ $? -ne 0 ]; then
			echo $0: ERROR sending \"$1\" 1>&2
			status=1
			exit
		fi
	fi
}


# applies privacy enhancements to messages and files appropriately

dopem() {
	if [ $MSG -eq 1 ]; then
		dopemmsg $*
	else
		dopemfile $*
	fi
}


## Processing BEGINs here

DOKEY=0
DOSND=1
MSG=1
VERBOSE=1

ARGS=
FILES=

TYPE=
STYPE=

doargs $*

domhl

if [ -z "$FILES" ]; then
	echo $0: no files specified\; unsupported option 1>&2
	status=1
	exit
fi

if [ $DOSIG -eq 0 -a $DOENC -eq 0 ]; then

	# what we need to do is look for PEM directives on the first line of
	# each file we're submitting for delivery

	for f in $FILES ; do
		LINE="`sed -n '1p' $f`"
		echo "$LINE" | egrep -s '^#'
		if [ $? -ne 0 ]; then
			nopem $f
		else
			# it would be nice to have only one doargs line
			# but I couldn't think of a way to do it.

			echo "$LINE" | egrep -s ';'
			if [ $? -ne 0 ]; then
				doargs `echo "$LINE" | \
					sed -e 's/$/%/' \
					    -e 's/#[ 	]*\([a-z]*\).*%/-\1/'`
			else
				doargs `echo "$LINE" | \
					sed -e 's/;/%;/' \
					    -e 's/#[ 	]*\([a-z]*\).*%/-\1/' \
					    -e 's/;/ /g'`
			fi

			if [ $DOSIG -eq 0 -a $DOENC -eq 0 ]; then
				nopem $f
			else
				chkpemargs

				# here again, normal UNIX utilities
				# don't work because they assume lines
				# are terminated with a NEWLINE

				cat $f | headskip -1 > $f.pem
				mv $f.pem $f
				dopem $f
			fi
		fi
	done
	status=0
	exit
fi

chkpemargs

for f in $FILES ; do
	dopem $f
done

status=0
exit
